/****************************************************************************
 *
 *	Copyright (c) 1999-2008, Watanabe Lab, School of Media Science,
 *	Tokyo University of Technology, All rights reserved.
 *
 *	Redistribution and use in source and binary forms,
 *	with or without modification, are permitted provided that the
 *	following conditions are met:
 *
 *		- Redistributions of source code must retain the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer.
 *
 *		- Redistributions in binary form must reproduce the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer in the documentation and/or
 *			other materials provided with the distribution.
 *
 *		- Neither the name of the copyright holders nor the names
 *			of its contributors may be used to endorse or promote
 *			products derived from this software without specific
 *			prior written permission.
 *
 *	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *	FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 *	COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 *	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 *	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 *	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 *	POSSIBILITY OF SUCH DAMAGE. 
 *
 ****************************************************************************/
/****************************************************************************
 *
 *	Copyright (c) 1999-2008, Watanabe Lab, School of Media Science,
 *	Tokyo University of Technology, All rights reserved.
 *
 *	本ソフトウェアおよびソースコードのライセンスは、基本的に
 *	「修正 BSD ライセンス」に従います。以下にその詳細を記します。
 *
 *	ソースコード形式かバイナリ形式か、変更するかしないかを問わず、
 *	以下の条件を満たす場合に限り、再頒布および使用が許可されます。
 *
 *	- ソースコードを再頒布する場合、上記の著作権表示、本条件一覧、
 *		および下記免責条項を含めること。
 *
 *	- バイナリ形式で再頒布する場合、頒布物に付属のドキュメント等の
 *		資料に、上記の著作権表示、本条件一覧、および下記免責条項を
 *		含めること。
 *
 *	- 書面による特別の許可なしに、本ソフトウェアから派生した製品の
 *		宣伝または販売促進に、本ソフトウェアの著作権者の名前または
 *		コントリビューターの名前を使用してはならない。
 *
 *	本ソフトウェアは、著作権者およびコントリビューターによって「現
 *	状のまま」提供されており、明示黙示を問わず、商業的な使用可能性、
 *	および特定の目的に対する適合性に関す暗黙の保証も含め、またそれ
 *	に限定されない、いかなる保証もないものとします。著作権者もコン
 *	トリビューターも、事由のいかんを問わず、損害発生の原因いかんを
 *	問わず、かつ責任の根拠が契約であるか厳格責任であるか(過失その
 *	他の)不法行為であるかを問わず、仮にそのような損害が発生する可
 *	能性を知らされていたとしても、本ソフトウェアの使用によって発生
 *	した(代替品または代用サービスの調達、使用の喪失、データの喪失、
 *	利益の喪失、業務の中断も含め、またそれに限定されない)直接損害、
 *	間接損害、偶発的な損害、特別損害、懲罰的損害、または結果損害に
 *	ついて、一切責任を負わないものとします。
 *
 ****************************************************************************/
#include "TFK_WindowBase.h"
#include "TFK_API.h"
#include <FK/Plane.h>
#include <FL/fl_ask.h>

using namespace std;

const double	ONE_FRAME_TIME = 0.001;

static void _callback1(Fl_Widget *argW, void *dummy)
{
	Fl::atclose((Fl_Window *)argW, dummy);
	return;
}

static void _callback2(Fl_Widget *argW, void *dummy)
{
	Fl_Window		*win = (Fl_Window *)argW;

	Fl_Widget::default_callback(argW, dummy);

	if(fl_choice("Really Quit?", "No", "Yes", 0L) == 1) {
		win->hide();
	}
	return;
}


TFK_WindowBase::TFK_WindowBase(void)
{
	baseWin = (Fl_Window *)NULL;
	fkWin = (fk_Window *)NULL;

	baseW = baseH = -1;
	fkX = fkY = fkW = fkH = -1;
	drawEvent = -1;

	putWin = (Fl_Window *)NULL;
	browser = (Fl_Multi_Browser *)NULL;

	TFK_API::setFont();

	return;
}

TFK_WindowBase::~TFK_WindowBase()
{
	Fl::check();
	delete baseWin;
	delete putWin;

	return;
}

unsigned long TFK_WindowBase::getNow(void)
{

#if defined(WIN32)

	return (unsigned long)timeGetTime();

#else

	struct timeval		tm;
	unsigned long		retVal;

	gettimeofday(&tm, NULL);
	retVal = (tm.tv_sec % 16777216) * 1000;
	retVal += tm.tv_usec/1000;
	return retVal;

#endif

}

void TFK_WindowBase::setScene(fk_Scene *argScene)
{
	scene = argScene;
	return;
}

void TFK_WindowBase::setBaseSize(int argW, int argH)
{
	if(argW > 0 && argW <= TFK_MAX_WINSIZE &&
	   argH > 0 && argH <= TFK_MAX_WINSIZE) {
		baseW = argW;
		baseH = argH;
	}
	return;
}

void TFK_WindowBase::setCanvas(int argX, int argY, int argW, int argH)
{
	if(argX >= 0 && argX <= baseW &&
	   argY >= 0 && argY <= baseH &&
	   argW > 0 && argW <= baseW - argX &&
	   argH > 0 && argH <= baseH - argY) {
		fkX = argX;
		fkY = argY;
		fkW = argW;
		fkH = argH;
	}
	return;
}

bool TFK_WindowBase::openWindow(void)
{
	if(baseW > 0 && baseH > 0) {
		baseWin = new Fl_Window(baseW, baseH, "TinyFK Window");
		if(fkW > 0 && fkH > 0) {
			fkWin = new fk_Window(fkX, fkY, fkW, fkH);
			fkWin->setScene(scene);
		} else {
			fkWin = (fk_Window *)NULL;
		}

		gui.create();

		baseWin->end();
		baseWin->show();
		if(fkWin != (fk_Window *)NULL) fkWin->show();

	} else {
		fl_alert("Error: Illegal Window Size");
		return false;
	}
	return true;
}

void TFK_WindowBase::drawWindow(void)
{
	if(baseWin == (Fl_Window *)NULL) return;
	if(baseWin->shown() == 0) return;
	if(baseWin->visible() == 0) {
		if(Fl::check() == 0) return;
		Fl::wait();
	} else {
		if(Fl::event_key() == FL_Escape) return;
		if(fkWin != (fk_Window *)NULL) {
			drawEvent = fkWin->drawWindow();
		}
	}

	return;
}

bool TFK_WindowBase::waitEvent(void)
{
	if(drawEvent == 0) return false;
	if(baseWin->shown() == 0) return false;
	if(Fl::wait() == 0) return false;
	return true;
}

bool TFK_WindowBase::waitEvent(double argT)
{
	// double		restTime;

	if(drawEvent == 0) return false;
	if(baseWin == (Fl_Window *)NULL) return false;
	if(baseWin->shown() == 0) return false;
	if(Fl::check() == 0) return false;
	/*
	restTime = argT - ONE_FRAME_TIME;
	if(Fl::wait(argT) < restTime) {
		if(Fl::check() == 0) return false;
	}
	*/
	if(Fl::wait(argT) < FK_EPS) return false;
	return true;
}

bool TFK_WindowBase::checkEvent(void)
{
	if(drawEvent == 0) return false;
	if(baseWin == (Fl_Window *)NULL) return false;
	if(baseWin->shown() == 0) return false;
	if(Fl::check() == 0) return false;
	return true;
}

void TFK_WindowBase::sleepTime(double argT)
{
	unsigned long		t1 = getNow();
	unsigned long		t2 = t1 + (unsigned long)(argT*1000.0);
	unsigned long		regTime;
	double				waitTime = 0.0;

#ifdef WIN32
	timeBeginPeriod(1);
#endif

	do {
		regTime = getNow();
		if(Fl::wait(argT - waitTime) < FK_EPS) break;
		drawWindow();
		waitTime = (getNow()-regTime)/1000.0;
	} while(t2 > getNow() && Fl::check() != 0);

#ifdef WIN32
	timeEndPeriod(1);
#endif

	return;
}

void TFK_WindowBase::setBlendMode(bool argMode)
{
	if(scene == (fk_Scene *)NULL) return;
	scene->setBlendStatus(argMode);
	return;
}

bool TFK_WindowBase::getBlendMode(void)
{
	if(scene == (fk_Scene *)NULL) return false;
	return scene->getBlendStatus();
}

bool TFK_WindowBase::getMouseStatus(int argButton)
{
	if(fkWin == (fk_Window *)NULL) return false;

	switch(argButton) {
	  case 0:
		return (fkWin->getMouseStatus(FK_MOUSE1, false) |
				fkWin->getMouseStatus(FK_MOUSE2, false) |
				fkWin->getMouseStatus(FK_MOUSE3, false));
	  case 1:
		return fkWin->getMouseStatus(FK_MOUSE1, false);
	  case 2:
		return fkWin->getMouseStatus(FK_MOUSE3, false);
	  case 3:
		return fkWin->getMouseStatus(FK_MOUSE2, false);

	  default:
		break;
	}
	return false;
}

fk_Vector TFK_WindowBase::getMousePosition(void)
{
	fk_Vector	org(0.0, 0.0, 0.0), norm(0.0, 0.0, -1.0), ret(0.0, 0.0, 0.0);
	fk_Plane	plane;
	int			x, y;

	if(fkWin == (fk_Window *)NULL) return ret;
	fkWin->getMousePosition(&x, &y, true);
	if(x == -1 && y == -1) {
		ret.set(-10000.0, -10000.0, -10000.0);
		return ret;
	}
	plane.setPosNormal(org, norm);
	fkWin->getProjectPosition(double(x), double(y), &plane, &ret);

	return ret;
}

fk_Vector TFK_WindowBase::waitMouse(int argButton)
{
	bool		oldStatus, curStatus;
	fk_Vector	dummy(0.0, 0.0, 0.0);

	if(baseWin == (Fl_Window *)NULL) return dummy;
	if(baseWin->shown() == 0) return dummy;
	if(Fl::check() == 0) return dummy;
	oldStatus = getMouseStatus(argButton);
	while(waitEvent() == true) {
		if(Fl::check() == 0) break;
		curStatus = getMouseStatus(argButton);
		if(curStatus == true && oldStatus == false) {
			return getMousePosition();
		}
		oldStatus = curStatus;
	}

	return dummy;
}

bool TFK_WindowBase::getKeyStatus(char argC)
{
	if(fkWin == (fk_Window *)NULL) return false;
	return fkWin->getKeyStatus(argC, false);
}

bool TFK_WindowBase::getSpecialKeyStatus(fk_SpecialKey argK)
{
	if(fkWin == (fk_Window *)NULL) return false;
	return fkWin->getSpecialKeyStatus(argK, false);
}

void TFK_WindowBase::setESCExitMode(bool argMode)
{
	if(argMode == true) {
		baseWin->callback(_callback1);
	} else {
		baseWin->callback(_callback2);
	}

	return;
}

void TFK_WindowBase::makeButton(int argID, int argType,
								int argX, int argY, int argW, int argH,
								string argCaption)
{
	gui.makeButton(argID, argType, argX, argY, argW, argH, argCaption);
	return;
}

bool TFK_WindowBase::getButtonStatus(int argID)
{
	return gui.getButtonStatus(argID);
}

void TFK_WindowBase::setButtonStatus(int argID, bool argStatus)
{
	gui.setButtonStatus(argID, argStatus);
	return;
}

void TFK_WindowBase::makeTextBox(int argID, int argX, int argY,
							 int argW, int argH, string argText)
{
	gui.makeTextBox(argID, argX, argY, argW, argH, argText);
	return;
}

string TFK_WindowBase::getTextBoxString(int argID)
{
	return gui.getTextBoxString(argID);
}

void TFK_WindowBase::setTextBoxString(int argID, string argText)
{
	gui.setTextBoxString(argID, argText);
	return;
}

void TFK_WindowBase::makeGroup(int argID, int argX, int argY,
							   int argW, int argH)
{
	gui.makeGroup(argID, argX, argY, argW, argH);
	return;
}

void TFK_WindowBase::entryButton(int argBID, int argGID)
{
	gui.entryButton(argBID, argGID);
	return;
}

string TFK_WindowBase::fileChooser(int argMode)
{
	Fl_File_Chooser		*fc;
	const char			*fCharP;
	string				fileName;
	bool				loopFlag;
	FILE				*fp;

	fileName.clear();

	loopFlag = true;
	while(loopFlag == true) {

		switch(argMode) {
		  case 1:
			fc = new Fl_File_Chooser(".", "*", Fl_File_Chooser::CREATE,
									 "TinyFK File Chooser(SAVE)");
			break;
		  case 2:
			fc = new Fl_File_Chooser(".", "*", Fl_File_Chooser::SINGLE,
									 "TinyFK File Chooser(LOAD)");
			break;
		  default:
			return fileName;
		}

		fc->show();
		while(fc->visible()) {
			Fl::wait();
		}

		fCharP = fc->value(1);
		if(fCharP == (const char *)NULL) {
			delete fc;
			return fileName;
		}
		fileName = fCharP;
		delete fc;

		if(argMode == 2) {
			loopFlag = false;
		} else {
			if((fp = fopen(fileName.c_str(), "rb")) != (FILE *)NULL) {
				if(fl_choice("Overwrite?", "No", "Yes", 0L) == 1) {
					loopFlag = false;
				}
				fclose(fp);
			} else {
				loopFlag = false;
			}
		}
	}


	return fileName;
}

fk_Color TFK_WindowBase::colorChooser(void)
{
	fk_Color		col;

	col.set(0.0, 0.0, 0.0);
	return colorChooser(col);
}

fk_Color TFK_WindowBase::colorChooser(fk_Color argCol)
{
	double		r, g, b;
	fk_Color	retCol;

	r = double(argCol.getR());
	g = double(argCol.getG());
	b = double(argCol.getB());

	if(fl_color_chooser("Color Chooser", r, g, b) == 0) {
		return argCol;
	}

	retCol.set(r, g, b);
	return retCol;
}

void TFK_WindowBase::entryTextBox(int argTID, int argGID)
{
	gui.entryTextBox(argTID, argGID);
	return;
}

void TFK_WindowBase::putString(string argStr)
{
	static const string		space = "			 ";
	string					output, str;
	string::size_type		index, old;

	if(putWin == (Fl_Window *)NULL) {
		putWin = new Fl_Window(320, 520, "TinyFK Put Window");
		browser = new Fl_Multi_Browser(10, 10, 300, 500);
		putWin->end();
		putWin->show();
	}

	index = argStr.find('\n');
	output = argStr.substr(0, index);
	while(index != string::npos) {
		str = output + space;
		browser->add(str.c_str());
		old = index+1;
		index = argStr.find('\n', old);
		if(index == string::npos) {
			output = argStr.substr(old, index);
		} else {
			output = argStr.substr(old, index-old);
		}
	}

	if(output.length() != 0) {
		str = output + space;
		browser->add(str.c_str());
	}

	browser->bottomline(99999999);
	return;
}

void TFK_WindowBase::openPutWindow(int argW, int argH)
{
	if(putWin == (Fl_Window *)NULL) {
		putWin = new Fl_Window(argW + 20, argH + 20, "TinyFK Put Window");
		browser = new Fl_Multi_Browser(10, 10, argW, argH);
		putWin->end();
		putWin->show();
	}

	return;
}

void TFK_WindowBase::closePutWindow(void)
{
	delete putWin;
	putWin = (Fl_Window *)NULL;

	return;
}

void TFK_WindowBase::clearPutWindow(void)
{
	if(putWin != (Fl_Window *)NULL) {
		browser->clear();
	}

	return;
}

bool TFK_WindowBase::snapImage(string argFName)
{
	return fkWin->snapImage(argFName);
}

void TFK_WindowBase::message(string argStr)
{
	fl_message(argStr.c_str());
	return;
}

void TFK_WindowBase::alert(string argStr)
{
	fl_alert(argStr.c_str());
	return;
}

bool TFK_WindowBase::ask(string argStr)
{
	if(fl_choice(argStr.c_str(), "No", "Yes", 0L) == 1) return true;
	return false;
}
