/****************************************************************************
 *
 *	Copyright (c) 1999-2011, Fine Kernel Project, All rights reserved.
 *
 *	Redistribution and use in source and binary forms,
 *	with or without modification, are permitted provided that the
 *	following conditions are met:
 *
 *		- Redistributions of source code must retain the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer.
 *
 *		- Redistributions in binary form must reproduce the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer in the documentation and/or
 *			other materials provided with the distribution.
 *
 *		- Neither the name of the copyright holders nor the names
 *			of its contributors may be used to endorse or promote
 *			products derived from this software without specific
 *			prior written permission.
 *
 *	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *	FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 *	COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 *	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 *	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 *	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 *	POSSIBILITY OF SUCH DAMAGE. 
 *
 ****************************************************************************/
/****************************************************************************
 *
 *	Copyright (c) 1999-2011, Fine Kernel Project, All rights reserved.
 *
 *	本ソフトウェアおよびソースコードのライセンスは、基本的に
 *	「修正 BSD ライセンス」に従います。以下にその詳細を記します。
 *
 *	ソースコード形式かバイナリ形式か、変更するかしないかを問わず、
 *	以下の条件を満たす場合に限り、再頒布および使用が許可されます。
 *
 *	- ソースコードを再頒布する場合、上記の著作権表示、本条件一覧、
 *		および下記免責条項を含めること。
 *
 *	- バイナリ形式で再頒布する場合、頒布物に付属のドキュメント等の
 *		資料に、上記の著作権表示、本条件一覧、および下記免責条項を
 *		含めること。
 *
 *	- 書面による特別の許可なしに、本ソフトウェアから派生した製品の
 *		宣伝または販売促進に、本ソフトウェアの著作権者の名前または
 *		コントリビューターの名前を使用してはならない。
 *
 *	本ソフトウェアは、著作権者およびコントリビューターによって「現
 *	状のまま」提供されており、明示黙示を問わず、商業的な使用可能性、
 *	および特定の目的に対する適合性に関す暗黙の保証も含め、またそれ
 *	に限定されない、いかなる保証もないものとします。著作権者もコン
 *	トリビューターも、事由のいかんを問わず、損害発生の原因いかんを
 *	問わず、かつ責任の根拠が契約であるか厳格責任であるか(過失その
 *	他の)不法行為であるかを問わず、仮にそのような損害が発生する可
 *	能性を知らされていたとしても、本ソフトウェアの使用によって発生
 *	した(代替品または代用サービスの調達、使用の喪失、データの喪失、
 *	利益の喪失、業務の中断も含め、またそれに限定されない)直接損害、
 *	間接損害、偶発的な損害、特別損害、懲罰的損害、または結果損害に
 *	ついて、一切責任を負わないものとします。
 *
 ****************************************************************************/
#include <FK/Model.h>
#include <FK/Material.h>
#include <FK/Shape.h>
#include <FK/Tree.h>
#include <FK/Quaternion.h>
#include <FK/Boundary.h>

static unsigned int		_globalModelID = 1;

fk_Model::fk_Model(fk_Shape *argShape)
	: fk_MatrixAdmin(FK_MODEL)
{
	material = new fk_Material;

	setDrawMode(FK_NONEMODE);
	setMaterialMode(FK_CHILD_MODE);

	parent = NULL;
	treeData = NULL;
	setShape(argShape);
	deleteMaterial();
	setSize(1.0);
	setWidth(1.0);
	setPickMode(false);
	setReverseDrawMode(false);

	pointColor = NULL;
	lineColor = NULL;

	_modelID = _globalModelID;
	_globalModelID++;

	treeFlag = false;
	smoothFlag = false;

	snapPos.set(0.0, 0.0, 0.0);
	snapInhPos.set(0.0, 0.0, 0.0);
	snapAngle.set(0.0, 0.0, 0.0);
	snapFlag = false;
	boundaryMode = FK_SPHERE_B;
	sbRadius = 0.0;

	return;
}

fk_Model::~fk_Model()
{
	deletePointColor();
	deleteLineColor();
	DeleteTree();

	delete material;
	return;
}

void fk_Model::setShape(fk_Shape *argShape)
{
	bool	drawModeFlag;

	if(argShape == NULL) {
		shape = NULL;
		return;
	}

	if(shape == NULL) {
		drawModeFlag = true;
	} else if(shape->getObjectType() != argShape->getObjectType()) {
		drawModeFlag = true;
	} else {
		drawModeFlag = false;
	}

	if(drawModeFlag == true) {
		switch(argShape->getObjectType()) {
		  case FK_POINT:
			drawMode = FK_POINTMODE;
			break;

		  case FK_LINE:
		  case FK_POLYLINE:
		  case FK_CLOSEDLINE:
			drawMode = FK_LINEMODE;
			break;

		  case FK_POLYGON:
		  case FK_BLOCK:
		  case FK_CIRCLE:
		  case FK_SPHERE:
		  case FK_PRISM:
		  case FK_CONE:
		  case FK_SOLID:
		  case FK_INDEXFACESET:
			drawMode = FK_POLYMODE;
			break;

		  case FK_RECTTEXTURE:
		  case FK_TRITEXTURE:
		  case FK_MESHTEXTURE:
		  case FK_IFSTEXTURE:
			drawMode = FK_TEXTUREMODE;
			break;

		  default:
			drawMode = FK_NONEMODE;
			break;
		}
	}

	shape = argShape;

	return;
}

void fk_Model::setMaterial(const fk_Material &argMate)
{
	*material = argMate;
	materialFlag = true;
	return;
}

void fk_Model::setPointColor(fk_Color *argColor)
{
	if(argColor == NULL) {
		return;
	}

	setPointColor(argColor->getR(), argColor->getG(), argColor->getB());
	return;
}

void fk_Model::setPointColor(float argR, float argG, float argB)
{
	if(pointColor == NULL) {
		pointColor = new fk_Color();
	}

	pointColor->set(argR, argG, argB);
	return;
}

void fk_Model::setLineColor(fk_Color *argColor)
{
	if(argColor == NULL) {
		return;
	}

	setLineColor(argColor->getR(), argColor->getG(), argColor->getB());
	return;
}

void fk_Model::setLineColor(float argR, float argG, float argB)
{
	if(lineColor == NULL) {
		lineColor = new fk_Color();
	}

	lineColor->set(argR, argG, argB);
	return;
}

fk_Shape * fk_Model::getShape(void) const
{
	return shape;
}

fk_Material * fk_Model::getMaterial(void)
{
	return material;
}

fk_Material * fk_Model::getInhMaterial(void)
{
	if(parent != NULL) {
		if(materialFlag == false) {
			return parent->getInhMaterial();
		} else {
			return material;
		}
	}
	return material;
}

fk_Color * fk_Model::getPointColor(void)
{
	return pointColor;
}

fk_Color * fk_Model::getInhPointColor(void)
{
	if(parent != NULL) {
		if(pointColor == NULL) {
			return parent->getInhPointColor();
		} else {
			return pointColor;
		}
	}
	return pointColor;
}

fk_Color * fk_Model::getLineColor(void)
{
	return lineColor;
}

fk_Color * fk_Model::getInhLineColor(void)
{
	if(parent != NULL) {
		if(lineColor == NULL) {
			return parent->getInhLineColor();
		} else {
			return lineColor;
		}
	}
	return lineColor;
}

void fk_Model::deleteMaterial(void)
{
	materialFlag = false;
	return;
}

void fk_Model::deletePointColor(void)
{
	delete pointColor;
	pointColor = NULL;
	return;
}

void fk_Model::deleteLineColor(void)
{
	delete lineColor;
	lineColor = NULL;
	return;
}

void fk_Model::setDrawMode(const fk_DrawMode argMode)
{
	drawMode = argMode;
	return;
}

fk_DrawMode fk_Model::getDrawMode(void) const
{
	return drawMode;
}

void fk_Model::setMaterialMode(const fk_MaterialMode argMode)
{
	materialMode = argMode;
	return;
}

fk_MaterialMode fk_Model::getMaterialMode(void) const
{
	return materialMode;
}

fk_Matrix fk_Model::getInhMatrix(void) const
{
	if(parent == NULL) return getMatrix();
	return (parent->getInhMatrix() * getMatrix());
}

fk_Matrix fk_Model::getInhInvMatrix(void) const
{
	if(parent == NULL) return getInvMatrix();
	return (getInvMatrix() * parent->getInhInvMatrix());
}

fk_OrthoMatrix fk_Model::getInhBaseMatrix(void) const
{
	if(parent == NULL) return OrthoMatrix;
	return (parent->getInhBaseMatrix() * OrthoMatrix);
}

fk_OrthoMatrix fk_Model::getInhInvBaseMatrix(void) const
{
	fk_OrthoMatrix	RetMat = OrthoMatrix;
	RetMat.inverse();
	if(parent == NULL) return RetMat;
	return (parent->getInhInvBaseMatrix() * RetMat);
}

fk_Vector fk_Model::getInhPosition(void) const
{
	fk_Vector retVec;

	if(parent == NULL) {
		retVec = Position;
	} else {
		retVec = parent->getInhMatrix() * Position;
	}
	return retVec;
}

fk_Vector fk_Model::getInhVec(void) const
{
	fk_HVector	hvec(Vec);

	if(parent == NULL) return Vec;

	hvec.isvec();
	return (parent->getInhMatrix() * hvec);
}
	
fk_Vector fk_Model::getInhUpvec(void) const
{
	fk_HVector	hvec(UpVec);

	if(parent == NULL) return UpVec;
	
	hvec.isvec();
	return (parent->getInhMatrix() * hvec);
}

fk_Vector fk_Model::getInhUpVec(void) const
{
	return getInhUpvec();
}

fk_Angle fk_Model::getInhAngle(void) const
{
	fk_Angle		retAngle;
	fk_Vector		vec, upvec;

	if(parent == NULL) return Angle;
	vec = getInhVec();
	upvec = getInhUpVec();
	VectorToAngle(&retAngle, &vec, &upvec);
	return retAngle;
}

void fk_Model::setSize(const double argSize)
{
	if(argSize <= FK_EPS) return;
	drawSize = argSize;
	return;
}

void fk_Model::setWidth(const double argWidth)
{
	if(argWidth <= FK_EPS) return;
	drawWidth = argWidth;
	return;
}

double fk_Model::getSize(void) const
{
	return drawSize;
}

double fk_Model::getWidth(void) const
{
	return drawWidth;
}

void fk_Model::setPickMode(const bool argFlg)
{
	pickFlag = argFlg;
	return;
}

bool fk_Model::getPickMode(void) const
{
	return pickFlag;
}

#ifndef _FREEBSD_

void fk_Model::setSmoothMode(const bool argFlg)
{
	smoothFlag = argFlg;

	return;
}

#else

void fk_Model::setSmoothMode(const bool)
{
	return;
}

#endif

bool fk_Model::getSmoothMode(void) const
{
	return smoothFlag;
}

void fk_Model::setReverseDrawMode(const bool argFlg)
{
	reverseFlag = argFlg;
	return;
}

bool fk_Model::getReverseDrawMode(void) const
{
	return reverseFlag;
}

unsigned int fk_Model::getID(void) const
{
	return _modelID;
}


void fk_Model::snapShot(void)
{
	snapPos = Position;
	snapInhPos = getInhPosition();
	snapAngle = Angle;
	snapFlag = true;
	return;
}

bool fk_Model::restore(void)
{
	if(snapFlag == false) return false;

	Angle = snapAngle;
	Position = snapPos;
	AdjustAngleToVec();
	UpdateMatrix(false);

	return true;
}

bool fk_Model::restore(double argT)
{
	fk_Quaternion		qOrg, qNew, qRet;

	if(snapFlag == false) return false;
	if(argT < FK_EPS || argT > 1.0 + FK_EPS) return false;

	qOrg.makeEuler(snapAngle);
	qNew.makeEuler(Angle);
	qRet = fk_Q_Inter_Sphere(qOrg, qNew, argT);
	Angle = qRet.getEuler();

	Position = (1.0 - argT)*snapPos +  argT*Position;
	AdjustAngleToVec();
	UpdateMatrix(false);

	return true;
}

void fk_Model::setBoundaryMode(fk_BoundaryMode argMode)
{
	boundaryMode = argMode;
	return;
}

fk_BoundaryMode fk_Model::getBoundaryMode(void)
{
	return boundaryMode;
}

void fk_Model::setSBRadius(double argRadius)
{
	if(argRadius > 0.0) sbRadius = argRadius;
	else argRadius = 0.0;

	return;
}

double fk_Model::getSBRadius(void)
{
	return sbRadius;
}

bool fk_Model::isInterference(fk_Model *argModel)
{
	fk_SphereBoundary		sb1, sb2;

	if(argModel == NULL) return false;
	sb1.setRadius(sbRadius);
	sb2.setRadius(argModel->sbRadius);
	sb1.setCurPos(getInhPosition());
	sb2.setCurPos(argModel->getInhPosition());

	return sb1.isInterference(sb2);
}

bool fk_Model::isCollision(fk_Model *argModel)
{
	fk_SphereBoundary		sb1, sb2;

	if(argModel == NULL) return false;
	if(snapFlag == false || argModel->snapFlag == false) return false;
	sb1.setRadius(sbRadius);
	sb2.setRadius(argModel->sbRadius);
	sb1.setCurPos(snapInhPos);
	sb2.setCurPos(argModel->snapInhPos);
	sb1.setNewPos(getInhPosition());
	sb2.setNewPos(argModel->getInhPosition());

	return sb1.isCollision(sb2);
}

double fk_Model::getCollisionTime(fk_Model *argModel)
{
	fk_SphereBoundary		sb1, sb2;

	if(argModel == NULL) return 0.0;
	if(snapFlag == false || argModel->snapFlag == false) return 0.0;
	sb1.setRadius(sbRadius);
	sb2.setRadius(argModel->sbRadius);
	sb1.setCurPos(snapInhPos);
	sb2.setCurPos(argModel->snapInhPos);
	sb1.setNewPos(getInhPosition());
	sb2.setNewPos(argModel->getInhPosition());

	return sb1.getCollisionTime(sb2);
}
