#include "stdafx.h"

#include "../include/FWatchCore2Obj.hpp"

#include "FileInfoCollectionImpl.hpp"
#include "ObjectLockUty.hpp"

using namespace FWatchCore2Lib;

//////////////////////////////////////////////////

HRESULT CFileInfoEnumerator::FinalConstruct() throw()
{
	pos_ = NULL;
	pParent_ = NULL;
	version_ = 0;

	return S_OK;
}

HRESULT __stdcall CFileInfoEnumerator::Reset() throw()
{
	if ( !pParent_) {
		ATLASSERT(false);
		return E_FAIL;
	}

	ThreadGuard<CFileInfoCollection> lock(pParent_);
	ThreadGuard<CFileInfoEnumerator> lock2(this);

	HRESULT hr = CheckVersion();
	if (FAILED(hr)) {
		return hr;
	}

	pos_ = pParent_->fileInfoMap_.GetStartPosition();

	return S_OK;
}

HRESULT __stdcall CFileInfoEnumerator::HasNext() throw()
{
	if ( !pParent_) {
		ATLASSERT(false);
		return E_FAIL;
	}

	ThreadGuard<CFileInfoCollection> lock(pParent_);
	ThreadGuard<CFileInfoEnumerator> lock2(this);

	HRESULT hr = CheckVersion();
	if (FAILED(hr)) {
		return hr;
	}

	return pos_ != NULL ? S_OK : S_FALSE;
}

HRESULT __stdcall CFileInfoEnumerator::Next(BSTR* v_pName, IFileInfo** v_ppFileInfo) throw()
{
	if ( !pParent_) {
		ATLASSERT(false);
		return E_FAIL;
	}

	ThreadGuard<CFileInfoCollection> lock(pParent_);
	ThreadGuard<CFileInfoEnumerator> lock2(this);

	if (v_ppFileInfo) {
		ATLASSERT(!*v_ppFileInfo);
		*v_ppFileInfo = NULL;
	}

	HRESULT hr = CheckVersion();
	if (FAILED(hr)) {
		return hr;
	}

	if (pos_ == NULL) {
		ATLASSERT(false);
		return E_FAIL;
	}

	FileInfoMap::CPair* pResult = pParent_->fileInfoMap_.GetNext(pos_);
	ATLASSERT(pResult);

	if (v_pName) {
		*v_pName = SysAllocString(pResult->m_key);
	}

	if (v_ppFileInfo) {
		IFileInfoPtr pFileInfo(pResult->m_value);
		ATLASSERT(pFileInfo);
		return pFileInfo.QueryInterface(v_ppFileInfo);
	}

	return S_OK;
}

HRESULT CFileInfoEnumerator::Init(CFileInfoCollection* v_pParent) throw()
{
	if ( !v_pParent) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ThreadGuard<CFileInfoCollection> lock(v_pParent);
	ThreadGuard<CFileInfoEnumerator> lock2(this);

	pParent_ = v_pParent;
	pos_ = pParent_->fileInfoMap_.GetStartPosition();
	version_ = pParent_->version_;

	return S_OK;
}

HRESULT CFileInfoEnumerator::CheckVersion() const throw()
{
	if ( !pParent_) {
		ATLASSERT(false);
		return E_FAIL;
	}

	if (version_ != pParent_->version_) {
		return E_ABORT;
	}

	return S_OK;
}

OBJECT_ENTRY_NON_CREATEABLE_EX_AUTO(__uuidof(CFileInfoEnumerator), CFileInfoEnumerator)

///////////////////////////////////////////////////////////

/// PRIVATE

HRESULT CFileInfoCollection::FinalConstruct() throw()
{
	version_ = 0;
	lastSavedVersion_ = 0;

	return S_OK;
}

/// IFileInfoCollection

HRESULT __stdcall CFileInfoCollection::get_Length(ULONG* v_pLength) throw()
{
	if ( !v_pLength) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ThreadGuard<CFileInfoCollection> lock(this);

	*v_pLength = static_cast<ULONG>(fileInfoMap_.GetCount());

	return S_OK;
}

HRESULT __stdcall CFileInfoCollection::new_Enum(IFileInfoEnumerator** v_ppFileInfoEnumerator) throw()
{
	if ( !v_ppFileInfoEnumerator) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ThreadGuard<CFileInfoCollection> lock(this);

	ATLASSERT(!*v_ppFileInfoEnumerator);
	*v_ppFileInfoEnumerator = NULL;

	HRESULT hr;
	CComObject<CFileInfoEnumerator>* pFileInfoEnumerator = NULL;
	hr = CComObject<CFileInfoEnumerator>::CreateInstance(&pFileInfoEnumerator);
	if (FAILED(hr)) {
		return hr;
	}
	ATLASSERT(pFileInfoEnumerator);
	pFileInfoEnumerator->AddRef();
	
	hr = pFileInfoEnumerator->Init(this);
	if (FAILED(hr)) {
		pFileInfoEnumerator->Release();
		return hr;
	}

	*v_ppFileInfoEnumerator = pFileInfoEnumerator;

	return S_OK;
}

HRESULT __stdcall CFileInfoCollection::get_Item(LPCWSTR v_name, IFileInfo** v_ppFileInfo) throw()
{
	if ( !v_name) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ThreadGuard<CFileInfoCollection> lock(this);

	FileInfoMap::CPair* pResult = fileInfoMap_.Lookup(v_name);
	if ( !pResult) {
		if (v_ppFileInfo) {
			ATLASSERT(!*v_ppFileInfo);
			*v_ppFileInfo = NULL;
		}
		return S_FALSE;
	}

	if (v_ppFileInfo) {
		IFileInfoPtr pFileInfo(pResult->m_value);
		ATLASSERT(pFileInfo);
		return pFileInfo.QueryInterface(v_ppFileInfo);
	}

	return S_OK;
}

HRESULT __stdcall CFileInfoCollection::set_Item(LPCWSTR v_pName, IFileInfo* v_pFileInfo) throw()
{
	if ( !v_pName || !v_pFileInfo) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ThreadGuard<CFileInfoCollection> lock(this);

	try {
		version_ += 1;
		fileInfoMap_.SetAt(v_pName, v_pFileInfo);
	}
	catch (...) {
		return E_OUTOFMEMORY;
	}

	return S_OK;
}

HRESULT __stdcall CFileInfoCollection::remove_Item(LPCWSTR v_pName) throw()
{
	if ( !v_pName) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ThreadGuard<CFileInfoCollection> lock(this);
	
	bool result = fileInfoMap_.RemoveKey(v_pName);

	return result ? S_OK : S_FALSE;
}

HRESULT __stdcall CFileInfoCollection::Clear() throw()
{
	ThreadGuard<CFileInfoCollection> lock(this);

	fileInfoMap_.RemoveAll();
	lastSavedVersion_ = version_;

	return S_OK;
}

HRESULT __stdcall CFileInfoCollection::Clone(IFileInfoCollection** v_ppClone) throw()
{
	if ( !v_ppClone) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr;

	CComObject<CFileInfoCollection>* pClone = NULL;
	hr = CComObject<CFileInfoCollection>::CreateInstance(&pClone);
	if (FAILED(hr)) {
		return hr;
	}
	ATLASSERT(pClone);
	CComPtr<IFileInfoCollection> pFileInfoCollection(pClone);

	ThreadGuard<CFileInfoCollection> lock(this);

	ATLASSERT(!*v_ppClone);
	*v_ppClone = NULL;

	POSITION pos = fileInfoMap_.GetStartPosition();
	while (pos) {
		FileInfoMap::CPair* pResult = fileInfoMap_.GetNext(pos);
		ATLASSERT(pResult);

		CComBSTR name(pResult->m_key);
		IFileInfoPtr pFileInfo(pResult->m_value);

		hr = pClone->set_Item(name, pFileInfo);
		if (FAILED(hr)) {
			return hr;
		}
	}

	ATLASSERT(pFileInfoCollection);
	return pFileInfoCollection.QueryInterface(v_ppClone);
}

/// IFileInfoReceiver

HRESULT __stdcall CFileInfoCollection::ReceiveFileInfo(LPCWSTR v_pPath, IFileInfo* v_pFileInfo) throw()
{
	return set_Item(v_pPath, v_pFileInfo);
}

HRESULT __stdcall CFileInfoCollection::Compare(IFileInfoCollection* v_pFileInfoCollection, IFileInfoComparator* v_pComparator) throw()
{
	if ( !v_pFileInfoCollection || !v_pComparator) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ThreadGuard<CFileInfoCollection> lock(this);

	HRESULT hr;

	// r

	ULONG length1;
	ULONG length2;

	length1 = (ULONG) fileInfoMap_.GetCount();
	hr = v_pFileInfoCollection->get_Length(&length2);
	if (FAILED(hr)) {
		return hr;
	}

	if (length1 != length2) {
		return S_FALSE;
	}

	// o̓erB

	POSITION pos = fileInfoMap_.GetStartPosition();
	while (pos) {
		const FileInfoMap::CPair* pItem = fileInfoMap_.GetNext(pos);
		ATLASSERT(pItem);

		const CComBSTR& name(pItem->m_key);
		const CComPtr<IFileInfo>& pFileInfo1(pItem->m_value);

		CComPtr<IFileInfo> pFileInfo2;
		hr = v_pFileInfoCollection->get_Item(name, &pFileInfo2);
		if (hr != S_OK) {
			// o^ĂȂ(S_FALSE)AsĂB
			return hr;
		}

		hr = v_pComparator->IsEquals(pFileInfo1, pFileInfo2);
		if (hr != S_OK) {
			// sv(S_FALSE)AsĂB
			return hr;
		}
	}

	return S_OK;
}

/// IPersist

HRESULT __stdcall CFileInfoCollection::GetClassID(CLSID *pClassID) throw()
{
	if ( !pClassID) {
		return E_INVALIDARG;
	}

	*pClassID = GetObjectCLSID();

	return S_OK;
}

/// IPersistStream

HRESULT __stdcall CFileInfoCollection::IsDirty() throw()
{
	ThreadGuard<CFileInfoCollection> lock(this);

	return (lastSavedVersion_ != version_) ? S_OK : S_FALSE;
}

HRESULT __stdcall CFileInfoCollection::Load(LPSTREAM v_pStm) throw()
{
	if ( !v_pStm) {
		return E_POINTER;
	}

	ThreadGuard<CFileInfoCollection> lock(this);
	
	Clear();

	HRESULT hr;

	// CLSID̓ǂݍ݂ƌ

	CLSID clsid;
	hr = ReadClassStm(v_pStm, &clsid);
	if (FAILED(hr)) {
		return hr;
	}

	if ( !IsEqualCLSID(clsid, GetObjectCLSID())) {
		return E_FAIL;
	}

	// ̎擾
	size_t count;
	hr = v_pStm->Read(&count, sizeof(size_t), NULL);
	if (FAILED(hr)) {
		return hr;
	}

	// vf̎擾
	for (size_t idx = 0; idx < count; idx++) {
		// pX̒̓ǂݍ
		unsigned int nameLen;
		hr = v_pStm->Read(&nameLen, sizeof(unsigned int), NULL);
		if (FAILED(hr)) {
			return hr;
		}

		// pX̓ǂݍ
		CComBSTR name(nameLen, (LPCWSTR) NULL);
		hr = v_pStm->Read(name, nameLen*2, NULL);
		if (FAILED(hr)) {
			return hr;
		}

		// FILETIME̓ǂݍ
		FILETIME lastModified;
		hr = v_pStm->Read(&lastModified, sizeof(FILETIME), NULL);
		if (FAILED(hr)) {
			return hr;
		}

		// t@CTCY̓ǂݍ
		ULARGE_INTEGER fileSize;
		hr = v_pStm->Read(&fileSize, sizeof(ULARGE_INTEGER), NULL);
		if (FAILED(hr)) {
			return hr;
		}

		// t@C̓ǂݍ
		DWORD attr;
		hr = v_pStm->Read(&attr, sizeof(DWORD), NULL);
		if (FAILED(hr)) {
			return hr;
		}
	}

	return S_OK;
}

HRESULT __stdcall CFileInfoCollection::Save(LPSTREAM v_pStm, BOOL v_fClearDirty) throw()
{
	if ( !v_pStm) {
		return E_POINTER;
	}

	ThreadGuard<CFileInfoCollection> lock(this);

	HRESULT hr;
	
	// CLSID̏
	hr = WriteClassStm(v_pStm, GetObjectCLSID());
	if (FAILED(hr)) {
		return hr;
	}

	// ̏
	const size_t count = fileInfoMap_.GetCount();
	hr = v_pStm->Write(&count, sizeof(size_t), NULL);
	if (FAILED(hr)) {
		return hr;
	}

	// ACȅ
	POSITION pos = fileInfoMap_.GetStartPosition();
	while (pos) {
		FileInfoMap::CPair* result = fileInfoMap_.GetNext(pos);
		const CComBSTR& name(result->m_key);
		const IFileInfoPtr& pFileInfo(result->m_value);

		// pX̒̏
		const unsigned int nameLen = name.Length();
		hr = v_pStm->Write(&nameLen, sizeof(unsigned int), NULL);
		if (FAILED(hr)) {
			return hr;
		}

		// pX̏
		const LPCWSTR pName = (LPCWSTR) name;
		hr = v_pStm->Write(pName, nameLen * 2, NULL);
		if (FAILED(hr)) {
			return hr;
		}

		// FILETIMȄ
		FILETIME lastModified;
		hr = pFileInfo->get_LastModified(&lastModified);
		if (FAILED(hr)) {
			return hr;
		}

		hr = v_pStm->Write(&lastModified, sizeof(FILETIME), NULL);
		if (FAILED(hr)) {
			return hr;
		}

		// t@CTCY̏
		ULARGE_INTEGER fileSize;
		hr = pFileInfo->get_FileSize(&fileSize);
		if (FAILED(hr)) {
			return hr;
		}

		hr = v_pStm->Write(&fileSize, sizeof(ULARGE_INTEGER), NULL);
		if (FAILED(hr)) {
			return hr;
		}

		// t@C̑̏
		DWORD attr;
		hr = pFileInfo->get_Attributes(&attr);
		if (FAILED(hr)) {
			return hr;
		}

		hr = v_pStm->Write(&attr, sizeof(DWORD), NULL);
		if (FAILED(hr)) {
			return hr;
		}
	}

	if (v_fClearDirty) {
		lastSavedVersion_ = version_;
	}

	return S_OK;
}

HRESULT __stdcall CFileInfoCollection::GetSizeMax(ULARGE_INTEGER* v_pcbSize) throw()
{
	if ( !v_pcbSize) {
		return E_POINTER;
	}

	ThreadGuard<CFileInfoCollection> lock(this);

	ULONGLONG siz = 0LL;

	siz += sizeof(CLSID);  // CLSID
	siz += sizeof(size_t); // ACě

	POSITION pos = fileInfoMap_.GetStartPosition();
	while (pos) {
		FileInfoMap::CPair* result = fileInfoMap_.GetNext(pos);
		const CComBSTR& name(result->m_key);
		const IFileInfoPtr& pFileInfo(result->m_value);
		
		siz += sizeof(unsigned int);   // pX̒
		siz += name.Length() * sizeof(wchar_t); // ̌ * wchar_t̃TCY

		siz += sizeof(FILETIME);       // FILE_INFÕt@C^C
		siz += sizeof(ULARGE_INTEGER); // FILE_INFÕt@CTCY
		siz += sizeof(DWORD);          // FILE_INFȎ
	}

	v_pcbSize->QuadPart = siz;

	return S_OK;
}

/// IPersistStreamInit

HRESULT __stdcall CFileInfoCollection::InitNew() throw()
{
	Clear();

	return S_OK;
}

	
OBJECT_ENTRY_AUTO(__uuidof(CFileInfoCollection), CFileInfoCollection)

////////////////////////////////////////////////////

extern "C" HRESULT __stdcall CreateFileInfoCollectionObject(IFileInfoCollection** v_ppFileInfoMap) throw()
{
	if ( !v_ppFileInfoMap) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ATLASSERT(!*v_ppFileInfoMap);
	*v_ppFileInfoMap = NULL;

	HRESULT hr;

	CComObject<CFileInfoCollection>* pCollectionImpl = NULL;
	hr = CComObject<CFileInfoCollection>::CreateInstance(&pCollectionImpl);
	if (FAILED(hr)) {
		return hr;
	}
	ATLASSERT(pCollectionImpl);
	CComPtr<IFileInfoCollection> pCollection(pCollectionImpl);

	return pCollection.QueryInterface(v_ppFileInfoMap);
}
