#!/usr/bin/perl
use strict;
use IO::Handle;
use IO::Select;

############################################################
#
# Grammar Manager (GRM)
# by Takuya Nishimoto, Galatea Project
# based on SRM_Main.pl of Galatea SRM
#
# Example:
#
#  $ ./GRM_Main.pl
#  set ConvertGrammar = ../phoenix/grm/renraku
#  tell 200 grammar conversion complete.
#
#  set ConvertGrammar = hogehoge (invalid file)
#  tell 100 error xxxx
#  tell 100 error in running XSLT processor: org.apache.xalan.xslt.Process
#  tell 400 grammar conversion failed.
#
############################################################
#
# Speech Recognition Main Module (engine-independent part)
#
# by speech recognition group, IPA agent project
#
#  ver.0.0  (2001/06/25) initial version (ri)
#  Ver.0.1  (2001/06/??) (internal) initial interaction with julian
#  Ver.0.2  (2001/07/04) (internal) alpha release
#  Ver.0.3  (2001/07/09) initial public release (ri)
#  Ver.0.4  (2002/01/29) (internal) support for switching grammars
#  Ver.0.5  (2002/02/20) config file support (ri)
#
# developed on perl version 5.005_03 built for i386-linux
#
# based on dummy module of Agent Manager (2001/06/04)
# below is the original header:
############################################################
#
#	dummy module for IPA	2001.06.04
#	code by matsusita yoshinori,	matsuy@jaist.ac.jp
#	checked on perl version 5.004_04 built for sun4-solaris
#
#	usage: perl dummy2.pl [initfile]
#	inq Run
#	rep LIVE
#
############################################################

# read in config variables
require "GRM_Config.pl";

use vars qw ( %CONF );

# read in engine-dependent module script
#require $CONF{"ENGINE_MODULE"};
require "GRM_xml2julgram.pl";
require "GRM_Julian.pl";

## no buffer ###############################################
select(STDIN);$|=1;
select(STDOUT);$|=1;

## grobal valiables ########################################
my( $debug ) = $CONF{"DEBUG_LEVEL"};
my( $initfile ) = $CONF{"GRM_INIT"};
my( %MACRO );	# macro
my( %COUNT );	# counter for various action

use vars qw ( %SLOT %PROPERTY ); # SLOT and PROPERTY values are holded in Sub Module

## main ####################################################
main();
exit 0;

sub main{
    init();     # initialize
    process_input();
}

## process input from stdin and subprocess socket #########
sub process_input {
    my ($nfound, $rin, $rout, $win, $wout, $ein, $eout);
    while(1) {
        # watch stdin and sub-process socket
        $rin = $win = "";
        vec($rin, fileno(STDIN), 1) = 1;
#        if (&SubCall_OutputAlive) { # sub process is running
#            vec($rin, fileno(&SubCall_OutputHandler), 1) = 1;
#        }
        $ein = $rin;
        $nfound = select($rout=$rin, $wout=$win, $eout=$ein, undef);
        if (vec($rout, fileno(STDIN), 1) == 1) { # STDIN input
            $_ = <STDIN>;
            analyzetoken($_);   # usual process
        } else {            # process output
#            &SubCall_ProcessOutput;
        }
    }
}

## initialize process using macro ##########################
sub init{
	print_debbug("initializing...\n");
	$initfile = $ARGV[0] if $ARGV[0];
	analyzetoken("def init < $initfile\ndo init\n");	# initialize by initfile
	foreach(sort keys %SLOT){
		print_debbug("$_ = $SLOT{$_}\n");	# list initialized slot & value for debuging
	}
	delete($MACRO{"init"});	# delete 'init' macro from %MACRO slot
	print_debbug("initialize completed.\n");
}

## analize command #########################################
sub analyzetoken{
	my($input) = @_;
	my( $command, $slot, $method, $alias);	# 'set/inq/def/do/...', targetslot, '=/</<<', 'save/rest'
		
	foreach(split("\n", $input)){
		s;#.*$;; ;	# delete comment
		next if m;^\s*$; ;	# ignore empty line

		if( s;\b(set|def|prop)\s+(\S+)\s+(=|\<\<|\<);;	){	# for type 'command slot method ...' instanse: set/def/prop
			$command = $1;
			$slot = $2;
			$method = $3;
			$_;
			cmd_method($command, $slot, $method, $_);
		}
		elsif( s;\b(save|rest)\s+(\S+)\s+(\S+);; ){	# for type 'command slot alias' instance: save/rest
			$command = $1;
			$slot = $2;
			$alias = $3;
			cmd_alias($command, $slot, $alias);
		}
		elsif( s;\b(inq|do|del)\s+(\S+);; ){	# for type 'command slot' instanse: inq/do/del
			$command = $1;
			$slot = $2;
			cmd_slot($command, $slot);
		}
		else{ print_tell("400 '$_' no match\n"); }
	}
}

## customized output #######################################

sub print_rep{ print("rep @_"); STDOUT->flush(); $COUNT{"rep"}++;}	# rep output
sub print_tell{ print("tell @_"); STDOUT->flush(); $COUNT{"tell"}++;}	# tell output

sub print_debbug{
	print( STDERR "debug:@_") if $debug ;	# debug out
}

## check slot ##############################################

sub check_slot{
	my($slotname, $type) = @_;
	if( exists $$type{$slotname} ){ return 1;}	# check $slotname existance
	else{	# if nomatch
		print_tell("400 invalid slot $slotname\n");
		return 0;
	}
	$COUNT{"check_slot"}++;
}

## command type 'set Run = INIT' ###########################
sub cmd_method{
	my($command, $slot, $method, $tail) = @_;
	my($value, $oldvalue);	# values to set
	my($ret);       # return value (by ri) 2001/06/25
	
	if($method eq "="){	# single line input
		$tail =~ s;^\s+(.*)\s*$;; ;
		$value = $1;
	}
	elsif($method eq "<"){	# file input 'set Run < conf.file'
		my($infile);
		$tail =~ s;^\s*(.*)\s*$;; ;
		$infile = $1;
		open(INFILE, $infile) || (print_tell("400 can not open $infile\n") && return );
		$value = join('',<INFILE>);	# read content of $infile
		close(INFILE);
	}
	elsif($method eq "<<"){	# multiline input 'set Run << END ...'
		my($endmarker);	# endmarker
		$tail =~ s;^\s*(\S+)\s*$;; ;
		$endmarker = $1;	# set endmarker
		while(<STDIN>){
			last if m;$endmarker; ;	# read until endmarker
			$value .= $_;	# set value
		}
	}
	else{ die; }	# illigal

	#### begin: call sub module (by ri) 2001/06/25
	if($command eq "set"){
	    $oldvalue = $SLOT{$slot};
	    $SLOT{$slot} = $value;
	    $ret = &SubCall_Set($slot, $value);# $SLOT{$slot} may change here
	    if ($ret == 0) {	# error
		$SLOT{$slot} = $oldvalue;
		return 0;
	    }
	    $COUNT{$command}++;
	}
	elsif($command eq "def"){ 
	    $MACRO{$slot} = $value;
	    $COUNT{$command}++;
	}
	elsif($command eq "prop"){
	    $PROPERTY{$slot} = $value;
	    $COUNT{$command}++;
	}
	else{ die ; }	# illigal

	print_debbug("slot $slot, value $SLOT{$slot}\n");
	#### end: call sub module (by ri) 2001/06/25

}

## command type 3 args #####################################
sub cmd_alias{
	my($command, $slot, $alias) = @_;
	my($oldvalue);		# current value
	my($ret);       # return value (by ri) 2001/06/25
	return unless check_slot($slot, \%SLOT);
	if($command eq "save"){ $SLOT{$alias} = $SLOT{$slot} }	# copy value
	elsif($command eq "rest"){
	    #### begin: check alias existence (by ri) 2001/06/25
	    return unless check_slot($alias, \%SLOT);
	    #### end: check alias existence (by ri) 2001/06/25
	    #### begin: call sub module (by ri) 2001/06/25
	    $oldvalue = $SLOT{$slot};
	    $SLOT{$slot} = $SLOT{$alias}; # copy value
	    $ret = &SubCall_Set($slot, $SLOT{$alias}); # slot value may change
	    if ($ret == 0) {	# error
		$SLOT{$slot} = $oldvalue;
		return 0;
	    }
	    #### end: call sub module (by ri) 2001/06/25
	}
	else{ die; }	# illigal
	$COUNT{$command}++;
}

## command type 2 args #####################################
sub cmd_slot{
	my($command, $slot) = @_;
	my($ret);       # return value (by ri) 2001/06/25
	
	if($command eq "inq"){
		$COUNT{$command}++;
		#### begin: call sub module (by ri) 2001/06/25
		$ret = SubCall_Inq($slot);
		if ($ret eq "") {
		    return 0;
		}
		print_rep("$slot = $ret\n"); }
		#### end: call sub module (by ri) 2001/06/25
	elsif($command eq "do" ){
		return unless check_slot($slot, \%MACRO); # check slot existance
		$COUNT{$command}++;
		analyzetoken($MACRO{$slot});
	}
	elsif($command eq "del" ){
		return unless check_slot($slot, \%SLOT); # check slot existance
		$COUNT{$command}++;
		delete($SLOT{$slot}) || print_tell("400 $slot is not deleted!\n");
	}
	else{ die; }	# illigal
}

## end of script ###########################################
