
/*	
	GameboyAdvanceVM 
		- Nintendo GameboyAdvance Emulator
	Copyright 2002 Y_N y_n@users.sourceforge.jp
	Homepage https://sourceforge.jp/projects/gbaemu/
*/


#define	DREG_T		(u8)(opcode&0x7)		/*Destination register*/
#define	SREG_T		(u8)((opcode>>3)&0x7)	/*Source register*/
#define	OREG_T		(u8)((opcode>>6)&0x7)

#define	OPCODE_T		agb_read_mem16(pc)


void CDisassemblerDlg::disasm_thumb_adc()
{	/*4*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X adc r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_add()
{	/*Z2*/
	u16	opcode, offset;
	u8	rd, rs;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= (opcode&0x7)>>3;
	rd		= DREG_T;
	rs		= SREG_T;

	if(opcode & BIT_10_){	/*1 = I = offset*/
		sprintf(str_offset, "#0x%X", offset);
	}else{					/*0 = I = register*/
		sprintf(str_offset, "r%d", offset);
	}

	sprintf(mne, "%08X %04X add r%d, r%d, %s", pc, opcode, rd, rs, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_add_imm()
{	/*ADD Rd,#Offset8 Z3*/
	u16	opcode, offset;
	u8	rd;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;	/*offset8*/
	rd		= (opcode&0x7)>>8;

	sprintf(mne, "%08X %04X add r%d, #0x%X", pc, opcode, rd, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_add_adr()
{	/*12*/
	u16	opcode, offset;
	u8	rd;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;	/*offset8*/
	rd		= (opcode&0x7)>>8;
	
	sprintf(mne, "%08X %04X add r%d, %s, #0x%X", 
		pc, opcode, rd, ((opcode &BIT_10_)?"sp":"pc"), offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_add_sp()
{	/*X^bN|C^ɃItZbgZ - 13*/
	u16	opcode, offset;
	u8	rd;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= opcode&0x7F;	/*offset7*/
	rd		= (opcode&0x7)>>8;
	
	sprintf(mne, "%08X %04X add r%d, #%s0x%X", 
		pc, opcode, rd, ((opcode&BIT_7_)?"-":""), offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_add_hi()
{	/*HiWX^ - 5*/
	u16	opcode;
	u8	rd, rs, hd, hs, h1, h2;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= SREG_T;
	hd		= rd;
	hs		= rs;

	if(opcode & BIT_7_){
		if(opcode & BIT_6_){	/*ADD Hd,Hs*/
			h1 = hd;
			h2 = hs;
		}else{					/*ADD Hd,Rs*/
			h1 = hd;
			h2 = rs;
		}
	}else{
		if(opcode & BIT_6_){	/*ADD Rd,Hs*/
			h1 = rd;
			h2 = hs;
		}else{
			h1 = 0;/*`H*/
			h2 = 0;
		}
	}

	sprintf(mne, "%08X %04X add r%d, r%d", pc, opcode, h1, h2);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_and()
{	/*4*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X and r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_asr()
{	/*4*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X asr r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_asr_imm()
{	/*1*/
	u16	opcode, offset;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	sprintf(mne, "%08X %04X asr r%d, r%d, #0x0%X", pc, opcode, rd, rs, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_b()
{	/* - 18*/
	u16	opcode, offset;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= opcode&0x7FF;

	sprintf(mne, "%08X %04X b #0x%X", pc, opcode, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_bxx()
{	/* - 16*/
	u16	opcode, offset;
	u8	cond;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;
	cond	= (opcode&0xF)>>8;

	sprintf(mne, "%08X %04X b%s #0x%X", pc, opcode, cond_table[cond], offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_bl()
{	/*ƃN - 19*/
	u16	opcode, offset;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= opcode&0x7FF;

	sprintf(mne, "%08X %04X bl #0x%X", pc, opcode, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_bic()
{	/*4*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X asr r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_bx()
{	/**/
	u16	opcode;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;

	sprintf(mne, "%08X %04X bx", pc, opcode);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_bx_hi()
{	/*Xe[gύX - 5*/
	u16	opcode;
	u8	rs, hs, h2;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rs		= SREG_T;
	hs		= rs;

	if(opcode & BIT_7_){	/*H1*/
		if(opcode & BIT_6_){	/*BX Hs*/
			h2 = hs;
		}else{					/*BX Rs*/
			h2 = rs;
		}
	}else{
		h2 = 0;/*`H*/
	}

	sprintf(mne, "%08X %04X bx r%d, r%d", pc, opcode, h2);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_cmn()
{	/*CMN Rd,Rs r4*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X cmn r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_cmp()
{	/*CMP Rd,Rs r - 4*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X asr r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_cmp_imm()
{	/*CMP Rd,#Offset8 r - 3*/
	u16	opcode, offset;
	u8	rd;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;	/*offset8*/
	rd		= (opcode&0x7)>>8;

	sprintf(mne, "%08X %04X cmp r%d, #0x%X", pc, opcode, rd, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_cmp_hi()
{	/*HiWX^ - 5*/
	u16	opcode;
	u8	rd, rs, hd, hs, h1, h2;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= SREG_T;
	hd		= rd;
	hs		= rs;

	if(opcode & BIT_7_){	/*H1*/
		if(opcode & BIT_6_){	/*CMP Hd,Hs*/
			h1 = hd;
			h2 = hs;
		}else{					/*CMP Hd,Rs*/
			h1 = hd;
			h2 = rs;
		}
	}else{
		if(opcode & BIT_6_){	/*CMP Rd,Hs*/
			h1 = rd;
			h2 = hs;
		}else{
			h1 = 0;/*`H*/
			h2 = 0;
		}
	}

	sprintf(mne, "%08X %04X cmp r%d, r%d", pc, opcode, h1, h2);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_eor()
{	/*4*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X eor r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_ldmia()
{	/*WX^̃[h^XgA - 15*/
	u32	bit, lst;
	u16	opcode, rlist;
	u8	rb;	
	char mne[STR_BUF_SIZE], str_lst[64], str_reg[16];

	opcode	= OPCODE_T;
	rlist	= opcode&0xFF;	/*Rlist8*/
	rb		= (opcode&0x07)>>8;

	strcpy(str_lst, "");
	for(bit=BIT_8_,lst=0; bit!=BIT_1_; bit>>=1){
		sprintf(str_reg, "r%d,", lst);
		strcat(str_lst, str_reg);
	}

	sprintf(mne, "%08X %04X ldmia r%d!, {%s}", pc, opcode, rb, str_lst);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_ldr()
{	/*WX^ItZbgɂ郍[h - 7*/
	u16	opcode;
	u8	rd, rb, ro;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

	sprintf(mne, "%08X %04X ldr r%d, [r%d, r%d]", pc, opcode, rd, rb, ro);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_ldr_imm()
{	/*C~fBGCgItZbgɂ郍[h^XgA - 9*/
	u16	opcode, offset;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	sprintf(mne, "%08X %04X ldr r%d, [r%d, #0x%X]", pc, opcode, rd, rs, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_ldrb()
{	/*WX^ItZbgɂ郍[h - 7*/
	u16	opcode;
	u8	rd, rb, ro;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

	sprintf(mne, "%08X %04X ldr r%d, [r%d, r%d]", pc, opcode, rd, rb, ro);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_ldrb_imm()
{	/*C~fBGCgItZbgɂ郍[h^XgA - 9*/
	u16	opcode, offset;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	sprintf(mne, "%08X %04X ldrb r%d, [r%d, #0x%X]", pc, opcode, rd, rs, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_ldrh()
{	/*oCg^n[t[h̃[h^XgAƕg - 8*/
	u16	opcode;
	u8	rd, rb, ro;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

	sprintf(mne, "%08X %04X ldrh r%d, [r%d, r%d]", pc, opcode, rd, rb, ro);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_ldrh_imm()
{	/**/
	u16	opcode, offset;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	sprintf(mne, "%08X %04X ldrh r%d, [r%d, #0x%X]", pc, opcode, rd, rs, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_ldr_pc()
{	/*PC΃[h - 6*/
	u16	opcode, offset;
	u8	rd;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;
	rd		= (opcode&0x7)>>8;

	sprintf(mne, "%08X %04X ldr r%d, [pc,#0x%X]", pc, opcode, rd, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_ldr_sp()
{	/**/
	u16	opcode, offset;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;

	sprintf(mne, "%08X %04X ldr [sp, #0x%X]", pc, opcode, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_ldsb()
{	/*oCg^n[t[h̃[h^XgAƕg - 8*/
	u16	opcode;
	u8	rd, rb, ro;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

	sprintf(mne, "%08X %04X ldsb r%d, [r%d, r%d]", pc, opcode, rd, rb, ro);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_ldsh()
{	/*oCg^n[t[h̃[h^XgAƕg - 8*/
	u16	opcode;
	u8	rd, rb, ro;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

	sprintf(mne, "%08X %04X ldsh r%d, [r%d, r%d]", pc, opcode, rd, rb, ro);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_lsl()
{	/*4*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X lsl r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_lsl_imm()
{	/*1*/
	u16	opcode, offset;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;	/*offset5*/
	rd		= DREG_T;
	rs		= SREG_T;

	sprintf(mne, "%08X %04X lsl r%d, r%d, #0x0%X", pc, opcode, rd, rs, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_lsr()
{	/*Rd := Rd >> Rs*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X lsr r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_lsr_imm()
{	/*1*/
	u16	opcode, offset;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	sprintf(mne, "%08X %04X lsr r%d, r%d, #0x0%X", pc, opcode, rd, rs, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_mov()
{	/*MOV Rd,#Offset8 ړ3*/
	u16	opcode, offset;
	u8	rd;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;	/*offset8*/
	rd		= (opcode&0x7)>>8;

	sprintf(mne, "%08X %04X mov r%d, #0x%X", pc, opcode, rd, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_mov_hi()
{	/*HiWX^ - 5*/
	u16	opcode;
	u8	rd, rs, hd, hs, h1, h2;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= SREG_T;
	hd		= rd;
	hs		= rs;

	if(opcode & BIT_7_){	/*H1*/
		if(opcode & BIT_6_){	/*MOV Hd,Hs*/
			h1 = hd;
			h2 = hs;
		}else{
			h1 = hd;			/*MOV Hd,Rs*/
			h2 = rs;
		}
	}else{
		if(opcode & BIT_6_){	/*MOV Rd,Hs*/
			h1 = rd;
			h2 = hs;
		}else{
			h1 = 0;/*`H*/
			h2 = 0;
		}
	}

	sprintf(mne, "%08X %04X movhi r%d, r%d", pc, opcode, h1, h2);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_mul()
{	/*4*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X mul r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_mvn()
{	/*4*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X mvn r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_neg()
{	/*4*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X neg r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_orr()
{	/*4*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X orr r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_pop()
{	/*WX^PUSH/POP - 14*/
	u32	bit, lst;
	u16	opcode, rlist;
	char mne[STR_BUF_SIZE], str_lst[64], str_reg[16];

	opcode	= OPCODE_T;
	rlist	= opcode&0xFF;	/*Rlist8*/

	strcpy(str_lst, "");
	for(bit=BIT_8_,lst=8; bit!=BIT_1_; bit>>=1){
		sprintf(str_reg, "r%d,", lst);
		strcat(str_lst, str_reg);
	}
	if(opcode&BIT_8_)strcat(str_lst, "pc");

	sprintf(mne, "%08X %04X pop {%s}", pc, opcode, str_lst);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_push()
{	/*WX^PUSH/POP - 14*/
	u32	bit, lst;
	u16	opcode, rlist;
	char mne[STR_BUF_SIZE], str_lst[64], str_reg[16];

	opcode	= OPCODE_T;
	rlist	= opcode&0xFF;	/*Rlist8*/

	strcpy(str_lst, "");
	for(bit=BIT_1_,lst=0; bit!=BIT_8_; bit<<=1){
		sprintf(str_reg, "r%d,", lst);
		strcat(str_lst, str_reg);
	}
	if(opcode&BIT_8_)strcat(str_lst, "lr");

	sprintf(mne, "%08X %04X push {%s}", pc, opcode, str_lst);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_ror()
{	/*1*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X ror r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_sbc()
{	/*4*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X sbc r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_stmia()
{	/*WX^̃[h^XgA - 15*/
	u32	bit, lst;
	u16	opcode, rlist;
	u8	rb;	
	char mne[STR_BUF_SIZE], str_lst[64], str_reg[16];

	opcode	= OPCODE_T;
	rlist	= opcode&0xFF;	/*Rlist8*/
	rb		= (opcode&0x07)>>8;

	strcpy(str_lst, "");
	for(bit=BIT_1_,lst=0; bit!=BIT_8_; bit<<=1){
		sprintf(str_reg, "r%d,", lst);
		strcat(str_lst, str_reg);
	}

	sprintf(mne, "%08X %04X stmia r%d!, {%s}", pc, opcode, rb, str_lst);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_str()
{	/*WX^ItZbgɂXgA - 7*/
	u16	opcode;
	u8	rd, rb, ro;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

	sprintf(mne, "%08X %04X str r%d, [r%d, r%d]", pc, opcode, rd, rb, ro);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_str_imm()
{	/*C~fBGCgItZbgɂ郍[h^XgA - 9*/
	u16	opcode, offset;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	sprintf(mne, "%08X %04X str r%d, [r%d, #0x%X]", pc, opcode, rd, rs, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_strb()
{	/*WX^ItZbgɂXgA - 7*/
	u16	opcode;
	u8	rd, rb, ro;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

	sprintf(mne, "%08X %04X strb r%d, [r%d, r%d]", pc, opcode, rd, rb, ro);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_strb_imm()
{	/*C~fBGCgItZbgɂ郍[h^XgA - 9*/
	u16	opcode, offset;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	sprintf(mne, "%08X %04X strb r%d, [r%d, #0x%X]", pc, opcode, rd, rs, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_strh()
{	/*oCg^n[t[h̃[h^XgAƕg - 8*/
	u16	opcode;
	u8	rd, rb, ro;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

	sprintf(mne, "%08X %04X strh r%d, [r%d, r%d]", pc, opcode, rd, rb, ro);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_strh_imm()
{	/**/
	u16	opcode, offset;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	sprintf(mne, "%08X %04X strh r%d, [r%d, #0x%X]", pc, opcode, rd, rs, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_str_sp()
{	/*SP΃[h^XgA - 11*/
	u16	opcode, offset;
	u8	rd;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;
	rd		= (opcode&0x7)>>8;

	sprintf(mne, "%08X %04X str r%d, [sp, #0x%X]", pc, opcode, rd, opcode);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_swi()
{	/*\tgEFA荞 - 17*/
	u16	opcode, offset;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;

	sprintf(mne, "%08X %04X swi #0x%X", pc, opcode, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_sub()
{	/*Z2*/
	u16	opcode, offset;
	u8	rd, rs;
	char mne[STR_BUF_SIZE], str_offset[STR_OFFSET_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= (opcode&0x7)>>3;
	rd		= DREG_T;
	rs		= SREG_T;

	if(opcode & BIT_10_){	/*1 = I = offset*/
		sprintf(str_offset, "#0x%X", offset);
	}else{					/*0 = I = register*/
		sprintf(str_offset, "r%d", offset);
	}

	sprintf(mne, "%08X %04X sub r%d, r%d, %s", pc, opcode, rd, rs, str_offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_sub_imm()
{	/*SUB Rd,#Offset8 Z3*/
	u16	opcode, offset;
	u8	rd;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;	/*offset8*/
	rd		= (opcode&0x7)>>8;

	sprintf(mne, "%08X %04X sub r%d, #0x%X", pc, opcode, rd, offset);
	m_lstDisasm.AddString(mne);
}

void CDisassemblerDlg::disasm_thumb_tst()
{	/*4*/
	u16	opcode;
	u8	rd, rs;
	char mne[STR_BUF_SIZE];

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	sprintf(mne, "%08X %04X tst r%d, r%d", pc, opcode, rd, rs);
	m_lstDisasm.AddString(mne);
}

