
/*	
	GameboyAdvanceVM 
		- Nintendo GameboyAdvance Emulator
	Copyright 2002 Y_N y_n@users.sourceforge.jp
	Homepage https://sourceforge.jp/projects/gbaemu/
*/


#define	DREG_T		(u8)(opcode&0x7)		/*Destination register*/
#define	SREG_T		(u8)((opcode>>3)&0x7)	/*Source register*/
#define	OREG_T		(u8)((opcode>>6)&0x7)

#define	OPCODE_T	agb_read_mem16(arm.reg[15]);


_inline void thumb_adc()
{	/*ADC Rd,Rs(L[tZ) - 4*/
	u32	temp32, cflag;
	u16	opcode;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	cflag = LSB_FC;
	temp32 = arm.reg[rd] + arm.reg[rs] + cflag;

	SetAddFlag(arm.reg[rd], arm.reg[rs] + cflag, temp32);

	arm.reg[rd] = temp32;

	arm.cycle = 1;
}

_inline void thumb_add()
{	/*ADD Rd,Rs,(Rn || #Offset3) (Z) - 2*/
	u16	opcode, offset;
	u8	rd, rs;

	opcode	= OPCODE_T;
	offset	= (opcode & 0x7)>>3;
	rd		= DREG_T;
	rs		= SREG_T;

	if(opcode & BIT_10_){	/*1 = I = offset*/
		arm.reg[rd] = arm.reg[rs] + offset;
	}else{					/*0 = I = register*/
		arm.reg[rd] = arm.reg[rs] + offset;
	}

	SetAddFlag(arm.reg[rs], offset, arm.reg[rd]);

	arm.cycle = 1;
}

_inline void thumb_add_imm()
{	/*ADD Rd,#Offset8 (Z) - 3*/
	u32	temp32;
	u16	opcode, offset;
	u8	rd;

	opcode	= OPCODE_T;
	offset	= opcode & 0xFF;	/*offset8*/
	rd		= (opcode & 0x7)>>8;

	temp32 = arm.reg[rd] + offset;

	SetAddFlag(arm.reg[rd], offset, temp32);

	arm.reg[rd] = temp32;

	arm.cycle = 1;
}

_inline void thumb_add_adr()
{	/*ADD Rd,(PC || SP),#imm (Z) - 12*/
	u16	opcode, offset;
	u8	rd;

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;	/*offset8*/
	rd		= (opcode&0x7)>>8;

	if(opcode & BIT_11_){
		/*if((SP + offset)>0xFFFFFFFF)SF(C_); else RF(C_);*/
		arm.reg[rd] = SP + offset;
		SetAddFlag(SP, offset, arm.reg[rd]);
	}else{
		/*if((PC + offset)>0xFFFFFFFF)SF(C_); else RF(C_);*/
		arm.reg[rd] = PC + offset;
		SetAddFlag(PC, offset, arm.reg[rd]);
	}

	arm.cycle = 1;
}

_inline void thumb_add_sp()
{	/*ADD SP,#+-imm (X^bN|C^ɃItZbgZ) - 13*/
	u16	opcode, offset;

	opcode	= OPCODE_T;
	offset	= opcode&0x7F;	/*offset7*/
	
	if(opcode & BIT_7_){
		if((SP + offset) > 0xFFFFFFFF)SF(C_); else RF(C_);
		SP = SP + offset;
	}else{
		if((SP - offset) > 0)SF(C_); else RF(C_);
		SP = SP - offset;
	}

	if(!SP)SF(Z_); else RF(Z_);
	if(opcode & BIT_31_)SF(N_); else RF(N_);

	arm.cycle = 1;
}

_inline void thumb_add_hi()
{	/*ADD Rd,Hs (Z) - 5*/
	u16	opcode;
	u8	rd, rs, hd, hs, h1, h2;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= SREG_T;
	hd		= rd + 8;
	hs		= rs + 8;

	if(opcode & BIT_7_){
		if(opcode & BIT_6_){	/*ADD Hd,Hs*/
			h1 = hd;
			h2 = hs;
		}else{					/*ADD Hd,Rs*/
			h1 = hd;
			h2 = rs;
		}
	}else{
		if(opcode & BIT_6_){	/*ADD Rd,Hs*/
			h1 = rd;
			h2 = hs;
		}else{
			h1 = 0;/*`H*/
			h2 = 0;
		}
	}

	arm.reg[h1] = arm.reg[h1] + arm.reg[h2];

	if((arm.reg[h1] + arm.reg[h2]) > 0xFFFFFFFF)SF(C_); else RF(C_);
	if(!arm.reg[h1])SF(Z_); else RF(Z_);
	if(opcode & BIT_31_)SF(N_); else RF(N_);

	arm.cycle = 1;
}

_inline void thumb_and()
{	/*AND Rd,Rs (_) - 4*/
	u16	opcode;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	arm.reg[rd] = arm.reg[rd] & arm.reg[rs];

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void thumb_asr()
{	/*ASR Rd,Rs (ZpEVtg) - 4*/
	u32	bit, temp32, shift;
	u16	opcode;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	if(arm.reg[rd] & (1<<arm.reg[rs]))SF(C_); else RF(C_);

	//arm.reg[rd] = arm.reg[rd]>>arm.reg[rs];

	shift = arm.reg[rs];

	temp32 = (arm.reg[rd]>>shift);
	if(arm.reg[rd] & BIT_31_){
		for(bit=BIT_31_; shift; bit>>=1){
			temp32 |= bit;
			shift--;
		}
	}
	arm.reg[rd] = temp32;

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void thumb_asr_imm()
{	/*ASR Rd,Rs,#Offset5 (ZpEVtg) - 1*/
	u32	bit, temp32;
	u16	opcode, offset;
	u8	rd, rs;

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	if(arm.reg[rd] & (1<<offset))SF(C_); else RF(C_);

	//arm.reg[rd] = arm.reg[rs]>>offset;

	temp32 = (arm.reg[rs]>>offset);
	if(arm.reg[rs] & BIT_31_){
		for(bit=BIT_31_; offset; bit>>=1){
			temp32 |= bit;
			offset--;
		}
	}
	arm.reg[rd] = temp32;

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void thumb_b()
{	/*B label () - 18*/
	u16	opcode;
	s16	offset;

	opcode	= OPCODE_T;
	offset	= opcode&0x7FF;

	PC = PC + offset;

	arm.cycle = 3;
}

_inline void thumb_bxx()
{	/*BXX label () - 16*/
	BOOL not_jump;
	u16	opcode, offset;
	u8	cond;

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;
	cond	= (opcode&0xF)>>8;

	not_jump = TRUE;
	switch(cond){	/*s - Condition Field*/
	case 0x0:
		if(FZ)not_jump = FALSE;
		break;		/*if Z set		- equal*/
	case 0x1:
		if(!FZ)not_jump = FALSE;
		break;		/*if Z clear	- not equal*/
	case 0x2:
		if(FC)not_jump = FALSE;
		break;		/*if C set		- unsigned higher or same*/
	case 0x3:
		if(!FC)not_jump = FALSE;
		break;		/*if C clear	- unsigne lower*/
	case 0x4:
		if(FN)not_jump = FALSE;
		break;		/*if N set		- negative*/
	case 0x5:
		if(!FN)not_jump = FALSE;
		break;		/*if N clear	- positive or zero*/
	case 0x6:
		if(FV)not_jump = FALSE;
		break;		/*if V set		- overflow*/
	case 0x7:
		if(!FV)not_jump = FALSE;
		break;		/*if V clear	- no overflow*/
	case 0x8:
		if(FC && !FZ)not_jump = FALSE;
		break;		/*if C set and Z clear	- unsigned higher*/
	case 0x9:
		if(!FC || FZ)not_jump = FALSE;
		break;		/*if C clear and Z set	- unsigned lower or same*/
	case 0xA:
		if((FN && FV) || (!FN && !FV))not_jump = FALSE;
		break;		/*if ((N set and V set) or (N clear and V clear))	- greater or equal*/
	case 0xB:
		if((!FN && FV) || (FN && !FV))not_jump = FALSE;
		break;		/*if ((N set and V clear) or (N clear and V set))	- less than*/
	case 0xC:
		if((!FZ && (FN || FV)) || (!FN || !FV))not_jump = FALSE;
		break;		/*if (Z clear and(N or V set) or (N or V clear))- greater than*/
	case 0xD:
		if((FZ || (FN && !FV)) || (!FN && FV))not_jump = FALSE;
		break;		/*if (Z set or(N set and V clear) or (N clear and V set))- less than or equal*/
	default:
		not_jump = FALSE;
		break;		/*(ignored)	- always */
	}

	if(!not_jump)PC = PC + offset;

	arm.cycle = 3;
}

_inline void thumb_bl()
{	/*BL label (ƃN) - 19*/
	u32	temp;
	u16	opcode, offset;

	opcode	= OPCODE_T;
	offset	= opcode & 0x7FF;

	if(opcode & BIT_11_){	/*H=1*/
		temp = PC + 2;
		PC = LR + (offset<<1);
		LR = temp | 1;
	}else{					/*H=0*/
		LR = PC + (offset<<12);
	}

	arm.cycle = 3;
}

_inline void thumb_bic()
{	/*BIC Rd,Rs (rbgNA) - 4*/
	u16	opcode;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	arm.reg[rd] = arm.reg[rd] & ~arm.reg[rs];

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void thumb_bx()
{	/*Bx (Xe[gύX)*/
	u16	opcode;

	opcode	= OPCODE_T;

	if(opcode & BIT_1_){
		CPSR &= ~T_;	/*ARMXe[gɕύX*/
	}else{
		/*CPSR |= T_;*/	/*THUMBXe[ĝ܂*/
	}
}

_inline void thumb_bx_hi()
{	/*BX Hs(Xe[gύX) - 5*/
	u16	opcode;
	u8	rs, hs, h2;

	opcode	= OPCODE_T;
	rs		= SREG_T;
	hs		= rs + 8;

	if(opcode & BIT_7_){	/*H1*/
		if(opcode & BIT_6_){	/*BX Hs*/
			h2 = hs;
		}else{					/*BX Rs*/
			h2 = rs;
		}
	}else{
		h2 = 0;/*`H*/
	}

	if(arm.reg[h2] & BIT_1_){
		CPSR &= ~T_;	/*ARMXe[gɕύX*/
	}else{
		/*CPSR |= T_;*/	/*THUMBXe[ĝ܂*/
	}

	arm.cycle = 3;
}

_inline void thumb_cmn()
{	/*CMN Rd,Rs (r) - 4*/
	u32	temp;
	u16	opcode;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	temp = arm.reg[rd] + arm.reg[rs];
/*
	if((arm.reg[rd] + arm.reg[rs]) > 0xFFFFFFFF)SF(C_); else RF(C_);
	if(!temp)SF(Z_); else RF(Z_);
	if(opcode & BIT_31_)SF(N_); else RF(N_);
*/
	SetAddFlag(arm.reg[rd], arm.reg[rs], temp);

	arm.cycle = 1;
}

_inline void thumb_cmp()
{	/*CMP Rd,Rs (r) - 4*/
	u16	opcode, temp;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	temp = arm.reg[rd] - arm.reg[rs];	
/*
	if((arm.reg[rd] - arm.reg[rs]) > 0)SF(C_); else RF(C_);
	if(!temp)SF(Z_); else RF(Z_);
	if(opcode & BIT_31_)SF(N_); else RF(N_);
*/
	SetSubFlag(arm.reg[rd], arm.reg[rs], temp);

	arm.cycle = 1;
}

_inline void thumb_cmp_imm()
{	/*CMP Rd,#Offset8 (r) - 3*/
	u16	opcode, offset, temp;
	u8	rd;

	opcode	= OPCODE_T;
	offset	= opcode & 0xFF;	/*offset8*/
	rd		= (opcode & 0x7)>>8;

	temp = arm.reg[rd] - offset;
/*
	if((arm.reg[rd] - offset) > 0)SF(C_); else RF(C_);
	if(!temp)SF(Z_); else RF(Z_);
	if(opcode & BIT_31_)SF(N_); else RF(N_);
*/
	SetSubFlag(arm.reg[rd], offset, temp);

	arm.cycle = 1;
}

_inline void thumb_cmp_hi()
{	/*CMP Rd,Hs (r,HiWX^) - 5*/
	u16	opcode,	temp;
	u8	rd, rs, hd, hs, h1, h2;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= SREG_T;
	hd		= rd + 8;
	hs		= rs + 8;

	if(opcode & BIT_7_){	/*H1*/
		if(opcode & BIT_6_){	/*CMP Hd,Hs*/
			h1 = hd;
			h2 = hs;
		}else{					/*CMP Hd,Rs*/
			h1 = hd;
			h2 = rs;
		}
	}else{
		if(opcode & BIT_6_){	/*CMP Rd,Hs*/
			h1 = rd;
			h2 = hs;
		}else{
			h1 = 0;/*`H*/
			h2 = 0;
		}
	}
	
	temp = arm.reg[h1] - arm.reg[h2];
/*
	if((arm.reg[h1] - arm.reg[h2]) > 0)SF(C_); else RF(C_);
	if(!temp)SF(Z_); else RF(Z_);
	if(opcode & BIT_31_)SF(N_); else RF(N_);
*/
	SetSubFlag(arm.reg[h1], arm.reg[h2], temp);

	arm.cycle = 1;
}

_inline void thumb_eor()
{	/*EOR Rd,Rs (rI_a) - 4*/
	u16	opcode;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	arm.reg[rd] = arm.reg[rd] ^ arm.reg[rs];
	
	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void thumb_ldmia()
{	/*LDMIA Rb!,{Rlist} (WX^̃[h) - 15*/
	u32	bit, lst;
	u16	opcode, rlist;
	u8	rb;	

	opcode	= OPCODE_T;
	rlist	= opcode&0xFF;	/*Rlist8*/
	rb		= (opcode&0x07)>>8;

	for(bit=BIT_8_,lst=0; bit!=BIT_1_; bit>>=1){
		if(opcode & bit){
			arm.reg[lst] = agb_read_mem32(arm.reg[rb]+=2);
		}
	}

	arm.cycle = 3;
}

_inline void thumb_ldr()
{	/*LDR Rd,[Rb,Ro] (WX^ItZbgɂ郍[h) - 7*/
	u16	opcode;
	u8	rd, rb, ro;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

	arm.reg[rd] = agb_read_mem32(arm.reg[rb] + arm.reg[ro]);

	arm.cycle = 3;
}

_inline void thumb_ldr_imm()
{	/*LDR Rd,[Rb,#imm](C~fBGCgItZbgɂ郍[h) - 9*/
	u16	opcode, offset;
	u8	rd, rs;

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	arm.reg[rd] = agb_read_mem32(arm.reg[rs] + offset);

	arm.cycle = 3;
}

_inline void thumb_ldrb()
{	/*LDRB Rd,[Rb,Ro] (WX^ItZbgɂ郍[h) - 7*/
	u16	opcode;
	u8	rd, rb, ro;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

	arm.reg[rd] = agb_read_mem8(arm.reg[rb] + arm.reg[ro]);

	arm.cycle = 3;
}

_inline void thumb_ldrb_imm()
{	/*LDRB Rd,[Rb,#imm] (C~fBGCgItZbgɂ郍[h) - 9*/
	u16	opcode, offset;
	u8	rd, rs;

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	arm.reg[rd] = agb_read_mem8(arm.reg[rs] + offset);

	arm.cycle = 3;
}

_inline void thumb_ldrh()
{	/*LDRH Rd,[Rb,Ro] (n[t[h̃[hƕg) - 8*/
	u16	opcode;
	u8	rd, rb, ro;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

	arm.reg[rd] = agb_read_mem16(arm.reg[rb] + arm.reg[ro]);

	arm.cycle = 3;
}

_inline void thumb_ldrh_imm()
{	/*LDRH Rd,[Rb,#imm] (n[t[h̃[h) - 10*/
	u16	opcode, offset;
	u8	rd, rs;

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	arm.reg[rd] = agb_read_mem32(arm.reg[rs] + offset);
}

_inline void thumb_ldr_pc()
{	/*LDR Rd,[PC,#imm] (PC΃[h) - 6*/
	u16	opcode, offset;
	u8	rd;

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;
	rd		= (opcode&0x7)>>8;

	arm.reg[rd] = agb_read_mem32(PC + offset);

	arm.cycle = 3;
}

_inline void thumb_ldr_sp()
{	/*LDR Rd,[SP,#imm] (SP΃[h)- 11*/
	u16	opcode, offset;
	u8	rd;

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;
	rd		= (opcode&0x7)>>8;

	arm.reg[rd] = agb_read_mem32(SP + offset);

	arm.cycle = 3;
}

_inline void thumb_ldsb()
{	/*LDSB Rd,[Rb,Ro] (oCg̃[hƕg) - 8*/
	u16	opcode;
	u8	rd, rb, ro;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

//	arm.reg[rd] = agb_read_mem8(arm.reg[rb] + arm.reg[ro]);
}

_inline void thumb_ldsh()
{	/*LDSH Rd,[Rb,Ro] (n[t[h̃[hƕg) - 8*/
	u16	opcode;
	u8	rd, rb, ro;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

//	arm.reg[rd] = agb_read_mem16(arm.reg[rb] + arm.reg[ro]);

	arm.cycle = 3;
}

_inline void thumb_lsl()
{	/*LSL Rd,Rs (_Vtg) - 4*/
	u16	opcode;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	if(arm.reg[rd] & (1<<arm.reg[rs]))SF(C_); else RF(C_);

	arm.reg[rd] = arm.reg[rd] << arm.reg[rs];

	SetDpFlags(arm.reg[rd]);
}

_inline void thumb_lsl_imm()
{	/*LSL Rd,Rs,#offset5 (_Vtg) - 1*/
	u16	opcode, offset;
	u8	rd, rs;

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;	/*offset5*/
	rd		= DREG_T;
	rs		= SREG_T;

	if(arm.reg[rd] & (1<<offset))SF(C_); else RF(C_);

	arm.reg[rd] = arm.reg[rs] << offset;

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void thumb_lsr()
{	/*LSR Rd,Rd,Rs (_EVtg) - 1*/
	u16	opcode;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	if(arm.reg[rd] & (1<<arm.reg[rs]))SF(C_); else RF(C_);

	arm.reg[rd] = arm.reg[rd] >> arm.reg[rs];

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void thumb_lsr_imm()
{	/*LSR Rd,Rs,#Offset5 (_EVtg) - 1*/
	u16	opcode, offset;
	u8	rd, rs;

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	if(arm.reg[rd] & (1<<offset))SF(C_); else RF(C_);

	arm.reg[rd] = arm.reg[rs] >> offset;

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void thumb_mov()
{	/*MOV Rd,#Offset8 (ړ) - 3*/
	u16	opcode, offset;
	u8	rd;

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;	/*offset8*/
	rd		= (opcode&0x7)>>8;

	arm.reg[rd] = offset;

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void thumb_mov_hi()
{	/*MOV Rd,Hs(ړ,HiWX^) - 5*/
	u16	opcode;
	u8	rd, rs, hd, hs, h1, h2;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= SREG_T;
	hd		= rd + 8;
	hs		= rs + 8;

	if(opcode & BIT_7_){	/*H1*/
		if(opcode & BIT_6_){	/*MOV Hd,Hs*/
			h1 = hd;
			h2 = hs;
		}else{
			h1 = hd;			/*MOV Hd,Rs*/
			h2 = rs;
		}
	}else{
		if(opcode & BIT_6_){	/*MOV Rd,Hs*/
			h1 = rd;
			h2 = hs;
		}else{
			h1 = 0;/*`H*/
			h2 = 0;
		}
	}

	arm.cycle = 1;
}

_inline void thumb_mul()
{	/*MUL Rd,Rs (ώZ) - 4*/
	u16	opcode;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	if((arm.reg[rd] * arm.reg[rs])>0xFFFFFFF)SF(C_); else RF(C_);
	
	arm.reg[rd] = arm.reg[rd] * arm.reg[rs];

	SetDpFlags(arm.reg[rd]);
}

_inline void thumb_mvn()
{	/*MVN Rd,Rs (rbg]) - 4*/
	u16	opcode;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	arm.reg[rd] = ~arm.reg[rs];

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void thumb_neg()
{	/*NEG Rd,Rs (]) - 4*/
	u16	opcode;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	arm.reg[rd] = arm.reg[rs];
	if(arm.reg[rd] & BIT_31_){
		arm.reg[rd] &= ~BIT_31_;
	}else{
		arm.reg[rd] |= BIT_31_;
	}

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void thumb_orr()
{	/*ORR Rd,Rs (_a) - 4*/
	u16	opcode;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	arm.reg[rd] = arm.reg[rd] | arm.reg[rs];

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void thumb_pop()
{	/*POP {Rlist} (WX^POP) - 14*/
	u32	bit, lst;
	u16	opcode, rlist;

	opcode	= OPCODE_T;
	rlist	= opcode&0xFF;	/*Rlist8*/

	for(bit=BIT_7_,lst=7; bit!=BIT_1_; bit>>=1){
		if(opcode & bit){
			arm.reg[lst] = agb_read_mem32(SP-=2);
		}
	}

	if(opcode&BIT_8_)PC = agb_read_mem32(SP-=2);

	arm.cycle = 3;
}

_inline void thumb_push()
{	/*PUSH {Rlist} (WX^PUSH) - 14*/
	u32	bit, lst;
	u16	opcode, rlist;

	opcode	= OPCODE_T;
	rlist	= opcode&0xFF;	/*Rlist8*/

	for(bit=BIT_1_,lst=0; bit!=BIT_8_; bit<<=1){
		if(opcode & bit){
			agb_write_mem32(SP-=2, arm.reg[lst]);
		}
	}

	if(opcode&BIT_8_)agb_write_mem32(SP-=2, LR);

	arm.cycle = 3;
}

_inline void thumb_ror()
{	/*ROR Rd,Rs (E[e[g) - 4*/
	u16	opcode;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	arm.reg[rd] = (arm.reg[rd] >> arm.reg[rs])|(arm.reg[rd] << (16-arm.reg[rs]));

	SetDpFlags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void thumb_sbc()
{	/*SBC Rd,Rs (L[tZ) - 4*/
	u32	temp32;
	u16	opcode;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	/*if((arm.reg[rd] - arm.reg[rs] - LSB_FC) > 0)SF(C_); else RF(C_);*/
	
	temp32 = arm.reg[rd] - (arm.reg[rs] + LSB_FC);

	SetSubFlag(arm.reg[rd], arm.reg[rs] + LSB_FC, temp32);

	arm.reg[rd] = temp32;

	arm.cycle = 1;
}

_inline void thumb_stmia()
{	/*STMIA Rb!,{Rlist} (WX^̃XgA) - 15*/
	u32	bit, lst;
	u16	opcode, rlist;
	u8	rb;	

	opcode	= OPCODE_T;
	rlist	= opcode&0xFF;	/*Rlist8*/
	rb		= (opcode&0x07)>>8;

	for(bit=BIT_1_,lst=0; bit!=BIT_8_; bit<<=1){
		if(opcode & bit){
			agb_write_mem32(arm.reg[rb]-=2, arm.reg[lst]);
		}
	}

	arm.cycle = 3;
}

_inline void thumb_str()
{	/*STR Rd,[Rb,Ro] (WX^ItZbgɂXgA) - 7*/
	u16	opcode;
	u8	rd, rb, ro;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

	agb_write_mem32(arm.reg[rd], arm.reg[rb] + arm.reg[ro]);

	arm.cycle = 3;
}

_inline void thumb_str_imm()
{	/*STR Rd,[Rb,#imm] (C~fBGCgItZbgɂXgA) - 9*/
	u16	opcode, offset;
	u8	rd, rs;

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	agb_write_mem32(arm.reg[rd], arm.reg[rs] + offset);

	arm.cycle = 3;
}

_inline void thumb_strb()
{	/*STRB Rd,[Rb,Ro] (WX^ItZbgɂXgA) - 7*/
	u16	opcode;
	u8	rd, rb, ro;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

	agb_write_mem8(arm.reg[rd], arm.reg[rb] + arm.reg[ro]);

	arm.cycle = 3;
}

_inline void thumb_strb_imm()
{	/*STRB Rd,[Rb,#imm] (C~fBGCgItZbgɂXgA) - 9*/
	u16	opcode, offset;
	u8	rd, rs;

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	agb_write_mem8(arm.reg[rd], arm.reg[rs] + offset);

	arm.cycle = 3;
}

_inline void thumb_strh()
{	/*STRH Rd,[Rb,Ro] (n[t[h̃XgAƕg - 8*/
	u16	opcode;
	u8	rd, rb, ro;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rb		= SREG_T;
	ro		= OREG_T;

	agb_write_mem16(arm.reg[rd], arm.reg[rb] + arm.reg[ro]);

	arm.cycle = 3;
}

_inline void thumb_strh_imm()
{	/*STRH Rd,[Rb,#imm] (n[t[h̃XgA)- 10*/
	u16	opcode, offset;
	u8	rd, rs;

	opcode	= OPCODE_T;
	offset	= (opcode&0x1F)>>6;
	rd		= DREG_T;
	rs		= SREG_T;

	agb_write_mem16(arm.reg[rd], arm.reg[rs] + offset);

	arm.cycle = 3;
}

_inline void thumb_str_sp()
{	/*STR Rd,[SP,#imm] (SP΃XgA) - 11*/
	u16	opcode, offset;
	u8	rd;

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;
	rd		= (opcode&0x7)>>8;

	agb_write_mem32(arm.reg[rd], SP + offset);

	arm.cycle = 3;
}

_inline void thumb_swi()
{	/*SWI Value8 (\tgEFA荞) - 17*/
	u16	opcode, offset;

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;

	arm.cycle = 3;
}

_inline void thumb_sub()
{	/*SUB Rd,Rs,Rn (Z) - 2*/
	u16	opcode, offset;
	u8	rd, rs;

	opcode	= OPCODE_T;
	offset	= (opcode&0x7)>>3;
	rd		= DREG_T;
	rs		= SREG_T;

	if(opcode & BIT_10_){	/*1 = I = offset*/
		arm.reg[rd] = arm.reg[rs] - offset;
		if((arm.reg[rd] - offset) > 0)SF(C_); else RF(C_);
	}else{					/*0 = I = register*/
		arm.reg[rd] = arm.reg[rs] - arm.reg[offset];
		if((arm.reg[rd] - arm.reg[offset]) > 0)SF(C_); else RF(C_);
	}

	SetSubFlag(arm.reg[rs], offset, arm.reg[rd]);

	arm.cycle = 1;
}

_inline void thumb_sub_imm()
{	/*SUB Rd,#Offset8 (Z) - 3*/
	u32	temp32;
	u16	opcode, offset;
	u8	rd;

	opcode	= OPCODE_T;
	offset	= opcode&0xFF;	/*offset8*/
	rd		= (opcode&0x7)>>8;
	
	/*if((arm.reg[rd] - offset)>0)SF(C_); else RF(C_);*/

	temp32 = arm.reg[rd] - offset;

	SetSubFlag(arm.reg[rd], offset, temp32);

	arm.reg[rd] = temp32;

	arm.cycle = 1;
}

_inline void thumb_tst()
{	/*TST Rd,Rs (rbgeXg) - 4*/
	u16	opcode, temp16;
	u8	rd, rs;

	opcode	= OPCODE_T;
	rd		= DREG_T;
	rs		= DREG_T;

	temp16 = arm.reg[rd] & arm.reg[rs];

	SetDpFlags(temp16);

	arm.cycle = 1;
}

static int exec_thumb_state()
{
	u16	opcode;
	u8	opcode_11_5, opcode_6_5, opcode_9_3;

	opcode = OPCODE_T;

	/*IyR[hw肳͈͂̃rbgo*/
	opcode_11_5	= (u8)((opcode>>11)&0x11);	/*11-15 5bit*/
	opcode_6_5	= (u8)((opcode>>6)&0x1F);
	opcode_9_3	= (u8)((opcode>>6)&0x07);

	switch(opcode_11_5){
	case 0x00:	/*00000*/
		thumb_lsl_imm();/*LSL Rd,Rs,#Offset5 - 1*/
		break;
	case 0x01:	/*00001*/
		thumb_lsr_imm();/*LSR Rd,Rs,#Offset5 - 1*/
		break;
	case 0x02:	/*00010*/
		thumb_asr_imm();/*ASR Rd,Rs,#Offset5 - 1*/
		break;
	case 0x03:	/*00011*/
		if(opcode & BIT_9_){
			thumb_sub();/*Z - 2*/
		}else{
			thumb_add();/*Z - 2*/
		}
		break;
	case 0x04:	/*00100*/
		thumb_mov();/*MOV Rd,#Offset8 ړ - 3*/
		break;
	case 0x05:	/*00101*/
		thumb_cmp();/*CMP Rd,#Offset8 r - 3*/
		break;
	case 0x06:	/*00110*/
		thumb_add_imm();/*ADD Rd,#Offset8 Z - 3*/
		break;
	case 0x07:	/*00111*/
		thumb_sub_imm();/*SUB Rd,#Offset8 Z - 3*/
		break;
	case 0x08:	/*01000*/
		switch(opcode_6_5){
		case 0x00:	/*HiWX^/Xe[g*/
		case 0x01:
		case 0x02:
			thumb_add_hi();	/*HiWX^ - 5*/
			break;
		case 0x03:
		case 0x04:
		case 0x05:
			thumb_cmp_hi();	/*HiWX^ - 5*/
			break;
		case 0x06:
		case 0x07:
		case 0x08:
			thumb_mov_hi();	/*HiWX^ - 5*/
			break;
		case 0x09:
		case 0x0a:
		case 0x0b:
			thumb_bx_hi();	/*ƃXe[gύX - 5*/
			break;
		case 0x10:	/*ALUZ - 4*/
			thumb_and();
			break;
		case 0x11:
			thumb_eor();
			break;
		case 0x12:
			thumb_lsl();
			break;
		case 0x13:
			thumb_lsr();
			break;
		case 0x14:
			thumb_asr();
			break;
		case 0x15:
			thumb_adc();
			break;
		case 0x16:
			thumb_sbc();
			break;
		case 0x17:
			thumb_ror();
			break;
		case 0x18:
			thumb_tst();
			break;
		case 0x19:
			thumb_neg();
			break;
		case 0x1A:
			thumb_cmp();
			break;
		case 0x1B:
			thumb_cmn();
			break;
		case 0x1C:
			thumb_orr();
			break;
		case 0x1D:
			thumb_mul();
			break;
		case 0x1E:
			thumb_bic();
			break;
		case 0x1F:
			thumb_mvn();
			break;
		}
		break;
	case 0x09:	/*01001*/
		thumb_ldr_pc();	/*PC΃[h - 6*/
		break;
	case 0x0A:	/*01010*/
	case 0x0B:	/*01011*/
		switch(opcode_9_3){
		/*WX^ItZbgɂ郍[h/XgA - 7*/
		case 0x0:	/*000 LB0*/
			thumb_str();
			break;
		case 0x2:	/*010 LB0*/
			thumb_strb();
			break;
		case 0x4:	/*100 LB0*/
			thumb_ldr();
			break;
		case 0x6:	/*110 LB0*/
			thumb_ldrb();
			break;
		/*oCg^n[t[h̃[h^XgAƕg - 8*/
		case 0x1:	/*000 HS0*/
			thumb_strh();
			break;
		case 0x3:	/*010 HS0*/
			thumb_ldsb();
			break;
		case 0x5:	/*100 HS0*/
			thumb_ldrh();
			break;
		case 0x7:	/*110 HS0*/
			thumb_ldsh();
			break;
		}
		break;	
	/*C~fBGCgItZbgɂ郍[h^XgA - 9*/
	case 0x0C:	/*01100 - BL=00*/
		thumb_str_imm();	/*str rd,[rb,#imm]*/
		break;
	case 0x0D:	/*01101 - BL=01*/
		thumb_ldr_imm();	/*ldr rd,[rb,#imm]*/
		break;
	case 0x0E:	/*01110 - BL=10*/
		thumb_strb_imm();/*strb rd,[rb,#imm]*/
		break;
	case 0x0F:	/*01111 - BL=11*/
		thumb_ldrb_imm();/*ldrb rd,[rb,#imm]*/
		break;
	/*n[t[h̃[h^XgA - 10*/
	case 0x10:	/*10000 - L=0*/
		thumb_strh_imm();/*strh rd,[rb,#imm]*/
		break;
	case 0x11:	/*10001 - L=1*/
		thumb_ldrh_imm();/*ldrh rd,[rb,#imm]*/
		break;
	/*SP΃[h^XgA - 11*/
	case 0x12:	/*10010 - S=0*/
		thumb_str_sp();/*str rd,[SP,#imm]*/
		break;
	case 0x13:	/*10011 - S=1*/
		thumb_ldr_sp();/*ldr rd,[SP,#imm]*/
		break;
	/*AhX̃[h - 12*/
	case 0x14:	/*10100 - S=0*//*add rd,PC,#imm*/
	case 0x15:	/*10101 - S=1*//*add rd,SP,#imm*/
		thumb_add_adr();
		break;
	case 0x16:	/*10110*/
	case 0x17:	/*10111*/
		if(opcode & BIT_10_){
	/*WX^PUSH/POP - 14*/
			if(opcode & BIT_11_){	/*L*/
				thumb_pop();/*POP {Rlist}*/
			}else{
				thumb_push();/*PUSH {Rlist}*/
			}
		}else{
	/*X^bN|C^ɃItZbgZ - 13*/
			if(!((opcode&0x7)>>8)){	/*000S*/
				thumb_add_sp();/*add SP,#+-imm*/
			}
			break;
		}
		break;
	/*WX^̃[h^XgA - 15*/
	case 0x18:	/*11000*/
		thumb_stmia();/*stmia rb!,{Rlist}*/
		break;
	case 0x19:	/*11001*/
		thumb_ldmia();/*ldmia rb!,{Rlist}*/
		break;
	case 0x1A:	/*11010*/
	case 0x1B:	/*11011*/
		if(((opcode&0xF)>>8)==0xF){
	/*\tgEFA荞 - 17*/
			thumb_swi();
		}else{
	/* - 16*/
			thumb_bxx();
		}
		break;
	/* - 18*/
	case 0x1C:	/*11100*/
		thumb_b();
		break;
	/*ƃN - 19*/
	case 0x1E:	/*11110*/
	case 0x1F:	/*11111*/
		thumb_bl();
		break;
	default:
		break;
	}

	PC += 2;

	return 0;
}

