
/*	
	GameboyAdvanceVM 
		- Nintendo GameboyAdvance Emulator
	Copyright 2002 Y_N y_n@users.sourceforge.jp
	Homepage https://sourceforge.jp/projects/gbaemu/
*/



_inline void arm7tdmi_thumb_adc()
{	/*ADC Rd,Rs(L[tZ) - 4*/
	u32	temp32, cflag;
	u16	opcode;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	cflag = ARM_LSB_FC;
	temp32 = arm.reg[rd] + arm.reg[rs] + cflag;

	arm7tdmi_set_add_flag(arm.reg[rd], arm.reg[rs] + cflag, temp32);

	arm.reg[rd] = temp32;

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_add()
{	/*ADD Rd,Rs,(Rn || #Offset3) (Z) - 2*/
	u16	opcode, offset;
	u32	rd, rs;

	opcode	= arm.opcode16;
	offset	= (opcode >> 3) & 0x7;
	rd		= THUMB_DREG;
	rs		= THUMB_SREG;

	if(opcode & BIT10){	/*1 = I = offset*/
		arm.reg[rd] = arm.reg[rs] + offset;
	}else{					/*0 = I = register*/
		arm.reg[rd] = arm.reg[rs] + offset;
	}

	arm7tdmi_set_add_flag(arm.reg[rs], offset, arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_add_imm()
{	/*ADD Rd,#Offset8 (Z) - 3*/
	u32	temp32;
	u16	opcode, offset;
	u32	rd;

	opcode	= arm.opcode16;
	offset	= opcode & 0xFF;	/*offset8*/
	rd		= (opcode >> 8) & 0x7;

	temp32 = arm.reg[rd] + offset;

	arm7tdmi_set_add_flag(arm.reg[rd], offset, temp32);

	arm.reg[rd] = temp32;

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_add_adr()
{	/*ADD Rd,(PC || SP),#imm (Z) - 12*/
	u16	opcode, offset;
	u32	rd;

	opcode	= arm.opcode16;
	offset	= opcode & 0xFF;	/*offset8*/
	rd		= (opcode >> 8) & 0x7;

	if(opcode & BIT11){
		/*if((ARM_SP + offset)>0xFFFFFFFF)ARM_SF(ARM_C); else ARM_RF(ARM_C);*/
		arm.reg[rd] = ARM_SP + offset;
		arm7tdmi_set_add_flag(ARM_SP, offset, arm.reg[rd]);
	}else{
		/*if((ARM_PC + offset)>0xFFFFFFFF)ARM_SF(ARM_C); else ARM_RF(ARM_C);*/
		arm.reg[rd] = ARM_PC + offset;
		arm7tdmi_set_add_flag(ARM_PC, offset, arm.reg[rd]);
	}

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_add_sp()
{	/*ADD SP,#+-imm (X^bN|C^ɃItZbgZ) - 13*/
	u16	opcode, offset;

	opcode	= arm.opcode16;
	offset	= opcode & 0x7F;	/*offset7*/
	
	if(opcode & BIT7){
		if((ARM_SP + offset) > 0xFFFFFFFF)ARM_SF(ARM_C); else ARM_RF(ARM_C);
		ARM_SP = ARM_SP + offset;
	}else{
		if((ARM_SP - offset) > 0)ARM_SF(ARM_C); else ARM_RF(ARM_C);
		ARM_SP = ARM_SP - offset;
	}

	if(!ARM_SP)ARM_SF(ARM_Z); else ARM_RF(ARM_Z);
	if(opcode & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_add_hi()
{	/*ADD Rd,Hs (Z) - 5*/
	u16	opcode;
	u32	rd, rs, hd, hs, h1, h2;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_SREG;
	hd		= rd + 8;
	hs		= rs + 8;

	if(opcode & BIT7){
		if(opcode & BIT6){	/*ADD Hd,Hs*/
			h1 = hd;
			h2 = hs;
		}else{					/*ADD Hd,Rs*/
			h1 = hd;
			h2 = rs;
		}
	}else{
		if(opcode & BIT6){	/*ADD Rd,Hs*/
			h1 = rd;
			h2 = hs;
		}else{
			h1 = 0;/*`H*/
			h2 = 0;
		}
	}

	arm.reg[h1] = arm.reg[h1] + arm.reg[h2];

	if((arm.reg[h1] + arm.reg[h2]) > 0xFFFFFFFF)ARM_SF(ARM_C); else ARM_RF(ARM_C);
	if(!arm.reg[h1])ARM_SF(ARM_Z); else ARM_RF(ARM_Z);
	if(opcode & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_and()
{	/*AND Rd,Rs (_) - 4*/
	u16	opcode;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	arm.reg[rd] = arm.reg[rd] & arm.reg[rs];

	arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_asr()
{	/*ASR Rd,Rs (ZpEVtg) - 4*/
	u32	bit, temp32, shift;
	u16	opcode;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	if(arm.reg[rd] & (1 << arm.reg[rs]))ARM_SF(ARM_C); else ARM_RF(ARM_C);

	//arm.reg[rd] = arm.reg[rd]>>arm.reg[rs];

	shift = arm.reg[rs];

	temp32 = (arm.reg[rd] >> shift);
	if(arm.reg[rd] & BIT31){
		for(bit = BIT31; shift; bit >>= 1){
			temp32 |= bit;
			shift--;
		}
	}
	arm.reg[rd] = temp32;

	arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_asr_imm()
{	/*ASR Rd,Rs,#Offset5 (ZpEVtg) - 1*/
	u32	bit, temp32;
	u16	opcode, offset;
	u32	rd, rs;

	opcode	= arm.opcode16;
	offset	= (opcode & 0x1F) >> 6;
	rd		= THUMB_DREG;
	rs		= THUMB_SREG;

	if(arm.reg[rd] & (1 << offset))ARM_SF(ARM_C); else ARM_RF(ARM_C);

	//arm.reg[rd] = arm.reg[rs]>>offset;

	temp32 = (arm.reg[rs] >> offset);
	if(arm.reg[rs] & BIT31){
		for(bit = BIT31; offset; bit >>= 1){
			temp32 |= bit;
			offset--;
		}
	}
	arm.reg[rd] = temp32;

	arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_b()
{	/*B label () - 18*/
	u16	opcode;
	s16	offset;

	opcode	= arm.opcode16;
	offset	= opcode & 0x7FF;

	ARM_PC = ARM_PC + offset;

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_bxx()
{	/*BXX label () - 16*/
	BOOL not_jump;
	u16	opcode, offset;
	u32	cond;

	opcode	= arm.opcode16;
	offset	= opcode & 0xFF;
	cond	= (opcode >> 8) & 0xF;

	not_jump = TRUE;
	switch(cond){	/*s - Condition Field*/
	case 0x0:
		if(ARM_FZ)not_jump = FALSE;
		break;		/*if Z set		- equal*/
	case 0x1:
		if(!ARM_FZ)not_jump = FALSE;
		break;		/*if Z clear	- not equal*/
	case 0x2:
		if(ARM_FC)not_jump = FALSE;
		break;		/*if C set		- unsigned higher or same*/
	case 0x3:
		if(!ARM_FC)not_jump = FALSE;
		break;		/*if C clear	- unsigne lower*/
	case 0x4:
		if(ARM_FN)not_jump = FALSE;
		break;		/*if N set		- negative*/
	case 0x5:
		if(!ARM_FN)not_jump = FALSE;
		break;		/*if N clear	- positive or zero*/
	case 0x6:
		if(ARM_FV)not_jump = FALSE;
		break;		/*if V set		- overflow*/
	case 0x7:
		if(!ARM_FV)not_jump = FALSE;
		break;		/*if V clear	- no overflow*/
	case 0x8:
		if(ARM_FC && !ARM_FZ)not_jump = FALSE;
		break;		/*if C set and Z clear	- unsigned higher*/
	case 0x9:
		if(!ARM_FC || ARM_FZ)not_jump = FALSE;
		break;		/*if C clear and Z set	- unsigned lower or same*/
	case 0xA:
		if((ARM_FN && ARM_FV) || (!ARM_FN && !ARM_FV))not_jump = FALSE;
		break;		/*if ((N set and V set) or (N clear and V clear))	- greater or equal*/
	case 0xB:
		if((!ARM_FN && ARM_FV) || (ARM_FN && !ARM_FV))not_jump = FALSE;
		break;		/*if ((N set and V clear) or (N clear and V set))	- less than*/
	case 0xC:
		if((!ARM_FZ && (ARM_FN || ARM_FV)) || (!ARM_FN || !ARM_FV))not_jump = FALSE;
		break;		/*if (Z clear and(N or V set) or (N or V clear))- greater than*/
	case 0xD:
		if((ARM_FZ || (ARM_FN && !ARM_FV)) || (!ARM_FN && ARM_FV))not_jump = FALSE;
		break;		/*if (Z set or(N set and V clear) or (N clear and V set))- less than or equal*/
	default:
		not_jump = FALSE;
		break;		/*(ignored)	- always */
	}

	if(!not_jump)ARM_PC = ARM_PC + offset;

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_bl()
{	/*BL label (ƃN) - 19*/
	u32	temp;
	u16	opcode, offset;

	opcode	= arm.opcode16;
	offset	= opcode & 0x7FF;

	if(opcode & BIT11){	/*H=1*/
		temp = ARM_PC + 2;
		ARM_PC = ARM_LR + (offset << 1);
		ARM_LR = temp | 1;
	}else{					/*H=0*/
		ARM_LR = ARM_PC + (offset << 12);
	}

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_bic()
{	/*BIC Rd,Rs (rbgNA) - 4*/
	u16	opcode;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	arm.reg[rd] = arm.reg[rd] & ~arm.reg[rs];

	arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_bx()
{	/*Bx (Xe[gύX)*/
	u16	opcode;

	opcode	= arm.opcode16;

	if(opcode & BIT1){
		ARM_CPSR &= ~ARM_T;	/*ARMXe[gɕύX*/
	}else{
		/*ARM_CPSR |= ARM_T;*/	/*THUMBXe[ĝ܂*/
	}
}

_inline void arm7tdmi_thumb_bx_hi()
{	/*BX Hs(Xe[gύX) - 5*/
	u16	opcode;
	u32	rs, hs, h2;

	opcode	= arm.opcode16;
	rs		= THUMB_SREG;
	hs		= rs + 8;

	if(opcode & BIT7){	/*H1*/
		if(opcode & BIT6){	/*BX Hs*/
			h2 = hs;
		}else{					/*BX Rs*/
			h2 = rs;
		}
	}else{
		h2 = 0;/*`H*/
	}

	if(arm.reg[h2] & BIT1){
		ARM_CPSR &= ~ARM_T;	/*ARMXe[gɕύX*/
	}else{
		/*ARM_CPSR |= ARM_T;*/	/*THUMBXe[ĝ܂*/
	}

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_cmn()
{	/*CMN Rd,Rs (r) - 4*/
	u32	temp;
	u16	opcode;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	temp = arm.reg[rd] + arm.reg[rs];
/*
	if((arm.reg[rd] + arm.reg[rs]) > 0xFFFFFFFF)ARM_SF(ARM_C); else ARM_RF(ARM_C);
	if(!temp)ARM_SF(ARM_Z); else ARM_RF(ARM_Z);
	if(opcode & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);
*/
	arm7tdmi_set_add_flag(arm.reg[rd], arm.reg[rs], temp);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_cmp()
{	/*CMP Rd,Rs (r) - 4*/
	u16	opcode, temp;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	temp = arm.reg[rd] - arm.reg[rs];	
/*
	if((arm.reg[rd] - arm.reg[rs]) > 0)ARM_SF(ARM_C); else ARM_RF(ARM_C);
	if(!temp)ARM_SF(ARM_Z); else ARM_RF(ARM_Z);
	if(opcode & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);
*/
	arm7tdmi_set_sub_flag(arm.reg[rd], arm.reg[rs], temp);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_cmp_imm()
{	/*CMP Rd,#Offset8 (r) - 3*/
	u16	opcode, offset, temp;
	u32	rd;

	opcode	= arm.opcode16;
	offset	= opcode & 0xFF;	/*offset8*/
	rd		= (opcode >> 8) & 0x7;

	temp = arm.reg[rd] - offset;
/*
	if((arm.reg[rd] - offset) > 0)ARM_SF(ARM_C); else ARM_RF(ARM_C);
	if(!temp)ARM_SF(ARM_Z); else ARM_RF(ARM_Z);
	if(opcode & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);
*/
	arm7tdmi_set_sub_flag(arm.reg[rd], offset, temp);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_cmp_hi()
{	/*CMP Rd,Hs (r,HiWX^) - 5*/
	u16	opcode,	temp;
	u32	rd, rs, hd, hs, h1, h2;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_SREG;
	hd		= rd + 8;
	hs		= rs + 8;

	if(opcode & BIT7){	/*H1*/
		if(opcode & BIT6){	/*CMP Hd,Hs*/
			h1 = hd;
			h2 = hs;
		}else{					/*CMP Hd,Rs*/
			h1 = hd;
			h2 = rs;
		}
	}else{
		if(opcode & BIT6){	/*CMP Rd,Hs*/
			h1 = rd;
			h2 = hs;
		}else{
			h1 = 0;/*`H*/
			h2 = 0;
		}
	}
	
	temp = arm.reg[h1] - arm.reg[h2];
/*
	if((arm.reg[h1] - arm.reg[h2]) > 0)ARM_SF(ARM_C); else ARM_RF(ARM_C);
	if(!temp)ARM_SF(ARM_Z); else ARM_RF(ARM_Z);
	if(opcode & BIT31)ARM_SF(ARM_N); else ARM_RF(ARM_N);
*/
	arm7tdmi_set_sub_flag(arm.reg[h1], arm.reg[h2], temp);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_eor()
{	/*EOR Rd,Rs (rI_a) - 4*/
	u16	opcode;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	arm.reg[rd] = arm.reg[rd] ^ arm.reg[rs];
	
	arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_ldmia()
{	/*LDMIA Rb!,{Rlist} (WX^̃[h) - 15*/
	u32	bit, lst;
	u16	opcode, rlist;
	u32	rb;	

	opcode	= arm.opcode16;
	rlist	= opcode & 0xFF;	/*Rlist8*/
	rb		= (opcode & 0x07) >> 8;

	for(bit = BIT8, lst = 0; bit != BIT1; bit >>= 1){
		if(opcode & bit){
			arm.reg[lst] = agb_read_mem32(arm.reg[rb] += 2);
		}
	}

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_ldr()
{	/*LDR Rd,[Rb,Ro] (WX^ItZbgɂ郍[h) - 7*/
	u16	opcode;
	u32	rd, rb, ro;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rb		= THUMB_SREG;
	ro		= THUMB_OREG;

	arm.reg[rd] = agb_read_mem32(arm.reg[rb] + arm.reg[ro]);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_ldr_imm()
{	/*LDR Rd,[Rb,#imm](C~fBGCgItZbgɂ郍[h) - 9*/
	u16	opcode, offset;
	u32	rd, rs;

	opcode	= arm.opcode16;
	offset	= (opcode >> 6) & 0x1F;
	rd		= THUMB_DREG;
	rs		= THUMB_SREG;

	arm.reg[rd] = agb_read_mem32(arm.reg[rs] + offset);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_ldrb()
{	/*LDRB Rd,[Rb,Ro] (WX^ItZbgɂ郍[h) - 7*/
	u16	opcode;
	u32	rd, rb, ro;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rb		= THUMB_SREG;
	ro		= THUMB_OREG;

	arm.reg[rd] = agb_read_mem8(arm.reg[rb] + arm.reg[ro]);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_ldrb_imm()
{	/*LDRB Rd,[Rb,#imm] (C~fBGCgItZbgɂ郍[h) - 9*/
	u16	opcode, offset;
	u32	rd, rs;

	opcode	= arm.opcode16;
	offset	= (opcode >> 6) & 0x1F;
	rd		= THUMB_DREG;
	rs		= THUMB_SREG;

	arm.reg[rd] = agb_read_mem8(arm.reg[rs] + offset);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_ldrh()
{	/*LDRH Rd,[Rb,Ro] (n[t[h̃[hƕg) - 8*/
	u16	opcode;
	u32	rd, rb, ro;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rb		= THUMB_SREG;
	ro		= THUMB_OREG;

	arm.reg[rd] = agb_read_mem16(arm.reg[rb] + arm.reg[ro]);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_ldrh_imm()
{	/*LDRH Rd,[Rb,#imm] (n[t[h̃[h) - 10*/
	u16	opcode, offset;
	u32	rd, rs;

	opcode	= arm.opcode16;
	offset	= (opcode >> 6) & 0x1F;
	rd		= THUMB_DREG;
	rs		= THUMB_SREG;

	arm.reg[rd] = agb_read_mem32(arm.reg[rs] + offset);
}

_inline void arm7tdmi_thumb_ldr_pc()
{	/*LDR Rd,[PC,#imm] (PC΃[h) - 6*/
	u16	opcode, offset;
	u32	rd;

	opcode	= arm.opcode16;
	offset	= opcode & 0xFF;
	rd		= (opcode >> 8) & 0x7;

	arm.reg[rd] = agb_read_mem32(ARM_PC + offset);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_ldr_sp()
{	/*LDR Rd,[SP,#imm] (SP΃[h)- 11*/
	u16	opcode, offset;
	u32	rd;

	opcode	= arm.opcode16;
	offset	= opcode & 0xFF;
	rd		= (opcode >> 8) & 0x7;

	arm.reg[rd] = agb_read_mem32(ARM_SP + offset);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_ldsb()
{	/*LDSB Rd,[Rb,Ro] (oCg̃[hƕg) - 8*/
	u16	opcode;
	u32	rd, rb, ro;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rb		= THUMB_SREG;
	ro		= THUMB_OREG;

//	arm.reg[rd] = agb_read_mem8(arm.reg[rb] + arm.reg[ro]);
}

_inline void arm7tdmi_thumb_ldsh()
{	/*LDSH Rd,[Rb,Ro] (n[t[h̃[hƕg) - 8*/
	u16	opcode;
	u32	rd, rb, ro;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rb		= THUMB_SREG;
	ro		= THUMB_OREG;

//	arm.reg[rd] = agb_read_mem16(arm.reg[rb] + arm.reg[ro]);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_lsl()
{	/*LSL Rd,Rs (_Vtg) - 4*/
	u16	opcode;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	if(arm.reg[rd] & (1<<arm.reg[rs]))ARM_SF(ARM_C); else ARM_RF(ARM_C);

	arm.reg[rd] = arm.reg[rd] << arm.reg[rs];

	arm7tdmi_set_dp_flags(arm.reg[rd]);
}

_inline void arm7tdmi_thumb_lsl_imm()
{	/*LSL Rd,Rs,#offset5 (_Vtg) - 1*/
	u16	opcode, offset;
	u32	rd, rs;

	opcode	= arm.opcode16;
	offset	= (opcode >> 6) & 0x1F;	/*offset5*/
	rd		= THUMB_DREG;
	rs		= THUMB_SREG;

	if(arm.reg[rd] & (1 << offset))ARM_SF(ARM_C); else ARM_RF(ARM_C);

	arm.reg[rd] = arm.reg[rs] << offset;

	arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_lsr()
{	/*LSR Rd,Rd,Rs (_EVtg) - 1*/
	u16	opcode;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	if(arm.reg[rd] & (1<<arm.reg[rs]))ARM_SF(ARM_C); else ARM_RF(ARM_C);

	arm.reg[rd] = arm.reg[rd] >> arm.reg[rs];

	arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_lsr_imm()
{	/*LSR Rd,Rs,#Offset5 (_EVtg) - 1*/
	u16	opcode, offset;
	u32	rd, rs;

	opcode	= arm.opcode16;
	offset	= (opcode >> 6) & 0x1F;
	rd		= THUMB_DREG;
	rs		= THUMB_SREG;

	if(arm.reg[rd] & (1 << offset))ARM_SF(ARM_C); else ARM_RF(ARM_C);

	arm.reg[rd] = arm.reg[rs] >> offset;

	arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_mov()
{	/*MOV Rd,#Offset8 (ړ) - 3*/
	u16	opcode, offset;
	u32	rd;

	opcode	= arm.opcode16;
	offset	= opcode & 0xFF;	/*offset8*/
	rd		= (opcode >> 8) & 0x7;

	arm.reg[rd] = offset;

	arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_mov_hi()
{	/*MOV Rd,Hs(ړ,HiWX^) - 5*/
	u16	opcode;
	u32	rd, rs, hd, hs, h1, h2;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_SREG;
	hd		= rd + 8;
	hs		= rs + 8;

	if(opcode & BIT7){	/*H1*/
		if(opcode & BIT6){	/*MOV Hd,Hs*/
			h1 = hd;
			h2 = hs;
		}else{
			h1 = hd;			/*MOV Hd,Rs*/
			h2 = rs;
		}
	}else{
		if(opcode & BIT6){	/*MOV Rd,Hs*/
			h1 = rd;
			h2 = hs;
		}else{
			h1 = 0;/*`H*/
			h2 = 0;
		}
	}

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_mul()
{	/*MUL Rd,Rs (ώZ) - 4*/
	u16	opcode;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	if((arm.reg[rd] * arm.reg[rs]) > 0xFFFFFFF)ARM_SF(ARM_C); else ARM_RF(ARM_C);
	
	arm.reg[rd] = arm.reg[rd] * arm.reg[rs];

	arm7tdmi_set_dp_flags(arm.reg[rd]);
}

_inline void arm7tdmi_thumb_mvn()
{	/*MVN Rd,Rs (rbg]) - 4*/
	u16	opcode;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	arm.reg[rd] = ~arm.reg[rs];

	arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_neg()
{	/*NEG Rd,Rs (]) - 4*/
	u16	opcode;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	arm.reg[rd] = arm.reg[rs];
	if(arm.reg[rd] & BIT31){
		arm.reg[rd] &= ~BIT31;
	}else{
		arm.reg[rd] |= BIT31;
	}

	arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_orr()
{	/*ORR Rd,Rs (_a) - 4*/
	u16	opcode;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	arm.reg[rd] = arm.reg[rd] | arm.reg[rs];

	arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_pop()
{	/*POP {Rlist} (WX^POP) - 14*/
	u32	bit, lst;
	u16	opcode, rlist;

	opcode	= arm.opcode16;
	rlist	= opcode & 0xFF;	/*Rlist8*/

	for(bit = BIT7, lst = 7; bit != BIT1; bit >>= 1){
		if(opcode & bit){
			arm.reg[lst] = agb_read_mem32(ARM_SP -= 4);
		}
	}

	if(opcode & BIT8)ARM_PC = agb_read_mem32(ARM_SP -= 4);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_push()
{	/*PUSH {Rlist} (WX^PUSH) - 14*/
	u32	bit, lst;
	u16	opcode, rlist;

	opcode	= arm.opcode16;
	rlist	= opcode & 0xFF;	/*Rlist8*/

	for(bit = BIT1, lst = 0; bit != BIT8; bit <<= 1){
		if(opcode & bit){
			agb_write_mem32(ARM_SP -= 4, arm.reg[lst]);
		}
	}

	if(opcode & BIT8)agb_write_mem32(ARM_SP -= 4, ARM_LR);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_ror()
{	/*ROR Rd,Rs (E[e[g) - 4*/
	u16	opcode;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	arm.reg[rd] = (arm.reg[rd] >> arm.reg[rs]) | (arm.reg[rd] << (16-arm.reg[rs]));

	arm7tdmi_set_dp_flags(arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_sbc()
{	/*SBC Rd,Rs (L[tZ) - 4*/
	u32	temp32;
	u16	opcode;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	/*if((arm.reg[rd] - arm.reg[rs] - LSB_FC) > 0)ARM_SF(ARM_C); else ARM_RF(ARM_C);*/
	
	temp32 = arm.reg[rd] - (arm.reg[rs] + ARM_LSB_FC);

	arm7tdmi_set_sub_flag(arm.reg[rd], arm.reg[rs] + ARM_LSB_FC, temp32);

	arm.reg[rd] = temp32;

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_stmia()
{	/*STMIA Rb!,{Rlist} (WX^̃XgA) - 15*/
	u32	bit, lst;
	u16	opcode, rlist;
	u32	rb;	

	opcode	= arm.opcode16;
	rlist	= opcode & 0xFF;	/*Rlist8*/
	rb		= (opcode >> 8) & 0x07;

	for(bit = BIT1, lst = 0; bit != BIT8; bit <<= 1){
		if(opcode & bit){
			agb_write_mem32(arm.reg[rb] -= 2, arm.reg[lst]);
		}
	}

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_str()
{	/*STR Rd,[Rb,Ro] (WX^ItZbgɂXgA) - 7*/
	u16	opcode;
	u32	rd, rb, ro;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rb		= THUMB_SREG;
	ro		= THUMB_OREG;

	agb_write_mem32(arm.reg[rd], arm.reg[rb] + arm.reg[ro]);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_str_imm()
{	/*STR Rd,[Rb,#imm] (C~fBGCgItZbgɂXgA) - 9*/
	u16	opcode, offset;
	u32	rd, rs;

	opcode	= arm.opcode16;
	offset	= (opcode >> 6) & 0x1F;
	rd		= THUMB_DREG;
	rs		= THUMB_SREG;

	agb_write_mem32(arm.reg[rd], arm.reg[rs] + offset);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_strb()
{	/*STRB Rd,[Rb,Ro] (WX^ItZbgɂXgA) - 7*/
	u16	opcode;
	u32	rd, rb, ro;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rb		= THUMB_SREG;
	ro		= THUMB_OREG;

	agb_write_mem8(arm.reg[rd], arm.reg[rb] + arm.reg[ro]);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_strb_imm()
{	/*STRB Rd,[Rb,#imm] (C~fBGCgItZbgɂXgA) - 9*/
	u16	opcode, offset;
	u32	rd, rs;

	opcode	= arm.opcode16;
	offset	= (opcode >> 6) & 0x1F;
	rd		= THUMB_DREG;
	rs		= THUMB_SREG;

	agb_write_mem8(arm.reg[rd], arm.reg[rs] + offset);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_strh()
{	/*STRH Rd,[Rb,Ro] (n[t[h̃XgAƕg - 8*/
	u16	opcode;
	u32	rd, rb, ro;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rb		= THUMB_SREG;
	ro		= THUMB_OREG;

	agb_write_mem16(arm.reg[rd], arm.reg[rb] + arm.reg[ro]);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_strh_imm()
{	/*STRH Rd,[Rb,#imm] (n[t[h̃XgA)- 10*/
	u16	opcode, offset;
	u32	rd, rs;

	opcode	= arm.opcode16;
	offset	= (opcode >> 6) & 0x1F;
	rd		= THUMB_DREG;
	rs		= THUMB_SREG;

	agb_write_mem16(arm.reg[rd], arm.reg[rs] + offset);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_str_sp()
{	/*STR Rd,[SP,#imm] (SP΃XgA) - 11*/
	u16	opcode, offset;
	u32	rd;

	opcode	= arm.opcode16;
	offset	= opcode & 0xFF;
	rd		= (opcode >> 8) & 0x7;

	agb_write_mem32(arm.reg[rd], ARM_SP + offset);

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_swi()
{	/*SWI Value8 (\tgEFA荞) - 17*/
	u16	opcode, offset;

	opcode	= arm.opcode16;
	offset	= opcode & 0xFF;

	arm.cycle = 3;
}

_inline void arm7tdmi_thumb_sub()
{	/*SUB Rd,Rs,Rn (Z) - 2*/
	u16	opcode, offset;
	u32	rd, rs;

	opcode	= arm.opcode16;
	offset	= (opcode >> 3) & 0x7;
	rd		= THUMB_DREG;
	rs		= THUMB_SREG;

	if(opcode & BIT10){	/*1 = I = offset*/
		arm.reg[rd] = arm.reg[rs] - offset;
		if((arm.reg[rd] - offset) > 0)ARM_SF(ARM_C); else ARM_RF(ARM_C);
	}else{					/*0 = I = register*/
		arm.reg[rd] = arm.reg[rs] - arm.reg[offset];
		if((arm.reg[rd] - arm.reg[offset]) > 0)ARM_SF(ARM_C); else ARM_RF(ARM_C);
	}

	arm7tdmi_set_sub_flag(arm.reg[rs], offset, arm.reg[rd]);

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_sub_imm()
{	/*SUB Rd,#Offset8 (Z) - 3*/
	u32	temp32;
	u16	opcode, offset;
	u32	rd;

	opcode	= arm.opcode16;
	offset	= opcode & 0xFF;	/*offset8*/
	rd		= (opcode >> 8) & 0x7;
	
	/*if((arm.reg[rd] - offset)>0)ARM_SF(ARM_C); else ARM_RF(ARM_C);*/

	temp32 = arm.reg[rd] - offset;

	arm7tdmi_set_sub_flag(arm.reg[rd], offset, temp32);

	arm.reg[rd] = temp32;

	arm.cycle = 1;
}

_inline void arm7tdmi_thumb_tst()
{	/*TST Rd,Rs (rbgeXg) - 4*/
	u16	opcode, temp16;
	u32	rd, rs;

	opcode	= arm.opcode16;
	rd		= THUMB_DREG;
	rs		= THUMB_DREG;

	temp16 = arm.reg[rd] & arm.reg[rs];

	arm7tdmi_set_dp_flags(temp16);

	arm.cycle = 1;
}

static int exec_thumb_state()
{
	u16	opcode;
	u8	opcode_11_5, opcode_6_5, opcode_9_3;

	opcode		= THUMB_OPCODE;
	arm.opcode16= opcode;

	/*IyR[hw肳͈͂̃rbgo*/
	opcode_11_5	= (u8)((opcode >> 11) & 0x11);	/*11-15 5bit*/
	opcode_6_5	= (u8)((opcode >> 6) & 0x1F);
	opcode_9_3	= (u8)((opcode >> 6) & 0x07);

	switch(opcode_11_5){
	case 0x00:	/*00000*/
		arm7tdmi_thumb_lsl_imm();/*LSL Rd,Rs,#Offset5 - 1*/
		break;
	case 0x01:	/*00001*/
		arm7tdmi_thumb_lsr_imm();/*LSR Rd,Rs,#Offset5 - 1*/
		break;
	case 0x02:	/*00010*/
		arm7tdmi_thumb_asr_imm();/*ASR Rd,Rs,#Offset5 - 1*/
		break;
	case 0x03:	/*00011*/
		if(opcode & BIT9){
			arm7tdmi_thumb_sub();/*Z - 2*/
		}else{
			arm7tdmi_thumb_add();/*Z - 2*/
		}
		break;
	case 0x04:	/*00100*/
		arm7tdmi_thumb_mov();/*MOV Rd,#Offset8 ړ - 3*/
		break;
	case 0x05:	/*00101*/
		arm7tdmi_thumb_cmp();/*CMP Rd,#Offset8 r - 3*/
		break;
	case 0x06:	/*00110*/
		arm7tdmi_thumb_add_imm();/*ADD Rd,#Offset8 Z - 3*/
		break;
	case 0x07:	/*00111*/
		arm7tdmi_thumb_sub_imm();/*SUB Rd,#Offset8 Z - 3*/
		break;
	case 0x08:	/*01000*/
		switch(opcode_6_5){
		case 0x00:	/*HiWX^/Xe[g*/
		case 0x01:
		case 0x02:
			arm7tdmi_thumb_add_hi();	/*HiWX^ - 5*/
			break;
		case 0x03:
		case 0x04:
		case 0x05:
			arm7tdmi_thumb_cmp_hi();	/*HiWX^ - 5*/
			break;
		case 0x06:
		case 0x07:
		case 0x08:
			arm7tdmi_thumb_mov_hi();	/*HiWX^ - 5*/
			break;
		case 0x09:
		case 0x0a:
		case 0x0b:
			arm7tdmi_thumb_bx_hi();	/*ƃXe[gύX - 5*/
			break;
		case 0x10:	/*ALUZ - 4*/
			arm7tdmi_thumb_and();
			break;
		case 0x11:
			arm7tdmi_thumb_eor();
			break;
		case 0x12:
			arm7tdmi_thumb_lsl();
			break;
		case 0x13:
			arm7tdmi_thumb_lsr();
			break;
		case 0x14:
			arm7tdmi_thumb_asr();
			break;
		case 0x15:
			arm7tdmi_thumb_adc();
			break;
		case 0x16:
			arm7tdmi_thumb_sbc();
			break;
		case 0x17:
			arm7tdmi_thumb_ror();
			break;
		case 0x18:
			arm7tdmi_thumb_tst();
			break;
		case 0x19:
			arm7tdmi_thumb_neg();
			break;
		case 0x1A:
			arm7tdmi_thumb_cmp();
			break;
		case 0x1B:
			arm7tdmi_thumb_cmn();
			break;
		case 0x1C:
			arm7tdmi_thumb_orr();
			break;
		case 0x1D:
			arm7tdmi_thumb_mul();
			break;
		case 0x1E:
			arm7tdmi_thumb_bic();
			break;
		case 0x1F:
			arm7tdmi_thumb_mvn();
			break;
		}
		break;
	case 0x09:	/*01001*/
		arm7tdmi_thumb_ldr_pc();	/*PC΃[h - 6*/
		break;
	case 0x0A:	/*01010*/
	case 0x0B:	/*01011*/
		switch(opcode_9_3){
		/*WX^ItZbgɂ郍[h/XgA - 7*/
		case 0x0:	/*000 LB0*/
			arm7tdmi_thumb_str();
			break;
		case 0x2:	/*010 LB0*/
			arm7tdmi_thumb_strb();
			break;
		case 0x4:	/*100 LB0*/
			arm7tdmi_thumb_ldr();
			break;
		case 0x6:	/*110 LB0*/
			arm7tdmi_thumb_ldrb();
			break;
		/*oCg^n[t[h̃[h^XgAƕg - 8*/
		case 0x1:	/*000 HS0*/
			arm7tdmi_thumb_strh();
			break;
		case 0x3:	/*010 HS0*/
			arm7tdmi_thumb_ldsb();
			break;
		case 0x5:	/*100 HS0*/
			arm7tdmi_thumb_ldrh();
			break;
		case 0x7:	/*110 HS0*/
			arm7tdmi_thumb_ldsh();
			break;
		}
		break;	
	/*C~fBGCgItZbgɂ郍[h^XgA - 9*/
	case 0x0C:	/*01100 - BL=00*/
		arm7tdmi_thumb_str_imm();	/*str rd,[rb,#imm]*/
		break;
	case 0x0D:	/*01101 - BL=01*/
		arm7tdmi_thumb_ldr_imm();	/*ldr rd,[rb,#imm]*/
		break;
	case 0x0E:	/*01110 - BL=10*/
		arm7tdmi_thumb_strb_imm();/*strb rd,[rb,#imm]*/
		break;
	case 0x0F:	/*01111 - BL=11*/
		arm7tdmi_thumb_ldrb_imm();/*ldrb rd,[rb,#imm]*/
		break;
	/*n[t[h̃[h^XgA - 10*/
	case 0x10:	/*10000 - L=0*/
		arm7tdmi_thumb_strh_imm();/*strh rd,[rb,#imm]*/
		break;
	case 0x11:	/*10001 - L=1*/
		arm7tdmi_thumb_ldrh_imm();/*ldrh rd,[rb,#imm]*/
		break;
	/*SP΃[h^XgA - 11*/
	case 0x12:	/*10010 - S=0*/
		arm7tdmi_thumb_str_sp();/*str rd,[SP,#imm]*/
		break;
	case 0x13:	/*10011 - S=1*/
		arm7tdmi_thumb_ldr_sp();/*ldr rd,[SP,#imm]*/
		break;
	/*AhX̃[h - 12*/
	case 0x14:	/*10100 - S=0*//*add rd,PC,#imm*/
	case 0x15:	/*10101 - S=1*//*add rd,SP,#imm*/
		arm7tdmi_thumb_add_adr();
		break;
	case 0x16:	/*10110*/
	case 0x17:	/*10111*/
		if(opcode & BIT10){
	/*WX^PUSH/POP - 14*/
			if(opcode & BIT11){	/*L*/
				arm7tdmi_thumb_pop();/*POP {Rlist}*/
			}else{
				arm7tdmi_thumb_push();/*PUSH {Rlist}*/
			}
		}else{
	/*X^bN|C^ɃItZbgZ - 13*/
			if(!((opcode >> 8) & 0x7)){	/*000S*/
				arm7tdmi_thumb_add_sp();/*add SP,#+-imm*/
			}
			break;
		}
		break;
	/*WX^̃[h^XgA - 15*/
	case 0x18:	/*11000*/
		arm7tdmi_thumb_stmia();/*stmia rb!,{Rlist}*/
		break;
	case 0x19:	/*11001*/
		arm7tdmi_thumb_ldmia();/*ldmia rb!,{Rlist}*/
		break;
	case 0x1A:	/*11010*/
	case 0x1B:	/*11011*/
		if(((opcode >> 8) & 0xF) == 0xF){
	/*\tgEFA荞 - 17*/
			arm7tdmi_thumb_swi();
		}else{
	/* - 16*/
			arm7tdmi_thumb_bxx();
		}
		break;
	/* - 18*/
	case 0x1C:	/*11100*/
		arm7tdmi_thumb_b();
		break;
	/*ƃN - 19*/
	case 0x1E:	/*11110*/
	case 0x1F:	/*11111*/
		arm7tdmi_thumb_bl();
		break;
	default:
		break;
	}

	ARM_PC += 2;

	return 0;
}

