
/*	
	GameboyVM - Nintendo Gameboy Emulator
		Copyright 2002 Y_N y_n@users.sourceforge.jp
		Homepage https://sourceforge.jp/projects/gbemu/
*/


#include "../include/defines.h"


_inline void POP(u16 *val)
{
	*val = RMem16(cpu.SP);
	cpu.SP+=2;
}

_inline void PUSH(u16 val)
{
	cpu.SP-=2;
	WMem16(cpu.SP, val);
}

_inline void ADC(u8 val)
{
	u8 tmp;

	tmp=cpu.A;
	RF(_N);
	if((((tmp) & 0x0F)+(val & 0x0F)+(LSB_FC))>0x0F)SF(_H);else RF(_H);
	cpu.A+=(val)+(LSB_FC);
	if(((tmp)+(val)+(LSB_FC))>0xFF)SF(_C);else RF(_C);
	if(cpu.A)RF(_Z);else SF(_Z);
}

_inline void ADD(u8 val)
{
	RF(_N);
	if((((cpu.A) & 0x0F)+((val) & 0x0F))>0x0F)SF(_H);else RF(_H);
	if(((cpu.A)+(val))>0xFF)SF(_C);else RF(_C);
	cpu.A+=val;
	if(cpu.A)RF(_Z);else SF(_Z);
}

_inline void ADDHL(u16 val)
{
	RF(_N);
	if((((cpu.HL) & 0x0FFF)+((val) & 0x0FFF))>0x0FFF)SF(_H);else RF(_H);
	if(((cpu.HL)+(val))>0xFFFF)SF(_C);else RF(_C);
	cpu.HL+=val;
	if(cpu.HL)RF(_Z);else SF(_Z);
}

_inline void AND(u8 dest)
{
	RF(_N|_C|_H);
	cpu.A&=dest;
	if(cpu.A)RF(_Z);else SF(_Z);
}

_inline void CALL()
{
	u16 tmp16;

	tmp16=GETW;
	cpu.PC+=3;	/*Opcode+Operand*2*/
	cpu.SP-=2;
	WMem16(cpu.SP, cpu.PC);
	cpu.PC=tmp16;
}

_inline void CP(u8 val)
{
	SF(_N);
	if((cpu.A)==val)SF(_Z);else RF(_Z);
	if(((cpu.A) & 0x0F)<((val) & 0x0F))SF(_H);else RF(_H);
	if((cpu.A)<(val))SF(_C);else RF(_C);
}

_inline void DAA()
{
	u8	tmp8, low, high;

	tmp8=0;
	low=(cpu.A) & 0x0F;
	high=(cpu.A) & 0xF0;

	if(FN){
		if(FH && low>=0x06){
			tmp8-=0x06;
			high+=0x10;
		}
		if(FC && high<=0x70)tmp8-=0x60;
	}else{
		if(FH && low<=0x02)tmp8=0x06;
		else if(!FH && low>=0x0A){
			tmp8=0x06;
			high+=0x10;
		}
		if((FC && high<=0x30)||(!FC && high>=0xA0))tmp8+=0x60;
	}

	if(((tmp8&0x0F)+(cpu.A&0x0F))>0x0F)SF(_H);else RF(_H);
	if((tmp8+(cpu.A))>0xFF)SF(_C);else RF(_C);
	cpu.A+=tmp8;
	if(cpu.A)RF(_Z);else SF(_Z);
}

#define DECB(dest)	SF(_N);if((dest) & 0x0F)RF(_H);else SF(_H);(dest)--;\
					if(dest)RF(_Z);else SF(_Z)
/*
_inline void DECB(u8 *dest){
	SF(_N);
	if((*dest) & 0x0F)RF(_H);else SF(_H);
	(*dest)--;
	if(*dest)RF(_Z);else SF(_Z);
}*/
/*
_inline void DECW(u8 *dest){
	(*dest)--;
}

_inline void INCW(u8 *dest){
	(*dest)++;
}
*/
_inline void JP()
{
	cpu.PC=GETW;
}

_inline void JR()
{
	cpu.PC+=(s8)GETB;
}

/*_inline void INCB(u8 *dest){
	RF(_N);
	if((*dest) & 0x0F)RF(_H);else SF(_H);
	(*dest)++;
	if(*dest)RF(_Z);else SF(_Z);
}
*/
#define INCB(dest)	RF(_N);if((dest) & 0x0F)RF(_H);else SF(_H);\
					(dest)++;if(dest)RF(_Z);else SF(_Z)

_inline void OR(u8 val)
{
	RF(_N|_H|_C);
	cpu.A|=val;
	if(cpu.A)RF(_Z);else SF(_Z);
}

_inline void RET()
{
	cpu.PC = RMem(cpu.SP)+((RMem((u16)(cpu.SP+1)))<<8);
	cpu.SP+=2;
}

_inline void RST(u8 val)
{
	PUSH(cpu.PC+1);
	cpu.PC = val;
}

_inline void SBC(u8 val)
{
	u8	tmp;

	tmp = cpu.A;
	SF(_N);
	if((tmp & 0x0F)<((val+(LSB_FC)) & 0x0F))SF(_H);else RF(_H);
	cpu.A-=(val)+(LSB_FC);
	if((tmp)<(val+(LSB_FC)))SF(_C);else RF(_C);
	if(cpu.A)RF(_Z);else SF(_Z);
}

_inline void SUB(u8 dest)
{
	SF(_N);
	if((cpu.A & 0x0F)<(dest & 0x0F))SF(_H);else RF(_H);
	if(cpu.A<dest)SF(_C);else RF(_C);
	cpu.A-=dest;
	if(cpu.A)RF(_Z);else SF(_Z);
}

_inline void XOR(u8 dest)
{
	RF(_N|_H|_C);
	cpu.A^=dest;
	if(cpu.A)RF(_Z);else SF(_Z);
}

_inline void UndefinedOpcode()
{
	u8 opcode;

	opcode = RMem(cpu.PC);

	CPUIsRunning=0;
#ifdef	_WIN32_GDI
		CString message;
		message.Format(IDS_UNDEFINE_OPCODE, opcode, cpu.PC, rom_mbc);
		GBEmu.MessageBox(message, NULL, MB_ICONSTOP);
#endif	/*_WIN32_GDI*/
}

/*m[}R[hTCNe[u*/
u8 Cycles[256]={
	 4,12, 8, 8, 4, 4, 8, 4,20, 8, 8, 8, 4, 4, 8, 4,
	 4,12, 8, 8, 4, 4, 8, 4, 8, 8, 8, 8, 4, 4, 8, 4,
	 8,12, 8, 8, 4, 4, 8, 4, 8, 8, 8, 8, 4, 4, 8, 4,
	 8,12, 8, 8,12,12,12, 4, 8, 8, 8, 8, 4, 4, 8, 4,
	 4, 4, 4, 4, 4, 4, 8, 4, 4, 4, 4, 4, 4, 4, 8, 4,
	 4, 4, 4, 4, 4, 4, 8, 4, 4, 4, 4, 4, 4, 4, 8, 4,
	 4, 4, 4, 4, 4, 4, 8, 4, 4, 4, 4, 4, 4, 4, 8, 4,
	 8, 8, 8, 8, 8, 8, 4, 8, 4, 4, 4, 4, 4, 4, 8, 4,
	 4, 4, 4, 4, 4, 4, 8, 4, 4, 4, 4, 4, 4, 4, 8, 4,
	 4, 4, 4, 4, 4, 4, 8, 4, 4, 4, 4, 4, 4, 4, 8, 4,
	 4, 4, 4, 4, 4, 4, 8, 4, 4, 4, 4, 4, 4, 4, 8, 4,
	 4, 4, 4, 4, 4, 4, 8, 4, 4, 4, 4, 4, 4, 4, 8, 4,
	 8, 8,12,12,12,16, 8,32, 8, 8,12, 0,12,12, 8,32,
	 8, 8,12, 0,12,16, 8,32, 8, 8,12, 0,12, 0, 8,32,
	12, 8, 8, 0, 0,16, 8,32,16, 4,16, 0, 0, 0, 8,32, 
	12, 8, 8, 4, 0,16, 8,32,12, 8,16, 4, 0, 0, 8,32,
};

/*CBvtBbNXR[hTCNe[u*/
u8 Cycles_cb[256]={
	 8, 8, 8, 8, 8, 8,16, 8, 8, 8, 8, 8, 8, 8,16, 8,
	 8, 8, 8, 8, 8, 8,16, 8, 8, 8, 8, 8, 8, 8,16, 8,
	 8, 8, 8, 8, 8, 8,16, 8, 8, 8, 8, 8, 8, 8,16, 8,
	 8, 8, 8, 8, 8, 8,16, 8, 8, 8, 8, 8, 8, 8,16, 8,
	 8, 8, 8, 8, 8, 8,12, 8, 8, 8, 8, 8, 8, 8,12, 8,
	 8, 8, 8, 8, 8, 8,12, 8, 8, 8, 8, 8, 8, 8,12, 8,
	 8, 8, 8, 8, 8, 8,12, 8, 8, 8, 8, 8, 8, 8,12, 8,
	 8, 8, 8, 8, 8, 8,12, 8, 8, 8, 8, 8, 8, 8,12, 8,
	 8, 8, 8, 8, 8, 8,16, 8, 8, 8, 8, 8, 8, 8,16, 8,
	 8, 8, 8, 8, 8, 8,16, 8, 8, 8, 8, 8, 8, 8,16, 8,
	 8, 8, 8, 8, 8, 8,16, 8, 8, 8, 8, 8, 8, 8,16, 8,
	 8, 8, 8, 8, 8, 8,16, 8, 8, 8, 8, 8, 8, 8,16, 8,
	 8, 8, 8, 8, 8, 8,16, 8, 8, 8, 8, 8, 8, 8,16, 8,
	 8, 8, 8, 8, 8, 8,16, 8, 8, 8, 8, 8, 8, 8,16, 8,
	 8, 8, 8, 8, 8, 8,16, 8, 8, 8, 8, 8, 8, 8,16, 8,
	 8, 8, 8, 8, 8, 8,16, 8, 8, 8, 8, 8, 8, 8,16, 8,
};

/*IyR[hTCYe[u(Wv߂͂O)*/
u8 PC_table[256]={
	1,3,1,1,1,1,2,1,3,1,1,1,1,1,2,1,/*0*/
	2,3,1,1,1,1,2,1,2,1,1,1,1,1,2,1,/*1*/
	2,3,1,1,1,1,2,1,2,1,1,1,1,1,2,1,/*2*/
	2,3,1,1,1,1,2,1,2,1,1,1,1,1,2,1,/*3*/
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,/*4*/
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,/*5*/
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,/*6*/
	1,1,1,1,1,1,0,1,1,1,1,1,1,1,1,1,/*7*/
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,/*8*/
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,/*9*/
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,/*A*/
	1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,/*B*/
	0,1,0,0,0,1,2,0,0,0,0,0,0,0,2,0,/*C*/
	0,1,0,0,0,1,2,0,0,0,0,0,0,0,2,0,/*D*/
	2,1,1,0,0,1,2,0,2,0,3,0,0,0,2,0,/*E*/
	2,1,1,1,0,1,2,0,2,1,3,1,0,0,2,0,/*F*/
/*	0,1,2,3,4,5,6,7,8,9,A,B,C,D,E,F*/
};


_inline void Cpu(){
	static u16 tmp16;
	static u8 tmp8;
	static u8 opcode;

	opcode = RMem(cpu.PC);

	switch(opcode){

	case 0x00 : break;						/*NOP*/
	case 0x01 : cpu.BC=GETW;break;			/*LD BC nn*/
	case 0x02 : WMem(cpu.BC,cpu.A);break;	/*LD (BC),A*/
	case 0x03 : cpu.BC++;break;				/*INC BC*/
	case 0x04 : INCB(cpu.B);break;			/*INC B*/
	case 0x05 : DECB(cpu.B);break;			/*DEC B*/
	case 0x06 : cpu.B=GETB;break;			/*LD B,n*/
	case 0x07 : RF(_N);RF(_H);
		if(cpu.A & 0x80)SF(_C);else RF(_C);
		cpu.A=((cpu.A<<1)|(cpu.A>>7));
		if(cpu.A)RF(_Z);else SF(_Z);break;	/*RLCA*/
	case 0x08 : tmp16=GETW;
		WMem16(tmp16, cpu.SP);break;		/*LD (nn),SP*/
	case 0x09 : ADDHL(cpu.BC);break;		/*ADD HL,BC*/
	case 0x0A : cpu.A=RMem(cpu.BC);break;	/*LD A,(BC)*/
	case 0x0B : cpu.BC--;break;				/*DEC BC*/
	case 0x0C : INCB(cpu.C);break;			/*INC C*/
	case 0x0D : DECB(cpu.C);break;			/*DEC C*/
	case 0x0E : cpu.C=GETB;break;			/*LD C,n*/
	case 0x0F : RF(_N|_H);
		if(cpu.A & 0x01)SF(_C);else RF(_C);
		cpu.A=((cpu.A>>1)|(cpu.A<<7));
		if(cpu.A)RF(_Z);else SF(_Z);break;	/*RRCA*/

	case 0x10 : if(GETB==0x00)cpu.stop=1;break;	/*STOP*/
	case 0x11 : cpu.DE=GETW;break;			/*LD DE,nn*/
	case 0x12 : WMem(cpu.DE,cpu.A);break;	/*LD (DE),A*/
	case 0x13 : cpu.DE++;break;				/*INC DE*/
	case 0x14 : INCB(cpu.D);break;			/*INC D*/
	case 0x15 : DECB(cpu.D);break;			/*DEC D*/
	case 0x16 : cpu.D=GETB;break;			/*LD D,n*/
	case 0x17 : tmp8=cpu.A;
		RF(_N|_H);
		cpu.A=((cpu.A<<1)|(LSB_FC));
		if(tmp8 & 0x80)SF(_C);else RF(_C);
		if(cpu.A)RF(_Z);else SF(_Z);break;	/*RLA*/
	case 0x18 : JR();break;					/*JR n*/
	case 0x19 : ADDHL(cpu.DE);break;		/*ADD HL,DE*/
	case 0x1A : cpu.A=RMem(cpu.DE);break;	/*LD A,(DE)*/
	case 0x1B : cpu.DE--;break;				/*DEC DE*/
	case 0x1C : INCB(cpu.E);break;			/*INC E*/
	case 0x1D : DECB(cpu.E);break;			/*DEC E*/
	case 0x1E : cpu.E=GETB;break;			/*LD E,n*/
	case 0x1F : tmp8=cpu.A;
		RF(_N|_H);
		cpu.A=(u8)((cpu.A>>1)|(MSB_FC));
		if(tmp8 & 0x01)SF(_C);else RF(_C);
		if(cpu.A)RF(_Z);else SF(_Z);break;	/*RRA*/

	case 0x20 : if(!FZ)JR();break;			/*JR nz*/
	case 0x21 : cpu.HL=GETW;break;			/*LD HL,nn*/
	case 0x22 : WMem(cpu.HL,cpu.A);
		cpu.HL++;break;						/*LDI (HL),A*/
	case 0x23 : cpu.HL++;break;				/*INC HL*/
	case 0x24 : INCB(cpu.H);break;			/*INC H*/
	case 0x25 : DECB(cpu.H);break;			/*DEC H*/
	case 0x26 : cpu.H=GETB;break;			/*LD H,n*/
	case 0x27 : DAA();break;				/*DAA*/
	case 0x28 : if(FZ)JR();break;			/*JR z*/
	case 0x29 : ADDHL(cpu.HL);break;		/*ADD HL,HL*/
	case 0x2A : cpu.A=RMem(cpu.HL);
		cpu.HL++;break;						/*LDI A,(HL)*/
	case 0x2B : cpu.HL--;break;				/*DEC HL*/
	case 0x2C : INCB(cpu.L);break;			/*INC L*/
	case 0x2D : DECB(cpu.L);break;			/*DEC L*/
	case 0x2E : cpu.L=GETB;break;			/*LD L,n*/
	case 0x2F : SF(_N|_H);
		cpu.A=~cpu.A;break;					/*CPL*/

	case 0x30 : if(!FC)JR();break;			/*JR nc*/
	case 0x31 : cpu.SP=GETW;break;			/*LD SP,nn*/
	case 0x32 : WMem(cpu.HL,cpu.A);
		cpu.HL--;break;						/*LDD (HL),A*/
	case 0x33 : cpu.SP++;break;				/*INC SP*/
	case 0x34 : tmp8=RMem(cpu.HL);
		RF(_N);
		if((tmp8 & 0x0F)==0x0F)SF(_H);else RF(_H);
		tmp8++;if(tmp8)RF(_Z);else SF(_Z);
		WMem(cpu.HL,tmp8);break;			/*INC (HL)*/
	case 0x35 :	tmp8=RMem(cpu.HL);
		SF(_N);if(tmp8 & 0x0F)RF(_H);else SF(_H);
		tmp8--;if(tmp8)RF(_Z);else SF(_Z);
		WMem(cpu.HL,tmp8);break;			/*DEC (HL)*/
	case 0x36 : WMem(cpu.HL,GETB);break;	/*LD (HL),n*/
	case 0x37 : RF(_N|_H);SF(_C);break;		/*SCF*/
	case 0x38 : if(FC)JR();break;			/*JR C*/
	case 0x39 : ADDHL(cpu.SP);break;		/*ADD HL,SP*/
	case 0x3A : cpu.A=RMem(cpu.HL);
		cpu.HL--;break;						/*LDD A,(HL)*/
	case 0x3B : cpu.SP--;break;				/*DEC SP*/
	case 0x3C : INCB(cpu.A);break;			/*INC A*/
	case 0x3D : DECB(cpu.A);break;			/*DEC A*/
	case 0x3E : cpu.A=GETB;break;			/*LD A,n*/
	case 0x3F : RF(_N|_H);
		if(FC)SF(_H);else RF(_H);
		if(FC)RF(_C);else SF(_C);
		break;								/*CCF L[tO̔]*/

	case 0x40 : /*cpu.B=cpu.B;*/break;
	case 0x41 : cpu.B=cpu.C;break;
	case 0x42 : cpu.B=cpu.D;break;
	case 0x43 : cpu.B=cpu.E;break;
	case 0x44 : cpu.B=cpu.H;break;
	case 0x45 : cpu.B=cpu.L;break;
	case 0x46 : cpu.B=RMem(cpu.HL);break;
	case 0x47 : cpu.B=cpu.A;break;

	case 0x48 : cpu.C=cpu.B;break;
	case 0x49 : /*cpu.C=cpu.C;*/break;
	case 0x4A : cpu.C=cpu.D;break;
	case 0x4B : cpu.C=cpu.E;break;
	case 0x4C : cpu.C=cpu.H;break;
	case 0x4D : cpu.C=cpu.L;break;
	case 0x4E : cpu.C=RMem(cpu.HL);break;
	case 0x4F : cpu.C=cpu.A;break;

	case 0x50 : cpu.D=cpu.B;break;
	case 0x51 : cpu.D=cpu.C;break;
	case 0x52 : /*cpu.D=cpu.D;*/break;
	case 0x53 : cpu.D=cpu.E;break;
	case 0x54 : cpu.D=cpu.H;break;
	case 0x55 : cpu.D=cpu.L;break;
	case 0x56 : cpu.D=RMem(cpu.HL);break;
	case 0x57 : cpu.D=cpu.A;break;

	case 0x58 : cpu.E=cpu.B;break;
	case 0x59 : cpu.E=cpu.C;break;
	case 0x5A : cpu.E=cpu.D;break;
	case 0x5B : /*cpu.E=cpu.E;*/break;
	case 0x5C : cpu.E=cpu.H;break;
	case 0x5D : cpu.E=cpu.L;break;
	case 0x5E : cpu.E=RMem(cpu.HL);break;
	case 0x5F : cpu.E=cpu.A;break;

	case 0x60 : cpu.H=cpu.B;break;
	case 0x61 : cpu.H=cpu.C;break;
	case 0x62 : cpu.H=cpu.D;break;
	case 0x63 : cpu.H=cpu.E;break;
	case 0x64 : /*cpu.H=cpu.H;*/break;
	case 0x65 : cpu.H=cpu.L;break;
	case 0x66 : cpu.H=RMem(cpu.HL);break;
	case 0x67 : cpu.H=cpu.A;break;

	case 0x68 : cpu.L=cpu.B;break;
	case 0x69 : cpu.L=cpu.C;break;
	case 0x6A : cpu.L=cpu.D;break;
	case 0x6B : cpu.L=cpu.E;break;
	case 0x6C : cpu.L=cpu.H;break;
	case 0x6D : /*cpu.L=cpu.L;*/break;
	case 0x6E : cpu.L=RMem(cpu.HL);break;
	case 0x6F : cpu.L=cpu.A;break;

	case 0x70 : WMem(cpu.HL,cpu.B);break;
	case 0x71 : WMem(cpu.HL,cpu.C);break;
	case 0x72 : WMem(cpu.HL,cpu.D);break;
	case 0x73 : WMem(cpu.HL,cpu.E);break;
	case 0x74 : WMem(cpu.HL,cpu.H);break;
	case 0x75 : WMem(cpu.HL,cpu.L);break;
	case 0x76 : cpu.halt=cpu.ime=1;break;	/*HALT*/
	case 0x77 : WMem(cpu.HL,cpu.A);break;

	case 0x78 : cpu.A=cpu.B;break;
	case 0x79 : cpu.A=cpu.C;break;
	case 0x7A : cpu.A=cpu.D;break;
	case 0x7B : cpu.A=cpu.E;break;
	case 0x7C : cpu.A=cpu.H;break;
	case 0x7D : cpu.A=cpu.L;break;
	case 0x7E : cpu.A=RMem(cpu.HL);break;
	case 0x7F : /*cpu.A=cpu.A;*/break;

	case 0x80 : ADD(cpu.B);break;
	case 0x81 : ADD(cpu.C);break;
	case 0x82 : ADD(cpu.D);break;
	case 0x83 : ADD(cpu.E);break;
	case 0x84 : ADD(cpu.H);break;
	case 0x85 : ADD(cpu.L);break;
	case 0x86 : tmp8=RMem(cpu.HL);ADD(tmp8);break;
	case 0x87 : ADD(cpu.A);break;
	
	case 0x88 : ADC(cpu.B);break;
	case 0x89 : ADC(cpu.C);break;
	case 0x8A : ADC(cpu.D);break;
	case 0x8B : ADC(cpu.E);break;
	case 0x8C : ADC(cpu.H);break;
	case 0x8D : ADC(cpu.L);break;
	case 0x8E : tmp8=RMem(cpu.HL);ADC(tmp8);break;
	case 0x8F : ADC(cpu.A);break;

	case 0x90 :	SUB(cpu.B);break;
	case 0x91 :	SUB(cpu.C);break;
	case 0x92 :	SUB(cpu.D);break;
	case 0x93 :	SUB(cpu.E);break;
	case 0x94 :	SUB(cpu.H);break;
	case 0x95 :	SUB(cpu.L);break;
	case 0x96 :	tmp8=RMem(cpu.HL);SUB(tmp8);break;
	case 0x97 :	SUB(cpu.A);break;

	case 0x98 :	SBC(cpu.B);break;
	case 0x99 :	SBC(cpu.C);break;
	case 0x9A :	SBC(cpu.D);break;
	case 0x9B :	SBC(cpu.E);break;
	case 0x9C :	SBC(cpu.H);break;
	case 0x9D :	SBC(cpu.L);break;
	case 0x9E :	tmp8=RMem(cpu.HL);SBC(tmp8);break;
	case 0x9F :	SBC(cpu.A);break;

	case 0xA0 : AND(cpu.B);break;
	case 0xA1 : AND(cpu.C);break;
	case 0xA2 : AND(cpu.D);break;
	case 0xA3 : AND(cpu.E);break;
	case 0xA4 : AND(cpu.H);break;
	case 0xA5 : AND(cpu.L);break;
	case 0xA6 : tmp8=RMem(cpu.HL);AND(tmp8);break;
	case 0xA7 : AND(cpu.A);break;

	case 0xA8 :	XOR(cpu.B);break;
	case 0xA9 :	XOR(cpu.C);break;
	case 0xAA :	XOR(cpu.D);break;
	case 0xAB :	XOR(cpu.E);break;
	case 0xAC :	XOR(cpu.H);break;
	case 0xAD :	XOR(cpu.L);break;
	case 0xAE :	tmp8=RMem(cpu.HL);XOR(tmp8);break;
	case 0xAF :	XOR(cpu.A);break;

	case 0xB0 :	OR(cpu.B);break;
	case 0xB1 :	OR(cpu.C);break;
	case 0xB2 :	OR(cpu.D);break;
	case 0xB3 :	OR(cpu.E);break;
	case 0xB4 :	OR(cpu.H);break;
	case 0xB5 :	OR(cpu.L);break;
	case 0xB6 :	tmp8=RMem(cpu.HL);OR(tmp8);break;
	case 0xB7 :	OR(cpu.A);break;
	
	case 0xB8 :	CP(cpu.B);break;
	case 0xB9 :	CP(cpu.C);break;
	case 0xBA :	CP(cpu.D);break;
	case 0xBB :	CP(cpu.E);break;
	case 0xBC :	CP(cpu.H);break;
	case 0xBD :	CP(cpu.L);break;
	case 0xBE :	tmp8=RMem(cpu.HL);
		CP(tmp8);break;						/*CP A,(HL)*/
	case 0xBF :	CP(cpu.A);break;			/*CP A,A*/

	case 0xC0 : if(!FZ){RET();}else cpu.PC++;break;		/*RET nz*/
	case 0xC1 : POP(&cpu.BC);break;			/*POP BC*/
	case 0xC2 : if(!FZ){JP();} else cpu.PC+=3;break;	/*JP nz*/
	case 0xC3 : JP();break;					/*JP*/
	case 0xC4 : if(!FZ){CALL();}else cpu.PC+=3;break;	/*CALL nz,nn*/
	case 0xC5 : PUSH(cpu.BC);break;			/*PUSH BC*/
	case 0xC6 : tmp8=GETB;ADD(tmp8);break;	/*ADD A,n*/
	case 0xC7 : RST(0x00);break;			/*RST 00h*/
	case 0xC8 : if(FZ){RET();}else cpu.PC++;break;		/*RET z*/
	case 0xC9 : RET();break;				/*RET*/
	case 0xCA : if(FZ){JP();} else cpu.PC+=3;break;		/*JP z,nn*/
	case 0xCB : UndefinedOpcode();break;	/*CB prefix*/
	case 0xCC : if(FZ){CALL();}else cpu.PC+=3;break;	/*CALL z,nn*/
	case 0xCD : CALL();break;				/*CALL*/
	case 0xCE : tmp8=GETB;ADC(tmp8);break;	/*ADC A,n*/
	case 0xCF : RST(0x08);break;			/*RST 08h*/

	case 0xD0 : if(!FC){RET();}else cpu.PC++; break;	/*RET nc*/
	case 0xD1 : POP(&cpu.DE);break;			/*POP DE*/
	case 0xD2 : if(!FC){JP();} else cpu.PC+=3; break;	/*JP nc,nn*/
	case 0xD3 : UndefinedOpcode();break;	/*`*/
	case 0xD4 : if(!FC){CALL();}else cpu.PC+=3;break;	/*CALL nc,nn*/
	case 0xD5 : PUSH(cpu.DE);break;			/*PUSH DE*/
	case 0xD6 :	tmp8=GETB;SUB(tmp8);break;	/*SUB A,n*/
	case 0xD7 : RST(0x10);break;			/*RST 10h*/
	case 0xD8 : if(FC){RET();}else cpu.PC++; break;		/*RET c*/
	case 0xD9 : RET();cpu.ime=1;EndInterrupt();break;	/*RETI*/
	case 0xDA : if(FC){JP();} else cpu.PC+=3; break;	/*JP c,nn*/
	case 0xDB : UndefinedOpcode();break;	/*`*/
	case 0xDC : if(FC){CALL();}else cpu.PC+=3; break;	/*CALL c,nn*/
	case 0xDD : UndefinedOpcode();break;	/*`*/
	case 0xDE :	tmp8=GETB;SBC(tmp8);break;	/*SBC A,n*/
	case 0xDF : RST(0x18);break;			/*RST 18h*/

	case 0xE0 :	WMem((u16)(0xFF00+GETB),cpu.A);break;	/*LD (ff00+n),A*/
	case 0xE1 : POP(&cpu.HL);break;			/*POP HL*/
	case 0xE2 : WMem((u16)(0xFF00+(cpu.C)),cpu.A);break;/*LD (ff00+C),A*/
	case 0xE3 : UndefinedOpcode();break;	/*`*/
	case 0xE4 : UndefinedOpcode();break;	/*`*/
	case 0xE5 : PUSH(cpu.HL);break;			/*PUSH HL*/
	case 0xE6 : tmp8=GETB;AND(tmp8);break;	/*AND A,n*/
	case 0xE7 : RST(0x20);break;			/*RST 20h*/
	case 0xE8 : tmp8=GETB;
		RF(_Z|_N|_H);
		if((cpu.SP+(s8)tmp8)>0xFFFF)SF(_C);else RF(_C);
		cpu.SP+=(s8)tmp8;break;				/*ADD SP,n(tItZbg)*/
	case 0xE9 : cpu.PC=cpu.HL;break;		/*JP HL*/
	case 0xEA : WMem(GETW,cpu.A);break;		/*LD (nn),A*/
	case 0xEB : UndefinedOpcode();break;	/*`*/
	case 0xEC : UndefinedOpcode();break;	/*`*/
	case 0xED : UndefinedOpcode();break;	/*`*/
	case 0xEE :	tmp8=GETB;XOR(tmp8);break;	/*XOR A,n*/
	case 0xEF : RST(0x28);break;			/*RST 28h*/

	case 0xF0 : cpu.A=RMem((u16)(0xFF00+GETB));break;	/*LD A,(n)*/
	case 0xF1 : POP(&cpu.AF);break;			/*POP AF*/
	case 0xF2 : cpu.A=RMem((u16)(0xFF00+(cpu.C)));break;/*LD A,(FF00h+C)*/
	case 0xF3 : cpu.ime=0;break;			/*DI*/
	case 0xF4 : UndefinedOpcode();break;	/*`*/
	case 0xF5 : PUSH(cpu.AF);break;			/*PUSH AF*/
	case 0xF6 :	OR(GETB);break;				/*OR A,n*/
	case 0xF7 : RST(0x30);break;			/*RST 30h*/
	case 0xF8 : tmp8=GETB;
		RF(_Z|_N|_H);
		if(((cpu.SP)+(s8)tmp8)>0xFFFF)SF(_C);else RF(_C);
		cpu.HL=(cpu.SP)+(s8)tmp8;break;		/*LD HL SP+n(tItZbg)*/
	case 0xF9 : cpu.SP=cpu.HL;break;		/*LD SP,HL*/
	case 0xFA : cpu.A=RMem(GETW);break;		/*LD A,(nn)*/
	case 0xFB : cpu.ime=1;EndInterrupt();break;			/*EI*/
	case 0xFC :	UndefinedOpcode();break;	/*`*/
	case 0xFD :	UndefinedOpcode();break;	/*`*/
	case 0xFE :	tmp8=GETB;CP(tmp8);break;
	case 0xFF : RST(0x38);break;			/*RST 38h*/

#ifdef	_WIN32_GDI
	default   : __assume(0);
#endif	/*_WIN32_GDI*/

	}

}

