# -*- coding: ascii -*-
#
#  logwindow.py - Log window for GBottler
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: logwindow.py,v 1.20 2004/09/27 22:39:07 atzm Exp $
#

import gtk
import os, time

from logfetchdialog import LogFetchDialog
from textmanager    import TextManager
from logbook        import LogBook
from loglist        import LogList
from logmanager     import XMLLogWriter, XMLLogOpener, LogFetchOperator

from common import *

class LogWindow:
	def __init__(self, app):
# === Definition of Basic Const === #
		self.app    = app
		accel_group = gtk.AccelGroup()

		self.set_self_msg()
		self.volley_canceler = False

# === Create Base Window === #
		self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
		self.window.add_accel_group(accel_group)
		self.window.connect("delete_event", self.close)
		self.window.set_title(unicode(_(APP), "utf-8") + ":" + \
							  unicode(_("Logs"), "utf-8"))
		self.window.set_size_request(450,300)

# === Create Entry for Search === #
		self.search_entry = gtk.Entry()
		self.search_entry.show()
		self.search_entry.set_text(self.msg['search'])
		self.search_entry.connect("activate", self.search)
		self.search_entry.connect("key-press-event", self.entry_keypress)
		self.search_entry.connect("focus-in-event", self.set_entry_guide, 'in')
		self.search_entry.connect("focus-out-event", self.set_entry_guide, 'out')

		self.s_arrow = gtk.Arrow(gtk.ARROW_DOWN, gtk.SHADOW_NONE)
		self.s_arrow.show()

		self.s_a_b = gtk.ToggleButton()
		self.s_a_b.show()
		self.s_a_b.add(self.s_arrow)
		self.s_a_b.connect('clicked',
						   lambda x: self.s_arrow.set((not self.s_arrow.get_property('arrow-type')),
													  gtk.SHADOW_NONE))
		self.s_a_b.connect('focus-in-event', lambda x, y: self.search_entry.grab_focus())

		self.sbar = gtk.Statusbar()
		self.sbar.show()
		self.sbar.set_has_resize_grip(gtk.TRUE)

		self.rowsbar = gtk.Statusbar()
		self.rowsbar.show()
		self.rowsbar.set_has_resize_grip(gtk.FALSE)

		self.curr_rowsbar = gtk.Statusbar()
		self.curr_rowsbar.show()
		self.curr_rowsbar.set_has_resize_grip(gtk.FALSE)

		search_hbox = gtk.HBox()
		search_hbox.show()
		search_hbox.pack_start(self.s_a_b, gtk.FALSE, gtk.FALSE, 0)
		search_hbox.pack_start(self.search_entry)
		search_hbox.pack_start(self.curr_rowsbar)
		search_hbox.pack_start(self.rowsbar)
		search_hbox.pack_start(self.sbar)

# === Create MessageBar === #
		btn = get_icon_button(gtk.STOCK_CLEAR, size=gtk.ICON_SIZE_MENU,
							  relief=gtk.RELIEF_NONE)
		btn.show()
		btn.connect('clicked', lambda x: self.clear_message())
		btn.connect('focus-in-event', self.focus_active_loglist)

		self.messagebar = gtk.Label()
		self.messagebar.show()
		msgbox = gtk.HBox()
		msgbox.show()
		msgbox.pack_start(self.messagebar)
		msgbox.pack_start(btn, gtk.FALSE)

		self.msgframe = gtk.Frame()
		self.msgframe.show()
		self.msgframe.set_shadow_type(gtk.SHADOW_ETCHED_IN)
		self.msgframe.add(msgbox)

		bottombox = gtk.VBox()
		bottombox.show()
		bottombox.pack_start(self.msgframe)
		bottombox.pack_start(search_hbox)

# === Create Menubar === #
		menu_items = self.get_menu_items()
		# can't set menu_items before set entry

		self.item_factory = gtk.ItemFactory(gtk.MenuBar, "<main>", accel_group)
		self.item_factory.create_items(menu_items)

		main_menubar = self.item_factory.get_widget("<main>")
		main_menubar.show()

# === Create Popup Menu for LogList === #
		p_menu = (
			( "/%s" % unicode(_("Play")), "<control>p", self.play, 0, "<StockItem>", gtk.STOCK_REFRESH ),
			( "/%s" % unicode(_("Volley")), None, self.volley, 0, None ),
			( '/sep1', None, None, 0, "<Separator>" ),
			( "/%s" % unicode(_("Cancel to play")), None, self.cancel_play, 0, "<StockItem>", gtk.STOCK_CANCEL ),
			( "/%s" % unicode(_("Cancel to play all at this page")), None, self.cancel_play_page, 0, "<StockItem>", gtk.STOCK_CANCEL ),
			( "/%s" % unicode(_("Cancel to play all")), None, self.cancel_play_all, 0, "<StockItem>", gtk.STOCK_CANCEL ),
			( '/sep2', None, None, 0, "<Separator>" ),
			( "/%s" % unicode(_("Vote")), "<control><alt>v", self.voting, 0, "<StockItem>", gtk.STOCK_YES ),
			( "/%s" % unicode(_("Agree")), "<control><alt>a", self.agreeing, 0, "<StockItem>", gtk.STOCK_NO ),
			( '/sep3', None, None, 0, "<Separator>" ),
			( "/%s" % self.msg["copy time"], None, self.edit_copy_time, 0, None ),
			( "/%s" % self.msg["copy ghostname"], None, self.edit_copy_ghostname, 0, None ),
			( "/%s" % self.msg["copy channel"], None, self.edit_copy_channel, 0, None ),
			( "/%s" % self.msg["copy script"], "<control>c", self.edit_copy_script, 0, "<StockItem>", gtk.STOCK_COPY ),
			( "/%s" % self.msg["copy all"], None, self.edit_copy_all, 0, None ),
			)
		self.p_ifact = gtk.ItemFactory(gtk.Menu, "<main>", accel_group)
		self.p_ifact.create_items(p_menu)
		self.popup_menu = self.p_ifact.get_widget("<main>")

		# keep MenuItems, vote and agree
		self.vi = self.p_ifact.get_item("/%s" % unicode(_("Vote"), "utf-8"))
		self.ai = self.p_ifact.get_item("/%s" % unicode(_("Agree"), "utf-8"))

# === Create Log Viewer === #
		self.textmanager = TextManager(self, self.app.prefs)
		self.textmanager.set_style_talk()
		self.textmanager.set_cursor_visible(gtk.FALSE)
		self.textmanager.set_editable(gtk.FALSE)
		self.textmanager.set_border_width(0)
		self.textmanager.connect("focus-in-event", self.focus_active_loglist)
		self.textmanager.show()

		self.sw2 = gtk.ScrolledWindow()
		self.sw2.show()
		self.sw2.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		self.sw2.set_border_width(0)
		self.sw2.add(self.textmanager)

		text_frame = gtk.Frame()
		text_frame.show()
		text_frame.set_shadow_type(gtk.SHADOW_IN)
		text_frame.set_border_width(0)
		text_frame.add(self.sw2)

# === Create Buttons === #
		fetch_button = get_icon_button(gtk.STOCK_CONVERT, gtk.ICON_SIZE_MENU)
		open_button  = get_icon_button(gtk.STOCK_OPEN,    gtk.ICON_SIZE_MENU)
		save_button  = get_icon_button(gtk.STOCK_SAVE_AS, gtk.ICON_SIZE_MENU)

		self.pb = get_icon_button(gtk.STOCK_REFRESH, gtk.ICON_SIZE_MENU)
		self.vb = get_icon_button(gtk.STOCK_YES,     gtk.ICON_SIZE_MENU)
		self.ab = get_icon_button(gtk.STOCK_NO,      gtk.ICON_SIZE_MENU)

		fetch_button.connect("clicked", self.fetch)
		open_button.connect("clicked", self.open_xml_log)
		save_button.connect("clicked", self.save_as_xml)

		self.pb.connect("clicked", self.play, None)
		self.vb.connect("clicked", self.voting, None)
		self.ab.connect("clicked", self.agreeing, None)

		self.vb.set_sensitive(gtk.FALSE)
		self.ab.set_sensitive(gtk.FALSE)

		for item in [fetch_button, open_button, save_button,
					 self.pb, self.vb, self.ab]:
			item.connect("focus-in-event", self.focus_active_loglist)

		self.tooltips = gtk.Tooltips()
		self.tooltips.set_tip(fetch_button, self.msg['fetch'])
		self.tooltips.set_tip(open_button, self.msg['open_xml'])
		self.tooltips.set_tip(save_button, self.msg['save'])

		self.tooltips.set_tip(self.pb, unicode(_("Play"), "utf-8"))
		self.tooltips.set_tip(self.vb, unicode(_("Vote"), "utf-8"))
		self.tooltips.set_tip(self.ab, unicode(_("Agree"), "utf-8"))

# === Create Labels and style === #
		style_desc = gtk.Label(unicode(_("Style"), "utf-8") + ": ")
		style_desc.show()

		self.style_l = gtk.Label(self.msg['talk style'])
		self.style_l.show()

		self.s_e_box = gtk.EventBox()
		self.s_e_box.show()
		self.s_e_box.set_events(gtk.gdk.BUTTON_PRESS_MASK)
		self.s_e_box.add(self.style_l)
		self.s_e_box.connect("button-press-event", self.button_popup,
							 self.item_factory.get_widget("/%s/%s" % (self.msg["view"][1],
																	  self.msg["style"][1])))

# === Bring Buttons and Labels together to HBox, Frame === #
		toolbox = gtk.HBox()
		toolbox.show()

		toolbox.pack_start(fetch_button, gtk.FALSE, gtk.TRUE, 0)
		toolbox.pack_start(open_button, gtk.FALSE, gtk.TRUE, 0)
		toolbox.pack_start(save_button, gtk.FALSE, gtk.TRUE, 0)
		vs = gtk.VSeparator()
		vs.show()
		toolbox.pack_start(vs, gtk.FALSE, gtk.TRUE, 3)

		toolbox.pack_start(self.pb, gtk.FALSE, gtk.TRUE, 0)
		toolbox.pack_start(self.vb, gtk.FALSE, gtk.TRUE, 0)
		toolbox.pack_start(self.ab, gtk.FALSE, gtk.TRUE, 0)
		vs = gtk.VSeparator()
		vs.show()
		toolbox.pack_start(vs, gtk.FALSE, gtk.TRUE, 3)

		toolbox.pack_start(style_desc, gtk.FALSE, gtk.TRUE, 0)
		toolbox.pack_start(self.s_e_box, gtk.FALSE, gtk.TRUE, 0)

		button_frame = gtk.Frame()
		button_frame.show()
		button_frame.set_border_width(0)
		button_frame.set_shadow_type(gtk.SHADOW_OUT)
		button_frame.add(toolbox)

# === Create Notebook === #
		label = unicode(_('Current'), 'utf-8')
		self.logbook = LogBook(label)
		self.logbook.set_tab_pos(int(self.app.prefs.get('logtabpos')))
		self.logbook.connect('switch-page', self.tab_changed)

		self.create_tab(label)

# === Bring Log Entry and Log Viewer together to VPaned === #
		main_paned = gtk.VPaned()
		main_paned.show()
		main_paned.pack1(self.logbook, gtk.TRUE)
		main_paned.pack2(text_frame, gtk.FALSE)

# === Bring all items together to Main Window Finally === #
		main_vbox = gtk.VBox(gtk.FALSE, 0)
		main_vbox.show()

		main_vbox.pack_start(main_menubar, gtk.FALSE, gtk.TRUE, 0)
		main_vbox.pack_start(button_frame, gtk.FALSE, gtk.TRUE, 0)
		main_vbox.pack_start(main_paned)
		main_vbox.pack_start(bottombox, gtk.FALSE, gtk.FALSE, 0)
		#main_vbox.pack_start(search_hbox, gtk.FALSE, gtk.FALSE, 0)

		self.window.add(main_vbox)

# === Initialization of LogWindow === #
		self.focus_active_loglist()

	def open(self, widget, data):
		self.window.show()

	def close(self, widget=None, e=None, data=None):
		self.window.hide()
		return gtk.TRUE

	def join(self):
		for item in [self.vb, self.ab, self.vi, self.ai]:
			item.set_sensitive(gtk.TRUE)

	def part(self):
		for item in [self.vb, self.ab, self.vi, self.ai]:
			item.set_sensitive(gtk.FALSE)

	def set_message(self, text):
		self.messagebar.set_text(text)

	def add_message(self, text):
		self.messagebar.set_text(self.messagebar.get_text() + text)

	def clear_message(self):
		self.messagebar.set_text('')

	def color_changed(self, script, sakura, kero, sync, error, URL):
		self.textmanager.color_changed(script, sakura, kero, sync, error, URL)
		list = self.logbook.get_active_loglist()
		if not list or not list.is_selected():
			return
		self.select(list)
	def font_changed(self, genfont, scfont):
		self.textmanager.font_changed(genfont, scfont)

	def autologging(self, path, compress=False):
		list = self.logbook.get_current_loglist()
		if not list:
			return
		filename = time.strftime(path, time.localtime(time.time()))

		dir = os.path.dirname(filename)
		if not os.path.exists(dir):
			try:
				os.makedirs(dir)
			except:
				return

		logger = XMLLogWriter(self, list, filename, compress, False)
		logger.start()

	def save_as_xml(self, widget=None, list=None):
		filew = CompressFileSelection()
		filew.set_select_multiple(gtk.FALSE)
		filew.connect("destroy", lambda x: filew.destroy())
		filew.set_filename(os.path.normpath(open_bottlecase() + '/*.xml'))

		filew.show()
		while True:
			res = filew.run()
			if not res or res == gtk.RESPONSE_CANCEL:
				filew.destroy()
				return

			filename = filew.get_filename()
			compress = filew.compress.get_active()

			if compress and filename[-3:] != '.gz':
				filename += '.gz'

			if os.path.exists(filename):
				if not open_overwrite_check_dialog(filew, filename):
					continue
			break
		filew.destroy()

		if not list or not isinstance(list, LogList):
			list = self.logbook.get_active_loglist()
		if not list:
			return

		logger = XMLLogWriter(self, list, filename, compress, True)
		logger.start()

	def open_xml_log(self, widget=None, data=None):
		filew = gtk.FileSelection()
		filew.set_select_multiple(gtk.FALSE)
		filew.connect("destroy", lambda x: filew.destroy())
		filew.set_filename(os.path.normpath(open_bottlecase() + '/*.xml'))
		filew.show()
		res = filew.run()
		filename = filew.get_filename()
		filew.destroy()

		if res == gtk.RESPONSE_OK:
			if os.path.exists(filename):
				logger = XMLLogOpener(self, filename,
									  self.app.prefs.get('logtabfocus'),
									  self.app.prefs.get('logpagepos'))
				logger.start()

	def fetch(self, widget=None, data=None):
		try:
			channels = self.app.client.channels
		except:
			channels = []

		d = LogFetchDialog(self.window, channels)
		if not d.request or not d.request_name:
			return

		fetcher = LogFetchOperator(self, d.request, d.request_name,
								   self.app.prefs.get('logtabfocus'),
								   self.app.prefs.get('logpagepos'))
		fetcher.start()

	def popup_from_logtab(self, widget, event, sender):
		if event.button == 3:
			sender.popup(None, None, None, event.button, event.time)

	def tab_drag_data_get(self, widget, context, selection, targetType, eventTime, sender):
		n = self.logbook.page_num(sender)
		selection.set(selection.target, 8, str(n))

	def tab_drop_received(self, widget, context, x, y, selection, targetType, time, receiver):
		sender_n = int(selection.data)
		sender = self.logbook.get_nth_page(sender_n)

		receiver_n = self.logbook.page_num(receiver)

		self.logbook.reorder_child(sender, receiver_n)
		self.logbook.reorder_child(receiver, sender_n)

	def edit_copy_ghostname(self, widget, data):
		self.edit_copy(widget, data, LogList.LISTSTORE_GHOST)
	def edit_copy_channel(self, widget, data):
		self.edit_copy(widget, data, LogList.LISTSTORE_CHANNEL)
	def edit_copy_script(self, widget, data):
		self.edit_copy(widget, data, LogList.LISTSTORE_SCRIPT)
	def edit_copy_time(self, widget, data):
		self.edit_copy(widget, data, LogList.LISTSTORE_DATETIME)
	def edit_copy_all(self, widget, data):
		self.edit_copy(widget, data, LogList.LISTSTORE_ALL)
	def edit_copy(self, widget, data, target):
		self.app.dispose_selection()
		self.app.copy_clipboard(self.logbook.edit_copy(widget, data, target))

	def connect_for_logtab(self, list, child, eventbox):
		save_menuitem = gtk.MenuItem(self.msg["save"])
		save_menuitem.show()
		save_menuitem.connect("activate", self.save_as_xml, list)
		close_menuitem = gtk.MenuItem(self.msg["close"])
		close_menuitem.show()
		if self.logbook.current_is(list):
			close_menuitem.connect("activate", self.close_current_tab, child)
		else:
			close_menuitem.connect("activate", self.close_tab, child)

		logtab_popup_menu = gtk.Menu()
		logtab_popup_menu.show()
		logtab_popup_menu.append(save_menuitem)
		logtab_popup_menu.append(close_menuitem)
		eventbox.connect("button-press-event", self.popup_from_logtab, logtab_popup_menu)

		# DND
		eventbox.connect("drag-data-get", self.tab_drag_data_get, child)
		eventbox.drag_source_set(gtk.gdk.BUTTON1_MASK, [ ( "text/plain", 0, 80 ) ], gtk.gdk.ACTION_MOVE)

		eventbox.connect("drag-data-received", self.tab_drop_received, child)
		eventbox.drag_dest_set(gtk.DEST_DEFAULT_MOTION|gtk.DEST_DEFAULT_HIGHLIGHT|gtk.DEST_DEFAULT_DROP,
							   [ ( "text/plain", 0, 80 ) ], gtk.gdk.ACTION_MOVE)

	def create_tab(self, label_text, pos=None, focus=False, list=None):
		close_button, eventbox, child = self.logbook.create_tab(label_text, pos, focus, list)

		list = child.get_child()
		self.connect_list(list)

		if self.logbook.current_is(list):
			close_button.connect('clicked', self.close_current_tab, child)
		else:
			close_button.connect('clicked', self.close_tab, child)
		close_button.connect('focus-in-event', self.focus_active_loglist)

		self.connect_for_logtab(list, child, eventbox)
		self.focus_active_loglist()

		if self.logbook.active_is(list):
			self.update_rows(list)
			if list.rows() > 0 and list.is_selected():
				self.select(list)

	def update(self, mid, channel, ghost, script, receive_time):
		datetime = time.strftime("%y/%m/%d %H:%M:%S", receive_time)
		close_button, eventbox, child = self.logbook.update(mid, channel, ghost, script, datetime,
															pos=self.app.prefs.get('logpagepos'),
															focus=self.app.prefs.get('logtabfocus'))

		if child and close_button and eventbox:
			# if Tab named `Current' Created
			close_button.connect('clicked', self.close_current_tab, child)
			close_button.connect('focus-in-event', self.focus_active_loglist)

			list = child.get_child()
			if list:
				self.connect_list(list)

				if list == self.logbook.get_active_loglist():
					if list.is_selected():
						self.select(list)
					self.update_rows(list)
			self.connect_for_logtab(list, child, eventbox)

		elif self.logbook.current_is(self.logbook.get_active_loglist()):
			self.update_rows(self.logbook.get_active_loglist())

		self.focus_active_loglist()

	def close_current_tab(self, widget, child):
		self.app.notify_log_current_tab_closed()
		self.close_tab(widget, child)

	def close_tab(self, widget, child):
		n = self.logbook.close_tab(widget, child)
		if n == -1:
			self.remove_status()

		self.volley_canceler = True
		list = child.get_children()[0]
		self.app.notify_play_cancel_page_of(list)

	def connect_list(self, list):
		list.connect("select-cursor-row",  self.select)
		list.connect("key-press-event",    self.loglist_keypress)
		list.connect("button-press-event", self.popup)

	def remove_status(self):
		self.sbar.pop(0)
		self.rowsbar.pop(0)
		self.curr_rowsbar.pop(0)
		self.textmanager.clear()

	def update_rows(self, list):
		if list and list.rows():
			self.rowsbar.pop(0)
			self.rowsbar.push(0, str(list.rows()) + unicode(_('Messages'), 'utf-8'))
		self.update_select_no(list)

	def update_select_no(self, list):
		if list and list.is_selected():
			self.curr_rowsbar.pop(0)
			self.curr_rowsbar.push(0, unicode('No.' + str(list.selection()+1), 'utf-8'))

	def tab_changed(self, notebook, page, pagenum):
		list = self.logbook.tab_changed(notebook, page, pagenum)
		if list is None:
			self.remove_status()
		else:
			self.select(list)
			self.update_rows(list)

	def tab_pos_changed(self, pos):
		if self.logbook.get_tab_pos() != pos:
			self.logbook.set_tab_pos(pos)

	def search(self, widget=None, data=None):
		self.logbook.search(self.s_arrow.get_property('arrow-type'),
							self.search_entry.get_text(),
							widget, data)

	def volley(self, widget=None, data=None):
		list = self.logbook.get_active_loglist()
		if not list or not list.is_selected():
			return

		self.volley_canceler = False
		gtk.timeout_add(WAIT_NEXT_REQUEST, self._volley, widget, data, list, list.selection(), 0)

	VOLLEY_INTERVAL = 500 # 0.5 sec
	def _volley(self, widget, data, list, target, min):
		if self.volley_canceler:
			self.volley_canceler = False
			return
		gtk.timeout_add(WAIT_NEXT_REQUEST, self.play, widget, data, list, target)
		if target > min:
			gtk.timeout_add(self.VOLLEY_INTERVAL, self._volley, widget, data, list, target-1, min)

	def play(self, widget=None, data=None, list=None, target=None):
		if list is None or target is None:
			list = self.logbook.get_active_loglist()
			if not list or not list.is_selected():
				return
			target = list.selection()

		script  = unicode(list.get_text(target, LogList.LISTSTORE_SCRIPT), 'utf-8')
		ifghost = unicode(list.get_text(target, LogList.LISTSTORE_GHOST), 'utf-8')
		channel = unicode(list.get_text(target, LogList.LISTSTORE_CHANNEL), 'utf-8')
		mid     = list.mids[target]

		if script is None or ifghost is None or not mid:
			return

		if self.get_bottle_state(mid, list) == STATE_BOTTLE_NONE: # ignore duplex
			self.app.send_local_message(channel, ifghost, script, mid=mid, list=list)

	def cancel_play(self, widget=None, data=None):
		mid = self.logbook.cancel_play(widget, data)
		if not self.app.notify_play_cancel(mid):
			open_error_dialog(unicode(_("Couldn't cancel to play message"), 'utf-8'), self.window)

	def cancel_play_page(self, widget=None, data=None):
		list = self.logbook.cancel_play_page()
		if list is not None:
			self.volley_canceler = True
			self.app.notify_play_cancel_page_of(list)
		else:
			open_error_dialog(unicode(_("Couldn't cancel to play message"), 'utf-8'), self.window)

	def cancel_play_all(self, widget=None, data=None):
		self.volley_canceler = True
		self.logbook.cancel_play_all()
		self.app.notify_play_cancel_all()

	def get_bottle_state(self, mid, list):
		return self.logbook.get_bottle_state(mid, list)

	def set_bottle_state(self, mid, list, stock=STATE_BOTTLE_NONE):
		self.logbook.set_bottle_state(mid, list, stock)

	def set_playing_bottle(self, mid, list, prev_list=None):
		if prev_list is not None:
			self.clear_playing_bottle(prev_list)
		self.logbook.set_playing_bottle(mid, list)

	def set_reserve_bottle(self, mid, list):
		self.set_bottle_state(mid, list, STATE_BOTTLE_RESERVED)

	def clear_playing_bottle(self, list=None):
		self.logbook.clear_playing_bottle(list)

	def log_votes(self, mid, type, num):
		self.logbook.log_votes(mid, type, num)

	def voting(self, widget, data):
		mid, type = self.logbook.voting(widget, data, self.window)
		if mid is None or type is None:
			return
		self.app.vote_message(mid, type)

	def agreeing(self, widget, data):
		mid, type = self.logbook.agreeing(widget, data, self.window)
		if mid is None or type is None:
			return
		self.app.vote_message(mid, type)

	def button_popup(self, widget, event, sender):
		if event.button in [1, 3]:
			sender.popup(None, widget, None, event.button, event.time)

	def set_entry_guide(self, widget, event, data):
		if data == 'in':
			if widget.get_text() == self.msg['search']:
				widget.set_text('')
		elif data == 'out':
			if not widget.get_text():
				widget.set_text(self.msg['search'])

	def popup(self, loglist, event):
		if event.button == 3:
			self.popup_menu.popup(None, loglist, None, event.button, event.time)
		elif event.type == 5:
			self.play(loglist, event)

	def select(self, loglist, se=gtk.FALSE):
		self.textmanager.clear()
		row = loglist.selection()
		script = unicode(loglist.get_text(row, LogList.LISTSTORE_SCRIPT), "utf-8", 'replace')
		self.textmanager.insert_with_color_from(script)
		self.sw2.emit('scroll-child', gtk.SCROLL_START, gtk.FALSE)
		self.sbar.pop(0)
		self.sbar.push(0, str(len(script.encode('sjis'))) + unicode(_('bytes'), 'utf-8'))
		self.update_select_no(loglist)

	def scroll(self, widget, event):
		if event.state == 1: # shift
			if event.keyval == 65362: # up
				self.sw2.emit('scroll-child', gtk.SCROLL_PAGE_UP, gtk.FALSE)
				return gtk.TRUE
			elif event.keyval == 65364: # down
				self.sw2.emit('scroll-child', gtk.SCROLL_PAGE_DOWN, gtk.FALSE)
				return gtk.TRUE
			return gtk.FALSE
		return gtk.FALSE

	def entry_keypress(self, widget, event):
		return self.scroll(widget, event)

	def loglist_keypress(self, widget, event):
		if self.scroll(widget, event) == gtk.FALSE:
			if event.keyval == 65293: # enter
				self.play()
				return gtk.TRUE
			return gtk.FALSE
		return gtk.TRUE

	def set_style(self, data, widget):
		if data == STYLE_TALK:
			label = self.msg['talk style']
		elif data == STYLE_SCRIPT:
			label = self.msg['script style']
		elif data == STYLE_SCRIPT_WITH_LINEFEED:
			label = "%s(%s)" % (self.msg['script style'], self.msg['linefeed'])

		if label == self.style_l.get_text():
			return
		self.style_l.set_text(label)
		self.textmanager.set_style(data)

		list = self.logbook.get_active_loglist()
		if not list or not list.is_selected():
			return
		self.select(list)

	def focus_active_loglist(self, widget=None, data=None):
		self.logbook.focus_active_loglist()

	def check_message_bar(self, data=None, widget=None):
		if widget.get_active():
			self.msgframe.hide()
		else:
			self.msgframe.show()

	def help(self, event=None):
		pass

	def get_menu_items(self):
		menu_items = (
			( "/%s" % self.app.msg["file"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s..." % (self.app.msg["file"][1], self.msg["open_xml"]), "<control>o", self.open_xml_log,
			  0, "<StockItem>", gtk.STOCK_OPEN ),
			( "/%s/%s..." % (self.app.msg["file"][1], self.msg["save"]), "<control>s", self.save_as_xml,
			  0, "<StockItem>", gtk.STOCK_SAVE_AS ),
			( "/%s/%s..." % (self.app.msg["file"][1], self.msg["fetch"]), None, self.fetch,
			  0, "<StockItem>", gtk.STOCK_CONVERT ),
			( "/%s/%s" % (self.app.msg["file"][1], self.msg["close"]), "<control>q", self.close,
			  0, "<StockItem>", gtk.STOCK_CLOSE ),

			( "/%s" % self.app.msg["edit"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["copy time"]),
			  None, self.edit_copy_time, 0, None ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["copy ghostname"]),
			  None, self.edit_copy_ghostname, 0, None ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["copy channel"]),
			  None, self.edit_copy_channel, 0, None ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["copy script"]),
			  "<control>c", self.edit_copy_script, 0, "<StockItem>", gtk.STOCK_COPY ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["copy all"]),
			  None, self.edit_copy_all, 0, None ),
			( "/%s/sep1" % self.app.msg["edit"][1], None, None, 0, "<Separator>" ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["search"]), "<control>f",
			  lambda x, y: self.search_entry.grab_focus(), 0, "<StockItem>", gtk.STOCK_FIND ),
			( "/%s/sep2" % self.app.msg["edit"][1], None, None, 0, "<Separator>" ),
			( "/%s/%s..." % (self.app.msg["edit"][1], self.app.msg["pref"][0]), None,
			  self.app.open_preference, 0, "<StockItem>", gtk.STOCK_PREFERENCES ),

			( "/%s" % self.msg["view"][0], None, None, 0, "<Branch>" ),

			( "/%s/%s" % (self.msg["view"][1], self.msg["style"][0]), None, None, 0, "<Branch>" ),
			( "/%s/%s/%s" % (self.msg["view"][1], self.msg["style"][1], self.msg["talk style"]),
			  None, self.set_style, STYLE_TALK, "<RadioItem>" ),
			( "/%s/%s/%s" % (self.msg["view"][1], self.msg["style"][1], self.msg["script style"]),
			  None, self.set_style, STYLE_SCRIPT,
			  "/%s/%s/%s" % (self.msg["view"][1],
							 self.msg["style"][1],
							 self.msg["talk style"])),
			( "/%s/%s/%s(%s)" % (self.msg["view"][1], self.msg["style"][1],
								 self.msg["script style"], self.msg["linefeed"]),
			  None, self.set_style, STYLE_SCRIPT_WITH_LINEFEED,
			  "/%s/%s/%s" % (self.msg["view"][1],
							 self.msg["style"][1],
							 self.msg["script style"])),
			( "/%s/%s" % (self.msg["view"][1], self.msg["hide_msg"]), None, self.check_message_bar, 0, "<CheckItem>" ),

			( "/%s" % self.app.msg["help"][0], None, None, 0, "<LastBranch>" ),
			( "/%s/%s..." % (self.app.msg["help"][1], self.app.msg["about"]), None, self.app.about,
			  0, "<StockItem>", gtk.STOCK_DIALOG_INFO ),
			)
		return menu_items

	def set_self_msg(self):
		self.msg = {
			"open_xml":  unicode(_("Open XML log"), "utf-8"),
			"save":  unicode(_("Save as XML"), "utf-8"),
			"fetch":  unicode(_("Fetch logs"), "utf-8"),
			"close":  unicode(_("Close"), "utf-8"),
			"copy script":   unicode(_("Copy script"), "utf-8"),
			"copy ghostname": unicode(_("Copy ghost name"), "utf-8"),
			"copy channel":   unicode(_("Copy channel"), "utf-8"),
			"copy time":   unicode(_("Copy time"), "utf-8"),
			"copy all":   unicode(_("Copy all"), "utf-8"),
			"search":   unicode(_("Search"), "utf-8"),

			"view":		   [unicode(_("View(_V)"), "utf-8"),
							unicode(_("View(V)"), "utf-8")],
			"update":	   unicode(_("Update"), "utf-8"),
			"style":	   [unicode(_("Style(_S)"), "utf-8"),
							unicode(_("Style(S)"), "utf-8")],
			"hide_msg":    unicode(_("Hide system messages"), "utf-8"),
			"talk style":  unicode(_("Talk style"), "utf-8"),
			"script style":unicode(_("Script style"), "utf-8"),
			"linefeed":	   unicode(_("with linefeed"), "utf-8"),
			}

class CompressFileSelection(gtk.FileSelection):
	def __init__(self):
		gtk.FileSelection.__init__(self)
		self.compress = gtk.CheckButton(unicode(_('GZIP Compress'), 'utf-8'))
		self.compress.show()
		self.main_vbox.pack_end(self.compress)
