# -*- coding: ascii -*-
#
#  phrases.py - Phrase manipulator for GBottler
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: phrases.py,v 1.22 2004/09/25 01:38:39 atzm Exp $
#

import os, sys, re, string
import gtk, gobject

from common import *

class PhraseManager:
	DBFILE = "phrases"
	DBWRITEFREQ = 10 # write phrases to file per 10 operations
	def __init__(self):
		self.dbpath = os.path.join(open_bottlecase(), self.DBFILE)
		self.dblife = self.DBWRITEFREQ
		self.load_database()

	def load_database(self):
		self.phrases = []
		try:
			file = open(self.dbpath)
		except IOError:
			return
		while 1:
			line = unicode(file.readline(), "utf-8")
			if not line:
				break
			elif line[0] == "#":
				continue
			elif line[-2:] == "\r\n":
				line = line[:-2]
			elif line[-1] in ["\r", "\n"]:
				line = line[:-1]
			self.phrases.append(line)
		file.close()

	def save_database(self):
		try:
			file = open(self.dbpath, "w")
		except IOError:
			sys.stderr.write("Error: cannot write %s\n" % self.dbpath)
			return
		file.write("# This is an auto-generated file.\n")
		file.write("# Don't edit this file while running %s!\n" % APP)
		for line in self.phrases:
			file.write(line.encode("utf-8") + "\n")
		file.close()

	def close(self):
		self.save_database()

	def sync(self):
		self.dblife = self.dblife - 1
		if self.dblife == 0:
			self.dblife = self.DBWRITEFREQ
			self.save_database()

	def overwrite(self, phrases):
		self.phrases = phrases

	def list(self):
		return self.phrases[:]

	def get(self, index):
		return self.phrases[index]

	def append(self, phrase):
		self.phrases.append(phrase)
		self.sync()

	def insert(self, index, phrase):
		self.phrases.insert(index, phrase)
		self.sync()

	def modify(self, index, phrase):
		self.phrases[index] = phrase
		self.sync()

	def delete(self, index):
		del self.phrases[index]
		self.sync()

	def move_up(self, index):
		if index - 1 < 0:
			return
		self.phrases[index], self.phrases[index - 1] = \
							 self.phrases[index - 1], self.phrases[index]
		self.sync()

	def move_down(self, index):
		if index + 1 >= len(self.phrases):
			return
		self.phrases[index], self.phrases[index + 1] = \
							 self.phrases[index + 1], self.phrases[index]
		self.sync()

class PhraseWindow(gtk.Window):
	TARGET_DEFINES = 0
	TARGET_SCRIPT  = 1

	def __init__(self, app):
		self.app = app
		self.manager = PhraseManager()
		accel_group = gtk.AccelGroup()

# === Create Menubar === #
		self.set_self_msg()
		menu_items = self.get_menu_items()

		self.item_factory = gtk.ItemFactory(gtk.MenuBar, "<main>", accel_group)
		self.item_factory.create_items(menu_items)

		self.menu_list = {}
		for m in menu_items:
			# keeping ref to each items
			key = re.sub("_", "", m[0])
			self.menu_list[key] = self.item_factory.get_item(key)

		menubar = self.item_factory.get_widget("<main>")

		# keep edit menu as popup menu
		self.popup_menu = self.item_factory.get_widget("/%s" % self.app.msg["edit"][1])

# === Create Buttons === #
		self.tooltips = gtk.Tooltips()
		bbox = gtk.HBox(gtk.FALSE, 0)
		for item in [
			[gtk.STOCK_NEW,     self.insert_new,     self.msg['new']],
			[gtk.STOCK_COPY,    self.edit_duplicate, self.msg['duplicate']],
			[gtk.STOCK_DELETE,  self.delete,         self.app.msg['delete']],
			[gtk.STOCK_GO_UP,   self.edit_up,        self.msg['up']],
			[gtk.STOCK_GO_DOWN, self.edit_down,      self.msg['down']],
			]:
			btn = get_icon_button(item[0])
			btn.connect('clicked', item[1])
			self.tooltips.set_tip(btn, item[2])
			bbox.pack_start(btn, gtk.FALSE, gtk.TRUE, 0)

		bf = gtk.Frame()
		bf.set_shadow_type(gtk.SHADOW_OUT)
		bf.add(bbox)

# === Create TreeView for Phrases === #
		liststore = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING)
		self.entries = gtk.TreeView(liststore)

		columns = [
			gtk.TreeViewColumn(unicode(_("Name"), "utf-8")),
			gtk.TreeViewColumn(unicode(_("Script"), "utf-8")),
			]
		for i in range(len(columns)):
			self.entries.append_column(columns[i])
			cell = gtk.CellRendererText()
			cell.set_property('editable', gtk.TRUE)
			cell.connect('edited', self.cell_edited, liststore, i)
			columns[i].pack_start(cell, gtk.FALSE)
			columns[i].add_attribute(cell, 'text', i)
			columns[i].set_resizable(gtk.TRUE)
			columns[i].set_clickable(gtk.TRUE)
			columns[i].connect('clicked', self.sort_column, liststore, i)

		self.entries.set_rules_hint(gtk.TRUE)
		self.entries.set_search_column(gtk.TRUE)
		self.entries.set_reorderable(gtk.TRUE)

		self.entries.connect("button-press-event", self.popup)
		self.entries.connect("key-press-event", self.keypress)

		sw = gtk.ScrolledWindow()
		sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		sw.set_shadow_type(gtk.SHADOW_IN)
		sw.add(self.entries)

# === Create Buttons, OK, APPLY, CANCEL === #
		self.ok  = gtk.Button(unicode(_("OK"), "utf8"), gtk.STOCK_OK)
		self.apl = gtk.Button(unicode(_("Apply"), "utf8"), gtk.STOCK_APPLY)
		self.can = gtk.Button(unicode(_("Cancel"), "utf8"), gtk.STOCK_CANCEL)

		self.ok.connect('clicked', self.okey)
		self.apl.connect('clicked', self.apply)
		self.can.connect('clicked', self.cancel)

# === Bring Buttons together to HBox === #
		buttons = gtk.HBox(gtk.TRUE)
		buttons.pack_end(self.ok, padding=3)
		buttons.pack_end(self.can, padding=3)
		buttons.pack_end(self.apl, padding=3)

# === Bring all items together to Main Window Finally === #
		self.main_vbox = gtk.VBox(gtk.FALSE, 0)

		self.main_vbox.pack_start(menubar, gtk.FALSE, gtk.TRUE, 0)
		self.main_vbox.pack_start(bf, gtk.FALSE, gtk.TRUE, 0)
		self.main_vbox.pack_start(sw, gtk.TRUE, gtk.TRUE, 0)
		self.main_vbox.pack_end(buttons, gtk.FALSE, gtk.FALSE, 3)

		gtk.Window.__init__(self, gtk.WINDOW_TOPLEVEL)
		self.add_accel_group(accel_group)
		self.set_default_size(420, 320)
		self.connect("delete-event", self.close)
		self.set_title(unicode(_(APP), "utf-8") + ":" + \
					   unicode(_("Phrases"), "utf-8"))
		self.add(self.main_vbox)

# === Initialization of PhraseWindow === #
		self.initialize()

	def sort_column(self, column, liststore, id):
		current_column_id = liststore.get_sort_column_id()
		if current_column_id[0] == id:
			liststore.set_sort_column_id(id, not current_column_id[1])
		else:
			liststore.set_sort_column_id(id, gtk.SORT_DESCENDING)

	def cell_edited(self, cell, path, new_text, liststore, column):
		liststore[path][column] = new_text

	def get_entry_rows(self):
		return len(self.entries.get_model())

	def entry_is_selected(self):
		try:
			treeselection = self.entries.get_selection()
			[liststore, treeiter] = treeselection.get_selected()
			if not treeiter:
				return False
		except:
			return False
		return True

	def get_entry_selection(self):
		if self.entry_is_selected():
			treeselection = self.entries.get_selection()
			[liststore, treeiter] = treeselection.get_selected()
			pathes = liststore.get_path(treeiter)
			return pathes[0]
		else:
			raise RuntimeError('List is not selected.')

	def get_entry_text(self, row, col):
		liststore = self.entries.get_model()
		iter = liststore.get_iter(row)
		return liststore.get_value(iter, col)

	def edit_copy_clipboard_defines(self, widget=None, event=None):
		self.edit_copy_clipboard(widget, event, self.TARGET_DEFINES)
	def edit_copy_clipboard_script(self, widget=None, event=None):
		self.edit_copy_clipboard(widget, event, self.TARGET_SCRIPT)
	def edit_copy_clipboard(self, widget, event, target):
		if not self.entry_is_selected():
			return
		self.app.dispose_selection()
		self.app.copy_clipboard(unicode(self.get_entry_text(self.get_entry_selection(), target), 'utf-8'))

	def keypress(self, widget, event=None):
		if type(widget) is gtk.TreeView:
			if event.keyval == 65535: # delete key
				self.delete(widget, event)

	def popup(self, widget, event):
		if event.button == 3:
			self.popup_menu.popup(None, None, None, event.button, event.time)
			return gtk.TRUE
		if event.type == 5:
			self.edit_copy(widget, event)
			return gtk.TRUE

	def edit_up(self, widget=None, event=None):
		if not self.entry_is_selected():
			return
		row = self.get_entry_selection()
		if row != 0:
			liststore = self.entries.get_model()
			src_iter  = liststore.get_iter(row)
			dest_iter = liststore.get_iter(row-1)
			liststore.swap(src_iter, dest_iter)

	def edit_down(self, widget=None, event=None):
		if not self.entry_is_selected():
			return
		row = self.get_entry_selection()
		if row+1 < self.get_entry_rows():
			liststore = self.entries.get_model()
			src_iter  = liststore.get_iter(row)
			dest_iter = liststore.get_iter(row+1)
			liststore.swap(dest_iter, src_iter)

	def edit_duplicate(self, widget=None, event=None):
		if not self.entry_is_selected():
			return
		row = self.get_entry_selection()
		name = self.get_entry_text(row, 0)
		script = self.get_entry_text(row, 1)

		liststore = self.entries.get_model()
		liststore.insert(row, (name, script))

	def edit_copy(self, widget=None, event=None):
		if not self.entry_is_selected():
			return
		script = self.get_entry_text(self.get_entry_selection(), 1)
		if not script:
			return
		self.app.insert(script)

	def initialize(self):
		liststore = self.entries.get_model()
		liststore.clear()
		for p in self.manager.list():
			p = string.split(p)
			liststore.append(p)
		self.entries.grab_focus()

	def delete(self, widget=None, event=None):
		row = 0
		if self.entry_is_selected():
			row = self.get_entry_selection()
		else:
			return
		liststore = self.entries.get_model()
		iter = liststore.get_iter(row)
		liststore.remove(iter)
		self.entries.grab_focus()

	def insert_new(self, widget=None, event=None):
		row = 0
		name = ''
		script = ''
		if self.entry_is_selected():
			row    = self.get_entry_selection()
			name   = self.get_entry_text(row, self.TARGET_DEFINES)
			script = self.get_entry_text(row, self.TARGET_SCRIPT)
		else:
			name   = unicode(_('Name'), 'utf-8')
			script = unicode(_('Script'), 'utf-8')
		liststore = self.entries.get_model()
		liststore.insert(row+1, (name, script))

	def apply(self, widget=None, event=None):
		p = []
		for r in range(self.get_entry_rows()):
			n = unicode(self.get_entry_text(r, 0), "utf-8")
			s = unicode(self.get_entry_text(r, 1), "utf-8")
			p.append("%s\t%s" % (n, s))
		self.manager.overwrite(p)
		self.manager.save_database()

	def okey(self, widget=None, event=None):
		self.apply(widget, event)
		self.initialize()
		self.close()

	def cancel(self, widget=None, event=None):
		self.initialize()
		self.close()

	def open(self, widget=None, data=None):
		self.show_all()
		self.entries.grab_focus()

	def close(self, widget=None, e=None, data=None):
		self.hide()
		return gtk.TRUE

	def set_self_msg(self):
		self.msg = {
			"close": unicode(_("Close"), "utf-8"),
			"copy to main editor": unicode(_("Copy to main editor"), "utf-8"),
			"copy name":   unicode(_("Copy name"), "utf-8"),
			"copy script":   unicode(_("Copy script"), "utf-8"),
			"cut":	 unicode(_("Cut"), "utf-8"),
			"new":   unicode(_("New"), "utf-8"),
			"duplicate": unicode(_("Duplicate"), "utf-8"),
			"up":    unicode(_("Up"), "utf-8"),
			"down":  unicode(_("Down"), "utf-8"),
			}

	def get_menu_items(self):
		menu_items = (
			( "/%s" % self.app.msg["file"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.app.msg["file"][1], self.msg["close"]), "<control>q", self.close, 0, "<StockItem>", gtk.STOCK_CLOSE ),

			( "/%s" % self.app.msg["edit"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["copy to main editor"]),
			  None, self.edit_copy, 0, None ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["copy name"]),
			  None, self.edit_copy_clipboard_defines, 0, None ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["copy script"]),
			  "<control>c", self.edit_copy_clipboard_script, 0, "<StockItem>", gtk.STOCK_COPY ),
			( "/%s/sep" % self.app.msg["edit"][1], None, None, 0, '<Separator>' ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["new"]), None, self.insert_new, 0, "<StockItem>", gtk.STOCK_NEW ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["duplicate"]), None, self.edit_duplicate, 0, None ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.app.msg["delete"]), None, self.delete, 0, "<StockItem>", gtk.STOCK_DELETE ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["up"]), None, self.edit_up, 0, "<StockItem>", gtk.STOCK_GO_UP ),
			( "/%s/%s" % (self.app.msg["edit"][1], self.msg["down"]), None, self.edit_down, 0, "<StockItem>", gtk.STOCK_GO_DOWN ),

			( "/%s" % self.app.msg["help"][0], None, None, 0, "<LastBranch>" ),
			( "/%s/%s..." % (self.app.msg["help"][1], self.app.msg["about"]), None, self.app.about, 0, "<StockItem>", gtk.STOCK_DIALOG_INFO ),
			)
		return menu_items
