# -*- coding: ascii -*-
#
#  prefs.py - Preference manipulator for GBottler
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: prefs.py,v 1.55 2004/09/15 06:09:09 atzm Exp $
#

import os, sys, re, string
import gtk
from common import *

class PreferenceDialog:
	def __init__(self, app, prefmanager, luid_info=None):
		self.prefs = {}
		self.app = app
		self.prefmanager = prefmanager
		if luid_info:
			self.luid_info = luid_info

# === Create General Tab === #
		self.prefs['auto_open_log'] = gtk.CheckButton(unicode(_("Open log window when startup")))
		self.prefs['auto_connect']  = gtk.CheckButton(unicode(_("Connect bottle server when startup")))
		self.prefs['auto_join']     = gtk.CheckButton(unicode(_("Join all channels when connect")))
		self.prefs['auto_clear']    = gtk.CheckButton(unicode(_("Clear editor after send")))
		self.prefs['sent_log']      = gtk.CheckButton(unicode(_("Logging history of sent script")))

		genbox = gtk.VBox()
		for name in ['auto_open_log', 'auto_connect', 'auto_join', 'auto_clear', 'sent_log']:
			self.prefs[name].set_border_width(5)
			genbox.pack_start(self.prefs[name], gtk.FALSE, gtk.FALSE, 0)

		genframe = gtk.Frame(unicode(_("General")))
		genframe.set_border_width(5)
		genframe.add(genbox)

# === Create SSTP Frame === #
		sendbox = gtk.VBox()

		self.prefs['forward'] = gtk.CheckButton(unicode(_("Forward")))
		self.prefs['accept_all'] = gtk.RadioButton(None, unicode(_("Accept All")))
		self.prefs['forward_listed'] = gtk.RadioButton(self.prefs['accept_all'],
													   unicode(_("Forward only listed Ghost")))
		for name in ['forward', 'accept_all', 'forward_listed']:
			self.prefs[name].set_border_width(5)
			sendbox.pack_start(self.prefs[name], gtk.FALSE, gtk.FALSE, 0)
		self.prefs['forward'].connect('toggled', self.forward_toggled)

		sendframe = gtk.Frame(unicode(_("SSTP server")))
		sendframe.set_border_width(5)
		sendframe.add(sendbox)

# === Create Initial Script Frame === #
		self.prefs['initialscript'] = gtk.Entry()

		iniscbox = gtk.VBox()
		iniscbox.pack_start(self.prefs['initialscript'], gtk.FALSE, gtk.FALSE, 5)
		iniscbox.pack_start(gtk.Label(unicode(_('A first | is cursor position'), 'utf-8')),
								 gtk.FALSE, gtk.FALSE, 5)

		iniscframe = gtk.Frame(unicode(_("Initial script")))
		iniscframe.set_border_width(5)
		iniscframe.add(iniscbox)

# === Bring General and SSTP Frames together to General Tab === #
		general_tab = gtk.VBox()
		for item in [genframe, sendframe, iniscframe]:
			general_tab.pack_start(item, gtk.FALSE, gtk.FALSE, 0)

# === Create Font Tab === #
		fonttable = gtk.Table(2, 3)
		list = ['general font', 'script font']
		for i in range(len(list)):
			self.prefs[list[i]] = gtk.Entry()
			fonttable.attach(gtk.Label(unicode(_(list[i]), 'utf-8')), 0, 1, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
			fonttable.attach(self.prefs[list[i]], 1, 2, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
			btn = get_icon_button(gtk.STOCK_SELECT_FONT, size=gtk.ICON_SIZE_BUTTON, relief=gtk.RELIEF_NORMAL)
			btn.connect('clicked', self.open_fontselection, self.prefs[list[i]])
			fonttable.attach(btn, 2, 3, i, i+1, gtk.FILL, gtk.FILL, 0, 3)

		fontbox = gtk.VBox()
		fontbox.pack_start(fonttable, gtk.FALSE, gtk.FALSE, 0)

		fontframe = gtk.Frame(unicode(_("Font settings")))
		fontframe.set_border_width(5)
		fontframe.add(fontbox)

		colortable = gtk.Table(6, 3)
		list = ['script', 'sakura', 'kero', 'sync', 'error', 'URL']
		for i in range(len(list)):
			self.prefs[list[i]] = gtk.Entry()
			colortable.attach(gtk.Label(unicode(_(list[i]), 'utf-8')), 0, 1, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
			colortable.attach(self.prefs[list[i]], 1, 2, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
			btn = get_icon_button(gtk.STOCK_SELECT_COLOR, size=gtk.ICON_SIZE_BUTTON, relief=gtk.RELIEF_NORMAL)
			btn.connect('clicked', self.open_colorselection, self.prefs[list[i]])
			colortable.attach(btn, 2, 3, i, i+1, gtk.FILL, gtk.FILL, 0, 3)

		colorbox = gtk.VBox()
		colorbox.pack_start(colortable, gtk.FALSE, gtk.FALSE, 0)

		colorframe = gtk.Frame(unicode(_("Color settings")))
		colorframe.set_border_width(5)
		colorframe.add(colorbox)

		fonttab = gtk.VBox()
		fonttab.pack_start(fontframe)
		fonttab.pack_start(colorframe)

# === Create GViewer Tab == #
		self.prefs['use_gviewer'] = gtk.CheckButton(unicode(_('Use GViewer'), 'utf-8'))
		self.prefs['use_gviewer'].set_border_width(2)
		self.prefs['use_gviewer'].connect('toggled', self.viewer_toggled)

		self.prefs['open_gviewer_when_play'] = gtk.CheckButton(unicode(_('Auto open GViewer when play script'), 'utf-8'))
		self.prefs['open_gviewer_when_play'].set_border_width(2)

		adj = gtk.Adjustment(1.0, 0.0, 500.0, 1.0, 5.0, 0.0)
		self.prefs['wait_1_char'] = gtk.SpinButton(adj, 0, 0)

		adj = gtk.Adjustment(1.0, 0.0, 500.0, 1.0, 5.0, 0.0)
		self.prefs['wait_w1_tag'] = gtk.SpinButton(adj, 0, 0)

		wait_table = gtk.Table(2, 3)
		wait_table.attach(gtk.Label(unicode(_('Wait for a character'), 'utf-8')), 0, 1, 0, 1, gtk.FILL, gtk.FILL, 3, 3)
		wait_table.attach(self.prefs['wait_1_char'], 1, 2, 0, 1, gtk.FILL, gtk.FILL, 3, 3)
		wait_table.attach(gtk.Label(unicode(_('milli seconds'), 'utf-8')), 2, 3, 0, 1, gtk.FILL, gtk.FILL, 3, 3)
		wait_table.attach(gtk.Label(unicode(_('Wait for \\w1'), 'utf-8')), 0, 1, 1, 2, gtk.FILL, gtk.FILL, 3, 3)
		wait_table.attach(self.prefs['wait_w1_tag'], 1, 2, 1, 2, gtk.FILL, gtk.FILL, 3, 3)
		wait_table.attach(gtk.Label(unicode(_('milli seconds'), 'utf-8')), 2, 3, 1, 2, gtk.FILL, gtk.FILL, 3, 3)

		viewbox = gtk.VBox()
		viewbox.pack_start(self.prefs['use_gviewer'], gtk.FALSE, gtk.FALSE, 0)
		viewbox.pack_start(self.prefs['open_gviewer_when_play'], gtk.FALSE, gtk.FALSE, 0)
		viewbox.pack_start(wait_table, gtk.FALSE, gtk.FALSE, 0)

		viewerframe = gtk.Frame(unicode(_('GViewer'), 'utf-8'))
		viewerframe.set_border_width(5)
		viewerframe.add(viewbox)

		viewer_tab = gtk.VBox()
		viewer_tab.pack_start(viewerframe, gtk.FALSE, gtk.FALSE, 0)
		self.viewer_toggled()

# === Create Logging Frame == #
		self.prefs['logging'] = gtk.CheckButton(unicode(_("Save `Current' log as XML when disconnect"), 'utf-8'))
		self.prefs['logging'].set_border_width(2)
		self.prefs['logging'].connect('toggled', self.logging_toggled)

		self.prefs['logging_current_tab_closed'] = gtk.CheckButton(unicode(_("Save log as XML when close `Current' tab"), 'utf-8'))
		self.prefs['logging_current_tab_closed'].set_border_width(2)
		self.prefs['logging_current_tab_closed'].connect('toggled', self.logging_toggled)

		self.prefs['logging_compress'] = gtk.CheckButton(unicode(_("GZIP Compress"), 'utf-8'))
		self.prefs['logging_compress'].set_border_width(2)
		self.prefs['logging_compress'].set_sensitive(gtk.FALSE)

		self.prefs['logging_path'] = gtk.Entry()
		self.prefs['logging_path'].set_sensitive(gtk.FALSE)

		lbox = gtk.VBox()
		lbox.pack_start(self.prefs['logging_path'], gtk.FALSE, gtk.FALSE, 3)
		lbox.pack_start(gtk.Label(unicode(_('Can use time format e.g. %H, %M, %S and etc.'), 'utf-8')),
						gtk.FALSE, gtk.FALSE, 3)

		lframe = gtk.Frame(unicode(_('Filename'), 'utf-8'))
		lframe.set_shadow_type(gtk.SHADOW_ETCHED_OUT)
		lframe.set_border_width(5)
		lframe.add(lbox)

		logbox1 = gtk.VBox()
		for item in [self.prefs['logging'], self.prefs['logging_current_tab_closed'],
					 self.prefs['logging_compress'], lframe]:
			logbox1.pack_start(item, gtk.FALSE, gtk.FALSE, 2)

		logframe1 = gtk.Frame(unicode(_("Logs")))
		logframe1.set_border_width(5)
		logframe1.add(logbox1)

# === Create LogTab behavior Frame == #
		self.prefs['logtabfocus']  = gtk.CheckButton(unicode(_("Focus new page automatically"), 'utf-8'))
		self.prefs['logtabfocus'].set_border_width(2)

		pagepos_table = gtk.Table(2, 2)
		prev = None
		list = [[
			[POS_NEXT,  "To current tab's next"],
			[POS_PREV,  "To current tab's previous"]
			],
				[
			[POS_END,   'As end tab'],
			[POS_FIRST, 'As first tab']
			]]
		for i in range(len(list)):
			for j in range(len(list[i])):
				pref = list[i][j][0]
				description = list[i][j][1]

				self.prefs[pref]  = gtk.RadioButton(prev, unicode(_(description)))
				self.prefs[pref].set_border_width(2)
				pagepos_table.attach(self.prefs[pref], j, j+1, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
				prev = self.prefs[pref]

		logbox2 = gtk.VBox()
		logbox2.pack_start(self.prefs['logtabfocus'], gtk.FALSE, gtk.FALSE, 2)
		logbox2.pack_start(pagepos_table, gtk.FALSE, gtk.FALSE, 2)

		logframe2 = gtk.Frame(unicode(_("When open new page...")))
		logframe2.set_border_width(5)
		logframe2.add(logbox2)

# === Create LogTab position Frame == #
		tabpos_table = gtk.Table(4, 1)
		prev = None
		list = [['Top', 'Bottom', 'Left', 'Right']]
		for i in range(len(list)):
			for j in range(len(list[i])):
				label = list[i][j]

				pref_name = None
				if label == 'Right':
					pref_name = "LOGTABPOS_%s" % gtk.POS_RIGHT
				elif label == 'Left':
					pref_name = "LOGTABPOS_%s" % gtk.POS_LEFT
				elif label == 'Bottom':
					pref_name = "LOGTABPOS_%s" % gtk.POS_BOTTOM
				else:
					pref_name = "LOGTABPOS_%s" % gtk.POS_TOP

				self.prefs[pref_name] = gtk.RadioButton(prev, unicode(_(label)))
				tabpos_table.attach(self.prefs[pref_name], j, j+1, i, i+1, gtk.FILL, gtk.FILL, 3, 3)
				prev = self.prefs[pref_name]

		logframe3 = gtk.Frame(unicode(_("Tab position")))
		logframe3.set_border_width(5)
		logframe3.add(tabpos_table)

# === Bring Logging, LogTab behavior, LogTab position frames together to Log Tab === #
		log_tab = gtk.VBox()
		log_tab.pack_start(logframe1, gtk.FALSE, gtk.FALSE, 0)
		log_tab.pack_start(logframe2, gtk.FALSE, gtk.FALSE, 0)
		log_tab.pack_start(logframe3, gtk.FALSE, gtk.FALSE, 0)

# === Create Browser Frame == #
		self.prefs['browser'] = gtk.Entry()

		brsbox = gtk.VBox()
		brsbox.pack_start(self.prefs['browser'], gtk.FALSE, gtk.FALSE, 5)
		brsbox.pack_start(gtk.Label(unicode(_('Replace %s to URL'), 'utf-8')), gtk.TRUE, gtk.TRUE, 3)

		brsframe = gtk.Frame(unicode(_("Command to run browser")))
		brsframe.set_border_width(5)
		brsframe.add(brsbox)

# === Create Download Frame === #
		self.prefs['autodl'] = gtk.CheckButton(unicode(_("Auto download when receive script contained URL")))
		self.prefs['autodl'].set_border_width(5)
		self.prefs['autodl'].connect('toggled', self.download_toggled)

		self.prefs['download_log'] = gtk.CheckButton(unicode(_("Logging history")))
		self.prefs['download_log'].set_border_width(5)
		self.prefs['download_log'].set_sensitive(gtk.FALSE)

		# make table
		rows = 4
		ex_table = gtk.Table(7, rows)

		keys = DL_SUFFIX_PATTERNS.keys()
		keys.sort()
		length = len(keys) / rows + 1
		list = [keys[length*i:length*(i+1)] for i in range(rows)]

		for i in range(len(list)):
			for j in range(len(list[i])):
				self.prefs[list[i][j]] = gtk.CheckButton(list[i][j])
				self.prefs[list[i][j]].set_sensitive(gtk.FALSE)
				ex_table.attach(self.prefs[list[i][j]], j, j+1, i, i+1, gtk.FILL, gtk.FILL, 3, 3)

		ex_frame = gtk.Frame(unicode(_('Supported file types'), 'utf-8'))
		ex_frame.set_shadow_type(gtk.SHADOW_ETCHED_OUT)
		ex_frame.set_border_width(5)
		ex_frame.add(ex_table)

		dlbox = gtk.VBox()
		dlbox.pack_start(self.prefs['autodl'], gtk.FALSE, gtk.FALSE, 0)
		dlbox.pack_start(self.prefs['download_log'], gtk.FALSE, gtk.FALSE, 0)
		dlbox.pack_start(ex_frame, gtk.FALSE, gtk.FALSE, 0)

		dlframe = gtk.Frame(unicode(_("Download"), 'utf-8'))
		dlframe.set_border_width(5)
		dlframe.add(dlbox)

# === Bring Browser, Download together to Log Tab === #
		net_tab = gtk.VBox()
		net_tab.pack_start(brsframe, gtk.FALSE, gtk.FALSE, 0)
		net_tab.pack_start(dlframe, gtk.FALSE, gtk.FALSE, 0)

# === Create LUID Tab === #
		if luid_info:
			luid_data = self.luid_info.get_all()

			if luid_data['info']:
				self.luidinfolist = gtk.CList(len(luid_data['info'][0]), luid_data['info'][0])
			else:
				self.luidinfolist = gtk.CList()
			self.luidinfolist.column_titles_passive()
			self.luidinfolist.set_column_auto_resize(1, gtk.TRUE)
			self.luidinfolist.set_shadow_type(gtk.SHADOW_IN)
			self.luidinfolist.set_selection_mode(gtk.SELECTION_BROWSE)
			self.luidinfolist.set_reorderable(gtk.TRUE)
			self.luidinfolist.set_column_width(0, 100)
			self.luidinfolist.set_column_auto_resize(1, gtk.TRUE)

			self.luidinfolist.freeze()
			self.luidinfolist.clear()
			for line in luid_data['info'][1:]:
				self.luidinfolist.append(tuple(line))
			self.luidinfolist.sort()
			self.luidinfolist.columns_autosize()
			self.luidinfolist.thaw()

			self.luidinfosw = gtk.ScrolledWindow()
			self.luidinfosw.set_size_request(300, 80)
			self.luidinfosw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
			self.luidinfosw.set_border_width(5)
			self.luidinfosw.add(self.luidinfolist)

			if luid_data['table']:
				self.luidlist = gtk.CList(len(luid_data['table'][0]), luid_data['table'][0])
			else:
				self.luidlist = gtk.CList()
			self.luidlist.column_titles_passive()
			self.luidlist.set_column_auto_resize(1, gtk.TRUE)
			self.luidlist.set_shadow_type(gtk.SHADOW_IN)
			self.luidlist.set_selection_mode(gtk.SELECTION_BROWSE)
			self.luidlist.set_reorderable(gtk.TRUE)
			self.luidlist.set_column_width(0, 100)
			self.luidlist.set_column_auto_resize(1, gtk.TRUE)

			self.luidlist.freeze()
			self.luidlist.clear()
			for line in luid_data['table'][1:]:
				self.luidlist.append(tuple(line))
			self.luidlist.sort()
			self.luidlist.columns_autosize()
			self.luidlist.thaw()

			self.luidsw = gtk.ScrolledWindow()
			self.luidsw.set_size_request(300, 100)
			self.luidsw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
			self.luidsw.set_border_width(5)
			self.luidsw.add(self.luidlist)

			luidbutton = gtk.Button(unicode(_('Update')))
			luidbutton.connect('clicked', self.refresh_luid_info)

			luidbox = gtk.VBox()
			luidbox.pack_start(self.luidinfosw, gtk.FALSE, gtk.FALSE, 0)
			luidbox.pack_start(self.luidsw, gtk.TRUE, gtk.TRUE, 0)
			luidbox.pack_end(luidbutton, gtk.FALSE, gtk.FALSE, 3)

# === Bring all Tabs together to Notebook === #
		main_notebook = gtk.Notebook()
		main_notebook.popup_enable()
		main_notebook.set_scrollable(gtk.TRUE)
		main_notebook.set_tab_pos(gtk.POS_TOP)

		for item in [
			[general_tab, gtk.Label(unicode(_("General")))],
			[log_tab,     gtk.Label(unicode(_("Logs")))],
			[net_tab,     gtk.Label(unicode(_("Network")))],
			[fonttab,     gtk.Label(unicode(_("Fonts")))],
			[viewer_tab,  gtk.Label(unicode(_("GViewer")))],
			]:
			main_notebook.append_page(item[0], item[1])

		if luid_info is not None:
			main_notebook.append_page(luidbox, gtk.Label(unicode(_("LUID"))))

# === Bring all items together to Main Window Finally === #
		self.window = gtk.Dialog(("%s: " % APP + unicode(_("Preferences"))), self.app.window,
								 gtk.DIALOG_DESTROY_WITH_PARENT|gtk.DIALOG_MODAL|gtk.DIALOG_NO_SEPARATOR,
								 (gtk.STOCK_APPLY, gtk.RESPONSE_APPLY,
								  gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT,
								  gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
		self.window.set_border_width(5)
		self.window.vbox.pack_start(main_notebook, gtk.TRUE, gtk.TRUE, 0)
		self.window.connect("destroy", self.close)
		self.window.show_all()

# === Initialization of PreferenceDialog === #
		self.prefmanager.load_database()
		self.set_prefs()

# === Start mainloop of Dialog === #
		while True:
			response = self.window.run()

			if response == gtk.RESPONSE_APPLY:
				self.prefmanager.save_database(self)
				self.app.notify_preferences_changed(self.get_prefs())
			elif response == gtk.RESPONSE_ACCEPT:
				self.prefmanager.save_database(self)
				self.app.notify_preferences_changed(self.get_prefs())
				self.close()
				break
			elif response == gtk.RESPONSE_REJECT:
				self.set_prefs()
				self.close()
				break

	def open_fontselection(self, widget=None, entry=None):
		fontsel = gtk.FontSelectionDialog(unicode(_('Chose font...'), 'utf-8'))
		fontsel.show()
		fontsel.set_font_name(entry.get_text())
		res = fontsel.run()
		fontsel.hide()
		if res == gtk.RESPONSE_OK:
			entry.set_text(fontsel.get_font_name())
		fontsel.destroy()

	def open_colorselection(self, widget=None, entry=None):
		colorsel = gtk.ColorSelectionDialog(unicode(_('Chose color...'), 'utf-8'))
		colorsel.show()
		colorsel.colorsel.set_current_color(gtk.gdk.color_parse(entry.get_text()))
		res = colorsel.run()
		colorsel.hide()
		if res == gtk.RESPONSE_OK:
			color = colorsel.colorsel.get_current_color()
			color_str = "#%04x%04x%04x" % (color.red, color.green, color.blue)
			entry.set_text(color_str)
		colorsel.destroy()

	def refresh_luid_info(self, widget=None, data=None):
		luid_data = self.luid_info.get_all(True)

		if luid_data['info'] and luid_data['table']:
			self.luidinfolist.freeze()
			self.luidinfolist.hide()
			self.luidinfolist.clear()
			self.luidinfosw.remove(self.luidinfolist)
			self.luidinfolist = gtk.CList(len(luid_data['info'][0]), luid_data['info'][0])
			self.luidinfolist.column_titles_passive()
			self.luidinfolist.set_column_auto_resize(1, gtk.TRUE)
			self.luidinfosw.add(self.luidinfolist)
			for line in luid_data['info'][1:]:
				self.luidinfolist.append(tuple(line))
			self.luidinfolist.columns_autosize()
			self.luidinfolist.thaw()
			self.luidinfolist.show()

			self.luidlist.freeze()
			self.luidlist.hide()
			self.luidlist.clear()
			self.luidsw.remove(self.luidlist)
			self.luidlist = gtk.CList(len(luid_data['table'][0]), luid_data['table'][0])
			self.luidlist.column_titles_passive()
			self.luidlist.set_column_auto_resize(1, gtk.TRUE)
			self.luidsw.add(self.luidlist)
			for line in luid_data['table'][1:]:
				self.luidlist.append(tuple(line))
			self.luidlist.columns_autosize()
			self.luidlist.thaw()
			self.luidlist.show()

	def close(self, widget=None, data=None):
		self.window.destroy()

	def get_prefs(self):
		prefs = {}

		# CheckButtons
		for name in ['auto_open_log', 'auto_connect', 'auto_join', 'autodl',
					 'forward', 'accept_all', 'forward_listed', 'auto_clear',
					 'logging', 'logging_current_tab_closed', 'logging_compress',
					 'open_gviewer_when_play', 'logtabfocus', 'download_log',
					 'sent_log', 'use_gviewer'] + DL_SUFFIX_PATTERNS.keys():
			prefs[name] = self.prefs[name].get_active()

		# SpinButtons
		for name in ['wait_w1_tag', 'wait_1_char']:
			prefs[name] = self.prefs[name].get_value_as_int()

		# Entries
		for name in ['browser', 'script', 'sakura',
					 'kero', 'sync', 'error', 'URL',
					 'initialscript', 'logging_path',
					 'general font', 'script font']:
			prefs[name] = self.prefs[name].get_text()

		# RadioButtons
		for name in [POS_NEXT, POS_PREV,
					 POS_FIRST, POS_END]:
			if self.prefs[name].get_active():
				prefs['logpagepos'] = name
				break
		for type in [gtk.POS_TOP, gtk.POS_BOTTOM, gtk.POS_LEFT, gtk.POS_RIGHT]:
			name = "LOGTABPOS_%s" % type
			if self.prefs[name].get_active():
				prefs['logtabpos'] = type
				break

		return prefs

	def set_prefs(self):
		for key in self.prefmanager.rc.keys():
			if self.prefs.has_key(key):
				# Entries
				if key in ['browser', 'script', 'sakura',
						   'kero', 'sync', 'error', 'URL',
						   'initialscript', 'logging_path',
						   'general font', 'script font']:
					self.prefs[key].set_text(self.prefmanager.rc[key])

				# SpinButtons
				elif key in ['wait_w1_tag', 'wait_1_char']:
					self.prefs[key].set_value(int(self.prefmanager.rc[key]))

				else:
					self.prefs[key].set_active(self.prefmanager.rc[key])

			if key == 'logpagepos':
				for type in [POS_NEXT, POS_PREV,
							 POS_FIRST, POS_END]:
					if self.prefmanager.rc[key] == type:
						self.prefs[type].set_active(gtk.TRUE)
						break
			elif key == 'logtabpos':
				for type in [gtk.POS_TOP, gtk.POS_BOTTOM, gtk.POS_LEFT, gtk.POS_RIGHT]:
					name = "LOGTABPOS_%s" % type
					if self.prefmanager.rc[key] == str(type):
						self.prefs[name].set_active(gtk.TRUE)
						break

		if not self.prefs['forward'].get_active():
			self.prefs['accept_all'].set_sensitive(gtk.FALSE)
			self.prefs['forward_listed'].set_sensitive(gtk.FALSE)

	def forward_toggled(self, widget=None, data=None):
		state = self.prefs['forward'].get_active()
		self.prefs['accept_all'].set_sensitive(state)
		self.prefs['forward_listed'].set_sensitive(state)

	def download_toggled(self, widget=None, data=None):
		state = self.prefs['autodl'].get_active()
		self.prefs['download_log'].set_sensitive(state)
		for p in DL_SUFFIX_PATTERNS.keys():
			self.prefs[p].set_sensitive(state)

	def logging_toggled(self, widget=None, data=None):
		logging_state = self.prefs['logging'].get_active()
		logcurr_state = self.prefs['logging_current_tab_closed'].get_active()

		if not logging_state and not logcurr_state:
			self.prefs['logging_path'].set_sensitive(gtk.FALSE)
			self.prefs['logging_compress'].set_sensitive(gtk.FALSE)
		else:
			self.prefs['logging_path'].set_sensitive(gtk.TRUE)
			self.prefs['logging_compress'].set_sensitive(gtk.TRUE)

	def viewer_toggled(self, widget=None, data=None):
		viewer_state = self.prefs['use_gviewer'].get_active()
		self.prefs['open_gviewer_when_play'].set_sensitive(viewer_state)
		self.prefs['wait_1_char'].set_sensitive(viewer_state)
		self.prefs['wait_w1_tag'].set_sensitive(viewer_state)


class PreferenceManager:
	DBFILE = "bottlerc"

	def __init__(self):
		self.dbpath = os.path.join(open_bottlecase(), self.DBFILE)
		self.rc = {}
		self.initialize()
		self.load_database()

	def initialize(self):
		for k, v in [['auto_open_log', False],
					 ['auto_connect', False],
					 ['auto_join', False],
					 ['autodl', False],
					 ['download_log', True],
					 ['forward', True],
					 ['accept_all', True],
					 ['forward_listed', False],
					 ['auto_clear', True],
					 ['sent_log', True],
					 ['use_gviewer', False],
					 ['open_gviewer_when_play', False],
					 ['initialscript', r'\t\u\s[10]\h\s[0]|\e'],
					 ['wait_1_char', 47],
					 ['wait_w1_tag', 50],
					 ['logging', False],
					 ['logging_current_tab_closed', False],
					 ['logging_compress', False],
					 ['logging_path', os.path.normpath(open_bottlecase() + '/autolog/%Y%m%d/%H-%M-%S.xml')],
					 ['logtabfocus', True],
					 ['logpagepos', POS_NEXT],
					 ['logtabpos', gtk.POS_TOP],
					 ['general font', 'Sans 10'],
					 ['script font', 'Monospace 10'],
					 ['browser', "mozilla -remote 'openFile(%s,new-tab)'"],
					 ['script', "dark green"],
					 ['sakura', "black"],
					 ['kero', "brown"],
					 ['sync', "blue"],
					 ['error', "red"],
					 ['URL', "purple"]] + [[i, True] for i in DL_SUFFIX_PATTERNS.keys()]:
			self.rc[k] = v

	def load_database(self):
		pat = re.compile('\s*:\s*')
		try:
			file = open(self.dbpath)
		except IOError:
			return
		while 1:
			line = unicode(file.readline(), "utf-8")
			if not line:
				break
			elif line[0] in ["#", ";"]:
				continue
			elif line[0:2] == "//":
				continue
			elif line[-2:] == "\r\n":
				line = line[:-2]
			elif line[-1] in ["\r", "\n"]:
				line = line[:-1]

			[name, value] = pat.split(line, 1)
			if value == 'True':
				value = True
			elif value == 'False':
				value = False
			self.rc[name] = value
		file.close()

	def save_database(self, prefdialog):
		try:
			file = open(self.dbpath, "w")
		except IOError:
			sys.stderr.write("Error: cannot write %s\n" % self.dbpath)
			return
		self.rc = prefdialog.get_prefs()
		file.write("# This is an auto-generated file.\n")
		file.write("# Don't edit this file while running %s!\n" % APP)
		for key in self.rc.keys():
			file.write("%s: %s\n" % (key.encode("utf-8"), self.rc[key]))
		file.close()

	def get(self, name):
		if not self.rc.has_key(name):
			raise ValueError("rc has no key %s\n" % name)
		return self.rc[name]

	def get_all(self):
		return self.rc
