# -*- coding: ascii -*-
#
#  client.py - bottle client part of GBottler
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: client.py,v 1.6 2004/11/01 12:06:50 atzm Exp $
#

import time, StringIO, mimetools

from bottlelib     import BottleClient
from downloader    import Downloader
from prefs         import PreferenceManager
from messagesender import MessageSender, VoteMessageSender
from common        import APP, VOTE, AGREE

class Bottler(BottleClient):
	def __init__(self, app, auto_download, download_patterns, download_logging):
		BottleClient.__init__(self)
		self.app = app

		self.auto_download    = auto_download
		self.download_pattern = download_patterns
		self.download_logging = download_logging

	def receive_preferences_changed(self, prefs):
		self.set_auto_download(prefs[PreferenceManager.PREF_DOWNLOAD])
		self.set_pattern(prefs)
		self.set_dl_logging(prefs[PreferenceManager.PREF_LOGGING_DOWNLOAD])

	def set_auto_download(self, setting):
		self.auto_download = bool(setting)

	def set_pattern(self, dic):
		self.download_pattern = dic

	def set_dl_logging(self, logging):
		self.download_logging = bool(logging)

	def handle_sstp_message(self, message, unicast, forced):
		# dispatch SSTP message to local SSTP server
		if not message:
			return

		# get unique values
		mid = self.headers.get('mid', '')
		sender = APP #self.headers.get('sender', 'SSTP Bottle')
		receive_time = time.localtime(int("0x%s" % mid[:8], 16)) # in few problem of accuracy
		if unicast:
			channel = 'unicast'
		elif forced:
			channel = 'broadcast'
		else:
			channel = self.headers.get('channel', '')

		# parse SSTP message
		file = StringIO.StringIO(message)
		request = file.readline()
		headers = mimetools.Message(file)

		try:
			script = headers["script"]
		except KeyError:
			return
		charset = headers.get("charset", "Shift_JIS")
		ifghost = headers.get('ifghost', 'default')

		# log message
		self.app.notify_log_update(mid[:], channel, ifghost, script, receive_time)

		# send local message
		self.app.send_local_message(channel, ifghost, script, sender, charset, mid[:])

		# download
		if self.auto_download:
			downloader = Downloader(verbose=True, dic=self.download_pattern,
									logging=self.download_logging, script=script)
			downloader.start()

	def handle_dialog_message(self, message):
		self.app.monitor_clear()
		self.app.monitor_insert(message)

	def handle_broadcast_information(self, type, forced):
		BottleClient.handle_broadcast_information(self, type, forced)
		if type in [VOTE, AGREE]:
			self.app.log_votes(self.headers["mid"], type, int(self.headers["num"]))

	def handle_all_users(self):
		self.app.show_users()

	def handle_channel_users(self, channel):
		self.app.show_users_of(channel)

	def handle_close_channel(self, channel):
		self.app.close_channel(channel)

	def send_message(self, auto_clear, sent_log, channel, script, ghost):
		ms = MessageSender(self.app, self, auto_clear, sent_log, channel, script, ghost)
		ms.start()

	def send_vote_message(self, logwindow, mid, type):
		vms = VoteMessageSender(self, logwindow, mid, type)
		vms.start()
