# -*- coding: ascii -*-
#
#  localmessenger.py - Communicates local SSTP Server
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: localmessenger.py,v 1.17 2004/10/01 22:16:55 atzm Exp $
#

import string, StringIO, re, time, socket
import gtk

from threading import Thread
from bottlelib import BottleClientError
from common    import APP

class LocalMessageSender(Thread):
	def __init__(self, messenger, logwindow, interval_default=1.2, interval_not_empty=1.0):
		self.messenger = messenger
		self.logwindow = logwindow
		self.interval_default = interval_default
		self.interval_not_empty = interval_not_empty
		self.message_queue = []

		Thread.__init__(self)
		self.setDaemon(True)

	def play_cancel_page_of(self, list):
		for i in range(len(self.message_queue)):
			if self.message_queue[i][2] is list:
				del self.message_queue[i]
				self.play_cancel_page_of(list)
				return

	def find_mid(self, items):
		return items[0]
	def get_index_from_mid(self, mid):
		try:
			return map(self.find_mid, self.message_queue).index(mid)
		except:
			return -1
	def cancel(self, mid):
		i = self.get_index_from_mid(mid)
		if i != -1:
			del self.message_queue[i]
			return True
		return False

	def cancel_all(self):
		self.message_queue = []
		self.logwindow.clear_playing_bottle()

	def enqueue(self, command, mid, list):
		if self.messenger.get_execute_check_queue() is not None: # check server
			self.logwindow.set_reserve_bottle(mid, list)
		self.message_queue.append([mid, command, list])

	SSTP_SUCCESS_REGEX = re.compile('SSTP/\d(\.\d)+ 2\d\d')
	def run(self):
		prev_list   = None  # keep prev list
		now_playing = False # to stop polling local SSTP server
		while True:
			if not self.message_queue and not now_playing:
				time.sleep(self.interval_default)
				continue

			status = self.messenger.get_execute_check_queue()
			if status == 0:
				if self.message_queue:
					mid, command, list = self.message_queue.pop(0)
					self.set_playing_bottle(mid, list, prev_list)
					result = self.messenger.send_sstp_message(command)

					if not self.SSTP_SUCCESS_REGEX.search(self.get_status(result)):
						self.logwindow.set_reserve_bottle(mid, list)
						self.message_queue.insert(0, [mid, command, list])
						continue

					prev_list = list
					now_playing = True
				else:
					self.clear_playing_bottle(prev_list)
					now_playing = False
					time.sleep(self.interval_default)
			elif status is None:
				if self.message_queue:
					del self.message_queue[0]
				now_playing = False
				time.sleep(self.interval_default)
			else:
				time.sleep(self.interval_not_empty)

	def set_playing_bottle(self, mid, list, prev_list):
		gtk.gdk.threads_enter()
		self.logwindow.set_playing_bottle(mid, list, prev_list)
		gtk.gdk.flush()
		gtk.gdk.threads_leave()

	def clear_playing_bottle(self, list=None):
		gtk.gdk.threads_enter()
		self.logwindow.clear_playing_bottle(list)
		gtk.gdk.flush()
		gtk.gdk.threads_leave()

	def get_status(self, result):
		if result is None:
			return 'SSTP/1.0 503 Service Unavailable'
		return string.split(result, '\r\n', 1)[0]


class LocalMessenger:
	SSTP_SEND = (
		"SEND SSTP/1.4\r\n"
		"Sender: %s / %s / %s\r\n"
		"IfGhost: %s\r\n"
		"Script: %s\r\n"
		"Option: notranslate\r\n"
		"Charset: %s\r\n"
		"\r\n")

	SSTP_EXECUTE = (
		"EXECUTE SSTP/1.0\r\n"
		"Sender: %s\r\n"
		"Command: %s\r\n"
		"Option: notranslate\r\n"
		"Charset: Shift_JIS\r\n"
		"\r\n")

	def __init__(self, host, port, logwindow):
		self.host = host
		self.port = port
		self.local_message_sender = LocalMessageSender(self, logwindow)
		self.local_message_sender.start()

	def send_local_message(self, channel, ifghost, script, sender=APP, charset='Shift_JIS', mid=None, list=None):
		command = self.SSTP_SEND % (sender, channel, ifghost, ifghost, script, charset)

		if mid is None or list is None:
			self.send_sstp_message(command)
		else:
			self.local_message_sender.enqueue(command, mid, list)

	def cancel_message_as(self, mid):
		return self.local_message_sender.cancel(mid)

	def cancel_message_all(self):
		self.local_message_sender.cancel_all()

	def notify_play_cancel_page_of(self, list):
		self.local_message_sender.play_cancel_page_of(list)

	def get_execute_check_queue(self):
		try:
			data = self.send_sstp_message(self.SSTP_EXECUTE % (APP, 'CheckQueue'))
		except BottleClientError, e:
			return 0

		if data is None:
			return None

		file = StringIO.StringIO(data)
		request = file.readline()

		if request[:12] != "SSTP/1.0 200":
			return 0
		for line in file:
			line = string.strip(line, '\r\n\r\n')
			if not line: continue
			return int(line)

	def get_execute(self, command):
		if command == 'GetNames':
			default_returns = []
		else:
			default_returns = [None, None]
			
		try:
			data = self.send_sstp_message(self.SSTP_EXECUTE % (APP, command))
		except BottleClientError, e:
			return default_returns

		file = StringIO.StringIO(data)
		request = file.readline()

		if request[:12] != "SSTP/1.0 200":
			return default_returns

		names = []
		charset = ''
		while True:
			line = file.readline()
			if not line:
				break
			line = string.strip(line, '\r\n\r\n')
			if not line:
				continue
			elif string.lower(line[:9]) == 'charset: ':
				charset = line[9:]
				continue
			names.append(line)

		if not charset:
			charset = 'sjis'

		for i in range(len(names)):
			names[i] = unicode(names[i], charset)

		if command == 'GetNames':
			return names
		else:
			return re.split('\s*,\s*', names[0], 1)

	def send_ghost_names(self):
		ghosts = self.get_execute('GetNames')
		if ghosts:
			return ghosts

		sakura, kero = self.get_execute('GetName')
		if sakura:
			return [sakura]

		return []

	def send_sstp_message(self, message):
		buffer = []
		try:
			message = message.encode("sjis")
		except UnicodeEncodeError, e:
			raise BottleClientError("Detected illegal code: " + str(e))
		s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
		try:
			s.connect((self.host, self.port))
			s.send(message)
			while 1:
				buffer.append(s.recv(1024))
				if not buffer[-1]:
					break
			s.close()
		except socket.error, (code, message):
			return None
		return string.join(buffer, '')
