#!/usr/bin/env python

import os
import sys
import time
import string
import re
import mimetools
import StringIO
import fcntl
import gettext
from types import *

if os.environ.has_key("DISPLAY"):
	if not sys.modules.has_key('gtk'):
		try:
			import pygtk
			pygtk.require("2.0")
		except ImportError:
			pass
	import gtk

from bottlelib import BottleClient, BottleClientError

try:
	from ninix.script import *
except:
	from script import *

gettext.install('gbottler')

LIBDIR = "~/.bottlecase"

# vote types
VOTE  = "Vote"
AGREE = "Agree"

APP = 'GBottler'
VER = '0.0.3'

class Bottler(BottleClient):
	def __init__(self, app):
		BottleClient.__init__(self, app.pbar)
		self.app = app

	def handle_sstp_message(self, message, unicast, forced):
		if not message:
			return
		# dispatch SSTP message to local SSTP server
		ifghost = re.search("IfGhost: ?(.+?)\r\n", message).group(1)

		# bad /-P
		m = re.split("(Sender: ?.+?)\r\n", message)
		m.insert(2, " / " + ifghost + "\r\n")
		message = string.join(m, "")

		if self.app.menu_list["/%s/%s/%s" % (self.app.msg["edit"][1], self.app.msg["pref"][1],
											 self.app.msg["forward"])].get_active():
			if not self.app.menu_list["/%s/%s/%s" % (self.app.msg["edit"][1],
													 self.app.msg["pref"][1],
													 self.app.msg["aa"])].get_active():
				if self.app.sakuraname == ifghost or \
					   self.app.target_ghost == ifghost or \
					   self.app.target_ghost == Application.DEFAULT_TARGET:
					BottleClient.handle_sstp_message(self, message, unicast, forced)
			else:
				BottleClient.handle_sstp_message(self, message, unicast, forced)

		# parse SSTP message
		file = StringIO.StringIO(message)
		request = file.readline()
		headers = mimetools.Message(file)
		charset = headers.get("charset", "Shift_JIS")
		if unicast:
			channel = "unicast"
		else:
			mid = self.headers["mid"]
			if forced:
				channel = "broadcast"
			else:
				channel = self.headers["channel"]
		script = headers["script"]

		# log message
		self.app.log_message(mid, channel, ifghost, script)

		# log: auto update view
		if self.app.log_window and \
			   self.app.log_window.item_factory.get_widget("/%s/%s" % (self.app.log_window.msg["view"][1],
																	   self.app.log_window.msg["auto update"])).get_active():
			self.app.log_window.update(None, None)

		#print "Debug msg: ", ifghost, ", ", script, "\n", message

	def handle_dialog_message(self, message):
		self.app.monitor_clear()
		self.app.monitor_insert(message)
		self.app.window.bell()

	def handle_broadcast_information(self, type, forced):
		BottleClient.handle_broadcast_information(self, type, forced)
		if type in [VOTE, AGREE]:
			self.app.log_votes(
				self.headers["mid"], type, int(self.headers["num"]))

	def handle_all_users(self):
		self.app.show_users()

	def handle_channel_users(self, channel):
		pass

	def handle_close_channel(self, channel):
		self.app.close_channel(channel)

class Application:
	def edit_cut(self, data, widget):
		self.main_text.emit("cut-clipboard")

	def edit_copy(self, data, widget):
		self.main_text.emit("copy-clipboard")

	def edit_paste(self, data, widget):
		self.main_text.emit("paste-clipboard")

	def edit_delete(self, data, widget):
		self.main_text.emit("delete-from-cursor", gtk.DELETE_CHARS, 1)

	def get_imglabelbox_with_stock(self, stock_id, size, label):
		h = gtk.HBox(gtk.FALSE, 0)
		i = gtk.Image()
		i.set_from_stock(stock_id, size)
		l = gtk.Label(label)
		h.pack_start(i, gtk.FALSE, gtk.TRUE, 0)
		h.pack_start(l, gtk.FALSE, gtk.TRUE, 0)
		i.show()
		l.show()
		h.show()
		return h

	DEFAULT_TARGET = unicode(_("default"), "utf-8")
	def __init__(self):
		self.msg = {
			"file":   [unicode(_("File(_F)"), "utf-8"),
					   unicode(_("File(F)"), "utf-8")],
			"join":   unicode(_("Join"), "utf-8"),
			"part":   unicode(_("Part"), "utf-8"),
			"quit":   unicode(_("Quit"), "utf-8"),
			"edit":   [unicode(_("Edit(_E)"), "utf-8"),
					   unicode(_("Edit(E)"), "utf-8")],
			"cut":    unicode(_("Cut"), "utf-8"),
			"copy":   unicode(_("Copy"), "utf-8"),
			"paste":  unicode(_("Paste"), "utf-8"),
			"delete": unicode(_("Delete"), "utf-8"),
			"clear":  unicode(_("Clear"), "utf-8"),
			"ca":     unicode(_("Clear All"), "utf-8"),
			"pref":   [unicode(_("Preference(_P)"), "utf-8"),
					   unicode(_("Preference(P)"), "utf-8")],
			"forward":unicode(_("Forward"), "utf-8"),
			"aa":     unicode(_("Accept All"), "utf-8"),
			"ceas":   unicode(_("Clear editor after send"), "utf-8"),
			"channel":[unicode(_("Channel(_C)"), "utf-8"),
					   unicode(_("Channel(C)"), "utf-8")],
			"ghost":  [unicode(_("Ghost(_G)"), "utf-8"),
					   unicode(_("Ghost(G)"), "utf-8")],
			"window": [unicode(_("Window(_W)"), "utf-8"),
					   unicode(_("Window(W)"), "utf-8")],
			"logs":   unicode(_("Logs"), "utf-8"),
			"help":   [unicode(_("Help(_H)"), "utf-8"),
					   unicode(_("Help(H)"), "utf-8")],
			"about":  unicode(_("About"), "utf-8"),
			}

		menu_items = (
			( "/%s" % self.msg["file"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.msg["file"][1], self.msg["join"]), "<control>J", self.join, 0, None ),
			( "/%s/%s" % (self.msg["file"][1], self.msg["part"]), "<control>P", self.part, 0, None ),
			( "/%s/sep1" % self.msg["file"][1], None, None, 0, "<Separator>" ),
			( "/%s/%s" % (self.msg["file"][1], self.msg["quit"]), "<control>Q", self.quit, 0, None ),

			( "/%s" % self.msg["edit"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["cut"]), "<control>X", self.edit_cut, 0, None ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["copy"]), "<control>C", self.edit_copy, 0, None ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["paste"]), "<control>V", self.edit_paste, 0, None ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["delete"]), "<control>D", self.edit_delete,0, None ),
			( "/%s/sep4" % self.msg["edit"][1], None, None, 0, "<Separator>" ),

			( "/%s/%s" % (self.msg["edit"][1], self.msg["clear"]), None, self.edit_clear, 0, None ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["ca"]), None, self.edit_clear_all, 0, None ),
			( "/%s/sep5" % self.msg["edit"][1], None, None, 0, "<Separator>" ),

			( "/%s/%s" % (self.msg["edit"][1], self.msg["pref"][0]), None, None, 0, "<Branch>" ),
			( "/%s/%s/%s" % (self.msg["edit"][1], self.msg["pref"][0], self.msg["forward"]), None, None, 0, "<CheckItem>" ),
			( "/%s/%s/%s" % (self.msg["edit"][1], self.msg["pref"][0], self.msg["aa"]), None, None, 0, "<CheckItem>" ),
			( "/%s/%s/%s" % (self.msg["edit"][1], self.msg["pref"][0], self.msg["ceas"]), None, None, 0, "<CheckItem>" ),

			( "/%s" % self.msg["channel"][0], None, None, 0, "<Branch>" ),
			( "/%s" % self.msg["ghost"][0], None, None, 0, "<Branch>" ),

			( "/%s" % self.msg["window"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.msg["window"][1], self.msg["logs"]), "<control>L", None, 0, None ),

			( "/%s" % self.msg["help"][0], None, None, 0, "<LastBranch>" ),
			( "/%s/%s" % (self.msg["help"][1], self.msg["about"]), None, self.about, 0, None ),
			)

		self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
		self.window.connect("destroy", self.quit, "WM destroy")
		self.window.set_title(unicode(_("GBottler"), "utf-8"))

		self.main_vbox = gtk.VBox(gtk.FALSE, 0)
		self.window.add(self.main_vbox)
		self.main_vbox.show()

		accel_group = gtk.AccelGroup()

		self.item_factory = gtk.ItemFactory(gtk.MenuBar, "<main>", accel_group)
		self.item_factory.create_items(menu_items)

		self.window.add_accel_group(accel_group)

		self.menu_list = {}
		for m in menu_items:
			key = re.sub("_", "", m[0])
			self.menu_list[key] = self.item_factory.get_item(key)

		for m in ["/%s/%s" % (self.msg["file"][1], self.msg["part"]),
				  "/%s" % self.msg["channel"][1],
				  "/%s" % self.msg["ghost"][1]]:
			self.menu_list[m].set_sensitive(gtk.FALSE)

		for m in ["/%s/%s/%s" % (self.msg["edit"][1], self.msg["pref"][1], self.msg["forward"]),
				  "/%s/%s/%s" % (self.msg["edit"][1], self.msg["pref"][1], self.msg["aa"]),
				  "/%s/%s/%s" % (self.msg["edit"][1], self.msg["pref"][1], self.msg["ceas"])]:
			self.menu_list[m].set_active(gtk.TRUE)

		self.menubar = self.item_factory.get_widget("<main>")

		self.main_vbox.pack_start(self.menubar, gtk.FALSE, gtk.TRUE, 0)
		self.menubar.show()
		# End of creating Menubar

		check_h = self.get_imglabelbox_with_stock(gtk.STOCK_SPELL_CHECK,
												  gtk.ICON_SIZE_BUTTON,
												  unicode(_("Check"), "utf-8"))
		checkButton = gtk.Button()
		checkButton.add(check_h)
		checkButton.connect("clicked", self.check, None)
		check_h.show()

		test_h = self.get_imglabelbox_with_stock(gtk.STOCK_REFRESH,
												 gtk.ICON_SIZE_BUTTON,
												 unicode(_("Test"), "utf-8"))
		testButton = gtk.Button()
		testButton.add(test_h)
		testButton.connect("clicked", self.test, None)
		test_h.show()

		send_h =  self.get_imglabelbox_with_stock(gtk.STOCK_OK,
												  gtk.ICON_SIZE_BUTTON,
												  unicode(_("Send"), "utf-8"))
		self.send_button = gtk.Button()
		self.send_button.add(send_h)
		self.send_button.connect("clicked", self.send, None)
		send_h.show()

		channelLabel	   = gtk.Label(unicode(_("Channel") + ": ", "utf-8"))

		self.channel	   = gtk.Label()
		self.c_e_box       = gtk.EventBox()
		self.c_e_box.set_events(gtk.gdk.BUTTON_PRESS_MASK)
		self.c_e_box.add(self.channel)
		self.c_e_box.show()

		self.channel_ghost = gtk.Label()
		self.g_e_box       = gtk.EventBox()
		self.g_e_box.set_events(gtk.gdk.BUTTON_PRESS_MASK)
		self.g_e_box.add(self.channel_ghost)
		self.g_e_box.show()

		usersLabel		 = gtk.Label("   " + unicode(_("Users") + ": ", "utf-8"))
		self.users		 = gtk.Label()

		group2 = gtk.HBox(gtk.FALSE, 0)
		group2.set_border_width(1)
		self.main_vbox.pack_start(group2, gtk.FALSE, gtk.TRUE, 0)
		group2.show()

		group2.pack_start(checkButton, gtk.FALSE, gtk.TRUE, 0)
		group2.pack_start(testButton, gtk.FALSE, gtk.TRUE, 0)
		group2.pack_start(self.send_button, gtk.FALSE, gtk.TRUE, 0)

		group2.pack_start(channelLabel, gtk.FALSE, gtk.TRUE, 0)
		group2.pack_start(self.c_e_box, gtk.FALSE, gtk.TRUE, 0)
		group2.pack_start(self.g_e_box, gtk.FALSE, gtk.TRUE, 0)
		group2.pack_start(usersLabel, gtk.FALSE, gtk.TRUE, 0)
		group2.pack_start(self.users, gtk.FALSE, gtk.TRUE, 0)

		checkButton.show()
		testButton.show()
		self.send_button.set_sensitive(gtk.FALSE)
		self.send_button.show()

		channelLabel.show()
		self.channel.show()
		self.channel_ghost.show()
		usersLabel.show()
		self.users.show()

		sw = gtk.ScrolledWindow()
		sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		self.main_text = gtk.TextView()
		self.main_text.set_cursor_visible(gtk.TRUE)
		self.main_text.set_size_request(500,100)
		self.editor = self.main_text.get_buffer()
		self.editor.set_text(r"\t\u\s[10]\h\s[0]\e")
		self.main_text.set_editable(gtk.TRUE)
		self.main_text.set_wrap_mode(gtk.WRAP_CHAR)
		sw.add(self.main_text)
		sw.show()
		#self.main_vbox.add(sw)
		self.main_text.show()
		self.main_text.grab_focus()
		self.main_text.emit("move-cursor", gtk.MOVEMENT_LOGICAL_POSITIONS, -2, gtk.FALSE)

		self.pbar = gtk.ProgressBar()
		self.pbar.set_size_request(500, 20)
		#self.main_vbox.pack_start(self.pbar, gtk.FALSE, gtk.TRUE, 0)
		self.pbar.show()

		self.paned = gtk.VPaned()
		self.paned.pack1(sw)
		self.paned.pack2(self.pbar)
		self.main_vbox.pack_start(self.paned, gtk.TRUE, gtk.TRUE, 0)
		self.paned.show()

		pmsg = {
			"sessions":     unicode(_("Sessions"), "utf-8"),
			"timecritical": unicode(_("TimeCritical"), "utf-8"),
			"quick":        unicode(_("Quick"), "utf-8"),
			"syncronized":  unicode(_("Syncronized"), "utf-8"),

			"sakura":       unicode(_("Sakura"), "utf-8"),
			"countenance":  unicode(_("Countenance"), "utf-8"),
			"general":      unicode(_("General"), "utf-8"),
			"shy":          unicode(_("Shy"), "utf-8"),
			"surprised":    unicode(_("Surprised"), "utf-8"),
			"angst":        unicode(_("Angst"), "utf-8"),
			"ease off":     unicode(_("Ease off"), "utf-8"),
			"smile":        unicode(_("Smile"), "utf-8"),
			"closed eyes":  unicode(_("Closed eyes"), "utf-8"),
			"angry":        unicode(_("Angry"), "utf-8"),
			"scorned":      unicode(_("Scorned"), "utf-8"),
			"signboard":    unicode(_("Signboard"), "utf-8"),
			"singing":      unicode(_("Singing"), "utf-8"),

			"unyu":         unicode(_("Unyu"), "utf-8"),
			"opened eyes":  unicode(_("Opened Eyes"), "utf-8"),

			"wait":         unicode(_("Wait"), "utf-8"),
			"sec":          unicode(_(" sec"), "utf-8"),

			"meta string":  unicode(_("Meta String"), "utf-8"),
			"user name":    unicode(_("User name"), "utf-8"),
			"self name":    unicode(_("Self name"), "utf-8"),
			"true":         unicode(_("True"), "utf-8"),
			"false":        unicode(_("False"), "utf-8"),
			"kero name":    unicode(_("Kero name"), "utf-8"),

			"new line":     unicode(_("New line"), "utf-8"),
			"clear":        unicode(_("Clear"), "utf-8"),

			"questions":    unicode(_("Questions"), "utf-8"),
			"start":        unicode(_("Start"), "utf-8"),
			"no scroll":    unicode(_("no scroll"), "utf-8"),

			"terminate":    unicode(_("Terminate"), "utf-8"),
			}

		popup_menu_items = (
			( "/sep0", None, None, 0, "<Separator>" ),
			( "/%s" % pmsg["sessions"], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (pmsg["sessions"], pmsg["timecritical"]), None, self.insert_t, 0, None ),
			( "/%s/%s" % (pmsg["sessions"], pmsg["quick"]), None, self.insert_q, 0, None ),
			( "/%s/%s" % (pmsg["sessions"], pmsg["syncronized"]), None, self.insert_s, 0, None ),
			( "/sep1", None, None, 0, "<Separator>" ),

			( "/%s" % pmsg["sakura"], None, self.insert_h, 0, None ),
			( "/%s" % pmsg["countenance"], None, None, 0, "<Branch>" ),
			( "/%s/[0]  %s" % (pmsg["countenance"], pmsg["general"]), "<control><alt>0", self.insert_s0,  0, None, ),
			( "/%s/[1]  %s" % (pmsg["countenance"], pmsg["shy"]), "<control><alt>1", self.insert_s1, 0, None, ),
			( "/%s/[2]  %s" % (pmsg["countenance"], pmsg["surprised"]), "<control><alt>2", self.insert_s2, 0, None, ),
			( "/%s/[3]  %s" % (pmsg["countenance"], pmsg["angst"]), "<control><alt>3", self.insert_s3, 0, None, ),
			( "/%s/[4]  %s" % (pmsg["countenance"], pmsg["ease off"]), "<control><alt>4", self.insert_s4, 0, None, ),
			( "/%s/[5]  %s" % (pmsg["countenance"], pmsg["smile"]), "<control><alt>5", self.insert_s5, 0, None, ),
			( "/%s/[6]  %s" % (pmsg["countenance"], pmsg["closed eyes"]), "<control><alt>6", self.insert_s6, 0, None, ),
			( "/%s/[7]  %s" % (pmsg["countenance"], pmsg["angry"]), "<control><alt>7", self.insert_s7, 0, None, ),
			( "/%s/[8]  %s" % (pmsg["countenance"], pmsg["scorned"]), "<control><alt>8", self.insert_s8, 0, None, ),
			( "/%s/[20] %s" % (pmsg["countenance"], pmsg["signboard"]), None, self.insert_s20, 0, None, ),
			( "/%s/[25] %s" % (pmsg["countenance"], pmsg["singing"]), None, self.insert_s25, 0, None, ),
			( "/sep2", None, None, 0, "<Separator>" ),

			( "/%s" % pmsg["unyu"], None, self.insert_u, 0, None ),
			( "/%s" % pmsg["countenance"], None, None, 0, "<Branch>" ),
			( "/%s/[10] %s" % (pmsg["countenance"], pmsg["general"]), "<control><alt><shift>g", self.insert_s10, 0, None ),
			( "/%s/[11] %s" % (pmsg["countenance"], pmsg["opened eyes"]), None, self.insert_s11, 0, None ),
			( "/%s/[19] %s" % (pmsg["countenance"], pmsg["singing"]), None, self.insert_s19, 0, None ),
			( "/sep3", None, None, 0, "<Separator>" ),

			( "/%s" % pmsg["wait"], None, None, 0, "<Branch>" ),
			( "/%s/0.1%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w2, 0, None ),
			( "/%s/0.2%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w4, 0, None ),
			( "/%s/0.3%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w6, 0, None ),
			( "/%s/0.4%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w8, 0, None ),
			( "/%s/0.5%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w10, 0, None ),
			( "/%s/0.6%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w12, 0, None ),
			( "/%s/0.9%s" % (pmsg["wait"], pmsg["sec"]), "<control><alt>w", self.insert_w18, 0, None ),
			( "/%s/1.0%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w20, 0, None ),
			( "/%s/1.2%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w24, 0, None ),
			( "/%s/1.5%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w30, 0, None ),
			( "/%s/2.0%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w40, 0, None ),
			( "/%s/2.5%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w50, 0, None ),
			( "/%s/3.0%s" % (pmsg["wait"], pmsg["sec"]), None, self.insert_w60, 0, None ),

			( "/%s" % pmsg["meta string"], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (pmsg["meta string"], pmsg["user name"]), None, self.insert_username, 0, None ),
			( "/%s/%s (%s)" % (pmsg["meta string"], pmsg["self name"], pmsg["true"]), None, self.insert_selfname, 0, None ),
			( "/%s/%s (%s)" % (pmsg["meta string"], pmsg["self name"], pmsg["false"]), None, self.insert_selfname2, 0, None),
			( "/%s/%s" % (pmsg["meta string"], pmsg["kero name"]), None, self.insert_keroname, 0, None ),
			( "/sep4", None, None, 0, "<Separator>" ),

			( "/%s" % pmsg["new line"], "<control><alt>n", self.insert_n, 0, None ),
			( "/%s" % pmsg["clear"], None, self.insert_c, 0, None ),
			( "/URL", None, self.insert_url, 0, None ),

			( "/%s" % pmsg["questions"], None, None, 0, "<Branch>" ),
			( "/%s/Q0" % pmsg["questions"], None, self.insert_q0, 0, None ),
			( "/%s/Q1" % pmsg["questions"], None, self.insert_q1, 0, None ),
			( "/%s/Q2" % pmsg["questions"], None, self.insert_q2, 0, None ),
			( "/%s/Q3" % pmsg["questions"], None, self.insert_q3, 0, None ),
			( "/%s/Q4" % pmsg["questions"], None, self.insert_q4, 0, None ),
			( "/%s/Q5" % pmsg["questions"], None, self.insert_q5, 0, None ),
			( "/%s/Q6" % pmsg["questions"], None, self.insert_q6, 0, None ),
			( "/%s/Q7" % pmsg["questions"], None, self.insert_q7, 0, None ),
			( "/%s/Q8" % pmsg["questions"], None, self.insert_q8, 0, None ),
			( "/%s/Q9" % pmsg["questions"], None, self.insert_q9, 0, None ),
			( "/%s/%s" % (pmsg["questions"], pmsg["start"]), None, self.insert_z, 0, None ),
			( "/%s/%s (%s)" % (pmsg["questions"], pmsg["start"], pmsg["no scroll"]), None, self.insert_y, 0, None),

			( "/%s" % pmsg["terminate"], "<control><alt>e", self.insert_e, 0, None ),
			)

		self.popup_item_factory = gtk.ItemFactory(gtk.Menu, "<main>", accel_group)
		self.popup_item_factory.create_items(popup_menu_items)
		self.popup_menu = self.popup_item_factory.get_widget("<main>")
		self.main_text.connect("button-press-event", self.popup)

		self.window.show()

		self.client = Bottler(self)
		self.client.debug = 2

		self.parser = Parser('loose')

		self.host = ""
		self.port = 9801
		self.local_ghost_last_notified = 0
		self.target_ghost = ""

		self.to_sjis = lambda x: x.encode("sjis")
		self.sjis_to = lambda x: unicode(x, "sjis")

		try:
			luid = open(os.path.join(open_bottlecase(), "luid")).read()
		except IOError:
			luid = None
		if luid and len(luid) == 82:
			self.client.luid = luid
		self.sakuraname = ""
		self.keroname = ""

		self.logger = LogManager()
		self.log_window = LogWindow(self, self.logger)
		if self.logger.is_disabled():
			d = gtk.Dialog(unicode(_("Warning"), "utf-8"), parent=self.window,
						   flags=gtk.DIALOG_NO_SEPARATOR|gtk.DIALOG_MODAL)
			d.connect("destroy", lambda w: w.destroy)
			d.set_border_width(10)
			l = gtk.Label(unicode(_("Warning"), "utf-8") + ": " + \
						  unicode(_("another bottler.py compatible SSTP bottle client is running!"), "utf-8") + "\n\n" + \
						  unicode(_("logging disabled."), "utf-8"))
			l.set_justify(gtk.JUSTIFY_RIGHT)
			l.show()
			d.vbox.add(l)
			d.add_button(unicode(_("OK"), "utf-8"), True)
			d.show()
			d.run()
			d.destroy()

		self.menu_list["/%s/%s" % (self.msg["window"][1],
								   self.msg["logs"])].connect("activate", self.log_window.open, None)

		self.about = None

		self.local_ghost = None
		self.local_ghost_last_notified = 0
		self.send_ghost_names()
		# initialize
		self.c_e_id = 1 # channel label eventbox handler id
		self.g_e_id = 1
		self.disable_channel()

		self.timeout_id = None

	def main(self):
		gtk.main()
		return 0

	def popup(self, widget, event):
		if event.button == 3:
			self.popup_menu.popup(None, None, None, event.button, 0)

	def join(self, data, widget):
		if not self.client.initialized:
			try:
				self.client.init()
			except BottleClientError, e:
				return
			filename = os.path.join(open_bottlecase(), "luid")
			try:
				open(filename, "w").write(self.client.luid)
			except IOError:
				self.monitor_insert("\n" + unicode(_("Warning"), "utf-8") + \
									": cannot write " + filename)
				self.window.bell()

		channels = ChannelDialog(self, self.window, self.client.channels).go()
		self.pbar.set_fraction(0.0)
		self.monitor_clear()

		if channels is None:
			return
		if self.client.join(channels):
			return

		self.pbar.set_fraction(0.0)
		self.monitor_clear()
		self.enable_channel()
		self.timeout_id = gtk.timeout_add(100, self.handle_event)

	def button_popup(self, widget, event, sender):
		if event.button in [1, 3]:
			sender.popup(None, None, None, event.button, 0)

	def enable_channel(self):
		# rebuild Channel menu
		channel_item = self.menu_list["/%s" % self.msg["channel"][1]]
		channel_item.deselect()
		channel_item.set_sensitive(gtk.FALSE)
		channel_item.remove_submenu()

		channels = []
		for n in self.client.in_channels:
			name = self.client.channels[n]["name"]
			channels.append(name)
			if not self.channel.get_text():
				self.channel.set_text(name)

		channel_menu = []
		for n in range(len(channels)):
			if n == 0:
				channel_menu.append(("/"+channels[n], None, None, 0, "<RadioItem>"))
			else:
				channel_menu.append(("/"+channels[n], None, None, 0, "/"+channels[n-1]))
		channel_menu = tuple(channel_menu)

		self.channel_factory = gtk.ItemFactory(gtk.Menu, "<main>")
		self.channel_factory.create_items(channel_menu)

		for n in range(len(channels)):
			self.channel_factory.get_item(channel_menu[n][0]).connect("toggled",
																	  self.update_channel,
																	  channels[n])

		channel_item.set_submenu(self.channel_factory.get_widget("<main>"))
		channel_item.set_sensitive(gtk.TRUE)

		if not self.c_e_box.handler_is_connected(self.c_e_id):
			self.c_e_id = self.c_e_box.connect("button-press-event", self.button_popup,
											   self.channel_factory.get_widget("<main>"))

		self.rebuild_ghostmenu()
		self.menu_list["/%s/%s" % (self.msg["file"][1], self.msg["part"])].set_sensitive(gtk.TRUE)
		self.menu_list["/%s/%s" % (self.msg["file"][1], self.msg["join"])].set_sensitive(gtk.FALSE)
		self.log_window.vb.set_sensitive(gtk.TRUE)
		self.log_window.ab.set_sensitive(gtk.TRUE)
		self.log_window.vi.set_sensitive(gtk.TRUE)
		self.log_window.ai.set_sensitive(gtk.TRUE)
		self.show_users()

	def update_channel(self, widget, data):
		self.channel.set_text(data)
		self.check_channel_permission()

	def rebuild_ghostmenu(self):
		# rebuild Ghost menu
		ghost_item = self.menu_list["/%s" % self.msg["ghost"][1]]
		ghost_item.deselect()
		ghost_item.set_sensitive(gtk.FALSE)

		if self.g_e_box.handler_is_connected(self.g_e_id):
			self.g_e_box.disconnect(self.g_e_id)

		current = self.target_ghost
		ghosts = [self.DEFAULT_TARGET]
		ghosts.append(self.sakuraname)
		name = ""
		for n in range(len(self.client.channels)):
			name = self.client.channels[n]["ghost"]
			if name and name not in ghosts:
				ghosts.append(name)

		ghost_menu = []
		for n in range(len(ghosts)):
			if n == 0:
				ghost_menu.append(("/"+ghosts[n], None, None, 0, "<RadioItem>"))
			else:
				ghost_menu.append(("/"+ghosts[n], None, None, 0, "/"+ghosts[n-1]))
		ghost_menu = tuple(ghost_menu)

		self.g_ifact = gtk.ItemFactory(gtk.Menu, "<main>")
		self.g_ifact.create_items(ghost_menu)

		all_ghost_items = {}
		for n in range(len(ghosts)):
			all_ghost_items[ghosts[n]] = self.g_ifact.get_item(ghost_menu[n][0])
			all_ghost_items[ghosts[n]].connect("toggled", self.update_ghost, ghosts[n])

		if current in ghosts:
			all_ghost_items[current].set_active(gtk.TRUE)
			all_ghost_items[current].toggled()
		else:
			if current != self.DEFAULT_TARGET and current != name:
				all_ghost_items[self.sakuraname].set_active(gtk.TRUE)
				all_ghost_items[self.sakuraname].toggled()
			else:
				all_ghost_items[self.DEFAULT_TARGET].set_active(gtk.TRUE)
				all_ghost_items[self.DEFAULT_TARGET].toggled()

		ghost_item.set_submenu(self.g_ifact.get_widget("<main>"))
		ghost_item.set_sensitive(gtk.TRUE)

		if not self.g_e_box.handler_is_connected(self.g_e_id):
			self.g_e_id = self.g_e_box.connect("button-press-event", self.button_popup, self.g_ifact.get_widget("<main>"))

		self.check_channel_permission()

	def update_ghost(self, widget, data):
		self.target_ghost = data
		self.channel_ghost.set_text("(" + self.target_ghost + ")")

	def check_channel_permission(self):
		channel = self.get_channel_id()
		if self.client.channels[channel]["nopost"]:
			self.send_button.set_sensitive(gtk.FALSE)
		else:
			self.send_button.set_sensitive(gtk.TRUE)

	def handle_event(self):
		if not self.client.connected:
			return
		handled = 0
		try:
			handled = self.client.handle_event()
		except BottleClientError, e:
			self.client.close()
			self.disable_channel()
			return

		now = time.time()
		if now - self.local_ghost_last_notified > 30: # about 30 seconds
			self.local_ghost_last_notified = now
			self.send_ghost_names()
			self.rebuild_ghostmenu()
		self.timeout_id = gtk.timeout_add(1000, self.handle_event)

	SSTP_EXECUTE = (
		"EXECUTE SSTP/1.0\r\n"
		"Sender: GBottler\r\n"
		"Command: GetName\r\n"
		"Option: notranslate\r\n"
		"Charset: Shift_JIS\r\n"
		"\r\n")

	def send_ghost_names(self):
		try:
			data = self.client.send_sstp_message(self.SSTP_EXECUTE,
												 self.host, self.port)
		except BottleClientError, e:
			return
		if not data or data[:12] != "SSTP/1.0 200":
			return
		pos = string.find(data, "\r\n\r\n")
		if pos < 0:
			return

		resdata = data[pos:]
		try:
			resdata = unicode(resdata, "utf8").encode("sjis")
		except UnicodeError:
			pass

		name = self.sjis_to(string.strip(resdata))
		if not name or self.local_ghost == name:
			return
		try:
			self.client.send_ghost_names([name])
		except BottleClientError, e:
			return
		self.local_ghost = name
		[self.sakuraname, self.keroname] = self.local_ghost.split("\n")[0].split(",")

	def monitor_clear(self):
		self.pbar.set_text("")

	def monitor_insert(self, text):
		setted = ""
		if self.pbar.get_text():
			setted = self.pbar.get_text()
		self.pbar.set_text(setted + text)

	def disable_channel(self):
		self.menu_list["/%s/%s" % (self.msg["file"][1], self.msg["part"])].set_sensitive(gtk.FALSE)
		self.menu_list["/%s/%s" % (self.msg["file"][1], self.msg["join"])].set_sensitive(gtk.TRUE)
		self.log_window.vb.set_sensitive(gtk.FALSE)
		self.log_window.ab.set_sensitive(gtk.FALSE)
		self.log_window.vi.set_sensitive(gtk.FALSE)
		self.log_window.ai.set_sensitive(gtk.FALSE)

		if self.c_e_box.handler_is_connected(self.c_e_id):
			self.c_e_box.disconnect(self.c_e_id)
			self.c_e_id = 1
		if self.g_e_box.handler_is_connected(self.g_e_id):
			self.g_e_box.disconnect(self.g_e_id)
			self.g_e_id = 1

		self.send_button.set_sensitive(gtk.FALSE)
		self.channel.set_text("")
		self.channel_ghost.set_text("")
		self.target_ghost = self.DEFAULT_TARGET
		self.users.set_text("")

		for m in ["/%s" % self.msg["ghost"][1],
				  "/%s" % self.msg["channel"][1]]:
			menu = self.menu_list[m]
			menu.deselect()
			menu.set_sensitive(gtk.FALSE)
			menu.remove_submenu()

	def get_channel_id(self):
		name = self.channel.get_text()
		for id in self.client.in_channels:
			if self.client.channels[id]["name"] == name:
				return id
		raise RuntimeError, "unknown channel name (%s)" % name

	def show_users(self):
		channel = self.get_channel_id()
		count = self.client.channels[channel]["count"]
		self.users.set_text("%d / %d " % (count, self.client.users))

	def part(self, data, widget):
		if not self.client.connected:
			return
		if self.timeout_id:
			gtk.timeout_remove(self.timeout_id)
			self.timeout_id = None
		self.client.close()
		self.logger.close()
		self.monitor_clear()
		self.monitor_insert(unicode(_("Connection closed."), "utf8"))
		self.disable_channel()

	def quit(self, data, widget):
		self.part(data, widget)
		gtk.mainquit()

	def edit_clear(self, data, widget):
		self.monitor_clear()
		self.editor.set_text(r"\t\u\s[10]\h\s[0]\e")
		self.main_text.emit("move-cursor", gtk.MOVEMENT_LOGICAL_POSITIONS, -2, gtk.FALSE)

	def edit_clear_all(self, data, widget):
		self.monitor_clear()
		self.editor.set_text("")

	def check(self, widget, data):
		self.monitor_clear()
		self.monitor_insert(unicode(_("Now checking script..."), "utf8"))
		[si, ei] = self.editor.get_bounds()
		lines = string.split(self.editor.get_text(si, ei), "\n")

		error = 0
		buffer = []
		i = 1.0 / len(lines)
		for n in range(len(lines)):
			if not lines[n]:
				break
			lines[n] = self.check_line(lines[n], n)
			if lines[n] is None:
				error = 1
				break
			buffer.extend(lines[n])
			self.pbar.set_fraction(self.pbar.get_fraction() + i)
		if error or self.check_tag_abuse(buffer):
			return 1
		else:
			self.pbar.set_fraction(0.0)
			self.monitor_insert(unicode(_("No error."), "utf8"))
			return 0

	def check_line(self, script, lineno):
		try:
			return self.parser.parse(script)
		except ParserError, e:
			self.pbar.set_fraction(0.0)
			self.monitor_insert(unicode(_("Error"), "utf8"))

			d = gtk.Dialog(title="Script Parser Error", parent=self.window, flags=1)
			d.set_has_separator(gtk.FALSE)
			d.connect("destroy", lambda w,data:d.destroy(), "WM destroy")
			d.vbox.set_border_width(10)

			h = gtk.HBox()
			i = gtk.Image()
			i.set_from_stock(gtk.STOCK_DIALOG_ERROR, gtk.ICON_SIZE_LARGE_TOOLBAR)
			i.show()
			h.pack_start(i, gtk.FALSE, gtk.TRUE, 0)
			l = gtk.Label("Script Parser Error")
			l.show()
			h.pack_start(l, gtk.FALSE, gtk.TRUE, 0)
			h.show()
			d.vbox.add(h)
			s = gtk.HSeparator()
			s.show()
			d.vbox.add(s)

			s = gtk.ScrolledWindow()
			s.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
			t = gtk.TextView()
			t.set_cursor_visible(gtk.TRUE)
			t.set_size_request(300, 100)
			b = t.get_buffer()
			b.set_text('')
			tag = b.create_tag()
			tag.set_property('font', 'Monospace')

			b.insert_with_tags(b.get_end_iter(), unicode(_("Line"), "utf8") + \
							   " %d: %s\n%s\n" % (lineno, e.message, script) +
							   " " * e.column + "^" * (e.length or 1), tag)

			t.set_editable(gtk.FALSE)
			t.set_wrap_mode(gtk.WRAP_CHAR)
			s.add(t)
			d.vbox.add(s)

			ok = gtk.Button(_("Close"), gtk.STOCK_CLOSE)
			ok.connect("clicked", lambda w,data:d.destroy(), None)
			ok.show()
			d.action_area.pack_start(ok, gtk.TRUE, gtk.TRUE, 0)

			t.show()
			s.show()
			d.show()

			return None

	RESERVED_SURFACES = (0, 1, 2, 3, 4, 5, 6, 7, 8, 10, 11, 19, 20, 25)

	def check_tag_abuse(self, script):
		scope = error = None
		surface_usage = [0, 10]

		msg_or  = unicode(_("or"), "utf8")
		msg_tag = unicode(_("tag"), "utf8")
		for node in script:
			if node[0] == SCRIPT_TAG:
				name, args = node[1], node[2:]
				if name == r"\h":
					scope = 0
				elif name == r"\u":
					scope = 1
				elif name == r"\s":
					try:
						id = int(args[0])
					except ValueError:
						error = unicode(_('bad use of surfaces'), 'utf-8')
						break
					if id in self.RESERVED_SURFACES:
						target = id / 10 % 2
					else:
						target = None
					if scope is None:
						if target is None:
							tag = r"\h %s \u %s" % (msg_or, msg_tag)
						elif target == 0:
							tag = r"\h %s" % msg_tag
						else:
							tag = r"\u %s" % msg_tag
						error = r"\s[%d] " % id + \
								unicode(_("is used without a preceding"), "utf8") + \
								" %s" % tag
						break
					if id in self.RESERVED_SURFACES:
						surface_usage[scope] = id
			elif node[0] == SCRIPT_TEXT:
				if scope is None:
					error = r"\h %s \u " % msg_or + \
							unicode(_("is needed before characters"), "utf8")
					break
		else:
			if surface_usage[0] / 10 % 2 == 1 or surface_usage[0] == 20 or \
			   surface_usage[1] / 10 % 2 == 0:
				error = unicode(_("bad use of surfaces"), "utf8")
		if error is not None:
			self.pbar.set_fraction(0.0)
			self.monitor_clear()
			self.monitor_insert(unicode(_("Error"), "utf8") + ": %s " % error)
			return 1
		return 0

	def test(self, widget, data):
		if self.check(widget, data):
			return
		self.send_local_message(self.get_script())

	def send(self, widget, data):
		if self.check(widget, data):
			return

		d = gtk.Dialog(unicode(_("Really?"), "utf-8"), parent=self.window,
					   flags=gtk.DIALOG_NO_SEPARATOR|gtk.DIALOG_MODAL)
		l = gtk.Label(unicode(_("Send to "), "utf8") + \
					  self.channel.get_text() + " / " + \
					  self.target_ghost + unicode(_(", OK?"), "utf8"))

		d.vbox.pack_start(l, gtk.TRUE, gtk.TRUE, 0)
		l.show()
		d.add_button(unicode(_("OK"), "utf8"), True)
		d.add_button(unicode(_("Cancel"), "utf8"), False)
		d.show()
		response = d.run()
		d.hide()
		d.destroy()

		if not response:
			return

		channel = self.get_channel_id()
		script = self.get_script()
		ghost = self.target_ghost
		if ghost == self.DEFAULT_TARGET:
			ghost = None
		self.monitor_clear()
		self.monitor_insert(unicode(_("Broadcasting..."), "utf8"))
		try:
			if self.client.send_broadcast(channel, script, ghost):
				error = self.client.headers["ExtraMessage"]
			else:
				error = None
		except BottleClientError, e:
			error = str(e)

		if error is not None:
			self.monitor_insert(unicode(_("failed!!")) + "\n" + \
								unicode(_("Error"), "utf8") + ": ")
			self.monitor_insert(unicode(error))
			self.window.bell()

		if self.menu_list["/%s/%s/%s" % (self.msg["edit"][1], self.msg["pref"][1], self.msg["ceas"])].get_active():
			self.edit_clear(data, widget)

		self.pbar.set_fraction(0.0)

	def get_script(self):
		[si, ei] = self.editor.get_bounds()
		script = string.replace(self.editor.get_text(si, ei), "\n", "")

		if string.rstrip(script)[-2:] != r"\e":
			script = script + r"\e"
		return unicode(script, "utf-8")

	def about(self, data, widget):
		if self.about:
			return

		def aboutdestroy(widget, data):
			self.about.destroy()
			self.about = None

		self.about = gtk.Dialog(unicode(_("About"), "utf-8"), parent=self.window,
								flags=gtk.DIALOG_NO_SEPARATOR)
		self.about.connect("destroy", aboutdestroy, "WM destroy")
		self.about.set_border_width(10)

		h = gtk.HBox()
		i = gtk.Image()
		i.set_from_stock(gtk.STOCK_DIALOG_INFO, gtk.ICON_SIZE_LARGE_TOOLBAR)
		i.show()
		h.pack_start(i, gtk.FALSE, gtk.TRUE, 0)
		l = gtk.Label(unicode(_("About"), 'utf-8'))
		l.show()
		h.pack_start(l, gtk.FALSE, gtk.TRUE, 0)
		h.show()
		self.about.vbox.add(h)
		s = gtk.HSeparator()
		s.show()
		self.about.vbox.add(s)

		frame = gtk.Frame(label=" %s %s " % (APP, VER))
		l = gtk.Label("\n"+
					  " Copyright(C) 2001-'04, Tamito KAJIYAMA \n"+
					  " Copyright(C) 2003-'04, Atzm WATANABE \n")
		l.set_justify(gtk.JUSTIFY_RIGHT)
		l.show()
		frame.add(l)
		frame.show()
		self.about.vbox.add(frame)

		frame = gtk.Frame(label=unicode(_(" License "), "utf-8"))
		l = gtk.Label("\n"+
					  " GNU General Public License version 2 or later \n")
		l.set_justify(gtk.JUSTIFY_RIGHT)
		l.show()
		frame.add(l)
		frame.show()
		self.about.vbox.add(frame)

		b = gtk.Button(unicode(_("OK"), "utf8"), gtk.STOCK_OK)
		b.connect("clicked", aboutdestroy, None)
		b.show()
		self.about.action_area.pack_start(b, gtk.TRUE, gtk.TRUE, 0)

		self.about.show()

	def log_message(self, mid, channel, ifghost, script):
		self.logger.log_message(mid, channel, ifghost, script)

	def log_votes(self, mid, type, num):
		self.logger.log_votes(mid, type, num)

	SSTP_SEND = (
		"SEND SSTP/1.1\r\n"
		"Sender: GBottler\r\n"
		"Script: %s\r\n"
		"Option: notranslate\r\n"
		"Charset: Shift_JIS\r\n"
		"\r\n")
	def send_local_message(self, script):
		self.client.send_sstp_message(self.SSTP_SEND % script,
									  self.host, self.port)

	def vote_message(self, mid, type):
		if self.client.vote_message(mid, type):
			print "voting error"
			print self.client.headers["ExtraMessage"]
		self.monitor_clear()
		self.pbar.set_fraction(0.0)

	def insert(self, pre, post=""):
		self.main_text.emit("insert-at-cursor", pre)
		if post:
			self.main_text.emit("insert-at-cursor", post)
			self.main_text.emit("move-cursor", gtk.MOVEMENT_LOGICAL_POSITIONS,
								-1 * len(post), gtk.FALSE)

	def insert_h(self, data, widget): self.insert("\n\\h")
	def insert_u(self, data, widget): self.insert("\n\\u")
	def insert_s0(self, data, widget): self.insert(r"\s[0]")
	def insert_s1(self, data, widget): self.insert(r"\s[1]")
	def insert_s2(self, data, widget): self.insert(r"\s[2]")
	def insert_s3(self, data, widget): self.insert(r"\s[3]")
	def insert_s4(self, data, widget): self.insert(r"\s[4]")
	def insert_s5(self, data, widget): self.insert(r"\s[5]")
	def insert_s6(self, data, widget): self.insert(r"\s[6]")
	def insert_s7(self, data, widget): self.insert(r"\s[7]")
	def insert_s8(self, data, widget): self.insert(r"\s[8]")
	def insert_s20(self, data, widget): self.insert(r"\s[20]")
	def insert_s25(self, data, widget): self.insert(r"\s[25]")
	def insert_s10(self, data, widget): self.insert(r"\s[10]")
	def insert_s11(self, data, widget): self.insert(r"\s[11]")
	def insert_s19(self, data, widget): self.insert(r"\s[19]")
	def insert_username(self, data, widget): self.insert("%username")
	def insert_selfname(self, data, widget): self.insert("%selfname")
	def insert_selfname2(self, data, widget): self.insert("%selfname2")
	def insert_keroname(self, data, widget): self.insert("%keroname")
	def insert_url(self, data, widget): self.insert(r"\URL[http://", "]")
	def insert_c(self, data, widget): self.insert(r"\c")
	def insert_n(self, data, widget): self.insert("\\n\n")
	def insert_w2(self, data, widget): self.insert(r"\w2")
	def insert_w4(self, data, widget): self.insert(r"\w4")
	def insert_w6(self, data, widget): self.insert(r"\w6")
	def insert_w8(self, data, widget): self.insert(r"\w8")
	def insert_w10(self, data, widget): self.insert(r"\w8\w2")
	def insert_w12(self, data, widget): self.insert(r"\w8\w4")
	def insert_w18(self, data, widget): self.insert(r"\w9\w9")
	def insert_w20(self, data, widget): self.insert(r"\w8\w8\w4")
	def insert_w24(self, data, widget): self.insert(r"\w8\w8\w8")
	def insert_w30(self, data, widget): self.insert(r"\w9\w9\w9\w3")
	def insert_w40(self, data, widget): self.insert(r"\w9\w9\w9\w9\w4")
	def insert_w50(self, data, widget): self.insert(r"\w9\w9\w9\w9\w9\w5")
	def insert_w60(self, data, widget): self.insert(r"\w9\w9\w9\w9\w9\w9\w6")
	def insert_q0(self, data, widget): self.insert(r"\q0[#][", "]")
	def insert_q1(self, data, widget): self.insert(r"\q1[#][", "]")
	def insert_q2(self, data, widget): self.insert(r"\q2[#][", "]")
	def insert_q3(self, data, widget): self.insert(r"\q3[#][", "]")
	def insert_q4(self, data, widget): self.insert(r"\q4[#][", "]")
	def insert_q5(self, data, widget): self.insert(r"\q5[#][", "]")
	def insert_q6(self, data, widget): self.insert(r"\q6[#][", "]")
	def insert_q7(self, data, widget): self.insert(r"\q7[#][", "]")
	def insert_q8(self, data, widget): self.insert(r"\q8[#][", "]")
	def insert_q9(self, data, widget): self.insert(r"\q9[#][", "]")
	def insert_z(self, data, widget): self.insert(r"\z")
	def insert_y(self, data, widget): self.insert(r"\y")
	def insert_t(self, data, widget): self.insert(r"\t")
	def insert_q(self, data, widget): self.insert(r"\_q")
	def insert_s(self, data, widget): self.insert(r"\_s")
	def insert_e(self, data, widget): self.insert(r"\e")

class ChannelDialog:
	def __init__(self, app, master, channels):
		self.selection = []
		self.channels  = channels

		self.window = gtk.Dialog(parent=master, flags=1)
		self.window.set_has_separator(gtk.FALSE)
		self.window.connect("destroy", self.cancel, "WM destroy")
		self.window.set_title(unicode(_("Select Channels"), "utf-8"))

		self.window.vbox.set_border_width(10)

		cbs = []
		for n in range(len(self.channels)):
			cb = gtk.CheckButton(self.channels[n]["name"])
			cbs.append(cb)
			cb.connect("toggled", self.selected, [self.channels[n], n])
			self.window.vbox.pack_start(cb, gtk.TRUE, gtk.TRUE, 2)
			cb.show()

		self.window.vbox.show()

		self.buffer   = gtk.TextBuffer()
		self.textview = gtk.TextView(self.buffer)
		self.textview.set_editable(gtk.FALSE)
		self.textview.set_justification(gtk.JUSTIFY_LEFT)
		self.window.vbox.pack_start(self.textview, gtk.TRUE, gtk.TRUE, 2)
		self.textview.show()

		bbox = gtk.HBox(gtk.FALSE, 1)
		bbox.set_border_width(2)
		ok = gtk.Button(unicode(_("OK"), "utf8"), gtk.STOCK_OK)
		ca = gtk.Button(unicode(_("Cancel"), "utf8"), gtk.STOCK_CANCEL)
		ok.connect("clicked", self.ok, None)
		ca.connect("clicked", self.cancel, None)
		bbox.pack_start(ok, gtk.TRUE, gtk.TRUE, 2)
		ok.show()
		bbox.pack_start(ca, gtk.TRUE, gtk.TRUE, 2)
		ca.show()
		self.window.vbox.pack_start(bbox, gtk.TRUE, gtk.TRUE, 2)
		bbox.show()

		self.window.show()

		cbs.reverse()
		for b in cbs:
			b.set_active(gtk.TRUE)

		ok.grab_focus()

	def selected(self, caller, data):
		n = data[1]
		data = data[0]
		buf = "info: " +  data["info"] + "\n" + "ghost: " + data["ghost"] + \
			  "\n" + "count: " + str(data["count"])
		self.buffer.set_text(buf)

		if n in self.selection:
			self.selection.remove(n)
		else:
			self.selection.append(n)
			self.selection.sort()

	def go(self):
		gtk.main()
		return self.selection

	def ok(self, widget, data):
		gtk.mainquit()
		self.window.hide()

	def cancel(self, widget, data):
		self.selection = None
		gtk.mainquit()
		self.window.hide()

def open_bottlecase(path=None):
	libdir = os.path.expanduser(path or LIBDIR)
	if not os.path.exists(libdir):
		os.mkdir(libdir)
	return libdir


class LogManager:
	DBFILE = "log_db"
	DBWRITEFREQ = 10 # write database to file per 10 votes
	# channel
	ALL_CHANNELS = "-"
	ALL_GHOSTS   = "-"
	# ranges
	RANGE_TODAY = 1
	RANGE_YESTERDAY = 2
	RANGE_LAST3DAYS = 3
	RANGE_LAST7DAYS = 4
	RANGE_SPECIFIED = 5
	RANGE_ALL = 6
	RANGE_LAST50BOTTLES = 7
	# orders
	ORDER_BY_NEWNESS  = 100
	ORDER_BY_DATETIME = 101
	ORDER_BY_VOTES	= 102

	def __init__(self):
		self.logdir = open_bottlecase()
		# lock database
		self.lockfile = open(os.path.join(self.logdir, ".lock"), "w")
		try:
			fcntl.flock(self.lockfile.fileno(), fcntl.LOCK_EX|fcntl.LOCK_NB)
		except IOError:
			self.disabled = 1
			return
		self.disabled = 0
		# load database
		self.dbpath = os.path.join(self.logdir, self.DBFILE)
		self.dblife = self.DBWRITEFREQ
		self.load_database()
		# defaults
		self.channel = None # all channels
		self.ghost   = None
		self.range   = self.RANGE_TODAY
		self.order   = self.ORDER_BY_NEWNESS

	def is_disabled(self):
		return self.disabled

	def load_database(self):
		self.aliases = {}
		self.votes = {}
		try:
			file = open(self.dbpath)
		except IOError:
			pass
		else:
			nchannels = int(file.readline())
			for n in range(nchannels):
				channel, alias = string.split(file.readline())
				channel = unicode(channel, "utf-8")
				self.aliases[channel] = alias
			while 1:
				line = file.readline()
				if not line:
					break
				record = string.split(line)
				try:
					dict = self.votes[record[0]]
				except KeyError:
					dict = self.votes[record[0]] = {}
				dict[VOTE] = int(record[1])
				if len(record) == 2:
					dict[AGREE] = 0
				else:
					dict[AGREE] = int(record[2])
		self.channels = self.aliases.keys()
		self.channels.sort()

	def save_database(self):
		try:
			file = open(self.dbpath, "w")
		except IOError, (errno, message):
			sys.stderr.write("Error: cannot write %s\n" % self.dbpath)
		else:
			file.write("%d\n" % len(self.channels))
			for channel, alias in self.aliases.items():
				file.write("%s %s\n" % (channel.encode("utf-8"), alias))
			for mid, dict in self.votes.items():
				v = dict.get(VOTE,  0)
				a = dict.get(AGREE, 0)
				file.write("%s %d %d\n" % (mid, v, a))
			file.close()

	def close(self):
		if self.disabled:
			return
		self.save_database()
		fcntl.flock(self.lockfile.fileno(), fcntl.LOCK_UN)

	def log_message(self, mid, channel, ifghost, script):
		if self.disabled:
			return
		# register an alias to channel if necessary

		if channel not in self.channels:
			self.channels.append(channel)
			self.channels.sort()
			self.aliases[channel] = "ch-%d" % len(self.channels)
		# write message to log file
		now = time.localtime(time.time())
		datetime = time.strftime("%Y%m%d%H%M%S", now)
		filename = "%s-%s.log" % (datetime[:8], self.aliases[channel])
		file = open(os.path.join(self.logdir, filename), "a")
		file.write("%s\t%s\t%s\t%s\t%s\n" % (datetime, mid.encode("utf-8"),
											 channel.encode("utf-8"),
											 ifghost.encode("utf-8"),
											 script.encode("utf-8")))
		file.close()

	def log_votes(self, mid, type, num):
		if self.disabled:
			return
		try:
			dict = self.votes[mid]
		except KeyError:
			dict = self.votes[mid] = {}
		dict[type] = num
		self.dblife = self.dblife - 1
		if self.dblife == 0:
			self.dblife = self.DBWRITEFREQ
			self.save_database()

	def read_log(self, path): # internal
		buffer = []
		file = open(path)
		while 1:
			line = unicode(file.readline(), "utf-8")
			if not line:
				break
			datetime, mid, channel, ifghost, script = string.split(line, "\t", 4)
			assert(len(datetime) == 14)
			dict = self.votes.get(mid, {})
			v = dict.get(VOTE,  0)
			a = dict.get(AGREE, 0)
			buffer.append((datetime, mid, v, a,
						   channel, ifghost, string.rstrip(script)))
		return buffer

	def reload(self):
		if self.disabled:
			return 1
		# check range
		now = time.localtime(time.time())
		e = time.mktime(now[0:3] + (0, 0, 0, 0, 0, 0)) + 86400
		s = None
		if self.range == self.RANGE_TODAY:
			s = e - 86400
		elif self.range == self.RANGE_YESTERDAY:
			s = e - 86400 * 2
			e = e - 86400 * 2
		elif self.range == self.RANGE_LAST3DAYS:
			s = e - 86400 * 3
		elif self.range == self.RANGE_LAST7DAYS:
			s = e - 86400 * 7
		elif self.range == self.RANGE_SPECIFIED:
			s = self.range_start
			e = self.range_end
		elif self.range in [self.RANGE_ALL, self.RANGE_LAST50BOTTLES]:
			s = 0
		range_start = time.strftime("%Y%m%d", time.localtime(s))
		range_end   = time.strftime("%Y%m%d", time.localtime(e))
		# read log files
		self.log = []
		self.selected_channels = []
		channels = {} # alias to channel
		for key, value in self.aliases.items():
			channels[value] = key

		list = os.listdir(self.logdir)
		list.sort()
		list.reverse()
		dateline = int(range_end) - 1
		for filename in list:
			# filename should be in the form of YYYYMMDD-CHANNEL.log
			if filename[-4:] == ".log":

				# Hmm... fixme
				if self.range == self.RANGE_LAST50BOTTLES and \
					   not int(filename[:8]) == dateline and \
					   len(self.log) >= 50:
					break

				if range_start <= filename[:8] <= range_end:
					alias = filename[9:-4]
					channel = channels.get(alias)
					if channel and channel not in self.selected_channels:
						self.selected_channels.append(channel)
					if self.channel == self.ALL_CHANNELS or \
						   self.aliases[self.channel] == alias:
						path = os.path.join(self.logdir, filename)
						self.log.extend(self.read_log(path))

				dateline = int(filename[:8])

		self.selected_channels.sort()
		# sort logs
		if self.order == self.ORDER_BY_NEWNESS:
			self.log.sort(lambda x, y: cmp(y[0], x[0]))
		elif self.order == self.ORDER_BY_DATETIME:
			self.log.sort(lambda x, y: cmp(x[0], y[0]))
		elif self.order == self.ORDER_BY_VOTES:
			self.log.sort(lambda x, y: cmp((y[2], y[0]), (x[2], x[0])))
		# format logs
		self.formatted_log = []
		if not self.range == self.RANGE_LAST50BOTTLES:
			for datetime, mid, votes, agreements, channel, ifghost, script in self.log:
				self.formatted_log.append(["%s/%s/%s %s:%s:%s" % \
										   (datetime[0:4], datetime[4:6], datetime[6:8],
											datetime[8:10], datetime[10:12], datetime[12:14]),
										   ifghost, channel, "%2d" % votes, "%2d" % agreements,
										   script, mid])
		else:
			for i in range(len(self.log)):
				[datetime, mid, votes, agreements, channel, ifghost, script] = self.log[i]
				self.formatted_log.append(["%s/%s/%s %s:%s:%s" % \
										   (datetime[0:4], datetime[4:6], datetime[6:8],
											datetime[8:10], datetime[10:12], datetime[12:14]),
										   ifghost, channel, "%2d" % votes, "%2d" % agreements,
										   script, mid])
				if i >= 50:
					break

		return 0

	def set_channel(self, channel):
		self.channel = channel

	def get_channel(self):
		return self.channel

	def set_ghost(self, ghost):
		self.ghost = ghost

	def get_ghost(self):
		return self.ghost

	def set_range(self, range, range_start=None, range_end=None):
		self.range = range
		if range == self.RANGE_SPECIFIED:
			self.range_start = range_start
			self.range_end   = range_end

	def get_range(self):
		return self.range

	def set_order(self, order):
		self.order = order

	def get_order(self):
		return self.order

	def get_formatted_logs(self):
		if self.disabled:
			return None
		return self.formatted_log

	def get_selected_channels(self):
		return self.selected_channels

	def get_mid(self, index):
		if self.disabled or index >= len(self.log):
			return None
		return self.log[index][1]

	def get_script(self, index):
		if self.disabled or index >= len(self.log):
			return None
		return self.log[index][6]

class LogWindow:
	STYLE_TALK = 1
	STYLE_SCRIPT = 2
	STYLE_SCRIPT_WITH_LINEFEED = 3

	def __init__(self, app, logger):
		self.app = app
		self.logger = logger
		self.parser = Parser('loose')

		self.labels = {
			LogManager.ALL_CHANNELS: unicode(_("All"), "utf-8"),
			LogManager.RANGE_TODAY: unicode(_("Today"), "utf-8"),
			LogManager.RANGE_YESTERDAY: unicode(_("Yesterday"), "utf-8"),
			LogManager.RANGE_LAST3DAYS: unicode(_("Last 3 days"), "utf-8"),
			LogManager.RANGE_LAST7DAYS: unicode(_("Last a week"), "utf-8"),
			LogManager.RANGE_SPECIFIED: unicode(_("Specified"), "utf-8"),
			LogManager.RANGE_ALL: unicode(_("All"), "utf-8"),
			LogManager.RANGE_LAST50BOTTLES: unicode(_("Last 50 bottles"), "utf-8"),
			LogManager.ORDER_BY_NEWNESS: unicode(_("By newness"), "utf-8"),
			LogManager.ORDER_BY_DATETIME: unicode(_("By datetime"), "utf-8"),
			LogManager.ORDER_BY_VOTES: unicode(_("By votes"), "utf-8"),
			}

		self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
		self.window.set_size_request(450,300)

		# create GUI
		self.msg = {
			"file":   [unicode(_("File(_F)"), "utf-8"),
					   unicode(_("File(F)"), "utf-8")],
			"close":  unicode(_("Close"), "utf-8"),

			"edit":   [unicode(_("Edit(_E)"), "utf-8"),
					   unicode(_("Edit(E)"), "utf-8")],
			"copy to main editor":   unicode(_("Copy to main editor"), "utf-8"),

			"view":        [unicode(_("View(_V)"), "utf-8"),
							unicode(_("View(V)"), "utf-8")],
			"update":      unicode(_("Update"), "utf-8"),
			"auto update": unicode(_("Auto update"), "utf-8"),
			"range":       [unicode(_("Range(_R)"), "utf-8"),
							unicode(_("Range(R)"), "utf-8")],
			"style":       [unicode(_("Style(_S)"), "utf-8"),
							unicode(_("Style(S)"), "utf-8")],
			"talk style":  unicode(_("Talk style"), "utf-8"),
			"script style":unicode(_("Script style"), "utf-8"),
			"linefeed":    unicode(_("with linefeed"), "utf-8"),

			"help":   [unicode(_("Help(_H)"), "utf-8"),
					   unicode(_("Help(H)"), "utf-8")],
			"about":  unicode(_("About"), "utf-8"),
			}

		menu_items = (
			( "/%s" % self.msg["file"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.msg["file"][1], self.msg["close"]), "<control>q", self.close, 0, None ),

			( "/%s" % self.msg["edit"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.msg["edit"][1], self.msg["copy to main editor"]), "<control>c", self.edit_copy, 0, None ),

			( "/%s" % self.msg["view"][0], None, None, 0, "<Branch>" ),
			( "/%s/%s" % (self.msg["view"][1], self.msg["update"]), "<control>u", self.update, 0, None ),
			( "/%s/%s" % (self.msg["view"][1], self.msg["auto update"]), None, None, 0, "<CheckItem>" ),
			( "/%s/sep1" % self.msg["view"][1], None, None, 0, "<Separator>" ),

			( "/%s/%s" % (self.msg["view"][1], self.msg["range"][0]), None, None, 0, "<Branch>" ),
			( "/%s/%s/" % (self.msg["view"][1], self.msg["range"][1]) + self.labels[LogManager.RANGE_LAST50BOTTLES],
			  None, self.initialize, LogManager.RANGE_LAST50BOTTLES, "<RadioItem>" ),
			( "/%s/%s/" % (self.msg["view"][1], self.msg["range"][1]) + self.labels[LogManager.RANGE_TODAY],
			  None, self.initialize, LogManager.RANGE_TODAY,
			  "/%s/%s/" % (self.msg["view"][1], self.msg["range"][1]) + self.labels[LogManager.RANGE_LAST50BOTTLES] ),
			( "/%s/%s/" % (self.msg["view"][1], self.msg["range"][1]) + self.labels[LogManager.RANGE_YESTERDAY],
			  None, self.initialize, LogManager.RANGE_YESTERDAY,
			  "/%s/%s/" % (self.msg["view"][1], self.msg["range"][1]) + self.labels[LogManager.RANGE_TODAY]),
			( "/%s/%s/" % (self.msg["view"][1], self.msg["range"][1]) + self.labels[LogManager.RANGE_LAST3DAYS],
			  None, self.initialize, LogManager.RANGE_LAST3DAYS,
			  "/%s/%s/" % (self.msg["view"][1], self.msg["range"][1]) + self.labels[LogManager.RANGE_YESTERDAY]),
			( "/%s/%s/" % (self.msg["view"][1], self.msg["range"][1]) + self.labels[LogManager.RANGE_LAST7DAYS],
			  None, self.initialize, LogManager.RANGE_LAST7DAYS,
			  "/%s/%s/" % (self.msg["view"][1], self.msg["range"][1]) + self.labels[LogManager.RANGE_LAST3DAYS]),
			( "/%s/%s/" % (self.msg["view"][1], self.msg["range"][1]) + self.labels[LogManager.RANGE_ALL],
			  None, self.initialize, LogManager.RANGE_ALL,
			  "/%s/%s/" % (self.msg["view"][1], self.msg["range"][1]) + self.labels[LogManager.RANGE_LAST7DAYS]),

			( "/%s/%s" % (self.msg["view"][1], self.msg["style"][0]), None, None, 0, "<Branch>" ),
			( "/%s/%s/%s" % (self.msg["view"][1], self.msg["style"][1],
							 self.msg["talk style"]), None, self.set_style, self.STYLE_TALK, "<RadioItem>" ),
			( "/%s/%s/%s" % (self.msg["view"][1], self.msg["style"][1], self.msg["script style"]),
			  None, self.set_style, self.STYLE_SCRIPT, "/%s/%s/%s" % (self.msg["view"][1],
																	  self.msg["style"][1],
																	  self.msg["talk style"])),
			( "/%s/%s/%s(%s)" % (self.msg["view"][1], self.msg["style"][1], self.msg["script style"], self.msg["linefeed"]),
			  None, self.set_style, self.STYLE_SCRIPT_WITH_LINEFEED, "/%s/%s/%s" % (self.msg["view"][1],
																					self.msg["style"][1],
																					self.msg["script style"])),

			( "/%s" % self.msg["help"][0], None, None, 0, "<LastBranch>" ),
			( "/%s/%s" % (self.msg["help"][1], self.msg["about"]), None, self.app.about, 0, None ),
			)

		self.window.connect("delete_event", self.close)
		self.window.set_title(unicode(_("GBottler"), "utf-8") + ":" + \
							  unicode(_("Logs"), "utf-8"))

		self.main_vbox = gtk.VBox(gtk.FALSE, 0)
		self.window.add(self.main_vbox)
		self.main_vbox.show()

		accel_group = gtk.AccelGroup()

		self.item_factory = gtk.ItemFactory(gtk.MenuBar, "<main>", accel_group)
		self.item_factory.create_items(menu_items)

		self.window.add_accel_group(accel_group)

		self.menu_list = {}
		for m in menu_items:
			key = re.sub("_", "", m[0])
			self.menu_list[key] = self.item_factory.get_item(key)

		self.menubar = self.item_factory.get_widget("<main>")
		self.main_vbox.pack_start(self.menubar, gtk.FALSE, gtk.TRUE, 0)
		self.menubar.show()
		self.item_factory.get_widget("/%s/%s" % (self.msg["view"][1], self.msg["auto update"])).set_active(gtk.TRUE)
		# End of creating Menubar

		self.hbox = gtk.HBox()
		pb_h = self.app.get_imglabelbox_with_stock(gtk.STOCK_REFRESH,
												   gtk.ICON_SIZE_BUTTON,
												   unicode(_("Play"), "utf-8"))
		self.pb = gtk.Button()
		self.pb.add(pb_h)
		self.pb.connect("clicked", self.play, None)
		pb_h.show()
		self.pb.show()
		self.hbox.pack_start(self.pb, gtk.FALSE, gtk.TRUE, 0)

		vb_h = self.app.get_imglabelbox_with_stock(gtk.STOCK_YES,
												   gtk.ICON_SIZE_BUTTON,
												   unicode(_("Vote"), "utf-8"))
		self.vb = gtk.Button()
		self.vb.add(vb_h)
		self.vb.connect("clicked", self.vote, None)
		vb_h.show()
		self.vb.set_sensitive(gtk.FALSE)
		self.vb.show()
		self.hbox.pack_start(self.vb, gtk.FALSE, gtk.TRUE, 0)

		ab_h = self.app.get_imglabelbox_with_stock(gtk.STOCK_NO,
												   gtk.ICON_SIZE_BUTTON,
												   unicode(_("Agree"), "utf-8"))
		self.ab = gtk.Button()
		self.ab.add(ab_h)
		self.ab.connect("clicked", self.agree, None)
		ab_h.show()
		self.ab.set_sensitive(gtk.FALSE)
		self.ab.show()
		self.hbox.pack_start(self.ab, gtk.FALSE, gtk.TRUE, 0)

		self.l = gtk.Label(" " + unicode(_("Range"), "utf-8") + ": ")
		self.range_l = gtk.Label(unicode(_("Last 50 bottles"), "utf-8"))
		self.r_e_box = gtk.EventBox()
		self.r_e_box.set_events(gtk.gdk.BUTTON_PRESS_MASK)
		self.r_e_box.add(self.range_l)
		self.hbox.pack_start(self.l, gtk.FALSE, gtk.TRUE, 0)
		self.hbox.pack_start(self.r_e_box, gtk.FALSE, gtk.TRUE, 0)
		self.l.show()
		self.range_l.show()
		self.r_e_box.show()
		self.r_e_box.connect("button-press-event", self.button_popup,
							 self.item_factory.get_widget("/%s/%s" % (self.msg["view"][1],
																	  self.msg["range"][1])))

		self.m = gtk.Label("    " + unicode(_("Style"), "utf-8") + ": ")
		self.style_l = gtk.Label(self.msg['talk style'])
		self.s_e_box = gtk.EventBox()
		self.s_e_box.set_events(gtk.gdk.BUTTON_PRESS_MASK)
		self.s_e_box.add(self.style_l)
		self.hbox.pack_start(self.m, gtk.FALSE, gtk.TRUE, 0)
		self.hbox.pack_start(self.s_e_box, gtk.FALSE, gtk.TRUE, 0)
		self.m.show()
		self.style_l.show()
		self.s_e_box.show()
		self.s_e_box.connect("button-press-event", self.button_popup,
							 self.item_factory.get_widget("/%s/%s" % (self.msg["view"][1],
																	  self.msg["style"][1])))

		self.main_vbox.pack_start(self.hbox, gtk.FALSE, gtk.TRUE, 0)
		self.hbox.show()

		self.sw1 = gtk.ScrolledWindow()
		self.sw1.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		self.cols = [
			[unicode(_("Datetime"), "utf-8"), gtk.SORT_DESCENDING],
			[unicode(_("Ghost"), "utf-8"), gtk.SORT_DESCENDING],
			[unicode(_("Channel"), "utf-8"), gtk.SORT_DESCENDING],
			[unicode(_("Votes"), "utf-8"), gtk.SORT_DESCENDING],
			[unicode(_("Agrees"), "utf-8"), gtk.SORT_DESCENDING],
			[unicode(_("Script"), "utf-8"), gtk.SORT_DESCENDING],
			]
		self.entries = gtk.CList(len(self.cols), [ self.cols[i][0] for i in range(len(self.cols)) ])
		self.entries.set_selection_mode(gtk.SELECTION_BROWSE)
		self.entries.set_reorderable(gtk.TRUE)
		self.entries.set_size_request(450, 150)
		self.entries.connect("select-row", self.select)

		self.entries.set_column_width(0, 100)
		self.entries.set_column_auto_resize(1, gtk.TRUE)
		self.sw1.add(self.entries)
		self.sw1.show()
		self.entries.show()

		for i in range(len(self.cols)):
			self.entries.get_column_widget(i).get_parent().connect("clicked", self.sort, i)

		p_menu = (
			( "/sep1", None, None, 0, "<Separator>" ),
			( "/%s" % unicode(_("Copy to main editor")), "<control>c", self.edit_copy, 0, None ),
			( "/%s" % unicode(_("Play")), "<control>p", self.play, 0, None ),
			( "/%s" % unicode(_("Vote")), "<control><alt>v", self.vote, 0, None ),
			( "/%s" % unicode(_("Agree")), "<control><alt>a", self.agree, 0, None ),
			)
		self.p_ifact = gtk.ItemFactory(gtk.Menu, "<main>", accel_group)
		self.p_ifact.create_items(p_menu)
		self.popup_menu = self.p_ifact.get_widget("<main>")
		self.entries.connect("button-press-event", self.popup)
		self.vi = self.p_ifact.get_item("/%s" % unicode(_("Vote"), "utf-8"))
		self.ai = self.p_ifact.get_item("/%s" % unicode(_("Agree"), "utf-8"))

		self.sw2 = gtk.ScrolledWindow()
		self.sw2.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		self.sw2.show()

		self.main_text = gtk.TextView()
		self.main_text.set_cursor_visible(gtk.TRUE)
		self.main_text.set_editable(gtk.FALSE)
		self.main_text.set_wrap_mode(gtk.WRAP_CHAR)
		self.main_text.set_border_width(2)
		self.main_text.show()
		self.sw2.add(self.main_text)

		self.pane = gtk.VPaned()
		self.pane.pack1(self.sw1)
		self.pane.pack2(self.sw2)
		self.pane.show()
		self.main_vbox.pack_start(self.pane)

		self.monitor = self.main_text.get_buffer()
		self.sakuratag = self.monitor.create_tag('sakura')
		self.sakuratag.set_property('foreground', 'black')
		self.kerotag = self.monitor.create_tag('kero')
		self.kerotag.set_property('foreground', 'brown')
		self.synctag = self.monitor.create_tag('sync')
		self.synctag.set_property('foreground', 'blue')
		self.scripttag = self.monitor.create_tag('script')
		self.scripttag.set_property('foreground', 'dark green')

		# initialize
		self.view_style = self.STYLE_TALK
		self.current_sort = 0
		self.initialize(None, self.window)

	def set_style(self, data, widget):
		if data == self.STYLE_TALK:
			self.style_l.set_text(self.msg['talk style'])
		elif data == self.STYLE_SCRIPT:
			self.style_l.set_text(self.msg['script style'])
		elif data == self.STYLE_SCRIPT_WITH_LINEFEED:
			self.style_l.set_text("%s(%s)" % (self.msg['script style'], self.msg['linefeed']))

		self.view_style = data
		self.select(None, self.entries.selection[0], None, None)

	def select(self, clist, row, col, event):
		current = 'sakura'
		self.monitor.set_text("")
		for chunk in self.parser.parse(unicode(self.entries.get_text(row, 5), "utf-8")):
			if chunk[0] == TEXT_STRING:
				if chunk[1][0][0] == SCRIPT_TEXT:
					self.monitor.insert_with_tags_by_name(self.monitor.get_end_iter(),
														  chunk[1][0][1], current)

			elif chunk[0] == TEXT_META:
				name, args = chunk[1], chunk[2:]

				if name in [r'\0', r'\h']:
					current = 'sakura'
				elif name in [r'\1', r'\u']:
					current = 'kero'
				elif name == r'\_s':
					current = 'sync'

				if self.view_style == self.STYLE_TALK:
					pre_end = self.monitor.get_end_iter()
					first = pre_end.backward_char()
					if name in [r'\0', r'\h', r'\1', r'\u', r'\_s'] and \
						   pre_end.get_char() != '\n' and first:
						self.monitor.insert_with_tags_by_name(self.monitor.get_end_iter(),
															  '\n', 'script')

					elif name == r'\URL':
						for n in range(len(args)):
							self.monitor.insert_with_tags_by_name(self.monitor.get_end_iter(),
																  "[%s]" % args[n][0][1], 'script')

				elif self.view_style == self.STYLE_SCRIPT:
					self.monitor.insert_with_tags_by_name(self.monitor.get_end_iter(),
														  name, 'script')
					for n in range(len(args)):
						self.monitor.insert_with_tags_by_name(self.monitor.get_end_iter(),
															  "[%s]" % args[n], 'script')

				elif self.view_style == self.STYLE_SCRIPT_WITH_LINEFEED:
					self.monitor.insert_with_tags_by_name(self.monitor.get_end_iter(),
														  name, 'script')
					for n in range(len(args)):
						self.monitor.insert_with_tags_by_name(self.monitor.get_end_iter(),
															  "[%s]" % args[n], 'script')
					if name == r'\n':
						self.monitor.insert_with_tags_by_name(self.monitor.get_end_iter(),
															  '\n', 'script')

		self.monitor.insert_with_tags_by_name(self.monitor.get_end_iter(),
											  '\n', 'script')

	def sort(self, widget, colnum):
		if type(widget) is gtk.Button:
			self.cols[colnum][1] = not self.cols[colnum][1]

		self.entries.freeze()
		self.entries.set_sort_type(self.cols[colnum][1])
		self.entries.set_sort_column(colnum)
		self.entries.sort()
		self.entries.thaw()
		self.entries.grab_focus()
		self.current_sort = colnum

		if self.entries.selection:
			self.entries.moveto(self.entries.selection[0], colnum, 0.5, 0)
			self.set_focus_row(self.entries.selection[0], self.entries.rows)

	def popup(self, widget, event):
		if event.button == 3:
			sel = self.entries.get_selection_info(int(event.x), int(event.y))
			self.set_focus_row(sel[0], self.entries.rows)
			self.popup_menu.popup(None, None, None, event.button, 0)

	def button_popup(self, widget, event, sender):
		if event.button in [1, 3]:
			sender.popup(None, None, None, event.button, 0)

	def open(self, widget, data):
		self.window.show()

	def close(self, widget=None, e=None, data=None):
		self.window.hide()
		return gtk.TRUE

	def update(self, data, widget):
		range = self.logger.get_range()
		if range:
			self.initialize(range, widget)
		else:
			self.initialize(None, widget)

	def initialize(self, changed, widget):
		# Bad /-P
		current_row = None
		if self.entries.selection:
			current_row = self.entries.get_row_data(self.entries.selection[0])

		self.monitor.set_text("")

		if changed in [None, 0]:
			self.logger.set_channel(LogManager.ALL_CHANNELS)
			self.logger.set_range(LogManager.RANGE_LAST50BOTTLES)
			self.range_l.set_text(self.labels[LogManager.RANGE_LAST50BOTTLES])

		elif type(changed) == StringType:
			self.logger.set_channel(changed)

		elif changed <= 100:
			self.logger.set_range(changed)
			self.range_l.set_text(self.labels[changed])

		error = self.logger.reload()
		if error:
			return

 		loglist = self.logger.get_formatted_logs()
		self.entries.freeze()
		self.entries.clear()
		for log in loglist:
			mid = log[-1]
			del log[-1]
			tuple(log)
			n = self.entries.append(log)
			self.entries.set_row_data(n, mid)
		self.sort(widget, self.current_sort)

		# Bad /-P
		if not current_row == None:
			for i in range(self.entries.rows):
				if self.entries.get_row_data(i) == current_row:
					current_row = i
					break
			if type(current_row) is int:
				self.set_focus_row(current_row, self.entries.rows)
				self.entries.moveto(current_row, 0, 0.5, 0)

		self.entries.columns_autosize()
		self.entries.thaw()
		self.entries.grab_focus()

		if self.entries.rows <= 0:
			self.pb.set_sensitive(gtk.FALSE)
		else:
			self.pb.set_sensitive(gtk.TRUE)

	def set_focus_row(self, row, row_count):
		if row_count == 0:
			return
		if row_count == 1:
			r = 0
		else:
			r = min((row+0.00001)/(row_count-1),1.0)
		self.entries.emit("scroll-vertical", gtk.SCROLL_JUMP, r)

	def play(self, widget, data):
		script = self.entries.get_text(self.entries.selection[0], 5)
		if script is None:
			return
		self.app.send_local_message(script)

	def vote(self, widget, data):
		mid = self.entries.get_row_data(self.entries.selection[0])
		if mid is None:
			return

		d = gtk.Dialog(unicode(_("Really?"), "utf-8"), parent=self.window,
					   flags=gtk.DIALOG_NO_SEPARATOR|gtk.DIALOG_MODAL)
		d.connect("destroy", lambda w: w.destroy)
		d.set_border_width(10)
		l = gtk.Label(unicode(_("Vote this bottle?"), "utf-8"))
		l.set_justify(gtk.JUSTIFY_RIGHT)
		l.show()
		d.vbox.add(l)
		d.add_button(unicode(_("OK"), "utf-8"), False)
		d.add_button(unicode(_("Cancel"), "utf-8"), True)
		d.show()
		cancel = d.run()
		d.destroy()

		if cancel:
			return
		self.app.vote_message(mid, VOTE)

	def agree(self, widget, data):
		mid = self.entries.get_row_data(self.entries.selection[0])
		if mid is None:
			return

		d = gtk.Dialog(unicode(_("Really?"), "utf-8"), parent=self.window,
					   flags=gtk.DIALOG_NO_SEPARATOR|gtk.DIALOG_MODAL)
		d.connect("destroy", lambda w: w.destroy)
		d.set_border_width(10)
		l = gtk.Label(unicode(_("Agree this bottle?"), "utf-8"))
		l.set_justify(gtk.JUSTIFY_RIGHT)
		l.show()
		d.vbox.add(l)
		d.add_button(unicode(_("OK"), "utf-8"), False)
		d.add_button(unicode(_("Cancel"), "utf-8"), True)
		d.show()
		cancel = d.run()
		d.destroy()

		if cancel:
			return
		self.app.vote_message(mid, AGREE)

	def edit_copy(self, widget, data):
		if not self.entries.selection:
			return
		script = self.entries.get_text(self.entries.selection[0], 5)
		if not script:
			return

		self.app.insert(script)

	def help(self, event=None):
		pass

if __name__ == "__main__":
	app = Application()
	app.main()
