/** @file
 */
#if defined(HAVE_CONFIG_H)
#  include "../config.h"
#endif
#include "../gettext.h"

#include <boost/bind.hpp>
#include <boost/format.hpp>
#include <glibmm/i18n.h>
#include <gtkmm/stock.h>
#include <gtkmm/clipboard.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>

#include "context_menu.hpp"

#if !defined(HAVE_GNOME_VFS_MIME_GET_DEFAULT_APPLICATION_FOR_URI)
#  define gnome_vfs_mime_get_default_application_for_uri(A, B) gnome_vfs_mime_get_default_application(B)
#endif
#if !defined(HAVE_GNOME_VFS_MIME_GET_ALL_APPLICATIONS_FOR_URI)
#  define gnome_vfs_mime_get_all_applications_for_uri(A, B) gnome_vfs_mime_get_all_applications(B)
#endif


namespace gdestraier {
  namespace gui {


    context_menu::context_menu() {
    }

    context_menu::~context_menu() {
    }




    /** @brief ファイルメニューを登録します
     *  @param filemenu 登録するメニュー
     *  ファイルメニューを登録しておくと、コンテキストに応じて更新されます
     */
    void context_menu::add_filemenu(Gtk::Menu* filemenu) {
      filemenu_type menudata;
      menudata.menu_ = filemenu;

      Gtk::Menu::MenuList& items = filemenu->items();
      for (Gtk::Menu::MenuList::iterator i = items.begin(); i != items.end(); i++) {
        Glib::ustring name = i->get_name();

        if (name == "menu_open") {
          i->set_sensitive(false);
        } else if (name == "menu_open_with") {
          i->set_sensitive(false);
          menudata.open_with_connection_ =
            i->signal_activate().connect(boost::bind(&signal_open_with_type::emit, &signal_open_with_));
        } else if (name == "menu_properties") {
          i->set_sensitive(false);
          menudata.properties_connection_ =
            i->signal_activate().connect(boost::bind(&signal_properties_type::emit, &signal_properties_));
        }
      }

      filemenus_.push_back(menudata);
    }

    

    /** @brief ファイルメニューを登録抹消します
     *  @param filemenu 抹消するメニュー
     */
    void context_menu::remove_filemenu(Gtk::Menu* filemenu)
    {
      for (filemenus_type::iterator i = filemenus_.begin(); i != filemenus_.end(); i++) {
        if (i->menu_ == filemenu) {
          i->open_connection_.disconnect();
          i->open_with_connection_.disconnect();
          i->properties_connection_.disconnect();

          filemenus_.erase(i);
        }
      }
    }



    void context_menu::update(Glib::ustring const& uri, Glib::ustring const& mime_type)
    {
      //
      // MIMEタイプを元にアプリケーションを列挙しつつ、ポップアップメニューを初期化します
      //
      typedef Gtk::Menu_Helpers::MenuElem menuelm_type;
      typedef Gtk::Menu_Helpers::StockMenuElem stockmenuelm_type;
      Gtk::Menu_Helpers::MenuList& menuitems = popup_.items();
      menuitems.erase(menuitems.begin(), menuitems.end());

      // Default application
      boost::format openfmt(_("Open with %1$s"));
      ::GnomeVFSMimeApplication* default_app =
          ::gnome_vfs_mime_get_default_application_for_uri(uri.c_str(), mime_type.c_str());
      if (default_app) {
        menuitems.push_back(menuelm_type((openfmt % default_app->name).str(),
                                         boost::bind(&signal_open_type::emit,
                                                     &signal_open_,
                                                     Glib::ustring(default_app->id)
                                                     ))  );

        menuitems.push_back(Gtk::Menu_Helpers::SeparatorElem()); // セパレータ
      }

      // その他のアプリケーション
      ::GList* applist = ::gnome_vfs_mime_get_all_applications_for_uri(uri.c_str(), mime_type.c_str());
      for (GList* p = applist; p != 0; p = g_list_next(p)) {
        ::GnomeVFSMimeApplication* app = static_cast<GnomeVFSMimeApplication*>(p->data);
        menuitems.push_back(menuelm_type((openfmt % app->name).str(),
                                         boost::bind(&signal_open_type::emit,
                                                     &signal_open_,
                                                     Glib::ustring(app->id)
                                                     )) );
      }


      // Add common items
      menuitems.push_back(menuelm_type(_("Open with (_A)..."),
                                       boost::bind(&signal_open_with_type::emit, &signal_open_with_) ));
      menuitems.push_back(Gtk::Menu_Helpers::SeparatorElem()); // セパレータ
      menuitems.push_back(stockmenuelm_type(Gtk::Stock::COPY,
                                            boost::bind(&context_menu::on_copy_activate, this, uri) ));
      menuitems.push_back(Gtk::Menu_Helpers::SeparatorElem()); // セパレータ
      menuitems.push_back(stockmenuelm_type(Gtk::Stock::PROPERTIES,
                                            boost::bind(&signal_properties_type::emit, &signal_properties_) ));


      //
      // ファイルメニューを更新します
      //
      for (filemenus_type::iterator imenu = filemenus_.begin(); imenu != filemenus_.end(); imenu++) {
        Gtk::Menu::MenuList& items = imenu->menu_->items();
        for (Gtk::Menu::MenuList::iterator i = items.begin(); i != items.end(); i++) {
          Glib::ustring name = i->get_name();

          if (name == "menu_open") {
            imenu->open_connection_.disconnect();
            i->set_sensitive(false);
            if (default_app) {
              imenu->open_connection_ =
                i->signal_activate().connect(boost::bind(&signal_open_type::emit,
                                                         &signal_open_,
                                                         Glib::ustring(default_app->id)
                                                         ));
              i->set_sensitive(true);
            }
          } else if (name == "menu_open_with" || name == "menu_properties")
            i->set_sensitive(true);
        }
      }
      if (default_app)
        ::gnome_vfs_mime_application_free(default_app);
      for (GList* p = applist; p != 0; p = g_list_next(p)) {
        ::gnome_vfs_mime_application_free(static_cast<GnomeVFSMimeApplication*>(p->data));
      }
      ::g_list_free(applist);
    }


    void context_menu::clear() {
      for (filemenus_type::iterator imenu = filemenus_.begin(); imenu != filemenus_.end(); imenu++) {
        Gtk::Menu::MenuList& items = imenu->menu_->items();
        for (Gtk::Menu::MenuList::iterator i = items.begin(); i != items.end(); i++) {
          Glib::ustring name = i->get_name();
          
          if (name == "menu_open") {
            i->set_sensitive(false);
          } else if (name == "menu_open_with") {
            i->set_sensitive(false);
          } else if (name == "menu_properties") {
            i->set_sensitive(false);
          }
        }
      }
    }


    void context_menu::on_copy_activate(Glib::ustring const& uri)
    {
      Gtk::Clipboard::get()->set_text(uri);
    }

  }
}
