/** @file
 */
#if defined(HAVE_CONFIG_H)
#  include "../config.h"
#endif

#include <iostream>
#include <vector>
#include <boost/format.hpp>

#include <gtkmm/textbuffer.h>
#include "../gettext.h"
#include <glibmm/i18n.h>
#include <glibmm/spawn.h>
#include <glibmm/main.h>

#include "gdestraier.hpp"
#include "model/preferences.hpp"
#include "index_build_dialog.hpp"

#include "glade-helper.hpp"

namespace gdestraier {
  namespace gui {

    namespace {
      class index_column_record : public Gtk::TreeModelColumnRecord {
      public:
        Gtk::TreeModelColumn<bool>           build_;
        Gtk::TreeModelColumn<Glib::ustring>  description_;
        Gtk::TreeModelColumn<gdestraier::model::preferences::indexes_type::const_iterator> raw_iter_;

      protected:
        index_column_record() { add(build_); add(description_); add(raw_iter_); }

      public:
        static index_column_record& instance() {
          static index_column_record inst;
          return inst;
        }
      };

    }



    index_build_dialog::index_build_dialog(Gtk::Dialog::BaseObjectType* cobject,
                                           ::GladeXML* glade)
      : Gtk::Dialog(cobject)
    {
      // Save precious widgets
      gdestraier::gui::get_widget(glade, "indexes_view", &indexes_view_);
      gdestraier::gui::get_widget(glade, "lock_timeout", &lock_timeout_);
      gdestraier::gui::get_widget(glade, "collect_overwrite_spaces", &collect_overwrite_spaces_);
      gdestraier::gui::get_widget(glade, "purge_nonexist_documents", &purge_nonexist_documents_);
      gdestraier::gui::get_widget(glade, "collect_deleted_spaces", &collect_deleted_spaces_);
      gdestraier::gui::get_widget(glade, "run_optimize", &run_optimize_);
      gdestraier::gui::get_widget(glade, "no_gather", &no_gather_);
      gdestraier::gui::get_widget(glade, "generate_keywords", &generate_keywords_);
      gdestraier::gui::get_widget(glade, "cachesize_button", &cache_size_button_);
      gdestraier::gui::get_widget(glade, "build_log", &log_view_);
      gdestraier::gui::get_widget(glade, "close", &close_button_);
      gdestraier::gui::get_widget(glade, "check_all_indexes", &check_all_indexes_button_);
      gdestraier::gui::get_widget(glade, "uncheck_all_indexes", &uncheck_all_indexes_button_);
      gdestraier::gui::get_widget(glade, "build_execute", &build_execute_button_);
      gdestraier::gui::get_widget(glade, "build_stop", &build_stop_button_);

      // Connect signals
      build_execute_button_->signal_clicked().connect(sigc::mem_fun(this, &index_build_dialog::on_build_execute_activate));
      build_stop_button_->signal_clicked().connect(sigc::mem_fun(this, &index_build_dialog::on_build_stop_activate));
      check_all_indexes_button_->signal_clicked().connect(sigc::mem_fun(this, &index_build_dialog::on_check_all_activate));
      uncheck_all_indexes_button_->signal_clicked().connect(sigc::mem_fun(this, &index_build_dialog::on_uncheck_all_activate));
      

      builder_.signal_stdout_.connect(sigc::mem_fun(this, &index_build_dialog::on_log_received));
      builder_.signal_stderr_.connect(sigc::mem_fun(this, &index_build_dialog::on_log_received));



      //
      // Initialize dialog data
      //

      // インデックスビューを初期化
      index_column_record& columns = index_column_record::instance();
      indexes_model_ = Gtk::ListStore::create(columns);
      indexes_view_->set_model(indexes_model_);
      indexes_view_->append_column_editable(_("Build"), columns.build_);
      indexes_view_->append_column(_("Desc."), columns.description_);

      // インデックスビューのモデルを初期化
      // @TODO TreeNodeChildrenのback_inserterが欲しい。切に...
      gdestraier::model::preferences const& pref = gdestraier::gui::get_preferences();
      typedef gdestraier::model::preferences::indexes_type indexes_type;
      for (indexes_type::const_iterator i = pref.indexes_.begin(); i != pref.indexes_.end(); i++) {

        if (i->document_path_.empty()) continue; // 文書ファイルと結び付いていなければ構築できない

        Gtk::TreeModel::iterator ii = indexes_model_->append();
        ii->set_value(columns.description_, Glib::ustring(i->description_));
        ii->set_value(columns.build_, false);
        ii->set_value(columns.raw_iter_, i);
      }
    }


    index_build_dialog::~index_build_dialog()
    {
    }




    void
    index_build_dialog::on_log_received(Glib::ustring const& line) {
      Glib::RefPtr<Gtk::TextBuffer> buff(log_view_->get_buffer());
      buff->insert(buff->end(), line);
    }



    void
    index_build_dialog::on_check_all_activate()
    {
      index_column_record& columns = index_column_record::instance();
      for (Gtk::TreeModel::Children::iterator i = indexes_model_->children().begin();
           i != indexes_model_->children().end(); i++)
        i->set_value(columns.build_, true);
    }


    void
    index_build_dialog::on_uncheck_all_activate()
    {
      index_column_record& columns = index_column_record::instance();
      for (Gtk::TreeModel::Children::iterator i = indexes_model_->children().begin();
           i != indexes_model_->children().end(); i++)
        i->set_value(columns.build_, false);
    }



    /** @brief 実行ボタンが押されました
     */
    void
    index_build_dialog::on_build_execute_activate()
    {
      // UI の有効/無効を切替えます
      build_execute_button_->set_sensitive(false);
      build_stop_button_->set_sensitive(true);
      close_button_->set_sensitive(false);
      indexes_view_->set_sensitive(false);
      collect_overwrite_spaces_->set_sensitive(false);
      purge_nonexist_documents_->set_sensitive(false);
      collect_deleted_spaces_->set_sensitive(false);
      run_optimize_->set_sensitive(false);
      no_gather_->set_sensitive(false);
      generate_keywords_->set_sensitive(false);
      check_all_indexes_button_->set_sensitive(false);
      uncheck_all_indexes_button_->set_sensitive(false);
      cache_size_button_->set_sensitive(false);


      // ビルダを再設定します
      builder_.reset();
      builder_.lock_timeout_             = lock_timeout_->get_value_as_int();
      builder_.collect_overwrite_spaces_ = collect_overwrite_spaces_->get_active();
      builder_.collect_deleted_spaces_   = collect_deleted_spaces_->get_active();
      builder_.gather_documents_         = ! no_gather_->get_active();
      builder_.purge_nonexist_documents_ = purge_nonexist_documents_->get_active();
      builder_.optimize_                 = run_optimize_->get_active();
      builder_.generate_keywords_        = generate_keywords_->get_active();
      builder_.cache_size_               = cache_size_button_->get_value_as_int();



      // @TODO インデックスの並列更新をサポートしたい
      index_column_record& columns = index_column_record::instance();
      typedef gdestraier::model::preferences::indexes_type indexes_type;
      for (Gtk::TreeModel::Children::iterator i = indexes_model_->children().begin();
           i != indexes_model_->children().end(); i++) {

        if (! i->get_value(columns.build_)) continue; // ビルド対象に選択されていなかった

        indexes_type::const_iterator index = i->get_value(columns.raw_iter_);

        builder_.launch(&*index);
      }


      // UI の有効/無効を切替えます
      build_execute_button_->set_sensitive(true);
      build_stop_button_->set_sensitive(false);
      close_button_->set_sensitive(true);
      indexes_view_->set_sensitive(true);
      collect_overwrite_spaces_->set_sensitive(true);
      purge_nonexist_documents_->set_sensitive(true);
      collect_deleted_spaces_->set_sensitive(true);
      run_optimize_->set_sensitive(true);
      no_gather_->set_sensitive(true);
      generate_keywords_->set_sensitive(true);
      check_all_indexes_button_->set_sensitive(true);
      uncheck_all_indexes_button_->set_sensitive(true);
      cache_size_button_->set_sensitive(true);
    }



    /** @brief 停止ボタンが押されました
     */
    void
    index_build_dialog::on_build_stop_activate()
    {
      builder_.cancel();
    }







  }
}
