/** @file
 */
#if defined(HAVE_CONFIG_H)
#  include "config.h"
#endif

#include "../gettext.h"
#include <glibmm/i18n.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/liststore.h>
#include <gtkmm/filechooserdialog.h>
#include <gtkmm/stock.h>
#include "gdestraier.hpp"
#include "index_edit_dialog.hpp"
#include "model/preferences.hpp"
#include "model/languages.hpp"
#include "model/encodings.hpp"

#include "glade-helper.hpp"

namespace gdestraier {
  namespace gui {


    namespace {
      /** @brief ファイルタイプコンボの為のカラムレコード */
      class filetypes_column_record : public Gtk::TreeModelColumnRecord {
      public:
        Gtk::TreeModelColumn<Glib::ustring> description_;
        Gtk::TreeModelColumn<gdestraier::model::filetype const*> object_;

      protected:
        filetypes_column_record() { add(description_); add(object_); }

      public:
        static filetypes_column_record const& instance() {
          static filetypes_column_record inst;
          return inst;
        }
      };


      /** @brief 言語コンボの為のカラムレコード */
      class languages_column_record : public Gtk::TreeModelColumnRecord {
      public:
        Gtk::TreeModelColumn<Glib::ustring> name_;
        Gtk::TreeModelColumn<Glib::ustring> id_;
      protected:
        languages_column_record() { add(name_); add(id_); }
      public:
        static languages_column_record const& instance() {
          static languages_column_record inst;
          return inst;
        }
      };


      /** @brief エンコーディングコンボの為のカラムレコード */
      class encodings_column_record : public Gtk::TreeModelColumnRecord {
      public:
        Gtk::TreeModelColumn<Glib::ustring> id_;
        Gtk::TreeModelColumn<gdestraier::model::encoding const*> object_;
      protected:
        encodings_column_record() { add(id_); add(object_); }
      public:
        static encodings_column_record const& instance() {
          static encodings_column_record inst;
          return inst;
        }
      };


    };


    index_edit_dialog::index_edit_dialog(Gtk::Dialog::BaseObjectType* cobject,
                                         ::GladeXML* glade) :
      Gtk::Dialog(cobject),
      has_error_(false)
    {
      //gdestraier::model::preferences& pref = gdestraier::gui::get_preferences();

      // Save precious widgets.
      gdestraier::gui::get_widget(glade, "description", &description_entry_);

      gdestraier::gui::get_widget(glade, "index_location_combo", &index_location_combo_);
      gdestraier::gui::get_widget(glade, "index_path_entry", &index_path_entry_);
      gdestraier::gui::get_widget(glade, "index_user_entry", &index_user_entry_);
      gdestraier::gui::get_widget(glade, "index_password_entry", &index_password_entry_);

      gdestraier::gui::get_widget(glade, "description_label1", &description_label_1_);
      gdestraier::gui::get_widget(glade, "document_folder", &document_path_entry_);
      gdestraier::gui::get_widget(glade, "choose_document_folder", &choose_document_path_button_);
      gdestraier::gui::get_widget(glade, "active", &active_checkbutton_);
      gdestraier::gui::get_widget(glade, "filetype", &filetype_combo_);
      gdestraier::gui::get_widget(glade, "document_encoding", &document_encoding_combo_);
      gdestraier::gui::get_widget(glade, "language", &language_combo_);
      gdestraier::gui::get_widget(glade, "filesystem_encoding", &filesystem_encoding_combo_);
      gdestraier::gui::get_widget(glade, "use_N-gram_for_all_languages", &use_Ngram_for_all_languages_);
      gdestraier::gui::get_widget(glade, "include_document_title_to_body", &include_document_title_to_body_);
      gdestraier::gui::get_widget(glade, "include_document_uri_to_body", &include_document_uri_to_body_);
      gdestraier::gui::get_widget(glade, "quick_build", &enable_quick_build_checkbutton_);

      gdestraier::gui::get_widget(glade, "description_label2", &description_label_2_);
      gdestraier::gui::get_widget(glade, "uri_replace_regex", &uri_replace_rule_regex_entry_);
      gdestraier::gui::get_widget(glade, "uri_replace_to", &uri_replace_rule_to_entry_);
      gdestraier::gui::get_widget(glade, "title_replace_regex", &title_replace_rule_regex_entry_);
      gdestraier::gui::get_widget(glade, "title_replace_to", &title_replace_rule_to_entry_);

      gdestraier::gui::get_widget(glade, "description_label3", &description_label_3_);
      gdestraier::gui::get_widget(glade, "depth_spinbutton", &depth_spinbutton_);
      gdestraier::gui::get_widget(glade, "max_documents_spinbutton", &max_documents_spinbutton_);
      gdestraier::gui::get_widget(glade, "use_snippet", &use_snippet_);

      gdestraier::gui::get_widget(glade, "ok_button", &ok_button_);


      // Connect signals.
      signal_response().connect(sigc::mem_fun(this, &index_edit_dialog::on_response));

      description_entry_->signal_changed()
        .connect(sigc::mem_fun(this, &index_edit_dialog::on_description_changed));
      index_location_combo_->signal_changed()
        .connect(sigc::mem_fun(this, &index_edit_dialog::on_location_changed));

      choose_document_path_button_->signal_clicked().connect(sigc::mem_fun(this, &index_edit_dialog::on_choose_document_folder));

      //
      // Initialize controls
      //

      { // Filetype combo box
        filetypes_column_record const& columns = filetypes_column_record::instance();
        Glib::RefPtr<Gtk::ListStore> model = Gtk::ListStore::create(columns);
        filetype_combo_->set_model(model);

        typedef gdestraier::model::filetypes_manager filetypes_manager_type;
        filetypes_manager_type& ftm = gdestraier::model::filetypes_manager::instance();

        for (filetypes_manager_type::iterator i = ftm.begin(); i != ftm.end(); i++) {
          Gtk::ListStore::iterator ii = model->append();
          ii->set_value(columns.description_, Glib::ustring((*i)->get_translated()));
          ii->set_value(columns.object_, *i);
        } 
      }

      { // Language combo box
        languages_column_record const& columns = languages_column_record::instance();
        Glib::RefPtr<Gtk::ListStore> model = Gtk::ListStore::create(columns);
        language_combo_->set_model(model);

        typedef gdestraier::model::languages langs_type;
        langs_type const& langs = langs_type::instance();
        for (langs_type::const_iterator i = langs.begin(); i != langs.end(); i++) {
          Gtk::ListStore::iterator ii = model->append();
          ii->set_value(columns.name_, Glib::ustring(i->name_ + " (" + i->id_ + ")"));
          ii->set_value(columns.id_, Glib::ustring(i->id_));
        }
      }

      { // Encoding combo box
        encodings_column_record const& columns = encodings_column_record::instance();
        Glib::RefPtr<Gtk::ListStore> doc_model = Gtk::ListStore::create(columns);
        document_encoding_combo_->set_model(doc_model);
        {
          Gtk::ListStore::iterator ii = doc_model->append();
          ii->set_value(columns.id_, Glib::ustring(_("Auto")));
          ii->set_value(columns.object_, static_cast<gdestraier::model::encoding const*>(0));
        }
        Glib::RefPtr<Gtk::ListStore> fs_model = Gtk::ListStore::create(columns);
        filesystem_encoding_combo_->set_model(fs_model);


        for (size_t i = 0; i < gdestraier::model::encoding::num_encodings; i++) {
          Gtk::ListStore::iterator ii = doc_model->append();
          ii->set_value(columns.id_, Glib::ustring(gdestraier::model::encoding::encodings[i].id_));
          ii->set_value(columns.object_, &gdestraier::model::encoding::encodings[i]);

          ii = fs_model->append();
          ii->set_value(columns.id_, Glib::ustring(gdestraier::model::encoding::encodings[i].id_));
          ii->set_value(columns.object_, &gdestraier::model::encoding::encodings[i]);
        }
      }

      filetype_combo_->set_active(0);
      document_encoding_combo_->set_active(0);
      language_combo_->set_active(0);
      filesystem_encoding_combo_->set_active(0);
    }


    index_edit_dialog::~index_edit_dialog()
    {
    }



    namespace {
      Glib::ustring combo_get_active_string(Gtk::ComboBox* combo, int column)
      {
        Glib::ustring str;
        combo->get_active()->get_value(column, str);
        return str;
      }


      void combo_set_active_string(Gtk::ComboBox* combo, int column,
                                   Glib::ustring const& activestr)
      {
        Glib::RefPtr<Gtk::TreeModel> model = combo->get_model();
        Gtk::TreeNodeChildren children = model->children();

        for (Gtk::TreeNodeChildren::const_iterator i = children.begin(); i != children.end(); i++) {
          Glib::ustring str;
          i->get_value(column, str);

          if (str == activestr) {
            combo->set_active(i);
            return;
          }
        }
      }
    }


    gdestraier::model::filetype const*
    index_edit_dialog::get_filetype() const {
      filetypes_column_record const& columns = filetypes_column_record::instance();
      return filetype_combo_->get_active()->get_value(columns.object_);
    }


    void
    index_edit_dialog::set_filetype(gdestraier::model::filetype const* type) {
      filetypes_column_record const& columns = filetypes_column_record::instance();

      Glib::RefPtr<Gtk::TreeModel> model = filetype_combo_->get_model();
      Gtk::TreeNodeChildren children = model->children();

      for (Gtk::TreeNodeChildren::const_iterator i = children.begin(); i != children.end(); i++) {
        if (i->get_value(columns.object_) == type) {
          filetype_combo_->set_active(i);
          return;
        }
      }
    }




    gdestraier::model::encoding const*
    index_edit_dialog::get_document_encoding() const {
      encodings_column_record const& columns = encodings_column_record::instance();
      return document_encoding_combo_->get_active()->get_value(columns.object_);
    }


    void
    index_edit_dialog::set_document_encoding(gdestraier::model::encoding const* enc) {
      encodings_column_record const& columns = encodings_column_record::instance();

      Glib::RefPtr<Gtk::TreeModel> model = document_encoding_combo_->get_model();
      Gtk::TreeNodeChildren children = model->children();

      for (Gtk::TreeNodeChildren::const_iterator i = children.begin(); i != children.end(); i++) {
        if (i->get_value(columns.object_) == enc) {
          document_encoding_combo_->set_active(i);
          return;
        }
      }
    }




    Glib::ustring
    index_edit_dialog::get_language() const {
      languages_column_record const& columns = languages_column_record::instance();
      return combo_get_active_string(language_combo_, columns.id_.index());
    }

    void
    index_edit_dialog::set_language(Glib::ustring const& lang) {
      languages_column_record const& columns = languages_column_record::instance();
      combo_set_active_string(language_combo_, columns.id_.index(), lang);
    }


    gdestraier::model::encoding const*
    index_edit_dialog::get_filesystem_encoding() const {
      encodings_column_record const& columns = encodings_column_record::instance();
      return filesystem_encoding_combo_->get_active()->get_value(columns.object_);
    }

    void
    index_edit_dialog::set_filesystem_encoding(gdestraier::model::encoding const* enc) {
      encodings_column_record const& columns = encodings_column_record::instance();

      Glib::RefPtr<Gtk::TreeModel> model = filesystem_encoding_combo_->get_model();
      Gtk::TreeNodeChildren children = model->children();

      for (Gtk::TreeNodeChildren::const_iterator i = children.begin(); i != children.end(); i++) {
        Glib::ustring str;
        if (i->get_value(columns.object_) == enc) {
          filesystem_encoding_combo_->set_active(i);
          return;
        }
      }
    }




    /** @brief descriptionが編集されました
     */
    void index_edit_dialog::on_description_changed()
    {
      Glib::ustring txt = description_entry_->get_text();
      ok_button_->set_sensitive(! txt.empty());

      description_label_1_->set_text(txt);
      description_label_2_->set_text(txt);
      description_label_3_->set_text(txt);
    }




    /** @brief インデックスの場所が変更されました
     */
    void index_edit_dialog::on_location_changed()
    {
      bool is_remote = index_location_combo_->get_active_row_number() == 1;
      index_user_entry_->set_sensitive(is_remote);
      index_password_entry_->set_sensitive(is_remote);
      depth_spinbutton_->set_sensitive(is_remote);
      use_Ngram_for_all_languages_->set_sensitive(! is_remote);
    }



    /** @brief 文書フォルダ選択ボタンが押下されました
     */
    void index_edit_dialog::on_choose_document_folder()
    {
      Gtk::FileChooserDialog dlg(_("Choose document directory"),
                                 Gtk::FILE_CHOOSER_ACTION_SELECT_FOLDER);
      dlg.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_OK);
      dlg.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
      dlg.set_local_only(false);
      dlg.set_uri(document_path_entry_->get_text());
      if (dlg.run() == Gtk::RESPONSE_OK)
        document_path_entry_->set_text(dlg.get_uri());
    }


    /** @biref ダイアログが終了しようとしています
     */
    void index_edit_dialog::on_response(int response_id)
    {
      has_error_ = false;

      if (response_id == Gtk::RESPONSE_OK) {
        if (get_description() == "") {
          has_error_ = true;
          Gtk::MessageDialog msg(_("Most specify a description."), false, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE, true);
          msg.run();
          response_id = 0;
        } else if (get_index_path() == get_document_path()) {
          has_error_ = true;
          Gtk::MessageDialog msg(_("Document and index must not same directory."),
                                 false, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE, true);
          msg.run();
        }
      }
    }
  }
}
