/** @file
 */
#if defined(HAVE_CONFIG_H)
#  include "../config.h"
#endif
#include <cassert>
#include <cstdio>
#include <boost/format.hpp>
#include <boost/bind.hpp>
#include "../gettext.h"
#include <glibmm/i18n.h>
#include <gtkmm/main.h>
#include <gtkmm/box.h>
#include <gtkmm/toolbar.h>
#include <libgnomeui/gnome-about.h>

#include <estraier.h>

#include "gdestraier.hpp"
#include "mainwindow.hpp"
#include "model/preferences.hpp"
#include "preferences_dialog.hpp"
#include "index_build_dialog.hpp"
#if defined(HAVE_DBUS_GLIB)
#  include "dbus-server.hpp"
#endif
#include "glade-helper.hpp"


namespace gdestraier {
  namespace gui {


    namespace {
      class where_index_column_record : public Gtk::TreeModelColumnRecord {
      public:
        Gtk::TreeModelColumn<Glib::ustring> description_;
        Gtk::TreeModelColumn<gdestraier::model::index_type const*> index_ptr_;

      protected:
        where_index_column_record() { add(description_); add(index_ptr_); }

      public:
        static where_index_column_record const& instance() {
          static where_index_column_record inst;
          return inst;
        }
      };
    }



    /** @brief デフォルトコンストラクタ
     */
    mainwindow::mainwindow()
    {
    }


    /** @brief Glade::Xmlから生成するコンストラクタ
     */
    mainwindow::mainwindow(GtkWindow* cobject, ::GladeXML* glade) :
      Gtk::Window(cobject),
      keyword_entry_(0), resultview_(0), search_button_(0)
    {
      // Save precious widgets.
      gdestraier::gui::get_widget(glade, "keyword_entry", &keyword_entry_);
      gdestraier::gui::get_widget(glade, "where_index_combobox", &where_index_combobox_);
      gdestraier::gui::get_widget_derived(glade, "resultview", &resultview_);
      gdestraier::gui::get_widget(glade, "search_button", &search_button_);
      gdestraier::gui::get_widget(glade, "statusbar", &statusbar_);
      Gtk::Menu* filemenu;
      gdestraier::gui::get_widget(glade, "menu_file_menu", &filemenu);
      resultview_->add_filemenu(filemenu);


      // Connect signals.
      signal_delete_event().connect(sigc::mem_fun(this, &mainwindow::on_close));

      search_button_->signal_clicked().connect(sigc::mem_fun(this, &mainwindow::on_search));
      keyword_entry_->signal_activate().connect(sigc::mem_fun(this, &mainwindow::on_search));

      gdestraier::gui::connect_clicked(glade, "menu_file", sigc::mem_fun(this, &mainwindow::on_menu_file_activate));
      gdestraier::gui::connect_clicked(glade, "menu_quit", sigc::mem_fun(this, &mainwindow::on_quit));
      gdestraier::gui::connect_clicked(glade, "menu_preference", boost::bind(&mainwindow::on_menu_preference_activate, this));
      gdestraier::gui::connect_clicked(glade, "menu_build_index", boost::bind(&mainwindow::on_menu_build_index_activate, this));
      gdestraier::gui::connect_clicked(glade, "menu_about", boost::bind(&mainwindow::on_menu_about, this));
      gdestraier::gui::connect_clicked(glade, "menu_quick_build", boost::bind(&mainwindow::on_quick_build_activate, this));
      gdestraier::gui::connect_clicked(glade, "quick_build", boost::bind(&mainwindow::on_quick_build_activate, this));


      // "Where :"
      {
        where_index_column_record const& cols = where_index_column_record::instance();
        where_index_model_ = Gtk::ListStore::create(cols);
        where_index_combobox_->set_model(where_index_model_);

        on_indexes_update();
      }


      gdestraier::model::preferences const& pref = gdestraier::gui::get_preferences();

      // GUI の状態を復帰します
      restore_position();


      //
      // グループ化切替えボタン
      //
      gdestraier::gui::get_widget(glade, "group_none", &grouping_buttons_.none_);
      gdestraier::gui::get_widget(glade, "group_by_index", &grouping_buttons_.by_index_);
      gdestraier::gui::get_widget(glade, "group_by_mime_type", &grouping_buttons_.by_mime_type_);
      gdestraier::gui::get_widget(glade, "group_by_author", &grouping_buttons_.by_author_);
      gdestraier::gui::get_widget(glade, "group_by_language", &grouping_buttons_.by_language_);
      gdestraier::gui::get_widget(glade, "group_by_last_modified", &grouping_buttons_.by_last_modified_);

      grouping_buttons_.by_index_->signal_toggled().connect(boost::bind(&mainwindow::on_grouping_changed, this));
      grouping_buttons_.by_mime_type_->signal_toggled().connect(boost::bind(&mainwindow::on_grouping_changed, this));
      grouping_buttons_.by_author_->signal_toggled().connect(boost::bind(&mainwindow::on_grouping_changed, this));
      grouping_buttons_.by_language_->signal_toggled().connect(boost::bind(&mainwindow::on_grouping_changed, this));
      grouping_buttons_.by_last_modified_->signal_toggled().connect(boost::bind(&mainwindow::on_grouping_changed, this));

      switch (pref.group_key_) {
      case gdestraier::model::preferences::GROUP_BY_NONE:
        grouping_buttons_.none_->set_active(true);
        break;
      case gdestraier::model::preferences::GROUP_BY_INDEX:
        grouping_buttons_.by_index_->set_active(true);
        break;
      case gdestraier::model::preferences::GROUP_BY_MIME_TYPE:
        grouping_buttons_.by_mime_type_->set_active(true);
        break;
      case gdestraier::model::preferences::GROUP_BY_AUTHOR:
        grouping_buttons_.by_author_->set_active(true);
        break;
      case gdestraier::model::preferences::GROUP_BY_LANGUAGE:
        grouping_buttons_.by_language_->set_active(true);
        break;
      case gdestraier::model::preferences::GROUP_BY_LAST_MODIFIED:
        grouping_buttons_.by_last_modified_->set_active(true);
        break;
      }
      resultview_->set_group_key(pref.group_key_);


      //
      // 順序切替えボタン
      //
      gdestraier::gui::get_widget(glade, "order_by_score", &order_buttons_.by_score_);
      gdestraier::gui::get_widget(glade, "order_by_last_modified", &order_buttons_.by_last_modified_);
      gdestraier::gui::get_widget(glade, "order_by_title", &order_buttons_.by_title_);
      order_buttons_.by_score_->signal_toggled().connect(sigc::mem_fun(this, &mainwindow::on_order_changed));
      order_buttons_.by_last_modified_->signal_toggled().connect(sigc::mem_fun(this, &mainwindow::on_order_changed));
      order_buttons_.by_title_->signal_toggled().connect(sigc::mem_fun(this, &mainwindow::on_order_changed));
      switch(pref.sort_key_) {
      case gdestraier::model::preferences::SORT_BY_SCORE:
        order_buttons_.by_score_->set_active(true);
        break;
      case gdestraier::model::preferences::SORT_BY_LAST_MODIFIED:
        order_buttons_.by_last_modified_->set_active(true);
        break;
      case gdestraier::model::preferences::SORT_BY_TITLE:
        order_buttons_.by_title_->set_active(true);
        break;
      }
      resultview_->set_sort_key(pref.sort_key_);


#if defined(HAVE_DBUS_GLIB)
      // Connect D-BUS service
      dbus::server::instance().signal_on_search_request_
        .connect(boost::bind(&mainwindow::search,
                             this,
                             _1));
#endif
    }


    /** @brief 仮想デストラクタ
     */
    mainwindow::~mainwindow()
    {
    }


    /**
     * Windowの位置とサイズを保存します
     */
    void
    mainwindow::save_position()
    {
      gdestraier::model::preferences& pref = gdestraier::gui::get_preferences();

      get_position(pref.GUI_.mainwindow_.left_, pref.GUI_.mainwindow_.top_);
      get_size(pref.GUI_.mainwindow_.width_, pref.GUI_.mainwindow_.height_);
      pref.save();
    }

    /**
     * Windowの位置とサイズを復帰します
     */
    void
    mainwindow::restore_position()
    {
      gdestraier::model::preferences const& pref = gdestraier::gui::get_preferences();

      int x = pref.GUI_.mainwindow_.left_;
      int y = pref.GUI_.mainwindow_.top_;
      int w = pref.GUI_.mainwindow_.width_;
      int h = pref.GUI_.mainwindow_.height_;


      if (w > 0 && h > 0) {
        int sw = ::gdk_screen_width();
        int sh = ::gdk_screen_height();

        if (x >= sw)         x = sw - 100;
        else if (x + w < 50) x = 100;
        if (y >= sh)         y = sh - 100;
        else if (y + h < 50) y = 100;

        move(x, y);
        set_default_size(w, h);
      }
    }



    void
    mainwindow::on_quit()
    {
      save_position();
      ::gtk_main_quit();
    }


    bool
    mainwindow::on_close(::GdkEventAny* event)
    {
      save_position();
      return false;
    }




    /** @brief 指定された文字列での検索を実行します
     */
    void
    mainwindow::search(char const* phrase)
    {
      keyword_entry_->set_text(phrase);
      on_search();
    }


    /** @brief 検索ボタンが押された時に呼び出されます
     */
    void
    mainwindow::on_search()
    {
      gdestraier::model::index_type const* target_index = 0;
      {
        where_index_column_record const& cols = where_index_column_record::instance();
        target_index = where_index_combobox_->get_active()->get_value(cols.index_ptr_);
      }
      double t1, t2;
      t1 = ::est_gettimeofday();
      resultview_->do_search(keyword_entry_->get_text(), target_index);
      t2 = ::est_gettimeofday();

      // 結果の件数を表示する
      unsigned int n = resultview_->get_found_documents();
      statusbar_->pop();
      statusbar_->push((boost::format((n > 1)?
                                      _("%1$d documents found in %2$d/%3$d indexes. (%4$d ms)") :
                                      _("%1$d document found in %2$d/%3$d indexes. (%4$d ms)")
                                      ) %
                        n % 
                        resultview_->get_found_indexes() %
                        resultview_->get_total_indexes() %
                        (t2 - t1)
                        ).str());
    }



    void mainwindow::on_menu_file_activate() {
      resultview_->update_context_menu();
    }

    /** @brief メニュー [Edit -> preferences ]
     */
    void
    mainwindow::on_menu_preference_activate()
    {
      resultview_->clear();  // 安全の為に検索結果をクリアしてしまう
      where_index_model_->clear();

      preferences_dialog* dlg;
      gdestraier::gui::load_toplevel_widget_derived(&dlg);
      dlg->run();
      delete dlg;

      on_indexes_update();
    }

    /** @brief メニュー [Edit -> Build index]
     */
    void
    mainwindow::on_menu_build_index_activate()
    {
      index_build_dialog* dlg;
      gdestraier::gui::load_toplevel_widget_derived(&dlg);
      while (dlg->run() == 0)
        ;
      delete dlg;
    }


    /** @brief メニュー [Help -> About]
     */
    void
    mainwindow::on_menu_about()
    {
      Glib::ustring translator = _("translator_credits");
      if (translator == "translator_credits") translator = "";

      static char const* authors[] = { "seagull <seagull@mitsuki.no-ip.com>", 0 };
      static char const* documenters[] = { 0 };

      ::GtkWidget* dlg =
          GTK_WIDGET(::gnome_about_new(PACKAGE_NAME,
                                       PACKAGE_VERSION,
                                       "Copyright (c) 2005 seagull",
                                       "",
                                       authors,
                                       documenters,
                                       translator.c_str(),
                                       0));
      ::gtk_widget_show(dlg);
    }



    /** @brief グループ化キーの変更
     *
     */
    void
    mainwindow::on_grouping_changed()
    {
      if (grouping_buttons_.none_->get_active())
        resultview_->set_group_key(gdestraier::model::preferences::GROUP_BY_NONE);
      else if (grouping_buttons_.by_index_->get_active())
        resultview_->set_group_key(gdestraier::model::preferences::GROUP_BY_INDEX);
      else if (grouping_buttons_.by_mime_type_->get_active())
        resultview_->set_group_key(gdestraier::model::preferences::GROUP_BY_MIME_TYPE);
      else if (grouping_buttons_.by_author_->get_active())
        resultview_->set_group_key(gdestraier::model::preferences::GROUP_BY_AUTHOR);
      else if (grouping_buttons_.by_language_->get_active())
        resultview_->set_group_key(gdestraier::model::preferences::GROUP_BY_LANGUAGE);
      else if (grouping_buttons_.by_last_modified_->get_active())
        resultview_->set_group_key(gdestraier::model::preferences::GROUP_BY_LAST_MODIFIED);
    }


    /** @brief 順序キーの変更
     */
    void
    mainwindow::on_order_changed()
    {
      if (order_buttons_.by_score_->get_active())
        resultview_->set_sort_key(gdestraier::model::preferences::SORT_BY_SCORE);
      else if (order_buttons_.by_last_modified_->get_active())
        resultview_->set_sort_key(gdestraier::model::preferences::SORT_BY_LAST_MODIFIED);
      else if (order_buttons_.by_title_->get_active())
        resultview_->set_sort_key(gdestraier::model::preferences::SORT_BY_TITLE);
    }


    /** @brief Handle log of quick build.
     */
    void
    mainwindow::on_quick_build_message(Glib::ustring const& msg) {
      statusbar_->pop();
      statusbar_->push(msg);
    }
    

    /** @brief QuickBuild
     */
    void
    mainwindow::on_quick_build_activate()
    {
      set_sensitive(false);

      gdestraier::builder::builder builder;
      builder.gather_documents_ = true;
      builder.signal_stdout_.connect(sigc::mem_fun(this, &mainwindow::on_quick_build_message));
      builder.signal_stderr_.connect(sigc::mem_fun(this, &mainwindow::on_quick_build_message));

      gdestraier::model::preferences const& pref = gdestraier::gui::get_preferences();
      for (gdestraier::model::preferences::indexes_type::const_iterator i = pref.indexes_.begin();
           i != pref.indexes_.end(); i++) {
        if (i->is_enable_quick_build_)
          builder.launch(&*i);
      }

      set_sensitive(true);

      statusbar_->pop();
      statusbar_->push(_("Quick build successfuly finished."));
    }



    /** @brief Update "Where:" combobox.
     */
    void
    mainwindow::on_indexes_update()
    {
      where_index_model_->clear();
      where_index_column_record const& cols = where_index_column_record::instance();
      Gtk::TreeModel::iterator ii = where_index_model_->append();
      ii->set_value(cols.description_, Glib::ustring(_("Everything")));
      ii->set_value(cols.index_ptr_, (gdestraier::model::index_type const*)0);

      gdestraier::model::preferences const& pref = gdestraier::gui::get_preferences();
      typedef gdestraier::model::preferences::indexes_type indexes_type;
      for (indexes_type::const_iterator i = pref.indexes_.begin(); i != pref.indexes_.end(); i++) {
        ii = where_index_model_->append();
        ii->set_value(cols.description_, Glib::ustring(i->description_));
        ii->set_value(cols.index_ptr_, &*i);
      }

      where_index_combobox_->set_active(0);
    }
  }
}
