/** @file
 *  @brief Thumbnail loader for gdestraier
 */
#if defined(HAVE_CONFIG_H)
#  include "../config.h"
#endif
#include <list>

#include <boost/thread.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/thread/condition.hpp>
#include <boost/scoped_ptr.hpp>
#include <libgnomeui/gnome-thumbnail.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <glib/gfileutils.h>

#include "gdestraier.hpp"
#include "thumbnail_loader.hpp"

namespace gdestraier {
  namespace gui {


    class thumbnail_loader_impl : public thumbnail_loader {
    protected:
      typedef boost::mutex::scoped_lock lock_type;

    protected:
      boost::mutex     lock_;
      boost::thread*   raw_thread_;

      boost::condition request_exists_;
      bool             terminate_required_;

      std::list<std::string>  queued_uris_;

      ::GnomeThumbnailFactory* factory_;

    public:
      thumbnail_loader_impl() :
        raw_thread_(0)
      {
        factory_ = ::gnome_thumbnail_factory_new(GNOME_THUMBNAIL_SIZE_NORMAL);
        if (factory_ == 0)
          ::g_warning("Failed to gnome_thumbnail_factory_new.\n");
      }


      virtual ~thumbnail_loader_impl() { stop(); }


      void operator()() {

        while (1) {

          std::string uri;
          while (1) {
            lock_type lk(lock_);
            if (terminate_required_) { return ; /* Thread terminate */ }
            if (! queued_uris_.empty()) { uri = queued_uris_.front(); queued_uris_.pop_front(); break; }

            request_exists_.wait(lk);
          }

          ::GnomeVFSFileInfo* info = ::gnome_vfs_file_info_new();
          ::GnomeVFSResult res = ::gnome_vfs_get_file_info(uri.c_str(), info,
                                                           GNOME_VFS_FILE_INFO_GET_MIME_TYPE);

          if (res == GNOME_VFS_OK && (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE)) {

            std::time_t mtime = 0;
            if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MTIME)
              mtime = info->mtime;


            bool already_exists = false;
            char* th_path = ::gnome_thumbnail_factory_lookup(factory_, uri.c_str(), mtime);
            if (th_path != 0 && ::g_file_test(th_path, ::G_FILE_TEST_EXISTS))
              already_exists = true;
            ::g_free(th_path);

            if (already_exists) {
              // TODO: Must be notify.
            } else {
              ::GdkPixbuf* pic = ::gnome_thumbnail_factory_generate_thumbnail(factory_,
                                                                              uri.c_str(),
                                                                              info->mime_type);
              if (pic)
                ::gnome_thumbnail_factory_save_thumbnail(factory_, pic, uri.c_str(), mtime);
              else
                // 作成失敗したのでマークする
                ::gnome_thumbnail_factory_create_failed_thumbnail(factory_, uri.c_str(), mtime);
              // TODO: Must be notify.
            }
          }

          ::gnome_vfs_file_info_unref(info);
        }
      }




      void start() {
        lock_type lk(lock_);
        if (raw_thread_ == 0 && factory_ != 0) {
          terminate_required_ = false;
          raw_thread_ = new boost::thread(boost::ref(*this));
        }
      }


      virtual void stop()
      {
        lock_type lk(lock_);
        terminate_required_ = true;
        request_exists_.notify_one();

        boost::thread* old = raw_thread_;
        raw_thread_ = 0;
        lk.unlock();

        if (old != 0) {
          old->join();
          delete old;
        }
      }





      virtual int load(Glib::RefPtr<Gdk::Pixbuf>* pixbuf,
                       char const* uri, std::time_t mtime, char const* mime_type)
      {
        *pixbuf = Glib::RefPtr<Gdk::Pixbuf>(0);

        if (::gnome_thumbnail_factory_has_valid_failed_thumbnail(factory_,
                                                                 uri, mtime))
          return false;


        int result;
        char* th_path = ::gnome_thumbnail_factory_lookup(factory_, uri, mtime);

        if (th_path != 0 && ::g_file_test(th_path, ::G_FILE_TEST_EXISTS)) {
          // 作成済みのキャッシュが存在するので、それをロードする
          ::GError* err = 0;
#if defined(HAVE_GDK_PIXBUF_NEW_FROM_FILE_AT_SCALE)
          gdestraier::model::preferences const& pref = gdestraier::gui::get_preferences();
          ::GdkPixbuf* pic = ::gdk_pixbuf_new_from_file_at_scale(th_path,
                                                                 pref.thumbnail_size_,
                                                                 pref.thumbnail_size_,
                                                                 true,
                                                                 &err);
          if (pic == 0)
            result = LOAD_FAILED;
          else {
            *pixbuf = Glib::wrap(pic);
            result = LOAD_SUCCESS;
          }
#else
          ::GdkPixbuf* pic = ::gdk_pixbuf_new_from_file(th_path, &err);
          if (err != 0) {
            ::g_message("%s: %s\n", th_path, err->message);
            ::g_error_free(err);
          }

          if (pic == 0)
            result = LOAD_FAILED;
          else {
            gdestraier::model::preferences const& pref = gdestraier::gui::get_preferences();
            *pixbuf = Glib::wrap(::gnome_thumbnail_scale_down_pixbuf(pic,
                                                                     pref.thumbnail_size_,
                                                                     pref.thumbnail_size_) );
            ::gdk_pixbuf_unref(pic);

            result = LOAD_SUCCESS;
          }
#endif
        } else if (::gnome_thumbnail_factory_can_thumbnail(factory_, uri, mime_type, mtime)) {
          // 作成されておらず、かつ作成可能なので作成ジョブをスケジュールする
          lock_type lk(lock_);
          if (std::find(queued_uris_.begin(), queued_uris_.end(), uri) == queued_uris_.end()) {
            queued_uris_.push_back(uri);
            request_exists_.notify_one();
          }
          result = LOAD_SCHEDULED;
        } else
          result = NO_THUMBNAIL;

        ::g_free(th_path);
        return result;
      }

    };



    thumbnail_loader::thumbnail_loader() { }
    thumbnail_loader::~thumbnail_loader() { }

    thumbnail_loader& thumbnail_loader::get_default() {
      static thumbnail_loader_impl l;

      l.start();

      return l;
    }
  }
}


