/* -*- Mode: C; tab-width: 3; indent-tabs-mode: nil; c-basic-offset: 3 -*- */

/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: image_view.h,v 1.35.2.13 2003/05/21 08:26:36 makeinu Exp $
 */

#ifndef __IMAGE_VIEW_H__
#define __IMAGE_VIEW_H__

#include "gimageview.h"

#include "fileload.h"
#include "gimv_image.h"
#include "image_info.h"
#include "image_loader.h"

#define IMAGEVIEW(obj)         GTK_CHECK_CAST (obj, imageview_get_type (), ImageView)
#define IMAGEVIEW_CLASS(klass) GTK_CHECK_CLASS_CAST (klass, imageview_get_type, ImageViewClass)
#define IS_IMAGEVIEW(obj)      GTK_CHECK_TYPE (obj, imageview_get_type ())

#define IMAGEVIEW_DEFAULT_VIEW_MODE N_("Default Image Viewer")

#define IMG_MIN_SCALE 5
#define IMG_MAX_SCALE 100000
#define IMG_SCALE_STEP 5


typedef enum
{
   IMAGEVIEW_ZOOM_100,  /* Real Size */
   IMAGEVIEW_ZOOM_IN,   /* Increase Image Size 5% */
   IMAGEVIEW_ZOOM_OUT,  /* Decrease Image Size 5% */
   IMAGEVIEW_ZOOM_FIT,  /* Fit to Widnow */
   IMAGEVIEW_ZOOM_10,
   IMAGEVIEW_ZOOM_25,
   IMAGEVIEW_ZOOM_50,
   IMAGEVIEW_ZOOM_75,
   IMAGEVIEW_ZOOM_125,
   IMAGEVIEW_ZOOM_150,
   IMAGEVIEW_ZOOM_175,
   IMAGEVIEW_ZOOM_200,
   IMAGEVIEW_ZOOM_300,
   IMAGEVIEW_ZOOM_FREE
} ImageViewZoomType;


typedef enum
{
   IMAGEVIEW_ROTATE_0,
   IMAGEVIEW_ROTATE_90,
   IMAGEVIEW_ROTATE_180,
   IMAGEVIEW_ROTATE_270
} ImageViewAngle;


typedef enum
{
   ImageViewPlayableDisable = -1,
   ImageViewPlayableStop,
   ImageViewPlayablePlay,
   ImageViewPlayablePause,
   ImageViewPlayableForward,
   ImageViewPlayableReverse,
} ImageViewPlayableStatus;

typedef enum
{
   ImageViewSeekBarDraggingFlag   = 1 << 0
} ImageViewPlayerFlags;


typedef enum
{
   ImageViewPlayerVisibleHide,
   ImageViewPlayerVisibleShow,
   ImageViewPlayerVisibleAuto
} ImageViewPlayerVisibleType;


typedef struct ImageViewClass_Tag   ImageViewClass;
typedef struct ImageViewPrivate_Tag ImageViewPrivate;
typedef struct ImageViewPlugin_Tag  ImageViewPlugin;
typedef struct ImageViewScalableInterFace_Tag   ImageViewScalableInterFace;
typedef struct ImageViewRotatableInterFace_Tag  ImageViewRotatableInterFace;
typedef struct ImageViewScrollableInterFace_Tag ImageViewScrollableInterFace;
typedef struct ImageViewPlayableInterFace_Tag   ImageViewPlayableInterFace;


typedef GList *(*ImageViewNextFn)     (ImageView *iv,
                                       gpointer   list_owner,
                                       GList     *list,
                                       gpointer   data);
typedef GList *(*ImageViewPrevFn)     (ImageView *iv,
                                       gpointer   list_owner,
                                       GList     *list,
                                       gpointer   data);
typedef GList *(*ImageViewNthFn)      (ImageView *iv,
                                       gpointer   list_owner,
                                       GList     *list,
                                       guint      nth,
                                       gpointer   data);
typedef void (*ImageViewRemoveListFn) (ImageView *iv,
                                       gpointer list_owner,
                                       gpointer data);

struct ImageView_Tag
{
   GtkVBox          parent;

   GtkWidget       *draw_area;
   ImageViewPlugin *draw_area_funcs;

   /* other widgets */
   GtkWidget       *table;
   GtkWidget       *hscrollbar;
   GtkWidget       *vscrollbar;
   GtkWidget       *nav_button;
   GtkWidget       *imageview_popup;
   GtkWidget       *zoom_menu;
   GtkWidget       *rotate_menu;
   GtkWidget       *movie_menu;
   GtkWidget       *view_modes_menu;
   GdkCursor       *cursor;
   GtkAdjustment   *hadj, *vadj;

   GtkWidget       *progressbar;

   GtkWidget       *player_container;
   GtkWidget       *player_toolbar;

   struct    /* buttons in player */
   {
      GtkWidget *rw;
      GtkWidget *play;
      GtkWidget *stop;
      GtkWidget *fw;
      GtkWidget *eject;
      GtkWidget *seekbar;
      GtkWidget *play_icon;
   } player;

   /* information about image */
   ImageInfo       *info;

   /* image */
   ImageLoader     *loader;
   GimvImage       *image;

   /* imageview status */
   GdkColor                  *bg_color;
   gboolean                   show_scrollbar;
   gboolean                   continuance_play;
   ImageViewPlayerFlags       player_flags;
   ImageViewPlayerVisibleType player_visible;

   /* image status */
   gfloat           x_scale;
   gfloat           y_scale;
   ImageViewAngle   rotate;
   gboolean         fit_to_frame_when_change_file;
   gboolean         fit_to_frame_zoom_out_only;
   gboolean         fit_to_frame;
   gboolean         keep_aspect;
   gboolean         ignore_alpha;
   gboolean         buffer;

   ImageViewPrivate *priv;
};


struct ImageViewClass_Tag
{
   GtkVBoxClass parent_class;

   void (*image_changed)     (ImageView *iv);
   void (*load_start)        (ImageView *iv, ImageInfo *info);
   void (*load_end)          (ImageView *iv, ImageInfo *info, gboolean cancel);
   void (*set_list)          (ImageView *iv);
   void (*unset_list)        (ImageView *iv);
   void (*rendered)          (ImageView *iv);
   void (*toggle_aspect)     (ImageView *iv, gboolean keep_aspect);
   void (*toggle_buffer)     (ImageView *iv, gboolean buffered);
   void (*thumbnail_created) (ImageView *iv, ImageInfo *info);

   gboolean (*image_pressed) (ImageView *iv, GdkEventButton *button);
   gboolean (*image_released)(ImageView *iv, GdkEventButton *button);
   gboolean (*image_clicked) (ImageView *iv, GdkEventButton *button);
};


GList     *imageview_get_list               (void);
guint      imageview_get_type               (void);
GtkWidget *imageview_new                    (ImageInfo   *info);

void       imageview_change_image           (ImageView   *iv,
                                             ImageInfo   *info);
void       imageview_change_image_info      (ImageView   *iv,
                                             ImageInfo   *info);
void       imageview_draw_image             (ImageView   *iv);
void       imageview_show_image             (ImageView   *iv);

void       imageview_create_thumbnail       (ImageView      *iv);

/* creating/show/hide child widgets */
void       imageview_change_view_mode       (ImageView      *iv,
                                             const gchar    *label);
GtkWidget *imageview_create_zoom_menu       (GtkWidget   *window,
                                             ImageView   *iv,
                                             const gchar *path);
GtkWidget *imageview_create_rotate_menu     (GtkWidget   *window,
                                             ImageView   *iv,
                                             const gchar *path);
GtkWidget *imageview_create_popup_menu      (GtkWidget   *window,
                                             ImageView   *iv,
                                             const gchar *path);
GtkWidget *imageview_create_movie_menu      (GtkWidget   *window,
                                             ImageView   *iv,
                                             const gchar *path);
GtkWidget *imageview_create_view_modes_menu (GtkWidget   *window,
                                             ImageView   *iv,
                                             const gchar *path);
void       imageview_set_progressbar        (ImageView   *iv,
                                             GtkWidget   *progressbar);
void       imageview_set_player_visible     (ImageView   *iv,
                                             ImageViewPlayerVisibleType type);
void       imageview_show_scrollbar         (ImageView      *iv);
void       imageview_hide_scrollbar         (ImageView      *iv);
void       imageview_popup_menu             (ImageView      *iv,
                                             GdkEventButton *event);
void       imageview_set_bg_color           (ImageView      *iv,
                                             gint            red,
                                             gint            green,
                                             gint            blue);
void       imageview_open_navwin            (ImageView      *iv,
                                             gfloat          x_root,
                                             gfloat          y_root);
void       imageview_set_fullscreen         (ImageView      *iv,
                                             GtkWindow      *fullscreen);
void       imageview_unset_fullscreen       (ImageView      *iv);

/* loading */
void       imageview_free_image_buf         (ImageView   *iv);
void       imageview_load_image_buf         (ImageView   *iv);
gboolean   imageview_is_loading             (ImageView   *iv);
void       imageview_cancel_loading         (ImageView   *iv);

/* scalable interface */
void       imageview_zoom_image             (ImageView   *iv,
                                             ImageViewZoomType zoom,
                                             gfloat       x_scale,
                                             gfloat       y_scale);
gboolean   imageview_get_image_size         (ImageView   *iv,
                                             gint        *width,
                                             gint        *height);

/* rotatable interface */
void       imageview_rotate_image           (ImageView   *iv,
                                             ImageViewAngle angle);
void       imageview_rotate_ccw             (ImageView   *iv);
void       imageview_rotate_cw              (ImageView   *iv);

/* scrollable interface */
/*
 * +----------->
 * | (0, 0)                               (width, 0)
 * |  +------------------------------------+
 * |  |                                    |
 * |  |         Image                      |
 * v  |                                    |
 *    |   (x, y)               (x + fwidth, y)
 *    |     +-------------------+          |
 *    |     |                   |          |
 *    |     |                   |          |
 *    |     |    View Port      |          |
 *    |     |                   |          |
 *    |     |                   |          |
 *    |     +-------------------+          |
 *    |   (x, y + fheight)     (x + fwidth, y + fheight)
 *    |                                    |
 *    +------------------------------------+
 *   (0, height)                          (width, height)
 */
void       imageview_get_image_frame_size   (ImageView   *iv,
                                             gint        *fwidth,
                                             gint        *fheight);
gboolean   imageview_get_view_position      (ImageView   *iv,
                                             gint        *x,
                                             gint        *y);
void       imageview_moveto                 (ImageView   *iv,
                                             gint         x,
                                             gint         y);
#define    imageview_set_view_position(iv, x, y) \
           imageview_moveto (iv, x, y)
void       imageview_reset_scrollbar        (ImageView   *iv);

/* playable interface */
gboolean   imageview_is_playable            (ImageView *iv);
void       imageview_playable_play          (ImageView *iv);
void       imageview_playable_stop          (ImageView *iv);
void       imageview_playable_pause         (ImageView *iv);
void       imageview_playable_forward       (ImageView *iv);
void       imageview_playable_reverse       (ImageView *iv);
void       imageview_playable_seek          (ImageView *iv,
                                             guint      pos);
void       imageview_playable_eject         (ImageView *iv);
ImageViewPlayableStatus
           imageview_playable_get_status    (ImageView *iv);
guint      imageview_playable_get_length    (ImageView *iv);
guint      imageview_playable_get_position  (ImageView *iv);

/* list interface */
void       imageview_set_list               (ImageView       *iv,
                                             GList           *list,
                                             GList           *current,
                                             gpointer         list_owner,
                                             ImageViewNextFn  next_fn,
                                             ImageViewPrevFn  prev_fn,
                                             ImageViewNthFn   nth_fn,
                                             ImageViewRemoveListFn remove_list_fn,
                                             gpointer         user_data);
void       imageview_remove_list            (ImageView       *iv,
                                             gpointer         list_owner);
void       imageview_set_list_self          (ImageView       *iv,
                                             GList           *list,
                                             GList           *current);
gboolean   imageview_has_list               (ImageView       *iv);
gint       imageview_image_list_length      (ImageView       *iv);
gint       imageview_image_list_position    (ImageView       *iv);
GList     *imageview_image_list_current     (ImageView       *iv);
void       imageview_next                   (ImageView       *iv);
void       imageview_prev                   (ImageView       *iv);
void       imageview_nth                    (ImageView       *iv,
                                             guint            nth);


/* for ImageView Embeder Plugin */
typedef gboolean   (*ImageViewDrawAreaIsSupportedFn)     (ImageView *iv,
                                                          ImageInfo *info);
typedef GtkWidget *(*ImageViewDrawAreaCreateFn)          (ImageView *iv);
typedef void       (*ImageViewDrawAreaCreateThumbnailFn) (ImageView *iv,
                                                          const gchar *type);
typedef void       (*ImageViewDrawAreaFullscreenFn)      (ImageView *iv);

/* playable */
typedef gboolean   (*ImageViewDrawAreaIsPlayableFn)      (ImageView *iv,
                                                          ImageInfo *info);
typedef gboolean   (*ImageViewDrawAreaIsSeekableFn)      (ImageView *iv);
typedef void       (*ImageViewDrawAreaPlayFn)            (ImageView *iv);
typedef void       (*ImageViewDrawAreaStopFn)            (ImageView *iv);
typedef void       (*ImageViewDrawAreaPauseFn)           (ImageView *iv);
typedef void       (*ImageViewDrawAreaForwardFn)         (ImageView *iv);
typedef void       (*ImageViewDrawAreaReverseFn)         (ImageView *iv);
typedef void       (*ImageViewDrawAreaSeekFn)            (ImageView *iv,
                                                          gfloat     pos); /* [%] */
typedef void       (*ImageViewDrawAreaEjectFn)           (ImageView *iv);
typedef ImageViewPlayableStatus
                   (*ImageViewDrawAreaGetStatusFn)       (ImageView *iv);
typedef guint      (*ImageViewDrawAreaGetLengthFn)       (ImageView *iv);
typedef guint      (*ImageViewDrawAreaGetPositionFn)     (ImageView *iv);


#define GIMV_IMAGE_VIEW_IF_VERSION 2

struct ImageViewPlugin_Tag
{
   const guint32                      if_version; /* plugin interface version */

   const gchar * const                label;

   gint priority_hint;

   ImageViewDrawAreaIsSupportedFn     is_supported_fn;
   ImageViewDrawAreaCreateFn          create_fn;
   ImageViewDrawAreaCreateThumbnailFn create_thumbnail_fn;
   ImageViewDrawAreaFullscreenFn      fullscreen_fn;

   ImageViewScalableInterFace        *scalable;
   ImageViewRotatableInterFace       *rotatable;
   ImageViewScrollableInterFace      *scrollable;

   struct ImageViewPlayableInterFace_Tag {
      ImageViewDrawAreaIsPlayableFn      is_playable_fn;
      ImageViewDrawAreaIsSeekableFn      is_seekable_fn;
      ImageViewDrawAreaPlayFn            play_fn;
      ImageViewDrawAreaStopFn            stop_fn;
      ImageViewDrawAreaPauseFn           pause_fn;
      ImageViewDrawAreaForwardFn         forward_fn;
      ImageViewDrawAreaReverseFn         reverse_fn;
      ImageViewDrawAreaSeekFn            seek_fn;
      ImageViewDrawAreaEjectFn           eject_fn;
      ImageViewDrawAreaGetStatusFn       get_status_fn;
      ImageViewDrawAreaGetLengthFn       get_length_fn;
      ImageViewDrawAreaGetPositionFn     get_position_fn;
   } *playable;
};

GList *imageview_plugin_get_list (void);


/* for internal use */
void imageview_playable_set_status   (ImageView *iv,
                                      ImageViewPlayableStatus status);
void imageview_playable_set_position (ImageView *iv,
                                      gfloat     pos); /* [%] */

#endif /* __IMAGE_VIEW_H__ */
