/* -*- Mode: C; tab-width: 3; indent-tabs-mode: nil; c-basic-offset: 3 -*- */

/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: thumbnail_view.c,v 1.120.2.24 2003/05/21 14:39:21 makeinu Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "gimageview.h"

#include "charset.h"
#include "comment_view.h"
#include "dirview.h"
#include "dnd.h"
#include "duplicates_finder.h"
#include "fileload.h"
#include "fileutil.h"
#include "gfileutil.h"
#include "gimv_image.h"
#include "gtk2-compat.h"
#include "gtk_prop.h"
#include "gtkutils.h"
#include "image_window.h"
#include "menu.h"
#include "prefs.h"
#include "gimv_scrolled.h"
#include "compare_similar.h"
#include "similar_window.h"
#include "thumbnail.h"
#include "thumbnail_support.h"
#include "thumbnail_view.h"
#include "thumbnail_window.h"
#include "thumbnail_view_album.h"
#ifdef ENABLE_EXIF
#   include "exif_view.h"
#endif /* ENABLE_EXIF */


typedef enum
{
   ACTION_NONE,
   ACTION_POPUP,
   ACTION_OPEN_AUTO,
   ACTION_OPEN_AUTO_FORCE,
   ACTION_OPEN_IN_PREVIEW,
   ACTION_OPEN_IN_PREVIEW_FORCE,
   ACTION_OPEN_IN_NEW_WIN,
   ACTION_OPEN_IN_SHARED_WIN,
   ACTION_OPEN_IN_SHARED_WIN_FORCE
} ButtonActionType;


/* callback functions */
static void   cb_open_image            (ThumbView      *tv,
                                        ThumbViewOpenImageType   action,
                                        GtkWidget      *menuitem);
static void   cb_open_image_by_external(GtkWidget      *menuitem,
                                        ThumbView      *tv);
static void   cb_open_image_by_script  (GtkWidget      *menuitem,
                                        ThumbView      *tv);
static void   cb_recreate_thumbnail    (ThumbView      *tv,
                                        guint           action,
                                        GtkWidget      *menuitem);
static void   cb_remove_thumbnail      (ThumbView      *tv,
                                        guint           action,
                                        GtkWidget      *menuitem);
static void   cb_file_property         (ThumbView      *tv,
                                        guint           action,
                                        GtkWidget      *menuitem);
#ifdef ENABLE_EXIF
static void   cb_exif                  (ThumbView      *tv,
                                        guint           action,
                                        GtkWidget      *menuitem);
#endif /* ENABLE_EXIF */
static void   cb_edit_comment          (ThumbView      *tv,
                                        guint           action,
                                        GtkWidget      *menuitem);
static void   cb_file_operate          (ThumbView      *tv,
                                        FileOperateType type,
                                        GtkWidget      *widget);
static void   cb_rename_file           (ThumbView      *tv,
                                        guint           action,
                                        GtkWidget      *menuitem);
static void   cb_remove_file           (ThumbView      *tv,
                                        guint           action,
                                        GtkWidget      *menuitem);
static void   cb_similar_win_destroy (GtkWidget      *window,
                                        ThumbView      *tv);
static void   cb_thumbview_scrollbar_value_changed (GtkWidget *widget,
                                                    ThumbView *tv);


/* compare functions */
static int comp_func_spel  (gconstpointer data1,
                            gconstpointer data2);
static int comp_func_size  (gconstpointer data1,
                            gconstpointer data2);
static int comp_func_atime (gconstpointer data1,
                            gconstpointer data2);
static int comp_func_mtime (gconstpointer data1,
                            gconstpointer data2);
static int comp_func_ctime (gconstpointer data1,
                            gconstpointer data2);
static int comp_func_type  (gconstpointer data1,
                            gconstpointer data2);
static int comp_func_width (gconstpointer data1,
                            gconstpointer data2);
static int comp_func_height(gconstpointer data1,
                            gconstpointer data2);
static int comp_func_area  (gconstpointer data1,
                            gconstpointer data2);

/* other private functions */
static void       thumbview_remove_mode_data    (ThumbView      *tv);
static GList     *thumbview_add_thumb_data      (ThumbView      *tv,
                                                 GList          *filelist);
static GList     *thumbview_add_thumb_data      (ThumbView      *tv,
                                                 GList          *filelist);
static gchar     *get_uri_list                  (GList          *thumblist);
static void       thumbview_button_action       (ThumbView      *tv,
                                                 Thumbnail      *thumb,
                                                 GdkEventButton *event,
                                                 gint            num);
static GtkWidget *create_progs_submenu          (ThumbView      *tv);
static GtkWidget *create_scripts_submenu        (ThumbView      *tv);
static void       thumbview_reset_load_priority (ThumbView      *tv);
static void       thumbview_set_scrollbar_callback    (ThumbView *tv);
static void       thumbview_remove_scrollbar_callback (ThumbView *tv);


/* reference popup menu for each thumbnail */
static GtkItemFactoryEntry thumb_button_popup_items [] =
{
   {N_("/_Open"),                     NULL,  cb_open_image,    THUMB_VIEW_OPEN_IMAGE_AUTO,        NULL},
   {N_("/Open in New _Window"),       NULL,  cb_open_image,    THUMB_VIEW_OPEN_IMAGE_NEW_WIN,     NULL},
   {N_("/Open in S_hared Window"),    NULL,  cb_open_image,    THUMB_VIEW_OPEN_IMAGE_SHARED_WIN,  NULL},
   {N_("/Open in E_xternal Program"), NULL,  NULL,             0,           "<Branch>"},
   {N_("/_Scripts"),                  NULL,  NULL,             0,           "<Branch>"},
   {N_("/---"),                       NULL,  NULL,             0,           "<Separator>"},
   {N_("/_Update Thumbnail"),         NULL,  cb_recreate_thumbnail,  0,     NULL},
   {N_("/Remo_ve from List"),         NULL,  cb_remove_thumbnail,    0,     NULL},
   {N_("/---"),                       NULL,  NULL,             0,           "<Separator>"},
   {N_("/_Property..."),              NULL,  cb_file_property, 0,           NULL},
#ifdef ENABLE_EXIF
   {N_("/Scan _EXIF Data..."),        NULL,  cb_exif,          0,           NULL},
#endif
   {N_("/E_dit Comment..."),          NULL,  cb_edit_comment,  0,           NULL},
   {N_("/---"),                       NULL,  NULL,             0,           "<Separator>"},
   {N_("/Re_name..."),                NULL,  cb_rename_file,   0,           NULL},
   {N_("/_Copy Files To..."),         NULL,  cb_file_operate,  FILE_COPY,   NULL},
   {N_("/_Move Files To..."),         NULL,  cb_file_operate,  FILE_MOVE,   NULL},
   {N_("/_Link Files To..."),         NULL,  cb_file_operate,  FILE_LINK,   NULL},
   {N_("/_Remove file..."),           NULL,  cb_remove_file,   0,           NULL},
   {NULL, NULL, NULL, 0, NULL},
};


static GList   *ThumbViewList = NULL;

static guint    button = 0;
static gboolean pressed = FALSE;
static gboolean dragging = FALSE;
static gint     drag_start_x = 0;
static gint     drag_start_y = 0;



/****************************************************************************
 *
 *  Plugin Management
 *
 ****************************************************************************/
static GList            *thumbview_plugin_list = NULL;
static GHashTable       *thumbview_modes = NULL;
static gchar           **thumbview_mode_labels = NULL;
extern ThumbViewPlugin   thumbalbum_modes[];
extern gint              thumbalbum_modes_num;

gint
thumbview_label_to_num(gchar *label)
{
   ThumbViewPlugin *view, *default_view;
   gint pos;

   g_return_val_if_fail (thumbview_modes, 0);

   if (!label || !*label) return 0;

   view = g_hash_table_lookup (thumbview_modes, label);

   if (!view) {
      default_view = g_hash_table_lookup (thumbview_modes,
                                          DEFAULT_DISP_MODE);
      g_return_val_if_fail (view && default_view, 0);
      pos = g_list_index (thumbview_plugin_list, view);
   } else {
      pos = g_list_index (thumbview_plugin_list, view);
   }

   if (pos < 0)
      return 0;
   else
      return pos;
}


gchar *
thumbview_num_to_label (gint num)
{
   ThumbViewPlugin *view;

   view = g_list_nth_data (thumbview_plugin_list, num);
   g_return_val_if_fail (view, NULL);

   return view->label;
}


GList *
thumbview_get_disp_mode_list (void)
{
   gint i;

   if (!thumbview_modes)
      thumbview_modes = g_hash_table_new (g_str_hash, g_str_equal);

   if (!thumbview_plugin_list) {
      for (i = 0; i < thumbalbum_modes_num; i++) {
         g_hash_table_insert (thumbview_modes,
                              thumbalbum_modes[i].label,
                              &thumbalbum_modes[i]);
         thumbview_plugin_list = g_list_append (thumbview_plugin_list,
                                                &thumbalbum_modes[i]);
      }
   }

   return thumbview_plugin_list;
}


gchar **
thumbview_get_disp_mode_labels (gint *length_ret)
{
   gint i, num;
   gchar **labels;
   GList *list;

   thumbview_get_disp_mode_list ();
   g_return_val_if_fail (thumbview_plugin_list, NULL);

   if (thumbview_mode_labels) {
      *length_ret = sizeof (thumbview_mode_labels) / sizeof (gchar *);
      return thumbview_mode_labels;
   }

   num = *length_ret = g_list_length (thumbview_plugin_list);
   g_return_val_if_fail (num > 0, NULL);

   thumbview_mode_labels = labels = g_new0 (gchar *, num + 1);
   list = thumbview_plugin_list;
   for (i = 0; list && i < num; i++) {
      ThumbViewPlugin *view = list->data;
      if (!view) {
         g_free (labels);
         return NULL;
      }
      labels[i] = g_strdup (view->label);
      list = g_list_next (list);
   }
   labels[num] = NULL;

   return labels;
}


static gint
comp_func_priority (ThumbViewPlugin *plugin1,
                    ThumbViewPlugin *plugin2)
{
   g_return_val_if_fail (plugin1, 1);
   g_return_val_if_fail (plugin2, -1);

   return plugin1->priority_hint - plugin2->priority_hint;
}


gboolean
thumbview_plugin_regist (const gchar *plugin_name,
                         const gchar *module_name,
                         gpointer impl,
                         gint     size)
{
   ThumbViewPlugin *plugin = impl;

   g_return_val_if_fail (module_name, FALSE);
   g_return_val_if_fail (plugin, FALSE);
   g_return_val_if_fail (size > 0, FALSE);
   g_return_val_if_fail (plugin->if_version == GIMV_THUMBNAIL_VIEW_IF_VERSION, FALSE);
   g_return_val_if_fail (plugin->label, FALSE);

   if (!thumbview_plugin_list)
      thumbview_get_disp_mode_list();

   g_hash_table_insert (thumbview_modes,
                        plugin->label,
                        plugin);
   thumbview_plugin_list = g_list_append (thumbview_plugin_list, plugin);
   thumbview_plugin_list = g_list_sort (thumbview_plugin_list,
                                        (GCompareFunc) comp_func_priority);

   return TRUE;
}


/******************************************************************************
 *
 *   Callback functions.
 *
 ******************************************************************************/
/*
 *  cb_open_image:
 *     @ Callback function for popup menu item at thumbnail button ("/Open").
 *     @ Open images on image window. If shared image window is existing,
 *       use it.
 *
 *  thumb    :
 *  action   :
 *  menuitem :
 */
static void
cb_open_image (ThumbView *tv, ThumbViewOpenImageType action, GtkWidget *menuitem)
{
   Thumbnail *thumb;
   GList *thumblist, *node;
   gint listnum;

   g_return_if_fail (tv);

   thumblist = thumbview_get_selection_list (tv);
   if (!thumblist) return;

   listnum = g_list_length (thumblist);

   if (action == THUMB_VIEW_OPEN_IMAGE_SHARED_WIN) {
      thumb = thumblist->data;
      thumbview_open_image (tv, thumb, action);
   } else {
      FilesLoader *files;

      files = files_loader_new ();
      if (listnum > 2)
         files_loader_create_progress_window (files);
      node = thumblist;

      while (node) {
         gint pos;
         gfloat progress;
         gchar buf[32];

         thumb = node->data;
         node = g_list_next (node);

         if (files->status == CANCEL || files->status == STOP) break;

         thumbview_open_image (tv, thumb, action);

         pos = g_list_position (thumblist, node) + 1;
         progress = (gfloat) pos / (gfloat) listnum;
         g_snprintf (buf, 32, "%d/%d files", pos, listnum);
         files_loader_progress_update (files, progress, buf);
      }

      files_loader_destroy_progress_window (files);
      files_loader_delete (files);
   }

   g_list_free (thumblist);
}


/*
 *  cb_open_image_by_external:
 *     @ Callback function for popup menu item at thumbnail button
 *       ("/Open in External Program").
 *     @ Open images by external program
 *
 *  thumb    :
 *  action   :
 *  menuitem :
 */
static void
cb_open_image_by_external (GtkWidget *menuitem, ThumbView *tv)
{
   Thumbnail *thumb;
   GList *thumblist, *node;
   gint action;
   gchar *user_cmd, *cmd = NULL, *tmpstr = NULL, **pair;
   gboolean show_dialog = FALSE;

   g_return_if_fail (menuitem && tv);

   thumblist = node = thumbview_get_selection_list (tv);
   if (!thumblist) return;

   action = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (menuitem), "num"));

   /* find command */
   if (action < sizeof (conf.progs) / sizeof (conf.progs[0])) {
      pair = g_strsplit (conf.progs[action], ",", 3);
      if (!pair[1]) {
         g_strfreev (pair);
         return;
      } else {
         cmd = g_strdup (pair[1]);
      }
      if (pair[2] && !g_strcasecmp (pair[2], "TRUE"))
         show_dialog = TRUE;
      g_strfreev (pair);
   } else {
      return;
   }

   /* create command string */
   while (node) {
      thumb = node->data;
      tmpstr = g_strconcat (cmd, " ",
                            "\"", image_info_get_path (thumb->info), "\"",
                            NULL);
      g_free (cmd);
      cmd = tmpstr;
      node = g_list_next (node);
   }
   tmpstr = g_strconcat (cmd, " &", NULL);
   g_free (cmd);
   cmd = tmpstr;
   tmpstr = NULL;

   tmpstr = cmd;
   cmd = charset_to_internal (cmd,
                              conf.charset_filename,
                              conf.charset_auto_detect_fn,
                              conf.charset_filename_mode);
   g_free (tmpstr);
   tmpstr = NULL;

   if (show_dialog) {
      user_cmd = gtkutil_popup_textentry (_("Execute command"),
                                          _("Please enter options:"),
                                          cmd, NULL, 400,
                                          TEXT_ENTRY_AUTOCOMP_PATH
                                          | TEXT_ENTRY_WRAP_ENTRY
                                          | TEXT_ENTRY_CURSOR_TOP,
                                          GTK_WINDOW (tv->thumb_window->window));
   } else {
      user_cmd = g_strdup (cmd);
   }
   g_free (cmd);
   cmd = NULL;

   /* exec command */
   if (user_cmd) {
      tmpstr = user_cmd;
      user_cmd = charset_internal_to_locale (user_cmd);
      g_free (tmpstr);
      tmpstr = NULL;
      system (user_cmd);
      g_free (user_cmd);
   }

   g_list_free (thumblist);
}


static void
cb_open_image_by_script (GtkWidget *menuitem, ThumbView *tv)
{
   Thumbnail *thumb;
   GList *thumblist, *node;
   gchar *script, *cmd = NULL, *tmpstr = NULL, *user_cmd;

   g_return_if_fail (menuitem && tv);

   thumblist = node = thumbview_get_selection_list (tv);
   if (!thumblist) return;

   script = gtk_object_get_data (GTK_OBJECT (menuitem), "script");
   if (!script || !script || !isexecutable (script)) goto ERROR;

   cmd = g_strdup (script);

   /* create command string */
   while (node) {
      thumb = node->data;
      tmpstr = g_strconcat (cmd, " ",
                            "\"", image_info_get_path (thumb->info), "\"",
                            NULL);
      g_free (cmd);
      cmd = tmpstr;
      node = g_list_next (node);
   }
   tmpstr = g_strconcat (cmd, " &", NULL);
   g_free (cmd);
   cmd = tmpstr;
   tmpstr = NULL;

   {
      tmpstr = cmd;
      cmd = charset_to_internal (cmd,
                                 conf.charset_filename,
                                 conf.charset_auto_detect_fn,
                                 conf.charset_filename_mode);
      g_free (tmpstr);
      tmpstr = NULL;

      if (conf.scripts_show_dialog) {
         user_cmd = gtkutil_popup_textentry (_("Execute script"),
                                             _("Please enter options:"),
                                             cmd, NULL, 400,
                                             TEXT_ENTRY_AUTOCOMP_PATH
                                             | TEXT_ENTRY_WRAP_ENTRY
                                             | TEXT_ENTRY_CURSOR_TOP,
                                             GTK_WINDOW (tv->thumb_window->window));
      } else {
         user_cmd = g_strdup (cmd);
      }
      g_free (cmd);
      cmd = NULL;

      tmpstr = user_cmd;
      user_cmd = charset_internal_to_locale (user_cmd);
      g_free (tmpstr);
      tmpstr = NULL;
   }

   /* exec command */
   if (user_cmd) {
      {   /********** convert charset **********/
         tmpstr = user_cmd;
         user_cmd = charset_internal_to_locale (user_cmd);
         g_free (tmpstr);
         tmpstr = NULL;
      }
      system (user_cmd);
      g_free (user_cmd);
      user_cmd = NULL;
   }

ERROR:
   g_list_free (thumblist);
}


/* FIXME: context of progress should be displayed */
/*
 *  cb_recreate_thumbnail:
 *     @ Callback function for popup menu item at thumbnail button
 *       ("/Recreate Thumbnail").
 *     @ Create thumbnail from original image. 
 *
 *  thumb    :
 *  action   :
 *  menuitem :
 */
static void
cb_recreate_thumbnail (ThumbView *tv, guint action, GtkWidget *menuitem)
{
   Thumbnail *thumb;
   GList *thumblist, *node;

   g_return_if_fail (tv);

   thumblist = thumbview_get_selection_list (tv);
   if (!thumblist) return;

   for (node = thumblist; node; node = g_list_next (node)) {
      thumb = node->data;
      if (!thumb) continue;

      thumbview_refresh_thumbnail (thumb, CREATE_THUMB);
   }

   g_list_free (thumblist);
}
/* END FIXME */


/*
 *  cb_remove_thumbnail:
 *     @ Callback function of popup menu. ("/Remove from List")
 *     @ Remove specified thumbnail from list.
 *
 *  thumb    :
 *  action   :
 *  menuitem :
 */
static void
cb_remove_thumbnail (ThumbView *tv, guint action, GtkWidget *menuitem)
{
   Thumbnail *thumb;
   GList *thumblist, *node;

   g_return_if_fail (tv);

   thumblist = thumbview_get_selection_list (tv);
   if (!thumblist) return;

   node = thumblist;
   while (node) {
      thumb = node->data;
      node = g_list_next (node);
      if (!thumb) continue;

      tv->thumblist = g_list_remove (tv->thumblist, thumb);
      thumbnail_unref (thumb);
   }

   thumbview_redraw (tv, tv->disp_mode, tv->container, NULL);

   g_list_free (thumblist);
}

/*
 *  cb_file_property:
 *     @ Callback function of popup menu. ("/Property")
 *     @ Display infomation of specified image file.
 *
 *  thumb    :
 *  action   :
 *  menuitem :
 */
static void
cb_file_property (ThumbView *tv, guint action, GtkWidget *menuitem)
{
   Thumbnail *thumb;
   GList *thumblist;
   ImageInfo *info;
   gint flags = 0;

   g_return_if_fail (tv);

   thumblist = thumbview_get_selection_list (tv);
   if (!thumblist || g_list_length (thumblist) > 1) return;

   thumb = thumblist->data;
   if (!thumb) return;

   info = image_info_ref (thumb->info);
   if (!info) return;

   if (tv->mode == THUMB_VIEW_MODE_ARCHIVE) {
      flags |= GTK_PROP_EDITABLE | GTK_PROP_NOT_DETECT_TYPE;
   }

   if (dlg_prop_from_image_info (info, flags))
      thumbview_refresh_list (tv);

   image_info_unref (info);
}


#ifdef ENABLE_EXIF
static void
cb_exif (ThumbView *tv, guint action, GtkWidget *menuitem)
{
   Thumbnail *thumb;
   GList *thumblist;

   g_return_if_fail (tv);

   thumblist = thumbview_get_selection_list (tv);
   if (!thumblist || g_list_length (thumblist) > 1) return;

   thumb = thumblist->data;
   if (!thumb) return;

   exif_view_create_window (image_info_get_path (thumb->info),
                            GTK_WINDOW (tv->thumb_window->window));
}
#endif /* ENABLE_EXIF */


static void
cb_edit_comment (ThumbView *tv, guint action, GtkWidget *menuitem)
{
   GList *thumblist, *node;

   g_return_if_fail (tv);

   node = thumblist = thumbview_get_selection_list (tv);
   if (!thumblist) return;

   while (node) {
      Thumbnail *thumb = node->data;
      node = g_list_next (node);

      if (!thumb) continue;

      comment_view_create_window (thumb->info);
   }
}


static gchar *previous_dir = NULL;


static void
cb_file_operate (ThumbView *tv, FileOperateType type, GtkWidget *widget)
{
   g_return_if_fail (tv);
   thumbview_file_operate (tv, type);
}


/*
 *  cb_rename_file:
 *     @ Callback function of popup menu. ("/Rename file")
 *     @ Rename specified thumbnail from list.
 *
 *  thumb    :
 *  action   :
 *  menuitem :
 */
static void
cb_rename_file (ThumbView *tv, guint action, GtkWidget *menuitem)
{
   g_return_if_fail (tv);
   thumbview_rename_file (tv);
}


/*
 *  cb_remove_file:
 *     @ Callback function of popup menu. ("/Remove file")
 *     @ Remove specified file.
 *
 *  thumb    :
 *  action   :
 *  menuitem :
 */
static void
cb_remove_file (ThumbView *tv, guint action, GtkWidget *menuitem)
{
   g_return_if_fail (tv);
   thumbview_delete_files (tv);
}


static void
cb_similar_win_destroy (GtkWidget *window, ThumbView *tv)
{
   g_return_if_fail (tv);

   tv->related_similar_win
      = g_list_remove (tv->related_similar_win, window);
}


static void
cb_thumbview_scrollbar_value_changed (GtkWidget *widget,
                                      ThumbView *tv)
{
   g_return_if_fail (tv);

   thumbview_reset_load_priority (tv);
}


typedef struct ThumbViewButtonAction_Tag
{
   ThumbView *tv;
   Thumbnail *thumb;
   GdkEventButton event;
   gint action;
} ThumbViewButtonAction;


static gint
idle_button_action (gpointer data)
{
   ThumbViewButtonAction *act = data;
   g_return_val_if_fail (act, 0);
   thumbview_button_action (act->tv, act->thumb, &act->event, act->action);
   return 0;
}


/*
 *  thumbview_thumb_button_press_cb:
 *     @ Callback function for mouse button pressed on thumbnail button event.
 *     @ If middle button pressed, open the image on image window.
 *     @ If right buton pressed, open popup menu.
 *
 *  widget :
 *  event  :
 *  thumb  :
 */
gboolean
thumbview_thumb_button_press_cb (GtkWidget *widget,
                                 GdkEventButton *event,
                                 Thumbnail *thumb)
{
   ThumbView   *tv;
   ThumbWindow *tw;
   gint num;

   g_return_val_if_fail (event, FALSE);

   button = event->button;
   pressed = TRUE;
   drag_start_x = event->x;
   drag_start_y = event->y;

   if (!thumb) goto ERROR;

   tv = thumbnail_get_parent_thumbview (thumb);
   g_return_val_if_fail (tv, FALSE);

   tw = tv->thumb_window;
   g_return_val_if_fail (tw, FALSE);

   thumbwin_notebook_drag_src_unset (tw);   /* FIXMEEEEEEEE!! */

   /* reset selection */
   if (event->type == GDK_BUTTON_PRESS
       && (event->button == 2 || event->button == 3))
   {
      if (!thumb->selected) {
         thumbview_set_selection_all (tv, FALSE);
         thumbview_set_selection (thumb, TRUE);
      }
   }

   while (gtk_events_pending()) gtk_main_iteration();

   num = prefs_mouse_get_num_from_event (event, conf.thumbview_mouse_button);
   if (event->type == GDK_2BUTTON_PRESS) {
      tv->button_2pressed_queue = num;
   } else if (num > 0) {
      ThumbViewButtonAction *act = g_new0 (ThumbViewButtonAction, 1);
      act->tv = tv;
      act->thumb = thumb;
      act->event = *event;
      act->action = num;
      gtk_idle_add_full (GTK_PRIORITY_REDRAW,
                         idle_button_action, NULL, act,
                         (GtkDestroyNotify) g_free);
   }

 ERROR:
   if (event->button == 3) /* for avoiding notebook's event */
      return TRUE;
   else
      return FALSE;
}


gboolean
thumbview_thumb_button_release_cb (GtkWidget *widget,
                                   GdkEventButton *event,
                                   Thumbnail *thumb)
{
   ThumbView   *tv;
   ThumbWindow *tw;
   gint num;

   g_return_val_if_fail (event, FALSE);
   if (!thumb) goto ERROR;

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, FALSE);

   tw = tv->thumb_window;
   g_return_val_if_fail (tw, FALSE);

   thumbwin_notebook_drag_src_reset (tw);   /* FIXMEEEEEEEEE!!! */

   if(pressed && !dragging) {
      if (tv->button_2pressed_queue) {
         num = tv->button_2pressed_queue;
         if (num > 0)
            num = 0 - num;
         tv->button_2pressed_queue = 0;
      } else {
         num = prefs_mouse_get_num_from_event (event,
                                               conf.thumbview_mouse_button);
      }
      if (num < 0) {
         ThumbViewButtonAction *act = g_new0 (ThumbViewButtonAction, 1);
         act->tv = tv;
         act->thumb = thumb;
         act->event = *event;
         act->action = num;
         gtk_idle_add_full (GTK_PRIORITY_REDRAW,
                            idle_button_action, NULL, act,
                            (GtkDestroyNotify) g_free);
      }
   }

 ERROR:
   button   = 0;
   pressed  = FALSE;
   dragging = FALSE;

   if (event->button == 3)   /* for avoiding notebook's callback */
      return TRUE;
   else
      return FALSE;
}


gboolean
thumbview_thumb_key_press_cb (GtkWidget *widget,
                              GdkEventKey *event,
                              Thumbnail *thumb)
{
   guint keyval, popup_key = 0;
   GdkModifierType modval, popup_mod = 0;

   g_return_val_if_fail (event, FALSE);

   keyval = event->keyval;
   modval = event->state;

   if (akey.common_popup_menu || *akey.common_popup_menu)
      gtk_accelerator_parse (akey.common_popup_menu, &popup_key, &popup_mod);

   if (keyval == popup_key && (!popup_mod || (modval & popup_mod))) {
      ThumbView *tv = thumbnail_get_parent_thumbview (thumb);
      g_return_val_if_fail (tv, FALSE);
      thumbview_popup_menu (tv, NULL, NULL);
      return TRUE;
   }

   return FALSE;
}


gboolean
thumbview_thumb_key_release_cb (GtkWidget *widget,
                                GdkEventKey *event,
                                Thumbnail *thumb)
{
   g_return_val_if_fail (event, FALSE);
   return FALSE;
}


gboolean
thumbview_motion_notify_cb (GtkWidget *widget,
                            GdkEventMotion *event,
                            Thumbnail *thumb)
{
   ThumbView   *tv;
   ThumbWindow *tw;
   gint dx, dy;

   g_return_val_if_fail (event, FALSE);

   if (thumb) {
      tv = thumb->thumb_view;
      g_return_val_if_fail (tv, FALSE);

      tw = tv->thumb_window;
      g_return_val_if_fail (tw, FALSE);
   }

   if (!pressed)
      return FALSE;

   dx = event->x - drag_start_x;
   dy = event->y - drag_start_y;

   if (!dragging && (abs (dx) > 2 || abs (dy) > 2))
      dragging = TRUE;

   /* return TRUE; */
   return FALSE;
}


void
thumbview_drag_begin_cb (GtkWidget *widget,
                         GdkDragContext *context,
                         gpointer data)
{
   ThumbView *tv = data;
   Thumbnail *thumb;
   GList *thumblist;
   GdkPixmap *pixmap;
   GdkBitmap *mask;
   GdkColormap *colormap;

   g_return_if_fail (tv && widget);

   thumblist = thumbview_get_selection_list (tv);
   if (!thumblist) return;

   thumb = thumblist->data;
   thumbnail_get_icon (thumb, &pixmap, &mask);
   if (g_list_length (thumblist) == 1 && pixmap) {
      colormap = gdk_colormap_get_system ();
      gtk_drag_set_icon_pixmap (context, colormap, pixmap, mask, -7, -7);
   }
}


void
thumbview_drag_data_get_cb (GtkWidget *widget,
                            GdkDragContext *context,
                            GtkSelectionData *seldata,
                            guint info,
                            guint time,
                            gpointer data)
{
   ThumbView *tv = data;
   GList *thumblist;
   gchar *uri_list;

   g_return_if_fail (tv && widget);

   thumblist = thumbview_get_selection_list (tv);

   if (!thumblist) {
      gtkutil_message_dialog (_("Error!!"), _("No files specified!!"),
                              GTK_WINDOW (tv->thumb_window->window));
      return;
   }

   switch (info) {
   case TARGET_URI_LIST:
      if (!thumblist) return;
      uri_list = get_uri_list (thumblist);
      gtk_selection_data_set(seldata, seldata->target,
                             8, uri_list, strlen(uri_list));
      g_free (uri_list);
      break;
   default:
      break;
   }
}


/*
 *  thumbview_drag_data_received_cb:
 *     @ reference "drag_data_received" event callback function.
 */
void
thumbview_drag_data_received_cb (GtkWidget *widget,
                                 GdkDragContext *context,
                                 gint x, gint y,
                                 GtkSelectionData *seldata,
                                 guint info,
                                 guint time,
                                 gpointer data)
{
   ThumbView *tv = data;
   FilesLoader *files;
   GList *list;
   GtkWidget *src_widget;
   ThumbView *src_tab = NULL, *dest_tab = NULL;

   g_return_if_fail (tv && widget);

   src_widget = gtk_drag_get_source_widget (context);
   if (src_widget == widget) return;

   if (src_widget)
      src_tab  = gtk_object_get_data (GTK_OBJECT (src_widget), "gimv-tab");
   dest_tab = gtk_object_get_data (GTK_OBJECT (widget), "gimv-tab");
   if (src_tab == dest_tab) return;

   if (tv->mode == THUMB_VIEW_MODE_DIR) {
      gchar *dirname;

      if (tv->dnd_destdir)
         dirname = tv->dnd_destdir;
      else
         dirname = tv->dirname;

      if (iswritable (dirname)) {
         dnd_file_operation (dirname, context, seldata,
                             time, tv->thumb_window);
      } else {
         gchar error_message[BUF_SIZE], *dir_internal;

         dir_internal = charset_to_internal (dirname,
                                             conf.charset_filename,
                                             conf.charset_auto_detect_fn,
                                             conf.charset_filename_mode);
         g_snprintf (error_message, BUF_SIZE,
                     _("Permission denied: %s"),
                     dir_internal);
         gtkutil_message_dialog (_("Error!!"), error_message,
                                 GTK_WINDOW (tv->thumb_window->window));

         g_free (dir_internal);
      }
      tv->dnd_destdir = NULL;

   } else if (tv->mode == THUMB_VIEW_MODE_COLLECTION) {
      list = dnd_get_file_list (seldata->data, seldata->length);
      files = files_loader_new ();
      files->filelist = list;
      thumbview_append_thumbnail (tv, files, FALSE);
      files_loader_delete (files);
   }
}


void
thumbview_drag_end_cb (GtkWidget *widget, GdkDragContext *drag_context,
                       gpointer data)
{
   ThumbView *tv = data;

   g_return_if_fail (tv);

   if (conf.dnd_refresh_list_always) {
      /* thumbview_refresh_list (tv); */
      /* to avoid gtk's bug, exec redraw after exit this callback function */
      gtk_idle_add (thumbview_refresh_list_idle, tv);
   }
}


void
thumbview_drag_data_delete_cb (GtkWidget *widget, GdkDragContext *drag_context,
                               gpointer data)
{
   ThumbView *tv = data;

   g_return_if_fail (tv);

   if (!conf.dnd_refresh_list_always)
      thumbview_refresh_list (tv);
}



/******************************************************************************
 *
 *   Compare functions.
 *
 ******************************************************************************/
/* sort by spel */
static int
comp_func_spel (gconstpointer data1, gconstpointer data2)
{
   Thumbnail *thumb1, *thumb2;
   const gchar *filename1, *filename2;
   gboolean ignore_dir;
   gint comp;
   SortItem item;
   SortFlag flags;

   thumb1 = (Thumbnail *) data1;
   thumb2 = (Thumbnail *) data2;

   filename1 = image_info_get_path (thumb1->info);
   filename2 = image_info_get_path (thumb2->info);

   item = thumbwin_get_sort_type (thumb1->thumb_view->thumb_window, &flags);
   ignore_dir = flags & SORT_DIR_INSENSITIVE;

   if (!filename1 || !*filename1)
      return -1;
   else if (!filename2 || !*filename2)
      return 1;

   if (filename1 && !strcmp ("..", g_basename (filename1))) {
      comp = -1;
   } else if (filename2 && !strcmp ("..", g_basename (filename2))) {
      comp = 1;
   } else if (!ignore_dir && isdir (filename1) && !isdir (filename2)) {
      comp = -1;
   } else if (!ignore_dir && !isdir (filename1) && isdir (filename2)) {
      comp = 1;
   } else {
      if (flags & SORT_CASE_INSENSITIVE)
         comp = g_strcasecmp ((gchar *) filename1, (gchar *) filename2);
      else
         comp = strcmp ((gchar *) filename1, (gchar *) filename2);
   }

   return comp;
}


/* sort by file size */
static int
comp_func_size (gconstpointer data1, gconstpointer data2)
{
   Thumbnail *thumb1, *thumb2;
   int retval;

   thumb1 = (Thumbnail *) data1;
   thumb2 = (Thumbnail *) data2;

   if (!thumb1)
      return -1;
   else if (!thumb2)
      return 1;

   retval = thumb1->info->st.st_size - thumb2->info->st.st_size;

   if (retval == 0)
      return comp_func_spel (data1, data2);
   else
      return retval;
}


/* sort by time of las access */
static int
comp_func_atime (gconstpointer data1, gconstpointer data2)
{
   Thumbnail *thumb1, *thumb2;
   int retval;

   thumb1 = (Thumbnail *) data1;
   thumb2 = (Thumbnail *) data2;

   if (!thumb1)
      return -1;
   else if (!thumb2)
      return 1;

   retval = thumb1->info->st.st_atime - thumb2->info->st.st_atime;

   if (retval == 0)
      return comp_func_spel (data1, data2);
   else
      return retval;
}


/* sort by time of last modification */
static int
comp_func_mtime (gconstpointer data1, gconstpointer data2)
{
   Thumbnail *thumb1, *thumb2;
   int retval;

   thumb1 = (Thumbnail *) data1;
   thumb2 = (Thumbnail *) data2;

   if (!thumb1)
      return -1;
   else if (!thumb2)
      return 1;

   retval = thumb1->info->st.st_mtime - thumb2->info->st.st_mtime;

   if (retval == 0)
      return comp_func_spel (data1, data2);
   else
      return retval;
}


/* sort by time of last change */
static int
comp_func_ctime (gconstpointer data1, gconstpointer data2)
{
   Thumbnail *thumb1, *thumb2;
   int retval;

   thumb1 = (Thumbnail *) data1;
   thumb2 = (Thumbnail *) data2;

   if (!thumb1)
      return -1;
   else if (!thumb2)
      return 1;

   retval = thumb1->info->st.st_ctime - thumb2->info->st.st_ctime;

   if (retval == 0)
      return comp_func_spel (data1, data2);
   else
      return retval;
}


/* sort by time of file name extension */
static int
comp_func_type (gconstpointer data1, gconstpointer data2)
{
   Thumbnail *thumb1, *thumb2;
   const gchar *file1, *file2, *ext1, *ext2;
   int retval;

   thumb1 = (Thumbnail *) data1;
   thumb2 = (Thumbnail *) data2;

   if (!thumb1)
      return -1;
   else if (!thumb2)
      return 1;

   file1 = image_info_get_path (thumb1->info);
   file2 = image_info_get_path (thumb2->info);

   if (!file1 || !*file1)
      return -1;
   else if (!file2 || !*file2)
      return 1;

   ext1 = strrchr (file1, '.');
   if (ext1 && *ext1)
      ext1++;
   else
      ext1 = NULL;

   ext2 = strrchr (file2, '.');
   if (ext2 && *ext2)
      ext2++;
   else
      ext2 = NULL;

   if ((!ext1 || !*ext1) && (!ext2 || !*ext2)) {
      return g_strcasecmp (file1, file2);
   } else if (!ext1 || !*ext1) {
      return -1;
   } else if (!ext2 || !*ext2) {
      return 1;
   }

   retval = g_strcasecmp (ext1, ext2);

   if (retval == 0)
      return comp_func_spel (data1, data2);
   else
      return retval;
}


/* sort by image width */
static int
comp_func_width (gconstpointer data1, gconstpointer data2)
{
   Thumbnail *thumb1, *thumb2;
   int retval;

   thumb1 = (Thumbnail *) data1;
   thumb2 = (Thumbnail *) data2;

   if (!thumb1)
      return -1;
   else if (!thumb2)
      return 1;

   retval = thumb1->info->width - thumb2->info->width;

   if (retval == 0)
      return comp_func_spel (data1, data2);
   else
      return retval;
}


/* sort by image height */
static int
comp_func_height (gconstpointer data1, gconstpointer data2)
{
   Thumbnail *thumb1, *thumb2;
   int retval;

   thumb1 = (Thumbnail *) data1;
   thumb2 = (Thumbnail *) data2;

   if (!thumb1)
      return -1;
   else if (!thumb2)
      return 1;

   retval = thumb1->info->height - thumb2->info->height;

   if (retval == 0)
      return comp_func_spel (data1, data2);
   else
      return retval;
}


/* sort by image width */
static int
comp_func_area (gconstpointer data1, gconstpointer data2)
{
   Thumbnail *thumb1, *thumb2;
   int retval, area1, area2;

   thumb1 = (Thumbnail *) data1;
   thumb2 = (Thumbnail *) data2;

   if (!thumb1)
      return -1;
   else if (!thumb2)
      return 1;

   area1 = thumb1->info->width * thumb1->info->height;
   if (thumb1->info->width < 0 && thumb1->info->height < 0)
      area1 = 0 - area1;
   area2 = thumb2->info->width * thumb2->info->height;
   if (thumb2->info->width < 0 && thumb2->info->height < 0)
      area2 = 0 - area2;

   retval = area1 - area2;

   if (retval == 0)
      return comp_func_spel (data1, data2);
   else
      return retval;
}



/******************************************************************************
 *
 *   Other Private Functions.
 *
 ******************************************************************************/
static gint
progress_timeout (gpointer data)
{
   gfloat new_val;
   GtkAdjustment *adj;

   adj = GTK_PROGRESS (data)->adjustment;

   new_val = adj->value + 1;
   if (new_val > adj->upper)
      new_val = adj->lower;

   gtk_progress_set_value (GTK_PROGRESS (data), new_val);

   return (TRUE);
}


static gboolean
thumbview_extract_archive_file (Thumbnail *thumb)
{
   ThumbWindow *tw;
   ThumbView *tv;
   FilesLoader *files;
   guint timer;
   gboolean success;

   g_return_val_if_fail (thumb, FALSE);

   tv = thumbnail_get_parent_thumbview (thumb);
   g_return_val_if_fail (tv, FALSE);

   if (tv->mode != THUMB_VIEW_MODE_ARCHIVE) return FALSE;

   tw = tv->thumb_window;
   g_return_val_if_fail (tw, FALSE);

   files = files_loader_new ();
   tv->status = THUMB_VIEW_STATUS_LOADING;
   fr_archive_ref (FR_ARCHIVE (thumb->info->archive));
   files->archive = thumb->info->archive;

   /* set progress bar */
   gtk_progress_set_activity_mode (GTK_PROGRESS (tw->progressbar), TRUE);
   timer = gtk_timeout_add (50, (GtkFunction)progress_timeout, tw->progressbar);

   /* extract */
   success = image_info_extract_archive (thumb->info);

   /* unset progress bar */
   gtk_timeout_remove (timer);
   gtk_progress_set_activity_mode (GTK_PROGRESS (tw->progressbar), FALSE);
   gtk_progress_bar_update (GTK_PROGRESS_BAR(tw->progressbar), 0.0);

   tv->status = THUMB_VIEW_STATUS_NORMAL;
   files_loader_delete (files);

   return success;
}

static void
thumbview_remove_mode_data (ThumbView *tv)
{
   GList *list;

   g_return_if_fail (tv);

   list = thumbview_plugin_list;
   while (list) {
      ThumbViewPlugin *view = list->data;
      if (view && view->thumbview_data_remove_func) 
         view->thumbview_data_remove_func (tv);      
      list = g_list_next (list);
   }
}


/*
 *  thumbview_add_thumb_data:
 *     @ Store file info to new Thumbnail struct, and add to thumblist GList
 *       in ThumbView struct. 
 *
 *  tv       : Pointer to ThumbView struct.
 *  filelist : file list.
 *  Return   : Pointer to head of added thumbnail list.
 */
static GList *
thumbview_add_thumb_data (ThumbView *tv, GList *filelist)
{
   Thumbnail  *thumb;
   GList *node, *retval = NULL;
   struct stat st;

   if (!tv || !filelist) return NULL;

   node = filelist;
   while (node) {
      gchar *filename = node->data;

      if (!stat(node->data, &st)) {
         ImageInfo *info = image_info_get (filename);
         thumb = thumbnail_new (info);
         image_info_unref (info);
         thumbnail_set_parent_thumbview (thumb, tv);
         tv->thumblist = g_list_append (tv->thumblist, thumb);

         /* update file num info */
         tv->thumb_window->filenum++;
         tv->thumb_window->filesize += st.st_size;
         tv->filenum++;
         tv->filesize += st.st_size;

         if (!retval)
            retval = g_list_find (tv->thumblist, thumb);
      }
      node = g_list_next (node);
   }
   return retval;
}


static GList *
thumbview_add_thumb_data_from_archive (ThumbView *tv, FRArchive *archive)
{
   Thumbnail  *thumb;
   FRCommand *command;
   GList *node, *retval = NULL;

   if (!tv || !archive) return NULL;

   command = archive->command;
   g_return_val_if_fail (command, NULL);
   node = command->file_list;
   while (node) {
      ImageInfo *info = node->data;

      node = g_list_next (node);

      if (!info) continue;

      /* detect filename extension */
      if (conf.detect_filetype_by_ext
          && !gimv_image_detect_type_by_ext (info->filename)
          && !fr_archive_utils_get_file_name_ext (info->filename))
      {
         continue;
      }
      
      thumb = thumbnail_new (info);
      thumbnail_set_parent_thumbview (thumb, tv);
      tv->thumblist = g_list_append (tv->thumblist, thumb);

      /* update file num info */
      tv->thumb_window->filenum++;
      tv->thumb_window->filesize += thumb->info->st.st_size;
      tv->filenum++;
      tv->filesize += thumb->info->st.st_size;

      if (!retval)
         retval = g_list_find (tv->thumblist, thumb);
   }
   return retval;
}


static gchar *
get_uri_list (GList *thumblist)
{
   gchar *path;
   gchar *uri;
   GList *node;
   Thumbnail *thumb;
 
   if (!thumblist) return NULL;

   uri = g_strdup ("");
   node = thumblist;
   while (node) {
      gchar *filename = NULL;

      thumb = node->data;

      filename = image_info_get_path_with_archive (thumb->info);
      path = g_strconcat (uri, "file:", filename, "\r\n", NULL);
      g_free (filename);
      g_free (uri);
      uri = g_strdup (path);
      g_free (path);

      node = g_list_next (node);
   }

   return uri;
}


typedef struct ChangeImageData_Tag
{
   ThumbWindow *tw;
   ImageWindow *iw;
   ImageView   *iv;
   Thumbnail   *thumb;
} ChangeImageData;


static gboolean
change_image_idle (gpointer user_data)
{
   ChangeImageData *data = user_data;

   if (data->iw) {
      imagewin_change_image (data->iw, data->thumb->info);
   } else {
      imageview_change_image (data->iv, data->thumb->info);
   }

   return FALSE;
}


static GList *
next_image (ImageView *iv, gpointer list_owner,
            GList *current, gpointer data)
{
   ImageWindow *iw = data;
   GList *next, *node;
   Thumbnail *thumb;
   ThumbView *tv = list_owner;
   ThumbWindow *tw;
   ChangeImageData *change_data;

   g_return_val_if_fail (iv, NULL);
   g_return_val_if_fail (current, NULL);

   node = g_list_find (ThumbViewList, tv);
   g_return_val_if_fail (node, NULL);

   node = g_list_find (tv->thumblist, current->data);
   g_return_val_if_fail (node, NULL);

   next = g_list_next (current);
   if (!next) next = g_list_first (current);
   g_return_val_if_fail (next, NULL);

   while (next && next != current) {
      Thumbnail *nthumb = next->data;
      GList *temp;

      if (!image_info_is_dir (nthumb->info)
          && !image_info_is_archive (nthumb->info))
      {
         break;
      }

      temp = g_list_next (next);
      if (!temp)
         next = g_list_first (next);
      else
         next = temp;
   }
   if (!next) next = current;

   thumb = next->data;
   g_return_val_if_fail (thumb, NULL);

   tw = tv->thumb_window;
   g_return_val_if_fail (tw, NULL);

   thumbview_set_selection_all (tv, FALSE);
   thumbview_set_selection (thumb, TRUE);
   thumbview_adjust (tv, thumb);

   if (tv->mode == THUMB_VIEW_MODE_ARCHIVE) {
      gboolean success =  thumbview_extract_archive_file (thumb);
      g_return_val_if_fail (success, next);
   }

   change_data = g_new0 (ChangeImageData, 1);
   change_data->tw = tw;
   change_data->iw = iw;
   change_data->iv = iv;
   change_data->thumb = thumb;
   gtk_idle_add_full (GTK_PRIORITY_DEFAULT,
                      change_image_idle,
                      NULL,
                      change_data,
                      (GtkDestroyNotify) g_free);

   if (!iw)
      comment_view_change_file (tw->cv, thumb->info);

   return next;
}


static GList *
prev_image (ImageView *iv, gpointer list_owner,
            GList *current, gpointer data)
{
   ImageWindow *iw = data;
   GList *prev, *node;
   Thumbnail *thumb;
   ThumbView *tv = list_owner;
   ThumbWindow *tw;
   ChangeImageData *change_data;

   g_return_val_if_fail (iv, NULL);
   g_return_val_if_fail (current, NULL);

   node = g_list_find (ThumbViewList, tv);
   g_return_val_if_fail (node, NULL);

   node = g_list_find (tv->thumblist, current->data);
   g_return_val_if_fail (node, NULL);

   prev = g_list_previous (current);
   if (!prev) prev = g_list_last (current);
   g_return_val_if_fail (prev, NULL);

   while (prev && prev != current) {
      Thumbnail *nthumb = prev->data;
      GList *temp;

      if (!image_info_is_dir (nthumb->info)
          && !image_info_is_archive (nthumb->info))
      {
         break;
      }

      temp = g_list_previous (prev);
      if (!temp)
         prev = g_list_last (prev);
      else
         prev = temp;
   }
   if (!prev) prev = current;

   thumb = prev->data;
   g_return_val_if_fail (thumb, NULL);

   tw = tv->thumb_window;
   g_return_val_if_fail (tw, NULL);

   thumbview_set_selection_all (tv, FALSE);
   thumbview_set_selection (thumb, TRUE);
   thumbview_adjust (tv, thumb);

   if (tv->mode == THUMB_VIEW_MODE_ARCHIVE) {
      gboolean success = thumbview_extract_archive_file (thumb);
      g_return_val_if_fail (success, prev);
   }

   change_data = g_new0 (ChangeImageData, 1);
   change_data->tw = tw;
   change_data->iw = iw;
   change_data->iv = iv;
   change_data->thumb = thumb;
   gtk_idle_add_full (GTK_PRIORITY_DEFAULT,
                      change_image_idle,
                      NULL,
                      change_data,
                      (GtkDestroyNotify) g_free);

   if (!iw)
      comment_view_change_file (tw->cv, thumb->info);

   return prev;
}


static GList *
nth_image (ImageView *iv,
           gpointer list_owner,
           GList *current,
           guint nth,
           gpointer data)
{
   ImageWindow *iw = data;
   GList *node;
   Thumbnail *thumb;
   ThumbView *tv = list_owner;
   ThumbWindow *tw;
   ChangeImageData *change_data;

   /* check check chek ... */
   g_return_val_if_fail (iv, NULL);
   g_return_val_if_fail (current, NULL);

   node = g_list_find (ThumbViewList, tv);
   g_return_val_if_fail (node, NULL);

   node = g_list_find (tv->thumblist, current->data);
   g_return_val_if_fail (node, NULL);

   tw = tv->thumb_window;
   g_return_val_if_fail (tw, NULL);

   /* get nth of list */
   node = g_list_nth (tv->thumblist, nth);
   g_return_val_if_fail (node, NULL);

   thumb = node->data;
   g_return_val_if_fail (thumb, NULL);

   thumbview_set_selection_all (tv, FALSE);
   thumbview_set_selection (thumb, TRUE);
   thumbview_adjust (tv, thumb);

   /* show image */
   if (tv->mode == THUMB_VIEW_MODE_ARCHIVE) {
      gboolean success = thumbview_extract_archive_file (thumb);
      g_return_val_if_fail (success, node);
   }

   change_data = g_new0 (ChangeImageData, 1);
   change_data->tw = tw;
   change_data->iw = iw;
   change_data->iv = iv;
   change_data->thumb = thumb;
   gtk_idle_add_full (GTK_PRIORITY_DEFAULT,
                      change_image_idle,
                      NULL,
                      change_data,
                      (GtkDestroyNotify) g_free);

   if (!iw)
      comment_view_change_file (tw->cv, thumb->info);

   return node;
}


static void
cb_imageview_thumbnail_created (ImageView *iv, ImageInfo *info, ThumbView *tv)
{
   GList *node;
   Thumbnail *thumb = NULL;

   g_return_if_fail (IS_IMAGEVIEW (iv));
   g_return_if_fail (info);
   g_return_if_fail (tv);

   node = tv->thumblist;
   while (node) {
      thumb = node->data;
      node = g_list_next (node);

      if (image_info_is_same (thumb->info, info))
         break;
      else
         thumb = NULL;
   }

   if (thumb)
      thumbview_refresh_thumbnail (thumb, LOAD_CACHE);
}


static void
remove_list (ImageView *iv, gpointer list_owner, gpointer data)
{
   ThumbView *tv = list_owner;
   GList *node;

   g_return_if_fail (iv);
   g_return_if_fail (tv);

   gtk_signal_disconnect_by_func (GTK_OBJECT (iv),
                                  GTK_SIGNAL_FUNC (cb_imageview_thumbnail_created),
                                  tv);

   node = g_list_find (ThumbViewList, tv);
   if (!node) return;

   tv->related_image_view = g_list_remove (tv->related_image_view, iv);
}


static void
thumbview_button_action (ThumbView *tv,
                         Thumbnail *thumb,
                         GdkEventButton *event,
                         gint num)
{
   ThumbWindow *tw;

   g_return_if_fail (tv);

   tw = tv->thumb_window;
   g_return_if_fail (tw);

   if (image_info_is_dir (thumb->info)
       || image_info_is_archive (thumb->info))
   {
      switch (abs (num)) {
      case ACTION_OPEN_AUTO:
         num = ACTION_OPEN_AUTO_FORCE;
         break;
      case ACTION_OPEN_IN_PREVIEW:
         num = ACTION_OPEN_IN_PREVIEW_FORCE;
         break;
      case ACTION_OPEN_IN_SHARED_WIN:
         num = ACTION_OPEN_IN_SHARED_WIN_FORCE;
         break;
      default:
         break;
      }
   }

   switch (abs (num)) {
   case ACTION_POPUP:
      thumbview_popup_menu (tv, thumb, event);

      /* FIXME */
      if (GIMV_IS_SCROLLED (GTK_BIN (tv->container)->child))
         gimv_scrolled_stop_auto_scroll (GIMV_SCROLLED (GTK_BIN (tv->container)->child));

      break;

   case ACTION_OPEN_AUTO:
      if (tw->show_preview || imagewin_get_shared_window())
         thumbview_open_image (tv, thumb, THUMB_VIEW_OPEN_IMAGE_AUTO);
      break;

   case ACTION_OPEN_AUTO_FORCE:
      thumbview_open_image (tv, thumb, THUMB_VIEW_OPEN_IMAGE_AUTO);
      break;

   case ACTION_OPEN_IN_PREVIEW:
      if (tw->show_preview)
         thumbview_open_image (tv, thumb, THUMB_VIEW_OPEN_IMAGE_PREVIEW);
      break;

   case ACTION_OPEN_IN_PREVIEW_FORCE:
      if (!tw->show_preview)
         thumbwin_open_preview (tw);
      thumbview_open_image (tv, thumb, THUMB_VIEW_OPEN_IMAGE_PREVIEW);
      break;

   case ACTION_OPEN_IN_NEW_WIN:
      thumbview_open_image (tv, thumb, THUMB_VIEW_OPEN_IMAGE_NEW_WIN);
      break;

   case ACTION_OPEN_IN_SHARED_WIN:
      if (imagewin_get_shared_window())
         thumbview_open_image (tv, thumb, THUMB_VIEW_OPEN_IMAGE_SHARED_WIN);
      break;

   case ACTION_OPEN_IN_SHARED_WIN_FORCE:
      thumbview_open_image (tv, thumb, THUMB_VIEW_OPEN_IMAGE_SHARED_WIN);
      break;

   default:
      break;
   }
}


static GtkWidget *
create_progs_submenu (ThumbView *tv)
{
   GtkWidget *menu;
   GtkWidget *menu_item;
   gint i, conf_num = sizeof (conf.progs) / sizeof (conf.progs[0]);
   gchar **pair;

   menu = gtk_menu_new();

   /* count items num */
   for (i = 0; i < conf_num; i++) {
      if (!conf.progs[i]) continue;

      pair = g_strsplit (conf.progs[i], ",", 3);

      if (pair[0] && pair[1]) {
         gchar *label;

         if (pair[2] && !strcasecmp (pair[2], "TRUE"))
            label = g_strconcat (pair[0], "...", NULL);
         else
            label = g_strdup (pair[0]);

         menu_item = gtk_menu_item_new_with_label (label);
         gtk_object_set_data (GTK_OBJECT (menu_item), "num", GINT_TO_POINTER (i));
         gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                             GTK_SIGNAL_FUNC (cb_open_image_by_external), tv);
         gtk_menu_append (GTK_MENU (menu), menu_item);
         gtk_widget_show (menu_item);

         g_free (label);
      }

      g_strfreev (pair);
   }

   return menu;
}


void
thumbview_open_image (ThumbView *tv, Thumbnail *thumb, gint type)
{
   ThumbWindow *tw;
   ImageWindow *iw = NULL;
   ImageView   *iv = NULL;
   GList *current, *node;
   const gchar *image_name, *ext;
   gchar *filename, *tmpstr;

   g_return_if_fail (tv && thumb);

   tw = tv->thumb_window;
   g_return_if_fail (tw);

   image_name = image_info_get_path (thumb->info);
   g_return_if_fail (image_name && *image_name);
   filename = g_strdup (image_name);

   if (!strcmp ("..", g_basename (filename))) {
      tmpstr = filename;
      filename = g_dirname (filename);
      g_free (tmpstr);
      if (filename) {
         tmpstr = filename;
         filename = g_dirname (filename);
         g_free (tmpstr);
      }
      tmpstr = NULL;
   }

   /* open directory */
   if (isdir (filename)) {
      open_dir_images (filename, tw, LOAD_CACHE, conf.scan_dir_recursive);
      g_free (filename);
      return;
   }

   /* extract image in archive */
   if (tv->mode == THUMB_VIEW_MODE_ARCHIVE) {
      gboolean success = thumbview_extract_archive_file (thumb);
      if (!success) {
         g_free (filename);
         return;
      }
   }

   /* open archive */
   ext = fr_archive_utils_get_file_name_ext (filename);
   if (ext) {
      open_archive_images (filename, tw, LOAD_CACHE);
      g_free (filename);
      return;
   }

   /* open the image */
   if ((type == THUMB_VIEW_OPEN_IMAGE_AUTO && tw->show_preview)
       || type == THUMB_VIEW_OPEN_IMAGE_PREVIEW)
   {
      iv = tw->iv;
      comment_view_change_file (tw->cv, thumb->info);
      imageview_change_image (iv, thumb->info);

   } else if (imagewin_get_shared_window() &&
              ((type == THUMB_VIEW_OPEN_IMAGE_AUTO && !conf.imgwin_open_new_win)
               ||type == THUMB_VIEW_OPEN_IMAGE_SHARED_WIN))
   {
      iw = imagewin_open_shared_window (thumb->info);
      if (iw)
         iv = iw->iv;

   } else {
      if (type == THUMB_VIEW_OPEN_IMAGE_SHARED_WIN) {
         iw = imagewin_open_shared_window (thumb->info);
      } else if (type == THUMB_VIEW_OPEN_IMAGE_AUTO) {
         iw = imagewin_open_window_auto (thumb->info);
      } else {
         iw = imagewin_open_window (thumb->info);
      }
      if (iw)
         iv = iw->iv;
   }

   if (iv && g_list_find (imageview_get_list(), iv)) {
      current = g_list_find (tv->thumblist, thumb);
      imageview_set_list (iv, tv->thumblist, current,
                          (gpointer) tv,
                          next_image,
                          prev_image,
                          nth_image,
                          remove_list,
                          iw);
      gtk_signal_connect (GTK_OBJECT (iv), "thumbnail_created",
                          GTK_SIGNAL_FUNC (cb_imageview_thumbnail_created), tv);
      node = g_list_find (tv->related_image_view, iv);
      if (!node) {
         gint num;
         tv->related_image_view = g_list_append (tv->related_image_view, iv);
         num = g_list_length (tv->related_image_view);
      }
   }

   g_free (filename);
}


void
thumbview_popup_menu (ThumbView *tv, Thumbnail *thumb, GdkEventButton *event)
{
   GtkWidget *popup_menu, *progs_submenu, *scripts_submenu;
   GList *thumblist = NULL, *node;
   guint n_menu_items;
   GtkItemFactory *ifactory;
   GtkWidget *menuitem;
   gchar *dirname;
   guint button;
   guint32 time;
   GtkMenuPositionFunc pos_fn = NULL;

   g_return_if_fail (tv);

   if (event) {
      button = event->button;
      time = gdk_event_get_time ((GdkEvent *) event);
   } else {
      button = 0;
      time = GDK_CURRENT_TIME;
      pos_fn = menu_calc_popup_position;
   }

   if (tv->popup_menu) {
      gtk_widget_unref (tv->popup_menu);
      tv->popup_menu = NULL;
   }

   thumblist = node = thumbview_get_selection_list (tv);
   if (!thumblist) return;

   if (!thumb) thumb = thumblist->data;
   g_return_if_fail (thumb);

   /* create popup menu */
   n_menu_items = sizeof(thumb_button_popup_items)
      / sizeof(thumb_button_popup_items[0]) - 1;
   popup_menu = menu_create_items(tv->thumb_window->window,
                                  thumb_button_popup_items, n_menu_items,
                                  "<ThumbnailButtonPop>", tv);

   /* set sensitive */
   ifactory = gtk_item_factory_from_widget (popup_menu);

   menuitem = gtk_item_factory_get_item (ifactory, "/Open");
   if (!image_info_is_dir (thumb->info)
       && !image_info_is_archive (thumb->info))
   {
      gtk_widget_hide (menuitem);
   }

   menuitem = gtk_item_factory_get_item (ifactory, "/Open in New Window");
   if (image_info_is_dir (thumb->info)
       || image_info_is_archive (thumb->info))
   {
      gtk_widget_hide (menuitem);
   }

   menuitem = gtk_item_factory_get_item (ifactory, "/Open in Shared Window");
   if (image_info_is_dir (thumb->info)
       || image_info_is_archive (thumb->info))
   {
      gtk_widget_hide (menuitem);
   } else if (g_list_length (thumblist) > 1) {
      gtk_widget_set_sensitive (menuitem, FALSE);
   }

   menuitem = gtk_item_factory_get_item (ifactory, "/Open in External Program");
   if (/* thumbnail_is_dir (thumb) || thumbnail_is_archive (thumb) */
      image_info_is_in_archive (thumb->info))
   {
      gtk_widget_hide (menuitem);
   } else {
      progs_submenu = create_progs_submenu (tv);
      menu_set_submenu (popup_menu, "/Open in External Program", progs_submenu);
   }

   menuitem = gtk_item_factory_get_item (ifactory, "/Scripts");
   if (/* thumbnail_is_dir (thumb) || thumbnail_is_archive (thumb) */
      image_info_is_in_archive (thumb->info))
   {
      gtk_widget_hide (menuitem);
   } else {
      scripts_submenu = create_scripts_submenu (tv);
      menu_set_submenu (popup_menu, "/Scripts", scripts_submenu);
   }

#warning FIXME!!
   menuitem = gtk_item_factory_get_item (ifactory, "/Update Thumbnail");
   if (image_info_is_dir (thumb->info)
       || image_info_is_archive (thumb->info)
       || image_info_is_movie (thumb->info))
   {
      gtk_widget_set_sensitive (menuitem, FALSE);
   }

   menuitem = gtk_item_factory_get_item (ifactory, "/Remove from List");
   if (tv->progress || tv->mode != THUMB_VIEW_MODE_COLLECTION) {
      gtk_widget_set_sensitive (menuitem, FALSE);
   }

   dirname = g_dirname (image_info_get_path (thumb->info));
   if (g_list_length (thumblist) < 1 || !iswritable (dirname)
       || tv->mode == THUMB_VIEW_MODE_ARCHIVE)
   {
      if (g_list_length (thumblist) < 1) {
         menuitem = gtk_item_factory_get_item (ifactory, "/Property...");
         gtk_widget_set_sensitive (menuitem, FALSE);
      }
      menuitem = gtk_item_factory_get_item (ifactory, "/Move Files To...");
      gtk_widget_set_sensitive (menuitem, FALSE);
      menuitem = gtk_item_factory_get_item (ifactory, "/Rename...");
      gtk_widget_set_sensitive (menuitem, FALSE);
      menuitem = gtk_item_factory_get_item (ifactory, "/Remove file...");
      gtk_widget_set_sensitive (menuitem, FALSE);
   }
   g_free (dirname);

   if (tv->mode == THUMB_VIEW_MODE_ARCHIVE) {
      menuitem = gtk_item_factory_get_item (ifactory, "/Copy Files To...");
      gtk_widget_set_sensitive (menuitem, FALSE);
      menuitem = gtk_item_factory_get_item (ifactory, "/Link Files To...");
      gtk_widget_set_sensitive (menuitem, FALSE);
   }

   if (g_list_length (thumblist) > 1) {
      menuitem = gtk_item_factory_get_item (ifactory, "/Property...");
      gtk_widget_set_sensitive (menuitem, FALSE);
      menuitem = gtk_item_factory_get_item (ifactory, "/Rename...");
      gtk_widget_set_sensitive (menuitem, FALSE);
   }

#ifdef ENABLE_EXIF
   {
      const gchar *img_name = image_info_get_path (thumb->info);
      const gchar *format = gimv_image_detect_type_by_ext (img_name);
      menuitem = gtk_item_factory_get_item (ifactory, "/Scan EXIF Data...");
#warning FIXME!!
      if (!format || !*format || g_strcasecmp(format, "image/jpeg")) {
         gtk_widget_hide (menuitem);
      }
      if (g_list_length (thumblist) > 1
          || tv->mode == THUMB_VIEW_MODE_ARCHIVE)
      {
         gtk_widget_set_sensitive (menuitem, FALSE);
      }
   }
#endif /* ENABLE_EXIF */

   /* popup */
   gtk_menu_popup(GTK_MENU(popup_menu), NULL, NULL,
                  NULL, NULL, button, time);

   tv->popup_menu = popup_menu;
#ifdef USE_GTK2
   gtk_object_ref (GTK_OBJECT (tv->popup_menu));
   gtk_object_sink (GTK_OBJECT (tv->popup_menu));
#endif

   g_list_free (thumblist);
}


void
thumbview_file_operate (ThumbView *tv, FileOperateType type)
{
   GList *list = NULL;
   gboolean success = FALSE;

   g_return_if_fail (tv);

   /* FIXME!! */
   /* not implemented yet */
   if (tv->mode == THUMB_VIEW_MODE_ARCHIVE) return;

   list = thumbview_get_selected_file_list (tv);
   if (!list) return;

   if (!previous_dir && tv->mode == THUMB_VIEW_MODE_DIR) {
      previous_dir = g_strdup (tv->dirname);
   }

   success = files2dir_with_dialog (list, &previous_dir, type,
                                    GTK_WINDOW (tv->thumb_window->window));

   if (success && type == FILE_MOVE) {
      thumbview_refresh_list (tv);
   }

   /* update dest side file list */
   tv = thumbview_find_opened_dir (previous_dir);
   if (tv) {
      thumbview_refresh_list (tv);
   }

   g_list_foreach (list, (GFunc) g_free, NULL);
   g_list_free (list);
}


void
thumbview_rename_file (ThumbView *tv)
{
   Thumbnail *thumb;
   gchar *cache_type;
   GList *thumblist;
   const gchar *src_file;
   gchar *dest_file, *dest_path;
   gchar *src_cache_path, *dest_cache_path;
   gchar *src_comment, *dest_comment;
   gchar message[BUF_SIZE], *dirname;
   ConfirmType confirm;
   gboolean exist;
   struct stat dest_st;

   g_return_if_fail (tv);

   /* FIXME!! */
   /* not implemented yet */
   if (tv->mode == THUMB_VIEW_MODE_ARCHIVE) return;

   thumblist = thumbview_get_selection_list (tv);
   if (!thumblist || g_list_length (thumblist) > 1) return;

   thumb = thumblist->data;
   if (!thumb) return;

   {   /********** convert charset **********/
      gchar *tmpstr, *src_file_internal;

      src_file = g_basename(image_info_get_path (thumb->info));
      src_file_internal = charset_to_internal (src_file,
                                               conf.charset_filename,
                                               conf.charset_auto_detect_fn,
                                               conf.charset_filename_mode);

      tmpstr = gtkutil_popup_textentry (_("Rename a file"),
                                        _("New file name: "),
                                        src_file_internal, NULL, -1, 0,
                                        GTK_WINDOW (tv->thumb_window->window));
      g_free (src_file_internal);
      src_file_internal = NULL;

      if (!tmpstr) return;

      dest_file = charset_internal_to_locale (tmpstr);
      g_free (tmpstr);
   }

   if (!strcmp (src_file, dest_file)) goto ERROR0;

   dirname = g_dirname (image_info_get_path (thumb->info));
   dest_path = g_strconcat (dirname, "/", g_basename (dest_file), NULL);
   g_free (dirname);
   exist = !lstat(dest_path, &dest_st);
   if (exist) {
      {   /********** convert charset **********/
         gchar *tmpstr;

         tmpstr = charset_to_internal (src_file,
                                       conf.charset_filename,
                                       conf.charset_auto_detect_fn,
                                       conf.charset_filename_mode);

         g_snprintf (message, BUF_SIZE,
                     _("File exist : %s\n\n"
                       "Overwrite?"), tmpstr);

         g_free (tmpstr);
      }

      confirm = gtkutil_confirm_dialog (_("File exist!!"), message, 0,
                                        GTK_WINDOW (tv->thumb_window->window));
      if (confirm == CONFIRM_NO) goto ERROR1;
   }

   /* rename file!! */
   if (rename (image_info_get_path (thumb->info), dest_path) < 0) {
      {   /********** convert charset **********/
         gchar *tmpstr;

         tmpstr = charset_to_internal (image_info_get_path (thumb->info),
                                       conf.charset_filename,
                                       conf.charset_auto_detect_fn,
                                       conf.charset_filename_mode);

         g_snprintf (message, BUF_SIZE,
                     _("Faild to rename file :\n%s"),
                     tmpstr);

         g_free (tmpstr);
      }

      gtkutil_message_dialog (_("Error!!"), message,
                              GTK_WINDOW (tv->thumb_window->window));
   }

   /* rename cache */
   cache_type = thumbnail_get_cache_type (thumb);
   if (cache_type > 0) {
      src_cache_path 
         = thumbsupport_get_thumb_cache_path (image_info_get_path (thumb->info),
                                              cache_type);
      dest_cache_path
         = thumbsupport_get_thumb_cache_path (dest_path, cache_type);
      if (rename (src_cache_path, dest_cache_path) < 0)
         g_print (_("Faild to rename cache file :%s\n"), dest_path);
      g_free (src_cache_path);
      g_free (dest_cache_path);
   }

   /* rename comment */
   src_comment = comment_find_file (image_info_get_path (thumb->info));
   if (src_comment) {
      dest_comment = comment_get_path (dest_path);
      if (rename (src_comment, dest_comment) < 0)
         g_print (_("Faild to rename comment file :%s\n"), dest_comment);
      g_free (src_comment);
      g_free (dest_comment);      
   }

   thumbview_refresh_list (tv);

ERROR1:
   g_free (dest_path);
ERROR0:
   g_free (dest_file);
}


gboolean
thumbview_delete_files (ThumbView *tv)
{
   ThumbWindow *tw;
   GList *selection;
   GList *filelist = NULL, *list;
   gboolean retval;

   g_return_val_if_fail (tv, FALSE);

   /* FIXME!! */
   /* not implemented yet */
   if (tv->mode == THUMB_VIEW_MODE_ARCHIVE) return FALSE;

   tw = tv->thumb_window;
   g_return_val_if_fail (tw, FALSE);

   selection = thumbview_get_selection_list (tv);
   if (!selection) return FALSE;

   /* convert to filename list */
   for (list = selection; list; list = g_list_next (list)) {
      Thumbnail *thumb = list->data;
      const gchar *filename;

      if (!thumb) continue;
      filename = image_info_get_path (thumb->info);
      if (filename && *filename)
         filelist = g_list_append (filelist, (gpointer) filename);
   }

   /* do delete */
   retval = delete_files (filelist, CONFIRM_ASK,
                          GTK_WINDOW (tv->thumb_window->window));

   if (retval) {
      thumbview_refresh_list (tv);
      if (tw->show_dirview)
         dirview_refresh_list (tw->dv);
   }

   g_list_free (filelist);
   g_list_free (selection);

   return retval;
}


static GtkWidget *
create_scripts_submenu (ThumbView *tv)
{
   GtkWidget *menu;
   GtkWidget *menu_item;
   GList *tmplist = NULL, *filelist = NULL, *list;
   const gchar *dirlist;
   gchar **dirs;
   gint i, flags;

   menu = gtk_menu_new();

   if (conf.scripts_use_default_search_dir_list)
      dirlist = SCRIPTS_DEFAULT_SEARCH_DIR_LIST;
   else
      dirlist = conf.scripts_search_dir_list;

   if (!dirlist || !*dirlist) return NULL;

   dirs = g_strsplit (dirlist, ",", -1);
   if (!dirs) return NULL;

   flags = 0 | GETDIR_FOLLOW_SYMLINK;
   for (i = 0; dirs[i]; i++) {
      if (!*dirs || !isdir (dirs[i])) continue;
      get_dir (dirs[i], flags, &tmplist, NULL);
      filelist = g_list_concat (filelist, tmplist);
   }
   g_strfreev (dirs);

   for (list = filelist; list; list = g_list_next (list)) {
      gchar *filename = list->data;
      gchar *label;

      if (!filename || !*filename || !isexecutable(filename)) continue;

      if (conf.scripts_show_dialog)
         label = g_strconcat (g_basename (filename), "...", NULL);
      else
         label = g_strdup (g_basename (filename));

      menu_item = gtk_menu_item_new_with_label (label);
      gtk_object_set_data_full (GTK_OBJECT (menu_item),
                                "script",
                                g_strdup (filename),
                                (GtkDestroyNotify) g_free);
      gtk_signal_connect (GTK_OBJECT (menu_item),
                          "activate",
                          GTK_SIGNAL_FUNC (cb_open_image_by_script),
                          tv);
      gtk_menu_append (GTK_MENU (menu), menu_item);
      gtk_widget_show (menu_item);

      g_free (label);
   }

   g_list_foreach (filelist, (GFunc) g_free, NULL);
   g_list_free (filelist);

   return menu;
}


static void
thumbview_reset_load_priority (ThumbView *tv)
{
   ThumbViewPlugin *view;
   GList *node, *tmp_list = NULL;
   gboolean in_view;

   g_return_if_fail (tv);

   if (!tv->load_list) return;
   node = g_list_last (tv->load_list);

   view = g_hash_table_lookup (thumbview_modes, tv->disp_mode);
   g_return_if_fail (view);
   if (!view->thumbnail_is_in_view_func) return;

   while (node) {
      Thumbnail *thumb = node->data;
      node = g_list_previous (node);

      in_view = view->thumbnail_is_in_view_func (tv, thumb);
      if (in_view) {
         tv->load_list = g_list_remove (tv->load_list, thumb);
         tmp_list = g_list_prepend (tmp_list, thumb);
      }
   }

   if (tmp_list)
      tv->load_list = g_list_concat (tmp_list, tv->load_list);
}


static void
thumbview_set_scrollbar_callback (ThumbView *tv)
{
   GtkAdjustment *hadj, *vadj;

   g_return_if_fail (tv);

   hadj = gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW (tv->container));
   vadj = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (tv->container));

   gtk_signal_connect (GTK_OBJECT (hadj),
                       "value_changed",
                       GTK_SIGNAL_FUNC (cb_thumbview_scrollbar_value_changed),
                       tv);
   gtk_signal_connect (GTK_OBJECT (vadj),
                       "value_changed",
                       GTK_SIGNAL_FUNC (cb_thumbview_scrollbar_value_changed),
                       tv);
}


static void
thumbview_remove_scrollbar_callback (ThumbView *tv)
{
   GtkAdjustment *hadj, *vadj;

   g_return_if_fail (tv);

   hadj = gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW (tv->container));
   vadj = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (tv->container));

   gtk_signal_disconnect_by_func (GTK_OBJECT (hadj),
                                  GTK_SIGNAL_FUNC (cb_thumbview_scrollbar_value_changed),
                                  tv);
   gtk_signal_disconnect_by_func (GTK_OBJECT (vadj),
                                  GTK_SIGNAL_FUNC (cb_thumbview_scrollbar_value_changed),
                                  tv);
}



/******************************************************************************
 *
 *   Public Functions.
 *
 ******************************************************************************/
GList *
thumbview_get_list (void)
{
   return ThumbViewList;
}


static void
thumbview_destroy_similar_win_relation (SimilarWin *sw,ThumbView *tv)
{
   g_return_if_fail (sw);
   g_return_if_fail (tv);

   similar_win_unset_relation (sw);
   gtk_signal_disconnect_by_func (GTK_OBJECT (sw),
                                  GTK_SIGNAL_FUNC (cb_similar_win_destroy),
                                  tv);
}


const gchar *
thumbview_get_path (ThumbView *tv)
{
   g_return_val_if_fail (tv, NULL);

   if (tv->mode == THUMB_VIEW_MODE_DIR) {
      return tv->dirname;
   } else if (tv->mode == THUMB_VIEW_MODE_ARCHIVE) {
      return tv->archive->filename;
   }

   return NULL;
}


/*
 *  thumbview_delete:
 *     @ Free ThumbView struct.
 *
 *  tv : Pointer to ThumbView struct to free.
 */
void
thumbview_delete (ThumbView *tv)
{
   ThumbWindow *tw;
   GList *node, *node_obs;

   if (!tv) return;

   thumbview_remove_scrollbar_callback (tv);

   tw = tv->thumb_window;
   if (tw) {
      tw->filenum -= tv->filenum;
      tw->filesize -= tv->filesize;
   }

   if (tv->mode == THUMB_VIEW_MODE_DIR && tv->dirname) {
      dirview_unset_opened_mark (tw->dv, tv->dirname);
   }

   ThumbViewList = g_list_remove (ThumbViewList, tv);

   if (GTK_BIN (tv->container)->child)
      gtk_widget_destroy (GTK_BIN (tv->container)->child);   

   if (tv->progress && tv->progress->status != WINDOW_DESTROYED) {
      tv->progress->status = CONTAINER_DESTROYED;
   }

   /* destroy relation */
   node = tv->related_image_view;
   while (node) {
      ImageView *iv = node->data;

      node = g_list_next (node);
      imageview_remove_list (iv, (gpointer) tv);
   }
   tv->related_image_view = NULL;

   g_list_foreach (tv->related_similar_win,
                   (GFunc) thumbview_destroy_similar_win_relation,
                   tv);
   g_list_free (tv->related_similar_win);
   tv->related_similar_win = NULL;

   node = g_list_first (tv->thumblist);

   while (node) {
      Thumbnail *thumb = node->data;

      thumbnail_unref (thumb);
      node->data = NULL;

      node_obs = node;
      node = g_list_next (node_obs);

      tv->thumblist = g_list_remove (tv->thumblist, node_obs);
   }

   if (tv->mode == THUMB_VIEW_MODE_ARCHIVE && tv->archive) {
      fr_archive_unref (FR_ARCHIVE (tv->archive));
      tv->archive = NULL;
   }

   if (tv->popup_menu) {
      gtk_widget_unref (tv->popup_menu);
      tv->popup_menu = NULL;
   }

   thumbview_remove_mode_data (tv);
   g_hash_table_destroy (tv->disp_mode_data);
   g_list_free (tv->thumblist);

   g_free (tv->dirname);
   g_free (tv->tabtitle);
   g_free (tv);
}


/*
 *  thumbview_find_opened_dir:
 *
 *  path   : directory name for find.
 *  Return : Pointer to ThumbView struct. If not found, return NULL.
 */
ThumbView *
thumbview_find_opened_dir (const gchar *path)
{
   ThumbView *tv;
   GList *node;

   if (!path) return NULL;

   node = g_list_first (ThumbViewList);
   while (node) {
      tv = node->data;
      if (tv->mode == THUMB_VIEW_MODE_DIR && !strcmp (path, tv->dirname)) {
         return tv;
      }
      node = g_list_next (node);
   }

   return NULL;
}


ThumbView *
thumbview_find_opened_archive (const gchar *path)
{
   ThumbView *tv;
   GList *node;

   if (!path) return NULL;

   node = g_list_first (ThumbViewList);
   while (node) {
      tv = node->data;
      if (tv->mode == THUMB_VIEW_MODE_ARCHIVE
          && tv->archive
          && !strcmp (path, tv->archive->filename))
      {
         return tv;
      }
      node = g_list_next (node);
   }

   return NULL;
}


/*
 *  thumbview_sort_data:
 *     @ sort thumbnail list.
 *
 *  tv : Pointer to ThumbView struct to sort thumbnails.
 */
void
thumbview_sort_data (ThumbView *tv)
{
   SortItem item;
   SortFlag flags;

   if (!tv) return;

   item = thumbwin_get_sort_type (tv->thumb_window, &flags);

   /* sort thumbnail */
   if (item == SORT_NAME)
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_spel);
   else if (item == SORT_SIZE)
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_size);
   else if (item == SORT_ATIME)
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_atime);
   else if (item == SORT_MTIME)
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_mtime);
   else if (item == SORT_CTIME)
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_ctime);
   else if (item == SORT_TYPE)
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_type);
   else if (item == SORT_WIDTH)
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_width);
   else if (item == SORT_HEIGHT)
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_height);
   else if (item == SORT_AREA)
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_area);

   if (flags & SORT_REVERSE)
      tv->thumblist = g_list_reverse (tv->thumblist);
}


/* #define ENABLE_THUMB_LOADER_TIMER 0 */
/*
 *  thumbview_load_thumbnails:
 *    @ Load specified thumbnails and draw to the thumbnail container.
 *
 *  tv     : Pointer to the ThumbView struct.
 *  start  : The thumbnail list.
 *  Return : TRUE if success.
 */
gboolean
thumbview_load_thumbnails (ThumbView *tv, GList *loadlist, gchar *dest_mode)
{
#ifdef ENABLE_THUMB_LOADER_TIMER
   GTimer *timer;
   gdouble etime;
   gulong mtime;
#endif /* ENABLE_THUMB_LOADER_TIMER */
   ThumbViewPlugin *view;
   FilesLoader *files = tv->progress;
   gint pos;

   g_return_val_if_fail (tv && loadlist, FALSE);
   g_return_val_if_fail (thumbview_plugin_list, FALSE);

#ifdef ENABLE_THUMB_LOADER_TIMER
   timer = g_timer_new ();
   g_timer_start (timer);
   g_print ("timer started\n");
#endif /* ENABLE_THUMB_LOADER_TIMER */

   view = g_hash_table_lookup (thumbview_modes, dest_mode);
   g_return_val_if_fail (view, FALSE);

   /* load thumbnail!! */
   tv->load_list = g_list_copy (loadlist);
   thumbview_reset_load_priority (tv);

   tv->progress->window = tv->thumb_window->window;
   tv->progress->progressbar = tv->thumb_window->progressbar;
   tv->progress->num = g_list_length (loadlist);
   tv->progress->pos = 0;
   thumbwin_set_statusbar_page_info (tv->thumb_window,
                                     THUMBWIN_CURRENT_PAGE);

   for (pos = 1; tv->load_list && files->status < CANCEL; pos++) {
      Thumbnail *thumb;

      if (tv->progress->pos % conf.thumbwin_redraw_interval == 0)
         while (gtk_events_pending()) gtk_main_iteration();

      thumb = tv->load_list->data;

      thumbnail_create (thumb, tv->ThumbnailSize, files->thumb_load_type);

      if (view->add_thumb_func)
         view->add_thumb_func (thumb, dest_mode, files->thumb_load_type);

      if(files->status < 0) {
         ThumbViewList = g_list_remove (ThumbViewList, tv);
         thumbview_delete (tv);
         break;
      } else {
         /* update progress info */
         tv->progress->now_file = image_info_get_path (thumb->info);
         tv->progress->pos = pos;
         thumbwin_loading_update_progress (tv->thumb_window,
                                           THUMBWIN_CURRENT_PAGE);
         /*
           thumbwin_set_statusbar_page_info (tv->thumb_window,
           THUMBWIN_CURRENT_PAGE);
         */
      }

      tv->load_list = g_list_remove (tv->load_list, thumb);
   }

   if (tv->load_list) {
      g_list_free (tv->load_list);
      tv->load_list = NULL;
   }

   gtk_progress_set_show_text(GTK_PROGRESS(files->progressbar), FALSE);
   gtk_progress_bar_update (GTK_PROGRESS_BAR(files->progressbar), 0.0);

#ifdef ENABLE_THUMB_LOADER_TIMER
   g_timer_stop (timer);
   g_print ("timer stopped\n");
   etime = g_timer_elapsed (timer, &mtime);
   g_print ("elapsed time = %f sec\n", etime);
   g_timer_destroy (timer);
#endif /* ENABLE_THUMB_LOADER_TIMER */

   return TRUE;
}


/*
 *  thumbview_append_thumbnail:
 *     @ Store file info to new Thumbnail struct, and add to thumblist GList
 *       in ThumbView struct. 
 *
 *  tv     : Pointer to ThumbView struct for open thumbnails.
 *  fieles : Pointer to FilesLoader struct.
 *  force  : Force append thumbnail unless any mode.
 *  Return : Retrun TRUE if success.
 */
gboolean
thumbview_append_thumbnail (ThumbView *tv, FilesLoader *files, gboolean force)
{
   ThumbViewPlugin *view;
   GList *start_pos, *loadlist;

   g_return_val_if_fail (tv && files, FALSE);

   view = g_hash_table_lookup (thumbview_modes, tv->disp_mode);
   g_return_val_if_fail (view, FALSE);

   if (!tv || !files || (tv->mode == THUMB_VIEW_MODE_DIR && !force)) {
      return FALSE;
   }

   tv->progress = files;

   start_pos = thumbview_add_thumb_data (tv, files->filelist);

   loadlist = view->add_thumb_frames_func (tv, start_pos, tv->disp_mode);

   if (loadlist) {
      thumbview_load_thumbnails (tv, loadlist, tv->disp_mode);
      g_list_free (loadlist);
   }

   tv->progress = NULL;

   if (files->status >= 0)
      thumbwin_set_statusbar_page_info (tv->thumb_window, THUMBWIN_CURRENT_PAGE);

   return TRUE;
}


/*
 *  thumbview_refresh_thumbnail:
 *     @ Create thumbnail from original image file.
 *
 *  thumb  : Pointer to the Thumbnail struct to refresh.
 *  Return : True if success.
 */
gboolean
thumbview_refresh_thumbnail (Thumbnail *thumb, ThumbLoadType type)
{
   ThumbView *tv = thumb->thumb_view;
   ThumbViewPlugin *view;
   gboolean retval = FALSE;

   g_return_val_if_fail (thumb, FALSE);

   view = g_hash_table_lookup (thumbview_modes, tv->disp_mode);
   g_return_val_if_fail (view, FALSE);

   thumbnail_create (thumb, tv->ThumbnailSize, type);

   if (view->refresh_thumb_func)
      retval = view->refresh_thumb_func (thumb, LOAD_CACHE);

   return retval;
}


typedef struct _SetCursorData
{
   ThumbView *tv;
   Thumbnail *cursor;
} SetCursorData;


static gboolean
idle_set_cursor(gpointer user_data)
{
   SetCursorData *data = user_data;

   if (data->cursor) {
      thumbview_set_focus (data->tv, data->cursor);
      thumbview_adjust (data->tv, data->cursor);
   }   

   g_free (data);

   return FALSE;
}


gboolean
thumbview_refresh_list (ThumbView *tv)
{
   FilesLoader *files;
   GList *selection, *thumbnode, *node;
   Thumbnail *thumb, *dest_cur[4];
   gchar *filename;
   gboolean exist = FALSE;
   struct stat st;
   gint i, flags;

   if (!tv)
      return FALSE;

   if (tv->progress)
      return FALSE;

   /* get dest cursor candiates */
   for (i = 0; i < sizeof (dest_cur) / sizeof (Thumbnail *); i++)
      dest_cur[i] = NULL;

   dest_cur[0] = thumbview_get_focus (tv);
   if (dest_cur[0]) {
      node = g_list_find (tv->thumblist, dest_cur[0]);
      if (node) {
         if (g_list_next (node))
            dest_cur[1] = g_list_next (node)->data;
         else if (g_list_previous (node))
            dest_cur[1] = g_list_previous (node)->data;
      }
   }

   selection = thumbview_get_selection_list (tv);

   if (selection) {
      node = g_list_find (tv->thumblist, selection->data);
      if (node) {
         dest_cur[2] = node->data;
         if (g_list_previous (node))
            dest_cur[3] = g_list_previous (node)->data;
      }

      for (; node; node = g_list_next (node)) {
         if (!g_list_find (selection, node->data)) {
            dest_cur[3] = node->data;
            break;
         }
      }
   }

   g_list_free (selection);

   /* search */
   files = files_loader_new ();

   if (tv->mode == THUMB_VIEW_MODE_DIR) {
      g_return_val_if_fail (tv->dirname, FALSE);

      flags = GETDIR_FOLLOW_SYMLINK;
      if (conf.read_dotfile)
         flags = flags | GETDIR_READ_DOT;
      if (conf.detect_filetype_by_ext)
         flags = flags | GETDIR_DETECT_EXT;
      if (conf.thumbview_show_archive)
         flags = flags | GETDIR_GET_ARCHIVE;

      get_dir (tv->dirname, flags, &files->filelist, &files->dirlist);

      if (tv->mode == THUMB_VIEW_MODE_DIR && conf.thumbview_show_dir) {
         gchar *parent = g_strconcat (tv->dirname, "..", NULL);
         files->filelist = g_list_concat (files->dirlist, files->filelist);
         files->filelist = g_list_prepend (files->filelist, parent);
         files->dirlist = NULL;
      }

      thumbnode = g_list_first (tv->thumblist);
      while (thumbnode) {
         thumb = thumbnode->data;
         thumbnode = g_list_next (thumbnode);

         /* remove same file from files->filelist */
         node = g_list_first (files->filelist);
         while (node) {
            filename = node->data;
            if (!strcmp (filename, image_info_get_path (thumb->info))) {
               /* check modification time */
               if (!stat (filename, &st)
                   && ((thumb->info->st.st_mtime != st.st_mtime)
                       ||(thumb->info->st.st_ctime != st.st_ctime)))
               {
                  exist = FALSE;
               } else {
                  exist = TRUE;
                  files->filelist = g_list_remove (files->filelist, filename);
                  g_free (filename);
               }
               break;
            }
            node = g_list_next (node);
         }

         /* FIXME!! */
         /* remove obsolete data */
         if (!exist) {
            tv->thumblist = g_list_remove (tv->thumblist, thumb);
            tv->filenum--;
            tv->thumb_window->filenum--;
            tv->filesize -= thumb->info->st.st_size;
            tv->thumb_window->filesize -= thumb->info->st.st_size;
            thumbnail_unref (thumb);
         }
         /* END FIXME!! */

         exist = FALSE;
      }

      thumbview_redraw (tv, tv->disp_mode, tv->container, NULL);

      /* append new files */
      if (files->filelist) {
         thumbview_append_thumbnail (tv, files, TRUE);
      }

   } else if (tv->mode == THUMB_VIEW_MODE_COLLECTION) {
   }

   files_loader_delete (files);

   thumbview_redraw (tv, tv->disp_mode, tv->container, NULL);
   thumbwin_set_statusbar_page_info (tv->thumb_window, THUMBWIN_CURRENT_PAGE);

   /* set cursor position */
   for (i = 0; i < sizeof (dest_cur) / sizeof (Thumbnail *); i++) {
      thumb = dest_cur[i];

      if (thumb && g_list_find (tv->thumblist, thumb)) {
         SetCursorData *data = g_new0 (SetCursorData, 1);
         data->tv = tv;
         data->cursor = thumb;
         gtk_idle_add (idle_set_cursor, data);
         break;
      }
   }

   /* check relation */
   node = tv->related_image_view;
   while (node) {
      ImageView *iv = node->data;
      GList *lnode;

      node = g_list_next (node);

      lnode = imageview_image_list_current (iv);
      if (lnode && !g_list_find (tv->thumblist, lnode->data))
         imageview_remove_list (iv, (gpointer) tv);
   }

   return TRUE;
}


gint
thumbview_refresh_list_idle (gpointer data)
{
   ThumbView *tv = data;
   thumbview_refresh_list (tv);
   return FALSE;
}


/*
 *  thumbview_redraw:
 *     @ Redraw thumbnail view or switch display mode or move to new container.
 *
 *  tv         : Pointer to the Thumbview struct.
 *  mode       : new display mode for switching display mode.
 *  scroll_win : new container for moveng thumbnail view.
 */
void
thumbview_redraw (ThumbView  *tv,
                  gchar      *mode,
                  GtkWidget  *scroll_win,
                  GList     **loadlist)
{
   ThumbViewPlugin *view;
   GList *node;
   GtkAdjustment *hadj, *vadj;

   g_return_if_fail (tv);

   node = g_list_find (ThumbViewList, tv);
   if (!node) return;

   if (!scroll_win)
      scroll_win = tv->container;

   thumbview_remove_scrollbar_callback (tv);

   /* sort thumbnail list */
   thumbview_sort_data (tv);

   /* remove current widget */
   view = g_hash_table_lookup (thumbview_modes, tv->disp_mode);
   g_return_if_fail (view);
   if (view->redraw_func)
      view->redraw_func (tv, mode, NULL, NULL);

   /* create new widget */
   view = g_hash_table_lookup (thumbview_modes, mode);
   g_return_if_fail (view);
   if (view->redraw_func)
      view->redraw_func (tv, mode, scroll_win, loadlist);

   tv->disp_mode = mode;
   tv->container = scroll_win;

   hadj = gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW (tv->container));
   vadj = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (tv->container));
   hadj->value = 0.0;
   vadj->value = 0.0;
   gtk_signal_emit_by_name (GTK_OBJECT(hadj), "value_changed"); 
   gtk_signal_emit_by_name (GTK_OBJECT(vadj), "value_changed"); 

   thumbview_set_scrollbar_callback (tv);

   thumbview_reset_load_priority (tv);

   gtk_widget_grab_focus (GTK_BIN (tv->container)->child);
}


/*
 *  thumbview_change_mode:
 *     @ Switch display mode.
 *
 *  tv         : Pointer to the Thumbview struct.
 *  mode       : new display mode for switching display mode.
 */
void
thumbview_change_mode (ThumbView *tv, gchar *mode)
{
   ThumbWindow *tw;
   GList *node, *loadlist = NULL;

   g_return_if_fail (tv);

   tw = tv->thumb_window;
   g_return_if_fail (tw);

   node = g_list_find (ThumbViewList, tv);
   if (!node) return;

   thumbview_redraw (tv, mode, tv->container, &loadlist);

   /* reload images if needed */
   if (loadlist) {
      FilesLoader *files;

      files = files_loader_new ();
      tv->progress = files;

      thumbview_load_thumbnails (tv, loadlist, tv->disp_mode);

      g_list_free (loadlist);
      files_loader_delete (files);
      tv->progress = NULL;

      thumbwin_set_statusbar_page_info (tw, THUMBWIN_CURRENT_PAGE);
   }
}


/*
 *  thumbview_resize:
 *     @ Resize thumbnail view widget. 
 *
 *  tv     : Pointer to the ThumbView struct.
 *  Return : New thumbnail view widget.
 */
GtkWidget *
thumbview_resize (ThumbView *tv)
{
   ThumbViewPlugin *view;
   GtkWidget *retval = NULL;

   if (!tv) return NULL;

   view = g_hash_table_lookup (thumbview_modes, tv->disp_mode);
   g_return_val_if_fail (view, FALSE);

   if (view->resize_func)
      retval = view->resize_func (tv);

   gtk_widget_grab_focus (GTK_BIN (tv->container)->child);

   return retval;
}


/*
 *  thumbview_adjust:
 *     @  
 *
 *  tv     : Pointer to the ThumbView struct.
 *  thumb  : Pointer to the Thumbnail struct.
 */
void
thumbview_adjust (ThumbView *tv, Thumbnail *thumb)
{
   ThumbViewPlugin *view;

   g_return_if_fail (tv);

   view = g_hash_table_lookup (thumbview_modes, tv->disp_mode);
   g_return_if_fail (view);

   if (view->adjust_func)
      view->adjust_func (tv, thumb);

   return;
}


GList *
thumbview_get_selection_list (ThumbView *tv)
{
   GList *list = NULL, *node;
   Thumbnail *thumb;

   g_return_val_if_fail (tv, NULL);

   node = tv->thumblist;
   if (!node) return NULL;

   while (node) {
      thumb = node->data;

      if (thumb->selected)
         list = g_list_append (list, thumb);

      node = g_list_next (node);
   }

   return list;
}


GList *
thumbview_get_selected_file_list (ThumbView *tv)
{
   GList *list = NULL, *filelist = NULL, *node;

   g_return_val_if_fail (tv, NULL);

   list = thumbview_get_selection_list (tv);
   if (!list) return NULL;

   node = list;
   while (node) {
      Thumbnail *thumb = node->data;

      if (thumb->info)
         filelist = g_list_append (filelist,
                                   g_strdup (image_info_get_path (thumb->info)));

      node = g_list_next (node);
   }

   return filelist;
}


gboolean
thumbview_set_selection (Thumbnail *thumb, gboolean select)
{
   ThumbViewPlugin *view;
   ThumbView *tv;

   g_return_val_if_fail (thumb, FALSE);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, FALSE);

   if (thumb->selected == select)
      return TRUE;

   view = g_hash_table_lookup (thumbview_modes, tv->disp_mode);
   g_return_val_if_fail (view, FALSE);

   if (view->set_selection_func)
      view->set_selection_func (thumb, select);
   else
      return FALSE;

   return TRUE;
}


gboolean
thumbview_set_selection_all (ThumbView *tv, gboolean select)
{
   Thumbnail *thumb;
   GList *node;

   g_return_val_if_fail (tv, FALSE);
   g_return_val_if_fail (tv->thumblist, FALSE);

   node = tv->thumblist;
   while (node) {
      thumb = node->data;
      thumbview_set_selection (thumb, select);
      node = g_list_next (node);
   }

   return TRUE;
}


void
thumbview_set_focus (ThumbView *tv, Thumbnail *thumb)
{
   ThumbViewPlugin *view;

   g_return_if_fail (tv);

   view = g_hash_table_lookup (thumbview_modes, tv->disp_mode);
   g_return_if_fail (view);

   if (view->set_focus_func)
      view->set_focus_func (tv, thumb);
}


Thumbnail *
thumbview_get_focus (ThumbView *tv)
{
   ThumbViewPlugin *view;

   g_return_val_if_fail (tv, NULL);

   view = g_hash_table_lookup (thumbview_modes, tv->disp_mode);
   g_return_val_if_fail (view, NULL);

   if (view->get_focus_func)
      return view->get_focus_func (tv);

   return NULL;
}


gboolean
thumbview_set_selection_multiple (Thumbnail *thumb,
                                  gboolean reverse, gboolean clear)
{
   ThumbView *tv;
   Thumbnail *thumb_tmp;
   GList *node, *current_node;
   gboolean retval = FALSE;

   g_return_val_if_fail (thumb, FALSE);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, FALSE);

   node = current_node = g_list_find (tv->thumblist, thumb);
   if (reverse)
      node = g_list_previous (node);
   else
      node = g_list_next (node);

   while (node) {
      thumb_tmp = node->data;
      if (thumb_tmp->selected) {
         if (clear)
            thumbview_set_selection_all (tv, FALSE);
         while (TRUE) {
            thumb_tmp = node->data;
            thumbview_set_selection (thumb_tmp, TRUE);
            if (node == current_node) break;
            if (reverse)
               node = g_list_next (node);
            else
               node = g_list_previous (node);
         }
         retval = TRUE;
         break;
      }
      if (reverse)
         node = g_list_previous (node);
      else
         node = g_list_next (node);
   }

   return retval;
}


void
thumbview_grab_focus (ThumbView *tv)
{
   g_return_if_fail(tv);
   g_return_if_fail(GTK_IS_BIN(tv->container));
   gtk_widget_grab_focus (GTK_BIN (tv->container)->child);
}


void
thumbview_find_duplicates (ThumbView *tv, Thumbnail *thumb, const gchar *type)
{
   DuplicatesFinder *finder;
   SimilarWin *sw = NULL;   
   ThumbWindow *tw;
   GList *node;

   g_return_if_fail (tv);

   tw = tv->thumb_window;
   g_return_if_fail (tv);

   /* create window */
   sw = similar_win_new (tv->ThumbnailSize);
   gtk_signal_connect (GTK_OBJECT (sw), "destroy",
                       GTK_SIGNAL_FUNC (cb_similar_win_destroy),
                       tv);
   similar_win_set_relation (sw, tv);
   tv->related_similar_win
      = g_list_append (tv->related_similar_win, sw);

   /* set finder */
   finder = sw->finder;
   duplicates_finder_set_algol_type (finder, type);

   for (node = tv->thumblist; node; node = g_list_next (node)) {
      duplicates_finder_append_dest (finder, node->data);
   }

   duplicates_finder_start (finder);
}


void
thumbview_reset_tab_label (ThumbView *tv, const gchar *title)
{
   gchar *tmpstr;
   const gchar *filename;

   /* set tab title */
   if (tv->mode == THUMB_VIEW_MODE_COLLECTION) {
      if (title) {
         g_free (tv->tabtitle);
         tv->tabtitle = g_strdup (title);
      }

   } else {
      if (tv->mode == THUMB_VIEW_MODE_DIR) {
         g_return_if_fail (tv->dirname && *tv->dirname);
         filename = tv->dirname;
      } else if (tv->mode == THUMB_VIEW_MODE_ARCHIVE) {
         g_return_if_fail (tv->archive);
         g_return_if_fail (tv->archive->filename && *tv->archive->filename);
         filename = tv->archive->filename;
      } else {
         return;
      }

      if (conf.thumbwin_tab_fullpath) {
         tmpstr = fileutil_home2tilde (filename);
      } else {
         if (tv->mode == THUMB_VIEW_MODE_ARCHIVE) {
            tmpstr = g_strdup (g_basename (filename));
         } else {
            gchar *dirname = g_dirname (filename);
            tmpstr = fileutil_dir_basename (dirname);
            g_free (dirname);
         }
      }

      g_free (tv->tabtitle);
      tv->tabtitle = charset_to_internal (tmpstr,
                                          conf.charset_filename,
                                          conf.charset_auto_detect_fn,
                                          conf.charset_filename_mode);

      g_free (tmpstr);

   }

   thumbwin_set_tab_label_text (tv->container, tv->tabtitle);
}


/*
 *  thumbview_initialize:
 *     @ initialize specified thumbnail view widget. 
 *
 *  tv        : Pointer to the ThumbView struct.
 *  dest_mode : Thumbnail display mode to initialize.
 */
static GtkWidget *
thumbview_initialize (ThumbView *tv, gchar *dest_mode)
{
   ThumbViewPlugin *view;
   GtkWidget *widget;

   g_return_val_if_fail (tv, FALSE);

   view = g_hash_table_lookup (thumbview_modes, dest_mode);
   g_return_val_if_fail (view, FALSE);

   g_return_val_if_fail (view->create_func, FALSE);

   widget = view->create_func (tv, dest_mode);

   return widget;
}


/*
 *  thumbview_create
 *     @ Create a thumbnail view. 
 *
 *  files     : Pointer to FilesLoader struct that store some infomation of
 *              opening imagee files.
 *  tw        : Pointer to the ThumbWindow struct.
 *  container : Pointer to GtkWidget to store thumbnail table.
 *  mode      : Thumbnail View mode (DIR or COLLECTION).
 *  Return    : Pointer to the ThumbView struct.
 */
ThumbView *
thumbview_create (FilesLoader *files, ThumbWindow *tw,
                  GtkWidget *container, ThumbViewMode mode)
{
   ThumbView *tv = NULL;
   GtkWidget *widget;
   gint page, current_page, this_page;
   GList *loadlist = NULL;
   gboolean is_scrollable;
   gchar buf[BUF_SIZE];

   thumbview_get_disp_mode_list ();

   g_return_val_if_fail (files, NULL);
   g_return_val_if_fail (mode == THUMB_VIEW_MODE_COLLECTION
                         || (mode == THUMB_VIEW_MODE_DIR && files->dirname)
                         || (mode == THUMB_VIEW_MODE_RECURSIVE_DIR && files->dirname)
                         || (mode == THUMB_VIEW_MODE_ARCHIVE && files->archive),
                         NULL);

   if (files->status >= CANCEL)
      return NULL;

   page = gtk_notebook_page_num (GTK_NOTEBOOK (tw->notebook), container);
   if (page < 0) {
      files->status = CANCEL;
      return NULL;
   }

   files->status = THUMB_LOADING;

   /* allocate new ThumbView struct and initialize */
   tv = g_new0 (ThumbView, 1);
   tv->progress = files;
   tv->thumb_window = tw;
   tv->container = container;
   tv->popup_menu = NULL;
   tv->dirname = NULL;
   tv->tabtitle = NULL;
   tv->archive = NULL;
   tv->filenum = 0;
   tv->filesize = 0;
   if (mode == THUMB_VIEW_MODE_RECURSIVE_DIR)
      tv->mode = THUMB_VIEW_MODE_COLLECTION;
   else
      tv->mode = mode;
   tv->disp_mode = tw->thumb_disp_mode;
   tv->disp_mode_data = g_hash_table_new (g_str_hash, g_str_equal);
   tv->related_image_view = NULL;
   tv->related_similar_win = NULL;
   tv->dnd_destdir = NULL;
   tv->load_list = NULL;
   tv->button_2pressed_queue = 0;

   /* set mode specific data */
   if (tv->mode == THUMB_VIEW_MODE_DIR || mode == THUMB_VIEW_MODE_RECURSIVE_DIR) {
      if (files->dirname[strlen (files->dirname) - 1] != '/')
         tv->dirname = g_strconcat (files->dirname, "/", NULL);
      else
         tv->dirname = g_strdup (files->dirname);

   } else if (tv->mode == THUMB_VIEW_MODE_ARCHIVE) {
      fr_archive_ref (FR_ARCHIVE (files->archive));
      tv->archive = files->archive;

   } else if (tv->mode == THUMB_VIEW_MODE_COLLECTION) {
      collection_page_count++;
   }

   /* set tab label */
   if (mode == THUMB_VIEW_MODE_RECURSIVE_DIR) {
      gchar *tmpstr, *dirname_internal;
      tmpstr = fileutil_home2tilde (tv->dirname);
      dirname_internal = charset_to_internal (tmpstr,
                                              conf.charset_filename,
                                              conf.charset_auto_detect_fn,
                                              conf.charset_filename_mode);
      g_snprintf (buf, BUF_SIZE, _("%s (Collection)"), dirname_internal);
      g_free (dirname_internal);
      g_free (tmpstr);
      thumbview_reset_tab_label (tv, buf);
   } else if (mode == THUMB_VIEW_MODE_COLLECTION) {
      g_snprintf (buf, BUF_SIZE, _("Collection %d"), collection_page_count);
      thumbview_reset_tab_label (tv, buf);
   } else {
      thumbview_reset_tab_label (tv, NULL);
   }
   thumbwin_set_tab_label_state (tv->container, GTK_STATE_SELECTED);

   /* set scrollbar callback */
   thumbview_set_scrollbar_callback (tv);

   /* get thumbnai size from toolbar */
   tv->ThumbnailSize =
      gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(tw->button.size_spin));

   ThumbViewList = g_list_append (ThumbViewList, tv);
   thumbwin_location_entry_set_text (tw, NULL);

   /* fetch infomation about image files */
   if (tv->mode == THUMB_VIEW_MODE_ARCHIVE) {
      thumbview_add_thumb_data_from_archive (tv, files->archive);
   } else {
      if (tv->mode == THUMB_VIEW_MODE_DIR && conf.thumbview_show_dir) {
         gchar *parent = g_strconcat (tv->dirname, "..", NULL);
         GList *dirlist = g_list_copy (files->dirlist);

         dirlist = g_list_prepend (dirlist, parent);
         thumbview_add_thumb_data (tv, dirlist);
         g_list_free(dirlist);
         g_free (parent);
      }
      thumbview_add_thumb_data (tv, files->filelist);
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_spel);
   }

   /* set window status */
   if (tv->mode == THUMB_VIEW_MODE_DIR && tv->dirname)
      dirview_set_opened_mark (tw->dv, tv->dirname);

   tw = tv->thumb_window;
   current_page = gtk_notebook_get_current_page (GTK_NOTEBOOK (tw->notebook));
   this_page = gtk_notebook_page_num (GTK_NOTEBOOK (tw->notebook), tv->container);

   if (current_page == this_page) {
      tw->status = THUMB_WIN_STATUS_LOADING;
      thumbwin_set_sensitive (tw, THUMB_WIN_STATUS_LOADING);
      if (tv->mode == THUMB_VIEW_MODE_DIR && tv->dirname)
         dirview_change_dir (tw->dv, tv->dirname);

   } else {
      tw->status = THUMB_WIN_STATUS_LOADING_BG;
      thumbwin_set_sensitive (tw, THUMB_WIN_STATUS_LOADING_BG);
   }

   /* load thumbnails */
   widget = thumbview_initialize (tv, tv->disp_mode);
   is_scrollable = GTK_WIDGET_GET_CLASS (widget)->set_scroll_adjustments_signal;
   if (is_scrollable)
      gtk_container_add (GTK_CONTAINER (tv->container), widget);
   else
      gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (tv->container),
                                             widget);     

   thumbview_redraw (tv, tv->disp_mode, tv->container, &loadlist);

   if (loadlist) {
      thumbview_load_thumbnails (tv, loadlist, tv->disp_mode);
      g_list_free (loadlist);
   }

   /* reset window status */
   if (files->status > 0 && files->status < CANCEL) {
      files->status = THUMB_LOAD_DONE;
   }

   if (files->status > 0) {
      tv->progress = NULL;
      /* reset status bar and tab label */
      tw->status = THUMB_WIN_STATUS_NORMAL;
      thumbwin_set_statusbar_page_info (tw, THUMBWIN_CURRENT_PAGE);
   }

   thumbview_resize (tv);

   thumbwin_set_tab_label_state (tv->container, GTK_STATE_NORMAL);

   return tv;
}
