/* -*- Mode: C; tab-width: 3; indent-tabs-mode: nil; c-basic-offset: 3 -*- */

/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: image_info.c,v 1.43.2.10 2003/05/31 17:34:21 makeinu Exp $
 */

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "gimageview.h"

#include "gfileutil.h"
#include "gimv_image.h"
#include "gimv_mime_types.h"
#include "fileutil.h"
#include "image_info.h"
#include "prefs.h"
#include "thumbnail.h"
#include "thumbnail_support.h"


GHashTable *ImageInfoTable = NULL;


/******************************************************************************
 *
 *   Private Functions.
 *
 ******************************************************************************/
static ImageInfo *
image_info_new (const gchar *filename)
{
   ImageInfo *info;

   info = g_new0 (ImageInfo, 1);
   g_return_val_if_fail (info, NULL);

   info->filename     = g_strdup (filename);
   info->archive_name = NULL;
   info->archive      = NULL;
   info->link         = NULL;
   info->format       = gimv_image_detect_type_by_ext (filename);
   info->width        = -1;
   info->height       = -1;
   info->depth        = -1;
   info->comment      = NULL;
   info->flags        = 0;

   info->ref_count = 1;

   return info;
}


/******************************************************************************
 *
 *   Public Functions.
 *
 ******************************************************************************/
/*
 *  image_info_get:
 *     @ Get infomation about normal file from cache.
 *       If it doesn't exist, create it.
 *
 *  filename:
 *  Return:
 */
ImageInfo *
image_info_get (const gchar *filename)
{
   ImageInfo *info;
   struct stat st;

   g_return_val_if_fail (filename, NULL);

   if (!ImageInfoTable) {
      ImageInfoTable = g_hash_table_new (g_str_hash, g_str_equal);
   }

   info = g_hash_table_lookup (ImageInfoTable, filename);
   if (!info) {
      if (!file_exists (filename)) return NULL;
      info = image_info_new (filename);
      if (!info) return NULL;
      stat (filename, &info->st);
      g_hash_table_insert (ImageInfoTable, info->filename, info);
   } else {
      if (stat (filename, &st)) return NULL;
      if (info->st.st_size != st.st_size
          || info->st.st_mtime != st.st_mtime
          || info->st.st_ctime != st.st_ctime)
      {
         info->st = st;
         info->width = -1;
         info->height = -1;
         info->flags &= ~IMAGE_INFO_SYNCED_FLAG;
      }
      image_info_ref (info);
   }

   return info;
}


/*
 *  image_info_get_url:
 *     @ Get infomation about remote (or Xine's MRL) file from cache.
 *       If it doesn't exist, create it.
 *
 *  filename:
 *  Return:
 */
ImageInfo *
image_info_get_url (const gchar *url)
{
   ImageInfo *info;

   g_return_val_if_fail (url, NULL);

   if (!ImageInfoTable) {
      ImageInfoTable = g_hash_table_new (g_str_hash, g_str_equal);
   }

   info = g_hash_table_lookup (ImageInfoTable, url);
   if (!info) {
      info = image_info_new (url);
      if (!info) return NULL;
      info->flags |= IMAGE_INFO_URL_FLAG;
      g_hash_table_insert (ImageInfoTable, info->filename, info);
   } else {
      image_info_ref (info);
   }

   return info;
}


ImageInfo *
image_info_get_with_archive (const gchar *filename,
                             FRArchive   *archive,
                             struct stat *st)
{
   ImageInfo *info;
   gchar buf[MAX_PATH_LEN];

   g_return_val_if_fail (filename, NULL);
   g_return_val_if_fail (archive, NULL);

   if (!ImageInfoTable) {
      ImageInfoTable = g_hash_table_new (g_str_hash, g_str_equal);
   }

   g_snprintf (buf, MAX_PATH_LEN, "%s/%s", archive->filename, filename);

   info = g_hash_table_lookup (ImageInfoTable, buf);
   if (!info) {
      info = image_info_new (filename);
      info->archive_name = g_strdup (archive->filename);
      info->archive = archive;
      g_hash_table_insert (ImageInfoTable, g_strdup (buf), info);
   } else {
      /* FIXME!! update if necessary */
      /* image_info_ref (info); */
   }

   if (st)
      info->st = *st;

   info->flags |= IMAGE_INFO_ARCHIVE_MEMBER_FLAG;

   return info;
}


ImageInfo *
image_info_lookup (const gchar *filename)
{
   ImageInfo *info;

   info = g_hash_table_lookup (ImageInfoTable, filename);

   if (info)
      image_info_ref (info);

   return info;
}


void
image_info_finalize (ImageInfo *info)
{
   guint num;
   gchar buf[MAX_PATH_LEN];

   g_return_if_fail (info);

   /* remove data from hash table */
   if (info->archive) {
      gchar *orig_key;
      ImageInfo *value;
      gboolean success;

      g_snprintf (buf, MAX_PATH_LEN, "%s/%s",
                  info->archive->filename, info->filename);
      success = g_hash_table_lookup_extended (ImageInfoTable, buf,
                                              (gpointer) &orig_key,
                                              (gpointer) &value);
      if (success) {
         g_hash_table_remove (ImageInfoTable, buf);
         g_free (orig_key);
      }
   } else {
      g_hash_table_remove (ImageInfoTable, info->filename);
   }

   if (info->filename) {
      g_free (info->filename);
      info->filename = NULL;
   }
   if (info->archive_name) {
      g_free (info->archive_name);
      info->archive_name = NULL;
   }
   if (info->link) {
      g_free (info->link);
      info->link = NULL;;
   }
   if (info->archive) {
      info->archive = NULL;
   }
   g_free (info);

   /* destroy hash table if needed */
   num = g_hash_table_size (ImageInfoTable);
   if (num < 1) {
      g_hash_table_destroy (ImageInfoTable);
      ImageInfoTable = NULL;
   }
}


ImageInfo *
image_info_ref (ImageInfo *info)
{
   g_return_val_if_fail (info, NULL);

   info->ref_count++;

   if (info->archive) {
      fr_archive_ref (FR_ARCHIVE (info->archive));
   }

   return info;
}


void
image_info_unref (ImageInfo *info)
{
   FRArchive *archive;

   g_return_if_fail (info);

   archive = info->archive;

   info->ref_count--;

   if (info->ref_count < 1) {
      image_info_finalize (info);
   }

   if (archive)
      fr_archive_unref (FR_ARCHIVE (archive));
}


/* used by fr-command */
void
image_info_unref_with_archive (ImageInfo *info)
{
   g_return_if_fail (info);

   info->ref_count--;
   /* info->archive = NULL; */

   if (info->ref_count < 1)
      image_info_finalize (info);
}


void
image_info_set_size (ImageInfo *info, gint width, gint height)
{
   g_return_if_fail (info);

   if (info->flags & IMAGE_INFO_SYNCED_FLAG) return;

   info->width  = width;
   info->height = height;
}


void
image_info_set_flags (ImageInfo *info, ImageInfoFlags flags)
{
   g_return_if_fail (info);

   if (!(info->flags & IMAGE_INFO_SYNCED_FLAG)
       || (flags & IMAGE_INFO_SYNCED_FLAG))
   {
      info->flags |= flags;
   }
}


void
image_info_unset_flags (ImageInfo *info, ImageInfoFlags flags)
{
   g_return_if_fail (info);

   if (!(info->flags & IMAGE_INFO_SYNCED_FLAG)
       || (flags & IMAGE_INFO_SYNCED_FLAG))
   {
      info->flags &= ~flags;
   }
}


gboolean
image_info_set_data_from_image (ImageInfo *info, GimvImage *image)
{
   gboolean is_exist;

   g_return_val_if_fail (info, FALSE);

   if (!image) return FALSE;

   if (image_info_is_in_archive (info)) {
      /* FIXME!! should we call stat? */
   } else {
      is_exist = !stat (info->filename, &info->st);
      if (!is_exist) return FALSE;
   }

   if (gimv_image_is_anim (image)) {
      info->flags |= IMAGE_INFO_ANIMATION_FLAG;
   }

   gimv_image_get_size (image, &info->width, &info->height);
   info->flags |= IMAGE_INFO_SYNCED_FLAG;

   return TRUE;
}


gboolean
image_info_is_dir (ImageInfo *info)
{
   g_return_val_if_fail (info, FALSE);

   if (isdir (image_info_get_path (info)))
      return TRUE;

   return FALSE;
}


gboolean
image_info_is_archive (ImageInfo *info)
{
   g_return_val_if_fail (info, FALSE);

   if (fr_archive_utils_get_file_name_ext (image_info_get_path (info)))
      return TRUE;

   return FALSE;
}


gboolean
image_info_is_in_archive (ImageInfo *info)
{
   g_return_val_if_fail (info, FALSE);

   if (info->archive_name)
      return TRUE;
   else
      return FALSE;
}


gboolean
image_info_is_url (ImageInfo *info)
{
   g_return_val_if_fail (info, FALSE);

   if (info->flags & IMAGE_INFO_URL_FLAG)
      return TRUE;
   else
      return FALSE;
}


gboolean
image_info_is_animation (ImageInfo *info)
{
   g_return_val_if_fail (info, FALSE);  

   if (info->flags & IMAGE_INFO_ANIMATION_FLAG)
      return TRUE;

   return FALSE;
}


gboolean
image_info_is_movie (ImageInfo *info)
{
   const gchar *ext;
   const gchar *type;

   g_return_val_if_fail (info, FALSE);

   if ((info->flags & IMAGE_INFO_MOVIE_FLAG)
       || (info->flags & IMAGE_INFO_MRL_FLAG))
   {
      return TRUE;
   }

   ext  = fileutil_get_extention (info->filename);
   type = gimv_mime_types_get_type_from_ext (ext);

   if (type 
       && (!g_strncasecmp (type, "video", 5))
       && g_strcasecmp (type, "video/x-mng")) /* FIXME!! */
   {
      return TRUE;
   }

   return FALSE;
}


gboolean
image_info_is_audio (ImageInfo *info)
{
   const gchar *ext;
   const gchar *type;

   g_return_val_if_fail (info, FALSE);

   ext  = gimv_mime_types_get_extension (info->filename);
   type = gimv_mime_types_get_type_from_ext (ext);

   if (type && !g_strncasecmp (type, "audio", 5))
      return TRUE;

   return FALSE;
}


gboolean
image_info_is_same (ImageInfo *info1, ImageInfo *info2)
{
   g_return_val_if_fail (info1, FALSE);
   g_return_val_if_fail (info2, FALSE);
   g_return_val_if_fail (info1->filename && *info1->filename, FALSE);
   g_return_val_if_fail (info2->filename && *info2->filename, FALSE);

   if (strcmp (info1->filename, info2->filename))
      return FALSE;

   if (!info1->archive && !info2->archive)
      return TRUE;

   if (info1->archive_name && *info2->archive_name
       && info2->archive_name && *info2->archive_name
       && !strcmp (info1->archive_name, info2->archive_name))
   {
      return TRUE;
   }

   return FALSE;
}


const gchar *
image_info_get_path (ImageInfo *info)
{
   g_return_val_if_fail (info, NULL);

   return info->filename;
}


gchar *
image_info_get_path_with_archive (ImageInfo *info)
{
   g_return_val_if_fail (info, NULL);

   if (!info->archive)
      return g_strdup (info->filename);

   return g_strconcat (info->archive_name, "/", info->filename, NULL);
}


const gchar *
image_info_get_archive_path (ImageInfo *info)
{
   g_return_val_if_fail (info, NULL);
   g_return_val_if_fail (info->archive, NULL);

   return info->archive->filename;
}


gboolean
image_info_need_temp_file (ImageInfo *info)
{
   g_return_val_if_fail (info, FALSE);

   if (image_info_is_in_archive (info))
      return TRUE;
   else
      return FALSE;
}


gchar *
image_info_get_temp_file_path (ImageInfo *info)
{
   FRArchive *archive;
   gchar *temp_dir;
   gchar *filename, buf[MAX_PATH_LEN];

   g_return_val_if_fail (info, NULL);

   archive = info->archive;
   g_return_val_if_fail (archive, NULL);

   filename = info->filename;

   temp_dir = gtk_object_get_data (GTK_OBJECT (archive), "temp-dir");

   g_return_val_if_fail (temp_dir && *temp_dir, NULL);

   g_snprintf (buf, MAX_PATH_LEN, "%s/%s",
               temp_dir, filename);

   return g_strdup (buf);
}


gchar *
image_info_get_temp_file (ImageInfo *info)
{
   gboolean success;
   gchar *filename;

   /* load the image */
   if (!image_info_need_temp_file (info))
      return NULL;

   filename = image_info_get_temp_file_path (info);
   g_return_val_if_fail (filename, NULL);

   if (file_exists (filename)) {
      return filename;
   }

   success = image_info_extract_archive (info);
   if (success)
      return filename;

   g_free (filename);
   return NULL;
}


/*
 *  image_info_extract_archive:
 *     @ extract an image from archive.
 *
 *  info    :
 *  Return  : Temporary file name. (shoud be free)
 */
/*
 *  FIXME!!
 *  If archive was already destroyed, create FRArchive object first.
 *  To detect whether archive was destroyed or not, check info->archive.
 *  When archive is destroyed, fr-archive class will set this value to NULL
 *  by using image_info_unref_with_archive ().
 */
gboolean
image_info_extract_archive (ImageInfo *info)
{
   FRArchive *archive;
   GList *filelist = NULL;
   gchar *temp_dir;
   gchar *filename, *temp_file, buf[MAX_PATH_LEN];

   g_return_val_if_fail (info, FALSE);

   archive = info->archive;
   g_return_val_if_fail (archive, FALSE);

   filename = info->filename;

   temp_dir = gtk_object_get_data (GTK_OBJECT (archive), "temp-dir");

   g_return_val_if_fail (temp_dir && *temp_dir, FALSE);

   g_snprintf (buf, MAX_PATH_LEN, "%s/%s",
               temp_dir, filename);
   temp_file = buf;

   if (!file_exists (temp_file) && !archive->process->running) {
      ensure_dir_exists (temp_dir);
      filelist = g_list_append (filelist, filename);

      fr_archive_extract (archive, filelist, temp_dir,
                          FALSE, TRUE, FALSE);

      gtk_main ();   /* will be quited by callback function
                        of archive (see fileload.c) */
   }

   if (file_exists (temp_file)) {
      return TRUE;
   } else {
      return FALSE;
   }
}


GimvIO *
image_info_get_gio (ImageInfo *info)
{
   gboolean need_temp;
   const gchar *filename;
   gchar *temp_file = NULL;
   GimvIO *gio = NULL;

   g_return_val_if_fail (info, NULL);

   need_temp = image_info_need_temp_file (info);
   if (need_temp) {
      temp_file = image_info_get_temp_file (info);
      filename = temp_file;
   } else {
      filename = image_info_get_path (info);
   }

   if (filename)
      gio = gimv_io_new (filename, "rb");

   g_free (temp_file);

   return gio;
}


const gchar *
image_info_get_format (ImageInfo *info)
{
   g_return_val_if_fail (info, NULL);

   return info->format;
}


void
image_info_get_image_size (ImageInfo *info, gint *width_ret, gint *height_ret)
{
   g_return_if_fail (width_ret && height_ret);
   *width_ret = -1;
   *height_ret = -1;

   g_return_if_fail (info);

   *width_ret  = info->width;
   *height_ret = info->height;
}


/* FIXME */
gboolean
image_info_rename_image (ImageInfo *info, const gchar *filename)
{
   struct stat st;
   gboolean success;
   gchar *cache_type;
   gchar *src_cache_path, *dest_cache_path;
   gchar *src_comment, *dest_comment;

   g_return_val_if_fail (info, FALSE);
   g_return_val_if_fail (info->filename, FALSE);
   g_return_val_if_fail (!info->archive, FALSE);
   g_return_val_if_fail (filename, FALSE);

   /* rename the file */
   success = !rename(info->filename, filename);
   if (!success) return FALSE;

   /* rename cache */
   src_cache_path = thumbnail_find_thumbcache (image_info_get_path (info),
                                               &cache_type);
   if (src_cache_path) {
      dest_cache_path
         = thumbsupport_get_thumb_cache_path (filename, cache_type);
      if (rename (src_cache_path, dest_cache_path) < 0)
         g_print (_("Faild to rename cache file :%s\n"), filename);
      g_free (src_cache_path);
      g_free (dest_cache_path);
   }

   /* rename comment */
   src_comment = comment_find_file (image_info_get_path (info));
   if (src_comment) {
      dest_comment = comment_get_path (filename);
      if (rename (src_comment, dest_comment) < 0)
         g_print (_("Faild to rename comment file :%s\n"), dest_comment);
      g_free (src_comment);
      g_free (dest_comment);      
   }

   if (stat (filename, &st)) return FALSE;
   info->st = st;
   g_free (info->filename);
   info->filename = (gchar *) g_strdup (filename);

   return success;
}
