/* -*- Mode: C; tab-width: 3; indent-tabs-mode: nil; c-basic-offset: 3 -*- */

/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: image_view_draw.c,v 1.1.2.7 2003/05/21 08:26:36 makeinu Exp $
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif /* HAVE_CONFIG_H */

#include "cursors.h"
#include "gimv_anim.h"
#include "image_view.h"
#include "prefs.h"
#include "thumbnail_support.h"


static gboolean cb_image_configure         (GtkWidget         *widget,
                                            GdkEventConfigure *event,
                                            ImageView         *iv);
static gboolean cb_image_expose            (GtkWidget         *widget,
                                            GdkEventExpose    *event,
                                            ImageView         *iv);

/* virtual functions */
static GtkWidget   *imageview_draw_create               (ImageView *iv);
static void         imageview_draw_create_thumbnail     (ImageView *iv,
                                                         const gchar *type);

static gboolean     imageview_draw_is_playable          (ImageView *iv,
                                                         ImageInfo *info);
static void         imageview_animation_play            (ImageView *iv);
static void         imageview_animation_stop            (ImageView *iv);
#if 0
static void         imageview_animation_pause           (ImageView *iv);
#endif
static ImageViewPlayableStatus
                    imageview_draw_get_status           (ImageView *iv);


static ImageViewPlayableInterFace imageview_draw_playable_table = {
   is_playable_fn:      imageview_draw_is_playable,
   is_seekable_fn:      NULL,
   play_fn:             imageview_animation_play,
   stop_fn:             imageview_animation_stop,
#if 0
   pause_fn:            imageview_animation_pause,
#else
   pause_fn:            NULL,
#endif
   forward_fn:          NULL,
   reverse_fn:          NULL,
   seek_fn:             NULL,
   eject_fn:            NULL,
   get_status_fn:       imageview_draw_get_status,
   get_length_fn:       NULL,
   get_position_fn:     NULL,
};


ImageViewPlugin imageview_draw_vfunc_table = {
   label:               IMAGEVIEW_DEFAULT_VIEW_MODE,
   priority_hint:       G_PRIORITY_LOW,
   is_supported_fn:     NULL,
   create_fn:           imageview_draw_create,
   create_thumbnail_fn: imageview_draw_create_thumbnail,
   fullscreen_fn:       NULL,

   scalable:            NULL,
   rotatable:           NULL,
   playable:            &imageview_draw_playable_table,
};


static GHashTable *animation_id_table        = NULL;
static GHashTable *create_thumbnail_id_table = NULL;


/*****************************************************************************
 *
 *   callback functions
 *
 *****************************************************************************/
static void
cb_destroy (GtkWidget *widget, ImageView *iv)
{
   g_return_if_fail (iv);

   imageview_animation_stop (iv);
}


static void
cb_load_end_create_thumbnail (ImageView *iv, ImageInfo *info,
                              gboolean cancel, gpointer data)
{
   GimvImage *imcache;
   ThumbCacheSaveInfo *save_info;
   gchar *filename;
   gboolean free_buf = GPOINTER_TO_INT (data);
   gpointer id_p;
   guint id;

   g_return_if_fail (IS_IMAGEVIEW (iv));

   id_p = g_hash_table_lookup (create_thumbnail_id_table, iv);
   id = GPOINTER_TO_UINT (id_p);
   if (id > 0)
      gtk_signal_disconnect (GTK_OBJECT (iv), id);
   g_hash_table_remove (create_thumbnail_id_table, iv);

   if (cancel) return;
   if (!iv->image) return;
   if (iv->info != iv->loader->info) return;

   filename = image_info_get_path_with_archive (iv->info);
   save_info = thumbsupport_save_info_new (iv->info, NULL);

   /* FIXME: conf.cache_write_type is hard coded */
   imcache = thumbsupport_save_thumbnail_cache (filename,
                                                conf.cache_write_type,
                                                iv->image,
                                                save_info);

   thumbsupport_save_info_delete (save_info);
   g_free (filename);

   if (free_buf)
      imageview_free_image_buf (iv);

   if (imcache) {
      gimv_image_unref (imcache);
      gtk_signal_emit_by_name (GTK_OBJECT (iv),
                               "thumbnail_created",
                               iv->info);
   }
}


static void
cb_draw_area_map (GtkWidget *widget, ImageView *iv)
{
   if (iv->bg_color) {
      imageview_set_bg_color (iv,
                              iv->bg_color->red,
                              iv->bg_color->green,
                              iv->bg_color->blue);
   }

   /* set cursor */
   if (!iv->cursor)
      iv->cursor = cursor_get (iv->draw_area->window, CURSOR_HAND_OPEN);
   gdk_window_set_cursor (iv->draw_area->window, iv->cursor);
}


static gboolean
cb_image_configure (GtkWidget *widget, GdkEventConfigure *event, ImageView *iv)
{
   gint width, height;
   gint fwidth, fheight;
   gint x_pos, y_pos;

   imageview_get_view_position (iv, &x_pos, &y_pos);
   imageview_get_image_size (iv, &width, &height);
   imageview_get_image_frame_size (iv, &fwidth, &fheight);

   if (fwidth < width) {
      if (x_pos < 0 || x_pos < 0 - fwidth || x_pos > width)
         x_pos = 0;
   } else {
      x_pos = (width - fwidth) / 2;
   }

   if (fheight < height) {
      if (y_pos < 0 || y_pos < 0 - fheight || y_pos > height)
         y_pos = 0;
   } else {

      y_pos = (height - fheight) / 2;
   }

   imageview_set_view_position (iv, x_pos, y_pos);
   imageview_draw_image (iv);

   return TRUE;
}


static gboolean
cb_image_expose (GtkWidget *widget, GdkEventExpose *event, ImageView *iv)
{
   imageview_draw_image (iv);
   return TRUE;
}



/*****************************************************************************
 *
 *   other private functions
 *
 *****************************************************************************/
static gboolean
timeout_animation (gpointer data)
{
   ImageView *iv = data;
   gint idx, interval;

   if (!iv->image) goto END;
   if (!gimv_image_is_anim (iv->image)) goto END;

   idx = gimv_anim_iterate ((GimvAnim *) iv->image);

   /* repeat */
   if (idx < 0) {
      if (!gimv_anim_seek ((GimvAnim *) iv->image, 0))
         goto END;
   }

   imageview_show_image (iv);

   interval = gimv_anim_get_interval ((GimvAnim *) iv->image);
   if (interval > 0) {
      guint timer = gtk_timeout_add (interval, timeout_animation, iv);
      g_hash_table_insert (animation_id_table,
                           iv, GUINT_TO_POINTER (timer));
   } else {
      goto END;
   }

   return FALSE;

END:
   g_hash_table_remove (animation_id_table, iv);
   imageview_playable_set_status (iv, ImageViewPlayableStop);
   return FALSE;
}


static gboolean
idle_animation_play (gpointer data)
{
   ImageView *iv = data;
   gint interval;

   if (!IS_IMAGEVIEW (iv)) goto END;
   if (!iv->info) goto END;

   if (!image_info_is_animation (iv->info)) goto END;

   imageview_animation_stop (iv);

   interval = gimv_anim_get_interval ((GimvAnim *) iv->image);
   if (interval > 0) {
      guint timer = gtk_timeout_add (interval, timeout_animation, iv);
      g_hash_table_insert (animation_id_table,
                           iv, GUINT_TO_POINTER (timer));
      imageview_playable_set_status (iv, ImageViewPlayablePlay);
   } else {
      goto END;
   }

   return FALSE;

END:
   g_hash_table_remove (animation_id_table, iv);
   imageview_playable_set_status (iv, ImageViewPlayableStop);
   return FALSE;
}



/*****************************************************************************
 *
 *   Virtual functions
 *
 *****************************************************************************/
static GtkWidget *
imageview_draw_create (ImageView *iv)
{
   GtkWidget *widget;

   if (!animation_id_table)
      animation_id_table
         = g_hash_table_new (g_direct_hash, g_direct_equal);

   if (!create_thumbnail_id_table)
      create_thumbnail_id_table
         = g_hash_table_new (g_direct_hash, g_direct_equal);

   widget = gtk_drawing_area_new ();

   gtk_signal_connect       (GTK_OBJECT (widget), "destroy",
                             GTK_SIGNAL_FUNC (cb_destroy), iv);
   gtk_signal_connect_after (GTK_OBJECT (widget), "map",
                             GTK_SIGNAL_FUNC (cb_draw_area_map), iv);
   gtk_signal_connect       (GTK_OBJECT (widget), "configure_event",
                             GTK_SIGNAL_FUNC (cb_image_configure), iv);
   gtk_signal_connect       (GTK_OBJECT (widget), "expose_event",
                             GTK_SIGNAL_FUNC (cb_image_expose), iv);

   gtk_widget_add_events (widget,
                          GDK_FOCUS_CHANGE
                          | GDK_BUTTON_PRESS_MASK | GDK_2BUTTON_PRESS
                          | GDK_KEY_PRESS | GDK_KEY_RELEASE
                          | GDK_BUTTON_RELEASE_MASK
                          | GDK_POINTER_MOTION_MASK
                          | GDK_POINTER_MOTION_HINT_MASK);

   return widget;
}


static void
imageview_draw_create_thumbnail (ImageView *iv, const gchar *cache_write_type)
{
   if (!iv->image) {
      gpointer id_p;
      guint id;

      id_p = g_hash_table_lookup (create_thumbnail_id_table, iv);
      id = GPOINTER_TO_UINT (id_p);
      if (id > 0)
         gtk_signal_disconnect (GTK_OBJECT (iv), id);
      id = gtk_signal_connect (GTK_OBJECT (iv), "load_end",
                               GTK_SIGNAL_FUNC (cb_load_end_create_thumbnail),
                               GINT_TO_POINTER (TRUE));
      g_hash_table_insert (create_thumbnail_id_table,
                           iv, GUINT_TO_POINTER (id));
      imageview_load_image_buf (iv);
   } else {
      cb_load_end_create_thumbnail (iv, iv->info,
                                    FALSE, GINT_TO_POINTER (FALSE));
   }
}


static gboolean
imageview_draw_is_playable (ImageView *iv, ImageInfo *info)
{
   g_return_val_if_fail (IS_IMAGEVIEW (iv), FALSE);
   if (!info) return FALSE;

   return image_info_is_animation (info);
}


static void
imageview_animation_play (ImageView *iv)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));
   gtk_idle_add (idle_animation_play, iv);
}


#if 0
static void
imageview_animation_stop (ImageView *iv)
{
   g_return_if_fail (iv);

   imageview_animation_pause (iv);

   if (!iv->info) return;
   if (!image_info_is_animation (iv->info)) return;

   gimv_anim_seek ((GimvAnim *) iv->image, 0);

   imageview_show_image (iv);

   imageview_playable_set_status (iv, ImageViewPlayableStop);
}


static void
imageview_animation_pause (ImageView *iv)
{
   gpointer id_p;
   guint id;

   g_return_if_fail (iv);

   if (!animation_id_table) return;

   id_p = g_hash_table_lookup (animation_id_table, iv);
   id = GPOINTER_TO_UINT (id_p);
   if (id > 0)
      gtk_timeout_remove (id);
   g_hash_table_remove (animation_id_table, iv);

   imageview_playable_set_status (iv, ImageViewPlayablePause);
}

#else

static void
imageview_animation_stop (ImageView *iv)
{
   gpointer id_p;
   guint id;

   g_return_if_fail (iv);

   if (!animation_id_table) return;

   id_p = g_hash_table_lookup (animation_id_table, iv);
   id = GPOINTER_TO_UINT (id_p);
   if (id > 0)
      gtk_timeout_remove (id);
   g_hash_table_remove (animation_id_table, iv);

   imageview_playable_set_status (iv, ImageViewPlayableStop);
}

#endif


static ImageViewPlayableStatus
imageview_draw_get_status (ImageView *iv)
{
   gpointer timer_p;
   guint timer;

   g_return_val_if_fail (IS_IMAGEVIEW (iv), ImageViewPlayableDisable);

   if (!iv->info || !image_info_is_animation (iv->info))
      return ImageViewPlayableDisable;

   timer_p = g_hash_table_lookup (animation_id_table, iv);
   timer = GPOINTER_TO_UINT (timer_p);
   if (timer > 0)
      return ImageViewPlayablePlay;
   else
      return ImageViewPlayableStop;
}
