/* -*- Mode: C; tab-width: 3; indent-tabs-mode: nil; c-basic-offset: 3 -*- */

/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: image_window.c,v 1.73.2.27 2003/06/03 16:31:35 makeinu Exp $
 */

#include <stdlib.h>
#include <gdk/gdkkeysyms.h>

#include "gimageview.h"

#include "charset.h"
#include "comment_view.h"
#include "cursors.h"
#include "gimv_icon_stock.h"
#include "gimv_prefs_win.h"
#include "gtk2-compat.h"
#include "gtkutils.h"
#include "help.h"
#include "image_view.h"
#include "image_window.h"
#include "menu.h"
#include "prefs.h"
#include "slideshow.h"
#include "thumbnail_window.h"

#define IMGWIN_FULLSCREEN_HIDE_CURSOR_DELAY 1000

enum {
   IMG_FIRST,
   IMG_PREV,
   IMG_NEXT,
   IMG_LAST
};

typedef enum {
   MOUSE_PRESS_NONE,
   MOUSE_PRESS_NEXT,
   MOUSE_PRESS_PREV,
   MOUSE_PRESS_POPUP,
   MOUSE_PRESS_ZOOM_IN,
   MOUSE_PRESS_ZOOM_OUT,
   MOUSE_PRESS_FIT,
   MOUSE_PRESS_ROTATE_CCW,
   MOUSE_PRESS_ROTATE_CW,
   MOUSE_PRESS_NAVWIN,
   MOUSE_PRESS_UP,
   MOUSE_PRESS_DOWN,
   MOUSE_PRESS_LEFT,
   MOUSE_PRESS_RIGHT
} ImageViewPressType;

static GList *ImageWinList = NULL;
static ImageWindow *shared_img_win;


static void  create_imageview_menus     (ImageWindow  *iw);
static GtkWidget *create_toolbar        (ImageWindow  *iw,
                                         GtkWidget    *container);

static void imagewin_set_win_size       (ImageWindow  *iw);
static void imagewin_set_window_title   (ImageWindow  *iw);
static void imagewin_set_statusbar_info (ImageWindow  *iw);

static void imagewin_fullscreen_new     (ImageWindow *iw);
static void imagewin_fullscreen_delete  (ImageWindow *iw);

/* callback functions for menubar */
static void cb_file_select          (gpointer     data,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_open_imagewin        (gpointer     data,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_open_thumbwin        (gpointer     data,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_toggle_buffer        (ImageWindow *iw,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_window_close         (ImageWindow *iw,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_ignore_alpha         (ImageWindow *iv,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_toggle_menubar       (ImageWindow *iw,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_toggle_player        (ImageWindow *iw,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_toggle_toolbar       (ImageWindow *iw,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_toggle_statusbar     (ImageWindow *iw,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_toggle_scrollbar     (ImageWindow *iw,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_switch_player        (ImageWindow *iw,
                                     ImageViewPlayerVisibleType visible,
                                     GtkWidget     *widget);
static void cb_toggle_maximize      (ImageWindow *iw,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_toggle_fullscreen    (ImageWindow *iw,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_fit_to_image         (ImageWindow *iw,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_edit_comment         (ImageWindow *iw,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_create_thumb         (ImageWindow *iw,
                                     guint        action,
                                     GtkWidget   *widget);
static void cb_options              (ImageWindow *iw);
static void cb_move_menu            (ImageWindow *iw,
                                     guint        action,
                                     GtkWidget   *widget);

/* callback functions for toolbar */
static void cb_toolbar_open_button     (GtkWidget   *widget);
static void cb_toolbar_prefs_button    (GtkWidget   *widget,
                                        ImageWindow *iw);
static void cb_toolbar_prev_button     (GtkWidget   *widget,
                                        ImageWindow *iw);
static void cb_toolbar_next_button     (GtkWidget   *widget,
                                        ImageWindow *iw);
static void cb_toolbar_no_zoom         (GtkWidget   *widget,
                                        ImageWindow *iw);
static void cb_toolbar_zoom_in         (GtkWidget   *widget,
                                        ImageWindow *iw);
static void cb_toolbar_zoom_out        (GtkWidget   *widget,
                                        ImageWindow *iw);
static void cb_toolbar_zoom_fit        (GtkWidget   *widget,
                                        ImageWindow *iw);
static void cb_toolbar_zoom            (GtkWidget   *widget,
                                        ImageWindow *iw);
static void cb_toolbar_keep_aspect     (GtkWidget   *widget,
                                        ImageWindow *iw);
static void cb_toolbar_fit_window      (GtkWidget   *widget,
                                        ImageWindow *iw);
static void cb_toolbar_fullscreen      (GtkWidget   *widget,
                                        ImageWindow *iw);
static gboolean cb_scale_spinner_key_press  (GtkWidget      *widget, 
                                             GdkEventKey    *event,
                                             ImageWindow    *iw);
static void     cb_rotate_menu              (GtkWidget      *widget,
                                             ImageWindow    *iw);
static gboolean cb_rotate_menu_button_press (GtkWidget      *widget,
                                             GdkEventButton *event,
                                             ImageWindow    *iw);

/* callback functions for slideshow toolbar */
static void cb_play_clicked  (GtkWidget   *button,
                              ImageWindow *iw);
static void cb_stop_clicked  (GtkWidget   *button,
                              ImageWindow *iw);
static void cb_prev_clicked  (GtkWidget   *button,
                              ImageWindow *iw);
static void cb_next_clicked  (GtkWidget   *button,
                              ImageWindow *iw);
static void cb_first_clicked (GtkWidget   *button,
                              ImageWindow *iw);
static void cb_last_clicked  (GtkWidget   *button,
                              ImageWindow *iw);

/* other callback functions */
static void     cb_window_closed       (GtkWidget   *widget,
                                        ImageWindow *iw);
static void     cb_set_list            (ImageView   *iv,
                                        ImageWindow *iw);
static void     cb_unset_list          (ImageView   *iv,
                                        ImageWindow *iw);
static void     cb_rendered            (ImageView   *iv,
                                        ImageWindow *iw);
static void     cb_toggle_aspect       (ImageView   *iv,
                                        gboolean     keep_aspect,
                                        ImageWindow *iw);
static gboolean cb_draw_area_key_press (GtkWidget   *widget, 
                                        GdkEventKey *event,
                                        ImageWindow *iw);
static gint     imageview_button_action(ImageView       *iv,
                                        GdkEventButton  *event,
                                        gint             num);
static gint     cb_imageview_pressed   (ImageView       *iv,
                                        GdkEventButton  *event,
                                        ImageWindow     *iw);
static gint     cb_imageview_clicked   (ImageView       *iv,
                                        GdkEventButton *event,
                                        ImageWindow    *iw);

/* for main menu */
GtkItemFactoryEntry imagewin_menu_items[] =
{
   {N_("/_File"),                        NULL,         NULL,              0,  "<Branch>"},
   {N_("/_File/_Open..."),               "<control>F", cb_file_select,    0,  NULL},
   {N_("/_File/Open _Image Window"),     "<control>I", cb_open_imagewin,  0,  NULL},
   {N_("/_File/Open _Thumbnail Window"), "<control>W", cb_open_thumbwin,  0,  NULL},
   {N_("/_File/---"),                    NULL,         NULL,              0,  "<Separator>"},
   {N_("/_File/Memory _Buffer"),         "<control>B", cb_toggle_buffer,  0,  "<ToggleItem>"},
   {N_("/_File/---"),                    NULL,         NULL,              0,  "<Separator>"},
   {N_("/_File/_Close"),                 "Q",          cb_window_close,   0,  NULL},
   {N_("/_File/_Quit"),                  "<control>C", gimv_quit,         0,  NULL},

   {N_("/_Edit"),                        NULL,         NULL,              0,  "<Branch>"},
   {N_("/_Edit/Edit _Comment..."),       NULL,         cb_edit_comment,   0,  NULL},
   {N_("/_Edit/Create _Thumbnail"),      "<shift>T",   cb_create_thumb,   0,  NULL},
   {N_("/_Edit/---"),                    NULL,         NULL,              0,  "<Separator>"},
   {N_("/_Edit/_Options..."),            "<control>O", cb_options,        0,  NULL},

   {N_("/_View"),                        NULL,         NULL,              0,  "<Branch>"},

   {N_("/_Move"),                        NULL,         NULL,              0,  "<Branch>"},

   {N_("/M_ovie"),                       NULL,         NULL,              0,  "<Branch>"},

   {N_("/_Help"),                        NULL,         NULL,              0,  "<LastBranch>"},
   {NULL, NULL, NULL, 0, NULL},
};


/* for "View" sub menu */
GtkItemFactoryEntry imagewin_view_items [] =
{
   {N_("/tear"),          NULL,        NULL,                 0,  "<Tearoff>"},
   {N_("/_Zoom"),         NULL,        NULL,                 0,  "<Branch>"},
   {N_("/_Rotate"),       NULL,        NULL,                 0,  "<Branch>"},
   {N_("/Ignore _Alpha Channel"), NULL,cb_ignore_alpha,      0,  "<ToggleItem>"},
   {N_("/---"),           NULL,        NULL,                 0,  "<Separator>"},
   {N_("/_Menu Bar"),     "M",         cb_toggle_menubar,    0,  "<ToggleItem>"},
   {N_("/_Tool Bar"),     "N",         cb_toggle_toolbar,    0,  "<ToggleItem>"},
   {N_("/Slide Show _Player"), "<shift>N", cb_toggle_player,     0,  "<ToggleItem>"},
   {N_("/St_atus Bar"),   "V",         cb_toggle_statusbar,  0,  "<ToggleItem>"},
   {N_("/_Scroll Bar"),   "<shift>S",  cb_toggle_scrollbar,  0,  "<ToggleItem>"},
   {N_("/_Player"),       NULL,        NULL,                 0,              "<Branch>"},  
   {N_("/_Player/_Show"), NULL,        cb_switch_player,     ImageViewPlayerVisibleShow, "<RadioItem>"},  
   {N_("/_Player/_Hide"), NULL,        cb_switch_player,     ImageViewPlayerVisibleHide, "/Player/Show"},  
   {N_("/_Player/_Auto"), NULL,        cb_switch_player,     ImageViewPlayerVisibleAuto, "/Player/Hide"},  
   {N_("/---"),           NULL,        NULL,                 0,  "<Separator>"},
   {N_("/_View modes"),   NULL,        NULL,                 0, "<Branch>"},
   {N_("/---"),           NULL,        NULL,                 0,  "<Separator>"},
   {N_("/_Fit to Image"), "I",         cb_fit_to_image,      0,  NULL},
   {N_("/Ma_ximize"),     "F",         cb_toggle_maximize,   0,  "<ToggleItem>"},
   {N_("/F_ull Screen"),  "<shift>F",  cb_toggle_fullscreen, 0,  "<ToggleItem>"},
   {NULL, NULL, NULL, 0, NULL},
};


GtkItemFactoryEntry imagewin_move_items [] =
{
   {N_("/_Next"),        "L",  cb_move_menu,  IMG_NEXT,  NULL},
   {N_("/_Previous"),    "H",  cb_move_menu,  IMG_PREV,  NULL},
   {N_("/_First"),       "K",  cb_move_menu,  IMG_FIRST, NULL},
   {N_("/_Last"),        "J",  cb_move_menu,  IMG_LAST,  NULL},
   {NULL, NULL, NULL, 0, NULL},
};


/*
 *  create_imageview_menus:
 *     @ create main menu and popup menu.
 *
 *  im : Pointer to ImageWindow structure.
 */
static void
create_imageview_menus (ImageWindow *iw)
{
   guint n_menu_items;
   GtkWidget *zoom_menu, *rotate_menu, *movie_menu, *vmode_menu;
   gboolean show_scrollbar, keep_buffer;

   /* main menu */
   n_menu_items = sizeof (imagewin_menu_items)
      / sizeof (imagewin_menu_items[0]) - 1;
   iw->menubar = menubar_create (iw->window, imagewin_menu_items, n_menu_items,
                                 "<ImageWinMainMenu>", iw);
   gtk_container_add(GTK_CONTAINER(iw->menubar_handle), iw->menubar);
#ifndef USE_GTK2
   gtk_menu_bar_set_shadow_type (GTK_MENU_BAR(iw->menubar), GTK_SHADOW_NONE);
#endif /* USE_GTK2 */
   gtk_widget_show (iw->menubar);

   /* sub menu */
   n_menu_items = sizeof(imagewin_view_items)
      / sizeof(imagewin_view_items[0]) - 1;
   iw->view_menu = menu_create_items(iw->window, imagewin_view_items,
                                     n_menu_items, "<ImageWinViewSubMenu>", iw);
   n_menu_items = sizeof(imagewin_move_items)
      / sizeof(imagewin_move_items[0]) - 1;
   iw->move_menu = menu_create_items(iw->window, imagewin_move_items,
                                     n_menu_items, "<ImageWinMoveSubMenu>", iw);

   zoom_menu   = imageview_create_zoom_menu   (iw->window,
                                               iw->iv,
                                               "<ImageWinZoomSubMenu>");
   rotate_menu = imageview_create_rotate_menu (iw->window,
                                               iw->iv,
                                               "<ImageWinRotateSubMenu>");

   movie_menu  = imageview_create_movie_menu (iw->window,
                                              iw->iv,
                                              "<ImageWinMovieSubMenu>");
   vmode_menu  = imageview_create_view_modes_menu (iw->window,
                                                   iw->iv,
                                                   "<ImageWinViewModesSubMenu>");
   iw->help_menu = gimvhelp_create_menu (iw->window);

   /* attach sub menus to parent menu */
   menu_set_submenu (iw->menubar,   "/View",       iw->view_menu);
   menu_set_submenu (iw->menubar,   "/Move",       iw->move_menu);
   menu_set_submenu (iw->menubar,   "/Help",       iw->help_menu);
   menu_set_submenu (iw->view_menu, "/Zoom",       zoom_menu);
   menu_set_submenu (iw->view_menu, "/Rotate",     rotate_menu);
   menu_set_submenu (iw->menubar,   "/Movie",      movie_menu);
   menu_set_submenu (iw->view_menu, "/View modes", vmode_menu);

   /* popup menu */
   iw->iv->imageview_popup = iw->view_menu;

   /* initialize menubar check items */
   gtk_object_get (GTK_OBJECT (iw->iv),
                   "show_scrollbar", &show_scrollbar,
                   "keep_buffer",    &keep_buffer,
                   NULL);

   menu_check_item_set_active (iw->menubar, "/File/Memory Buffer", keep_buffer);

   menu_check_item_set_active (iw->view_menu, "/Menu Bar",
                               iw->flags & ImgWinShowMenuBarFlag);
   menu_check_item_set_active (iw->view_menu, "/Tool Bar",
                               iw->flags & ImgWinShowToolBarFlag);
   menu_check_item_set_active (iw->view_menu, "/Slide Show Player",
                               iw->flags & ImgWinShowPlayerFlag);
   menu_check_item_set_active (iw->view_menu, "/Status Bar",
                               iw->flags & ImgWinShowStatusBarFlag);
   menu_check_item_set_active (iw->view_menu, "/Scroll Bar",  show_scrollbar);
   menu_check_item_set_active (iw->view_menu, "/Maximize",
                               iw->flags & ImgWinMaximizeFlag);
   menu_check_item_set_active (iw->view_menu, "/Full Screen",
                               iw->flags & ImgWinFullScreenFlag);

   menu_item_set_sensitive (iw->move_menu, "/Next",     FALSE);
   menu_item_set_sensitive (iw->move_menu, "/Previous", FALSE);
   menu_item_set_sensitive (iw->move_menu, "/First",    FALSE);
   menu_item_set_sensitive (iw->move_menu, "/Last",     FALSE);

   switch (iw->player_visible) {
   case ImageViewPlayerVisibleShow:
      menu_check_item_set_active (iw->view_menu, "/Player/Show", TRUE);
      break;
   case ImageViewPlayerVisibleHide:
      menu_check_item_set_active (iw->view_menu, "/Player/Hide", TRUE);
      break;
   case ImageViewPlayerVisibleAuto:
   default:
      menu_check_item_set_active (iw->view_menu, "/Player/Auto", TRUE);
      break;
   }
}


/*
 *  create_toolbar:
 *     @ create toolbar and buttons.
 *
 *  im        : Pointer to ImageWindow structure.
 *  container : Container to store toolbar.
 *  Return    : Pointer to toolbar widget.
 */
static GtkWidget *
create_toolbar (ImageWindow *iw, GtkWidget *container)
{
   GtkWidget *toolbar, *button, *spinner, *iconw, *menu;
   GtkAdjustment *adj;
   const gchar *rotate_labels[] = {
      N_("0"),
      N_("90"),
      N_("180"),
      N_("270"),
      NULL,
   };
   gfloat x_scale, y_scale;

   toolbar = gtkutil_create_toolbar ();

   /* file open button */
   iconw = gimv_icon_stock_get_widget ("nfolder");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                                    _("Open"),
                                    _("File Open"),
                                    _("File Open"),
                                    iconw,
                                    GTK_SIGNAL_FUNC (cb_toolbar_open_button),
                                    NULL);

   /* preference button */
   iconw = gimv_icon_stock_get_widget ("prefs");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                                    _("Prefs"),
                                    _("Preference"),
                                    _("Preference"),
                                    iconw,
                                    GTK_SIGNAL_FUNC (cb_toolbar_prefs_button),
                                    iw);

   gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));

   /* prev button */
   iconw = gimv_icon_stock_get_widget ("back");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                                    _("Prev"),
                                    _("Previous Image"),
                                    _("Previous Image"),
                                    iconw,
                                    GTK_SIGNAL_FUNC (cb_toolbar_prev_button),
                                    iw);
   iw->button.prev = button;
   /* gtk_widget_set_sensitive (button, FALSE); */

   /* next button */
   iconw = gimv_icon_stock_get_widget ("forward");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                                    _("Next"),
                                    _("Next Image"),
                                    _("Next Image"),
                                    iconw,
                                    GTK_SIGNAL_FUNC (cb_toolbar_next_button),
                                    iw);
   iw->button.next = button;
   /* gtk_widget_set_sensitive (button, FALSE); */

   gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));

   /* no zoom button */
   iconw = gimv_icon_stock_get_widget ("no_zoom");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                                    _("No Zoom"),
                                    _("No Zoom"),
                                    _("No Zoom"),
                                    iconw,
                                    GTK_SIGNAL_FUNC (cb_toolbar_no_zoom),
                                    iw);

   /* zoom in button */
   iconw = gimv_icon_stock_get_widget ("zoom_in");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                                    _("Zoom in"),
                                    _("Zoom in"),
                                    _("Zoom in"),
                                    iconw,
                                    GTK_SIGNAL_FUNC (cb_toolbar_zoom_in),
                                    iw);

   /* zoom out button */
   iconw = gimv_icon_stock_get_widget ("zoom_out");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                                    _("Zoom out"),
                                    _("Zoom out"),
                                    _("Zoom out"),
                                    iconw,
                                    GTK_SIGNAL_FUNC (cb_toolbar_zoom_out),
                                    iw);

   /* zoom fit button */
   iconw = gimv_icon_stock_get_widget ("zoom_fit");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                                    _("Zoom fit"),
                                    _("Zoom fit"),
                                    _("Zoom fit"),
                                    iconw,
                                    GTK_SIGNAL_FUNC (cb_toolbar_zoom_fit),
                                    iw);

   gtk_object_get (GTK_OBJECT (iw->iv),
                   "x_scale", &x_scale,
                   "y_scale", &y_scale,
                   NULL);

   /* x scale spinner */
   adj = (GtkAdjustment *) gtk_adjustment_new (x_scale,
                                               MIN_THUMB_SIZE, MAX_THUMB_SIZE,
                                               1.0, 5.0, 0.0);
   spinner = gtkutil_create_spin_button (adj);
   gtk_widget_set_name (spinner, "XScaleSpinner");
   gtk_toolbar_append_widget (GTK_TOOLBAR (toolbar), spinner,
                              _("X Scale"), _("X Scale"));
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
                       GTK_SIGNAL_FUNC (cb_toolbar_keep_aspect), iw);
   iw->button.xscale = spinner;
   gtk_signal_connect (GTK_OBJECT(spinner), "key-press-event",
                       GTK_SIGNAL_FUNC(cb_scale_spinner_key_press), iw);

   /* y scale spinner */
   adj = (GtkAdjustment *) gtk_adjustment_new (y_scale,
                                               MIN_THUMB_SIZE, MAX_THUMB_SIZE,
                                               1.0, 5.0, 0.0);
   spinner = gtkutil_create_spin_button (adj);
   gtk_widget_set_name (spinner, "YScaleSpinner");
   gtk_toolbar_append_widget (GTK_TOOLBAR (toolbar), spinner,
                              _("Y Scale"), _("Y Scale"));
   iw->button.yscale = spinner;

   /* zoom button */
   iconw = gimv_icon_stock_get_widget ("zoom");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                                    _("Zoom"),
                                    _("Zoom"),
                                    _("Zoom"),
                                    iconw,
                                    GTK_SIGNAL_FUNC (cb_toolbar_zoom),
                                    iw);

   gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));

   /* rotate button */
   menu = create_option_menu (rotate_labels,
                              0, cb_rotate_menu, iw);
   gtk_signal_connect (GTK_OBJECT (menu), "button_press_event",
                       GTK_SIGNAL_FUNC (cb_rotate_menu_button_press), iw);
   gtk_toolbar_append_widget (GTK_TOOLBAR (toolbar), menu,
                              _("Rotate"), _("Rotate the image"));
   iw->button.rotate = menu;

   gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));

   /* resize button */
   iconw = gimv_icon_stock_get_widget ("resize");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                                    _("Resize"),
                                    _("Fit Window Size to Image"),
                                    _("Fit Window Size to Image"),
                                    iconw,
                                    GTK_SIGNAL_FUNC (cb_toolbar_fit_window),
                                    iw);

   /* fullscreen button */
   iconw = gimv_icon_stock_get_widget ("fullscreen");
   /* button = gtk_toggle_button_new (); */
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                                    _("Full"),
                                    _("Fullscreen"),
                                    _("Fullscreen"),
                                    iconw,
                                    GTK_SIGNAL_FUNC (cb_toolbar_fullscreen),
                                    iw);

   gtk_widget_set_sensitive (iw->button.prev, FALSE);
   gtk_widget_set_sensitive (iw->button.next, FALSE);

   return toolbar;
}


/*
 *  create_player_toolbar:
 *     @ create toolbar and buttons.
 *
 *  im        : Pointer to ImageWindow structure.
 *  container : Container to store toolbar.
 *  Return    : Pointer to toolbar widget.
 */
static GtkWidget *
create_player_toolbar (ImageWindow *iw, GtkWidget *container)
{
   GtkWidget *toolbar;
   GtkWidget *button;
   GtkWidget *iconw;

   toolbar = gtkutil_create_toolbar ();

   /* previous button */
   iconw = gimv_icon_stock_get_widget ("prev_t");
   button = gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
                                     _("First"),
                                     _("First"), _("First"),
                                     iconw,
                                     GTK_SIGNAL_FUNC (cb_first_clicked), iw);
   iw->player.prev = button;

   /* Reverse button */
   iconw = gimv_icon_stock_get_widget ("rw");
   button = gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
                                     _("Prev"),
                                     _("Previous"), _("Previous"),
                                     iconw,
                                     GTK_SIGNAL_FUNC (cb_prev_clicked), iw);
   iw->player.rw = button;

   /* play button */
   iconw = gimv_icon_stock_get_widget ("play");
   button = gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
                                     _("Play"),
                                     _("Play Slide Show"), _("Play Slide Show"),
                                     iconw,
                                     GTK_SIGNAL_FUNC (cb_play_clicked), iw);
   iw->player.play = button;

   /* stop button */
   iconw = gimv_icon_stock_get_widget ("stop2");
   button = gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
                                     _("Stop"),
                                     _("Stop Slide Show"), _("Stop Slide Show"),
                                     iconw,
                                     GTK_SIGNAL_FUNC (cb_stop_clicked), iw);
   iw->player.stop = button;

   /* Forward button */
   iconw = gimv_icon_stock_get_widget ("ff");
   button = gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
                                     _("Next"),
                                     _("Next"), _("Next"),
                                     iconw,
                                     GTK_SIGNAL_FUNC (cb_next_clicked), iw);
   iw->player.fw = button;

   /* Next button */
   iconw = gimv_icon_stock_get_widget ("next_t");
   button = gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
                                     _("Last"),
                                     _("Last"), _("Last"),
                                     iconw,
                                     GTK_SIGNAL_FUNC (cb_last_clicked), iw);
   iw->player.next = button;

   return toolbar;
}


/*
 *  imageview_set_win_size:
 *     @ Set image window size to image size.
 *
 *  im    : Pointer to ImageWindow structure.
 */
static void
imagewin_set_win_size (ImageWindow *iw)
{
   gint x_size, y_size, width, height;
   gfloat x_scale, y_scale, tmp;
   gboolean show_scrollbar;
   ImageViewOrientation rotate;

   if (!iw->window) return;
   if (iw->flags & ImgWinMaximizeFlag) return;
   if (iw->flags & ImgWinFullScreenFlag) return;

   gtk_object_get (GTK_OBJECT (iw->iv),
                   "x_scale", &x_scale,
                   "y_scale", &y_scale,
                   NULL);
   if (x_scale < 0.001 || y_scale < 0.001) return;

   rotate = imageview_get_orientation (iw->iv);
   if (rotate == 0 || rotate == 2) {
      width =  iw->iv->info->width;
      height = iw->iv->info->height;
   } else {
      tmp = x_scale;
      x_scale = y_scale;
      y_scale = tmp;
      width  = iw->iv->info->height;
      height = iw->iv->info->width;
   }

   x_size = width  * x_scale / 100.0 + 0.5;
   y_size = height * y_scale / 100.0 + 0.5;

   if (iw->flags & ImgWinShowMenuBarFlag)
      y_size += iw->menubar_handle->allocation.height;

   if (iw->flags & ImgWinShowToolBarFlag)
      y_size += iw->toolbar_handle->allocation.height;

   if (iw->flags & ImgWinShowPlayerFlag)
      y_size += iw->player_handle->allocation.height;

   if (iw->flags & ImgWinShowStatusBarFlag)
      y_size += iw->status_bar_container->allocation.height;

   gtk_object_get (GTK_OBJECT (iw->iv),
                   "show_scrollbar", &show_scrollbar,
                   NULL);
   if (show_scrollbar) {
      x_size += iw->iv->vscrollbar->allocation.width;
      y_size += iw->iv->hscrollbar->allocation.height;
   }

   if (x_size < 1)
      x_size = conf.imgwin_width;
   if (y_size < 1)
      y_size = conf.imgwin_height;

   gdk_window_resize (iw->window->window, x_size, y_size);
   imageview_set_view_position (iw->iv, 0, 0);
   imageview_draw_image (iw->iv);
}


/*
 *  imageview_set_window_title:
 *     @ set window title (using image file name).
 *
 *  im : Pointer to ImageWindow structure.
 */
static void
imagewin_set_window_title (ImageWindow *iw)
{
   gchar buf[BUF_SIZE];
   const gchar *filename = NULL;
   gchar *dirname = NULL, *tmpstr1, *tmpstr2;
   gboolean keep_buffer;

   g_return_if_fail (iw);
   if (!g_list_find (ImageWinList, iw)) return;

   g_return_if_fail (iw->iv);
   if (!g_list_find (imageview_get_list(), iw->iv)) return;

   if (iw->iv->info && image_info_get_path (iw->iv->info)) {
      filename = g_basename (iw->iv->info->filename);
      dirname = g_dirname (iw->iv->info->filename);
   } else {
      gtk_window_set_title (GTK_WINDOW (iw->window), GIMV_PROG_NAME);
      return;
   }

   tmpstr1 = charset_to_internal (filename, conf.charset_filename,
                                  conf.charset_auto_detect_fn,
                                  conf.charset_filename_mode);
   tmpstr2 = charset_to_internal (dirname, conf.charset_filename,
                                  conf.charset_auto_detect_fn,
                                  conf.charset_filename_mode);

   gtk_object_get (GTK_OBJECT (iw->iv),
                   "keep_buffer", &keep_buffer,
                   NULL);

   if (tmpstr1 && tmpstr2 && *tmpstr1 && *tmpstr2) {
      if (keep_buffer)
         g_snprintf (buf, BUF_SIZE, _("%s (%s) - %s - "),
                     tmpstr1, tmpstr2, GIMV_PROG_NAME);
      else
         g_snprintf (buf, BUF_SIZE, _("* %s (%s) - %s - *"),
                     tmpstr1, tmpstr2, GIMV_PROG_NAME);
      gtk_window_set_title (GTK_WINDOW (iw->window), buf);
   } else {
      if (keep_buffer)
         g_snprintf (buf, BUF_SIZE, _("- %s -"), GIMV_PROG_NAME);
      else
         g_snprintf (buf, BUF_SIZE, _("* - %s - *"), GIMV_PROG_NAME);
      gtk_window_set_title (GTK_WINDOW (iw->window), buf);
   }

   g_free (tmpstr1);
   g_free (tmpstr2);
   g_free (dirname);
}


/*
 *  imageview_set_statusbar_info:
 *     @ set statubar strings.
 *
 *  im : Pointer to ImageWindow structure.
 */
static void
imagewin_set_statusbar_info (ImageWindow *iw)
{
   gchar buf[BUF_SIZE], *tmpstr;
   const gchar *filename = _("NONE");
   const gchar *format = _("UNKNOWN");
   gint width = 0, height = 0;
   gboolean keep_buffer;

   g_return_if_fail (iw);
   if (!g_list_find (ImageWinList, iw)) return;

   g_return_if_fail (iw->iv);
   if (!g_list_find (imageview_get_list(), iw->iv)) return;

   if (iw->iv->info) {
      filename = image_info_get_path (iw->iv->info);
      format = image_info_get_format (iw->iv->info);
      image_info_get_image_size (iw->iv->info, &width, &height);
   }

   tmpstr = charset_to_internal (filename, conf.charset_filename,
                                 conf.charset_auto_detect_fn,
                                 conf.charset_filename_mode);

   gtk_object_get (GTK_OBJECT (iw->iv),
                   "keep_buffer", &keep_buffer,
                   NULL);

   if (tmpstr && *tmpstr) {
      if (keep_buffer)
         g_snprintf (buf, BUF_SIZE, _("%s (Buffer ON)"), tmpstr);
      else
         g_snprintf (buf, BUF_SIZE, _("%s (Buffer OFF)"), tmpstr);

      gtk_statusbar_push (GTK_STATUSBAR (iw->status_bar1), 1, buf);
   } else {
      if (keep_buffer)
         g_snprintf (buf, BUF_SIZE, _("(Buffer ON)"));
      else
         g_snprintf (buf, BUF_SIZE, _("(Buffer OFF)"));
      gtk_statusbar_push (GTK_STATUSBAR (iw->status_bar1), 1, buf);
   }
   g_free (tmpstr);

   g_snprintf (buf, BUF_SIZE, "%s %d x %d", format, width, height);
   gtk_statusbar_push (GTK_STATUSBAR (iw->status_bar2), 1, buf);
}



/******************************************************************************
 *
 *   Callback functions for menubar.
 *
 ******************************************************************************/
/*
 *  cb_file_select:
 *     @ Callback function for main menu item (/File/Open).
 *       Open file browser.
 *
 *  data   :
 *  action :
 *  widget :
 */
static void
cb_file_select (gpointer data, guint action, GtkWidget *widget)
{
   create_filebrowser (NULL);
}


static void
cb_open_imagewin (gpointer data, guint action, GtkWidget *widget)
{
   imagewin_open_window (NULL);   
}


/*
 *  cb_open_thumbwin:
 *     @ Callback function for main menu item (/File/Open Thumbnail Window).
 *       Open new thumbnail window.
 *
 *  data   :
 *  action :
 *  widget :
 */
static void
cb_open_thumbwin (gpointer data, guint action, GtkWidget *widget)
{
   thumbwin_open_window();
}


/*
 *  cb_toggle_buffer:
 *     @ Callback function for main menu item (/File/Memory Buffer).
 *       Keep original image on memory or not.
 *
 *  im     : Pointer to ImageWindow structure.
 *  action :
 *  widget :
 */
static void
cb_toggle_buffer (ImageWindow *iw, guint action, GtkWidget *widget)
{
   gboolean keep_buffer;

   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      imageview_load_image_buf (iw->iv);
      keep_buffer = TRUE;
   } else {
      keep_buffer = FALSE;
      imageview_free_image_buf (iw->iv);
   }

   gtk_object_set (GTK_OBJECT (iw->iv),
                   "keep_buffer", keep_buffer,
                   NULL);

   imagewin_set_window_title (iw);
   imagewin_set_statusbar_info (iw);
}


/*
 *  cb_window_close:
 *     @ Callback function for main menu item (/File/Close).
 *       Image window close.
 *
 *  im     : Pointer to ImageWindow structure.
 *  action :
 *  widget :
 */
static void
cb_window_close (ImageWindow *iw, guint action, GtkWidget *widget)
{
   if (iw->flags & ImgWinCreatingFlag) return;

   if (iw->flags & ImgWinFullScreenFlag) {
      menu_check_item_set_active (iw->view_menu, "/Full Screen", FALSE);
      return; /* or close window completly? */
   }

   gtk_widget_destroy (iw->window);
}


static void
cb_ignore_alpha (ImageWindow *iw, guint action, GtkWidget *widget)
{
   gtk_object_set (GTK_OBJECT (iw->iv),
                   "ignore_alpha", GTK_CHECK_MENU_ITEM (widget)->active,
                   NULL);
   imageview_show_image (iw->iv);
}


/*
 *  cb_toggle_menubar:
 *     @ Callback function for main menu item (/View/Menu Bar).
 *       Show/Hide menu bar.
 *
 *  im     : Pointer to ImageWindow structure.
 *  action :
 *  widget :
 */
static void
cb_toggle_menubar (ImageWindow *iw, guint action, GtkWidget *widget)
{
   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      gtk_widget_show (iw->menubar_handle);
      iw->flags |= ImgWinShowMenuBarFlag;
   } else {
      gtk_widget_hide (iw->menubar_handle);
      iw->flags &= ~ImgWinShowMenuBarFlag;
   }
}


/*
 *  cb_toggle_player:
 *     @ Callback function for main menu item (/View/Slide Show Player).
 *       Show/Hide Player toolbar.
 *
 *  im     : Pointer to ImageWindow structure.
 *  action :
 *  widget :
 */
static void
cb_toggle_player (ImageWindow *iw, guint action, GtkWidget *widget)
{
   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      gtk_widget_show (iw->player_handle);
      iw->flags |= ImgWinShowPlayerFlag;
   } else {
      gtk_widget_hide (iw->player_handle);
      iw->flags &= ~ImgWinShowPlayerFlag;
   }
}


/*
 *  cb_toggle_toolbar:
 *     @ Callback function for main menu item (/View/Tool Bar).
 *       Show/Hide tool bar.
 *
 *  im     : Pointer to ImageWindow structure.
 *  action :
 *  widget :
 */
static void
cb_toggle_toolbar (ImageWindow *iw, guint action, GtkWidget *widget)
{
   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      gtk_widget_show (iw->toolbar_handle);
      iw->flags |= ImgWinShowToolBarFlag;
   } else {
      gtk_widget_hide (iw->toolbar_handle);
      iw->flags &= ~ImgWinShowToolBarFlag;
   }
}


static void
cb_toggle_statusbar (ImageWindow *iw, guint action, GtkWidget *widget)
{
   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      gtk_widget_show (iw->status_bar_container);
      iw->flags |= ImgWinShowStatusBarFlag;
   } else {
      gtk_widget_hide (iw->status_bar_container);
      iw->flags &= ~ImgWinShowStatusBarFlag;
   }
}


/*
 *  cb_toggle_statusbar:
 *     @ Callback function for main menu item (/View/Status Bar).
 *       Show/Hide status bar.
 *
 *  im     : Pointer to ImageWindow structure.
 *  action :
 *  widget :
 */
static void
cb_toggle_scrollbar (ImageWindow *iw, guint action, GtkWidget *widget)
{
   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      imageview_show_scrollbar (iw->iv);
   } else {
      imageview_hide_scrollbar (iw->iv);
   }
}


static void
cb_switch_player (ImageWindow *iw,
                  ImageViewPlayerVisibleType visible,
                  GtkWidget     *widget)
{
   g_return_if_fail (iw);
   g_return_if_fail (IS_IMAGEVIEW (iw->iv));

   imageview_set_player_visible (iw->iv, visible);
   iw->player_visible = imageview_get_player_visible (iw->iv);
}


/*
 *  cb_toggle_maximize:
 *     @ Callback function for main menu item (/View/Maximize).
 *
 *  im     : Pointer to ImageWindow structure.
 *  action :
 *  widget :
 */
static void
cb_toggle_maximize (ImageWindow *iw, guint action, GtkWidget *widget)
{
   GdkWindow *gdk_window = iw->window->window;
   gint client_x, client_y, root_x, root_y;
   gint width, height;

   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      gdk_window_get_origin (gdk_window, &root_x, &root_y);
      gdk_window_get_geometry (gdk_window, &client_x, &client_y,
                               &width, &height, NULL);

      gdk_window_move_resize (gdk_window, -client_x, -client_y,
                              gdk_screen_width (), gdk_screen_height ());
      gdk_window_raise (gdk_window);
      gdk_window_focus (gdk_window, GDK_CURRENT_TIME);

      iw->flags |= ImgWinMaximizeFlag;
      iw->win_x = root_x - client_x;
      iw->win_y = root_y - client_y;
      iw->win_width  = width;
      iw->win_height = height;
   } else {
      gdk_window_move_resize (gdk_window, iw->win_x, iw->win_y,
                              iw->win_width, iw->win_height);
      iw->flags &= ~ImgWinMaximizeFlag;
   }
}


static void
cb_toggle_fullscreen (ImageWindow *iw, guint action, GtkWidget *widget)
{
   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      imagewin_fullscreen_new (iw);
   } else {
      imagewin_fullscreen_delete (iw);
   }
}


/*
 *  cb_fit_to_image:
 *     @ Callback function for main menu item (/View/Fit to image).
 *     @ Fit window size to image.
 */
static void
cb_fit_to_image (ImageWindow *iw, guint action, GtkWidget *widget)
{
   imagewin_set_win_size (iw);
}


static void
cb_edit_comment (ImageWindow *iw, guint action, GtkWidget *menuitem)
{
   g_return_if_fail (iw);
   g_return_if_fail (iw->iv);

   if (iw->iv->info)
      comment_view_create_window (iw->iv->info);
}


static void
cb_create_thumb (ImageWindow *iw,
                 guint        action,
                 GtkWidget   *widget)
{
   g_return_if_fail (iw);
   g_return_if_fail (iw->iv);

   imageview_create_thumbnail (iw->iv);
}


/*
 *  cb_options:
 *     @ Callback function for main menu item (/Preference/Options).
 *     @ Open preference window.
 */
static void
cb_options (ImageWindow *iw)
{
   gimv_prefs_win_open_idle ("/Image Window", GTK_WINDOW (iw->window));
}


static void
cb_move_menu (ImageWindow *iw,
              guint        action,
              GtkWidget   *widget)
{
   gint num;

   g_return_if_fail (iw);

   switch (action) {
   case IMG_FIRST:
      imageview_nth (iw->iv, 0);
      break;
   case IMG_PREV:
      imageview_prev (iw->iv);
      break;
   case IMG_NEXT:
      imageview_next (iw->iv);
      break;
   case IMG_LAST:
      if (!imageview_has_list (iw->iv)) return;
      num = imageview_image_list_length (iw->iv);
      imageview_nth (iw->iv, num - 1);
      break;
   default:
      break;
   }
}


/******************************************************************************
 *
 *  callback functions for toolbar.
 *
 ******************************************************************************/
static void
cb_toolbar_open_button (GtkWidget *widget)
{
   create_filebrowser (NULL);
}


static void
cb_toolbar_prefs_button (GtkWidget *widget, ImageWindow *iw)
{
   gimv_prefs_win_open_idle ("/Image Window", GTK_WINDOW (iw->window));
}


static void
cb_toolbar_prev_button  (GtkWidget *widget, ImageWindow *iw)
{
   g_return_if_fail (iw);
   imageview_prev (iw->iv);
}


static void
cb_toolbar_next_button  (GtkWidget *widget, ImageWindow *iw)
{
   g_return_if_fail (iw);
   imageview_next (iw->iv);
}


static void
cb_toolbar_no_zoom (GtkWidget *widget, ImageWindow *iw)
{
   g_return_if_fail (iw);
   imageview_zoom_image (iw->iv, IMAGEVIEW_ZOOM_100, 0, 0);
}


static void
cb_toolbar_zoom_in (GtkWidget *widget, ImageWindow *iw)
{
   g_return_if_fail (iw);
   imageview_zoom_image (iw->iv, IMAGEVIEW_ZOOM_IN, 0, 0);
}


static void
cb_toolbar_zoom_out (GtkWidget *widget, ImageWindow *iw)
{
   g_return_if_fail (iw);
   imageview_zoom_image (iw->iv, IMAGEVIEW_ZOOM_OUT, 0, 0);
}


static void
cb_toolbar_zoom_fit (GtkWidget *widget, ImageWindow *iw)
{
   g_return_if_fail (iw);
   imageview_zoom_image (iw->iv, IMAGEVIEW_ZOOM_FIT, 0, 0);
}


static void
cb_toolbar_zoom (GtkWidget *widget, ImageWindow *iw)
{
   gfloat x_scale, y_scale;
   GtkSpinButton *xspin, *yspin;

   g_return_if_fail (iw);

   xspin = GTK_SPIN_BUTTON(iw->button.xscale);
   yspin = GTK_SPIN_BUTTON(iw->button.yscale);

   x_scale = gtk_spin_button_get_value_as_float (xspin);
   y_scale = gtk_spin_button_get_value_as_float (yspin);

   imageview_zoom_image (iw->iv, IMAGEVIEW_ZOOM_FREE, x_scale, y_scale);
}


static void
cb_toolbar_keep_aspect (GtkWidget *widget, ImageWindow *iw)
{
   gfloat scale;
   gboolean keep_aspect;

   g_return_if_fail (iw);

   gtk_object_get (GTK_OBJECT (iw->iv),
                   "keep_aspect", &keep_aspect,
                   NULL);

   if (!keep_aspect)
      return;

   scale = gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (iw->button.xscale));
   gtk_spin_button_set_value (GTK_SPIN_BUTTON (iw->button.yscale), scale);
}


static void
cb_toolbar_fit_window (GtkWidget *widget, ImageWindow *iw)
{
   g_return_if_fail (iw);
   imagewin_set_win_size (iw);
}


static void
cb_toolbar_fullscreen (GtkWidget *widget, ImageWindow *iw)
{
   g_return_if_fail (iw);
   menu_check_item_set_active (iw->view_menu, "/Full Screen",
                               !(iw->flags & ImgWinFullScreenFlag));
}


static gboolean
cb_scale_spinner_key_press (GtkWidget *widget, 
                            GdkEventKey *event,
                            ImageWindow *iw)
{
   g_return_val_if_fail (iw, FALSE);
        
   switch (event->keyval) {
   case GDK_Escape:
      gtk_window_set_focus (GTK_WINDOW (iw->window), NULL);
      return FALSE;
   }

   return FALSE;
}


static void
cb_rotate_menu (GtkWidget *widget, ImageWindow *iw)
{
   gint angle;
   g_return_if_fail (GTK_IS_MENU_ITEM (widget));
   angle = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (widget), "num"));
   imageview_rotate_image (iw->iv, angle);
}


static gboolean
cb_rotate_menu_button_press (GtkWidget *widget,
                             GdkEventButton *event,
                             ImageWindow *iw)
{
   g_return_val_if_fail (GTK_IS_OPTION_MENU (widget), FALSE);

   switch (event->button) {
   case 2:
      imageview_rotate_ccw (iw->iv);
      return TRUE;
   case 3:
      imageview_rotate_cw (iw->iv);
      return TRUE;
   default:
      break;
   }

   return FALSE;
}



/******************************************************************************
 *
 *  callback functions for slideshow player
 *
 ******************************************************************************/
static void
cb_play_clicked (GtkWidget *button, ImageWindow *iw)
{
   g_return_if_fail (iw);

   imagewin_slideshow_play (iw);
}


static void
cb_stop_clicked (GtkWidget *button, ImageWindow *iw)
{
   g_return_if_fail (iw);

   imagewin_slideshow_stop (iw);
}


static void
cb_prev_clicked (GtkWidget *button, ImageWindow *iw)
{
   g_return_if_fail (iw);

   imageview_prev (iw->iv);
}


static void
cb_next_clicked (GtkWidget *button, ImageWindow *iw)
{
   g_return_if_fail (iw);

   imageview_next (iw->iv);
}


static void
cb_first_clicked (GtkWidget *button, ImageWindow *iw)
{
   g_return_if_fail (iw);

   imageview_nth (iw->iv, 0);
}


static void
cb_last_clicked (GtkWidget *button, ImageWindow *iw)
{
   gint num;

   g_return_if_fail (iw);
   g_return_if_fail (IS_IMAGEVIEW (iw->iv));
   g_return_if_fail (imageview_has_list (iw->iv));

   num = imageview_image_list_length (iw->iv);
   imageview_nth (iw->iv, num - 1);
}


static gboolean
cb_seekbar_pressed (GtkWidget *widget,
                    GdkEventButton *event,
                    ImageWindow *iw)
{
   g_return_val_if_fail (iw, FALSE);

   iw->flags |= ImgWinSlideShowSeekBarDraggingFlag;

   return FALSE;
}


static gboolean
cb_seekbar_released (GtkWidget *widget,
                     GdkEventButton *event,
                     ImageWindow *iw)
{
   GtkAdjustment *adj;
   gint pos;

   g_return_val_if_fail (iw, FALSE);

   adj = gtk_range_get_adjustment (GTK_RANGE (iw->player.seekbar));
   pos = adj->value + 0.5;

   imageview_nth (iw->iv, pos);

   iw->flags &= ~ImgWinSlideShowSeekBarDraggingFlag;

   return FALSE;
}



/******************************************************************************
 *
 *  other callback functions.
 *
 ******************************************************************************/
/*
 *  cb_window_closed:
 *     @ Callback function for Image Window destroy event.
 *
 *  widget :
 *  im     : Pointer to ImageWindow structure.
 */
static void
cb_window_closed (GtkWidget *widget, ImageWindow *iw)
{
   g_return_if_fail (iw);

   imagewin_slideshow_stop (iw);

   if (g_list_length (ImageWinList) == 1 && conf.imgwin_save_win_state)
      imagewin_store_win_state_to_config (iw);

   gtk_signal_disconnect_by_func (GTK_OBJECT (iw->iv),
                                  (GtkSignalFunc) cb_set_list,   iw);
   gtk_signal_disconnect_by_func (GTK_OBJECT (iw->iv),
                                  (GtkSignalFunc) cb_unset_list, iw);

   iw->iv = NULL;

   /* update linked list */
   ImageWinList = g_list_remove (ImageWinList, iw);

   if (iw == shared_img_win)
      shared_img_win = NULL;

   /* free mem for image view window info */
   g_free(iw);

   /* quit when last window */
   if (!imagewin_get_list() && !thumbwin_get_list()) {
      gimv_quit();
   }
}


static gboolean
cb_window_delete (GtkWidget *widget, GdkEvent *event, ImageWindow *iw)
{
   g_return_val_if_fail (iw, FALSE);

   /* if (imageview_is_loading (iw->iv)) return TRUE; */

   return FALSE;
}


static void
cb_image_changed (ImageView *iv, ImageWindow *iw)
{
   g_return_if_fail (iw);

   if (!(iw->flags & ImgWinSlideShowSeekBarDraggingFlag)) {
      gint pos;
      GtkAdjustment *adj;

      pos = imageview_image_list_position (iv);
      adj = gtk_range_get_adjustment (GTK_RANGE (iw->player.seekbar));
      adj->value = pos;
      gtk_adjustment_changed (adj);
   }

   /* set statu bar */
   imagewin_set_statusbar_info (iw);
   /* set title */
   imagewin_set_window_title (iw);
}


static void
cb_load_start (ImageView *iv, ImageInfo *info, ImageWindow *iw)
{
   g_return_if_fail (iw);

   imagewin_set_sensitive (iw, FALSE);
}


static void
cb_load_end (ImageView *iv,
             ImageInfo *info,
             gboolean cancel,
             ImageWindow *iw)
{
   g_return_if_fail (iw);

   imagewin_set_sensitive (iw, TRUE);
}


static void
cb_set_list (ImageView *iv, ImageWindow *iw)
{
   gint num, pos;
   GtkAdjustment *adj;

   g_return_if_fail (iw);
   g_return_if_fail (IS_IMAGEVIEW (iv));

   gtk_widget_set_sensitive (iw->button.prev, TRUE);
   gtk_widget_set_sensitive (iw->button.next, TRUE);

   /* player */
   gtk_widget_set_sensitive (iw->player.play,    TRUE);
   gtk_widget_set_sensitive (iw->player.stop,    FALSE);
   gtk_widget_set_sensitive (iw->player.rw,      TRUE);
   gtk_widget_set_sensitive (iw->player.fw,      TRUE);
   gtk_widget_set_sensitive (iw->player.prev,    TRUE);
   gtk_widget_set_sensitive (iw->player.next,    TRUE);
   gtk_widget_set_sensitive (iw->player.seekbar, TRUE);

   num = imageview_image_list_length (iv);
   pos = imageview_image_list_position (iv);

   adj = gtk_range_get_adjustment (GTK_RANGE (iw->player.seekbar));

   adj->lower          = 0.0;
   adj->upper          = num;
   adj->value          = pos;
   adj->step_increment = 1.0;
   adj->page_increment = 1.0;
   adj->page_size      = 1.0;
   gtk_adjustment_changed (adj);

   menu_item_set_sensitive (iw->move_menu, "/Next",     TRUE);
   menu_item_set_sensitive (iw->move_menu, "/Previous", TRUE);
   menu_item_set_sensitive (iw->move_menu, "/First",    TRUE);
   menu_item_set_sensitive (iw->move_menu, "/Last",     TRUE);
}


static void
cb_unset_list (ImageView *iv, ImageWindow *iw)
{
   GtkAdjustment *adj;

   g_return_if_fail (iw);
   g_return_if_fail (IS_IMAGEVIEW (iv));

   gtk_widget_set_sensitive (iw->button.prev, FALSE);
   gtk_widget_set_sensitive (iw->button.next, FALSE);

   /* player */
   gtk_widget_set_sensitive (iw->player.play,    FALSE);
   gtk_widget_set_sensitive (iw->player.stop,    FALSE);
   gtk_widget_set_sensitive (iw->player.rw,      FALSE);
   gtk_widget_set_sensitive (iw->player.fw,      FALSE);
   gtk_widget_set_sensitive (iw->player.prev,    FALSE);
   gtk_widget_set_sensitive (iw->player.next,    FALSE);
   gtk_widget_set_sensitive (iw->player.seekbar, FALSE);

   adj = gtk_range_get_adjustment (GTK_RANGE (iw->player.seekbar));
   adj->value = 0;
   gtk_adjustment_changed (adj);

   menu_item_set_sensitive (iw->move_menu, "/Next",     FALSE);
   menu_item_set_sensitive (iw->move_menu, "/Previous", FALSE);
   menu_item_set_sensitive (iw->move_menu, "/First",    FALSE);
   menu_item_set_sensitive (iw->move_menu, "/Last",     FALSE);
}


static void
cb_rendered (ImageView *iv, ImageWindow *iw)
{
   gfloat x_scale, y_scale;
   ImageViewOrientation rotate;

   g_return_if_fail (iw);
   if (!g_list_find (ImageWinList, iw)) return;

   g_return_if_fail (iw->iv);
   if (!g_list_find (imageview_get_list(), iw->iv)) return;

   /* set statu bar */
   imagewin_set_statusbar_info (iw);

   /* set title */
   imagewin_set_window_title (iw);

   /* resize window */
   if (!(iw->flags & ImgWinMaximizeFlag) && conf.imgwin_fit_to_image)
      imagewin_set_win_size (iw);

   gtk_object_get (GTK_OBJECT (iw->iv),
                   "x_scale",     &x_scale,
                   "y_scale",     &y_scale,
                   "orientation", &rotate,
                   NULL);

   if (iw->button.xscale)
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (iw->button.xscale), x_scale);
   if (iw->button.yscale)
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (iw->button.yscale), y_scale);
   if (iw->button.rotate)
      gtk_option_menu_set_history (GTK_OPTION_MENU (iw->button.rotate),
                                   rotate);
}


static void
cb_toggle_aspect (ImageView *iv, gboolean keep_aspect, ImageWindow *iw)
{
   g_return_if_fail (iw);

   gtk_widget_set_sensitive (iw->button.yscale, !keep_aspect);
   if (keep_aspect)
      gtk_widget_hide (iw->button.yscale);
   else
      gtk_widget_show (iw->button.yscale);
}


static gboolean
cb_draw_area_key_press (GtkWidget *widget, 
                        GdkEventKey *event,
                        ImageWindow *iw)
{
   g_return_val_if_fail (iw, FALSE);
        
   switch (event->keyval) {
   case GDK_Escape:
      if (iw->slideshow) {
         gtk_widget_destroy (iw->window);
      } else if (iw->flags & ImgWinMaximizeFlag) {
         menu_check_item_set_active (iw->view_menu, "/Maximize",
                                     !(iw->flags & ImgWinMaximizeFlag));
      }
      return FALSE;
   default:
      break;
   }

   return FALSE;
}


static gint
imageview_button_action (ImageView *iv, GdkEventButton *event, gint num)
{
   gint mx, my;

   imageview_get_view_position (iv, &mx, &my);

   switch (abs (num)) {
   case MOUSE_PRESS_NEXT:
      imageview_next (iv);
      break;
   case MOUSE_PRESS_PREV:
      imageview_prev (iv);
      break;
   case MOUSE_PRESS_POPUP:
      imageview_popup_menu (iv, event);
      break;
   case MOUSE_PRESS_ZOOM_IN:
      imageview_zoom_image (iv, IMAGEVIEW_ZOOM_IN, 0, 0);
      break;
   case MOUSE_PRESS_ZOOM_OUT:
      imageview_zoom_image (iv, IMAGEVIEW_ZOOM_OUT, 0, 0);
      break;
   case MOUSE_PRESS_FIT:
      imageview_zoom_image (iv, IMAGEVIEW_ZOOM_FIT, 0, 0);
      break;
   case MOUSE_PRESS_ROTATE_CCW:
      imageview_rotate_ccw (iv);
      break;
   case MOUSE_PRESS_ROTATE_CW:
      imageview_rotate_cw (iv);
      break;
   case MOUSE_PRESS_NAVWIN:
      imageview_open_navwin (iv, event->x_root, event->y_root);
      break;
   case MOUSE_PRESS_UP:
      my += 20;
      imageview_moveto (iv, mx, my);
      break;
   case MOUSE_PRESS_DOWN:
      my -= 20;
      imageview_moveto (iv, mx, my);
      break;
   case MOUSE_PRESS_LEFT:
      mx += 20;
      imageview_moveto (iv, mx, my);
      break;
   case MOUSE_PRESS_RIGHT:
      mx -= 20;
      imageview_moveto (iv, mx, my);
      break;
   default:
      break;
   }

   return TRUE;
}


static gint
cb_imageview_pressed (ImageView *iv, GdkEventButton *event, ImageWindow *iw)
{
   gint num;

   g_return_val_if_fail (iv, TRUE);
   g_return_val_if_fail (event, TRUE);

   num = prefs_mouse_get_num_from_event (event, conf.imgview_mouse_button);
   if (num > 0)
      return imageview_button_action (iv, event, num);

   return TRUE;
}


static gint
cb_imageview_clicked (ImageView *iv, GdkEventButton *event, ImageWindow *iw)
{
   gint num;

   g_return_val_if_fail (iv, TRUE);
   g_return_val_if_fail (event, TRUE);

   num = prefs_mouse_get_num_from_event (event, conf.imgview_mouse_button);
   if (num < 0)
      return imageview_button_action (iv, event, num);

   return TRUE;
}



/******************************************************************************
 *
 *   public functions.
 *
 ******************************************************************************/
GList *
imagewin_get_list (void)
{
   return ImageWinList;
}


ImageWindow *
imagewin_get_shared_window (void)
{
   return shared_img_win;
}


void
imagewin_player_set_sensitive_all (ImageWindow *iw, gboolean sensitive)
{
   g_return_if_fail (iw);

   gtk_widget_set_sensitive (iw->player.prev,    sensitive);
   gtk_widget_set_sensitive (iw->player.rw,      sensitive);
   gtk_widget_set_sensitive (iw->player.play,    sensitive);
   gtk_widget_set_sensitive (iw->player.stop,    sensitive);
   gtk_widget_set_sensitive (iw->player.fw,      sensitive);
   gtk_widget_set_sensitive (iw->player.next,    sensitive);
   gtk_widget_set_sensitive (iw->player.seekbar, sensitive);
}


void
imagewin_change_image (ImageWindow *iw, ImageInfo *info)
{
   GList *node;
   gboolean fit_to_frame;

   g_return_if_fail (iw);

   node = g_list_find (ImageWinList, iw);
   if (!node) return;

   if (conf.imgwin_raise_window && !(iw->flags & ImgWinFullScreenFlag)) {
      gdk_window_raise (iw->window->window);
      gdk_window_focus (iw->window->window, GDK_CURRENT_TIME);
   }

   imageview_change_image (iw->iv, info);

   node = g_list_find (ImageWinList, iw);
   if (!node) return;

   gtk_object_get (GTK_OBJECT (iw->iv),
                   "fit_when_change", &fit_to_frame,
                   NULL);
   if (fit_to_frame && conf.imgwin_fit_to_image
       && !(iw->flags & ImgWinMaximizeFlag))
   {
      gdk_window_resize (iw->window->window,
                         conf.imgwin_width,
                         conf.imgwin_height);
      while (gtk_events_pending ()) gtk_main_iteration ();
      node = g_list_find (ImageWinList, iw);
      if (!node) return;
   }
}


void
imagewin_set_sensitive (ImageWindow *iw, gboolean sensitive)
{
   g_return_if_fail (iw);

   gtk_widget_set_sensitive (iw->menubar, sensitive);
}


void
imagewin_store_win_state_to_config (ImageWindow *iw)
{
   g_return_if_fail (iw && iw->window);

   if (iw->slideshow) return;

   if (!(iw->flags & ImgWinMaximizeFlag)) {
      conf.imgwin_width  = iw->window->allocation.width;
      conf.imgwin_height = iw->window->allocation.height;
   } else {
      conf.imgwin_width  = iw->win_width;
      conf.imgwin_height = iw->win_height;
   }

   conf.imgwin_show_menubar       = iw->flags & ImgWinShowMenuBarFlag;
   conf.imgwin_show_toolbar       = iw->flags & ImgWinShowToolBarFlag;
   conf.imgwin_show_player        = iw->flags & ImgWinShowPlayerFlag;
   conf.imgwin_show_statusbar     = iw->flags & ImgWinShowStatusBarFlag;
   conf.imgview_player_visible    = iw->player_visible;
   gtk_object_get (GTK_OBJECT (iw->iv),
                   "show_scrollbar",   &conf.imgview_scrollbar,
                   "continuance_play", &conf.imgview_movie_continuance,
                   NULL);
}


/*
 *   imagewin_initialize:
 *      @ allocate new ImageWindow & ImageView struct, and initialize it.
 *
 *   info   :
 *   Return : Pointer to ImageWindow struct.
 */
ImageWindow *
imagewin_initialize (ImageInfo *info)
{
   ImageWindow *iw;
   ImageView   *iv;

   iv = IMAGEVIEW (imageview_new (info));

   g_return_val_if_fail (iv, NULL);

   iw = g_new0 (ImageWindow, 1);
   g_return_val_if_fail (iw, NULL);

   iw->iv = iv;
   iw->flags              = 0;
   iw->flags              |= ImgWinCreatingFlag;
   if (conf.imgwin_show_menubar)
      iw->flags |= ImgWinShowMenuBarFlag;
   if (conf.imgwin_show_toolbar)
      iw->flags |= ImgWinShowToolBarFlag;
   if (conf.imgwin_show_player)
      iw->flags |= ImgWinShowPlayerFlag;
   if (conf.imgwin_show_statusbar)
      iw->flags |= ImgWinShowStatusBarFlag;
   /*
     iw->flags              &= ~ImgWinMaximizeFlag;
     iw->hide_frame         &= ~ImgWinHideFrameFlag;
     iw->seekbar_dragging   &= ~ImgWinSeekBarDraggingFlag;
     iw->is_movie           &= ~ImgWinMovieFlag;
   */
   iw->player_visible       = conf.imgview_player_visible;
   imageview_set_player_visible (iw->iv, iw->player_visible);

   iw->fullscreen           = NULL;
   iw->hide_cursor_timer_id = 0;
   iw->slideshow            = NULL;
   iw->slideshow_interval   = conf.slideshow_interval * 1000;
   iw->slideshow_timer_id   = 0;
   if (conf.slideshow_repeat)
      iw->flags |= ImgWinSlideShowRepeatFlag;

   /* set bg color */
   if (conf.imgwin_set_bg) {
      imageview_set_bg_color (iv,
                              conf.imgwin_bg_color[0],
                              conf.imgwin_bg_color[1],
                              conf.imgwin_bg_color[2]);
   }

   ImageWinList = g_list_append (ImageWinList, iw);

   return iw;
}


/*
 *   imagewin_create_widnow.
 *      @ Create new iamge window. 
 *        ImageWindow struct must be initialized before call this function.
 *
 *   iw     : Pointer to ImageWindow structure.
 *   Return : Return TRUE if success.
 */
gboolean
imagewin_create_window (ImageWindow *iw)
{
   GtkWidget *vbox, *hbox;
   GtkObject *adj;
   GList *node;

   if (!iw)
      return FALSE;

   /* Image Window */
   iw->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
   gtk_window_set_wmclass(GTK_WINDOW(iw->window), "imagewin", "GImageView");
   gtk_widget_realize (iw->window);
   if (iw->flags & ImgWinHideFrameFlag)
      gdk_window_set_decorations (iw->window->window, 0);
   gtk_window_set_policy(GTK_WINDOW(iw->window), TRUE, TRUE, FALSE);
   gtk_window_set_default_size (GTK_WINDOW(iw->window),
                                conf.imgwin_width, conf.imgwin_height);
   gtk_signal_connect (GTK_OBJECT (iw->window), "destroy",
                       GTK_SIGNAL_FUNC (cb_window_closed), iw);
   gtk_signal_connect (GTK_OBJECT (iw->window), "delete_event",
                       GTK_SIGNAL_FUNC (cb_window_delete), iw);
   imagewin_set_window_title (iw);

   gtk_widget_show (iw->window);

   gimv_icon_stock_set_window_icon (iw->window->window, "gimv_icon");

   /* Main vbox */
   vbox = gtk_vbox_new (FALSE, 0);
   iw->main_vbox = vbox;
   if (iw == shared_img_win)
      gtk_widget_set_name (iw->main_vbox, "SharedImageWin");
   else
      gtk_widget_set_name (iw->main_vbox, "ImageWin");
   gtk_container_add (GTK_CONTAINER (iw->window), vbox);
   gtk_widget_show (vbox);

   /* Menu Bar */
   iw->menubar_handle = gtk_handle_box_new();
   gtk_widget_set_name (iw->menubar_handle, "MenuBarContainer");
   gtk_container_set_border_width (GTK_CONTAINER(iw->menubar_handle), 2);
   gtk_box_pack_start(GTK_BOX(vbox), iw->menubar_handle, FALSE, FALSE, 0);

   if (!(iw->flags & ImgWinShowMenuBarFlag))
      gtk_widget_hide (iw->menubar_handle);
   else
      gtk_widget_show (iw->menubar_handle);

   /* toolbar */
   iw->toolbar_handle = gtk_handle_box_new();
   gtk_widget_set_name (iw->toolbar_handle, "ToolBarContainer");
   gtk_container_set_border_width (GTK_CONTAINER(iw->toolbar_handle), 2);
   gtk_box_pack_start(GTK_BOX(vbox), iw->toolbar_handle, FALSE, FALSE, 0);
   iw->toolbar = create_toolbar(iw, iw->toolbar_handle);
   gtk_container_add(GTK_CONTAINER(iw->toolbar_handle), iw->toolbar);
   gtk_widget_show_all (iw->toolbar);

   gtk_toolbar_set_style (GTK_TOOLBAR(iw->toolbar), conf.imgwin_toolbar_style);

   if (!(iw->flags & ImgWinShowToolBarFlag))
      gtk_widget_hide (iw->toolbar_handle);
   else
      gtk_widget_show (iw->toolbar_handle);

   /* player toolbar */
   iw->player_handle = gtk_handle_box_new();
   gtk_widget_set_name (iw->player_handle, "PlayerToolBarContainer");
   gtk_container_set_border_width (GTK_CONTAINER(iw->player_handle), 2);
   gtk_box_pack_start(GTK_BOX(vbox), iw->player_handle, FALSE, FALSE, 0);

   hbox = gtk_hbox_new (FALSE, 0);
   gtk_container_add(GTK_CONTAINER(iw->player_handle), hbox);
   gtk_widget_show (hbox);

   iw->player_bar = create_player_toolbar(iw, iw->player_handle);
   gtk_box_pack_start (GTK_BOX (hbox), iw->player_bar, FALSE, FALSE, 0);
   gtk_widget_show_all (iw->player_bar);

   adj = gtk_adjustment_new (0.0, 0.0, 100.0, 0.1, 1.0, 1.0);

   /* draw area */
   gtk_box_pack_start (GTK_BOX (iw->main_vbox), GTK_WIDGET (iw->iv),
                       TRUE, TRUE, 0);
   gtk_widget_show (GTK_WIDGET (iw->iv));

   gtk_signal_connect (GTK_OBJECT (iw->iv), "image_changed",
                       GTK_SIGNAL_FUNC (cb_image_changed), iw);
   gtk_signal_connect (GTK_OBJECT (iw->iv), "load_start",
                       GTK_SIGNAL_FUNC (cb_load_start), iw);
   gtk_signal_connect (GTK_OBJECT (iw->iv), "load_end",
                       GTK_SIGNAL_FUNC (cb_load_end), iw);
   gtk_signal_connect (GTK_OBJECT (iw->iv), "set_list",
                       GTK_SIGNAL_FUNC (cb_set_list), iw);
   gtk_signal_connect (GTK_OBJECT (iw->iv), "unset_list",
                       GTK_SIGNAL_FUNC (cb_unset_list), iw);
   gtk_signal_connect (GTK_OBJECT (iw->iv), "rendered",
                       GTK_SIGNAL_FUNC (cb_rendered), iw);
   gtk_signal_connect (GTK_OBJECT (iw->iv), "toggle_aspect",
                       GTK_SIGNAL_FUNC (cb_toggle_aspect), iw);
   gtk_signal_connect (GTK_OBJECT (iw->iv), "image_pressed",
                       GTK_SIGNAL_FUNC (cb_imageview_pressed), iw);
   gtk_signal_connect (GTK_OBJECT (iw->iv), "image_clicked",
                       GTK_SIGNAL_FUNC (cb_imageview_clicked), iw);
   gtk_signal_connect (GTK_OBJECT (iw->iv->draw_area), "key_press_event",
                       GTK_SIGNAL_FUNC (cb_draw_area_key_press), iw);

   iw->player.seekbar = gtk_hscale_new (GTK_ADJUSTMENT (adj));
   gtk_scale_set_draw_value (GTK_SCALE (iw->player.seekbar), FALSE);
   gtk_box_pack_start (GTK_BOX (hbox), iw->player.seekbar, TRUE, TRUE, 0);
   gtk_widget_show (iw->player.seekbar);

   gtk_signal_connect (GTK_OBJECT (iw->player.seekbar), "button_press_event",
                       GTK_SIGNAL_FUNC (cb_seekbar_pressed), iw);
   gtk_signal_connect (GTK_OBJECT (iw->player.seekbar), "button_release_event",
                       GTK_SIGNAL_FUNC (cb_seekbar_released), iw);

   gtk_toolbar_set_style (GTK_TOOLBAR(iw->player_bar), conf.imgwin_toolbar_style);

   if (!(iw->flags & ImgWinShowPlayerFlag))
      gtk_widget_hide (iw->player_handle);
   else
      gtk_widget_show (iw->player_handle);

   /* status bar */
   hbox = gtk_hbox_new (FALSE, 0);
   iw->status_bar_container = hbox;
   gtk_box_pack_end (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
   gtk_widget_show (hbox);

   iw->status_bar1 = gtk_statusbar_new ();
   gtk_widget_set_name (iw->status_bar1, "StatuBar1");
   gtk_container_border_width (GTK_CONTAINER (iw->status_bar1), 1);
   gtk_widget_set_usize(iw->status_bar1, 200, -1);
   gtk_box_pack_start (GTK_BOX (hbox), iw->status_bar1, TRUE, TRUE, 0);
   gtk_statusbar_push(GTK_STATUSBAR (iw->status_bar1), 1, _("New Window"));
   gtk_widget_show (iw->status_bar1);

   iw->status_bar2 = gtk_statusbar_new ();
   gtk_widget_set_name (iw->status_bar1, "StatuBar2");
   gtk_container_border_width (GTK_CONTAINER (iw->status_bar2), 1);
   gtk_widget_set_usize(iw->status_bar2, 50, -1);
   gtk_box_pack_start (GTK_BOX (hbox), iw->status_bar2, TRUE, TRUE, 0);
   gtk_widget_show (iw->status_bar2);

   iw->progressbar = gtk_progress_bar_new();
   gtk_widget_set_name (iw->progressbar, "ProgressBar");
   gtk_box_pack_end (GTK_BOX (hbox), iw->progressbar, FALSE, FALSE, 0);
   gtk_widget_show (iw->progressbar);

   imageview_set_progressbar (iw->iv, iw->progressbar);

   imagewin_player_set_sensitive_all (iw, FALSE);

   if (!(iw->flags & ImgWinShowStatusBarFlag))
      gtk_widget_hide (iw->status_bar_container);
   else
      gtk_widget_show (iw->status_bar_container);

   /* create menus */
   create_imageview_menus (iw);

   while (gtk_events_pending ()) gtk_main_iteration ();

   node = g_list_find (ImageWinList, iw);
   if (!node) return FALSE;

   /* Show Image */
   /* imageview_show_image(iw->iv); */

   iw->flags &= ~ImgWinCreatingFlag;

   gtk_widget_grab_focus (iw->iv->draw_area);

   return TRUE;
}


ImageWindow *
imagewin_open_window (ImageInfo *info)
{
   ImageWindow *iw;

   iw = imagewin_initialize (info);
   imagewin_create_window (iw);

   return iw;
}


ImageWindow *
imagewin_open_shared_window (ImageInfo *info)
{
   ImageWindow *iw;

   if (shared_img_win) {
      iw = shared_img_win;
      imagewin_change_image (iw, info);
   } else {
      iw = imagewin_initialize (info);
      shared_img_win = iw;
      imagewin_create_window (iw);
   }

   return iw;
}


ImageWindow *
imagewin_open_window_auto (ImageInfo *info)
{
   ImageWindow *iw;

   if (conf.imgwin_open_new_win) {
      iw = imagewin_open_window (info);
   } else {
      iw = imagewin_open_shared_window (info);
   }

   return iw;
}


static gboolean
timeout_slideshow (ImageWindow *iw)
{
   GList *current;

   current = imageview_image_list_current (iw->iv);

   if ((current && !g_list_next (current) && !(iw->flags & ImgWinSlideShowRepeatFlag))
       || !(iw->flags & ImgWinSlideShowPlayingFlag))
   {
      imagewin_slideshow_stop (iw);
      return FALSE;
   }

   imageview_next (iw->iv);

   return TRUE;
}


void
imagewin_slideshow_play (ImageWindow *iw)
{
   g_return_if_fail (iw);

   iw->flags |= ImgWinSlideShowPlayingFlag;
   iw->slideshow_timer_id
      = gtk_timeout_add (iw->slideshow_interval,
                         (GtkFunction) timeout_slideshow, iw);

   gtk_widget_set_sensitive (iw->player.play, FALSE);
   gtk_widget_set_sensitive (iw->player.stop, TRUE);
}


void
imagewin_slideshow_stop (ImageWindow *iw)
{
   g_return_if_fail (iw);

   iw->flags &= ~ImgWinSlideShowPlayingFlag;

   if (iw->slideshow_timer_id)
      gtk_timeout_remove (iw->slideshow_timer_id);
   iw->slideshow_timer_id = 0;

   gtk_widget_set_sensitive (iw->player.play, TRUE);
   gtk_widget_set_sensitive (iw->player.stop, FALSE);
}


void
imagewin_slideshow_set_interval (ImageWindow *iw, guint interval)
{
   g_return_if_fail (iw);

   iw->slideshow_interval = interval;
   if (iw->flags & ImgWinSlideShowPlayingFlag) {
      imagewin_slideshow_stop (iw);
      imagewin_slideshow_play (iw);
   }
}


void
imagewin_slideshow_set_repeat (ImageWindow *iw, gboolean repeat)
{
   g_return_if_fail (iw);

   if (repeat)
      iw->flags |= ImgWinSlideShowRepeatFlag;
   else
      iw->flags &= ~ImgWinSlideShowRepeatFlag;
}


/******************************************************************************
 *
 *   fullscreen related functions.
 *
 ******************************************************************************/
static void
show_cursor (ImageWindow *iw)
{
   GdkCursor *cursor;

   g_return_if_fail (iw);
   g_return_if_fail (iw->iv);
   g_return_if_fail (iw->iv->draw_area);
   g_return_if_fail (GTK_WIDGET_MAPPED (iw->iv->draw_area));

   cursor = cursor_get (iw->iv->draw_area->window,
                        CURSOR_HAND_OPEN);
   gdk_window_set_cursor (iw->iv->draw_area->window, 
                          cursor);
   gdk_cursor_destroy (cursor);
}


static void
hide_cursor (ImageWindow *iw)
{
   GdkCursor *cursor;

   g_return_if_fail (iw);
   g_return_if_fail (iw->iv);
   g_return_if_fail (iw->iv->draw_area);
   g_return_if_fail (GTK_WIDGET_MAPPED (iw->iv->draw_area));

   cursor = cursor_get (iw->iv->draw_area->window,
                        CURSOR_VOID);
   gdk_window_set_cursor (iw->iv->draw_area->window, 
                          cursor);
   gdk_cursor_destroy (cursor);
}


static gint 
timeout_hide_cursor (gpointer data)
{
   ImageWindow *iw = data;

   hide_cursor (iw);
   iw->hide_cursor_timer_id = 0;

   return FALSE;
}


static gboolean
cb_fullscreen_key_press (GtkWidget *widget, 
                         GdkEventKey *event,
                         ImageWindow *iw)
{
   g_return_val_if_fail (iw, FALSE);
        
   switch (event->keyval) {
   case GDK_Escape:
      if (iw->flags & ImgWinCreatingFlag) return TRUE;
      menu_check_item_set_active (iw->view_menu, "/Full Screen", FALSE);
      return TRUE;
   default:
      break;
   }

#ifdef USE_GTK2
   gtk_accel_groups_activate (G_OBJECT (iw->window),
                              event->keyval, event->state);
#else /* USE_GTK2 */
   gtk_accel_groups_activate (GTK_OBJECT (iw->window),
                              event->keyval, event->state);
#endif /* USE_GTK2 */

   return TRUE;
}


static gboolean
cb_fullscreen_motion_notify (GtkWidget *widget, 
                             GdkEventButton *bevent, 
                             gpointer data)
{
   ImageWindow *iw = data;

   gdk_window_get_pointer (widget->window, NULL, NULL, NULL);
   show_cursor (iw);

   if (iw->hide_cursor_timer_id != 0)
      gtk_timeout_remove (iw->hide_cursor_timer_id);
   iw->hide_cursor_timer_id
      = gtk_timeout_add (IMGWIN_FULLSCREEN_HIDE_CURSOR_DELAY,
                         timeout_hide_cursor,
                         iw);

   return FALSE;
}


static void
set_menu_sensitive (ImageWindow *iw, gboolean sensitive)
{
   g_return_if_fail (iw);

   menu_item_set_sensitive (iw->menubar,   "/File/Open...",     sensitive);
   menu_item_set_sensitive (iw->menubar,   "/File/Open Image Window",     sensitive);
   menu_item_set_sensitive (iw->menubar,   "/File/Open Thumbnail Window", sensitive);
   menu_item_set_sensitive (iw->menubar,   "/File/Quit",        sensitive);
   menu_item_set_sensitive (iw->menubar,   "/Edit/Options...",  sensitive);
   menu_item_set_sensitive (iw->view_menu, "/Menu Bar",         sensitive);
   menu_item_set_sensitive (iw->view_menu, "/Tool Bar",         sensitive);
   menu_item_set_sensitive (iw->view_menu, "/Slide Show Player",sensitive);
   menu_item_set_sensitive (iw->view_menu, "/Status Bar",       sensitive);
   menu_item_set_sensitive (iw->view_menu, "/Scroll Bar",       sensitive);
   menu_item_set_sensitive (iw->view_menu, "/Fit to Image",     sensitive);
   menu_item_set_sensitive (iw->view_menu, "/Maximize",         sensitive);
   gtk_widget_set_sensitive (iw->help_menu, sensitive);
}


static void
imagewin_fullscreen_new (ImageWindow *iw)
{ 
   g_return_if_fail (iw);

   iw->flags |= ImgWinFullScreenFlag;

   /* create window */
   iw->fullscreen = gtk_window_new (GTK_WINDOW_POPUP);

   gtk_window_set_wmclass (GTK_WINDOW (iw->fullscreen), "",
                           "gimv_fullscreen");
   gtk_widget_set_uposition (iw->fullscreen, 0, 0);
   gtk_widget_set_usize (iw->fullscreen, 
                         gdk_screen_width (), 
                         gdk_screen_height ());

   gtk_signal_connect (GTK_OBJECT (iw->fullscreen), 
                       "key_press_event",
                       (GtkSignalFunc) cb_fullscreen_key_press, 
                       iw);
   gtk_signal_connect (GTK_OBJECT (iw->fullscreen),
                       "motion_notify_event",
                       (GtkSignalFunc) cb_fullscreen_motion_notify,
                       iw);

   /* set draw widget */
   if (!(iw->slideshow && conf.slideshow_set_bg)
       && conf.imgwin_fullscreen_set_bg)
   {
      GtkStyle *style;
      GdkColor *color = g_new0 (GdkColor, 1);

      /* save current color */
      style = gtk_widget_get_style (iw->iv->draw_area);
      *color = style->bg[GTK_STATE_NORMAL];
      gtk_object_set_data_full (GTK_OBJECT (iw->fullscreen),
                                "ImgWinFullScreen::OrigColor",
                                color,
                                (GtkDestroyNotify) g_free);

      /* set bg color */
      imageview_set_bg_color (iw->iv,
                              conf.imgwin_fullscreen_bg_color[0],
                              conf.imgwin_fullscreen_bg_color[1],
                              conf.imgwin_fullscreen_bg_color[2]);
   }

   gtk_widget_show (iw->fullscreen);

   imageview_set_fullscreen (iw->iv, GTK_WINDOW (iw->fullscreen));

   set_menu_sensitive (iw, FALSE);

   gdk_keyboard_grab (iw->fullscreen->window, TRUE, GDK_CURRENT_TIME);
   gtk_grab_add (iw->fullscreen);
   gtk_widget_grab_focus (GTK_WIDGET (iw->iv));

   /* enable auto hide cursor stuff */
   iw->hide_cursor_timer_id
      = gtk_timeout_add (IMGWIN_FULLSCREEN_HIDE_CURSOR_DELAY,
                         timeout_hide_cursor,
                         iw);
}


static void
imagewin_fullscreen_delete (ImageWindow *iw)
{
   GdkColor *color;

   g_return_if_fail (iw);

   /* restore draw widget */
   color = gtk_object_get_data (GTK_OBJECT (iw->fullscreen),
                                "ImgWinFullScreen::OrigColor");
   if (color)
      imageview_set_bg_color (iw->iv, color->red, color->green, color->blue);

   imageview_unset_fullscreen (iw->iv);

   gtk_widget_grab_focus (GTK_WIDGET (iw->iv));

   /* disable auto hide cursor stuff */
   if (iw->hide_cursor_timer_id != 0)
      gtk_timeout_remove (iw->hide_cursor_timer_id);
   show_cursor (iw);

   /* restore sensitivity */
   set_menu_sensitive (iw, TRUE);

   gtk_widget_destroy (iw->fullscreen);
   iw->fullscreen = NULL;
   iw->flags &= ~ImgWinFullScreenFlag;

   if (iw->slideshow)
      gtk_widget_destroy (iw->window);
}
