/* -*- Mode: C; tab-width: 3; indent-tabs-mode: nil; c-basic-offset: 3 -*- */

/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: thumbtable.c,v 1.1.2.6 2003/05/14 19:32:11 makeinu Exp $
 */

#include <string.h>

#include "gimageview.h"

#include "dnd.h"
#include "fileload.h"
#include "fileutil.h"
#include "gfileutil.h"
#include "gimv_plugin.h"
#include "gtk2-compat.h"
#include "thumbnail.h"
#include "thumbnail_view.h"
#include "thumbnail_window.h"
#include "thumbtable.h"
#include "thumbtable_prefs.h"


static ThumbViewPlugin thumb_table_modes[] =
{
   {GIMV_THUMBNAIL_VIEW_IF_VERSION,
    N_("Thumbnail"),
    300,
    thumbtable_create,
    thumbtable_append_thumb_frames,
    thumbtable_add_thumbnail,
    thumbtable_refresh_thumbnail,
    NULL,
    thumbtable_redraw,
    thumbtable_resize,
    thumbtable_adjust,
    thumbtable_remove_thumbview_data,
    thumbtable_remove_thumbnail_data,
    thumbtable_set_selection,
    thumbtable_set_focus,
    thumbtable_get_focus,
    thumbtable_thumbnail_is_in_viewport},

   {GIMV_THUMBNAIL_VIEW_IF_VERSION,
    N_("Rename Mode"),
    310,
    thumbtable_create,
    thumbtable_append_thumb_frames,
    thumbtable_add_thumbnail,
    thumbtable_refresh_thumbnail,
    NULL,
    thumbtable_redraw,
    thumbtable_resize,
    thumbtable_adjust,
    thumbtable_remove_thumbview_data,
    thumbtable_remove_thumbnail_data,
    thumbtable_set_selection,
    thumbtable_set_focus,
    thumbtable_get_focus,
    thumbtable_thumbnail_is_in_viewport},
};

GIMV_PLUGIN_GET_IMPL(thumb_table_modes, GIMV_PLUGIN_THUMBVIEW_EMBEDER)

GimvPluginInfo gimv_plugin_info =
{
   if_version:    GIMV_PLUGIN_IF_VERSION,
   name:          N_("Thumbnail View Thumbnail Mode"),
   version:       "0.0.1",
   author:        N_("Takuro Ashie"),
   description:   NULL,
   get_implement: gimv_plugin_get_impl,
   get_mime_type: NULL,
   get_prefs_ui:  gimv_prefs_ui_thumbtable_get_page,
};


typedef struct Thumbata_Tag
{
   GtkWidget *button;
   GtkWidget *vbox;
   GtkWidget *pixmap;
   GtkWidget *entry;
   GtkTooltips *tooltips;
} ThumbData;


typedef struct ThumbTableData_Tag
{
   GtkWidget   *table;
   GtkWidget   *hbox;
   GtkWidget   *event_box;
   gint         colnum;

   Thumbnail   *focused;   /* focused thumbnail */
   gfloat       page_pos_x;
   gfloat       page_pos_y;
} ThumbTableData;


/* callback functions */
static void       cb_thumbbutton_enter          (GtkWidget      *button,
                                                 gpointer        data);
static void       cb_thumbbutton_toggle         (GtkWidget      *button,
                                                 gpointer        data);
static void       cb_thumbbutton_toggle         (GtkWidget      *button,
                                                 gpointer        data);
static gboolean   cb_button_focus_in            (GtkWidget      *button,
                                                 GdkEventFocus  *event,
                                                 Thumbnail      *thumb);
static gboolean   cb_expose                     (GtkWidget      *widget,
                                                 GdkEventExpose *event,
                                                 ThumbView      *tv);
static void       cb_entry_activate             (GtkWidget      *entry,
                                                 Thumbnail      *thumb);
static gboolean   cb_entry_focus_in             (GtkWidget      *entry,
                                                 GdkEventFocus  *event,
                                                 Thumbnail      *thumb);
static gboolean   cb_entry_focus_out            (GtkWidget      *entry,
                                                 GdkEventFocus  *event,
                                                 Thumbnail      *thumb);
static gboolean   cb_entry_key_press            (GtkWidget      *entry,
                                                 GdkEventKey    *event,
                                                 Thumbnail      *thumb);

/* private functions */
static ThumbTableData *thumbtable_new           (ThumbView *tv);
static guint      calc_thumbtable_col_row_num   (ThumbView *tv,
                                                 gint       num);
static gboolean   calc_thumbbutton_pos          (Thumbnail *thumb,
                                                 gint      *col,
                                                 gint      *row);
static GtkWidget *create_thumbnail_button       (Thumbnail *thumb,
                                                 gint       thumb_size,
                                                 gchar     *dest_mode);
static gboolean   thumbtable_append_thumb_frame (ThumbView *tv,
                                                 Thumbnail *thumb,
                                                 gchar     *dest_mode);
static gint       idle_thumbtable_redraw        (gpointer   data);


GtkTargetEntry thumbtable_dnd_targets[] = {
   {"text/uri-list",              0, TARGET_URI_LIST},
};
const gint thumbtable_dnd_targets_num = sizeof(thumbtable_dnd_targets) / sizeof(GtkTargetEntry);


/* static GtkStyle *normal_style = NULL; */


/******************************************************************************
 *
 *   Callback functions.
 *
 ******************************************************************************/
static gboolean
cb_thumb_button_press (GtkWidget *button, GdkEventButton *event, gpointer data)
{
   Thumbnail *thumb = data;
   ThumbView *tv;
   gboolean found = FALSE;

   g_return_val_if_fail (button && thumb, FALSE);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, FALSE);

   /* set selection */
   if (event->type == GDK_BUTTON_PRESS && (event->button == 1)) {
      if (event->state & GDK_SHIFT_MASK) {
         found = thumbview_set_selection_multiple (thumb, TRUE, FALSE);
         if (!found)
            thumbview_set_selection_multiple (thumb, FALSE, FALSE);
         thumbview_set_selection (thumb, FALSE);
      }
   }

   /* relay management to reference callback function */
   return thumbview_thumb_button_press_cb (button, event, thumb);
}


static gboolean
cb_thumb_button_release (GtkWidget *button, GdkEventButton *event, gpointer data)
{
   Thumbnail *thumb = data;

   return thumbview_thumb_button_release_cb (button, event, thumb);
}


/*
 *  cb_button_selected:
 *     @ Callback function for thumbnail button selected.
 *     @ Currently, do nothing.
 *
 *  widget :
 *  data   :
 */
static void
cb_thumbbutton_enter (GtkWidget *button, gpointer data)
{
   /*
     g_print ("enter\n");
   */
}


static void
cb_thumbbutton_toggle (GtkWidget *button, gpointer data)
{
   Thumbnail *thumb = data;

   g_return_if_fail (thumb);

   thumb->selected = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
   /*
   if (thumb->selected) {
      hilight_thumbnail (button);
   } else {
      if (!normal_style)
      normal_style = gtk_style_copy (gtk_widget_get_style (button));
      set_style(button, normal_style);
   }
   */
}


static gboolean
cb_button_focus_in (GtkWidget *button, GdkEventFocus *event, Thumbnail *thumb)
{
   ThumbView *tv;
   ThumbTableData *tt;
   ThumbData *thumb_data;

   g_return_val_if_fail (thumb, FALSE);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, FALSE);

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   g_return_val_if_fail (tt && thumb_data, FALSE);

   if (tt->focused == thumb) return FALSE;
   tt->focused = thumb;

   thumbtable_adjust (tv, thumb);

   return FALSE;
}


static gboolean
cb_expose (GtkWidget *widget, GdkEventExpose *event, ThumbView *tv)
{
   thumbview_resize (tv);

   return FALSE;
}


static void
cb_thumbview_drag_begin (GtkWidget *widget,
                         GdkDragContext *context,
                         gpointer data)
{
   Thumbnail *thumb = data;
   ThumbView *tv;

   g_return_if_fail (thumb);
   tv = thumb->thumb_view;
   g_return_if_fail (tv);

   if (!thumb->selected) {
      thumbview_set_selection_all (tv, FALSE);
      thumbview_set_selection (thumb, TRUE);
   }

   thumbview_drag_begin_cb (widget, context, thumb->thumb_view);
}


static void
cb_entry_activate (GtkWidget *entry, Thumbnail *thumb)
{
   ThumbWindow *tw;
   ThumbView *tv;
   const gchar *filename;
   gchar *dest_filename = NULL, *dest_path, *dirname;
   gchar error_message[BUF_SIZE];
   gboolean success;

   g_return_if_fail (thumb);
   g_return_if_fail (!image_info_is_in_archive (thumb->info));

   tv = thumbnail_get_parent_thumbview (thumb);
   g_return_if_fail (tv);

   tw = tv->thumb_window;
   g_return_if_fail (tw);

   filename = image_info_get_path (thumb->info);
   if (!filename || !*filename) goto ERROR0;
   if (!file_exists (filename)) {
      g_snprintf (error_message, BUF_SIZE,
                  _("File not exist!!:\n%s"),
                  filename);
      gtkutil_message_dialog ("Error!!", error_message,
                              GTK_WINDOW (tw->window));
      goto ERROR0;
   }

   filename = g_basename (gtk_entry_get_text (GTK_ENTRY (entry)));
   if (!filename || !*filename) goto ERROR0;
   dest_filename = gimv_filename_to_locale (filename);

   dirname = g_dirname (image_info_get_path (thumb->info));
   if (!dirname) goto ERROR0;
   if (!*dirname) goto ERROR1;

   if (!iswritable (dirname)) {
      g_snprintf (error_message, BUF_SIZE,
                  _("Permission denied!!:\n%s"),
                  dirname);
      gtkutil_message_dialog ("Error!!", error_message,
                              GTK_WINDOW (tw->window));
      goto ERROR1;
   }

   dest_path = g_strconcat (dirname, "/", dest_filename, NULL);
   if (file_exists (dest_path)) {
      g_snprintf (error_message, BUF_SIZE,
                  _("File exist!!:\n%s"),
                  dest_path);
      gtkutil_message_dialog (_("Error!!"), error_message,
                              GTK_WINDOW (tw->window));
      goto ERROR2;
   }

   /* do rename !! */
   success = image_info_rename_image (thumb->info, dest_path);

   if (!success) {
      g_snprintf (error_message, BUF_SIZE,
                  _("Faild to rename!!"));
      gtkutil_message_dialog (_("Error!!"), error_message,
                              GTK_WINDOW (tw->window));
   }

ERROR2:
   g_free (dest_path);
ERROR1:
   g_free (dirname);
ERROR0:
   g_free (dest_filename);
   filename = g_basename (image_info_get_path (thumb->info));

   if (filename && *filename) {
      gchar *tmpstr;
      tmpstr = gimv_filename_to_internal (filename);
      if (tmpstr)
         gtk_entry_set_text (GTK_ENTRY (entry), tmpstr);
      g_free (tmpstr);
   }
}


static gboolean
cb_entry_focus_in (GtkWidget *widget, GdkEventFocus *event, Thumbnail *thumb)
{
#ifdef USE_GTK2
   thumbwin_remove_key_accel (thumb->thumb_view->thumb_window);
#endif /* USE_GTK2 */
   return FALSE;
}


static gboolean
cb_entry_focus_out (GtkWidget *entry, GdkEventFocus *event, Thumbnail *thumb)
{
   const gchar *filename;

   g_return_val_if_fail (thumb, FALSE);

   filename = g_basename (image_info_get_path (thumb->info));

   if (filename && *filename) {
      gchar *tmpstr;
      tmpstr = gimv_filename_to_internal (filename);
      if (tmpstr)
         gtk_entry_set_text (GTK_ENTRY (entry), tmpstr);
      g_free (tmpstr);
   }

#ifdef USE_GTK2
   thumbwin_reset_key_accel (thumb->thumb_view->thumb_window);
#endif /* USE_GTK2 */
   return FALSE;
}


static gboolean
cb_entry_key_press (GtkWidget *entry,
                    GdkEventKey *event,
                    Thumbnail *thumb)
{
   ThumbView *tv;
   ThumbWindow *tw;
   Thumbnail *n_thumb;
   ThumbData *thumb_data;
   GList *next, *current;

   g_return_val_if_fail (thumb, FALSE);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, FALSE);

   tw = tv->thumb_window;
   g_return_val_if_fail (tw, FALSE);

   switch (event->keyval) {
   case GDK_Escape:
      gtk_window_set_focus (GTK_WINDOW (thumb->thumb_view->thumb_window->window), NULL);
      return TRUE;
   case GDK_Tab:
   case GDK_ISO_Left_Tab:
      current = g_list_find (tv->thumblist, thumb);
      if (!current) break;

      if (event->state & GDK_SHIFT_MASK)
         next = g_list_previous (current);
      else
         next = g_list_next (current);

      if (!next) {
         if (event->state & GDK_SHIFT_MASK)
            next = g_list_last (tv->thumblist);
         else
            next = tv->thumblist;
      }

      n_thumb = next->data;
      if (!n_thumb) break;

      thumb_data = g_hash_table_lookup (n_thumb->mode_data, THUMB_TABLE_LABEL);

      gtk_widget_grab_focus (thumb_data->entry);

      thumbtable_adjust (tv, n_thumb);
      return TRUE;
   default:
      break;
   }

   return TRUE;
}


static gboolean
cb_thumb_key_press (GtkWidget *widget,
                    GdkEventKey *event,
                    Thumbnail *thumb)
{
   ThumbView *tv;

   g_return_val_if_fail (event, FALSE);
   if (!thumb) return FALSE;

   tv = thumbnail_get_parent_thumbview (thumb);
   g_return_val_if_fail (tv, FALSE);

   if (thumbview_thumb_key_press_cb(widget, event, thumb))
      return FALSE;

   {
      ThumbTableData *tt;
      gint pos = g_list_index (tv->thumblist, thumb);
      gint col, row;

      tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
      g_return_val_if_fail (tt, FALSE);

      col = pos % tt->colnum;
      row = pos / tt->colnum;

      switch (event->keyval) {
      case GDK_Left:
         if (col == 0) return TRUE;
         break;
      case GDK_Right:
         if (col == tt->colnum - 1 || pos == tv->filenum - 1) return TRUE;
         break;
      case GDK_Up:
         if (row == 0) return TRUE;
         break;
      case GDK_Down:
         if (row == tv->filenum / tt->colnum
             || (row + 1) * tt->colnum + col >= tv->filenum)
         {
            return TRUE;
         }
         break;
      case GDK_Return:
         thumbview_open_image (tv, thumb, 0);
         break;
      case GDK_Delete:
         thumbview_delete_files (tv);
         break;
      default:
         break;
      }
   }

   return FALSE;
}


/******************************************************************************
 *
 *   private functions.
 *
 ******************************************************************************/
static ThumbTableData *
thumbtable_new (ThumbView *tv)
{
   ThumbTableData *tt = NULL;

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   if (!tt) {
      tt = g_new0 (ThumbTableData, 1);
      tt->focused = NULL;
      tt->page_pos_x = 0.0;
      tt->page_pos_y = 0.0;
      g_hash_table_insert (tv->disp_mode_data, THUMB_TABLE_LABEL, tt);
   }

   return tt;
}


/*
 *  calc_thumbtable_col_row_num:
 *     @ Calculate thumbnail table's columns and rows num. Calculated col num will be
 *       stored to ThumbView structure, and calcurated rows num is return value.
 *
 *  tv     : Pointer to ThumbView structure.
 *  num    : Number of thumbnails.
 *  Return : Rows number.
 */
static guint
calc_thumbtable_col_row_num (ThumbView *tv, gint num)
{
   ThumbWindow *tw;
   ThumbTableData *tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   GtkAdjustment *hadj;
   gint table_rows = 0, container_width;
   gint celwidth;
   gint ncol_min, ncol_max, col_space, button_border_width;

   tw = tv->thumb_window;
   hadj = gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW (tv->container));

   container_width = hadj->page_size;
   if (container_width < 10)
      container_width = tw->window->allocation.width;

   thumbtable_prefs_get_value ("colnum_min", (gpointer) &ncol_min);
   thumbtable_prefs_get_value ("colnum_max", (gpointer) &ncol_max);
   thumbtable_prefs_get_value ("col_space",  (gpointer) &col_space);
   thumbtable_prefs_get_value ("button_border_width",
                               (gpointer) &button_border_width);

   celwidth = tv->ThumbnailSize + 6
      + button_border_width * 2 + col_space;
   tt->colnum = (container_width /*+ celwidth * 2*/) / (celwidth);

   if (tt->colnum < ncol_min)
      tt->colnum = ncol_min;
   if (tt->colnum > ncol_max)
      tt->colnum = ncol_max;

   if (num)
      table_rows = num / tt->colnum + 1;
   
   return table_rows;
}


/*
 *  calc_thumbbutton_pos:
 *     @ calculate thumbnail position in the table (column and row).
 *
 *  thumb  : Pointer to the Thumbnail widget.
 *  col    : column num for return. 
 *  row    : row num for return.
 *  Return : TRUE if success.
 */
static gboolean
calc_thumbbutton_pos (Thumbnail *thumb, gint *col, gint *row)
{
   ThumbView *tv;
   ThumbTableData *tt;
   GList *node;
   gint pos;

   if (!thumb) return FALSE;

   tv = thumb->thumb_view;
   if (!tv) return FALSE;

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   if (!tt) return FALSE;

   node = g_list_find (tv->thumblist, thumb);

   if (!node)
      return FALSE;

   pos = g_list_position (tv->thumblist, node);

   *col = pos % tt->colnum;
   *row = pos / tt->colnum;

   return TRUE;
}


/*
 *  create_thumbnail_button:
 *     @ Create a thumbnail button. 
 *
 *  thumb      : Pointer to Thumbnail struture.
 *  thumb_size : Size of thumbnail.
 *  mode       :
 *  Return     : Pointer to GtkButton widget.
 */
static GtkWidget *
create_thumbnail_button (Thumbnail *thumb, gint thumb_size, gchar *dest_mode)
{
   GtkWidget *button;
   GtkWidget *label;
   GtkTooltips *tooltips;
   gchar *mode_label, *str;
   gchar buf[BUF_SIZE];
   const gint x_pad = 8, y_pad = 8;
   gint width, height;
   ThumbView *tv;
   ThumbData *thumb_data;
   const gchar *filename;
   gint button_border_width;

   g_return_val_if_fail (thumb, NULL);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, NULL);

   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   if (!thumb_data) return NULL;

   thumb_data->vbox = gtk_vbox_new (FALSE, 0);
   gtk_widget_show (thumb_data->vbox);

   button = gtk_toggle_button_new ();
   thumb_data->button = button;
   gtk_box_pack_start (GTK_BOX (thumb_data->vbox), button, TRUE, TRUE, 0);
   gtk_widget_show (button);

   width  = tv->ThumbnailSize + x_pad * 2;
   height = tv->ThumbnailSize + y_pad * 2;
   gtk_widget_set_usize (button, width, height);

   /* set signals */
   gtk_signal_connect (GTK_OBJECT(button), "enter",
                       GTK_SIGNAL_FUNC(cb_thumbbutton_enter), thumb);
   gtk_signal_connect (GTK_OBJECT(button), "toggled",
                       GTK_SIGNAL_FUNC(cb_thumbbutton_toggle), thumb);
   gtk_signal_connect (GTK_OBJECT (button), "key-press-event",
                       GTK_SIGNAL_FUNC (cb_thumb_key_press), thumb);
   gtk_signal_connect (GTK_OBJECT (button),"button_press_event",
                       GTK_SIGNAL_FUNC(cb_thumb_button_press), thumb);
   SIGNAL_CONNECT_TRANSRATE_SCROLL (button);
   gtk_signal_connect (GTK_OBJECT (button),"button_release_event",
                       GTK_SIGNAL_FUNC(cb_thumb_button_release),
                       thumb);
   gtk_signal_connect (GTK_OBJECT (button),"motion_notify_event",
                       GTK_SIGNAL_FUNC(thumbview_motion_notify_cb), thumb);
   gtk_signal_connect (GTK_OBJECT (button),"focus_in_event",
                       GTK_SIGNAL_FUNC(cb_button_focus_in), thumb);

   /* for drag file list */
   dnd_src_set (button, thumbtable_dnd_targets, thumbtable_dnd_targets_num);
   gtk_signal_connect (GTK_OBJECT (button), "drag_begin",
                       GTK_SIGNAL_FUNC (cb_thumbview_drag_begin),
                       thumb);
   gtk_signal_connect (GTK_OBJECT (button), "drag_data_get",
                       GTK_SIGNAL_FUNC (thumbview_drag_data_get_cb),
                       thumb->thumb_view);
   gtk_signal_connect (GTK_OBJECT (button), "drag-data-delete",
                       GTK_SIGNAL_FUNC (thumbview_drag_data_delete_cb),
                       thumb->thumb_view);
   gtk_signal_connect (GTK_OBJECT (button), "drag_end",
                       GTK_SIGNAL_FUNC (thumbview_drag_end_cb),
                       thumb->thumb_view);
   gtk_object_set_data (GTK_OBJECT (button), "gimv-tab", tv);

   /* set tooltip */
   filename = image_info_get_path (thumb->info);

   str = gimv_filename_to_internal (filename);

   g_snprintf(buf, BUF_SIZE, "%s (%dkB)", str,
              (gint) (thumb->info->st.st_size) / 1024);

   g_free (str);
   str = NULL;

   tooltips = gtk_tooltips_new();
#ifdef USE_GTK2
   gtk_object_ref (GTK_OBJECT (tooltips));
   gtk_object_sink (GTK_OBJECT (tooltips));
#endif
   gtk_tooltips_set_tip(GTK_TOOLTIPS (tooltips), button,
                        buf, NULL);
   if (thumb_data->tooltips)
      gtk_object_unref (GTK_OBJECT (thumb_data->tooltips));
   thumb_data->tooltips = tooltips;

   /* initialize button state */
   gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), thumb->selected);

   /* get filename */
   filename = g_basename (image_info_get_path (thumb->info));
   str = gimv_filename_to_internal (filename);

   mode_label = dest_mode;

   thumbtable_prefs_get_value ("button_border_width",
                               (gpointer) &button_border_width);

   /* normal mode */
   if (mode_label && !strcmp (THUMB_TABLE_LABEL, mode_label)) {
      label = gtk_label_new (str);
      gtk_widget_set_usize(label, thumb_size + button_border_width * 2, -1);
      gtk_box_pack_end (GTK_BOX(thumb_data->vbox), label, FALSE, FALSE, 0);
      gtk_widget_show (label);

      /* rename mode */
   } else if (mode_label && !strcmp (RENAME_MODE_LABEL, mode_label)) {
      gchar *dirname;

      dirname = g_dirname (image_info_get_path (thumb->info));

      thumb_data->entry = gtk_entry_new ();
      if (str)
         gtk_entry_set_text (GTK_ENTRY (thumb_data->entry), str);
      gtk_widget_set_usize(thumb_data->entry,
                           thumb_size + button_border_width * 2, -1);
      gtk_box_pack_end (GTK_BOX(thumb_data->vbox), thumb_data->entry, FALSE, FALSE, 0);
      gtk_widget_show (thumb_data->entry);

      if (!iswritable (dirname) || image_info_is_in_archive (thumb->info)) {
         gtk_widget_set_sensitive (thumb_data->entry, FALSE);
      } else {
         gtk_signal_connect (GTK_OBJECT(thumb_data->entry), "activate",
                             GTK_SIGNAL_FUNC(cb_entry_activate), thumb);	 
         gtk_signal_connect (GTK_OBJECT(thumb_data->entry),"focus_in_event",
                             GTK_SIGNAL_FUNC(cb_entry_focus_in), thumb);
         gtk_signal_connect (GTK_OBJECT(thumb_data->entry),"focus_out_event",
                             GTK_SIGNAL_FUNC(cb_entry_focus_out), thumb);
         gtk_signal_connect_after (GTK_OBJECT(thumb_data->entry), "key-press-event",
                                   GTK_SIGNAL_FUNC(cb_entry_key_press), thumb);
      }

      g_free (dirname);
   }

   g_free (str);

   return thumb_data->vbox;
}


/*
 *  thumbtable_append_thumb_frame:
 *     @ append a thumbnail button to the table.
 *
 *  tv     : Pointer to the ThumbView struct.
 *  thumb  : Pointer to the Thumbnail struct to create frame.
 *  Return : Added thumbnail button widget.
 */
static gboolean
thumbtable_append_thumb_frame (ThumbView *tv, Thumbnail *thumb, gchar *dest_mode)
{
   GtkWidget *button;
   GdkPixmap *pixmap;
   GdkBitmap *mask;
   gint col, row;
   ThumbTableData *tt;
   ThumbData *thumb_data;
   gboolean retval = FALSE;

   g_return_val_if_fail (tv, FALSE);

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   g_return_val_if_fail (tt, FALSE);

   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   if (!thumb_data) {
      thumb_data = g_new0 (ThumbData, 1);
      g_hash_table_insert (thumb->mode_data, THUMB_TABLE_LABEL, thumb_data);
   } else {
      if (thumb_data->tooltips)
         gtk_object_unref (GTK_OBJECT (thumb_data->tooltips));
   }
   thumb_data->button      = NULL;
   thumb_data->pixmap      = NULL;
   thumb_data->tooltips    = NULL;

   button = create_thumbnail_button (thumb, tv->ThumbnailSize, dest_mode);

   thumbnail_get_thumb (thumb, &pixmap, &mask);
   if (pixmap) {
      thumbtable_add_thumbnail (thumb, dest_mode, LOAD_CACHE);
      retval = TRUE;
   }

   calc_thumbbutton_pos (thumb, &col, &row);
   gtk_table_attach (GTK_TABLE (tt->table), button,
                     col, col + 1, row, row + 1,
                     GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
   gtk_widget_show (button);	 

   return retval;
}


static gint
idle_thumbtable_redraw (gpointer data)
{
   ThumbView *tv = data;
   ThumbTableData *tt;
   GtkScrolledWindow *scrollwin;
   GtkAdjustment *hadj, *vadj;
   GList *node;

   g_return_val_if_fail (tv, FALSE);

   node = g_list_find (thumbview_get_list(), tv);
   if (!node) return FALSE;

   if (!(!strcmp (tv->disp_mode, THUMB_TABLE_LABEL)
         || !strcmp (tv->disp_mode, RENAME_MODE_LABEL)))
   {
      return FALSE;
   }

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   if (!tt) return FALSE;

   gtk_widget_queue_draw (tt->table);

   /* reset page position */
   scrollwin = GTK_SCROLLED_WINDOW (tv->container);
   hadj = gtk_scrolled_window_get_hadjustment (scrollwin);
   vadj = gtk_scrolled_window_get_vadjustment (scrollwin);

   if (tt->page_pos_x < 0 - 0.1)
      tt->page_pos_x = 0;
   if (tt->page_pos_x > hadj->upper - hadj->page_size + 0.1)
      tt->page_pos_x = hadj->upper - hadj->page_size;

   if (tt->page_pos_y < 0 - 0.1)
      tt->page_pos_y = 0;
   if (tt->page_pos_y > vadj->upper - vadj->page_size + 0.1)
      tt->page_pos_y = vadj->upper - vadj->page_size;

   gtk_adjustment_set_value (hadj, 0.0);
   gtk_adjustment_set_value (vadj, 0.0);
   gtk_adjustment_set_value (hadj, tt->page_pos_x);
   gtk_adjustment_set_value (vadj, tt->page_pos_y);

   gtk_widget_queue_draw (tt->table);

   return FALSE;
}



/******************************************************************************
 *
 *   public functions.
 *
 ******************************************************************************/
/*
 *  thumbtable_remove_thumbview_data:
 *
 *  tv : Pointer to the ThumbView struct.
 */
void
thumbtable_remove_thumbview_data (ThumbView *tv)
{
   ThumbTableData *tt;

   if (!tv) return;

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   if (!tt) return;

   g_hash_table_remove (tv->disp_mode_data, THUMB_TABLE_LABEL);
   g_free (tt);
}


/*
 *  thumbtable_remove_thumbnail_data:
 *
 *  thumb : Pointer to the Thumbnail struct.
 */
void
thumbtable_remove_thumbnail_data (Thumbnail *thumb)
{
   ThumbData *thumb_data;

   if (!thumb) return;

   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   if (!thumb_data) return;

   g_hash_table_remove (thumb->mode_data, THUMB_TABLE_LABEL);
   if (thumb_data->tooltips)
      gtk_object_unref (GTK_OBJECT (thumb_data->tooltips));
   g_free (thumb_data);
}


/*
 *  thumbtable_append_thumb_frames:
 *     @ add thumbnail buttons to thumbnail table
 *       (and add thumbnail to button if exist).
 *
 *  tv     : Pointer to the ThumbView struct.
 *  start  : Pointer to Start of the Thumbnail list.
 *  Return : Return FALSE if canceled by user.
 */
GList *
thumbtable_append_thumb_frames (ThumbView *tv, GList *start, gchar *dest_mode)
{
   Thumbnail  *thumb;
   GList *node, *loadlist = NULL;
   gboolean thumb_loaded = FALSE;

   g_return_val_if_fail (tv, FALSE);
   if (!start) return FALSE;

   node = start;
   while (node) {
      thumb = (Thumbnail *) node->data;

      /* add button */
      thumb_loaded = thumbtable_append_thumb_frame (tv, thumb, dest_mode);
      if (!thumb_loaded)
         loadlist = g_list_append (loadlist, thumb);

      node = g_list_next (node);
   }

   return loadlist;
}


/*
 *  thumbtable_add_thumbnail:
 *     @ append thumbnail to thumbnail frame.
 *
 *  thumb     : Pointer to the Thumbnail struct.
 *  dest_mode : New thumbnail view display mode.
 *  Return    : Added thumbnail pixmap widget.
 */
GtkWidget *
thumbtable_add_thumbnail (Thumbnail *thumb, gchar *dest_mode,
                          ThumbLoadType type)
{
   ThumbView *tv = thumb->thumb_view;
   ThumbData *thumb_data;
   GtkWidget *pixmap = NULL;
   GdkPixmap *thumbnail;
   GdkBitmap *mask;

   g_return_val_if_fail (thumb, NULL);

   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   g_return_val_if_fail (thumb_data, NULL);

   thumbnail_get_thumb (thumb, &thumbnail, &mask);
   if (!thumbnail)
      if (thumbnail_create (thumb, tv->ThumbnailSize, type))
         thumbnail_get_thumb (thumb, &thumbnail, &mask);

   if (thumbnail) {
      if (thumb_data->pixmap) {
#ifdef USE_GTK2
         gtk_image_set_from_pixmap (GTK_IMAGE (thumb_data->pixmap),
                                    thumbnail, mask);
#else /* USE_GTK2 */
         gtk_pixmap_set (GTK_PIXMAP (thumb_data->pixmap), thumbnail, mask);
#endif /* USE_GTK2 */
      } else {
         pixmap = thumbnail_get_thumb_by_widget (thumb);
         gtk_container_add (GTK_CONTAINER (thumb_data->button), pixmap);
         gtk_widget_show (pixmap);
         thumb_data->pixmap = pixmap;
      }
      
   }

   return pixmap;
}


/*
 *  thumbtable_redraw:
 *     @ For resize table or sort files list, once tear off thumbnail buttons
 *       from thumbnail table, and attach to new resized table.
 *
 *  tv        : Pointer to ThumbView struct for redraw.
 *  dest_mode : New thumbnail view display mode.
 *  Return    : Pointer to the new GtkTable widget.
 */
void
thumbtable_redraw (ThumbView *tv, gchar *dest_mode,
                   GtkWidget *scroll_win, GList **loadlist)
{
   ThumbTableData *tt;
   GtkScrolledWindow *scrollwin;
   GtkAdjustment *vadj;
   GList *node;
   gint colnum;

   g_return_if_fail (tv);

   node = g_list_find (thumbview_get_list(), tv);
   if (!node) return;

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   if (!tt) {
      thumbtable_create (tv, dest_mode);
      tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   }

   if (tv->container && !strcmp (tv->disp_mode, THUMB_TABLE_LABEL)) {
      scrollwin = GTK_SCROLLED_WINDOW (tv->container);
      vadj = gtk_scrolled_window_get_vadjustment (scrollwin);
      tt->page_pos_y = vadj->value;
   }

   colnum = tt->colnum;
   calc_thumbtable_col_row_num (tv, 0);

   thumbtable_create (tv, dest_mode);

   /* redraw */
   if (scroll_win) {
      scrollwin = GTK_SCROLLED_WINDOW (scroll_win);
      if (GTK_BIN (tv->container)->child)
         gtk_widget_destroy(GTK_BIN (tv->container)->child);   
      gtk_scrolled_window_add_with_viewport (scrollwin, tt->event_box);
   }

   if (!loadlist) return;

   /* create thumbnail load list */
   *loadlist = NULL;
   node = tv->thumblist;
   while (node) {
      Thumbnail *thumb = node->data;
      GdkPixmap *pixmap = NULL;
      GdkBitmap *mask = NULL;

      thumbnail_get_thumb (thumb, &pixmap, &mask);
      if (!pixmap)
         *loadlist = g_list_append (*loadlist, thumb);
      node = g_list_next (node);
   }

   if (!strcmp (dest_mode, THUMB_TABLE_LABEL)
       || !strcmp (dest_mode, RENAME_MODE_LABEL))
   {
      gtk_idle_add (idle_thumbtable_redraw, tv);
   }
}


/*
 *  thumbtable_refresh_thumbnail:
 *     @ Create thumbnail from original image, and recreate thumbnail button.
 *
 *  thumb  : Pointer to the Thumbnail struct.
 *  Return : TRUE if success.
 */
gboolean
thumbtable_refresh_thumbnail (Thumbnail *thumb, ThumbLoadType type)
{
   ThumbView *tv;
   ThumbTableData *tt;
   GtkWidget *button;
   gint col, row;
   ThumbData *thumb_data;

   g_return_val_if_fail (thumb, FALSE);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, FALSE);

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   g_return_val_if_fail (tt, FALSE);

   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   g_return_val_if_fail (thumb_data, FALSE);

   gtk_widget_destroy (thumb_data->button);
   thumb_data->button = NULL;
   thumb_data->pixmap = NULL;

   button = create_thumbnail_button (thumb, tv->ThumbnailSize, tv->disp_mode);

   if (!button)
      return FALSE;

   thumbtable_add_thumbnail (thumb, tv->disp_mode, type);
   calc_thumbbutton_pos (thumb, &col, &row);
   gtk_table_attach (GTK_TABLE (tt->table), button,
                     col, col + 1, row, row + 1,
                     GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
   gtk_widget_show (button);

   thumb_data->button = button;

   if (button)
      return TRUE;
   else
      return FALSE;
}


/*
 *  thumbtable_resize:
 *     @ Resize thumbnail table (only call thumbtable_redraw ())
 *
 *  tv     : Pointer to the ThumbView struct.
 *  Return : Pointer to the New thumbnail table widget.
 */
GtkWidget *
thumbtable_resize (ThumbView *tv)
{
   ThumbTableData *tt;
   GList *node;
   gint colnum;

   g_return_val_if_fail (tv, NULL);

   node = g_list_find (thumbview_get_list(), tv);
   g_return_val_if_fail (node, NULL);

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   if (!tt) return NULL;

   colnum = tt->colnum;
   calc_thumbtable_col_row_num (tv, 0);

   if (colnum == tt->colnum) {
      return tt->event_box;
   } else {
      thumbtable_redraw (tv, tv->disp_mode, tv->container, NULL);

      return tt->event_box;
   }
}


void
thumbtable_adjust (ThumbView *tv, Thumbnail *thumb)
{
   ThumbTableData *tt;
   ThumbData *thumb_data;   
   GList *node;
   GtkWidget *button;
   GtkScrolledWindow *scrollwin;
   GtkAdjustment *hadj, *vadj;
   gint top, buttom, left, right;
   gint frame_top, frame_buttom, frame_left, frame_right;
   gfloat pos;

   g_return_if_fail (tv);
   g_return_if_fail (thumb);

   node = g_list_find (thumbview_get_list(), tv);
   if (!node) return;

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   g_return_if_fail (tt);

   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   g_return_if_fail (thumb_data);

   button = thumb_data->vbox;
   /* button = thumb_data->button; */

   scrollwin = GTK_SCROLLED_WINDOW (tv->container);
   hadj = gtk_scrolled_window_get_hadjustment (scrollwin);
   vadj = gtk_scrolled_window_get_vadjustment (scrollwin);

   left = button->allocation.x;
   right = left + button->allocation.width;
   top = button->allocation.y;
   buttom = top + button->allocation.height;

   frame_left = hadj->value;
   frame_right = frame_left + hadj->page_size;
   frame_top = vadj->value;
   frame_buttom = frame_top + vadj->page_size;

   if (right > frame_right) {
      pos = right - (int) hadj->page_size;
      gtk_adjustment_set_value (hadj, pos);
   } else if (left < frame_left) {
      gtk_adjustment_set_value (hadj, left);
   }

   if (buttom > frame_buttom) {
      pos = buttom - (int) vadj->page_size;
      gtk_adjustment_set_value (vadj, pos);
   } else if (top < frame_top) {
      gtk_adjustment_set_value (vadj, top);
   }
}


gboolean
thumbtable_set_selection (Thumbnail *thumb, gboolean select)
{
   ThumbData *thumb_data;
   GtkWidget *button;

   g_return_val_if_fail (thumb, FALSE);

   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   g_return_val_if_fail (thumb_data, FALSE);

   thumb->selected = select;
   button = thumb_data->button;
   gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), thumb->selected);

   /*
     if (thumb->selected) {
     hilight_thumbnail (button);
     } else {
     if (!normal_style)
     normal_style = gtk_style_copy (gtk_widget_get_style(button));
     set_style(button, normal_style);
     }
   */

   return TRUE;
}


void
thumbtable_set_focus (ThumbView *tv, Thumbnail *thumb)
{
   ThumbTableData *tt;
   ThumbData *thumb_data;

   g_return_if_fail (tv);

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   g_return_if_fail (tt);

   if (thumb) {
      thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
      g_return_if_fail (thumb_data);
      if (GTK_IS_WIDGET (thumb_data->button))
         gtk_widget_grab_focus (thumb_data->button);
   } else if (tt->focused) {
         gtk_widget_grab_focus (tt->event_box);
   }
}


Thumbnail *
thumbtable_get_focus (ThumbView *tv)
{
   ThumbTableData *tt;

   g_return_val_if_fail (tv, NULL);

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   g_return_val_if_fail (tt, NULL);

   return tt->focused;
}


gboolean
thumbtable_thumbnail_is_in_viewport (ThumbView *tv, Thumbnail *thumb)
{
   ThumbData *thumb_data;
   GdkRectangle area, cell_area, intersect_area;
   GtkAdjustment *vadj;

   g_return_val_if_fail (tv, FALSE);
   g_return_val_if_fail (thumb, FALSE);

   thumb_data = g_hash_table_lookup (thumb->mode_data, THUMB_TABLE_LABEL);
   g_return_val_if_fail (thumb_data, FALSE);

   /* table area */
   gtkutil_get_widget_area (tv->container, &area);

   /* button area */
   gtkutil_get_widget_area (thumb_data->vbox, &cell_area);
   vadj = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (tv->container));
   cell_area.y -= vadj->value;

   /* intersect? */
   if (gdk_rectangle_intersect (&area, &cell_area, &intersect_area))
      return TRUE;
   else
      return FALSE;
}


/*
 *  thumbtable_create:
 *     @ Create thumbnail table widget.
 *
 *  tv        : Pointer to the ThumbView struct.
 *  dest_mode : New display mode.
 *  Return    : GtkTable widget.
 */
GtkWidget *
thumbtable_create (ThumbView *tv, gchar *dest_mode)
{
   ThumbTableData *tt;
   gint table_rows;
   gint row_space, col_space;
   gboolean centering;

   g_return_val_if_fail (tv, NULL);

   tt = g_hash_table_lookup (tv->disp_mode_data, THUMB_TABLE_LABEL);
   if (!tt)
      tt = thumbtable_new (tv);

   table_rows = calc_thumbtable_col_row_num (tv, 0);

   tt->event_box = gtk_event_box_new ();

   if (!strcmp (THUMB_TABLE_LABEL, dest_mode)) {
      gtk_widget_set_name (tt->event_box, "ThumbnailMode");
   } else if (!strcmp (RENAME_MODE_LABEL, dest_mode)) {
      gtk_widget_set_name (tt->event_box, "RenameMode");
   }

   thumbtable_prefs_get_value ("row_space", (gpointer) &row_space);
   thumbtable_prefs_get_value ("col_space", (gpointer) &col_space);
   thumbtable_prefs_get_value ("centering", (gpointer) &centering);

   tt->hbox = gtk_hbox_new (centering, 0);
   tt->table = gtk_table_new (table_rows, tt->colnum, FALSE);
   gtk_container_set_border_width (GTK_CONTAINER (tt->table), 5);
   gtk_container_add (GTK_CONTAINER (tt->event_box), tt->hbox);
   gtk_box_pack_start (GTK_BOX (tt->hbox), tt->table, FALSE, FALSE, 0);

   gtk_table_set_row_spacings (GTK_TABLE (tt->table), row_space);
   gtk_table_set_col_spacings (GTK_TABLE (tt->table), col_space);

   gtk_widget_show (tt->event_box);
   gtk_widget_show (tt->hbox);
   gtk_widget_show (tt->table);

   /* set callback */
   gtk_signal_connect (GTK_OBJECT (tt->event_box), "expose_event",
                       GTK_SIGNAL_FUNC (cb_expose), tv);

   /* for drag and drop */
   dnd_dest_set (tt->event_box, thumbtable_dnd_targets, thumbtable_dnd_targets_num);
   gtk_signal_connect(GTK_OBJECT (tt->event_box), "drag_data_received",
                      GTK_SIGNAL_FUNC (thumbview_drag_data_received_cb), tv);
   gtk_signal_connect(GTK_OBJECT (tt->event_box), "drag_end",
                      GTK_SIGNAL_FUNC (thumbview_drag_end_cb), tv);
   gtk_object_set_data (GTK_OBJECT (tt->event_box), "gimv-tab", tv);

   /* create buttons */
   thumbtable_append_thumb_frames (tv, tv->thumblist, dest_mode);

   return tt->event_box;
}
