/* -*- Mode: C; tab-width: 3; indent-tabs-mode: nil; c-basic-offset: 3 -*- */

/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: listview.c,v 1.3 2003/06/18 09:26:54 makeinu Exp $
 */

#include <string.h>

#include "gimageview.h"

#include "dnd.h"
#include "fileutil.h"
#include "gimv_thumb.h"
#include "thumbnail_view.h"
#include "gimv_image_info.h"
#include "gimv_plugin.h"
#include "gtk2-compat.h"
#include "icon_widget.h"
#include "listview.h"
#include "listview_prefs.h"

#define RAW_SPACE 1
#define COL_SPACE 15
#define BORDER_WIDTH 5


static ThumbViewPlugin list_view_modes[] =
{
   {GIMV_THUMBNAIL_VIEW_IF_VERSION,
    N_("List (Icon)"),
    200,
    listview_create,
    list_view_append_thumb_frames,
    listview_add_thumbnail,
    listview_refresh_thumbnail,
    NULL,
    listview_redraw,
    listview_resize,
    listview_adjust,
    listview_remove_thumbview_data,
    listview_remove_thumbnail_data,
    listview_set_selection,
    listview_set_focus,
    listview_get_focus,
    listview_thumbnail_is_in_viewport},

   {GIMV_THUMBNAIL_VIEW_IF_VERSION,
    N_("List (Thumbnail)"),
    210,
    listview_create,
    list_view_append_thumb_frames,
    listview_add_thumbnail,
    listview_refresh_thumbnail,
    NULL,
    listview_redraw,
    listview_resize,
    listview_adjust,
    listview_remove_thumbview_data,
    listview_remove_thumbnail_data,
    listview_set_selection,
    listview_set_focus,
    listview_get_focus,
    listview_thumbnail_is_in_viewport},
};

GIMV_PLUGIN_GET_IMPL(list_view_modes, GIMV_PLUGIN_THUMBVIEW_EMBEDER)

GimvPluginInfo gimv_plugin_info =
{
   if_version:    GIMV_PLUGIN_IF_VERSION,
   name:          N_("Thumbnail View List Mode"),
   version:       "0.0.2",
   author:        N_("Takuro Ashie"),
   description:   NULL,
   get_implement: gimv_plugin_get_impl,
   get_mime_type: NULL,
   get_prefs_ui:  gimv_prefs_ui_listview_get_page,
};


typedef struct ThumbViewData_Tag
{
   GtkWidget *table;
   GtkWidget *eventbox;
   GtkWidget *hbox;
   gint       rownum;
   gint       colnum;
   gchar     *dest_mode;

   GimvThumb *focused;
} ThumbViewData;


typedef struct GimvThumbData_Tag
{
   GtkWidget *widget;
} GimvThumbData;


/* for converting thumbnail data to string */
typedef gchar *(*ListViewDataStr) (GimvThumb *thumb);

typedef struct _ListViewDisplayData
{
   gchar             *title;
   ListViewDataStr    func;
} ListViewDisplayData;


static gchar *label_filename   (GimvThumb *thumb);
static gchar *label_size       (GimvThumb *thumb);
static gchar *label_mtime      (GimvThumb *thumb);
static gchar *label_image_type (GimvThumb *thumb);

static ThumbViewData *listview_new                    (ThumbView *tv);
static void           calc_col_row_num                (ThumbView *tv,
                                                       gchar     *mode,
                                                       gint      *colnum,
                                                       gint      *rownum);
gchar                *listview_create_label_str       (GimvThumb *thumb);
static gboolean       listview_append_thumb_frame     (ThumbView *tv,
                                                       GimvThumb *thumb,
                                                       gchar     *dest_mode);
void                  listview_create_title_idx_list  ();

static ListViewDisplayData listview_display_data [] =
{
   {N_("Name"),  label_filename},
   {N_("Size"),  label_size},
   {N_("Time"),  label_mtime},
   {N_("Type"),  label_image_type},
};
static gint listview_display_data_num
= sizeof (listview_display_data) / sizeof (ListViewDisplayData);

GtkTargetEntry listview_dnd_targets[] = {
   {"text/uri-list",              0, TARGET_URI_LIST},
};
const gint listview_dnd_targets_num = sizeof(listview_dnd_targets) / sizeof(GtkTargetEntry);

static gchar *config_order_string = NULL;
static GList *listview_title_idx_list      = NULL;
static gint   listview_title_idx_list_num  = 0;
static gboolean listview_dragging = FALSE;


/******************************************************************************
 *
 *   callback functions.
 *
 ******************************************************************************/
static gboolean
cb_icon_key_press (GtkWidget *widget,
                   GdkEventKey *event,
                   GimvThumb *thumb)
{
   ThumbView *tv;

   g_return_val_if_fail (event, FALSE);

   if (!thumb) return FALSE;

   tv = gimv_thumb_get_parent_thumbview (thumb);
   g_return_val_if_fail (tv, FALSE);

   if (thumbview_thumb_key_press_cb(widget, event, thumb))
      return FALSE;

   {
      ThumbViewData *tv_data;
      gint pos = g_list_index (tv->thumblist, thumb);
      gint col, row;

      tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
      g_return_val_if_fail (tv_data, FALSE);

      col = pos / tv_data->rownum;
      row = pos % tv_data->rownum;

      switch (event->keyval) {
      case GDK_Left:
         if (col == 0) return TRUE;
         break;
      case GDK_Right:
         if (col == tv->filenum / tv_data->rownum
             || (col + 1) * tv_data->rownum + row >= tv->filenum)
         {
            return TRUE;
         }
         break;
      case GDK_Up:
         if (row == 0) return TRUE;
         break;
      case GDK_Down:
         if (row == tv_data->rownum - 1 || pos ==  tv->filenum - 1)
         {
            return TRUE;
         }
         break;
      case GDK_Return:
         if (event->state & GDK_SHIFT_MASK || event->state & GDK_CONTROL_MASK) {
            /* is there somteing to do? */
         } else {
            thumbview_set_selection_all (tv, FALSE);
         }
         thumbview_set_selection (thumb, TRUE);
         thumbview_open_image (tv, thumb, 0);
         break;
      case GDK_space:
         thumbview_set_selection (thumb, !thumb->selected);
         break;
      case GDK_Delete:
         thumbview_delete_files (tv);
         break;
      default:
         break;
      }
   }

   return FALSE;
}


static gboolean
cb_icon_button_press (GtkWidget *icon, GdkEventButton *event, gpointer data)
{
   GimvThumb *thumb = data;
   ThumbView *tv;
   gboolean clear = TRUE, found = FALSE;

   listview_dragging = FALSE;

   g_return_val_if_fail (icon && thumb, TRUE);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, TRUE);

   /* set selection */
   if (event->type == GDK_BUTTON_PRESS && (event->button == 1)) {
      if (event->state & GDK_SHIFT_MASK) {
         if (event->state & GDK_CONTROL_MASK)
            clear = FALSE;

         found = thumbview_set_selection_multiple (thumb, TRUE, clear);
         if (!found)
            thumbview_set_selection_multiple (thumb, FALSE, clear);
      } else if (!thumb->selected) {
         if (~event->state & GDK_CONTROL_MASK)
            thumbview_set_selection_all (tv, FALSE);
         thumbview_set_selection (thumb, TRUE);
      } else if (thumb->selected) {
         if (event->state & GDK_CONTROL_MASK)
            thumbview_set_selection (thumb, FALSE);
      }
   }

   gtk_widget_grab_focus (icon);

   /* relay management to reference callback function */
   thumbview_thumb_button_press_cb (icon, event, thumb);

   return TRUE;
}


static gboolean
cb_icon_button_release (GtkWidget *icon, GdkEventButton *event, gpointer data)
{
   GimvThumb *thumb = data;
   ThumbView *tv;

   g_return_val_if_fail (icon && thumb, TRUE);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, TRUE);

   /* set selection */
   if (event->type == GDK_BUTTON_RELEASE
       && (event->button == 1)
       && !(event->state & GDK_SHIFT_MASK)
       && !(event->state & GDK_CONTROL_MASK)
       && !listview_dragging)
   {
      thumbview_set_selection_all (tv, FALSE);
      thumbview_set_selection (thumb, TRUE);
   }

   gtk_widget_grab_focus (icon);

   /* relay management to reference callback function */
   thumbview_thumb_button_release_cb (icon, event, thumb);

   return TRUE;
}


static gboolean
cb_icon_motion_notify (GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
   GimvThumb *thumb = data;

   listview_dragging = TRUE;

   g_return_val_if_fail (thumb, FALSE);

   return thumbview_motion_notify_cb (widget, event, thumb);
}


static gboolean
cb_icon_focus_in (GtkWidget *icon, GdkEventFocus *event, GimvThumb *thumb)
{
   ThumbView *tv;
   ThumbViewData *tv_data;
   GimvThumbData *thumb_data;

   g_return_val_if_fail (thumb, FALSE);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, FALSE);

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   thumb_data = g_hash_table_lookup (thumb->mode_data, LIST_VIEW_LABEL);
   g_return_val_if_fail (tv_data && thumb_data, FALSE);

   if (tv_data->focused == thumb) return FALSE;
   tv_data->focused = thumb;

   listview_adjust (tv, thumb);

   return FALSE;
}


static gboolean
cb_expose (GtkWidget *widget, GdkEventExpose *event, ThumbView *tv)
{
   thumbview_resize (tv);

   return FALSE;
}



/******************************************************************************
 *
 *   private functions.
 *
 ******************************************************************************/
static gchar *
label_filename (GimvThumb *thumb)
{
   const gchar *filename;
   gchar *tmpstr;
   gchar buf[BUF_SIZE];
   gboolean show_title;

   g_return_val_if_fail (thumb, NULL);

   filename = g_basename(gimv_image_info_get_path (thumb->info));

   tmpstr = gimv_filename_to_internal (filename);

   listview_prefs_get_value("show_title", (gpointer *) &show_title);
   if (show_title)
      g_snprintf (buf, BUF_SIZE, _("Name : %s"), tmpstr);
   else
      return tmpstr;

   g_free (tmpstr);

   return g_strdup (buf);
}


static gchar *
label_size (GimvThumb *thumb)
{
   gchar *size_str, buf[BUF_SIZE];
   gboolean show_title;

   g_return_val_if_fail (thumb, NULL);

   size_str = fileutil_size2str (thumb->info->st.st_size, FALSE);
   listview_prefs_get_value("show_title", (gpointer *) &show_title);
   if (show_title)
      g_snprintf (buf, BUF_SIZE, _("Size : %s bytes"), size_str);
   else
      g_snprintf (buf, BUF_SIZE, _("%s bytes"), size_str);
   g_free (size_str);

   return g_strdup (buf);
}


static gchar *
label_mtime (GimvThumb *thumb)
{
   gchar *time_str, *str;
   gboolean show_title;

   g_return_val_if_fail (thumb, NULL);

   time_str = fileutil_time2str (thumb->info->st.st_mtime);
   listview_prefs_get_value("show_title", (gpointer *) &show_title);
   if (show_title)
      str = g_strconcat (_("Time : "), time_str, NULL);
   else
      return time_str;
   g_free (time_str);

   return str;
}


static gchar *
label_image_type (GimvThumb *thumb)
{
   gchar buf[BUF_SIZE];
   const gchar *filename;
   gboolean show_title;

   g_return_val_if_fail (thumb, NULL);

   filename = gimv_image_info_get_path (thumb->info);
   listview_prefs_get_value("show_title", (gpointer *) &show_title);
   if (show_title)
      g_snprintf (buf, BUF_SIZE, _("Type: %s"),
                  gimv_image_detect_type_by_ext (filename));
   else
      return g_strdup (gimv_image_detect_type_by_ext (filename));

   return g_strdup (buf);
}


static ThumbViewData *
listview_new (ThumbView *tv)
{
   ThumbViewData *tv_data = NULL;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   if (!tv_data) {
      tv_data = g_new0 (ThumbViewData, 1);
      tv_data->rownum = 20;
      tv_data->colnum = 20;
      g_hash_table_insert (tv->disp_mode_data, LIST_VIEW_LABEL, tv_data);
   }

   return tv_data;
}


static void
calc_col_row_num (ThumbView *tv, gchar *mode, gint *colnum, gint *rownum)
{
   GtkScrolledWindow *scrollwin;
   GtkAdjustment *vadj;
   GList *node;
   gint frame_height, icon_size;
   gchar *label;

   g_return_if_fail (tv);

   node = g_list_find (thumbview_get_list(), tv);
   if (!node) return;

   label = mode;
   if (!strcmp (label, LIST_THUMB_LABEL))
      icon_size = tv->ThumbnailSize;
   else
      icon_size = ICON_SIZE;

   scrollwin = GTK_SCROLLED_WINDOW (tv->container);
   vadj = gtk_scrolled_window_get_vadjustment (scrollwin);
   frame_height = vadj->page_size;

   if (frame_height > 0.0)
      *rownum = frame_height / (icon_size + RAW_SPACE + 4);
   else
      *rownum = 10;
   if (!strcmp (label, LIST_THUMB_LABEL))
      ++*rownum;

   if (*rownum < 1) *rownum = 1;

   *colnum = g_list_length (tv->thumblist) / *rownum + 1;
}


static gboolean
calc_thumbframe_pos (GimvThumb *thumb, gint *col, gint *row)
{
   ThumbView *tv;
   ThumbViewData *tv_data;
   GList *node;
   gint pos;

   if (!thumb) return FALSE;

   tv = thumb->thumb_view;
   if (!tv) return FALSE;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   if (!tv_data) return FALSE;

   node = g_list_find (tv->thumblist, thumb);

   if (!node)
      return FALSE;

   pos = g_list_position (tv->thumblist, node);

   *row = pos % tv_data->rownum;
   *col = pos / tv_data->rownum;

   return TRUE;
}


gchar *
listview_create_label_str (GimvThumb *thumb)
{
   GList *node;
   gchar *label;

   g_return_val_if_fail (thumb, NULL);

   /* create icon label string */
   node = listview_title_idx_list;
   label = NULL;
   while (node) {
      gchar *tmpstr, *oldstr;
      gint idx = GPOINTER_TO_INT (node->data);

      if (!label) {
         label = listview_display_data[idx].func (thumb);
      } else {
         tmpstr = listview_display_data[idx].func (thumb);
         oldstr = label;
         label = g_strconcat (label, "\n", tmpstr, NULL);
         g_free (tmpstr);
         g_free (oldstr);
      }

      node = g_list_next (node);
   }

   return label;
}


static gboolean
listview_append_thumb_frame (ThumbView *tv, GimvThumb *thumb, gchar *dest_mode)
{
   ThumbViewData *tv_data;
   GimvThumbData *thumb_data;
   GdkPixmap *thumbnail = NULL, *icon = NULL;
   GdkBitmap *thumbnail_mask = NULL, *icon_mask = NULL;
   GtkWidget *widget = NULL;
   gint col, row, frame_size;
   gchar *label, *icon_label;
   gboolean retval = FALSE;

   g_return_val_if_fail (tv, FALSE);

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   g_return_val_if_fail (tv_data, FALSE);

   thumb_data = g_hash_table_lookup (thumb->mode_data, LIST_VIEW_LABEL);
   if (!thumb_data) {
      thumb_data = g_new0 (GimvThumbData, 1);
      g_hash_table_insert (thumb->mode_data, LIST_VIEW_LABEL, thumb_data);
   }

   label = dest_mode;
   if (!strcmp (label, LIST_THUMB_LABEL)) {
      frame_size = tv->ThumbnailSize;
      icon_label = listview_create_label_str (thumb);
   } else {
      const gchar *filename = g_basename(gimv_image_info_get_path (thumb->info));
      icon_label = gimv_filename_to_internal (filename);
      frame_size = ICON_SIZE;
   }

   thumb_data->widget = icon_widget_new (NULL, NULL,
                                         icon_label,
                                         frame_size, frame_size);
   g_free (icon_label);

   gimv_thumb_get_icon  (thumb, &icon, &icon_mask);
   gimv_thumb_get_thumb (thumb, &thumbnail, &thumbnail_mask);

   if (!strcmp (label, LIST_VIEW_LABEL)) {
      if (icon) {
         icon_widget_set_pixmap (ICON_WIDGET (thumb_data->widget),
                                 icon, icon_mask);
         retval = TRUE;
      }
   } else if (!strcmp (label, LIST_THUMB_LABEL)) {
      ICON_WIDGET (thumb_data->widget)->xpad = 4;
      ICON_WIDGET (thumb_data->widget)->ypad = 4;
      if (thumbnail) {
         icon_widget_set_pixmap (ICON_WIDGET (thumb_data->widget),
                                 thumbnail, thumbnail_mask);
         retval = TRUE;
      }
   }

   gtk_widget_show (thumb_data->widget);

   /* set callback */
   gtk_signal_connect_after (GTK_OBJECT (thumb_data->widget), "key-press-event",
                             GTK_SIGNAL_FUNC (cb_icon_key_press), thumb);
   gtk_signal_connect_after (GTK_OBJECT (thumb_data->widget),"button_press_event",
                             GTK_SIGNAL_FUNC (cb_icon_button_press), thumb);
   SIGNAL_CONNECT_TRANSRATE_SCROLL (thumb_data->widget);
   gtk_signal_connect (GTK_OBJECT(thumb_data->widget),"button_release_event",
                       GTK_SIGNAL_FUNC(cb_icon_button_release), thumb);
   gtk_signal_connect (GTK_OBJECT(thumb_data->widget),"motion_notify_event",
                       GTK_SIGNAL_FUNC(cb_icon_motion_notify), thumb);
   gtk_signal_connect (GTK_OBJECT(thumb_data->widget),"focus_in_event",
                       GTK_SIGNAL_FUNC(cb_icon_focus_in), thumb);

   /* for drag file list */
   dnd_src_set (thumb_data->widget, listview_dnd_targets, listview_dnd_targets_num);
   gtk_signal_connect (GTK_OBJECT (thumb_data->widget), "drag_begin",
                       GTK_SIGNAL_FUNC (thumbview_drag_begin_cb), tv);
   gtk_signal_connect (GTK_OBJECT (thumb_data->widget), "drag_data_get",
                       GTK_SIGNAL_FUNC (thumbview_drag_data_get_cb), tv);
   gtk_signal_connect (GTK_OBJECT (thumb_data->widget), "drag-data-delete",
                       GTK_SIGNAL_FUNC (thumbview_drag_data_delete_cb), tv);
   gtk_signal_connect (GTK_OBJECT (thumb_data->widget), "drag_end",
                       GTK_SIGNAL_FUNC (thumbview_drag_end_cb), tv);
   gtk_object_set_data (GTK_OBJECT (thumb_data->widget), "gimv-tab", tv);

   /* attach to table */
   calc_thumbframe_pos (thumb, &col, &row);
   if (!strcmp (label, LIST_VIEW_LABEL)) {
      widget = gtk_hbox_new (FALSE, 0);
      gtk_widget_show (widget);
      gtk_box_pack_start (GTK_BOX (widget), thumb_data->widget, FALSE, FALSE, 0);
   } else if (!strcmp (label, LIST_THUMB_LABEL)) {
      widget = thumb_data->widget;
   }
   gtk_table_attach (GTK_TABLE (tv_data->table), widget,
                     col, col + 1, row, row + 1,
                     GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

   listview_set_selection (thumb, thumb->selected);

   return retval;
}



/******************************************************************************
 *
 *   public functions.
 *
 ******************************************************************************/
/*
 *  listview_remove_thumbnail_data:
 *
 *  tv : Pointer to the ThumbView struct.
 */
void
listview_remove_thumbview_data (ThumbView *tv)
{
   ThumbViewData *tv_data;

   if (!tv) return;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   if (!tv_data) return;

   g_hash_table_remove (tv->disp_mode_data, LIST_VIEW_LABEL); 
   g_free (tv_data);
}


/*
 *  listview_remove_thumbnail_data:
 *
 *  thumb : Pointer to the GimvThumb struct.
 */
void
listview_remove_thumbnail_data (GimvThumb *thumb)
{
   GimvThumbData *thumb_data;

   if (!thumb) return;

   thumb_data = g_hash_table_lookup (thumb->mode_data, LIST_VIEW_LABEL);
   if (!thumb_data) return;

   g_hash_table_remove (thumb->mode_data, LIST_VIEW_LABEL); 
   g_free (thumb_data);
}


GList *
list_view_append_thumb_frames (ThumbView *tv, GList *start, gchar *dest_mode)
{
   GimvThumb  *thumb;
   GList *node, *loadlist = NULL;
   gboolean thumb_loaded;

   g_return_val_if_fail (tv, FALSE);

   node = start;
   while (node) {
      thumb = (GimvThumb *) node->data;

      thumb_loaded = listview_append_thumb_frame (tv, thumb, dest_mode);
      if (!thumb_loaded)
         loadlist = g_list_append (loadlist, thumb);

      node = g_list_next (node);
   }

   return loadlist;
}


GtkWidget *
listview_add_thumbnail (GimvThumb *thumb, gchar *dest_mode,
                        ThumbLoadType type)
{
   GimvThumbData *thumb_data;
   GdkPixmap *icon = NULL, *thumbnail = NULL;
   GdkBitmap *icon_mask = NULL, *thumbnail_mask = NULL;
   gchar *label;

   g_return_val_if_fail (thumb, NULL);

   thumb_data = g_hash_table_lookup (thumb->mode_data, LIST_VIEW_LABEL);
   g_return_val_if_fail (thumb_data, NULL);

   label = dest_mode;

   gimv_thumb_get_icon (thumb, &icon, &icon_mask);
   gimv_thumb_get_thumb (thumb, &thumbnail, &thumbnail_mask);

   if (!strcmp (label, LIST_VIEW_LABEL) && icon) {
      icon_widget_set_pixmap (ICON_WIDGET (thumb_data->widget),
                              icon, icon_mask);
      return NULL;
   } else if (!strcmp (label, LIST_THUMB_LABEL) && thumbnail) {
      icon_widget_set_pixmap (ICON_WIDGET (thumb_data->widget),
                              thumbnail, thumbnail_mask);
      return NULL;
   }

   return NULL;
}


void
listview_redraw (ThumbView *tv, gchar *dest_mode,
                 GtkWidget *scroll_win, GList **loadlist)
{
   ThumbViewData *tv_data;
   GtkScrolledWindow *scrollwin;
   GList *node;

   g_return_if_fail (tv);

   node = g_list_find (thumbview_get_list(), tv);
   if (!node) return;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   if (!tv_data)
      tv_data = listview_new (tv);

   tv_data->dest_mode = dest_mode;

   if (tv_data->eventbox) {
      gtk_widget_destroy (tv_data->eventbox);
      tv_data->eventbox = NULL;
   }

   if (scroll_win) {
      if (GTK_BIN (tv->container)->child)
         gtk_widget_destroy (GTK_BIN (tv->container)->child);   
      tv_data->eventbox = listview_create (tv, dest_mode);
      scrollwin = GTK_SCROLLED_WINDOW (scroll_win);
      gtk_scrolled_window_add_with_viewport (scrollwin, tv_data->eventbox);
   }

   if (!loadlist) return;

   /* create thumbnail load list */
   *loadlist = NULL;
   node = tv->thumblist;
   while (node) {
      GimvThumb *thumb = node->data;
      GdkPixmap *pixmap = NULL;
      GdkBitmap *mask = NULL;

      gimv_thumb_get_thumb (thumb, &pixmap, &mask);
      if (!pixmap)
         *loadlist = g_list_append (*loadlist, thumb);
      node = g_list_next (node);
   }
}


/*
 *  listview_refresh_thumbnail:
 *     @ Create thumbnail from original image and draw it.
 *
 *  thumb  : Pointer to the GimvThumb struct.
 *  Return : TRUE if success.
 */
gboolean
listview_refresh_thumbnail (GimvThumb *thumb, ThumbLoadType type)
{
   GtkWidget *widget;
   ThumbView *tv;

   if (!thumb) return FALSE;

   tv = thumb->thumb_view;
   if (!tv) return FALSE;

   widget = listview_add_thumbnail (thumb, tv->disp_mode, type);

   if (widget)
      return TRUE;
   else
      return FALSE;
}


GtkWidget *
listview_resize (ThumbView *tv)
{
   ThumbViewData *tv_data;
   gint col, row;
   GList *node;

   g_return_val_if_fail (tv, NULL);

   node = g_list_find (thumbview_get_list(), tv);
   if (!node) return NULL;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   g_return_val_if_fail (tv_data, NULL);

   calc_col_row_num (tv, tv->disp_mode, &col, &row);
   if (row == tv_data->rownum) {
      return tv_data->eventbox;
   } else {
      tv_data->colnum = col;
      tv_data->rownum = row;
   }

#ifdef USE_GTK2
   gtk_container_foreach (GTK_CONTAINER(tv_data->table),
                          (GtkCallback) gtk_widget_destroy, NULL);
#else
   gtk_container_foreach (GTK_CONTAINER(tv_data->table),
                          GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);
#endif
   gtk_table_resize (GTK_TABLE (tv_data->table),
                     tv_data->rownum, tv_data->colnum);
   list_view_append_thumb_frames (tv, tv->thumblist, tv->disp_mode);

   return tv_data->eventbox;
}


void
listview_adjust (ThumbView *tv, GimvThumb *thumb)
{
   ThumbViewData *tv_data;
   GimvThumbData *thumb_data;   
   GList *node;
   GtkWidget *icon;
   GtkScrolledWindow *scrollwin;
   GtkAdjustment *hadj, *vadj;
   gint top, buttom, left, right;
   gint frame_top, frame_buttom, frame_left, frame_right;
   gfloat pos;

   g_return_if_fail (tv);

   node = g_list_find (thumbview_get_list(), tv);
   if (!node) return;

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   g_return_if_fail (tv_data);

   thumb_data = g_hash_table_lookup (thumb->mode_data, LIST_VIEW_LABEL);
   g_return_if_fail (thumb_data);

   icon = thumb_data->widget;
   g_return_if_fail (icon);

   scrollwin = GTK_SCROLLED_WINDOW (tv->container);
   hadj = gtk_scrolled_window_get_hadjustment (scrollwin);
   vadj = gtk_scrolled_window_get_vadjustment (scrollwin);

   left = icon->allocation.x;
   right = left + icon->allocation.width;
   top = icon->allocation.y;
   buttom = top + icon->allocation.height;

   frame_left = hadj->value;
   frame_right = frame_left + hadj->page_size;
   frame_top = vadj->value;
   frame_buttom = frame_top + vadj->page_size;

   if (right > frame_right) {
      pos = right - (int) hadj->page_size;
      gtk_adjustment_set_value (hadj, pos);
   } else if (left < frame_left) {
      gtk_adjustment_set_value (hadj, left);
   }

   if (buttom > frame_buttom) {
      pos = buttom - (int) vadj->page_size;
      gtk_adjustment_set_value (vadj, pos);
   } else if (top < frame_top) {
      gtk_adjustment_set_value (vadj, top);
   }
}


gboolean
listview_set_selection (GimvThumb *thumb, gboolean select)
{
   GimvThumbData *thumb_data;

   g_return_val_if_fail (thumb, FALSE);

   thumb_data = g_hash_table_lookup (thumb->mode_data, LIST_VIEW_LABEL);
   g_return_val_if_fail (thumb_data, FALSE);

   thumb->selected = select;
   if (thumb->selected) {
      gtk_widget_set_state (thumb_data->widget, GTK_STATE_SELECTED);
   } else {
      gtk_widget_set_state (thumb_data->widget, GTK_STATE_NORMAL);
   }

   return TRUE;
}


void
listview_set_focus (ThumbView *tv, GimvThumb *thumb)
{
   ThumbViewData *tv_data;
   GimvThumbData *thumb_data;

   g_return_if_fail (tv);

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   g_return_if_fail (tv_data);

   if (thumb) {
      thumb_data = g_hash_table_lookup (thumb->mode_data, LIST_VIEW_LABEL);
      g_return_if_fail (thumb_data);
      if (GTK_IS_WIDGET (thumb_data->widget))
         gtk_widget_grab_focus (thumb_data->widget);
   } else if (tv_data->focused) {
         gtk_widget_grab_focus (tv_data->eventbox);
   }
}


GimvThumb *
listview_get_focus (ThumbView *tv)
{
   ThumbViewData *tv_data;

   g_return_val_if_fail (tv, NULL);

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   g_return_val_if_fail (tv_data, NULL);

   return tv_data->focused;
}


gboolean
listview_thumbnail_is_in_viewport (ThumbView *tv, GimvThumb *thumb)
{
   GimvThumbData *thumb_data;
   GdkRectangle area, cell_area, intersect_area;
   GtkAdjustment *hadj;

   g_return_val_if_fail (tv, FALSE);
   g_return_val_if_fail (thumb, FALSE);

   thumb_data = g_hash_table_lookup (thumb->mode_data, LIST_VIEW_LABEL);
   g_return_val_if_fail (thumb_data, FALSE);

   /* table area */
   gtkutil_get_widget_area (tv->container, &area);

   /* cell area */
   gtkutil_get_widget_area (thumb_data->widget, &cell_area);
   hadj = gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW (tv->container));
   cell_area.x -= hadj->value;

   /* intersect? */
   if (gdk_rectangle_intersect (&area, &cell_area, &intersect_area))
      return TRUE;
   else
      return FALSE;
}


static void
set_style (GtkWidget *widget, GtkStyle *style)
{
   gtk_widget_set_style (widget, style);

   if (GTK_IS_CONTAINER(widget))
#ifdef USE_GTK2
      gtk_container_foreach (GTK_CONTAINER (widget),
                             (GtkCallback) set_style, style);
#else
   gtk_container_foreach (GTK_CONTAINER (widget),
                          GTK_SIGNAL_FUNC(set_style), style);
#endif
}


GtkWidget *
listview_create (ThumbView *tv, gchar *dest_mode)
{
   ThumbViewData *tv_data = NULL;
   GtkWidget *hbox;

   g_return_val_if_fail (tv, NULL);

   tv_data = g_hash_table_lookup (tv->disp_mode_data, LIST_VIEW_LABEL);
   if (!tv_data) {
      tv_data = listview_new (tv);
      g_return_val_if_fail (tv_data, NULL);
   }

   listview_create_title_idx_list ();

   calc_col_row_num (tv, dest_mode, &tv_data->colnum, &tv_data->rownum);

   tv_data->eventbox = gtk_event_box_new ();

   if (!strcmp (LIST_VIEW_LABEL, dest_mode)) {
      gtk_widget_set_name (tv_data->eventbox, "ListIconMode");
   } else if (!strcmp (LIST_THUMB_LABEL, dest_mode)) {
      gtk_widget_set_name (tv_data->eventbox, "ListThumbMode");
   }

   tv_data->hbox = hbox = gtk_hbox_new (FALSE, 0);
   tv_data->table = gtk_table_new (tv_data->rownum, tv_data->colnum, FALSE);
   gtk_table_set_row_spacings (GTK_TABLE (tv_data->table), RAW_SPACE);
   gtk_table_set_col_spacings (GTK_TABLE (tv_data->table), COL_SPACE);
   gtk_container_set_border_width (GTK_CONTAINER (tv_data->table), BORDER_WIDTH);

   gtk_container_add (GTK_CONTAINER (tv_data->eventbox), hbox);
   gtk_box_pack_start (GTK_BOX (hbox), tv_data->table, FALSE, FALSE, 0);

   gtk_widget_show (tv_data->table);
   gtk_widget_show (hbox);
   gtk_widget_show (tv_data->eventbox);

   /* set callback */
   gtk_signal_connect (GTK_OBJECT (tv_data->eventbox), "expose_event",
                       GTK_SIGNAL_FUNC (cb_expose), tv);

   /* for drag and drop */
   dnd_dest_set (tv_data->eventbox, listview_dnd_targets, listview_dnd_targets_num);
   gtk_signal_connect(GTK_OBJECT (tv_data->eventbox), "drag_data_received",
                      GTK_SIGNAL_FUNC (thumbview_drag_data_received_cb), tv);
   gtk_signal_connect(GTK_OBJECT (tv_data->eventbox), "drag_end",
                      GTK_SIGNAL_FUNC (thumbview_drag_end_cb), tv);
   gtk_object_set_data (GTK_OBJECT (tv_data->eventbox), "gimv-tab", tv);

   /* set icons */
   list_view_append_thumb_frames (tv, tv->thumblist, dest_mode);

   return tv_data->eventbox;
}



/******************************************************************************
 *
 *   for preference.
 *
 ******************************************************************************/
void
listview_create_title_idx_list (oid)
{
   gchar *data_order, **titles;
   gint i = 0;

   listview_prefs_get_value("data_order", (gpointer *) &data_order);

   if (!data_order) {
      config_order_string = NULL;
      if (listview_title_idx_list)
         g_list_free (listview_title_idx_list);
      listview_title_idx_list_num = 0;
      return;
   }

   if (data_order == config_order_string) return;

   if (listview_title_idx_list) g_list_free (listview_title_idx_list);
   listview_title_idx_list = NULL;

   titles = g_strsplit (data_order, ",", -1);

   g_return_if_fail (titles);

   listview_title_idx_list_num = 0;
   config_order_string = data_order;

   while (titles[i]) {
      gint idx;
      idx = listview_get_title_idx (titles[i]);
      if (idx >= 0) {
         listview_title_idx_list = g_list_append (listview_title_idx_list,
                                                  GINT_TO_POINTER (idx));
         listview_title_idx_list_num++;
      }
      i++;
   }

   g_strfreev (titles);
}


gint
listview_get_titles_num (void)
{
   return listview_display_data_num;
}


gchar *
listview_get_title (gint idx)
{
   g_return_val_if_fail (idx >= 0 && idx < listview_display_data_num, NULL);

   return listview_display_data[idx].title;
}


gint
listview_get_title_idx (const gchar *title)
{
   gint i;

   g_return_val_if_fail (title, -1);

   for (i = 0; i < listview_display_data_num; i++) {
      if (!listview_display_data[i].title) continue;
      if (!strcmp (listview_display_data[i].title, title))
         return i;
   }

   return -1;
}


void
listview_apply_config (void)
{
   GList *node;

   listview_create_title_idx_list ();

   node = thumbview_get_list();

   while (node) {
      ThumbView *tv = node->data;

      if (!strcmp (tv->disp_mode, LIST_THUMB_LABEL)) {
         thumbview_redraw (tv, tv->disp_mode, tv->container, NULL);
      }

      node = g_list_next (node);
   }
}
