#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# This file is part of G-language Genome Analysis Environment package
#
#     Copyright (C) 2001-2014 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
#   $Id: Util.pm,v 1.1.1.1 2002/04/02 20:25:43 gaou Exp $
#
# G-language GAE is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
# 
# G-language GAE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with G-language GAE -- see the file COPYING.
# If not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
#END_HEADER
#
# written by Kazuharu Arakawa <gaou@sfc.keio.ac.jp> at
# G-language Project, Institute for Advanced Biosciences, Keio University.
#

package G::Seq::GenomeMap;

use SubOpt;
use G::Messenger;
use G::Tools::GPAC;
use G::Tools::GMap;
use G::Tools::Statistics;
use G::Seq::GCskew;
use G::DB::SDB;

use GD;
#use GD::SVG;
use SVG;

use strict;
use base qw(Exporter);
use autouse 'Cwd'=>qw(getcwd);
use SelfLoader;

our @EXPORT = qw(
		 dnawalk
		 genome_map
		 genome_map2
		 genome_map3	 
		 plasmid_map
		 circular_map
		 seq2png
		 );


#:::::::::::::::::::::::::::::::::
#       Perldoc
#:::::::::::::::::::::::::::::::::


=head1 NAME

  G::Seq::GenomeMap - Collection of methods for creating genome maps.

=head1 DESCRIPTION

    This class is a part of G-language Genome Analysis Environment, 
    collecting methods for creating genome maps.

=cut

#::::::::::::::::::::::::::::::
#        Constants
#::::::::::::::::::::::::::::::

my %COG_fcode = (
		 J=>"Translation, ribosomal structure and biogenesis",
		 K=>"Transcription",
		 L=>"DNA replication, recombination and repair",
		 D=>"Cell division and chromosome partitioning",
		 O=>"Posttranslational modification, protein turnover, chaperones",
		 M=>"Cell envelope biogenesis, outer membrane",
		 N=>"Cell motility and secretion",
		 P=>"Inorganic ion transport and metabolism",
		 T=>"Signal transduction mechanisms",
		 C=>"Energy production and conservation",
		 G=>"Carbohydrate transport and metabolism",
		 E=>"Amino acid transport and metabolism",
		 F=>"Nucleotide transport and metabolism",
		 H=>"Coenzyme metabolism",
		 I=>"Lipid metabolism",
		 Q=>"Secondary metabolites biosynthesis, transport and catabolism",
		 R=>"General function prediction only",
		 S=>"Function unknown",
		 '-'=>"Non COG"
		 );

my %COG_fcolor = (
		  J=>"plum",
		  K=>"fuchsia",
		  L=>"pink",
		  D=>"lightgreen",
		  O=>"green",
		  M=>"khaki",
		  N=>"greenyellow",
		  P=>"darkkhaki",
		  T=>"cyan",
		  C=>"blue",
		  G=>"mediumturquoise",
		  E=>"lightskyblue",
		  F=>"mediumpurple",
		  H=>"aqua",
		  I=>"blueviolet",
		  Q=>"lightskyblue",
		  R=>"gainsboro",
		  S=>"darkgrey",
		  '-'=>"aliceblue"
		  );

my $cx = 4000;
my $cy = 4000;
my $pi = 4 * atan2(1,1);



#::::::::::::::::::::::::::::::
#        Methods Start
#::::::::::::::::::::::::::::::


=head2 circular_map

  Name: circular_map   -   draws circular map of the genome

  Description:
   This method creates a circular map of the genome using SVG,
   suitable for plasmids and circular bacterial chromosomes.

   From the outer ring inwards, genes on direct strand (pink), 
   genes on complementary strand (yellow), tRNAs (green arrows), 
   rRNAs (pink or orange stripes depending on the strand), 
   GC content (brown lines), GC skew (yellow lines). Replication 
   origin and terminus predicted from the GC skew shift points 
   are also labeled. 

   Generated SVG is very large (8000x8000 pixels), and it is
   suited to be converted to Google Map View with generate_gmap().

   Output file is created in the "graph" directory, with 
   the accession number as file name and ".svg" as extension.

  Usage:
    NULL = circular_map($gb);

 Options:
    -filename       output filename (default:'circular_map.svg')  
    -gmap           1 to create Google Map view, 0 to only generate PNG image
                    (default:1)
    -optionalText   optional text to describe the genome.

  Author: 
    Nobuhiro Kido (t07505nk@sfc.keio.ac.jp)

  History:
   20120824-01 added -filename option
   20100318-01 major update including support for linear chromosomes,
               change of gene color by strand in circular genomes,
               addition of accession number and optional text,
               updated color coding for GC content and GC skew.
   20080414-01 added gmap option
   20071023-01 initial posting

=cut


sub circular_map{
    opt_default("gmap"=>1, "filename"=>"circular_map.svg", "output"=>"g");
    my @args     = opt_get(@_);
    my $gb       = opt_as_gb(shift @args);
    my $filename = opt_val("filename");
    my $gmap     = opt_val("gmap");
    my $optionalText = opt_val("optionalText");
    mkdir( "data",  0777 );
    mkdir( "graph", 0777 );

    msg_error("Map generation will take a few minutes. Please make yourself comfortable until the image is generated ;-\)\n\n") if ($gmap);

    my $wfile = $gb->{LOCUS}->{id};
    my ( $ori, $ter ) = find_ori_ter( $gb, -output => "/dev/null" );
    my $bp_length = int(length( $gb->{SEQ} ));
    my ( $count_a, $count_b );

    my $svg = SVG->new(
        width         => $cx * 2,
        height        => $cy * 2,
        'xmlns:xlink' => 'http://www.w3.org/1999/xlink'
    );

    my $tag = $svg->rect(
        width  => $cx * 2,
        height => $cy * 2,
        x      => 0,
        y      => 0,
        fill   => 'black'
    );

    my $y = $svg->group(
        id => 'group_y',
        style =>
          { 'stroke' => '#ff9966', 'fill' => 'none', 'stroke-width' => 200 }
    );

    $svg->circle(
		 id  => "cdsringbackground",
		 cx  => $cx,
		 cy  => $cy,
		 r   => 3250,
		 'stroke' => '#222222', 
		 'opacity' => 0.8,
		 'fill' => 'none', 
		 'stroke-width' => 500
		 );

    my $ry  = $cy - 3250; #middle = 350

    if($gb->{LOCUS}->{circular} == 0){
	my $r = 3550;
	my ( $rx, $ry2 ) = &get_pos( int(length($gb->{SEQ}) * 47/48), $r, $bp_length );
	$svg->polygon(
		      points=>"$cx,$ry $rx,$ry2 $cx,$ry2",
		      stroke=>"#000000",
		      fill=>'#000000',
		      'stroke-width'=>30
		      );
	
	$r = 2950;
	( $rx, $ry2 ) = &get_pos( int(length($gb->{SEQ}) * 47/48), $r, $bp_length );
	$svg->polygon(
		      points=>"$cx,$ry $rx,$ry2 $cx,$ry2",
		      stroke=>"#000000",
		      fill=>'#000000',
		      'stroke-width'=>30
		      );
    }

    
    #$y->circle(cx=>$cx, cy=>$cy,r=>6500, id=>'main_c');

    my $scla = $svg->group(
        id => 'scale_a',
        style =>
          { 'stroke' => '#333333', 'fill' => 'none', 'stroke-width' => 1 }
    );
    my $sclb = $svg->group(
        id => 'scale_b',
        style =>
          { 'stroke' => '#ffffff', 'fill' => 'none', 'stroke-width' => 2 }
    );
    my $sclc = $svg->group(
        id => 'scale_c',
        style =>
          { 'stroke' => '#ffffff', 'fill' => 'none', 'stroke-width' => 2 }
    );

    my $cdsa = $svg->group(
        id => 'cds_a',
        style =>
          { 'stroke' => '#cc0066', 'fill' => 'none', 'stroke-width' => 300 }
    );
    my $cdsb = $svg->group(
        id => 'cds_b',
        style =>
          { 'stroke' => '#cc0033', 'fill' => 'none', 'stroke-width' => 300 }
    );
    my $cdsx = $svg->group(
        id => 'cds_x',
        style =>
          { 'stroke' => '#ccff33', 'fill' => 'none', 'stroke-width' => 300 }
    );
    my $cdsy = $svg->group(
        id => 'cds_y',
        style =>
          { 'stroke' => '#99cc00', 'fill' => 'none', 'stroke-width' => 300 }
    );
    my $cdst = $svg->group( id => 'cds_texts', style => { fill => '#ccccff' } );

    my $trnaa = $svg->group(
        id => 'trna_a',
        style =>
          { 'stroke' => '#3399ff', 'fill' => 'none', 'stroke-width' => 200 }
    );
    my $trnax = $svg->group(
        id => 'trna_x',
        style =>
          { 'stroke' => '#00ffcc', 'fill' => 'none', 'stroke-width' => 200 }
    );

    my $yaji = $svg->group(
        id => 'yajirushi',
        style =>
          { 'stroke' => '#ccffcc', 'fill' => 'none', 'stroke-width' => 5 }
    );

    my $rrnaa = $svg->group(
        id => 'rrna_a',
        style =>
          { 'stroke' => '#ff6699', 'fill' => 'none', 'stroke-width' => 200 }
    );
    my $rrnax = $svg->group(
        id => 'rrna_x',
        style =>
          { 'stroke' => '#ff9933', 'fill' => 'none', 'stroke-width' => 200 }
    );

    my $gcska = $svg->group(
        id => 'gccontent',
        style =>
          { 'stroke' => '#ff9933', 'fill' => 'none', 'stroke-width' => 3 }
    );
    my $gcskc = $svg->group(
        id => 'gccontent_b',
        style =>
          { 'stroke' => '#cc6611', 'fill' => 'none', 'stroke-width' => 3 }
    );

    my $gcskb = $svg->group(
        id => 'gcskew',
        style =>
          { 'stroke' => '#ccff66', 'fill' => 'none', 'stroke-width' => 3 }
    );
    my $gcskd = $svg->group(
        id => 'gcskew_b',
        style =>
          { 'stroke' => '#aadd44', 'fill' => 'none', 'stroke-width' => 3 }
    );

    my $oriter = $svg->group(
        id => 'ori_ter',
        style =>
          { 'stroke' => '#cccc33', 'fill' => 'none', 'stroke-width' => 20 }
    );


    my $title = $gb->{FEATURE0}->{organism};

    my $text1 = $svg->text(
        id            => 'title',
        'text-anchor' => 'middle',
        x             => $cx,
        y             => $cy,
        fill          => '#cccccc',
        'font-size'   => 200,
        'font-family' => 'Gill Sans'
    )->cdata($title);

    my ( $xe, $ye ) = &get_pos( 0, 3700, $bp_length );

    if(0){
    my ( $xe2, $ye2 ) = &get_pos( 0, 2850, $bp_length );
    $oriter->path( 'd' => "M$xe2,$ye2 L$xe,$ye" );

    ( $xe, $ye ) = &get_pos( 0, 2200, $bp_length );
    ( $xe2, $ye2 ) = &get_pos( 0, 1300, $bp_length );
    $oriter->path( 'd' => "M$xe2,$ye2 L$xe,$ye" );

    my ( $xe, $ye ) = &get_pos( length($gb->{SEQ}), 3700, $bp_length );
    my ( $xe2, $ye2 ) = &get_pos( length($gb->{SEQ}), 2850, $bp_length );
    $oriter->path( 'd' => "M$xe2,$ye2 L$xe,$ye" );

    ( $xe, $ye ) = &get_pos( length($gb->{SEQ}), 2200, $bp_length );
    ( $xe2, $ye2 ) = &get_pos( length($gb->{SEQ}), 1300, $bp_length );
    $oriter->path( 'd' => "M$xe2,$ye2 L$xe,$ye" );
}

    open( PWRITER, ">./data/" . $wfile . ".coord" );

    if($gb->{LOCUS}->{circular} == 1){

	( $xe, $ye ) = &get_pos( $ori, 4000, $bp_length );
	$oriter->path( 'd' => "M$cx,$cy L$xe,$ye" );
	
	if ( $xe < $cx ) {
	    $cdst->text(
			'text-anchor' => 'end',
			x             => $xe,
			y             => $ye,
			'font-size'   => 100
			)->cdata("ORIGIN");
	}
	else {
	    $cdst->text( x => $xe, y => $ye, 'font-size' => 100 )->cdata("ORIGIN");
	}
	
	($xe, $ye ) = &get_pos( $ter, 4000, $bp_length );
	$oriter->path( 'd' => "M$cx,$cy L$xe,$ye" );

	if ( $xe < $cx ) {
	    $cdst->text(
			'text-anchor' => 'end',
			x             => $xe,
			y             => $ye,
			'font-size'   => 100
			)->cdata("TERMINUS");
	} else {
	    $cdst->text( x => $xe, y => $ye, 'font-size' => 100 )->cdata("TERMINUS");
	}
	
    }

    my $num = $bp_length;
    while ( $num =~ s/(.*\d)(\d\d\d)/$1,$2/ ) { }

    my $text2 = $svg->text(
        id            => 'chromosome',
        'text-anchor' => 'middle',
        x             => $cx,
        y             => $cy + 200,
        fill          => '#cccccc',
        'font-size'   => 150,
        'font-family' => 'Gill Sans'
    )->cdata( sprintf("%s: %s", $gb->id(), $optionalText) ) if(length($optionalText));

    my $text2 = $svg->text(
        id            => 'bplength',
        'text-anchor' => 'middle',
        x             => $cx,
        y             => $cy + 400,
        fill          => '#cccccc',
        'font-size'   => 150,
        'font-family' => 'Gill Sans'
    )->cdata( $num . " bp" );

    my $i = 0;
    $sclb->circle( cx => $cx, cy => $cy, r => 3750 );
    $sclc->circle( cx => $cx, cy => $cy, r => 3680 );

    while ( $i < $bp_length ) {
        ( $xe, $ye ) = &get_pos( $i, 3700, $bp_length );
        $scla->path( 'd' => "M$cx,$cy L$xe,$ye" );
        $cdst->text( x => $xe, y => $ye )->cdata( $i . "bp" );
        ( $xe, $ye ) = &get_pos( $i, 2850, $bp_length );
        $cdst->text( x => $xe, y => $ye )->cdata( $i . "bp" );
        ( $xe, $ye ) = &get_pos( $i, 1800, $bp_length );
        $cdst->text( x => $xe, y => $ye )->cdata( $i . "bp" );

        if ( $i % 250000 == 0 ) {
            my ( $xs, $ys ) = &get_pos( $i, 3630, $bp_length );
            my ( $xe, $ye ) = &get_pos( $i, 3680, $bp_length );
            $sclc->path( 'd' => "M$xs,$ys L$xe,$ye" );

            $cdst->text(
                'text-anchor' => 'middle',
                x             => $xe,
                y             => $ye,
                'font-size'   => 80
            )->cdata( $i . "bp" );
        }
        $i = $i + 25000;
    }
    for ( $i = 0 ; $i < 100 ; $i++ ) {
        my ( $xs, $ys ) = &get_posp( $i, 3700, $bp_length );
        my ( $xe, $ye ) = &get_posp( $i, 3750, $bp_length );
        $sclc->path( 'd' => "M$xs,$ys L$xe,$ye" );

        if ( $i % 10 == 0 ) {
            if ( $xe < $cx ) {
                $cdst->text(
                    'text-anchor' => 'end',
                    x             => $xe,
                    y             => $ye,
                    'font-size'   => 80
                )->cdata( $i, "\%" );
            } else {
                $cdst->text( x => $xe, y => $ye, 'font-size' => 80 )->cdata( $i, "\%" );
            }
        }
    }

    for my $feature ( $gb->feature() ) {
	my($xs, $ys, $xe, $ye);
	my $strokeW = 300;
	my $r = 3400;

        if ( $gb->{$feature}->{type} eq "gene" and $gb->{$feature}->{direction} eq "direct" ) {
            $r = 3400;
            ( $xs, $ys ) = &get_pos( $gb->{$feature}->{start}, $r, $bp_length );
            ( $xe, $ye ) = &get_pos( $gb->{$feature}->{end}, $r, $bp_length );

            if ($gb->{LOCUS}->{circular} == 0){

		if($gb->{$feature}->{start}/(length($gb->{SEQ})) > 47/48){
		    my $startbp = $gb->{$feature}->{start} - int(length($gb->{SEQ}) * 47/48);
		    my $startpx = $startbp / int(length($gb->{SEQ}) * 1/48);
		    $strokeW = 300 - ($startpx * 300);
		    $r += int($strokeW/2) - 150;
		    ( $xs, $ys ) = &get_pos( $gb->{$feature}->{start}, $r, $bp_length );
		    ( $xe, $ye ) = &get_pos( $gb->{$feature}->{end},   $r, $bp_length );
		}
            }

        }elsif ( $gb->{$feature}->{type} eq "gene" and $gb->{$feature}->{direction} eq "complement" ){
            $r = 3100;
            ( $xs, $ys ) = &get_pos( $gb->{$feature}->{start}, $r, $bp_length );
            ( $xe, $ye ) = &get_pos( $gb->{$feature}->{end}, $r, $bp_length );

            if ($gb->{LOCUS}->{circular} == 0){

		if($gb->{$feature}->{start}/(length($gb->{SEQ})) > 47/48){
		    my $startbp = $gb->{$feature}->{start} - int(length($gb->{SEQ}) * 47/48);
		    my $startpx = $startbp / int(length($gb->{SEQ}) * 1/48);
		    $strokeW = 300 - ($startpx * 300);
		    $r -= int($strokeW/2) - 150;
		    ( $xs, $ys ) = &get_pos( $gb->{$feature}->{start}, $r, $bp_length );
		    ( $xe, $ye ) = &get_pos( $gb->{$feature}->{end}, $r, $bp_length );
		}
	    }
        }

	if($gb->{$feature}->{type} eq "gene"){
	    if ($gb->{LOCUS}->{circular} == 1){
		if(query_strand($gb, $feature) eq 'leading'){
		    $count_a++;
		    if($count_a % 2 == 1 ) {
			$cdsa->path(
				    id  => "a" . $count_a,
				    'd' => "M$xs $ys A$r $r,0,0 1,$xe $ye",
				    'stroke-width'=>$strokeW
				    );
		    }else{
			$cdsb->path(
				    id  => "a" . $count_a,
				    'd' => "M$xs $ys A$r $r,0,0 1,$xe $ye",
				    'stroke-width'=>$strokeW
				    );
		    }
		}else{
		    $count_b++;
		    if($count_a % 2 == 1 ) {
			$cdsx->path(
				    id  => "b" . $count_b,
				    'd' => "M$xs $ys A$r $r,0,0 1,$xe $ye",
				    'stroke-width' => $strokeW
				    );
		    } else {
			$cdsy->path(
				    id  => "b" . $count_b,
				    'd' => "M$xs $ys A$r $r,0,0 1,$xe $ye",
				    'stroke-width' => $strokeW
				    );
		    }
		}
	    }else{
		if($gb->{$feature}->{direction} eq 'direct'){
		    $count_a++;
		    if($count_a % 2 == 1 ) {
			$cdsa->path(
				    id  => "a" . $count_a,
				    'd' => "M$xs $ys A$r $r,0,0 1,$xe $ye",
				    'stroke-width'=>$strokeW
				    );
		    }else{
			$cdsb->path(
				    id  => "a" . $count_a,
				    'd' => "M$xs $ys A$r $r,0,0 1,$xe $ye",
				    'stroke-width'=>$strokeW
				    );
		    }
		}else{
		    $count_b++;
		    if($count_a % 2 == 1 ) {
			$cdsx->path(
				    id  => "b" . $count_b,
				    'd' => "M$xs $ys A$r $r,0,0 1,$xe $ye",
				    'stroke-width' => $strokeW
				    );
		    } else {
			$cdsy->path(
				    id  => "b" . $count_b,
				    'd' => "M$xs $ys A$r $r,0,0 1,$xe $ye",
				    'stroke-width' => $strokeW
				    );
		    }
		}
	    }
	    my ( $xm, $ym ) = &get_mapp( ( $xs + $xe ) / 2, ( $ys + $ye ) / 2, $bp_length );
	    my $feat = $gb->next_feature($feature);
	    printf PWRITER "%s\t%s\t%s\t%d\t%d\n", $feat, $gb->{$feat}->{type}, $gb->{$feat}->{gene}, $xm, $ym;
	}

        if ( $gb->{$feature}->{type} eq "tRNA" and $gb->{$feature}->{direction} eq "direct" ){
            my $r = 2600;
            my ( $xs, $ys ) = &get_pos( $gb->{$feature}->{start}, $r, $bp_length );
            my ( $xe, $ye ) = &get_pos( $gb->{$feature}->{end}, $r, $bp_length );
            $trnaa->path( 'd' => "M$xs $ys A$r $r,0,0 1,$xe $ye" );
            my ( $xxs, $yys, $xxe, $yye, $xl, $yl, $xr, $yr ) = &get_arrow( $gb->{$feature}->{start}, $r + 50, $bp_length );
            $yaji->path( 'd' => "M$xxs $yys L$xxe $yye" );
            $yaji->path( 'd' => "M$xxe $yye L$xl $yl" );
            $yaji->path( 'd' => "M$xxe $yye L$xr $yr" );
        }

        if ( $gb->{$feature}->{type} eq "tRNA" and $gb->{$feature}->{direction} eq "complement" ){
            my $r = 2600;
            my ( $xs, $ys ) = &get_pos( $gb->{$feature}->{start}, $r, $bp_length );
            my ( $xe, $ye ) = &get_pos( $gb->{$feature}->{end}, $r, $bp_length );
            $trnax->path( 'd' => "M$xs $ys A$r $r,0,0 1,$xe $ye" );
            my ( $xxs, $yys, $xxe, $yye, $xl, $yl, $xr, $yr ) = &get_arrowc( $gb->{$feature}->{start}, $r - 50, $bp_length );
            $yaji->path( 'd' => "M$xxs $yys L$xxe $yye" );
            $yaji->path( 'd' => "M$xxs $yys L$xl $yl" );
            $yaji->path( 'd' => "M$xxs $yys L$xr $yr" );
        }

        if ( $gb->{$feature}->{type} eq "rRNA" and $gb->{$feature}->{direction} eq "direct" ){
            my $r = 2400;
            my ( $xs, $ys ) = &get_pos( $gb->{$feature}->{start}, $r, $bp_length );
            my ( $xe, $ye ) = &get_pos( $gb->{$feature}->{end}, $r, $bp_length );
            $rrnaa->path( 'd' => "M$xs $ys A$r $r,0,0 1,$xe $ye" );
        }

        if ( $gb->{$feature}->{type} eq "rRNA" and $gb->{$feature}->{direction} eq "complement" ){
            my $r = 2400;
            my ( $xs, $ys ) = &get_pos( $gb->{$feature}->{start}, $r, $bp_length );
            my ( $xe, $ye ) = &get_pos( $gb->{$feature}->{end}, $r, $bp_length );
            $rrnax->path( 'd' => "M$xs $ys A$r $r,0,0 1,$xe $ye" );
        }
    }

    if($gb->{LOCUS}->{circular} == 0){
	my $ye1 = $cy - 3000;
	my $ye2 = $cy - 3500;
	$oriter->path( 'd' => "M$cx,$ye1 L$cx,$ye2", opacity=>0.5 );
	
	my $r = 3550;
	my ( $rx, $ry2 ) = &get_pos( int(length($gb->{SEQ}) * 47/48), $r, $bp_length );
	$oriter->path( 'd' => "M$cx,$ry L$rx,$ry2", opacity=>0.2 );
	
	my $r = 2950;
	my ( $rx, $ry2 ) = &get_pos( int(length($gb->{SEQ}) * 47/48), $r, $bp_length );
	$oriter->path( 'd' => "M$cx,$ry L$rx,$ry2", opacity=>0.2 );
    }


    my $bline  = 2000;
    my $blineg = 1500;
    my ( $px,  $py )  = &get_pos( 0, $bline,  $bp_length );
    my ( $pxg, $pyg ) = &get_pos( 0, $blineg, $bp_length );
    my $pgcc  = 0.5;
    my $pgcs  = 0.0;

    my $range = 1000;
    my $wsize = 2000;           #window_bp_size
    my $step  = 1000;

    $tag = $svg->circle(
        cx     => $cx,
        cy     => $cy,
        r      => $bline,
        stroke => '#33ff99',
        fill   => 'none'
    );

    $tag = $svg->circle(
        cx     => $cx,
        cy     => $cy,
        r      => $blineg,
        stroke => '#66ffff',
        fill   => 'none'
    );

    for ( my $i = 0 ; $i + $wsize <= $bp_length ; $i += $step ) {
        my $temp = substr( $gb->{SEQ}, $i, $wsize );
        my $c_g = $temp =~ tr/g/g/;
        my $c_c = $temp =~ tr/c/c/;
        my $c_a = $temp =~ tr/a/a/;
        my $c_t = $temp =~ tr/t/t/;

	my $opacity = 1;
	if($i > (length($gb->{SEQ}) * 95/96)){# || $i < length($gb->{SEQ}) * 1/96){
	    $opacity = 0.1 if($gb->{LOCUS}->{circular} == 0);
	}

        my $gc_content = 0;
        $gc_content = ( $c_g + $c_c ) / ( $c_g + $c_c + $c_a + $c_t )
          if ( $c_g + $c_c + $c_a + $c_t );

        my ( $x, $y ) = &get_pos( $i, ( $bline + ( $gc_content - 0.5 ) * $range * 2 ), $bp_length );

	if($pgcc >= 0.5 && $gc_content >= 0.5){
	    $gcska->path( 'd' => "M$px,$py L$x,$y", opacity=>$opacity );
	}elsif($pgcc <= 0.5 && $gc_content <= 0.5){
	    $gcskc->path( 'd' => "M$px,$py L$x,$y", opacity=>$opacity );
	}else{
	    my ( $xn, $yn ) = &get_pos( $i, $bline, $bp_length );
	    if($gc_content > 0.5){
		$gcska->path( 'd' => "M$xn,$yn L$x,$y", opacity=>$opacity );
		$gcskc->path( 'd' => "M$px,$py L$xn,$yn", opacity=>$opacity );
	    }else{
		$gcska->path( 'd' => "M$px,$py L$xn,$yn", opacity=>$opacity );
		$gcskc->path( 'd' => "M$xn,$yn L$x,$y", opacity=>$opacity );
	    }
	}

        my $gc_skew = 0;
        $gc_skew = ( $c_c - $c_g ) / ( $c_g + $c_c ) if ( $c_g + $c_c );

        my ( $xg, $yg ) = &get_pos( $i, ( $blineg + ($gc_skew) * $range * 2 ), $bp_length );

	if($pgcs >= 0 && $gc_skew >= 0){
	    $gcskb->path( 'd' => "M$pxg,$pyg L$xg,$yg", opacity=>$opacity );
	}elsif($pgcs <= 0 && $gc_skew <= 0){
	    $gcskd->path( 'd' => "M$pxg,$pyg L$xg,$yg", opacity=>$opacity );
	}else{
	    my ( $xn, $yn ) = &get_pos( $i, $blineg, $bp_length );
	    if($gc_skew > 0){
		$gcskb->path( 'd' => "M$xn,$yn L$xg,$yg",   opacity=>$opacity );
		$gcskd->path( 'd' => "M$pxg,$pyg L$xn,$yn", opacity=>$opacity );
	    }else{
		$gcskb->path( 'd' => "M$pxg,$pyg L$xn,$yn", opacity=>$opacity );
		$gcskd->path( 'd' => "M$xn,$yn L$xg,$yg",   opacity=>$opacity );
	    }
	}

#        $gcskb->path( 'd' => "M$pxg,$pyg L$xg,$yg", opacity=>$opacity );


        $px = $x;
        $py = $y;

        $pxg = $xg;
        $pyg = $yg;

	$pgcc = $gc_content;
	$pgcs = $gc_skew;
    }

    $cdst->text(
        'text-anchor' => 'middle',
        x             => $cx,
        y             => 1750,
        'font-size'   => 120
    )->cdata("GC-contents");

    $cdst->text(
        'text-anchor' => 'middle',
        x             => $cx,
        y             => 3000,
        'font-size'   => 120
    )->cdata("GC-skew");

    close PWRITER;
    open( WRITER, ">./graph/" . $filename );
    print WRITER $svg->xmlify;
    close WRITER;

    if($gmap){
	generateGMap("graph/$filename");
    }
}


sub dec2hex {
    my $num = shift;
    return $num if ($num < 10);

    $num = 'a' if ($num == 10);
    $num = 'b' if ($num == 11);
    $num = 'c' if ($num == 12);
    $num = 'd' if ($num == 13);
    $num = 'e' if ($num == 14);
    $num = 'f' if ($num == 15);

    return $num;
}
    
sub get_pos {
    my ( $n, $r, $bp_length ) = @_;
    my $x = $cx + ( sin( 2 * $pi * $n / $bp_length ) * $r );
    my $y = $cy - ( cos( 2 * $pi * $n / $bp_length ) * $r );

    return ( $x, $y );
}

sub get_posp {
    my ( $n, $r, $bp_length ) = @_;
    my $x = $cx + ( sin( 2 * $pi * $n / 100 ) * $r );
    my $y = $cy - ( cos( 2 * $pi * $n / 100 ) * $r );

    return ( $x, $y );
}

sub get_mapp {
    my ( $n, $r, $bp_length ) = @_;
    my $x = $n * 8192 / 8000;
    my $y = $r * 8192 / 8000;

    return ( $x, $y );
}

sub get_arrow {
    my ( $n, $r, $bp_length ) = @_;

    my $rr = 40;                                                #length
    my $rl = 20;                                                #hand_length
    my $x  = $cx + ( sin( 2 * $pi * $n / $bp_length ) * $r );
    my $y  = $cy - ( cos( 2 * $pi * $n / $bp_length ) * $r );
    my $th  = 2 * $pi * ( 1 + $n ) / $bp_length;
    my $xxs = $x - cos($th) * $rr;
    my $yys = $y - sin($th) * $rr;
    my $xxe = $x + cos($th) * $rr;
    my $yye = $y + sin($th) * $rr;

    $th -= $pi * 3 / 4;
    my $xl = $xxe + cos($th) * $rl;
    my $yl = $yye + sin($th) * $rl;
    $th -= $pi * 2 / 4;
    my $xr = $xxe + cos($th) * $rl;
    my $yr = $yye + sin($th) * $rl;

    return ( $xxs, $yys, $xxe, $yye, $xl, $yl, $xr, $yr );
}

sub get_arrowc {
    my ( $n, $r, $bp_length ) = @_;

    my $rr = 40;                                                #length
    my $rl = 20;                                                #hand_length
    my $x  = $cx + ( sin( 2 * $pi * $n / $bp_length ) * $r );
    my $y  = $cy - ( cos( 2 * $pi * $n / $bp_length ) * $r );
    my $th  = 2 * $pi * ( 1 + $n ) / $bp_length;
    my $xxs = $x - cos($th) * $rr;
    my $yys = $y - sin($th) * $rr;
    my $xxe = $x + cos($th) * $rr;
    my $yye = $y + sin($th) * $rr;
    
    $th -= $pi * 1 / 4;
    my $xl = $xxs + cos($th) * $rl;
    my $yl = $yys + sin($th) * $rl;
    $th += $pi * 2 / 4;
    my $xr = $xxs + cos($th) * $rl;
    my $yr = $yys + sin($th) * $rl;

    return ( $xxs, $yys, $xxe, $yye, $xl, $yl, $xr, $yr );
}








=head2 genome_map

  Name: genome_map   -   draws the map of the genome

  Description:
    This method creates a map of the genome, showing the local 
    nucleotide contents and positions of genes.
    A is shown in red, T is shown in green, 
    G is shown in yellow, and C is shown in blue. 

    This method generates the entire genome map using 
    multiple 800x600 pixel image in PNG (via GD library).

  Usage:
    NULL = genome_map($gb);

 Options:
    -name      print gene name (default: 1)
    -window    number of nucleotides represented by one pixel (default: 50)
    -amp       height of the nucleotide content graph (default: 1.5)
    -output    "g" for graph, "show" for display (default: "show") 

  Author: 
    Kazuharu Arakawa (gaou@sfc.keio.ac.jp)

  History:
    20010906-01 initial posting

=cut

sub genome_map {
    &opt_default(output=>"show", name=>1, window=>50, amp=>1.5);
    my @args = opt_get(@_);
    my $gb = opt_as_gb(shift @args);
    my $acnum = $gb->{LOCUS}->{id};
    my $output = opt_val("output");
    my $name = opt_val("name");
    my $filename;
    my $topmargin = 30;
    my $sidemargin = 80;
    my $hblock = 100;
    my $vblock = 10;
    my $page = 1;
    my $start;
    my $width = 800;
    my $height = 600;
    my $i = 0;
    my $cds = 1;
    my $window=opt_val("window");
    my $amp = opt_val("amp");

    mkdir ("graph", 0777);
    for ($start = 1; $start <= length($gb->{SEQ}); $start += $window * 700 * 10){ 
	my $end = $start + 10 * $window * 700 - 1;
	
	# GD constant
	my $im = new GD::Image($width, $height);
	my $white = $im->colorAllocate(255,255,255);
	my $black = $im->colorAllocate(0,0,0);
	my $gray = $im->colorAllocate(180,180,180);  
	my $red = $im->colorAllocate(255,0,0);       #A
	my $yellow = $im->colorAllocate(255,255,0);  #G
	my $green = $im->colorAllocate(0,150,0);     #T
	my $blue = $im->colorAllocate(0,0,255);      #C
	my $aqua = $im->colorAllocate(120, 160, 255);

	my $gred = $im->colorAllocate(255,150,150);       #A for graph
	my $gyellow = $im->colorAllocate(255,255,50);     #G for graph
	my $ggreen = $im->colorAllocate(150,150,150);     #T for graph
	my $gblue = $im->colorAllocate(150,150,255);      #C for graph


	# Draw Base Graph
	for ($i = $sidemargin; $i <= $sidemargin + $hblock * 7; $i += $hblock){
	    $im->line($i, $topmargin, $i, $topmargin + 5 * 11 * $vblock, 
		      $gray);
	}
	for ($i = $topmargin; $i <= $topmargin + $vblock * 5 * 11; $i += 
	     $vblock){
	    $im->line($sidemargin, $i, $sidemargin + 7 * $hblock, $i, $gray);
	}
	for ($i = $topmargin + $vblock * 5; 
	     $i < $topmargin + $vblock * 5 * 11; 
	     $i += $vblock * 5){
	    $im->line($sidemargin, $i - 1, $sidemargin + 7 * $hblock, $i - 1, 
		      $black);
	    $im->line($sidemargin, $i + 1, $sidemargin + 7 * $hblock, $i + 1, 
		      $black);
	}
	$im->string(gdSmallFont, $width - 110, 5, 
		    "G-language Project", $black);
	$im->string(gdSmallFont, $width - 110 - 50, 5, "A", $red);
	$im->string(gdSmallFont, $width - 110 - 40, 5, "T", $green);
	$im->string(gdSmallFont, $width - 110 - 30, 5, "G", $yellow);
	$im->string(gdSmallFont, $width - 110 - 20, 5, "C", $blue);

	my $j = 0;
	for ($i = $topmargin + $vblock * 5; 
	     $i <= $topmargin + $vblock * 5 * 10; 
	     $i += $vblock * 5){
	    my $num = $start + $j * $window * 700;
	    $im->string(gdTinyFont, 10, $i, "$num", $black);
	    $j ++;
	}
	$im->string(gdSmallFont, 5,  5, "$acnum : from $start to $end", 
		    $black);

	my ($pa, $pt, $pg, $pc, $num, $color);
	my $locus = 0;
	for ($i = $start - 1; $i <= $start + 700 * 10 * $window - 1; $i += $window){
	    last if ($i + $window >= length($gb->{SEQ}));
	    my $seq = $gb->getseq($i, $i + $window - 1);
	    my $a = $seq =~ tr/a/a/;
	    my $t = $seq =~ tr/t/t/;
	    my $g = $seq =~ tr/g/g/;
	    my $c = $seq =~ tr/c/c/;

	    # Draw DNA
	    if ($a >= $t && $a >= $g && $a >= $c){
		my $num = int($a / $window * 100);
		$color = $red;
	    }elsif ($t >= $a && $t >= $g && $t >= $c){
		my $num = int($t / $window * 100);
		$color = $green;
	    }elsif ($g >= $a && $g >= $t && $g >= $c){
		my $num = int($g / $window * 100);
		$color = $yellow;
	    }elsif ($c >= $a && $c >= $t && $c >= $g){
		my $num = int($c / $window * 100);
		$color = $blue;
	    }
	    $im->setPixel($sidemargin + 1 + $locus % (700), 
			  $topmargin + (int($locus / 700) + 1) * $vblock * 5,
			  $color);

	    my $dist = 7;
	    if ($locus % $dist == $dist - 1){
		# Draw A content graph
		$num = int($a / int ($window * $amp) * 100) + 5;
		$im->line($sidemargin - $dist + $locus % (700), $pa,
			  $sidemargin + $locus % (700),
			  $topmargin - $num
			  + (int($locus / 700) + 1) * $vblock * 5,
			  $gred);
		$pa = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;

		# Draw T content graph
		$num = int($t / int ($window * $amp) * 100) + 5;
		$im->line($sidemargin - $dist + $locus % (700), $pt,
			  $sidemargin + $locus % (700),
			  $topmargin - $num
			  + (int($locus / 700) + 1) * $vblock * 5,
			  $ggreen);
		$pt = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;

		# Draw G content graph
		$num = int($g / int ($window * $amp) * 100) + 5;
		$im->line($sidemargin - $dist + $locus % (700), $pg,
			  $sidemargin + $locus % (700),
			  $topmargin - $num
			  + (int($locus / 700) + 1) * $vblock * 5,
			  $gyellow);
		$pg = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;

		# Draw C content graph
		$num = int($c / int ($window * $amp) * 100) + 5;
		$im->line($sidemargin - $dist + $locus % (700), $pc,
			  $sidemargin + $locus % (700),
			  $topmargin - $num
			  + (int($locus / 700) + 1) * $vblock * 5,
			  $gblue);
		$pc = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;
	    }elsif($locus % 700 == 0){
		$num = int($a / int ($window * $amp) * 100) + 5;
		$pa = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;
		$num = int($t / int ($window * $amp) * 100) + 5;
		$pt = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;
		$num = int($g / int ($window * $amp) * 100) + 5;
		$pg = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;
		$num = int($c / int ($window * $amp) * 100) + 5;
		$pc = $topmargin - $num +(int($locus / 700) + 1)
		    * $vblock * 5;
	    }
	    $locus ++;
	}

	# Draw Genes
	my $flag = 0;
	my $before = -5000;
	my $before2 = -10000;
	while (%{$gb->{"CDS$cds"}}){
	    my $cdsstart = $gb->{"CDS$cds"}->{start};
	    my $cdsend = $gb->{"CDS$cds"}->{end};
	    my $cdsdir = $gb->{"CDS$cds"}->{direction};
	    my $cdsdiff = $cdsstart - $before;
	    my $cdsdiff2 = $cdsstart - $before2;
	    if ($flag == 0){
		if (int($cdsdiff / $window) < 20){
		    $flag = 1;
		}
	    }elsif ($flag == 1){
		if (int($cdsdiff / $window) < 20){
		    if (int($cdsdiff2 / $window) < 20){
			$flag = 2;
		    }else{
			$flag = 0;
		    }
		}else{
		    $flag = 0;
		}
	    }elsif ($flag == 2){
		if (int($cdsdiff2 / $window) < 20){
		    $flag = 1;
		}else{
		    $flag = 0;
		}
	    }
		
	    if ($cdsstart < $start && $cdsend >$start){
		my $dif1 = -3; 
		my $dif2 = -2 - 3;
		if ($cdsdir eq 'complement'){
		    $dif1 *= -1;
		    $dif2 *= -1;
		}
		my $k;
		for ($k = 1; $k <= $cdsend - $start; $k ++){
		    my $l = int ($k / $window); 
		    $im->line($sidemargin + 1 + $l % 700,
			      $topmargin + $dif1 + (int($l/700)+1)*$vblock * 5,
			      $sidemargin + 1 + $l % 700,
			      $topmargin + $dif2 + (int($l/700)+1)*$vblock * 5,
			      $aqua);
		}
		$cds ++;
	    }else{
		last if ($cdsstart >= $end);
		my $feat = $gb->{"CDS$cds"}->{feature};
		my $genename = $gb->{"FEATURE$feat"}->{gene};
		
		my $dif1 = -3; 
		my $dif2 = -2 - 3;
		if ($cdsdir eq 'complement'){
		    $dif1 *= -1;
		    $dif2 *= -1;
		}

		my $k;
		for ($k = $cdsstart-$start; $k <= $cdsend - $start; $k += $window){
		    last if ($k + $start > $end);
		    my $l = int ($k / $window);
		    $im->line($sidemargin + 1 + $l % 700,
			      $topmargin + $dif1 + (int($l/700)+1)*$vblock * 5,
			      $sidemargin + 1 + $l % 700,
			      $topmargin + $dif2 + (int($l/700)+1)*$vblock * 5,
			      $aqua);
		}
		last if ($k + $start > $end);
		
		$k = int(($cdsstart - $start)/$window);
		
		$dif1 = -2; 
		$dif2 = -2 - 4;
		my $dif3 = -2 - 4 - 9 + (-9 * $flag);
		if ($cdsdir eq 'complement'){
		    $dif1 *= -1;
		    $dif2 *= -1;
		    $dif3 *= -1;
		    $dif3 -= 7;
		}
		$im->line($sidemargin + 1 + $k % 700,
			  $topmargin + $dif1 + (int($k/700)+1)*$vblock * 5,
			  $sidemargin + 1 + $k % 700,
			  $topmargin + $dif2 + (int($k/700)+1)*$vblock * 5,
			  $black);

		$im->string(gdTinyFont, $sidemargin + 1 + $k %700,
			    $topmargin + $dif3 + (int($k/700)+1)*$vblock * 5,
			    "$genename", $black) if ($name);

		if ($cdsend > $end){
		    last;
		}else{
		    $before = $cdsstart;
		    $before2 = $before;
		    $cds ++;
		}
	    }
	}

	open(OUT, '>graph/' . $acnum . '-'. $page . '.png');
	binmode OUT;
	print OUT $im->png;
	close(OUT);
        msg_gimv("graph/$acnum" . '-' . $page . '.png') if ($output eq 'show');
	$page ++;
    }
}

=head2 genome_map2

  Name: genome_map2   -   draws the map of the genome (version 2)

  Description:
    This method creates a map of the genome, showing the local 
    nucleotide contents and positions of genes using SVG.
    A is shown in red, T is shown in green, 
    G is shown in yellow, and C is shown in blue. 

    Map of the specified position (with -start and -end options)
    is generated in SVG format.

  Usage:
    NULL = genome_map2($gb);

 Options:
    -window    length of one pixel (default: 50)
    -type      type of features to print (default: CDS)
    -output    "g" for graph, "show" for display (default: "show") 
    -start     start position (default: 1)
    -end       end position (default: 100000)
    -filename  output file name (default: genome_map.svg)
    -ptt       ptt filename
    
  Author: 
    Kazuharu Arakawa (gaou@sfc.keio.ac.jp)

  History:
    20020906-01 initial posting

=cut


sub genome_map2 {
    require SVG;

    &opt_default(output=>"show", window=>50, type=>"CDS", 
		 start=>1, end=>100000, filename=>"genome_map.svg", ptt=>'', cgi=>0);
    my @args = opt_get(@_);
    my $gb = shift @args;
    my $acnum = $gb->{LOCUS}->{id};
    my $output = opt_val("output");
    my $filename = opt_val("filename");
    my $topmargin = 60;
    my $sidemargin = 80;
    my $hblock = 100;
    my $vblock = 10;
    my $page = 1;
    my $i = 0;
    my $cds = 1;
    my $window=opt_val("window");
    my $start = opt_val("start");
    my $end = opt_val("end");
    my $ptt = opt_val("ptt");
    my $cgi = opt_val("cgi");

    my @type = split(/ /, opt_val("type"));
    my $j2 = 0;
    my $width = (int(($end - $start) / $hblock / $window) + 1) * $hblock  + $sidemargin * 2;
    my $height = $hblock/2 + $topmargin * 2 + scalar(@type) * $hblock - $vblock;

    eval{
	$ptt = set_gpac($gb, -ptt=>$ptt);
	 };

    unless ($cgi){
	sdb_save(\$ptt, "ptt");
	sdb_save($gb, "gb");
    }

    my $svg = SVG->new(width=>$width, height=>$height);
    
    # Draw Base Graph
    for ($i = $sidemargin; $i <= $width - $sidemargin; $i += $hblock){
	$svg->line(id=>"bvline$i", x1=>$i, 
		   y1=>$topmargin, x2=>$i, y2=>($hblock * (scalar(@type) + 1) + $vblock),
		   style=>{
		       stroke=>"darkgray", 'stroke-width'=>1,
		       'stroke-opacity'=>0.2
		       });
    }
    for ($i = $sidemargin; $i < $width - $sidemargin; $i += $hblock){
	$svg->text(
		   id=>"num$i", 
		   x=>$i, y=>($topmargin - 5), fill=>"black", 
		   'font-size'=>8,
		   )->cdata(((($i - $sidemargin) + 1) * $window - 49 + $start - 1) . " bp");
    }
    for ($i = $topmargin; $i <= $topmargin + $vblock * 5 ; $i += $vblock){
	$svg->line(id=>"bhline$i", x1=>$sidemargin, 
		   y1=>$i, x2=>($width - $sidemargin), y2=>$i,
		   style=>{
		       stroke=>"darkgray", 'stroke-width'=>1,
		       'stroke-opacity'=>0.2
		       });
    }
    for ($i = $topmargin + $vblock * 5; 
	 $i <= $topmargin + $hblock * (scalar(@type) + 1); $i += $hblock){
	$svg->line(id=>"bhlline$i", x1=>$sidemargin, 
		   y1=>$i, x2=>($width - $sidemargin), y2=>$i,
		   style=>{
		       stroke=>"darkgray", 'stroke-width'=>1,
		       'stroke-opacity'=>0.2
		       });
    }
    for ($i = $topmargin + $vblock * 5 + $hblock / 2; 
	 $i <= $topmargin + $hblock * scalar(@type); $i += $hblock){
	$svg->line(id=>"dnau$i", x1=>$sidemargin, 
		   y1=>($i - 1), x2=>($width - $sidemargin), y2=>($i - 1),
		   style=>{
		       stroke=>"black", 'stroke-width'=>1,
		       'stroke-opacity'=>1
		       });
	$svg->line(id=>"dnad$i", x1=>$sidemargin, 
		   y1=>($i + 1), x2=>($width - $sidemargin), y2=>($i + 1),
		   style=>{
		       stroke=>"black", 'stroke-width'=>1,
		       'stroke-opacity'=>1
		       });
    }

    $svg->text(
	       id=>"A", 
	       x=>20, y=>($topmargin + $vblock * 3), fill=>"red", 
	       'font-size'=>10,
	       )->cdata("A");
    $svg->text(
	       id=>"T", 
	       x=>30, y=>($topmargin + $vblock * 3), fill=>"green", 
	       'font-size'=>10,
	       )->cdata("T");
    $svg->text(
	       id=>"G", 
	       x=>40, y=>($topmargin + $vblock * 3), fill=>"gold", 
	       'font-size'=>10,
	       )->cdata("G");
    $svg->text(
	       id=>"C", 
	       x=>50, y=>($topmargin + $vblock * 3), fill=>"blue", 
	       'font-size'=>10,
	       )->cdata("C");
    
    my ($pa, $pt, $pg, $pc, $num, $color);
    my (@gra, @grt, @grc, @grg);
    my $locus = 0;
    for ($i = $start - 1; $i <= $end; $i += $window){
	my $seq = $gb->getseq($i, $i + $window * 2 - 1);
	my $a = $seq =~ tr/a/a/;
	my $t = $seq =~ tr/t/t/;
	my $g = $seq =~ tr/g/g/;
	my $c = $seq =~ tr/c/c/;
	my $x = $seq =~ tr/x/x/;
	my $n = $seq =~ tr/n/n/;
	
	if ($locus % 2 != 1){
    	    # Draw DNA
	    if ($a >= $t && $a >= $g && $a >= $c){
		my $num = int($a / $window * 100);
		$color = 'red';
		$color = 'white' if ($a == 0);
	    }elsif ($t >= $a && $t >= $g && $t >= $c){
		my $num = int($t / $window * 100);
		$color = 'green';
	    }elsif ($g >= $a && $g >= $t && $g >= $c){
		my $num = int($g / $window * 100);
		$color = 'yellow';
	    }elsif ($c >= $a && $c >= $t && $c >= $g){
		my $num = int($c / $window * 100);
		$color = 'blue';
	    }

	    my $j;
	    for ($j = $topmargin + $vblock * 5 + $hblock / 2; 
		 $j <= $topmargin + $hblock * scalar(@type); 
		 $j += $hblock
		 ){

		last if ($locus >= int($width - $sidemargin * 2) - 3);
		$j2 ++;

		$svg->line(id=>"dnac$j-$j2", x1=>($sidemargin + 1 + $locus), 
			   y1=>$j, x2=>($sidemargin + 3 + $locus), y2=>$j,
			   style=>{
			       stroke=>$color, 'stroke-width'=>1,
			       'stroke-opacity'=>1
			       });
	    }
	}
    
	my $dist = 5;
	if ($locus % $dist == $dist -1 || $locus == 0){
	    last if ($a == 0 && $t == 0 && $c == 0 && $g == 0 && $x == 0 && $n == 0);
	    push (@gra, sprintf("%d,%d", $sidemargin + $locus,
				$topmargin + $vblock * 5 - int($a / 2 / $window * 100 ) + 5));
	    push (@grt, sprintf("%d,%d", $sidemargin + $locus,
				$topmargin + $vblock * 5 - int($t / 2 / $window * 100 ) + 5));
	    push (@grg, sprintf("%d,%d", $sidemargin + $locus,
				$topmargin + $vblock * 5 - int($g / 2 / $window * 100 ) + 5));
	    push (@grc, sprintf("%d,%d", $sidemargin + $locus,
				$topmargin + $vblock * 5 - int($c / 2 / $window * 100 ) + 5));
	}

	$locus ++;
    }

    $svg->polyline(id=>"gra", points=>join(" ", @gra), 
	       style=>{
		   stroke=>'red', 'stroke-width'=>1,
		   'stroke-opacity'=>0.2, fill=>'none'
		   });
    $svg->polyline(id=>"grt", points=>join(" ", @grt), 
	       style=>{
		   stroke=>'green', 'stroke-width'=>1,
		   'stroke-opacity'=>0.2, fill=>'none'
		   });
    $svg->polyline(id=>"grg", points=>join(" ", @grg), 
	       style=>{
		   stroke=>'yellow', 'stroke-width'=>1,
		   'stroke-opacity'=>0.2, fill=>'none'
		   });
    $svg->polyline(id=>"grc", points=>join(" ", @grc), 
	       style=>{
		   stroke=>'blue', 'stroke-width'=>1,
		   'stroke-opacity'=>0.2, fill=>'none'
		   });


    # Draw Genes
    my $flag = 0;
    my $before = -5000;
    my $before2 = -10000;
    my @colors = qw(J K L D O M N P T C G E F H I Q R S -);
    
    my $col = 0;
    foreach my $pat (@type){
	$svg->text(
		   id=>$pat, 
		   x=>20, y=>($topmargin + $vblock * 5 + $hblock * $col + $hblock/2 + 3),
		   'font-size'=>10,
		   )->cdata($pat);
	
	foreach my $cds ($gb->feature()){
	    next unless($gb->{$cds}->{end} >= $start && $gb->{$cds}->{start} <= $end);
	    next unless($gb->{$cds}->{type} eq $pat);
	    my $cdsstart = $gb->{$cds}->{start};
	    my $cdsend = $gb->{$cds}->{end};
	    my $cdsdir = $gb->{$cds}->{direction};
	    my $cdsdiff = $cdsstart - $before;
	    my $genename = $gb->{$cds}->{gene};
	    my $code = $gb->{$cds}->{code} || $colors[$col];

	    my $dif1 = -10;
	    my $sign = -1;
	    $dif1 = -25 if ($cdsdiff / $window < 15);
	    
	    if ($cdsdir eq 'complement'){
		$dif1 *= -1;
		$sign *= -1;
	    }
	    
	    $cdsstart = $start if ($cdsstart < $start);
	    $cdsend = $end if ($cdsend > $end);
	    
	    if ($cdsdir eq 'direct'){
		$svg->rect(
			   id=>"liner$cds", x=>($sidemargin + ($cdsstart - $start)/$window), 
			   y=>($topmargin + $dif1 + $vblock * 5 + $hblock / 2 - 3 + $hblock * $col),
			   height=>(abs($dif1) + 3),
			   width=>(($cdsend - $cdsstart)/$window),
			   style=>{
			       stroke=>"gray", 'stroke-width'=>1,
			       'stroke-opacity'=>0.2, 'fill'=>'none'
			       });
	    }else{
		$svg->rect(
			   id=>"liner2$cds", x=>($sidemargin + ($cdsstart - $start)/$window), 
			   y=>($topmargin + $vblock * 5 + $hblock / 2 - 1 + $hblock * $col),
			   height=>(abs($dif1) + 3),
			   width=>(($cdsend - $cdsstart)/$window),
			   style=>{
			       stroke=>"gray", 'stroke-width'=>1,
			       'stroke-opacity'=>0.2, 'fill'=>'none'
			       });
	    }
	    
	    $svg->anchor(
			 -href=>'http://localhost/g-language/genome_view.cgi?pos=' . $cds,
			 -target=>'_blank'
			 )->rect(
				 id=>"line$cds", x=>($sidemargin + ($cdsstart - $start)/$window), 
				 y=>($topmargin + $dif1 + $vblock * 5 + $hblock / 2 - 3 + $hblock * $col),
				 height=>5,
				 width=>(($cdsend - $cdsstart)/$window),
				 style=>{
				     stroke=>"green", 'stroke-width'=>1,
				     'stroke-opacity'=>0.2, 'fill'=>$COG_fcolor{$code},
				     'fill-opacity'=>0.2
				     });
	    
	    $svg->text(
		       id=>"cds$cds", 
		       x=>($sidemargin + ($cdsstart - $start)/$window), 
		       y=>($topmargin + (abs($dif1) + 6) * $sign + $vblock * 5 + $hblock / 2 + 1  + $hblock * $col), 
		       fill=>$COG_fcolor{$code}, 
		       'font-size'=>5,
		       )->cdata($genename);
	    
	    $before = $cdsstart;
	}
	$col ++;
    }

    my $topline = $svg->line(id=>"top", x1=>0, y1=>10, x2=>$width, y2=>10,
			     style=>{
				 stroke=>"plum", 'stroke-width'=>3,
				 'stroke-opacity'=>0.3
				 });
    
    $topline->animate(
		      attributeName=>"stroke", 
		      values=>join(';', values(%COG_fcolor)), dur=>"90s",
		      repeatDur=>'indefinite'
		      );
    
    my $bottomline = $svg->line(id=>"bottom", x1=>0, y1=>($height-10), x2=>$width, y2=>($height-10),
				style=>{
				    stroke=>"plum", 'stroke-width'=>3,
				    'stroke-opacity'=>0.3
				    });
    
    $bottomline->animate(
			 attributeName=>"stroke", 
			 values=>join(';', values(%COG_fcolor)), dur=>"90s",
			 repeatDur=>'indefinite'
			 );
    
    $svg->anchor(
		 -href=>"http://www.g-language.org/",
		 -target=>"_blank"
		 )->text(
			 id=>"credits", 
			 x=>($width - 300), y=>($height - 15), fill=>"darkgray", 
			 'font-size'=>8, 'font-style'=>'italic',
			 )->cdata(
				  "generated by genome_map2, G-language Genome Analysis Environment"
				  );
    
    $svg->text(
	       id=>"locus1", 
	       x=>30, y=>26, fill=>"navy", 
	       'font-size'=>10,
	       )->cdata("Accession Number: ", $gb->{LOCUS}->{id});
    
    $svg->text(
	       id=>"locus2", 
	       x=>250, y=>26, fill=>"navy", 
	       'font-size'=>10,
	       )->cdata("Organism: ", $gb->{FEATURE0}->{organism});
    
    $svg->text(
	       id=>"locus3", 
	       x=>500, y=>26, fill=>"navy", 
	       'font-size'=>10,
	       )->cdata(length($gb->{SEQ}) . " bp");
    

    my $leftarrow = $svg->anchor(
				 -href=>'http://localhost/g-language/genome_view.cgi?pos=' . ($start - 50000)
				 )->polygon(
				   id=>"larrow", 
				   points=>sprintf("%d,%d %d,%d %d,%d", 50,
						   $topmargin + $vblock * 5 + $hblock * scalar(@type) - 10,
						   50,
						   $topmargin + $vblock * 5 + $hblock * scalar(@type) + 20,
						   15, 
						   $topmargin + $vblock * 5 + $hblock * scalar(@type) + 5
						   ),
				   style=>{
				       fill=>"plum", stroke=>"plum", 'stroke-width'=>5,
				       'fill-opacity'=>0.1, 'stroke-opacity'=>0.3
				       }
				   );
    
    $leftarrow->animate(
		      attributeName=>"stroke", 
		      values=>join(';', reverse(values(%COG_fcolor))), dur=>"90s",
		      repeatDur=>'indefinite'
		      );

    $leftarrow->animate(
		      attributeName=>"fill", 
		      values=>join(';', reverse(values(%COG_fcolor))), dur=>"90s",
		      repeatDur=>'indefinite'
		      );

    my $rightarrow = $svg->anchor(
				 -href=>'http://localhost/g-language/genome_view.cgi?pos=' . ($end + 50001)
				 )->polygon(
				   id=>"rarrow", 
				   points=>sprintf("%d,%d %d,%d %d,%d", $width - 50,
						   $topmargin + $vblock * 5 + $hblock * scalar(@type) - 10,
						   $width - 50,
						   $topmargin + $vblock * 5 + $hblock * scalar(@type) + 20,
						   $width - 15, 
						   $topmargin + $vblock * 5 + $hblock * scalar(@type) + 5
						   ),
				   style=>{
				       fill=>"plum", stroke=>"plum", 'stroke-width'=>5,
				       'fill-opacity'=>0.1, 'stroke-opacity'=>0.3
				       }
				   );
    
    $rightarrow->animate(
		      attributeName=>"stroke", 
		      values=>join(';', reverse(values(%COG_fcolor))), dur=>"90s",
		      repeatDur=>'indefinite'
		      );

    $rightarrow->animate(
		      attributeName=>"fill", 
		      values=>join(';', reverse(values(%COG_fcolor))), dur=>"90s",
		      repeatDur=>'indefinite'
		      );

    mkdir ("graph", 0777);
    open(OUT, '>graph/' . $filename) || msg_error($!);
    print OUT $svg->xmlify;
    close(OUT);
    msg_gimv('graph/' . $filename) if ($output eq 'show');
    
    return 1;
}







=head2 genome_map3

  Name: genome_map3   -   draws the map of the genome (version 3)

  Description:
    This method creates a map of the genome, showing the local 
    nucleotide contents and positions of genes.
    A is shown in red, T is shown in green, 
    G is shown in yellow, and C is shown in blue. 

    Entire genome is prepresented in a very large PNG image
    (8192x8192 pixel by default), and the number of nucleotides
    represented by one pixel is automatically calculated. Then 
    the large image is converted to a zoomable image usinig 
    Google Map API.

  Usage:
    NULL = genome_map($gb);

 Options:
    -width          image width (default: 8192)
    -height         image height (default: 8192)
    -filename       output filename (default: 'genome_map3.png')
    -output         "g" for graph, "show" for display (default: "show") 
    -datafilename   coordinate file name (default: "[accessionNumber].coord")
    -gmap           1 to create Google Map view, 0 to only generate PNG image
                    (default:1)

  Author: 
    Kazuharu Arakawa (gaou@sfc.keio.ac.jp)

  History:
   20120824-01 added -filename option
   20100319-01 added support for exons
   20071008-01 added -gmap option
   20070628-01 added -datafilename option
   20070623-01 initial posting

=cut

sub genome_map3 {
    &opt_default(output=>"show", width=>8192, height=>8192, datafilename=>'', gmap=>1, filename=>"genome_map3.png");
    my @args         = opt_get(@_);
    my $gb           = opt_as_gb(shift @args);
    my $acnum        = $gb->{LOCUS}->{id};
    my $datafilename = opt_val("datafilename") || $gb->id() . '.coord';
    my $output       = opt_val("output");
    my $filename     = opt_val("filename");
    my $topmargin    = 50;
    my $sidemargin   = 150;
    my $hblock       = 100;
    my $vblock       = 10;
    my $width        = opt_val("width");
    my $height       = opt_val("height");
    my $gmap         = opt_val("gmap");

    my $numHBlock = int(($width - $sidemargin - 50) / $hblock);
    my $numVBlock = int(($height - $topmargin - 50) / ($vblock * 5));
    my $window    = int(length($gb->{SEQ}) / ($numHBlock * $numVBlock * $hblock)) + 1;

    msg_error("Image size: $width x $height pixels.\nUsing Window size = $window bp to fit the whole genome...\n\n");
    msg_error("Map generation will take a few minutes. Please make yourself comfortable until the image is generated ;-\)\n\n") if ($gmap);

    my $start = 1;
    my $end = length $gb->{SEQ};
    my $i = 0;
    my $cds = 1;

    mkdir ("graph", 0777);
    mkdir ("data", 0777);
    open(OUT, '>' . 'data/' . $datafilename) || die($!);

    # GD constant
    my $im = new GD::Image($width, $height);
    my $white  = $im->colorAllocate(255,255,255);
    my $black  = $im->colorAllocate(0,0,0);
    my $gray   = $im->colorAllocate(180,180,180);  
    my $red    = $im->colorAllocate(255,0,0);    #A
    my $yellow = $im->colorAllocate(255,255,0);  #G
    my $green  = $im->colorAllocate(0,150,0);    #T
    my $blue   = $im->colorAllocate(0,0,255);    #C
    my $aqua   = $im->colorAllocate(120, 160, 255);
    my $rrna   = $im->colorAllocate(255, 160, 160);
    my $trna   = $im->colorAllocate(160, 255, 160);

    my $gred    = $im->colorAllocate(255,200,200);   #A for graph
    my $gyellow = $im->colorAllocate(255,255,100);   #G for graph
    my $ggreen  = $im->colorAllocate(180,220,180);   #T for graph
    my $gblue   = $im->colorAllocate(200,200,255);   #C for graph
  
    # Draw Base Graph
    for ($i = $sidemargin; $i <= $sidemargin + $hblock * $numHBlock; $i += $hblock){
	$im->line($i, $topmargin, $i, $topmargin + 5 * ($numVBlock + 1) * $vblock, $gray);
    }
    for ($i = $topmargin; $i <= $topmargin + $vblock * 5 * ($numVBlock + 1); $i += $vblock){
	$im->line($sidemargin, $i, $sidemargin + $numHBlock * $hblock, $i, $gray);
    }
    for ($i = $topmargin + $vblock * 5; $i < $topmargin + $vblock * 5 * ($numVBlock + 1); $i += $vblock * 5){
	$im->line($sidemargin, $i - 1, $sidemargin + $numHBlock * $hblock, $i - 1, $black);
	$im->line($sidemargin, $i + 1, $sidemargin + $numHBlock * $hblock, $i + 1, $black);
    }

    $im->string(gdGiantFont, $width - 210, 5, "G-language Project", $black);
    $im->string(gdGiantFont, $width - 210 - 50, 5, "A", $red);
    $im->string(gdGiantFont, $width - 210 - 40, 5, "T", $green);
    $im->string(gdGiantFont, $width - 210 - 30, 5, "G", $yellow);
    $im->string(gdGiantFont, $width - 210 - 20, 5, "C", $blue);

    my $j = 0;
    for ($i = $topmargin + $vblock * 5; $i <= $topmargin + $vblock * 5 * $numVBlock; $i += $vblock * 5){
	my $num = $start + $j * $window * $numHBlock * $numVBlock * $hblock;
	$im->string(gdLargeFont, 10, $i, "$num", $black);
	$j ++;
    }
    $im->string(gdGiantFont, 50,  5, "$acnum : from $start to $end", $black);

    my ($pa, $pt, $pg, $pc, $num, $color);
    my $locus = 0;
    for ($i = $start - 1; $i + $window <= $end; $i += $window){
	my $seq = $gb->getseq($i, $i + $window - 1);
	my $a = $seq =~ tr/a/a/;
	my $t = $seq =~ tr/t/t/;
	my $g = $seq =~ tr/g/g/;
	my $c = $seq =~ tr/c/c/;

	# Draw DNA
	if ($a >= $t && $a >= $g && $a >= $c){
	    my $num = int($a / $window * 100);
	    $color = $red;
	}elsif ($t >= $a && $t >= $g && $t >= $c){
	    my $num = int($t / $window * 100);
	    $color = $green;
	}elsif ($g >= $a && $g >= $t && $g >= $c){
	    my $num = int($g / $window * 100);
	    $color = $yellow;
	}elsif ($c >= $a && $c >= $t && $c >= $g){
	    my $num = int($c / $window * 100);
	    $color = $blue;
	}
	$im->setPixel($sidemargin + 1 + $locus % (($hblock * $numHBlock)), $topmargin + (int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5, $color);

	my $dist = 7;
	if (($locus % (($hblock * $numHBlock))) % $dist == $dist - 1){
	    # Draw A content graph
	    $num = int(($a / $window) * $vblock * 3.8) + 2;

	    $im->line($sidemargin - $dist + 1 + $locus % (($hblock * $numHBlock)), $pa, 
		      $sidemargin + $locus % (($hblock * $numHBlock)),
		      $topmargin - $num + (int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5, $gred);
	    $pa = $topmargin - $num + (int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;
	    
	    # Draw T content graph
	    $num = int(($t / $window) * $vblock * 3.8) + 2;
	    $im->line($sidemargin - $dist + 1 + $locus % (($hblock * $numHBlock)), $pt, 
		      $sidemargin + $locus % (($hblock * $numHBlock)),
		      $topmargin - $num + (int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5, $ggreen);
	    $pt = $topmargin - $num +(int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;
	    
	    # Draw G content graph
	    $num = int(($g / $window) * $vblock * 3.8) + 2;
	    $im->line($sidemargin - $dist + 1 + $locus % (($hblock * $numHBlock)), 
		      $pg, $sidemargin + $locus % (($hblock * $numHBlock)),
		      $topmargin - $num + (int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5, $gyellow);
	    $pg = $topmargin - $num +(int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;
	    
	    # Draw C content graph
	    $num = int(($c / $window) * $vblock * 3.8) + 2;
	    $im->line($sidemargin - $dist + 1 + $locus % (($hblock * $numHBlock)), 
		      $pc, $sidemargin + $locus % (($hblock * $numHBlock)),
		      $topmargin - $num + (int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5, $gblue);
	    $pc = $topmargin - $num +(int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;

	}elsif($locus % ($hblock * $numHBlock) == 0){

	    $num = int(($a / $window) * $vblock * 3.8) + 2;
	    $pa = $topmargin - $num +(int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;
	    $num = int(($t / $window) * $vblock * 3.8) + 2;
	    $pt = $topmargin - $num +(int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;
	    $num = int(($g / $window) * $vblock * 3.8) + 2;
	    $pg = $topmargin - $num +(int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;
	    $num = int(($c / $window) * $vblock * 3.8) + 2;
	    $pc = $topmargin - $num +(int($locus / ($hblock * $numHBlock)) + 1) * $vblock * 5;
	}
	$locus ++;
    }

    # Draw Genes
    my $flag = 0;
    my $before = -5000;
    my $before2 = -10000;
    foreach my $feat ($gb->feature()){
	next unless($gb->{$feat}->{type} eq 'CDS' || $gb->{$feat}->{type} eq 'rRNA' || $gb->{$feat}->{type} eq 'tRNA');
	my $cdsstart = $gb->{$feat}->{start};
	my $cdsend   = $gb->{$feat}->{end};
	my $cdsdir   = $gb->{$feat}->{direction};
	my $cdsdiff  = $cdsstart - $before;
	my $cdsdiff2 = $cdsstart - $before2;
	my $genename = $gb->{$feat}->{gene};

	if ($flag == 0){
	    if (int($cdsdiff / $window) < 25){
		$flag = 1;
	    }
	}elsif ($flag == 1){
	    if (int($cdsdiff / $window) < 25){
		if (int($cdsdiff2 / $window) < 25){
		    $flag = 2;
		}else{
		    $flag = 0;
		}
		$flag = 0;
	    }else{
		$flag = 0;
	    }
	}elsif ($flag == 2){
	    if (int($cdsdiff2 / $window) < 25){
		$flag = 1;
	    }else{
		$flag = 0;
	    }
	}

	my $blockcolor = $aqua;
	$blockcolor = $rrna if ($gb->{$feat}->{type} eq 'rRNA');
	$blockcolor = $trna if ($gb->{$feat}->{type} eq 'tRNA');

	unless ($cdsstart >= $end){
	    
	    my $dif1 = -2;
	    my $dif2 = -7 - 2;
	    my $dif3 = $dif2 - 5;
	    if ($cdsdir eq 'complement'){
		$dif1 *= -1;
		$dif2 *= -1;
		$dif3 *= -1;
	    }

	    if($cdsdir eq 'complement'){
		printf OUT "%s\t%s\t%s\t%s\t%s\t%s\n", $feat, $gb->{$feat}->{type}, $sidemargin + ($cdsstart / $window) % ($hblock * $numHBlock),
		$topmargin + $dif1 + (int(($cdsstart / $window) / ($hblock * $numHBlock))+1) * $vblock * 5,
		$sidemargin + ($cdsend / $window) % ($hblock * $numHBlock),
		$topmargin + $dif2 + (int(($cdsend / $window) / ($hblock * $numHBlock))+1) * $vblock * 5;
	    }else{
		printf OUT "%s\t%s\t%s\t%s\t%s\t%s\n", $feat, $gb->{$feat}->{type}, $sidemargin + ($cdsstart / $window) % ($hblock * $numHBlock),
		$topmargin + $dif2 + (int(($cdsstart / $window) / ($hblock * $numHBlock))+1) * $vblock * 5,
		$sidemargin + ($cdsend / $window) % ($hblock * $numHBlock),
		$topmargin + $dif1 + (int(($cdsend / $window) / ($hblock * $numHBlock))+1) * $vblock * 5;
	    }

	    my $k;
	    if($gb->{$feat}->{join}){
		my $prev = 0;
		for my $exon (split(/,/, $gb->{$feat}->{join})){
		    my ($exonstart, $exonend) = split(/\.\./, $exon, 2);
		    for ($k = $exonstart-$start; $k <= $exonend - $start; $k += $window){
			last if ($k + $start > $end);
			my $l = int ($k / $window);
			$im->line($sidemargin + 1 + $l % ($hblock * $numHBlock),
				  $topmargin + $dif1 + (int($l/($hblock * $numHBlock))+1)*$vblock * 5,
				  $sidemargin + 1 + $l % ($hblock * $numHBlock),
				  $topmargin + $dif2 + (int($l/($hblock * $numHBlock))+1)*$vblock * 5,
				  $blockcolor);
		    }
		    unless($prev == 0){
			my $l1 = int (($prev - $start) / $window);
			my $l2 = int (($exonstart - $start) / $window);
			my $l3 = $l1 + int(($l2 - $l1)/2);
			if((int($l1/($hblock * $numHBlock))+1)*$vblock * 5 != (int($l2/($hblock * $numHBlock))+1)*$vblock * 5){
			    $im->line($sidemargin + 1 + $l1 % ($hblock * $numHBlock),
				      $topmargin + $dif2 + (int($l1/($hblock * $numHBlock))+1)*$vblock * 5,
				      $sidemargin + 1 + $hblock * $numHBlock,
				      $topmargin + $dif3 + (int($l1/($hblock * $numHBlock))+1)*$vblock * 5,
				      $black);
			    $im->line($sidemargin + 1,
				      $topmargin + $dif3 + (int($l2/($hblock * $numHBlock))+1)*$vblock * 5,
				      $sidemargin + 1 + $l2 % ($hblock * $numHBlock),
				      $topmargin + $dif2 + (int($l2/($hblock * $numHBlock))+1)*$vblock * 5,
				      $black);
			}else{
			    $im->line($sidemargin + 1 + $l1 % ($hblock * $numHBlock),
				      $topmargin + $dif2 + (int($l3/($hblock * $numHBlock))+1)*$vblock * 5,
				      $sidemargin + 1 + $l3 % ($hblock * $numHBlock),
				      $topmargin + $dif3 + (int($l3/($hblock * $numHBlock))+1)*$vblock * 5,
				      $black);
			    $im->line($sidemargin + 1 + $l2 % ($hblock * $numHBlock),
				      $topmargin + $dif2 + (int($l3/($hblock * $numHBlock))+1)*$vblock * 5,
				      $sidemargin + 1 + $l3 % ($hblock * $numHBlock),
				      $topmargin + $dif3 + (int($l3/($hblock * $numHBlock))+1)*$vblock * 5,
				      $black);
			}
		    }

		    $prev = $exonend;
		}
	    }else{
		for ($k = $cdsstart-$start; $k <= $cdsend - $start; $k += $window){
		    last if ($k + $start > $end);
		    my $l = int ($k / $window);
		    $im->line($sidemargin + 1 + $l % ($hblock * $numHBlock),
			      $topmargin + $dif1 + (int($l/($hblock * $numHBlock))+1)*$vblock * 5,
			      $sidemargin + 1 + $l % ($hblock * $numHBlock),
			      $topmargin + $dif2 + (int($l/($hblock * $numHBlock))+1)*$vblock * 5,
			      $blockcolor);
		}
		last if ($k + $start > $end);
	    }
	    $k = int(($cdsstart - $start)/$window);
	    
	    $dif1 = -2; 
	    $dif2 = -2 - 11;
	    my $dif3 = -2 - 15 - 9 + (-12 * $flag);
	    if ($cdsdir eq 'complement'){
		$dif1 *= -1;
		$dif2 *= -1;
		$dif3 *= -1;
		$dif3 -= 15;
	    }
	    $im->line($sidemargin + 1 + $k % ($hblock * $numHBlock),
		      $topmargin + $dif1 + (int($k/($hblock * $numHBlock))+1)*$vblock * 5,
		      $sidemargin + 1 + $k % ($hblock * $numHBlock),
		      $topmargin + $dif2 + (int($k/($hblock * $numHBlock))+1)*$vblock * 5,
		      $black);
	    
	    $im->string(gdMediumBoldFont, $sidemargin + 1 + $k %($hblock * $numHBlock),
			$topmargin + $dif3 + (int($k/($hblock * $numHBlock))+1)*$vblock * 5,
			"$genename", $black);
	    
	    if ($cdsend > $end){
		last;
	    }else{
		$before = $cdsstart;
		$before2 = $before;
	    }
	}
    }
    close(OUT);

    open(IMAGE, '>graph/' . $filename);
    binmode IMAGE;
    print IMAGE $im->png;
    close(IMAGE);

    generateGMap("graph/$filename") if($gmap);
}






=head2 seq2png

 Name: seq2png   -   converts a sequence to PNG image

 Description:
  Converts a sequence to a png image, by representing nucleotide 
  sequences with representative pixels. 
  A is shown in red, T is shown in green, 
  G is shown in yellow, and C is shown in blue. 

  Usage: 
    NULL = seq2png(G instance); 

 Options:
    -width     width of the image (default:640)
    -window    window size of a sequence to represent each pixel (default: 20)
    -filename  output filename (default:'seq.png')
    -output    "g" for graph, "show" for display


  Author: 
    Kazuharu Arakawa (gaou@sfc.keio.ac.jp)

  History:
    20060714 added -window option
    20010906 update with options
    20010830 initial posting

=cut




sub seq2png {
    &opt_default(width=>640, filename=>"seq.png", output=>"show", window=>"20");
    my @args = opt_get(@_);
    my $gb = opt_as_gb(shift @args);
    my $width = opt_val("width");
    my $window = opt_val("window");
    my $output = opt_val("output");
    my $filename = opt_val("filename");
    my $height = int((length($gb->{SEQ})/$window)/$width)+1;
    my $im = new GD::Image($width, $height);

    my $white = $im->colorAllocate(255,255,255);
    my $red = $im->colorAllocate(255,0,0);
    my $yellow = $im->colorAllocate(255,255,0);
    my $green = $im->colorAllocate(0,150,0);
    my $blue = $im->colorAllocate(0,0,255);

    my ($x, $y);
    my $count = 0;
    for ($y = 0; $y <= $height; $y ++){
	for ($x = 0; $x <= $width; $x ++){
	    my $color=$white;

	    my $g = substr($gb->{SEQ}, $count * $window, $window) =~ tr/g/g/;
	    my $c = substr($gb->{SEQ}, $count * $window, $window) =~ tr/c/c/;
	    my $a = substr($gb->{SEQ}, $count * $window, $window) =~ tr/a/a/;
	    my $t = substr($gb->{SEQ}, $count * $window, $window) =~ tr/t/t/;

	    if($g >= $c && $g >= $a && $g >= $t){
		$color=$yellow;
	    }elsif($c >= $g && $c >= $a && $c >= $t){
		$color = $blue;
	    }elsif($a >= $g && $a >= $c && $a >= $t){
		$color = $red;
	    }elsif($t >= $g && $t >= $c && $t >= $a){
		$color = $green;
	    }
	    
	    $im->setPixel($x,$y,$color);
	    last if ($count == length($gb->{SEQ}));
	    $count ++;
	}
    }

    mkdir ('graph', 0777);
    open(OUT, '>graph/' . $filename);
    binmode OUT;
    print OUT $im->png;
    close(OUT);

    msg_gimv("graph/$filename") if ($output eq 'show');
}




=head2 plasmid_map

  Name: plasmid_map   -   draws circular map of the genome

  Description:
   This method creates a circular map of the genome using SVG,
   suitable for plasmids and circular bacterial chromosomes.

  Usage:
    NULL = plasmid_map($gb);

 Options:
   -ptt       ptt file name
   -output    g|show (default:show)
   -filename  output file name (default:plasmid_map.svg)

  Author: 
    Kazuharu Arakawa (gaou@sfc.keio.ac.jp)

  History:
   20071023-01 fixed SVG-related bug
   20020905-01 initial posting

=cut


sub plasmid_map{
    require SVG;

    opt_default(ptt=>'', output=>'show', filename=>"plasmid_map.svg", cgi=>0);
    my @args = opt_get(@_);
    my $gb = shift;
    my $ptt = opt_val("ptt");
    my $output = opt_val("output");
    my $filename = opt_val("filename");
    my $cgi = opt_val("cgi");

    eval{
	$ptt = set_gpac($gb, -ptt=>$ptt);
	 };

    unless ($cgi){
	sdb_save(\$ptt, "ptt");
	sdb_save($gb, "gb");
    }

    my $svg = SVG->new(width=>640, height=>400);
    my $pi = atan2(1,1) * 4;
    my $seqlen = length $gb->{SEQ};
    my $maxlen;
    foreach my $cds ($gb->cds()){
	my $tmplen = length ($gb->get_geneseq($cds));
	$maxlen = $tmplen if ($tmplen > $maxlen);
    }
    
    foreach my $cds ($gb->cds()){
	my $id = $cds;
	my $gene = $gb->get_geneseq($cds);
	my $length = length $gene;
	my $gc = int(log($maxlen) / log($length) * 30);
	
	my $rad = 2 * $pi * $gb->{$cds}->{start} / $seqlen * -1 + $pi;
	
	my ($x1, $x2, $y1, $y2);
	if ($gb->{$cds}->{direction} eq 'direct'){
	    $x1 = (100 + 5) * sin($rad) + 200;
	    $y1 = (100 + 5) * cos($rad) + 200;
	    $x2 = (100 + $gc) * sin($rad) + 200;
	    $y2 = (100 + $gc) * cos($rad) + 200;
	}else{
	    $x1 = (100 - $gc) * sin($rad) + 200;
	    $y1 = (100 - $gc) * cos($rad) + 200;
	    $x2 = (100 - 5) * sin($rad) + 200;
	    $y2 = (100 - 5) * cos($rad) + 200;
	}
	
	my $stroke = $COG_fcolor{$gb->{$cds}->{code}};
	
	my $cdsstart = $gb->{$cds}->{start};
	$svg->anchor(
		     -href=>'http://localhost/g-language/genome_view.cgi?pos=' . $cdsstart,
		     -target=>"_blank"
		     )->line(
			     id=>$id, x1=>$x1, 
			     y1=>$y1, x2=>$x2, y2=>$y2,
			     style=>{
				 stroke=>$stroke, 'stroke-width'=>1,
				 'stroke-opacity'=>0.3
				 });
    }
    
    $svg->circle(id=>'genome', cx=>200, cy=>200, r=>100, 
		 style=>{
		     stroke=>'blue', fill=>'none', 'stroke-width'=>5,
		     'stroke-opacity'=>0.2
		     });
    
    my $x = 400;
    my $y = 123;
    my $y2 = 130;
    
    $svg->anchor(
		 -href=>"http://www.ncbi.nih.gov/cgi-bin/COG/palox?fun=all",
		 -target=>"_blank"
		 )->text(
			 id=>"COG link", 
			 x=>($x - 20), y=>($y2 - 30), 
			 'font-size'=>9
			 )->cdata(
				  "Gene Classification based on NCBI COG functional categories"
				  );
    
    foreach my $category (qw(J K L D O M N P T C G E F H I Q R S -)){
	$svg->rect(id=>$category, x=>$x, y=>$y, width=>9, height=>8,
		   style=>{
		       stroke=>'none', fill=>$COG_fcolor{$category},
		       opacity=>0.3
		       });
	
	my $id = $category . '_text';
	
	$svg->text(id=>$id, x=>($x + 15), y=>$y2, 'font-size'=>7)->cdata($COG_fcode{$category});
	
	$y += 12;
	$y2 += 12;
    }
    
    my $topline = $svg->line(id=>"top", x1=>0, y1=>10, x2=>640, y2=>10,
			     style=>{
				 stroke=>"blue", 'stroke-width'=>3,
				 'stroke-opacity'=>0.3
				 });
    
    $topline->animate(
		      attributeName=>"stroke", 
		      values=>join(';', values(%COG_fcolor)), dur=>"90s",
		      repeatDur=>'indefinite'
		      );
    
    my $bottomline = $svg->line(id=>"bottom", x1=>0, y1=>390, x2=>640, y2=>390,
				style=>{
				    stroke=>"blue", 'stroke-width'=>3,
				    'stroke-opacity'=>0.3
				    });
    
    $bottomline->animate(
			 attributeName=>"stroke", 
			 values=>join(';', values(%COG_fcolor)), dur=>"90s",
			 repeatDur=>'indefinite'
			 );
    
    $svg->anchor(
		 -href=>"http://www.g-language.org/",
		 -target=>"_blank"
		 )->text(
			 id=>"credits", 
			 x=>380, y=>385, fill=>"darkgray", 
			 'font-size'=>8, 'font-style'=>'italic',
			 )->cdata(
				  "generated by plasmid map, G-language Genome Analysis Environment"
				  );
    
    $svg->text(
	       id=>"locus1", 
	       x=>30, y=>30, fill=>"navy", 
	       'font-size'=>10,
	       )->cdata("Accession Number: ", $gb->{LOCUS}->{id});
    
    $svg->text(
	       id=>"locus2", 
	       x=>250, y=>30, fill=>"navy", 
	       'font-size'=>10,
	       )->cdata("Organism: ", $gb->{FEATURE0}->{organism});
    
    $svg->text(
	       id=>"locus3", 
	       x=>500, y=>30, fill=>"navy", 
	       'font-size'=>10,
	       )->cdata("$seqlen bp");
    
    mkdir('graph', 0777);
    open(OUT, '>graph/' . $filename) || msg_error($!);
    print OUT $svg->xmlify;
    close(OUT);

    msg_gimv('graph/' . $filename) if ($output eq "show");

    return 1;
}









=head2 dnawalk

  Name: dnawalk   -   draws DNA Walk map of the genome

  Description:
   This method draws the DNA Walk map of the given genome.
   DNA Walk is drawn by moving a single pixel per nucleotide, 
   in the direction specified for each base. Here A is moved
   upward, T downward, G to the right, and C to the left.
   Position zero (first letter of the genome) is indicated
   by the crossing of thin axes.
   
   Generated PNG is very large (8000x8000 pixels), and it is
   suited to be converted to Google Map View with generate_gmap().

   Output file is created in the "graph" directory, with 
   the accession number as file name and ".png" as extension.

  Usage:
    NULL = dnawalk($gb);

 Options:
    -filename       output filename (default:'dnawalk.png')  
    -gmap           1 to create Google Map view, 0 to only generate PNG image
                    (default:1)

  Author: 
    Keita Ikegami (t06056ki@sfc.keio.ac.jp)

  History:
   20120824-01 added -filename option
   20080414-01 added gmap option
   20071023-01 initial posting

=cut


sub dnawalk{
    opt_default("gmap"=>1, "filename"=>'dnawalk.png', "output"=>"g");
    my @args = opt_get(@_);

    my $gb = opt_as_gb(shift @args);
    my $name = $gb->{LOCUS}->{id};
    my $segments = 10;
    my $gmap = opt_val("gmap");
    my $filename = opt_val("filename");

    mkdir("data", 0777);
    mkdir("graph", 0777);
    
    my ($XX, $YY) = (8192, 8192);
    my $im = new GD::Image($XX,$YY);    
    my $margin = 50;
    
    my ($cds_x, $cds_y);
    my ($out, @out);
    
    msg_error("Map generation will take a few minutes. Please make yourself comfortable until the image is generated ;-\)\n\n") if ($gmap);

    my $black   = $im->colorAllocate(0  ,0  ,0  ); ### black
    my $white   = $im->colorAllocate(255,255,255); ### white
    my $momo    = $im->colorAllocate(255,127,127); ### momo
    my $h_p     = $im->colorAllocate(255,127,191); ### hard_pink
    my $s_p     = $im->colorAllocate(255,127,255); ### soft_pink
    my $ppl     = $im->colorAllocate(191,127,255); ### purple
    my $h_b     = $im->colorAllocate(127,127,255); ### hard_blue
    my $s_b     = $im->colorAllocate(127,191,255); ### soft_blue
    my $ss_b    = $im->colorAllocate(127,255,255); ### soft+soft_blue
    my $s_g     = $im->colorAllocate(127,255,191); ### soft_green
    my $h_g     = $im->colorAllocate(127,255,127); ### hard_green
    my $g_y     = $im->colorAllocate(191,255,127); ### green_yallow
    my $sb_y    = $im->colorAllocate(255,255,127); ### sibui_yellow
    my $p_y     = $im->colorAllocate(255,191,127); ### pink_yallew
    my $black   = $im->colorAllocate(0,  0,  0, ); ### black
    
    my @colors = ($momo, $h_p, $s_p, $ppl, $h_b, $s_b, $ss_b, $s_g, $h_g, $g_y);
    
    my ( $x, $y, $color ) = ( 0, 0, $black );
    my ( @pos_x, @pos_y) ;
    my $nuc;
    
    for (my $i=0; $i<=$segments; $i++){
	my $tmpseq = substr( $gb->{SEQ},  $i * int(length($gb->{SEQ})/$segments),  int(length($gb->{SEQ})/$segments) );
	$tmpseq =~ tr[atgc][0286];
	
	foreach my $word (split(//, $tmpseq)){
	    $nuc++;
	    
	    if($word < 5){
		$y += ($word - 1);
	    }else{
		$x += ($word - 7);
	    }
	    
	    push (@pos_x, $x);
	    push (@pos_y, $y); 
	}
    }
    
    my $walk_width  = max(@pos_x) - min(@pos_x);
    my $walk_height = max(@pos_y) - min(@pos_y);

    my $compress;
    my ($cx, $cy);
    
    if( $walk_width > $walk_height ){
	$compress = $walk_width / (8192 - $margin);
	
	if( $compress < 1  &&  $compress > 0 ){
	    $compress = 1;
	    
	    $cx = int (-min(@pos_x)/ $compress + ( 8192 - $walk_width/$compress) / 2);
	    $cy = int (-min(@pos_y)/ $compress + ( 8192 - $walk_height/$compress)/ 2) ;
	}else{
	    $cx = int (- min(@pos_x) / $compress) + $margin/2;
	    $cy = int (- min(@pos_y) / $compress + ( 8192 - $walk_height/$compress) / 2);
	}

    }else{
	
	$compress = $walk_height / (8192 - $margin);
	
	if( $compress < 1  &&  $compress > 0 ){
	    $compress = 1;
	    
	    $cx = int (-min(@pos_x)/ $compress + ( 8192 - $walk_width/$compress) / 2);
	    $cy = int (-min(@pos_y)/ $compress + ( 8192 - $walk_height/$compress)/ 2) ;
	}else{
	    $cx = int (-min(@pos_x)/ $compress + ( 8192 - $walk_width/$compress) / 2);
	    $cy = int (-min(@pos_y) / $compress) +$margin/2;
	}
    }
    
    $im->line(0,   $cy, 8192, $cy,  $white); # yoko
    $im->line($cx, 0,   $cx,  8192, $white); # tate
    
    my $j = 0;
    my $k = 0;
    
    for (my $i=0; $i<=length($gb->{SEQ}); $i++){
	$j++;
	if ($j > length($gb->{SEQ}) / $segments){
	    $k++;
	    $j = 0;
	}
	$color = $colors[$k];

	my $xs = int( $cx + $pos_x[$i] / $compress );
	my $ys = int( $cy + $pos_y[$i] / $compress );
	
	$im->setPixel( $xs, $ys, $color );
    }
    
    open DATA, ">./data/$name.coord";
    
    for my $cds ($gb->feature()){
	$cds_x = int ($cx + @pos_x[$gb->{$cds}->{start}] / $compress);
	$cds_y = int ($cy + @pos_y[$gb->{$cds}->{start}] / $compress);
	my $gene = $gb->{$cds}->{gene} or $gb->{$cds}->{locus_tag};
	my $type = $gb->{$cds}->{type};
	print DATA "$cds\t$type\t$gene\t$cds_x\t$cds_y\n";	
    }
    close DATA;   
    
    open NEW, ">./graph/" . $filename;
    binmode NEW;
    print NEW $im->png;
    close NEW;

    generateGMap("graph/$filename") if($gmap);
}

1;
