/**********************************************************************
 
	Copyright (C) 2007- Hirohisa MORI <joshua@globalbase.org>
 
	This program is free software; you can redistribute it 
	and/or modify it under the terms of the GLOBALBASE 
	Library General Public License (G-LGPL) as published by 

	http://www.globalbase.org/
 
	This program is distributed in the hope that it will be 
	useful, but WITHOUT ANY WARRANTY; without even the 
	implied warranty of MERCHANTABILITY or FITNESS FOR A 
	PARTICULAR PURPOSE.

**********************************************************************/

#include	<stdio.h>
#include	<math.h>
#include	"geo.h"

typedef struct e_work {
	ELIPSOID *	ep;
	double		phi;
	double		lambda;
	double		N;
	double		M;
	double		S;
}E_WORK;


void
print_elipsoid(ELIPSOID * ep)
{
	printf("a=%f b=%f e=%f m0=%f org_X0=%f org_Y0=%f org_X=%f org_phi=%f org_lambda=%f org_X_enable=%i\n",
		ep->a,ep->b,ep->e,ep->m0,ep->org_X0,ep->org_Y0,ep->org_X,ep->org_phi,ep->org_lambda,ep->org_X_enable);
}


double
get_e(ELIPSOID * e)
{
double a2;
double b2;
	a2 = e->a*e->a;
	b2 = e->b*e->b;
	e->e =  sqrt(e->e2 = (a2-b2)/a2);
	return e->e;
}

void
get_r(E_WORK * w)
{
ELIPSOID * ep;
double sin_phi;
double te,te2;
	ep = w->ep;
	sin_phi = sin(w->phi);
	te2 = sqrt(te = 1 - ep->e2*sin_phi*sin_phi);
	w->N = ep->a/te2;
	w->M = ep->a*(1 - ep->e2)/(te*te2);
}

double
get_S(E_WORK * w)
{
double tan_phi;
double sin_2phi;
double sin_4phi;
double sin_6phi;
double sin_8phi;
double e4,e6,e8;
ELIPSOID * ep;
	ep = w->ep;
	tan_phi = tan(w->phi);
	sin_2phi = sin(2*w->phi);
	sin_4phi = sin(4*w->phi);
	sin_6phi = sin(6*w->phi);
	sin_8phi = sin(8*w->phi);
	e4 = ep->e2*ep->e2;
	e6 = e4 * ep->e2;
	e8 = e6 * ep->e2;
	w->S = ep->a * (
			w->phi +
			(- 1.0/4* w->phi - 3.0/8*sin_2phi)*ep->e2 +
			(- 3.0/64* w->phi - 3.0/32*sin_2phi + 15.0/256*sin_4phi)*e4 +
			(- 5.0/256*w->phi - 45.0/1024*sin_2phi + 45.0/1024*sin_4phi - 35.0/3072*sin_6phi)*e6 +
			(- 175.0/16384*w->phi - 105.0/4096*sin_2phi + 525.0/16384*sin_4phi - 175.0/12288*sin_6phi
						+ 315.0/131072*sin_8phi)*e8
		);
	return w->S;
}

/* GSI ALOS PRISM */
void
get_XY_1(double * X_ptr,double *Y_ptr,E_WORK * w)
{
double X,Y;
double d_lambda;
double lambda2;
double lambda4;
double lambda6;
double nyu;
double nyu2;
double nyu4;
double cos_phi;
double cos_phi2;
double cos_phi4;
double tan_phi;
double tan_phi2;
double tan_phi4;
ELIPSOID * ep;
	ep = w->ep;
	d_lambda = w->lambda - ep->org_lambda;
	lambda2 = d_lambda*d_lambda;
	lambda4 = lambda2 * lambda2;
	lambda6 = lambda4 * lambda2;
	
	cos_phi = cos(w->phi);
	cos_phi2 = cos_phi*cos_phi;
	cos_phi4 = cos_phi2*cos_phi2;
	tan_phi = tan(w->phi);
	tan_phi2 = tan_phi*tan_phi;
	tan_phi4 = tan_phi2*tan_phi2;
	
	nyu = ep->e2*cos_phi/(1-ep->e2);
	nyu2 = nyu*nyu;
	nyu4 = nyu2*nyu2;

	X = w->S + w->N*sin(w->phi)*cos_phi*
		(
		lambda2/2 + 
		lambda4*cos_phi2/24*(5 - tan_phi2 + 9*nyu2 + 4*nyu4) +
		lambda6*cos_phi4/720*(61 - 58*tan_phi2 + 270*nyu2)
		);
	Y = w->N*cos_phi*(
		d_lambda + 
		d_lambda*lambda2*cos_phi2/6*(1 - tan_phi2 + nyu2) +
		d_lambda*lambda4*cos_phi4/120*(5 - 18*tan_phi2 + 14*nyu2)
		);

	*Y_ptr = Y*ep->m0 + ep->org_Y0;

	/* origin */
	
	if ( ep->org_X_enable == 0 ) {

		cos_phi = cos(ep->org_phi);
		cos_phi2 = cos_phi*cos_phi;
		cos_phi4 = cos_phi2*cos_phi2;
		tan_phi = tan(ep->org_phi);
		tan_phi2 = tan_phi*tan_phi;
		tan_phi4 = tan_phi2*tan_phi2;
		
		nyu = ep->e2*cos_phi/(1-ep->e2);
		nyu2 = nyu*nyu;
		nyu4 = nyu2*nyu2;

		ep->org_X = w->S + w->N*sin(ep->org_phi)*cos_phi*
			(
			lambda2/2 + 
			lambda4*cos_phi2/24*(5 - tan_phi2 + 9*nyu2 + 4*nyu4) +
			lambda6*cos_phi4/720*(61 - 58*tan_phi2 + 270*nyu2)
			);
		ep->org_X_enable = 1;
	}
	*X_ptr = ep->m0*(X - ep->org_X) + ep->org_X0;
}

/* 
	Nakamura Hideo
	Shimizu Eihan
	"Sokryo-Gaku"
	Giho-Do Shuppan 2000.
	pp.30-31.
	ISBN 4-7655-1568-0 C 3051
*/
void
get_XY_2(double * X_ptr,double *Y_ptr,E_WORK * w)
{
ELIPSOID * ep;
double X,Y;
double sin_phi;
double cos_phi;
double cos_phi2;
double cos_phi3;
double cos_phi5;
double tan_phi;
double tan_phi2;
double tan_phi4;
double d_lambda;
double lambda2;
double lambda4;
double lambda6;

	ep = w->ep;

	sin_phi = sin(w->phi);
	cos_phi = cos(w->phi);
	cos_phi2 = cos_phi*cos_phi;
	cos_phi3 = cos_phi*cos_phi2;
	cos_phi5 = cos_phi3*cos_phi2;

	tan_phi = tan(w->phi);
	tan_phi2 = tan_phi*tan_phi;
	tan_phi4 = tan_phi2*tan_phi2;

	d_lambda = w->lambda - ep->org_lambda;
	lambda2 = d_lambda*d_lambda;
	lambda4 = lambda2*lambda2;
	lambda6 = lambda4*lambda2;
	
	X = w->S + w->N/2*sin_phi*cos_phi*lambda2
		+ w->N/24*sin_phi*cos_phi3*w->N/w->M*(4*w->N/w->M + 1)*lambda4
		+ w->N/720*sin_phi*cos_phi5*(61-58*tan_phi2 + tan_phi4)*lambda6;
	Y = w->N*cos_phi*d_lambda
		+ w->N/6*cos_phi3*(w->N/w->M - tan_phi2)*d_lambda*lambda2
		+ w->N/120*cos_phi5*(5-18*tan_phi2 + tan_phi4)*d_lambda*lambda4;

	*Y_ptr = Y*ep->m0 + ep->org_Y0;

	/* origin */
	
	if ( ep->org_X_enable == 0 ) {
		sin_phi = sin(ep->org_phi);
		cos_phi = cos(ep->org_phi);
		cos_phi2 = cos_phi*cos_phi;
		cos_phi3 = cos_phi*cos_phi2;
		cos_phi5 = cos_phi3*cos_phi2;

		tan_phi = tan(ep->org_phi);
		tan_phi2 = tan_phi*tan_phi;
		tan_phi4 = tan_phi2*tan_phi2;

		ep->org_X = w->S + w->N/2*sin_phi*cos_phi*lambda2
			+ w->N/24*sin_phi*cos_phi3*w->N/w->M*(4*w->N/w->M + 1)*lambda4
			+ w->N/720*sin_phi*cos_phi5*(61-58*tan_phi2 + tan_phi4)*lambda6;

		ep->org_X_enable = 1;
	}
	*X_ptr = ep->m0*(X - ep->org_X) + ep->org_X0;
}


void
utm2bl(double * X_ptr,double * Y_ptr,double lambda,double phi,ELIPSOID * ep)
{
E_WORK w;
	w.ep = ep;
	w.phi = phi;
	w.lambda = lambda;
	get_e(ep);
	get_r(&w);
	get_S(&w);
	get_XY_2(X_ptr,Y_ptr,&w);
}

#ifdef EXAMPLE_1
int
main()
{
ELIPSOID e;
double X,Y,lambda,phi;

	/* Setup of GRS80 */
	e.a = 6378.137;
	e.b = 6356.752;
	e.org_X0 = 0;
	e.org_Y0 = 500;
	e.m0 = 0.9996;
	e.org_X = 0;
	e.org_phi = 0;

	/* Setup Zone 31 North */
	e.org_lambda = 3*2*M_PI/360;
	e.org_X_enable = 1;
	
	lambda = 0;
	phi = 0;
	utm2bl(&X,&Y,lambda*2*M_PI/360,phi*2*M_PI/360,&e);
	printf(">> %f %f => %f %f\n",lambda,phi,X,Y);
	
	lambda = 2;
	phi = 30;
	utm2bl(&X,&Y,lambda*2*M_PI/360,phi*2*M_PI/360,&e);
	printf(">> %f %f => %f %f\n",lambda,phi,X,Y);

	lambda = 2.5;
	phi = 90;
	utm2bl(&X,&Y,lambda*2*M_PI/360,phi*2*M_PI/360,&e);
	printf(">> %f %f => %f %f\n",lambda,phi,X,Y);

	lambda = 5;
	phi = 30;
	utm2bl(&X,&Y,lambda*2*M_PI/360,phi*2*M_PI/360,&e);
	printf(">> %f %f => %f %f\n",lambda,phi,X,Y);

	/* Setup Zone 32 North */
	e.org_lambda = 9*2*M_PI/360;

	lambda = 6;
	phi = 35;
	utm2bl(&X,&Y,lambda*2*M_PI/360,phi*2*M_PI/360,&e);
	printf(">> %f %f => %f %f\n",lambda,phi,X,Y);
}
#endif

#ifdef EXAMPLE_2
/* Convert Speed Checking */
main()
{
int phi_grid,lambda_grid;
double phi,lambda;
ELIPSOID e;
double X,Y;
int tim;
int st;
int total;

	/* Setup of GRS80 */
	e.a = 6378.137;
	e.b = 6356.752;
	e.org_X0 = 0;
	e.org_Y0 = 500;
	e.m0 = 0.9996;
	e.org_X = 0;
	e.org_phi = 0;

	/* Setup Zone 31 North */
	e.org_lambda = 3*2*M_PI/360;
	e.org_X_enable = 1;
	

	st = time(0);
	for ( phi_grid = 0 ; phi_grid < 90*3600 ; phi_grid ++ ) {
		phi = phi_grid/3600.0*2*M_PI/360;
		for ( lambda_grid = 0 ; lambda_grid < 6*3600 ; lambda_grid ++ ) {
			lambda = lambda_grid/3600.0*2*M_PI/360;
			utm2bl(&X,&Y,lambda,phi,&e);
			if ( tim != time(0) ) {
				tim = time(0);
				printf(">> %f %f -> %f %f\n",phi_grid/3600.0,lambda_grid/3600.0,X,Y);
			}
		}
	}
	total = time(0) - st;
	printf("TOTAL TIME %i %g\n",total,total/(90.0*3600*6*3600));
}
#endif

/* RESULT

utm2bl.exe
0 0 -> 0.0 166021.4431
2 30 -> 3319206.2227 403549.8474
2.5 90 -> 9997964.9429 500000
5 30 -> 3320469.2864 692915.1052
6 35 -> 3877156.6915 226201.9037

get_XY_1 GSI ALOS PRISM algorithm
[j3:~/proj/chizu/utm] upa005% !./a
./a.out
>> 0.000000 0.000000 => 0.000000 166.022466
>> 2.000000 30.000000 => 3319.205937 403.549865
>> 2.500000 90.000000 => 9997.964697 500.000000
>> 5.000000 30.000000 => 3320.468995 692.914960
>> 6.000000 35.000000 => 3877.156349 226.202278

Nakamura Hideo et.al. algorithm
[j3:~/proj/chizu/utm] upa005% !./a
./a.out
>> 0.000000 0.000000 => 0.000000 166.021445
>> 2.000000 30.000000 => 3319.205940 403.549846
>> 2.500000 90.000000 => 9997.964697 500.000000
>> 5.000000 30.000000 => 3320.469043 692.915108
>> 6.000000 35.000000 => 3877.156684 226.201899
[j3:~/proj/chizu/utm] upa005% 
*/
