#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "gmet/gmet.h"

typedef struct Op
{
  char *sym;
  int pri;                      // priority
} Op;

static Op opCL = { "(", 7 };
static Op opCR = { ")", 0 };
static Op opPlus = { "+", 3 };
static Op opMinus = { "-", 3 };
static Op opMul = { "*", 5 };
static Op opDiv = { "/", 5 };
static Op opLt = { ">", 1 };
static Op opRt = { "<", 1 };
static Op opAnd = { "&&", 1 };
static Op opOr = { "||", 1 };
static Op opEq = { "==", 1 };
static Op opNeq = { "!=", 1 };
static Op opRe = { ">=", 1 };
static Op opLe = { "<=", 1 };

typedef struct Token
{
  int type;
  union
  {
    int num;
    Op *op;
  } data;
} Token;

enum TokenType
{
  typeNum,
  typeOp
};

static int
isName( char c )
{
  if ( ( c >= 'a' && c <= 'z' ) || ( c >= 'A' && c <= 'Z' ) || c == '_' )
    return 1;
  return 0;
}

static int
isNum( char c )
{
  if ( ( c >= '1' && c <= '9' ) || c == '0' )
  {
    return 1;
  }
  return 0;
}

static int
isOp( char c )
{
  switch ( c )
  {
  case '+':
  case '-':
  case '*':
  case '/':
  case '&':
  case '=':
  case '<':
  case '>':
  case '!':
  case '|':
    return 1;
  default:
    break;
  }
  return 0;
}

static Op *
getOp( char *exp, int size, int *ret )
{
  *ret = 0;
  switch ( *exp )
  {
  case '!':
    *ret = 2;
    return &opNeq;
  case '+':
    *ret = 1;
    return &opPlus;
  case '-':
    *ret = 1;
    return &opMinus;
  case '*':
    *ret = 1;
    return &opMul;
  case '/':
    *ret = 1;
    return &opDiv;
  case '>':
  case '<':
    {
      if ( size > 1 && *( exp + 1 ) == '=' )
      {
        *ret = 2;
        if ( *exp == '>' )
          return &opLe;
        else
          return &opRe;
      }
      *ret = 1;
      if ( *exp == '>' )
        return &opLt;
      else
        return &opRt;
    }
  case '=':
  case '&':
  case '|':
    {
      if ( size > 1 && *exp == *( exp + 1 ) )
      {
        *ret = 2;
        switch ( *exp )
        {
        case '=':
          return &opEq;
        case '&':
          return &opAnd;
        case '|':
          return &opOr;
        }
      }
      *ret = 0;
    }
  default:
    break;
  }
  return NULL;
}

static int
getNum( char *exp, int size, int *ofs )
{
  int ret = 0;
  *ofs = 0;

  while ( *( exp + *ofs ) != 0 && *ofs < size && isNum( *( exp + *ofs ) ) )
  {
    ret *= 10;
    ret += ( *( exp + *ofs ) - '0' );
    ( *ofs )++;
  }

  return ret;
}

static int
getNameSize( char *name, int size )
{
  int ret = 0;

  while ( ret < size && isName( *( name + ret ) ) )
    ret++;

  return ret;
}

static int gmetExpGet_( GmetData * parent, char *exp, int size );

static int
getRef( GmetData * parent, char *exp, int size, int *ofs )
{
  GmetVector *ref = gmetNewVector( 128 );
  int ret = 0;
  *ofs = 0;

  while ( *( exp + *ofs ) && *ofs < size )
  {
    GmetArray *ar = ( GmetArray * ) malloc( sizeof( GmetArray ) );
    ar->name = exp + *ofs;
    ar->size = getNameSize( ar->name, size - *ofs );
    ar->idx = 0;
    ( *ofs ) += ar->size;
    if ( ( *ofs < size ) && *( exp + *ofs ) == '[' )
    {
      ( *ofs )++;
      // search corresponding ']'
      int i = 0, j = 1;
      char *next = exp + *ofs;
      for ( ; *ofs + i < size && *next; i++, next++ )
      {
        if ( *next == '[' )
          j++;
        else if ( *next == ']' )
        {
          j--;
          if ( j == 0 )
            break;
        }
      }
      ar->idx = gmetExpGet_( parent, exp + *ofs, i );
      if ( ar->idx < 0 )
        return -1;
      ( *ofs ) += i + 1;
    }
    if ( gmetDebug )
    {
      char tmp[32];
      memset( tmp, 0, 32 );
      memcpy( tmp, ar->name, ( ar->size < 32 ? ar->size : 31 ) );
      fprintf( stderr, "getRef: name = \"%s\" idx = %d\n", tmp, ar->idx );
    }
    gmetVectorPush( ref, ar );
    if ( ( *ofs + 2 > size ) || strncmp( exp + *ofs, "->", 2 ) )
      break;
    *ofs += 2;
  }

  ret = gmetRefGet( parent, ref );

  {
    GmetArray *ar;
    while ( ar = gmetVectorPop( ref ) )
      free( ar );
    gmetDeleteVector( ref );
  }

  return ret;
}

static GmetVector *
char2tokens( GmetData * parent, char *exp, int size )
{
  GmetVector *ret = gmetNewVector( 16 );        // tokens vector
  int ofs = 0;
  int cnt = 0;

  if ( gmetDebug )
    fprintf( stderr, "char2tokens: \"%s\" size=%d\n", exp, size );

  while ( ofs < size )
  {
    Token *token = ( Token * ) malloc( sizeof( Token ) );
    int pas;

    if ( isName( *( exp + ofs ) ) )
    {
      if ( gmetDebug )
        fprintf( stderr, "char2tokens: isName\n" );
      token->type = typeNum;
      token->data.num = getRef( parent, exp + ofs, size - ofs, &pas );
    }
    else if ( isOp( *( exp + ofs ) ) )
    {
      if ( gmetDebug )
        fprintf( stderr, "char2tokens: isOp\n" );
      token->type = typeOp;
      token->data.op = getOp( exp + ofs, size - ofs, &pas );
    }
    else if ( isNum( *( exp + ofs ) ) )
    {
      if ( gmetDebug )
        fprintf( stderr, "char2tokens: isNum" );
      token->type = typeNum;
      token->data.num = getNum( exp + ofs, size - ofs, &pas );
      if ( gmetDebug )
        fprintf( stderr, " = %d\n", token->data.num );
    }
    else if ( isspace( *( exp + ofs ) ) )
    {
      ofs++;
      continue;
    }
    else
      fprintf( stderr, "char2tokens: illegal string: %c\n", *( exp + ofs ) );

    ofs += pas;
    gmetVectorPush( ret, token );
  }

  return ret;
}

static int
culcOp2( Op * op, int a, int b )
{
  switch ( op->sym[0] )
  {
  case '+':
    return a + b;
  case '-':
    return a - b;
  case '*':
    return a * b;
  case '/':
    return a / b;
  case '=':
    return ( a == b );
  case '!':
    return ( a != b );
  case '&':
    return ( a && b );
  case '|':
    return ( a || b );
  case '>':
    if ( strlen( op->sym ) == 1 )
      return ( a > b );
    else
      return ( a >= b );
  case '<':
    if ( strlen( op->sym ) == 1 )
      return ( a < b );
    else
      return ( a <= b );
  default:
    return 0;
  }
  return 0;
}

static int
culcExpTokens( GmetVector * v )
{
  GmetVector *opStack = gmetNewVector( 128 );
  GmetVector *numStack = gmetNewVector( 128 );
  Token *cur;
  int i, max = gmetVectorSize( v );
  int ret = -1;

  if ( gmetDebug )
    fprintf( stderr, "culcExpTokens: start\n" );

  for ( i = 0; i < max; i++ )
  {
    cur = gmetVectorGet( v, i );
    switch ( cur->type )
    {
    case typeOp:
      gmetVectorPush( opStack, cur->data.op );
      break;
    case typeNum:
      {
        gmetVectorPush( numStack, ( void * ) cur->data.num );
        while ( gmetVectorSize( opStack ) )
        {
          if ( i + 1 != max )
          {                     // next reading
            Token *next = gmetVectorGet( v, i + 1 );
            Op *prev = gmetVectorGet( opStack, -1 );

            if ( !next || next->type != typeOp )
            {
              gmetDeleteVector( numStack );
              gmetDeleteVector( opStack );
              return -1;
            }
            if ( prev == &opCL && next->data.op == &opCR )
            {
              gmetVectorPop( opStack );
              gmetVectorPop( opStack );
              continue;
            }
            if ( prev->pri < next->data.op->pri )
              break;
          }
          {
            Op *op = gmetVectorPop( opStack );
            int b = ( int ) gmetVectorPop( numStack );
            int a = ( int ) gmetVectorPop( numStack );
            int ret = culcOp2( op, a, b );
            gmetVectorPush( numStack, ( void * ) ret );
          }
        }
        break;
      }
    default:
      break;
    }
  }

  if ( gmetVectorSize( opStack )
       || gmetVectorSize( numStack ) != 1 || i != max )
  {
    return -1;
  }

  ret = ( int ) gmetVectorGet( numStack, 0 );
  gmetDeleteVector( numStack );
  gmetDeleteVector( opStack );
  return ret;
}

static int
gmetExpGet_( GmetData * parent, char *exp, int size )
{
  GmetVector *v = char2tokens( parent, exp, size );
  int ret = culcExpTokens( v );
  Token *tmp;

  if ( gmetDebug )
    fprintf( stderr, "gmetExpGet_: size = %d\n", size );

  while ( tmp = gmetVectorPop( v ) )
    free( tmp );
  gmetDeleteVector( v );

  return ret;
}

int
gmetExpGet( GmetData * parent, char *exp )
{
  int ret;

  if ( gmetDebug )
    fprintf( stderr, "gmetExpGet: start expression \"%s\"\n", exp );
  ret = gmetExpGet_( parent, exp, strlen( exp ) );

  if ( gmetDebug )
    fprintf( stderr, "gmetExpGet: return %d\n", ret );

  return ret;
}
