/* $Id: gnodempa.c,v 1.2 2002/10/25 09:20:54 iiojun Exp $ */

/* GNODEMPA
 * Copyright (C) 1999-2002 Kazuo HIYANE, Jun IIO, and Tomoyuki YATABE
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <gnome.h>
#include <getopt.h>
#include <malib/malib.h>
#include "intl.h"
#include "face_region.h"
#include "background.h"
#include "alphablend.h"
#include "range2d.h"

#define DEFAULT_A_MEAN 	"/gnodempa/color_params/a_mean=0.5"
#define DEFAULT_B_MEAN 	"/gnodempa/color_params/b_mean=0.5"
#define DEFAULT_A_COEF 	"/gnodempa/color_params/a_coef=0.05"
#define DEFAULT_B_COEF 	"/gnodempa/color_params/b_coef=0.05"

/* mode definition */
enum {
  RUN_MODE, CONFIG_MODE, MODE_NUM
};

/* static variable(s) (internal use only) */
static unsigned char* bg_image = PKGDATADIR"/bgimage.jpg";

/* signal setup */
void
sigint_handler (int sig)
{
  g_warning (_("SIGINT caught"));

  exit (-1);
}

/* filter setup */
void
read_filter_settings_from_file (FaceRegion* face, GtkRange2d* range2d)
{
  double a_mean, b_mean;
  double a_coef, b_coef;
  
  a_mean = gnome_config_get_float (DEFAULT_A_MEAN);
  b_mean = gnome_config_get_float (DEFAULT_B_MEAN);
  a_coef = gnome_config_get_float (DEFAULT_A_COEF);
  b_coef = gnome_config_get_float (DEFAULT_B_COEF);

  face_region_set_color_params (face, a_mean, b_mean, a_coef, b_coef);

  if (range2d != NULL)
    {
      gtk_range2d_set_value (range2d, a_mean, b_mean, a_coef, b_coef);
    }
}

/* propagate changed values */

void
change_value (GtkRange2d* range2d, FaceRegion* face)
{
  double a_mean, b_mean;
  double a_coef, b_coef;
  gtk_range2d_get_value (range2d, &a_mean, &b_mean, &a_coef, &b_coef);
  face_region_set_color_params (face, a_mean, b_mean, a_coef, b_coef);
}

/* setup the configure color range widget (CONFIG_MODE) */
GtkRange2d*
color_config_show_window (FaceRegion* face)
{
  GtkWidget* range2d = gtk_range2d_new ();
  gtk_widget_show (range2d);
  gtk_signal_connect (GTK_OBJECT (range2d), "value_changed", 
		      GTK_SIGNAL_FUNC (change_value), face);
  return (GtkRange2d*) range2d;
}

/* show help message and exit program */
void
show_usage_and_exit ()
{
  fprintf(stderr, "Usage - gnodempa [options]\n");
  fprintf(stderr, "  -b filename, --bg filename       set bgimages\n");
  fprintf(stderr, "  -c, --config                     configuration mode\n");
  fprintf(stderr, "  -h, -?, --help                   show this help\n");

  exit (-1);
}

/* check what mode this command running */
int
check_option (int argc, char** argv)
{
  int mode = RUN_MODE;

  int c;
  int this_option_optind = optind ? optind : 1;
  int option_index = 0;
  static struct option long_options[] =
  {
    {"bg", 0, 0, 'b'},
    {"config", 1, 0, 'c'},
    {"help", 0, 0, 'h'},
    {0, 0, 0, 0}
  };

  while (1)
    {
      c = getopt_long (argc, argv, "b:ch", long_options, &option_index);
      if (c == -1) break;
      
      switch (c)
	{
	case 'b':	  bg_image = optarg;	  break;
	case 'c':	  mode = CONFIG_MODE;	  break;
	case 'h':
	case '?':	  show_usage_and_exit (); break;
	default:
	  fprintf (stderr, "?? getopt returned character code 0%o ??\n", c);
	}
      
      if (optind < argc)
	{
	  show_usage_and_exit ();
	}
    }

  return mode;
}

/* create the process link */
void
create_link (MalibSource** src_p, FaceRegion** face_p, 
	     MalibGtkDisplay** disp_p, int mode)
{
  MalibSource* src;
  MalibGtkDisplay* disp;
  MalibSource* fil1, * fil2, * fil3;
  MalibBuffer* buf1, * buf2, * buf3;

  src = (MalibSource*) malib_bttv_new ();
  buf1 = (MalibBuffer*) malib_plainbuf_new_with_source (src);
  fil1 = (MalibSource*) face_region_new_with_buf (buf1);
  buf2 = (MalibBuffer*) malib_plainbuf_new_with_source (fil1);

  if (mode == CONFIG_MODE)
    {
      fil3 = (MalibSource*) malib_negative_new_with_buf (buf1);
      //  fil3 = (MalibSource*) 
      //    malib_intensity_new_with_buf (MALIB_INTENSITY_DARKER, buf1);
      buf3 = (MalibBuffer*) malib_plainbuf_new_with_source (fil3);
      
      fil2 = (MalibSource*) alphablend_new_with_bufs (buf3, buf2);
    }
  else /* mode == RUN_MODE */
    {
      fil2 = (MalibSource*) background_new_with_image_and_buf (bg_image, buf2);
    }

  disp = malib_gtkdisplay_new_with_source (fil2);

  /* store check points */
  *src_p = src;
  *face_p = (FaceRegion*) fil1;
  *disp_p = disp;
}

/* main routine */
int
main (int argc, char** argv)
{
  MalibGtkDisplay* disp;
  MalibSource* src;
  FaceRegion* face;
  GtkRange2d* range2d = NULL;
  int mode;

  /* gettext setup */
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  /* signal setup */
  signal (SIGINT, sigint_handler);

  /* malib setup */
  malib_init (&argc, &argv);

  /* check command line options */
  mode = check_option (argc, argv);

  /* create the process link */
  create_link (&src, &face, &disp, mode);

  /* config window setup (CONFIG_MODE) */
  if (mode == CONFIG_MODE)
    {
      range2d = color_config_show_window (face);
    }

  /* filter configuration */
  read_filter_settings_from_file (face, range2d);

  /* start capture & display */
  malib_bttv_start_capture ((MalibBttv*) src);
  malib_gtkdisplay_auto_play (disp);

  return 0;
}
