#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::Mail;
use Lib::App::Conf;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $user = Lib::User->new();
my $conf = Lib::App::Conf->new();

sub act() {
    my ($act, $cgi) = @_;
    my ($operation, $object) = split(/_/, $act);

    my $sessionid = $cgi->cookie('cookie59t');
    if ($sessionid eq '') {
        $error->print_error('00003e');
    }
    my $session_userid = $user->check_session($sessionid);
    if ($session_userid eq '') {
        $error->print_error('00003e');
    }

    if ($act eq 'input_user') {
        _input_user($operation);
    }
    elsif ($act eq 'add_user') {
        _add_user($operation, $cgi);
    }
    elsif ($act eq 'edit_user') {
        _edit_user($operation, $session_userid, $cgi);
    }
    elsif ($act eq 'update_user') {
        _update_user($operation, $session_userid, $cgi);
    }
    elsif ($act eq 'confirm_user') {
        _confirm_user();
    }
    elsif ($act eq 'leave_user') {
        _leave_user($session_userid, $cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

sub _input_user() {
    my ($operation) = @_;

    my $userdata;
    $userdata->{userid}   = '';
    $userdata->{password} = '';
    $userdata->{mailaddr} = '';
    $userdata->{hpurl}    = '';
    $userdata->{username} = '';
    $userdata->{auth}     = 1;
    $userdata->{status}   = 1;
    $userdata->{ad1}      = '';
    $userdata->{ad2}      = '';
    $userdata->{ad3}      = '';
    $userdata->{parts}    = '';

    # 入力画面の表示
    _show_page($operation, $userdata, '', '');
}

sub _add_user() {
    my ($operation, $cgi) = @_;

    # パラメーターの取得
    my $userdata;
    $userdata->{userid}   = $cgi->param('userid');
    $userdata->{password} = $cgi->param('password');
    $userdata->{mailaddr} = $cgi->param('mailaddr');
    $userdata->{hpurl}    = $cgi->param('hpurl');
    $userdata->{username} = $cgi->param('username');
    $userdata->{auth}     = $cgi->param('auth');
    $userdata->{status}   = $cgi->param('status');
    $userdata->{ad1}      = $cgi->param('ad1');
    $userdata->{ad2}      = $cgi->param('ad2');
    $userdata->{ad3}      = $cgi->param('ad3');
    $userdata->{parts}    = $cgi->param('parts');

    # 入力チェック
    _check_input($operation, $userdata, '');

    # ユーザーデータを追加
    $user->set_userid($userdata->{userid});
    $user->set_password($userdata->{password});
    $user->set_mailaddr($userdata->{mailaddr});
    $user->set_hpurl($userdata->{hpurl});
    $user->set_username($userdata->{username});
    $user->set_auth($userdata->{auth});
    $user->set_status($userdata->{status});
    $user->set_ad1($userdata->{ad1});
    $user->set_ad2($userdata->{ad2});
    $user->set_ad3($userdata->{ad3});
    $user->set_parts($userdata->{parts});
    $user->add();

    my $nextpage = './?act=view_userlist';
    print "Location: $nextpage", "\n\n";
}

sub _edit_user() {
    my ($operation, $session_userid, $cgi) = @_;

    # 既存データの読み込み
    my $userid = $cgi->param('userid');
    $user->load($userid);

    my $userdata;
    $userdata->{userid}   = $userid;
    $userdata->{password} = '********';
    $userdata->{mailaddr} = $user->get_mailaddr();
    $userdata->{hpurl}    = $user->get_hpurl();
    $userdata->{username} = $user->get_username();
    $userdata->{auth}     = $user->get_auth();
    $userdata->{status}   = $user->get_status();
    $userdata->{ad1}      = $user->get_ad1();
    $userdata->{ad2}      = $user->get_ad2();
    $userdata->{ad3}      = $user->get_ad3();
    $userdata->{parts}    = $user->get_parts();

    # 管理者以外は権限、ステータスは変更不可
    my $disabled = 'disabled';
    if ($user->is_admin($session_userid)) {
         $disabled = '';
    }

    # 編集画面の表示
    _show_page($operation, $userdata, $disabled, '');
}

sub _update_user() {
    my ($operation, $session_userid, $cgi) = @_;

    # パラメーターの取得
    my $userdata;
    $userdata->{userid}   = $cgi->param('userid');
    $userdata->{password} = $cgi->param('password');
    $userdata->{mailaddr} = $cgi->param('mailaddr');
    $userdata->{hpurl}    = $cgi->param('hpurl');
    $userdata->{username} = $cgi->param('username');
    $userdata->{auth}     = $cgi->param('auth');
    $userdata->{status}   = $cgi->param('status');
    $userdata->{ad1}      = $cgi->param('ad1');
    $userdata->{ad2}      = $cgi->param('ad2');
    $userdata->{ad3}      = $cgi->param('ad3');
    $userdata->{parts}    = $cgi->param('parts');

    # 管理者以外は権限、ステータスは変更不可
    my $nextpage = './';
    my $disabled = 'disabled';
    if ($user->is_admin($session_userid)) {
         $nextpage .= '?act=view_userlist';
         $disabled = '';
    }

    # 入力チェック
    _check_input($operation, $userdata, $disabled);

    # 既存データの読み込み
    $user->load($userdata->{userid});

    # ユーザーデータを更新
    $user->set_mailaddr($userdata->{mailaddr});
    $user->set_hpurl($userdata->{hpurl});
    $user->set_username($userdata->{username});
    if ($disabled eq '') {
        $user->set_auth($userdata->{auth});
        $user->set_status($userdata->{status});
    }
    $user->set_ad1($userdata->{ad1});
    $user->set_ad2($userdata->{ad2});
    $user->set_ad3($userdata->{ad3});
    $user->set_parts($userdata->{parts});
    $user->update();

    print "Location: $nextpage", "\n\n";
}

sub _check_input() {
    my ($operation, $userdata, $disabled) = @_;

    # ユーザーID
    if ($userdata->{userid} eq '') {
        _show_page($operation, $userdata, $disabled, 'ユーザーIDが未入力です');
    }
    # ユーザーIDが英数字かチェック
    if ($userdata->{userid} =~ /[^0-9a-zA-Z]/) {
        _show_page($operation, $userdata, $disabled, 'ユーザーIDは半角英数字で入力してください');
    }

    if ($operation eq 'add') {
        # ユーザーIDの重複チェック
        if (!$user->check_valid($userdata->{userid})) {
            _show_page($operation, $userdata, $disabled, '指定のユーザーIDは既に登録されています');
        }
        # パスワード
        if ($userdata->{password} eq '') {
            _show_page($operation, $userdata, $disabled, 'パスワードが未入力です');
        }
        # パスワードの長さをチェック
        my $strlen = length($userdata->{password});
        if ($strlen < 4 || $strlen > 9) {
            _show_page($operation, $userdata, $disabled, 'パスワードは半角英数字4文字～9文字で入力してください');
        }
        # パスワードが英数字かチェック
        if ($userdata->{password} =~ /[^0-9a-zA-Z]/) {
            _show_page($operation, $userdata, $disabled, 'パスワードは半角英数字で入力してください');
        }
    }

    # メールアドレス
    if ($userdata->{mailaddr} eq '') {
        _show_page($operation, $userdata, $disabled, 'メールアドレスが未入力です');
    }
    # メールアドレス形式が正しいかチェック
    if ($userdata->{mailaddr} !~ /[-_0-9a-z]+\@[-_0-9a-z]+\.[-_0-9a-z.]+/) {
        _show_page($operation, $userdata, $disabled, 'メールアドレスに誤りがあります');
    }

    # 名前
    if ($userdata->{username} eq '') {
        _show_page($operation, $userdata, $disabled, '名前が未入力です');
    }

    if ($disabled eq '') {
        # 権限
        if ($userdata->{auth} eq '') {
            _show_page($operation, $userdata, $disabled, '権限が選択されていません');
        }

        # ステータス
        if ($userdata->{status} eq '') {
            _show_page($operation, $userdata, $disabled, 'ステータスが選択されていません');
        }
    }
}

# 入力画面表示
sub _show_page() {
    my ($operation, $userdata, $disabled, $msg) = @_;

    $msg = "<p><strong><font class=\"message\" size=\"+1\">$msg</font></strong></p>";

    my $caption = 'ユーザーの追加';
    my $action = 'add_user';
    my $readonly = '';
    my $button = '追加';
    if (($operation eq 'edit') || ($operation eq 'update')) {
        $caption = 'ユーザー情報の編集';
        $action = 'update_user';
        $readonly = 'readonly';
        $button = '更新';
    }

    my $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
$caption
</strong></p>
END_PATH
    if ($disabled eq '') {
        $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;</font>&nbsp;
<a href="./?act=view_userlist" class="path">ユーザー管理</a>&nbsp;
&gt;&nbsp;
$caption
</strong></p>
END_PATH
    }

    # 権限
    my $authlist = '';
    if ($userdata->{auth} == 0) {
        $authlist = <<"END_AUTH";
<select name="auth" $disabled>
<option selected value="0">管理者
<option value="1">編集者
</select>
END_AUTH
    }
    else {
        $authlist = <<"END_AUTH";
<select name="auth" $disabled>
<option value="0">管理者
<option selected value="1">編集者
</select>
END_AUTH
    }

    # 状態
    my $statuslist = '';
    if ($userdata->{status} == 1) {
        $statuslist = <<"END_STATUS";
<select name="status" $disabled>
<option selected value="1">有効
<option value="0">無効
</select>
END_STATUS
    }
    else {
        $statuslist = <<"END_STATUS";
<select name="status" $disabled>
<option value="1">有効
<option selected value="0">無効
</select>
END_STATUS
    }

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/user.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$ACT\$/$action/g;
    $template =~ s/\$READONLY\$/$readonly/g;
    $template =~ s/\$BUTTON\$/$button/g;
    $template =~ s/\$USERID\$/$userdata->{userid}/g;
    $template =~ s/\$PASSWORD\$/$userdata->{password}/g;
    $template =~ s/\$MAILADDR\$/$userdata->{mailaddr}/g;
    $template =~ s/\$HPURL\$/$userdata->{hpurl}/g;
    $template =~ s/\$USERNAME\$/$userdata->{username}/g;
    $template =~ s/\$AUTH\$/$authlist/g;
    $template =~ s/\$STATUS\$/$statuslist/g;
    $template =~ s/\$AD1\$/$userdata->{ad1}/g;
    $template =~ s/\$AD2\$/$userdata->{ad2}/g;
    $template =~ s/\$AD3\$/$userdata->{ad3}/g;
    $template =~ s/\$PARTS\$/$userdata->{parts}/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

# 退会確認画面
sub _confirm_user() {
    _show_confirm('');
}

# ユーザーアカウントを無効化
sub _leave_user() {
    my ($session_userid, $cgi) = @_;

    # ユーザー情報を取得
    $user->load($session_userid);
    my $mailaddr = $user->get_mailaddr();

    # ステータスを無効にする
    $user->set_status(0);
    $user->update();

    # セッションを削除
    my $sessionid = $cgi->cookie('cookie59t');
    $user->logout($sessionid);

    my $mail = Lib::Mail->new();

    my $sitetitle = $conf->get_site_title();
    my $adminaddr = $conf->get_admin_address();

    # ユーザーに退会通知メールを送信
    my $subject = "退会のおしらせ";
    my $body = <<"END_BODY";
$sitetitle から退会しました。

もし、このメールに心当たりがない場合は、お手数ですが以下の
メールアドレス宛にその旨をご連絡ください。
mailto:$adminaddr
END_BODY

    $mail->send($subject, $body, $mailaddr);

    # 管理人にメールを送信
    $subject = "$session_userid さんが退会しました";
    $body = <<"END_BODY";
$session_userid さんが退会しました。

メールアドレス：$mailaddr
END_BODY

    $mail->send($subject, $body, $conf->get_admin_address());

    my $title = "$sitetitle から退会しました";

    # メッセージを表示
    my $msg = '';
    $msg = "<p><strong><font class=\"message\">$msg</font></strong></p>";
    my $pageexit = "<strong><a href=\"./\">ホーム</a></strong>";

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/message.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$TITLE\$/$title/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$PAGEEXIT\$/$pageexit/g;

    print "Content-Type: text/html\n\n";
    print $template;
}

# 退会確認画面の表示
sub _show_confirm() {
    my ($msg) = @_;

    $msg = "<p><strong><font class=\"message\" size=\"+1\">$msg</font></strong></p>";

    my $caption = '退会確認';
    my $sitetitle = $conf->get_site_title();

    my $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    my $content = <<"END_CONTENT";
<p>
<strong>$sitetitle から退会しますか？</strong><br />
<br />
退会する場合は、退会ボタンを押下してください。<br />
なお、同じユーザーIDでの再登録はできませんのでご注意ください。
</p>
END_CONTENT

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/leave.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$CONTENT\$/$content/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

1;
