/*
 * Grain Core - A XForms processor for mobile terminals.
 * Copyright (C) 2005-2006 HAW International Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Created on 2006/01/14 10:50:00
 * 
 */
package jp.grain.sprout.ui;

import java.io.IOException;
import java.util.Hashtable;
import java.util.Stack;
import java.util.Vector;

import jp.grain.spike.BinaryXMLParser;
import jp.grain.spike.CDataNode;
import jp.grain.spike.Document;
import jp.grain.spike.Element;
import jp.grain.spike.Node;
import jp.grain.spike.event.EventListener;
import jp.grain.spike.xpath.XPathEvaluator;
import jp.grain.spike.xpath.XPathExpr;
import jp.grain.sprout.model.Instance;
import jp.grain.sprout.model.InstanceElement;
import jp.grain.sprout.model.Model;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

/**
 * tH[hLg𐶐Builder
 * 
 * @version $Id: FormBuilder.java 392 2006-06-29 06:24:23Z nakajo $
 * @author Go Takahashi
 */
public class FormBuilder {

	public static final String XML_EVENTS = "http://www.w3.org/2001/xml-events";
	public static final String XHTML = "http://www.w3.org/1999/xhtml";
	public static final String GUD = "http://grain.jp/gud/";
    public static final String XPATH = "http://grain.jp/xpath/";

    private XmlPullParser parser;
	private Document form;
	private static Hashtable componentMap = new Hashtable();
	// private static Hashtable modelMap = new Hashtable();
	private XPathExpr locator;
	private Stack stack = new Stack();
	private String _url;
	private boolean _gud;
	private Element _currentElem;
	private boolean _closurePush;
	private Vector _closurePrams;
	
	/**
	 * @param class1
	 */
	public static void registerComponent(String name, Class clazz) {
		componentMap.put(name, clazz);
	}

	public static void init() {

		if (!componentMap.isEmpty()) return;
		try {
			Form.init();
			Model.init();
			Instance.init();

			componentMap.put("submission", Class.forName("jp.grain.sprout.model.Submission"));
			componentMap.put("model", Class.forName("jp.grain.sprout.model.Model"));
			componentMap.put("instance", Class.forName("jp.grain.sprout.model.Instance"));
			componentMap.put("bind", Class.forName("jp.grain.sprout.model.Bind"));
			
			componentMap.put("view", Class.forName("jp.grain.sprout.ui.Block"));
			componentMap.put("button", Class.forName("jp.grain.sprout.ui.Button"));
			componentMap.put("upload", Class.forName("jp.grain.sprout.ui.Upload"));
			componentMap.put("image", Class.forName("jp.grain.sprout.platform.doja.ui.Image"));
			componentMap.put("textbox", Class.forName("jp.grain.sprout.ui.TextBox"));
			componentMap.put("block", Class.forName("jp.grain.sprout.ui.Block"));
			componentMap.put("inline-block", Class.forName("jp.grain.sprout.ui.InlineBlock"));
			componentMap.put("inline-group", Class.forName("jp.grain.sprout.ui.InlineGroup"));
			componentMap.put("inline", Class.forName("jp.grain.sprout.ui.CharactorSequence"));
			componentMap.put("label", Class.forName("jp.grain.sprout.ui.CharactorSequence"));
			componentMap.put("choice", Class.forName("jp.grain.sprout.ui.Choice"));
			componentMap.put("select", Class.forName("jp.grain.sprout.ui.Select"));
			componentMap.put("tab-block", Class.forName("jp.grain.sprout.ui.TabBlock"));
			componentMap.put("item", Class.forName("jp.grain.sprout.ui.Item"));
			componentMap.put("choices", Class.forName("jp.grain.sprout.ui.Choices"));
			componentMap.put("itemset", Class.forName("jp.grain.sprout.ui.Itemset"));
			componentMap.put("repeat", Class.forName("jp.grain.sprout.ui.Repeat"));

			componentMap.put("action", Class.forName("jp.grain.sprout.action.Action"));

			// componentMap.put("action",
			// Class.forName("jp.grain.sprout.ui.Inline"));
			// componentMap.put("send",
			// Class.forName("jp.grain.sprout.ui.ActionElement"));
		} catch (ClassNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

	}

	/**
	 * @param parser
	 *            tH[hLg̐ɗpparser
	 * @param uri
	 *            tH[hLgʂURI
	 */
	public FormBuilder(XmlPullParser parser, String url, boolean gud) {
		this.parser = parser;
		_url = url;
		_gud = gud;
	}

	/**
	 * ꂽtH[hLg擾B {Ibuild\bȟĂяoɗpB
	 * 
	 * @return ꂽtH[hLgBbuild\bhĂяoĂȂꍇrootElementnullB
	 */
	public Document getDocument() {
		if (this.form != null && this.form instanceof Form) {
			((Form) this.form).setBaseUrl(_url);
		}
		return this.form;
	}

	/**
	 * tH[hLg𐶐B
	 * 
	 * @throws IOException
	 *             tH[hLg̐ɓo̓G[ꍇB
	 * @throws XmlPullParserException
	 *             XML̃p[Xɍ\G[ȂǂꍇB
	 */
	public void build() throws IOException, XmlPullParserException {
		for (int et = this.parser.getEventType(); et != BinaryXMLParser.END_DOCUMENT; et = this.parser.next()) {
			if (_gud) {
				if (et == BinaryXMLParser.START_DOCUMENT) continue;

				String namespace = this.parser.getNamespace();
                System.out.println("%%%% namespace = " + namespace);
				if (XPATH.equals(namespace)) {
					processXPath();
				} else {
					processGud();
				}
			} else {
				processNode();
			}
		}
		if (!this.stack.empty()) throw new XmlPullParserException("Illegal xml data");
	}

	/**
	 * @throws XmlPullParserException
	 * @throws IOException
	 * 
	 */
	private void processNode() throws XmlPullParserException, IOException {
		int et = this.parser.getEventType();
		System.out.println("processNode : et = " + et);
		switch (et) {
		case BinaryXMLParser.START_DOCUMENT:
			this.form = new Document();
			break;
		case BinaryXMLParser.START_TAG:
			String elemName = this.parser.getName();
			String elemPrefix = this.parser.getPrefix();
			String elemNs = this.parser.getNamespace();
			Element element = new InstanceElement(elemName, elemPrefix, elemNs);
			for (int i = 0; i < this.parser.getAttributeCount(); ++i) {
				String attrName = this.parser.getAttributeName(i);
				String attrValue = this.parser.getAttributeValue(i);
				String attrPrefix = this.parser.getAttributePrefix(i);
				String attrNs = this.parser.getAttributeNamespace(i);
				element.addAttribute(attrName, attrPrefix, attrNs, attrValue);
			}
			_currentElem.addChild(element);
			_currentElem = element;
			break;
		case BinaryXMLParser.END_TAG:
			_currentElem = _currentElem.getParentElement();
			break;
		case BinaryXMLParser.TEXT:
			_currentElem.setText(this.parser.getText());
			break;
		case BinaryXMLParser.CDSECT:
			CDataNode cdata = new CDataNode();
			cdata.setData(this.parser.getCData());
			_currentElem.addChild(cdata);
		}
	}

	/**
	 * @throws XmlPullParserException
	 * 
	 */
	private void processXPath() throws XmlPullParserException {
		int et = this.parser.getEventType();
        System.out.println("%%%%%%%%%% process XPath event = " + et);
		switch (et) {
		case BinaryXMLParser.START_TAG:
			String name = this.parser.getName();
            System.out.println("% start tag = " + name);
			if (name.equals(XPathExpr.NAME)) {
				this.locator = new XPathExpr();
				for (int i = 0; i < this.parser.getAttributeCount(); ++i) {
					String attrName = this.parser.getAttributeName(i);
					String attrValue = this.parser.getAttributeValue(i);
					String attrPrefix = this.parser.getAttributePrefix(i);
					String attrNs = this.parser.getAttributeNamespace(i);
					this.locator.addAttribute(attrName, attrPrefix, attrNs, attrValue);
				}
			} else if(name.equals(XPathEvaluator.OP_CLSPUSH)) {
				this._closurePush = true;
				this._closurePrams = new Vector();
			} else {
				Object[] op = createOperand();
				if(this._closurePush) {
					this._closurePrams.addElement(op);
				} else {
					if (op != null) this.locator.addOperand(op);
				}
			}
			break;
		case BinaryXMLParser.END_TAG:
			name = this.parser.getName();
            System.out.println("% end tag = " + name);
			if (name.equals(XPathExpr.NAME)) {
                System.out.println("% peek = " + this.stack.peek());
                this.locator.postProcess((Node)this.stack.peek());
				//((BindingElement) this.stack.peek()).setBindingExpr(this.locator);
				this.locator = null;
			} else if(name.equals(XPathEvaluator.OP_CLSPUSH)) {
				Object[] params = new Object[this._closurePrams.size()];
				this._closurePrams.copyInto(params);
				this.locator.addPredicate(XPathExpr.createOperand("push", params));
				
				this._closurePush = false;
				this._closurePrams = null;
			}
			break;
		case BinaryXMLParser.TEXT:
			// if (this.locator == null) throw new XmlPullParserException("there
			// is no root element but text is found");
			// this.locator.setValue(this.parser.getText());
			break;
		}
	}

	private Object[] createOperand() throws XmlPullParserException {
		String name = this.parser.getName();
		if (name.equals("spush")) {
			return XPathExpr.createOperand("push", this.parser.getAttributeValue(0));
		} else if (name.startsWith("true")) {
            return XPathExpr.createOperand("push", new Boolean(true));
		} else if (name.startsWith("false")) {
            return XPathExpr.createOperand("push", new Boolean(false));
		} else if (name.startsWith("_")) {
            return XPathExpr.createOperand("push", name.substring(1));
		} else if (name.startsWith("N")) {
			return XPathExpr.createOperand("push", Integer.valueOf(name.substring(1)));
		} else {
			int paramNum = this.parser.getAttributeCount();
			switch (paramNum) {
			case 0:
                return XPathExpr.createOperand(name);
			case 1:
				return XPathExpr.createOperand(name, this.parser.getAttributeValue(0));
			case 2: {
				Object[] params = new Object[paramNum];
				for (int i = 0; i < paramNum; i++) {
					int index = Integer.parseInt(this.parser.getAttributeName(i).substring(1));
					String value = this.parser.getAttributeValue(i);
					params[index] = "".equals(value) ? null : value;
				}
                return XPathExpr.createOperand(name, params[0], params[1]);
			}
			case 3: {
				Object[] params = new Object[paramNum];
				for (int i = 0; i < paramNum; i++) {
					int index = Integer.parseInt(this.parser.getAttributeName(i).substring(1));
					params[index] = this.parser.getAttributeValue(i);
				}
                return XPathExpr.createOperand(name, params[0], params[1]);
			}
			}
		}
        return null;
	}

	void processGud() throws XmlPullParserException {
		int et = this.parser.getEventType();
		String name = this.parser.getName();
		switch (et) {
		case BinaryXMLParser.START_TAG:
			System.out.print("<" + name);
			Node gud = createGudElement(name);
			System.out.println(">");
			if (gud instanceof Form && this.form == null) {
				this.form = (Form) gud;
			} else if (gud instanceof Instance) {
				System.out.println("prefix mapping for instanceNode");
				Instance instance = (Instance) gud;
				for (int i = 0; i < this.parser.getNamespaceCount(this.parser.getDepth() + 1); i++) {
					String prefix = this.parser.getNamespacePrefix(i);
					String nsUri = this.parser.getNamespaceUri(i);
					instance.addAncesterPrefixMapping(prefix, nsUri);
					System.out.println("add prefix = " + prefix + ", nsUri" + nsUri);
				}
			}
			if (!this.stack.empty()) {
				gud.preProcess((Node) this.stack.peek());
			}
			this.stack.push(gud);
			break;
		case BinaryXMLParser.END_TAG:
			System.out.println("</" + name + ">");
			Node child = (Node) this.stack.pop();
			if (!this.stack.empty()) {
				child.postProcess((Node) this.stack.peek());
			}
			if(child instanceof EventListener && this.form != null) {
				((Form)this.form).addEventListener((EventListener)child);
			}
			break;
		case BinaryXMLParser.TEXT:
			System.out.println("*** text");
			if (this.stack.empty()) throw new XmlPullParserException("there is no root element but text is found");
			((Element) this.stack.peek()).setText(this.parser.getText());
			break;
		}
	}

	private Node createGudElement(String name) throws XmlPullParserException {
		Node box = null;
		try {
			if (GUD.equals(this.parser.getNamespace())) {
				Class clazz = (Class) componentMap.get(name);
				if (clazz == null) {
					box = new GudAttribute(name);
				} else {
					box = (Node) clazz.newInstance();
				}
			} else {
				String namespace = this.parser.getNamespace();
				String prefix = this.parser.getPrefix();
				box = new InstanceElement(name, prefix, namespace);
			}
			for (int i = 0; i < this.parser.getAttributeCount(); ++i) {
				String attrName = this.parser.getAttributeName(i);
				String attrPrefix = this.parser.getAttributePrefix(i);
				String attrNS = this.parser.getAttributeNamespace(i);
				String attrValue = this.parser.getAttributeValue(i);
				System.out.println(" " + attrName + "=\"" + attrValue + "\"");
				((Element) box).addAttribute(attrName, attrPrefix, attrNS, attrValue);
				// TODO check attribure type is xsd:id
				if (attrName.equals("id")) {
					((Form) this.form).registerNode(attrValue, box);
				}
			}
			return box;
		} catch (InstantiationException e) {
			throw new XmlPullParserException(e.toString());
		} catch (IllegalAccessException e) {
			throw new XmlPullParserException(e.toString());
		}
	}

}