/*
 * Grain Core - A XForms processor for mobile terminals.
 * Copyright (C) 2005 HAW International Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Created on 2005/06/16 14:11:18
 * 
 */
package jp.grain.spike.xpath;

import j2meunit.framework.Test;
import j2meunit.framework.TestCase;
import j2meunit.framework.TestMethod;
import j2meunit.framework.TestSuite;

import java.io.IOException;

import jp.grain.spike.DefaultElement;
import jp.grain.spike.Document;
import jp.grain.spike.Element;
import jp.grain.spike.Node;
//import jp.grain.spike.Text;

/**
 * BinaryXMLParserTest
 * @version $Id: BinaryXMLSerializerTest.java 242 2006-04-09 08:19:37Z go $
 * @author Go Takahashi
 */
public class XPathEvaluatorTest extends TestCase {
	
    private Document doc;
    
    public XPathEvaluatorTest() {
    }

    public XPathEvaluatorTest(String name, TestMethod method) {
        super(name, method);
        
        // <haw type="now" xforms:unit="yen">
        //   <xforms:output id="id1" ref="test">abc</xforms:output>
        //   <xforms:output ref="x" xmlns=""/>
        //   <xforms:model>
        //     <scf>
        //       <d id="id2">test</d>word
        //     </scf>
        //     5
        //   </xforms:model>
        //   1
        // </haw>

        this.doc = new Document();
        Element haw = new DefaultElement("haw", "", "http://haw.co.jp/haw");
        haw.addPrefixMapping("xforms","http://www.w3.org/2002/xforms");
        haw.addPrefixMapping(null,"http://haw.co.jp/haw");
        haw.addAttribute("type", "now");
        haw.addAttribute("unit", "xforms", "http://www.w3.org/2002/xforms", "yen");
        this.doc.setRootElement(haw);

        Element output=new DefaultElement("output", "xforms", "http://www.w3.org/2002/xforms");
        output.addAttribute("id","id1");
        output.addAttribute("ref","test");
        haw.addChild(output);
        
        output.addChildText("abc");
               
        Element output2 = new DefaultElement("output", "xforms", "http://www.w3.org/2002/xforms");
        output2.addAttribute("ref", "x");
        haw.addChild(output2);
            
        Element model = new DefaultElement("model", "xforms", "http://www.w3.org/2002/xforms");
        model.addPrefixMapping(null, "");
        haw.addChild(model);
        haw.addChildText("1");
        
        Element scf = new DefaultElement("scf");
        model.addChild(scf);
        model.addChildText("5");
        
        Element d = new DefaultElement("d");
        scf.addChild(d);
        scf.addChildText("word");
        
        d.addAttribute("id","id2");
        d.addChildText("test");
        
        this.doc.registerNode("id1", output);
        this.doc.registerNode("id2", d);
    }
    
    public void testChild() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( scf ) 
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "child") ;
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", "scf", null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "word", result);
    } 
    
    public void testParent() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( parent::*/model ) 
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "parent");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", null, null);
        addOperand(expr, "step", "model", null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "5", result);
    } 
    
    public void testAncestor() {
        Element ctxnode = this.doc.getRootElement().getChildElement(2).getChildElement(0).getChildElement(0); // d
        assertEquals("ctxnode name", "d", ctxnode.getName());
        
        // ( ancestor::haw ) 
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "ancestor");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", "haw", null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "1", result);
    }

    public void testDescendant() {
        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( descendant::d ) 
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "descendant");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", "d", null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "test", result);
    }    
    
    public void testFollowingSibling() {
        Element ctxnode = this.doc.getRootElement().getChildElement(1); // xforms:output
        assertEquals("ctxnode name", "output", ctxnode.getName());
        
        // ( following-sibling::* ) 
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "following-sibling");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", null, null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "5", result);
    }   

    public void testPrecedingSibling() {
        Element ctxnode = this.doc.getRootElement().getChildElement(1); // xforms:output
        assertEquals("ctxnode name", "output", ctxnode.getName());
        
        // ( preceding-sibling::* ) 
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "preceding-sibling");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", null, null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "abc", result);
    }   
    
    public void testGrandchild() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( scf/d )
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "child");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", "scf", null);
        addOperand(expr, "step", "d", null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "test", result);
    } 
    
    public void testAbsolute() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( /haw/model/scf/d )
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "absolute");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", null, null);        
        addOperand(expr, "step", "haw", null);
        addOperand(expr, "step", "model", null);
        addOperand(expr, "step", "scf", null);
        addOperand(expr, "step", "d", null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "test", result);
    }
    
    public void testAttribute() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( /haw/model/@ref )
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "attribute");
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "absolute");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", null, null);        
        addOperand(expr, "step", "haw", null);
        addOperand(expr, "step", "model", null);
        addOperand(expr, "step", "ref", null);
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "test", result);
    } 

    public void testCalculate() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        assertEquals("ctxnode simpleContent", "5", ctxnode.getSimpleContent());
        
        // ( 3 + 4 * 5 div 2 mod 4 ) -> -(- (+ 3 (mod (div (* 4 5) 2) 4) 4))
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", new Integer(3));
        addOperand(expr, "push", new Integer(4));
        Node[] ns = {ctxnode};
        addOperand(expr, "push", ns);
        addOperand(expr, "mul");
        addOperand(expr, "push", new Integer(2));
        addOperand(expr, "div");
        addOperand(expr, "push", new Integer(4));
        addOperand(expr, "mod");
        addOperand(expr, "add");
        addOperand(expr, "push", "4");
        addOperand(expr, "sub");
        addOperand(expr, "neg");
        
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "-1", result);
    } 
        
    public void testCompare() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( 5 < 6 and 4 > 3 and 5 = 5 or 3 != 8 and 4 <= 4 and 5 >= 7 ) 
        // (or (and (and (< 5 6) (> 4 3)) (= 5 5)) (and (and (!= 3 8) (<= 4 4)) (>= 5 7)))
        //  (or (and (and     t      t)      t ) (and  (and        t        t)      f   ))
        //  (or (and  t t) (and t f))
        //  (or  t  f)
        //  t
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", new Integer(5));
        addOperand(expr, "push", "6");
        addOperand(expr, "lt");
        addOperand(expr, "push", "4");
        addOperand(expr, "push", "3");
        addOperand(expr, "gt");
        addOperand(expr, "and");
        Node[] ns = {ctxnode};
        addOperand(expr, "push", ns);
        addOperand(expr, "push", new Integer(5));
        addOperand(expr, "eq");
        addOperand(expr, "and");

        addOperand(expr, "push", "3");
        addOperand(expr, "push", new Integer(8));
        addOperand(expr, "ne");
        addOperand(expr, "push", new Integer(4));
        addOperand(expr, "push", new Integer(4));
        addOperand(expr, "le");
        addOperand(expr, "and");
        
        addOperand(expr, "push", "5");
        addOperand(expr, "push", new Integer(7));
        addOperand(expr, "ge");        
        addOperand(expr, "and");
        addOperand(expr, "or");        
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "true", result);
    } 
    
    public void testNumberedFilter() {

        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( ../*[3] ) 
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", new Integer(3));        
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "parent");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", null, null);
        addOperand(expr, "step", null, null);
        addOperand(expr, "nfilter");
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "5", result);
    } 
    
    public void testFilter() {
        try {
        Element ctxnode = this.doc.getRootElement().getChildElement(2); // xforms:model
        assertEquals("ctxnode name", "model", ctxnode.getName());
        
        // ( ../*[position() = last() - 1] ) 
        Object[] closure = new Object[5];
        closure[0] = XPathExpr.createOperand("ctxpload", "1");
        closure[1] = XPathExpr.createOperand("ctxpload", "0");
        closure[2] = XPathExpr.createOperand("push", new Integer(1));
        closure[3] = XPathExpr.createOperand("sub");
        closure[4] = XPathExpr.createOperand("eq");
        
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", closure);        
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "parent");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", null, null);
        addOperand(expr, "step", null, null);
        addOperand(expr, "filter");
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "", result);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    public void testConcat() {
        try {
        Element ctxnode = this.doc.getRootElement();
        assertEquals("ctxnode name", "haw", ctxnode.getName());
        
        // ( concat ( output[1] , model/scf/d ) ) 
        
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", new Integer(1));        
        addOperand(expr, "push", "child");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", "output", null);
        addOperand(expr, "nfilter");
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "child");
        addOperand(expr, "push", "child");
        addOperand(expr, "ctxnload");
        addOperand(expr, "step", "model", null);
        addOperand(expr, "step", "scf", null);
        addOperand(expr, "step", "d", null);
        addOperand(expr, "fcall", "concat", "2");        
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "abctest", result);

        expr = new XPathExpr();
        addOperand(expr, "push", "abc");        
        addOperand(expr, "push", "def");
        addOperand(expr, "push", "ghi");
        addOperand(expr, "push", "jkl");
        addOperand(expr, "fcall", "concat", "4");        
        
        result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "abcdefghijkl", result);
        
        } catch (Exception e) {
            e.printStackTrace();
        }
    }    
    
    public void testId() {
        try {
        Element ctxnode = this.doc.getRootElement();
        assertEquals("ctxnode name", "haw", ctxnode.getName());
        
        // ( concat ( id ("id1") , id ("id2")) ) 
        
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "id1");
        addOperand(expr, "fcall", "id");
        addOperand(expr, "push", "id2");
        addOperand(expr, "fcall", "id");
        addOperand(expr, "fcall", "concat", "2");
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("check result", "abctest", result);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }   
    
    /* ǉ
    public static final String FUNC_CONTAINS = "contains";
    public static final String FUNC_SUBSTRING_BEFORE = "substring-before";
    public static final String FUNC_SUBSTRING_AFTER = "substring-after";
    public static final String FUNC_SUBSTRING = "substring";
    public static final String FUNC_STRING_LENGTH = "string-length";
    public static final String FUNC_NORMALIZE_SPACE = "normalize-space";
    public static final String FUNC_TRANSLATE = "translate";
     */
    
    public void testContains() {
        try {
        Element ctxnode = this.doc.getRootElement();
        assertEquals("ctxnode name", "haw", ctxnode.getName());
        
        // contains ( "abcdefg", "cde" ) 
        
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "abcdefg");
        addOperand(expr, "push", "cde");
        addOperand(expr, "fcall", "contains");   
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("true", result);
        
        addOperand(expr, "push", "abcdefg");
        addOperand(expr, "push", "fgh");
        addOperand(expr, "fcall", "contains", "2");
        
        result = expr.evaluate(ctxnode).string();
        assertEquals("false", result);
        
        } catch (Exception e) {
            e.printStackTrace();
        }
    }  
    
    public void testSubstring_before() {
        try {
        Element ctxnode = this.doc.getRootElement();
        assertEquals("ctxnode name", "haw", ctxnode.getName());
        
        // substring-before ( "abcdefg", "cde" ) 
        
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "abcdefg");
        addOperand(expr, "push", "cde");
        addOperand(expr, "fcall", "substring-before", "2");   
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("ab", result);
        
        addOperand(expr, "push", "abcdefg");
        addOperand(expr, "push", "abcdeg");
        addOperand(expr, "fcall", "substring-before");
        
        result = expr.evaluate(ctxnode).string();
        assertEquals("", result);
        
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    public void testSubstring_after() {
        try {
        Element ctxnode = this.doc.getRootElement();
        assertEquals("ctxnode name", "haw", ctxnode.getName());
        
        // substring-before ( "abcdefg", "cde" ) 
        
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "abcdefg");
        addOperand(expr, "push", "cde");
        addOperand(expr, "fcall", "substring-after", "2");   
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("fg", result);
        
        addOperand(expr, "push", "abcdefg");
        addOperand(expr, "push", "gh");
        addOperand(expr, "fcall", "substring-after");
        
        result = expr.evaluate(ctxnode).string();
        assertEquals("", result);
        
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    public void testSubstring() {
        try {
        Element ctxnode = this.doc.getRootElement();
        assertEquals("ctxnode name", "haw", ctxnode.getName());
        
        // substring( "abcdefg", 1, 3 ) 
        
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "abcdefg");
        addOperand(expr, "push", "1");
        addOperand(expr, "push", "3");
        addOperand(expr, "fcall", "substring", "3");   
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("abc", result);
        
        expr = new XPathExpr();
        addOperand(expr, "push", "abcdefg");
        addOperand(expr, "push", "-1");
        addOperand(expr, "push", "10");
        addOperand(expr, "fcall", "substring", "3");
        
        result = expr.evaluate(ctxnode).string();
        assertEquals("abcdefg", result);
        
        expr = new XPathExpr();
        addOperand(expr, "push", "abcdefg");
        addOperand(expr, "push", "3");
        addOperand(expr, "fcall", "substring", "2");
        
        result = expr.evaluate(ctxnode).string();
        assertEquals("cdefg", result);

        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    public void testString_length() {
        try {
        Element ctxnode = this.doc.getRootElement();
        assertEquals("ctxnode name", "haw", ctxnode.getName());
        
        // string-length( "abcdefg" ) 
        
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "abcdefg");
        addOperand(expr, "fcall", "string-length", "1");   
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("7", result);
        
        expr = new XPathExpr();
        addOperand(expr, "fcall", "string-length");   
        
        result = expr.evaluate(ctxnode).string();
        assertEquals("1", result);

        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    public void testNormalize_space() {
        try {
        Element ctxnode = this.doc.getRootElement();
        assertEquals("ctxnode name", "haw", ctxnode.getName());
        
        // normalize-space( " a,   b,      c, d  " ) 
        
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", " \ta, \t  b,  \r\n    c, d \r ");
        addOperand(expr, "fcall", "normalize-space", "1");   
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("a, b, c, d", result);
        
        expr = new XPathExpr();
        addOperand(expr, "fcall", "normalize-space");   
        
        result = expr.evaluate(ctxnode).string();
        assertEquals("1", result);

        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    public void testTranslate() {
        try {
        Element ctxnode = this.doc.getRootElement();
        assertEquals("ctxnode name", "haw", ctxnode.getName());
        
        // translate("bar","abc","ABC") 
        
        XPathExpr expr = new XPathExpr();
        addOperand(expr, "push", "bar");
        addOperand(expr, "push", "abc");
        addOperand(expr, "push", "ABC");
        addOperand(expr, "fcall", "translate");   
        
        String result = expr.evaluate(ctxnode).string();
        assertEquals("BAr", result);
        
        // translate("--aaa--","abc-n","ABC")
        
        addOperand(expr, "push", "--aaa--");
        addOperand(expr, "push", "abc-n");
        addOperand(expr, "push", "ABC");
        addOperand(expr, "fcall", "translate");   
        
        result = expr.evaluate(ctxnode).string();
        assertEquals("AAA", result);
        
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    
    private void addOperand(XPathExpr expr, String op) {
        expr.addOperand(XPathExpr.createOperand(op));
    }
    
    private void addOperand(XPathExpr expr, String op, Object param0) {
        expr.addOperand(XPathExpr.createOperand(op, param0));
    }
    
    private void addOperand(XPathExpr expr, String op, Object param0, Object param1) {
        expr.addOperand(XPathExpr.createOperand(op, param0, param1));
    }

    private void addOperand(XPathExpr expr, String op, Object param0, Object param1, Object param2) {
        expr.addOperand(XPathExpr.createOperand(op, param0, param1, param2));
    }

    public Test suite() {
        TestSuite suite = new TestSuite();
        suite.addTest(new XPathEvaluatorTest("testXPath", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testChild(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testGrandchild", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testGrandchild(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testAbsolute", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testAbsolute(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testParent", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testParent(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testAncestor", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testAncestor(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testFollowingSibling", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testFollowingSibling(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testPrecedingSibling", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testPrecedingSibling(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testDescendant", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testDescendant(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testCalculate", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testCalculate(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testCompare", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testCompare(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testNumberedFilter", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testNumberedFilter(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testFilter", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testFilter(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testConcat", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testConcat(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testId", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testId(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testContains", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testContains(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testSubstring_before", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testSubstring_before(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testSubstring_after", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testSubstring_after(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testSubstring", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testSubstring(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testString_length", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testString_length(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testNormalize_space", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testNormalize_space(); }
        }));
        suite.addTest(new XPathEvaluatorTest("testTranslate", new TestMethod() {
            public void run(TestCase tc) throws IOException{ ((XPathEvaluatorTest)tc).testTranslate(); }
        }));
        return suite;
    }
    
}