
#==============================================================================#
# gruby/graph/chart/pie.rb
# $Id: pie.rb,v 1.6 2002/09/19 16:33:29 yuya Exp $
#==============================================================================#

#require 'ggcl/math/math'
#require 'ggcl/extension/array'
#require 'ggcl/extension/enumerable'

class GRb

  # Graph Module
  module Graph

    # Chart Module
    module Chart

      # PieBase Class
      class PieBase < Base

        # Initialize Methods

        def initialize(attribute)
          super()
          @attribute = attribute
          @values    = []
        end

        # Private Instance Methods

        def calculate_box(box, stretch)
          dx = ( stretch ? box.dx : [box.dx, box.dy].min ) # ΰ
          dy = ( stretch ? box.dy : [box.dx, box.dy].min ) # ΰι⤵
          rx = dx / 2.0                                    # ߤȾ(X)
          ry = dy / 2.0                                    # ߤȾ(Y)
          cx = box.x + box.dx / 2.0                        # ߤ濴(X)
          cy = box.y + box.dy / 2.0                        # ߤ濴(Y)
          return [dx, dy, rx, ry, cx, cy]
        end
        private :calculate_box

        def calculate_total(data)
           sum = 0

           data.each { |item|
             sum += item.value
           }

           return sum
        end
        private :calculate_total

        def calculate_ratio(data, total)
          return data.collect { |value|
            Ratio.new(value, value.value.to_f / total * 360)
          }
        end
        private :calculate_ratio

        def draw_label(image, cx, cy, rx, ry, angle, text)
          length1    = @attribute.label.line.length1
          length2    = @attribute.label.line.length2
          line_color = @attribute.label.line.color
          font       = @attribute.label.font.create_font
          spacing    = 4

          e1 = length1 - Math.sin(angle).abs * length1
          e2 = length2

          ex0 = cx + Math.cos(angle - 90) * rx
          ey0 = cy + Math.sin(angle - 90) * ry
          ex1 = cx + Math.cos(angle - 90) * (rx + e1)
          ey1 = cy + Math.sin(angle - 90) * (ry + e1)

          case angle % 360
          when 0...180
            ex2   = cx + rx + e2
            ey2   = ey1
            lx    = spacing
            align = Text::ALIGN_LEFT
          when 180...360
            ex2   = cx - rx - e2
            ey2   = ey1
            lx    = -spacing
            align = Text::ALIGN_RIGHT
          else
            raise 'bug?'
          end

          image.line(ex0, ey0, ex1, ey1, line_color)
          image.line(ex1, ey1, ex2, ey2, line_color)
          Text.new(font, text, align, Text::VALIGN_MIDDLE).draw2(image, ex2 + lx, ey2)
        end
        private :draw_label

        # Instance Methods

        def add_value(name, value, color)
          @values << Value.new(name, value, color)
        end

        # Value Class
        class Value

          # Initialize Method

          def initialize(name, value, color)
            @name  = name
            @value = value
            @color = color
          end

          # Accessor

          attr_accessor :name, :value, :color

          # Instance Methods

          def <=>(other)
            return @value <=> other.value
          end

        end # Value

        # Ratio Class
        class Ratio

          # Initialize Method

          def initialize(value, ratio)
            @value = value
            @ratio = ratio
          end

          # Accessor

          attr_reader :value, :ratio

          # Instance Methods

          def <=>(other)
            return @ratio <=> other.ratio
          end

        end # Raito

      end # PieBase

      #================================================================#

      # Pie Class
      class Pie < PieBase

        # Initialize Method

        def initialize(attribute)
          super(attribute)
        end

        # Private Instance Methods

        def draw_sector(image, cx, cy, rx, ry, angle_start, angle_end, fill)
          s = (angle_start + 270) % 360
          e = (angle_end + 270)   % 360

          fill_color = image.set(fill)
          image.image.filledArc(cx, cy, rx * 2, ry * 2, s, e, fill_color, GD::Arc)

          if @attribute.edge.visible
            line_color = image.set(@attribute.edge.color)
            image.image.filledArc(cx, cy, rx * 2, ry * 2, s, e, line_color, GD::Arc | GD::NoFill | GD::Edged)
          end
        end
        private :draw_sector

        # Instance Methods

        def draw(image, box)
          dx, dy, rx, ry, cx, cy = calculate_box(box, @attribute.stretch)
          total = calculate_total(@values)
          ratio = calculate_ratio(@values, total)

          current = 0

          ratio.each { |item|
            angle_start  = current.ceil
            angle_end    = (current + item.ratio).ceil
            angle_median = ((angle_end - angle_start) / 2.0 + angle_start).ceil

            draw_sector(image, cx, cy, rx, ry, angle_start, angle_end, item.value.color)

            if @attribute.label.visible
              case @attribute.label.type
              when :name;  label = item.value.name
              when :value; label = item.value.value.to_s
              when :raito; label = format('%.01f%%', item.ratio / 360 * 100)
              else
                raise 'invalid label type'
              end

              draw_label(image, cx, cy, rx, ry, angle_median, label)
            end

            current += item.ratio
          }

        end

      end # Pie

      #================================================================#

      # Pie3D Class
      class Pie3D < PieBase

        # Initialize Method

        def initialize(attribute)
          super(attribute)
        end

        # Private Instance Methods

        def draw_sector(image, cx, cy, rx, ry, rz, begin_r, end_r, line_color, fill)
          fill_color = fill.allocate(image)

          angle_start  = begin_r % 360
          angle_end    = end_r   % 360
          angle_median = (angle_end - angle_start) / 2 + angle_start

          angle_side_start  = ( angle_start <  90 ?  90 : angle_start )
          angle_side_end    = ( angle_end   > 270 ? 270 : angle_end   )
          angle_side_median = (angle_side_end - angle_side_start) / 2 + angle_side_start

          begin_x = cx + Math.cos(angle_start - 90)       * (rx - 1)
          begin_y = cy + Math.sin(angle_start - 90)       * (ry - 1)
          end_x   = cx + Math.cos(angle_end - 90)         * (rx - 1)
          end_y   = cy + Math.sin(angle_end - 90)         * (ry - 1)
          fill_x  = cx + Math.cos(angle_median - 90)      * (rx - 3)
          fill_y  = cy + Math.sin(angle_median - 90)      * (ry - 3)
          side_x  = cx + Math.cos(angle_side_median - 90) * (rx - 3)
          side_y  = cy + Math.sin(angle_side_median - 90) * (ry - 3)

          image.filledArc(cx, cy, rx * 2, ry * 2, (angle_start + 270) % 360, (angle_end + 270) % 360, fill_color, GD::Arc)
          image.filledArc(cx, cy, rx * 2, ry * 2, (angle_start + 270) % 360, (angle_end + 270) % 360, line_color, GD::Arc | GD::NoFill | GD::Edged)

          if angle_start > 90 && angle_start < 270
            image.line(begin_x, begin_y, begin_x, begin_y + rz, line_color)
          end

          if angle_end > 90 && angle_end < 270
            image.line(end_x, end_y, end_x, end_y + rz, line_color)
          end

          if (angle_start > 90 && angle_start < 270) || (angle_end > 90 && angle_end < 270) || (angle_start < 90 && angle_end > 270)
            shade_color = (fill - @attribute.shade).allocate(image)
            image.fillToBorder(side_x, side_y + rz, line_color, shade_color)
          end
        end
        private :draw_sector

        # Instance Methods

        def draw(image, box)
          dx, dy, rx, ry, cx, cy = calculate_box(box, @attribute.stretch)
          rz    = @attribute.height
          total = calculate_total(@values)
          ratio = calculate_ratio(@values, total)

          line = @attribute.edge.color.allocate(image)

          image.arc(cx, cy,      rx * 2, ry * 0.8, 0, 360, line)
          image.arc(cx, cy + rz, rx * 2, ry * 0.8, 0, 180, line)
          image.line(cx - rx, cy, cx - rx, cy + rz, line)
          image.line(cx + rx, cy, cx + rx, cy + rz, line)

          ratio.inject(0) { |current, item|
            angle_start  = current.ceil
            angle_end    = (current + item.ratio).ceil
            angle_median = (angle_end - angle_start) / 2 + angle_start

            draw_sector(image, cx, cy, rx, ry * 0.4, rz, angle_start, angle_end, line, item.value.color)
            draw_label(image, cx, cy, rx, ry * 0.4, angle_median, item.value.name)

            current + item.ratio
          }
        end

      end # Pie3D

    end # Chart

  end # Graph

end # GRb

#==============================================================================#
#==============================================================================#
