/*
copyright (c) 2006-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/
#include <windows.h>
#include <tchar.h>
#ifndef COBJMACROS
# define COBJMACROS 1
#endif /* not COBJMACROS */
#include <mlang.h>
#include <errno.h>


typedef struct _ICONV_INFO
{
  DWORD dwMode;
  DWORD dwSrcEncoding;
  DWORD dwDstEncoding;
} ICONV_INFO;
typedef ICONV_INFO *iconv_t;


static LPMULTILANGUAGE lpIMultiLanguage;
static LPSTR lpszCanonical = NULL;


const int _libiconv_version = 0x010b;


static UINT
libiconv_get_codepage (const char *code)
{
  if (code)
    {
      int i, nLeng;
      LPCSTR lpszPrefix[] = {"", "CP", "CSIBM", "DOS", "IBM",
                             "MS", "MSCP", "WINDOWS", NULL};
      LPWSTR lpszCode;
      BSTR bsDst, bsCode;
      MIMECSETINFO mcsiCode;
      HRESULT hResult;

      nLeng = MultiByteToWideChar (CP_ACP, 0, code, -1, NULL, 0);
      lpszCode = HeapAlloc (GetProcessHeap (), 0, nLeng * sizeof (WCHAR));
      MultiByteToWideChar (CP_ACP, 0, code, -1, lpszCode, nLeng);
      bsCode = SysAllocString (lpszCode);
      HeapFree (GetProcessHeap (), 0, lpszCode);
      hResult = IMultiLanguage_GetCharsetInfo (lpIMultiLanguage,
                                                            bsCode, &mcsiCode);
      SysFreeString (bsCode);
      if (hResult == S_OK)
        return mcsiCode.uiInternetEncoding;
      for (i = 0; lpszPrefix[i]; i++)
        {
          int j;

          for (j = 0; (lpszPrefix[i])[j] != '\0'; j++)
            if ((CHAR)CharUpperA ((LPSTR)code[j]) != (lpszPrefix[i])[j])
              break;
          if ((lpszPrefix[i])[j] == '\0')
            {
              int k;

              if (j > 0 && (code[j] == '-' || code[j] == '_'))
                j++;
              for (k = j; code[k] != '\0'; k++)
                if (!IsCharAlphaNumericA (code[k]) || IsCharAlphaA (code[k]))
                  break;
              if (k > j && code[k] == '\0')
                {
                  UINT uiCodePage = 0;

                  for (k = j; code[k] != '\0'; k++)
                    uiCodePage = uiCodePage * 10 + code[k] - '0';
                  return uiCodePage;
                }
            }
        }
    }
  return 0;
}


iconv_t
libiconv_open (const char *tocode,
               const char *fromcode)
{
  if (lpIMultiLanguage && tocode && fromcode)
    {
      DWORD dwDst, dwSrc;

      dwDst = libiconv_get_codepage (tocode);
      dwSrc = libiconv_get_codepage (fromcode);
      if (dwDst != 0 && dwSrc != 0
                            && IMultiLanguage_IsConvertible (lpIMultiLanguage,
                                                        dwSrc, dwDst) == S_OK)
        {
          iconv_t cd;

          cd = HeapAlloc (GetProcessHeap (), 0, sizeof (ICONV_INFO));
          cd->dwMode = 0;
          cd->dwSrcEncoding = dwSrc;
          cd->dwDstEncoding = dwDst;
          return cd;
        }
    }
  errno = EINVAL;
  return (iconv_t)-1;
}


int
libiconv_close (iconv_t cd)
{
  if (cd && cd != (iconv_t)-1)
    HeapFree (GetProcessHeap (), 0, cd);
  return 0;
}


size_t
libiconv (iconv_t   cd,
          char    **inbuf,
          size_t   *inbytesleft,
          char    **outbuf,
          size_t   *outbytesleft)
{
  if (!cd || cd == (iconv_t)-1)
    {
      errno = EINVAL;
      return (size_t)-1;
    }
  if (inbuf && *inbuf && inbytesleft)
    {
      INT nSrc, nDst;

      if (!outbuf || !*outbuf || !outbytesleft)
        {
          errno = E2BIG;
          return (size_t)-1;
        }
      nSrc = *inbytesleft;
      nDst = *outbytesleft;
      if (IMultiLanguage_ConvertString (lpIMultiLanguage, &cd->dwMode,
                                        cd->dwSrcEncoding, cd->dwDstEncoding,
                                        *inbuf, &nSrc, *outbuf, &nDst) != S_OK)
        {
          errno = nDst < nSrc * 8 ? E2BIG : EILSEQ;
          return (size_t)-1;
        }
      *inbuf += nSrc;
      *inbytesleft -= nSrc;
      *outbuf += nDst;
      *outbytesleft -= nDst;
    }
  else
    {
      cd->dwMode = 0;
    }
  return 0;
}


const char *
iconv_canonicalize (const char *name)
{
  DWORD dwCode;
  MIMECPINFO mcpiCode;

  if (lpszCanonical)
    {
      HeapFree (GetProcessHeap (), 0, lpszCanonical);
      lpszCanonical = NULL;
    }
  dwCode = libiconv_get_codepage (name);
  if (dwCode != 0
        && IMultiLanguage_GetCodePageInfo (lpIMultiLanguage, dwCode, &mcpiCode)
                                                                    == S_OK)
    {
      int nLeng;

      nLeng = WideCharToMultiByte (CP_ACP, 0, mcpiCode.wszWebCharset, -1,
                                                        NULL, 0, NULL, NULL);
      lpszCanonical = HeapAlloc (GetProcessHeap (), 0, nLeng * sizeof (CHAR));
      WideCharToMultiByte (CP_ACP, 0, mcpiCode.wszWebCharset, -1,
                                            lpszCanonical, nLeng, NULL, NULL);
    }
  else
    {
      lpszCanonical = HeapAlloc (GetProcessHeap (), 0,
                                        (lstrlenA (name) + 1) * sizeof (CHAR));
      lstrcpyA (lpszCanonical, name);
    }
  return lpszCanonical;
}


void
libiconvlist (int (*do_one) (unsigned int        namescount,
                             const char * const *names,
                             void               *data),
              void *data)
{
  LPENUMCODEPAGE lpIEnumCodePage;

  if (IMultiLanguage_EnumCodePages (lpIMultiLanguage,
                                            0, &lpIEnumCodePage) == NOERROR)
    {
      ULONG ulCelt;
      MIMECPINFO mcpiCode;

      while (IEnumCodePage_Next (lpIEnumCodePage,
                                                1, &mcpiCode, &ulCelt) == S_OK)
        {
          int nLeng, nResult;
          LPSTR lpszName;

          nLeng = WideCharToMultiByte (CP_ACP, 0, mcpiCode.wszWebCharset, -1,
                                                        NULL, 0, NULL, NULL);
          lpszName = HeapAlloc (GetProcessHeap (), 0, nLeng * sizeof (CHAR));
          WideCharToMultiByte (CP_ACP, 0, mcpiCode.wszWebCharset, -1,
                                                lpszName, nLeng, NULL, NULL);
          nResult = do_one (1, &lpszName, data);
          HeapFree (GetProcessHeap (), 0, lpszName);
          if (nResult != 0)
            break;
        }
      IEnumCodePage_Release (lpIEnumCodePage);
    }
}


int
libiconvctl (iconv_t  cd,
             int      request,
             void    *argument)
{
  errno = EINVAL;
  return -1;
}


void
libiconv_set_relocation_prefix (const char *orig_prefix,
                                const char *curr_prefix)
{
}


BOOL WINAPI
DllMain (HINSTANCE hinstDLL,
         DWORD     fdwReason,
         LPVOID    lpvReserved)
{
  switch (fdwReason)
    {
      case DLL_PROCESS_ATTACH:
        return CoInitialize (NULL) == S_OK
                && CoCreateInstance (&CLSID_CMultiLanguage, NULL, CLSCTX_ALL,
                    &IID_IMultiLanguage, (LPVOID *)&lpIMultiLanguage) == S_OK;
      case DLL_PROCESS_DETACH:
        if (lpszCanonical)
          HeapFree (GetProcessHeap (), 0, lpszCanonical);
        IMultiLanguage_Release (lpIMultiLanguage);
        CoUninitialize ();
    }
  return TRUE;
}
