/*
    IA32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "cpu.h"
#include "message.h"
#include "stack.h"
#include "system.h"


/******************************************************************************
*                                                                             *
* ja:CPU内部関数群                                                            *
*                                                                             *
******************************************************************************/
#define reg_get_l(r) ((r)&0x000000ff)
#define reg_get_h(r) (((r)>>8)&0x0000ffff)
#define reg_get_x(r) ((r)&0x0000ffff)
#define reg_set_l(r,v) ((r)=((r)&0xffffff00)|((v)&0x000000ff))
#define reg_set_h(r,v) ((r)=((r)&0xffff00ff)|(((v)<<8)&0x0000ff00))
#define reg_set_x(r,v) ((r)=((r)&0xffff0000)|((v)&0x0000ffff))
#define FLAG_CF  0
#define FLAG_PF  2
#define FLAG_AF  4
#define FLAG_ZF  6
#define FLAG_SF  7
#define FLAG_TF  8
#define FLAG_IF  9
#define FLAG_DF 10
#define FLAG_OF 11
#define flag_get(r,f) ((r)&(1<<(f))?1:0)
#define flag_set(r,f) ((r)|=1<<(f))
#define flag_clr(r,f) ((r)&=~(1<<(f)))
#define flag_put(r,f,v) ((v)?flag_set(r,f):flag_clr(r,f))


static guint
flag_parity (guint8 value)
{
  gint i, n = 1;

  for (i = 0; i < 8; i++)
    if (value & (1 << i))
      n++;
  return n & 1;
}


static guint8
reg_get8 (Ia32Thread   *thread,
          const guint8  reg)
{
  switch (reg & 7)
    {
      case 0: return reg_get_l (thread->reg.eax);
      case 1: return reg_get_l (thread->reg.ecx);
      case 2: return reg_get_l (thread->reg.edx);
      case 3: return reg_get_l (thread->reg.ebx);
      case 4: return reg_get_h (thread->reg.eax);
      case 5: return reg_get_h (thread->reg.ecx);
      case 6: return reg_get_h (thread->reg.edx);
      case 7: return reg_get_h (thread->reg.ebx);
    }
  return 0;
}


static guint16
reg_get16 (Ia32Thread   *thread,
           const guint8  reg)
{
  switch (reg & 7)
    {
      case 0: return reg_get_x (thread->reg.eax);
      case 1: return reg_get_x (thread->reg.ecx);
      case 2: return reg_get_x (thread->reg.edx);
      case 3: return reg_get_x (thread->reg.ebx);
      case 4: return reg_get_x (thread->reg.esp);
      case 5: return reg_get_x (thread->reg.ebp);
      case 6: return reg_get_x (thread->reg.esi);
      case 7: return reg_get_x (thread->reg.edi);
    }
  return 0;
}


static guint32
reg_get32 (Ia32Thread   *thread,
           const guint8  reg)
{
  switch (reg & 7)
    {
      case 0: return thread->reg.eax;
      case 1: return thread->reg.ecx;
      case 2: return thread->reg.edx;
      case 3: return thread->reg.ebx;
      case 4: return thread->reg.esp;
      case 5: return thread->reg.ebp;
      case 6: return thread->reg.esi;
      case 7: return thread->reg.edi;
    }
  return 0;
}


static void
reg_set8 (Ia32Thread    *thread,
           const guint8  reg,
           const guint8  value)
{
  switch (reg & 7)
    {
      case 0: reg_set_l (thread->reg.eax, value); break;
      case 1: reg_set_l (thread->reg.ecx, value); break;
      case 2: reg_set_l (thread->reg.edx, value); break;
      case 3: reg_set_l (thread->reg.ebx, value); break;
      case 4: reg_set_h (thread->reg.eax, value); break;
      case 5: reg_set_h (thread->reg.ecx, value); break;
      case 6: reg_set_h (thread->reg.edx, value); break;
      case 7: reg_set_h (thread->reg.ebx, value);
    }
}


static void
reg_set16 (Ia32Thread    *thread,
           const guint8   reg,
           const guint16  value)
{
  switch (reg & 7)
    {
      case 0: reg_set_x (thread->reg.eax, value); break;
      case 1: reg_set_x (thread->reg.ecx, value); break;
      case 2: reg_set_x (thread->reg.edx, value); break;
      case 3: reg_set_x (thread->reg.ebx, value); break;
      case 4: reg_set_x (thread->reg.esp, value); break;
      case 5: reg_set_x (thread->reg.ebp, value); break;
      case 6: reg_set_x (thread->reg.esi, value); break;
      case 7: reg_set_x (thread->reg.edi, value);
    }
}


static void
reg_set32 (Ia32Thread    *thread,
           const guint8   reg,
           const guint32  value)
{
  switch (reg & 7)
    {
      case 0: thread->reg.eax = value; break;
      case 1: thread->reg.ecx = value; break;
      case 2: thread->reg.edx = value; break;
      case 3: thread->reg.ebx = value; break;
      case 4: thread->reg.esp = value; break;
      case 5: thread->reg.ebp = value; break;
      case 6: thread->reg.esi = value; break;
      case 7: thread->reg.edi = value;
    }
}


/*  ja:セグメント変換
     thread,スレッド構造体
    address,アドレス
       pseg,セグメントレジスタプリフィックス
        RET,リニアアドレス                                                  */
guint32
ia32_cpu_seg (Ia32Thread    *thread,
              const guint32  address,
              const guint8   pseg)
{
  guint16 selector;
  Ia32CpuDesc desc;

  switch (pseg)
    {
      case IA32_CPU_PSEG_CS: selector = thread->reg.cs; break;
      case IA32_CPU_PSEG_ES: selector = thread->reg.es; break;
      case IA32_CPU_PSEG_FS: selector = thread->reg.fs; break;
      case IA32_CPU_PSEG_GS: selector = thread->reg.gs; break;
      case IA32_CPU_PSEG_SS: selector = thread->reg.ss; break;
      default:               selector = thread->reg.ds;
    }
  if (ia32_memory_read (thread->process, thread->reg.gdtr.base
                    + ia32_cpu_seg_get_index (selector) * sizeof (Ia32CpuDesc),
                                            &desc, sizeof (Ia32CpuDesc)) >= 0)
    return address + ia32_cpu_desc_get_base (&desc);
  return address;
}


typedef struct _ModRM
{
  gsize leng;
  gint mod, regop, regmem;
  guint32 address;
} ModRM;


/*  ja:ModR/Mを解析する
     thread,スレッド構造体
    address,仮想アドレス
       padr,アドレスサイズプレフィックス(TRUE:16ビット,FALSE:32ビット)
       pseg,セグメントレジスタプリフィックス
      modrm,ModR/M
        RET,TRUE:正常終了,FALSE:エラー                                      */
static gboolean
cpu_modrm (Ia32Thread     *thread,
           const guint32   address,
           const gboolean  padr,
           const guint8    pseg,
           ModRM          *m)
{
  guint8 c, pseg_real;

  if (ia32_memory_read_byte (thread->process, address, &c) < 0)
    return FALSE;
  pseg_real = pseg;
  m->leng = 1;
  m->mod = (c >> 6) & 3;
  m->regop = (c >> 3) & 7;
  m->regmem = c & 7;
  m->address = 0;
  if (m->mod == 3)
    return TRUE;
  if (padr)
    {
      if (!(m->mod == 0 && m->regmem == 6))
        switch (m->regmem)
          {
            case 0: m->address = reg_get_x (thread->reg.ebx)
                               + reg_get_x (thread->reg.esi);
                    break;
            case 1: m->address = reg_get_x (thread->reg.ebx)
                               + reg_get_x (thread->reg.edi);
                    break;
            case 2: m->address = reg_get_x (thread->reg.ebp)
                               + reg_get_x (thread->reg.esi);
                    if (pseg_real == IA32_CPU_PSEG_NONE)
                      pseg_real = IA32_CPU_PSEG_SS;
                    break;
            case 3: m->address = reg_get_x (thread->reg.ebp)
                               + reg_get_x (thread->reg.edi);
                    if (pseg_real == IA32_CPU_PSEG_NONE)
                      pseg_real = IA32_CPU_PSEG_SS;
                    break;
            case 4: m->address = reg_get_x (thread->reg.esi);
                    break;
            case 5: m->address = reg_get_x (thread->reg.edi);
                    break;
            case 6: m->address = reg_get_x (thread->reg.ebp);
                    if (pseg_real == IA32_CPU_PSEG_NONE)
                      pseg_real = IA32_CPU_PSEG_SS;
                    break;
            case 7: m->address = reg_get_x (thread->reg.ebx);
          }
      if (m->mod == 1)
        {
          guint8 disp;

          if (ia32_memory_read_byte (thread->process,
                                                address + m->leng, &disp) < 0)
            return FALSE;
          m->leng++;
          m->address += (gint8)disp;
        }
      else if ((m->mod == 0 && m->regmem == 6) || m->mod == 2)
        {
          guint16 disp;

          if (ia32_memory_read_word (thread->process,
                                                address + m->leng, &disp) < 0)
            return FALSE;
          m->leng += 2;
          m->address += disp;
        }
    }
  else
    {
      guint8 ss = 0, index = 0, base = 0;

      if (!((m->mod == 0 && m->regmem == 5) || m->regmem == 4))
        switch (m->regmem)
          {
            case 0: m->address = thread->reg.eax; break;
            case 1: m->address = thread->reg.ecx; break;
            case 2: m->address = thread->reg.edx; break;
            case 3: m->address = thread->reg.ebx; break;
            case 5: m->address = thread->reg.ebp;
                    if (pseg_real == IA32_CPU_PSEG_NONE)
                      pseg_real = IA32_CPU_PSEG_SS;
                    break;
            case 6: m->address = thread->reg.esi; break;
            case 7: m->address = thread->reg.edi;
          }
      if (m->regmem == 4)
        {
          if (ia32_memory_read_byte (thread->process,
                                                    address + m->leng, &c) < 0)
            return FALSE;
          m->leng++;
          ss = (c >> 6) & 3;
          index = (c >> 3) & 7;
          base = c & 7;
        }
      if (m->mod == 1)
        {
          guint8 disp;

          if (ia32_memory_read_byte (thread->process,
                                                address + m->leng, &disp) < 0)
            return FALSE;
          m->leng++;
          m->address += (gint8)disp;
        }
      else if ((m->mod == 0 && m->regmem == 4 && base == 5)
                            || (m->mod == 0 && m->regmem == 5) || m->mod == 2)
        {
          guint32 disp;

          if (ia32_memory_read_dword (thread->process,
                                                address + m->leng, &disp) < 0)
            return FALSE;
          m->leng += 4;
          m->address += disp;
        }
      if (m->regmem == 4)
        {
          switch (index)
            {
              case 0: m->address += thread->reg.eax << ss; break;
              case 1: m->address += thread->reg.ecx << ss; break;
              case 2: m->address += thread->reg.edx << ss; break;
              case 3: m->address += thread->reg.ebx << ss; break;
              case 5: m->address += thread->reg.ebp << ss;
                    if (pseg_real == IA32_CPU_PSEG_NONE)
                      pseg_real = IA32_CPU_PSEG_SS;
                    break;
              case 6: m->address += thread->reg.esi << ss; break;
              case 7: m->address += thread->reg.edi << ss;
            }
          switch (base)
            {
              case 0: m->address += thread->reg.eax; break;
              case 1: m->address += thread->reg.ecx; break;
              case 2: m->address += thread->reg.edx; break;
              case 3: m->address += thread->reg.ebx; break;
              case 4: m->address += thread->reg.esp; break;
              case 5: if (m->mod != 0)
                        {
                          m->address += thread->reg.ebp;
                          if (pseg_real == IA32_CPU_PSEG_NONE)
                            pseg_real = IA32_CPU_PSEG_SS;
                        }
                      break;
              case 6: m->address += thread->reg.esi; break;
              case 7: m->address += thread->reg.edi;
            }
        }
    }
  m->address = ia32_cpu_seg (thread, m->address, pseg_real);
  return TRUE;
}


/*  ja:ModR/Mから8ビット値を取得する
    thread,スレッド構造体
         m,ModR/M
     value,8ビット値
       RET,TRUE:正常終了,FALSE:エラー                                       */
static gboolean
modrm_get8 (Ia32Thread *thread,
            ModRM      *m,
            guint8     *value)
{
  if (m->mod == 3)
    *value = reg_get8 (thread, m->regmem);
  else if (ia32_memory_read_byte (thread->process, m->address, value) < 0)
    return FALSE;
  return TRUE;
}


/*  ja:ModR/Mから16ビット値を取得する
    thread,スレッド構造体
         m,ModR/M
     value,16ビット値
       RET,TRUE:正常終了,FALSE:エラー                                       */
static gboolean
modrm_get16 (Ia32Thread *thread,
             ModRM      *m,
             guint16    *value)
{
  if (m->mod == 3)
    *value = reg_get16 (thread, m->regmem);
  else if (ia32_memory_read_word (thread->process, m->address, value) < 0)
    return FALSE;
  return TRUE;
}


/*  ja:ModR/Mから32ビット値を取得する
    thread,スレッド構造体
         m,ModR/M
     value,32ビット値
       RET,TRUE:正常終了,FALSE:エラー                                       */
static gboolean
modrm_get32 (Ia32Thread *thread,
             ModRM      *m,
             guint32    *value)
{
  if (m->mod == 3)
    *value = reg_get32 (thread, m->regmem);
  else if (ia32_memory_read_dword (thread->process, m->address, value) < 0)
    return FALSE;
  return TRUE;
}


/*  ja:ModR/Mから8ビット値を設定する
    thread,スレッド構造体
         m,ModR/M
     value,8ビット値
     level,レベル(-1:変更なし)
       RET,TRUE:正常終了,FALSE:エラー                                       */
static gboolean
modrm_set8 (Ia32Thread    *thread,
            ModRM         *m,
            const guint8   value,
            const guint8   level)
{
  if (m->mod == 3)
    reg_set8 (thread, m->regmem, value);
  else if (ia32_memory_write_byte (thread->process,
                                                m->address, value, level) < 0)
    return FALSE;
  return TRUE;
}


/*  ja:ModR/Mから16ビット値を設定する
    thread,スレッド構造体
         m,ModR/M
     value,16ビット値
     level,レベル(-1:変更なし)
       RET,TRUE:正常終了,FALSE:エラー                                       */
static gboolean
modrm_set16 (Ia32Thread    *thread,
             ModRM         *m,
             const guint16  value,
             const guint8   level)
{
  if (m->mod == 3)
    reg_set16 (thread, m->regmem, value);
  else if (ia32_memory_write_word (thread->process,
                                                m->address, value, level) < 0)
    return FALSE;
  return TRUE;
}


/*  ja:ModR/Mから32ビット値を設定する
    thread,スレッド構造体
         m,ModR/M
     value,32ビット値
     level,レベル(-1:変更なし)
       RET,TRUE:正常終了,FALSE:エラー                                       */
static gboolean
modrm_set32 (Ia32Thread    *thread,
             ModRM         *m,
             const guint32  value,
             const guint8   level)
{
  if (m->mod == 3)
    reg_set32 (thread, m->regmem, value);
  else if (ia32_memory_write_dword (thread->process,
                                                m->address, value, level) < 0)
    return FALSE;
  return TRUE;
}


/*  ja:8ビット演算
    thread,スレッド構造体
        v0,8ビット値
        v1,8ビット値
      type,タイプ
       RET,演算結果                                                         */
static guint8
calc8 (Ia32Thread   *thread,
       const guint8  v0,
       const guint8  v1,
       const guint8  type)
{
  guint8 result = 0;
  guint32 eflags;

  eflags = thread->reg.eflags;
  flag_clr (eflags, FLAG_CF);
  flag_clr (eflags, FLAG_OF);
  switch (type)
    {
      case 2: /* adc */
        result = flag_get (thread->reg.eflags, FLAG_CF);
      case 0: /* add */
        if ((guint16)v0 + v1 + result > G_MAXUINT8)
          flag_set (eflags, FLAG_CF);
        result = v0 + v1 + result;
        if ((v0 ^ result) & (v1 ^ result) & 0x80)
          flag_set (eflags, FLAG_OF);
        break;
      case 1: /* or */
        result = v0 | v1;
          break;
      case 3: /* sbb */
        result = flag_get (thread->reg.eflags, FLAG_CF);
      case 5: /* sub */
      case 7: /* cmp */
        if (v0 < (guint16)v1 + result)
          flag_set (eflags, FLAG_CF);
        result = v0 - v1 - result;
        if ((v0 ^ result) & (~v1 ^ result) & 0x80)
          flag_set (eflags, FLAG_OF);
        break;
      case 4: /* and */
        result = v0 & v1;
        break;
      case 6: /* xor */
        result = v0 ^ v1;
    }
  flag_put (eflags, FLAG_PF, flag_parity (result));
  flag_put (eflags, FLAG_ZF, result == 0);
  flag_put (eflags, FLAG_SF, result & 0x80);
  thread->reg.eflags = eflags;
  return result;
}


/*  ja:16ビット演算
    thread,スレッド構造体
        v0,16ビット値
        v1,16ビット値
      type,タイプ
       RET,演算結果                                                         */
static guint16
calc16 (Ia32Thread    *thread,
        const guint16  v0,
        const guint16  v1,
        const guint8   type)
{
  guint16 result = 0;
  guint32 eflags;

  eflags = thread->reg.eflags;
  flag_clr (eflags, FLAG_CF);
  flag_clr (eflags, FLAG_OF);
  switch (type)
    {
      case 2: /* adc */
        result = flag_get (thread->reg.eflags, FLAG_CF);
      case 0: /* add */
        if ((guint32)v0 + v1 + result > G_MAXUINT16)
          flag_set (eflags, FLAG_CF);
        result = v0 + v1 + result;
        if ((v0 ^ result) & (v1 ^ result) & 0x8000)
          flag_set (eflags, FLAG_OF);
        break;
      case 1: /* or */
        result = v0 | v1;
          break;
      case 3: /* sbb */
        result = flag_get (eflags, FLAG_CF);
      case 5: /* sub */
      case 7: /* cmp */
        if (v0 < (guint32)v1 + result)
          flag_set (eflags, FLAG_CF);
        result = v0 - v1 - result;
        if ((v0 ^ result) & (~v1 ^ result) & 0x8000)
          flag_set (eflags, FLAG_OF);
        break;
      case 4: /* and */
        result = v0 & v1;
        break;
      case 6: /* xor */
        result = v0 ^ v1;
    }
  flag_put (eflags, FLAG_PF, flag_parity (result));
  flag_put (eflags, FLAG_ZF, result == 0);
  flag_put (eflags, FLAG_SF, result & 0x8000);
  thread->reg.eflags = eflags;
  return result;
}


/*  ja:32ビット演算
    thread,スレッド構造体
        v0,32ビット値
        v1,32ビット値
      type,タイプ
       RET,演算結果                                                         */
static guint32
calc32 (Ia32Thread    *thread,
        const guint32  v0,
        const guint32  v1,
        const guint8   type)
{
  guint32 eflags, result = 0;

  eflags = thread->reg.eflags;
  flag_clr (eflags, FLAG_CF);
  flag_clr (eflags, FLAG_OF);
  switch (type)
    {
      case 2: /* adc */
        result = flag_get (thread->reg.eflags, FLAG_CF);
      case 0: /* add */
        if ((guint64)v0 + v1 + result > G_MAXUINT32)
          flag_set (eflags, FLAG_CF);
        result = v0 + v1 + result;
        if ((v0 ^ result) & (v1 ^ result) & 0x80000000)
          flag_set (eflags, FLAG_OF);
        break;
      case 1: /* or */
        result = v0 | v1;
          break;
      case 3: /* sbb */
        result = flag_get (thread->reg.eflags, FLAG_CF);
      case 5: /* sub */
      case 7: /* cmp */
        if (v0 < (guint64)v1 + result)
          flag_set (eflags, FLAG_CF);
        result = v0 - v1 - result;
        if ((v0 ^ result) & (~v1 ^ result) & 0x80000000)
          flag_set (eflags, FLAG_OF);
        break;
      case 4: /* and */
        result = v0 & v1;
        break;
      case 6: /* xor */
        result = v0 ^ v1;
    }
  flag_put (eflags, FLAG_PF, flag_parity (result));
  flag_put (eflags, FLAG_ZF, result == 0);
  flag_put (eflags, FLAG_SF, result & 0x80000000);
  thread->reg.eflags = eflags;
  return result;
}


/*  ja:8ビット演算(inc/dec)
    thread,スレッド構造体
     value,8ビット値
    incdec,タイプ(TRUE:inc,FALSE:dec)
       RET,演算結果                                                         */
static guint8
incdec8 (Ia32Thread     *thread,
         const guint8    value,
         const gboolean  incdec)
{
  guint8 result;

  if (incdec)
    {
      result = value + 1;
      flag_put (thread->reg.eflags, FLAG_OF,
                                (result & 0x80) != 0 && (value & 0x80) == 0);
    }
  else
    {
      result = value - 1;
      flag_put (thread->reg.eflags, FLAG_OF,
                                (result & 0x80) == 0 && (value & 0x80) != 0);
    }
  flag_put (thread->reg.eflags, FLAG_PF, flag_parity (result));
  flag_put (thread->reg.eflags, FLAG_ZF, result == 0);
  flag_put (thread->reg.eflags, FLAG_SF, result & 0x80);
  return result;
}


/*  ja:16ビット演算(inc/dec)
    thread,スレッド構造体
     value,16ビット値
    incdec,タイプ(TRUE:inc,FALSE:dec)
       RET,演算結果                                                         */
static guint16
incdec16 (Ia32Thread     *thread,
          const guint16   value,
          const gboolean  incdec)
{
  guint16 result;

  if (incdec)
    {
      result = value + 1;
      flag_put (thread->reg.eflags, FLAG_OF,
                            (result & 0x8000) != 0 && (value & 0x8000) == 0);
    }
  else
    {
      result = value - 1;
      flag_put (thread->reg.eflags, FLAG_OF,
                            (result & 0x8000) == 0 && (value & 0x8000) != 0);
    }
  flag_put (thread->reg.eflags, FLAG_PF, flag_parity (result));
  flag_put (thread->reg.eflags, FLAG_ZF, result == 0);
  flag_put (thread->reg.eflags, FLAG_SF, result & 0x8000);
  return result;
}


/*  ja:32ビット演算(inc/dec)
    thread,スレッド構造体
     value,32ビット値
    incdec,タイプ(TRUE:inc,FALSE:dec)
       RET,演算結果                                                         */
static guint32
incdec32 (Ia32Thread     *thread,
          const guint32   value,
          const gboolean  incdec)
{
  guint32 result;

  if (incdec)
    {
      result = value + 1;
      flag_put (thread->reg.eflags, FLAG_OF,
                    (result & 0x80000000) != 0 && (value & 0x80000000) == 0);
    }
  else
    {
      result = value - 1;
      flag_put (thread->reg.eflags, FLAG_OF,
                    (result & 0x80000000) == 0 && (value & 0x80000000) != 0);
    }
  flag_put (thread->reg.eflags, FLAG_PF, flag_parity (result));
  flag_put (thread->reg.eflags, FLAG_ZF, result == 0);
  flag_put (thread->reg.eflags, FLAG_SF, result & 0x80000000);
  return result;
}


/******************************************************************************
*                                                                             *
* ja:CPU関数群                                                                *
*                                                                             *
******************************************************************************/
static gint
ia32_cpu_extend (Ia32Thread     *thread,
                 guint32         eip,
                 const gboolean  pope,
                 const gboolean  padr,
                 const guint8    pseg,
                 const guint8    prep)
{
  gint8 level;
  guint8 code;

  while ((level = ia32_memory_read_byte (thread->process, eip, &code)) >= 0)
    switch (code)
      {
        case 0x40: case 0x41: case 0x42: case 0x43: /* cmovcc */
        case 0x44: case 0x45: case 0x46: case 0x47:
        case 0x48: case 0x49: case 0x4a: case 0x4b:
        case 0x4c: case 0x4d: case 0x4e: case 0x4f:
          {
            gboolean result = FALSE;
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            switch (code)
              {
                case 0x40: /* cmovo */
                  if (flag_get (thread->reg.eflags, FLAG_OF) == 1)
                    result = TRUE;
                  break;
                case 0x41: /* cmovno */
                  if (flag_get (thread->reg.eflags, FLAG_OF) == 0)
                    result = TRUE;
                  break;
                case 0x42: /* cmovb */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 1)
                    result = TRUE;
                  break;
                case 0x43: /* cmovae */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 0)
                    result = TRUE;
                  break;
                case 0x44: /* cmovz */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 1)
                    result = TRUE;
                  break;
                case 0x45: /* cmovnz */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 0)
                    result = TRUE;
                  break;
                case 0x46: /* cmovbe */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 1
                                || flag_get (thread->reg.eflags, FLAG_ZF) == 1)
                    result = TRUE;
                  break;
                case 0x47: /* cmova */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 0
                                && flag_get (thread->reg.eflags, FLAG_ZF) == 0)
                    result = TRUE;
                  break;
                case 0x48: /* cmovs */
                  if (flag_get (thread->reg.eflags, FLAG_SF) == 1)
                    result = TRUE;
                  break;
                case 0x49: /* cmovns */
                  if (flag_get (thread->reg.eflags, FLAG_SF) == 0)
                    result = TRUE;
                  break;
                case 0x4a: /* cmovp */
                  if (flag_get (thread->reg.eflags, FLAG_PF) == 1)
                    result = TRUE;
                  break;
                case 0x4b: /* cmovnp */
                  if (flag_get (thread->reg.eflags, FLAG_PF) == 0)
                    result = TRUE;
                  break;
                case 0x4c: /* cmovl */
                  if (flag_get (thread->reg.eflags, FLAG_SF)
                                    != flag_get (thread->reg.eflags, FLAG_OF))
                    result = TRUE;
                  break;
                case 0x4d: /* cmovge */
                  if (flag_get (thread->reg.eflags, FLAG_SF)
                                    == flag_get (thread->reg.eflags, FLAG_OF))
                    result = TRUE;
                  break;
                case 0x4e: /* cmovle */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 1
                                    || flag_get (thread->reg.eflags, FLAG_SF)
                                    != flag_get (thread->reg.eflags, FLAG_OF))
                    result = TRUE;
                  break;
                case 0x4f: /* cmovg */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 0
                                    && flag_get (thread->reg.eflags, FLAG_SF)
                                    == flag_get (thread->reg.eflags, FLAG_OF))
                    result = TRUE;
              }
            if (result)
              {
                if (pope)
                  {
                    guint16 temp;

                    if (!modrm_get16 (thread, &m, &temp))
                      return IA32_ERR_GP;
                    reg_set16 (thread, m.regop, temp);
                  }
                else
                  {
                    guint32 temp;

                    if (!modrm_get32 (thread, &m, &temp))
                      return IA32_ERR_GP;
                    reg_set32 (thread, m.regop, temp);
                  }
              }
            thread->reg.eip = eip + m.leng + 1;
          }
          goto loop;
        case 0x80: case 0x81: case 0x82: case 0x83: /* jcc */
        case 0x84: case 0x85: case 0x86: case 0x87:
        case 0x88: case 0x89: case 0x8a: case 0x8b:
        case 0x8c: case 0x8d: case 0x8e: case 0x8f:
          {
            gint32 relative;

            if (pope)
              {
                guint16 imm16;

                if (ia32_memory_read_word (thread->process,
                                                        eip + 1, &imm16) < 0)
                  return IA32_ERR_GP;
                thread->reg.eip = eip + 3;
                relative = (gint16)imm16;
              }
            else
              {
                guint32 imm32;

                if (ia32_memory_read_dword (thread->process,
                                                        eip + 1, &imm32) < 0)
                  return IA32_ERR_GP;
                thread->reg.eip = eip + 5;
                relative = (gint32)imm32;
              }
            switch (code)
              {
                case 0x80: /* jo */
                  if (flag_get (thread->reg.eflags, FLAG_OF) == 1)
                    thread->reg.eip += relative;
                  break;
                case 0x81: /* jno */
                  if (flag_get (thread->reg.eflags, FLAG_OF) == 0)
                    thread->reg.eip += relative;
                  break;
                case 0x82: /* jb */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 1)
                    thread->reg.eip += relative;
                  break;
                case 0x83: /* jae */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 0)
                    thread->reg.eip += relative;
                  break;
                case 0x84: /* jz */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 1)
                    thread->reg.eip += relative;
                  break;
                case 0x85: /* jnz */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 0)
                    thread->reg.eip += relative;
                  break;
                case 0x86: /* jbe */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 1
                                || flag_get (thread->reg.eflags, FLAG_ZF) == 1)
                    thread->reg.eip += relative;
                  break;
                case 0x87: /* ja */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 0
                                && flag_get (thread->reg.eflags, FLAG_ZF) == 0)
                    thread->reg.eip += relative;
                  break;
                case 0x88: /* js */
                  if (flag_get (thread->reg.eflags, FLAG_SF) == 1)
                    thread->reg.eip += relative;
                  break;
                case 0x89: /* jns */
                  if (flag_get (thread->reg.eflags, FLAG_SF) == 0)
                    thread->reg.eip += relative;
                  break;
                case 0x8a: /* jp */
                  if (flag_get (thread->reg.eflags, FLAG_PF) == 1)
                    thread->reg.eip += relative;
                  break;
                case 0x8b: /* jnp */
                  if (flag_get (thread->reg.eflags, FLAG_PF) == 0)
                    thread->reg.eip += relative;
                  break;
                case 0x8c: /* jl */
                  if (flag_get (thread->reg.eflags, FLAG_SF)
                                    != flag_get (thread->reg.eflags, FLAG_OF))
                    thread->reg.eip += relative;
                  break;
                case 0x8d: /* jge */
                  if (flag_get (thread->reg.eflags, FLAG_SF)
                                    == flag_get (thread->reg.eflags, FLAG_OF))
                    thread->reg.eip += relative;
                  break;
                case 0x8e: /* jle */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 1
                                    || flag_get (thread->reg.eflags, FLAG_SF)
                                    != flag_get (thread->reg.eflags, FLAG_OF))
                    thread->reg.eip += relative;
                  break;
                case 0x8f: /* jg */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 0
                                    && flag_get (thread->reg.eflags, FLAG_SF)
                                    == flag_get (thread->reg.eflags, FLAG_OF))
                    thread->reg.eip += relative;
              }
          }
          goto loop;
        case 0x90: case 0x91: case 0x92: case 0x93: /* setcc */
        case 0x94: case 0x95: case 0x96: case 0x97:
        case 0x98: case 0x99: case 0x9a: case 0x9b:
        case 0x9c: case 0x9d: case 0x9e: case 0x9f:
          {
            gboolean result = FALSE;
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            switch (code)
              {
                case 0x90: /* seto */
                  if (flag_get (thread->reg.eflags, FLAG_OF) == 1)
                    result = TRUE;
                  break;
                case 0x91: /* setno */
                  if (flag_get (thread->reg.eflags, FLAG_OF) == 0)
                    result = TRUE;
                  break;
                case 0x92: /* setb */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 1)
                    result = TRUE;
                  break;
                case 0x93: /* setae */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 0)
                    result = TRUE;
                  break;
                case 0x94: /* setz */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 1)
                    result = TRUE;
                  break;
                case 0x95: /* setnz */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 0)
                    result = TRUE;
                  break;
                case 0x96: /* setbe */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 1
                                || flag_get (thread->reg.eflags, FLAG_ZF) == 1)
                    result = TRUE;
                  break;
                case 0x97: /* seta */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 0
                                && flag_get (thread->reg.eflags, FLAG_ZF) == 0)
                    result = TRUE;
                  break;
                case 0x98: /* sets */
                  if (flag_get (thread->reg.eflags, FLAG_SF) == 1)
                    result = TRUE;
                  break;
                case 0x99: /* setns */
                  if (flag_get (thread->reg.eflags, FLAG_SF) == 0)
                    result = TRUE;
                  break;
                case 0x9a: /* setp */
                  if (flag_get (thread->reg.eflags, FLAG_PF) == 1)
                    result = TRUE;
                  break;
                case 0x9b: /* setnp */
                  if (flag_get (thread->reg.eflags, FLAG_PF) == 0)
                    result = TRUE;
                  break;
                case 0x9c: /* setl */
                  if (flag_get (thread->reg.eflags, FLAG_SF)
                                    != flag_get (thread->reg.eflags, FLAG_OF))
                    result = TRUE;
                  break;
                case 0x9d: /* setge */
                  if (flag_get (thread->reg.eflags, FLAG_SF)
                                    == flag_get (thread->reg.eflags, FLAG_OF))
                    result = TRUE;
                  break;
                case 0x9e: /* setle */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 1
                                    || flag_get (thread->reg.eflags, FLAG_SF)
                                    != flag_get (thread->reg.eflags, FLAG_OF))
                    result = TRUE;
                  break;
                case 0x9f: /* setg */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 0
                                    && flag_get (thread->reg.eflags, FLAG_SF)
                                    == flag_get (thread->reg.eflags, FLAG_OF))
                    result = TRUE;
              }
            if (!modrm_set8 (thread, &m, result ? 1 : 0, level))
              return IA32_ERR_GP;
            thread->reg.eip = eip + m.leng + 1;
          }
          goto loop;
        case 0xa2: /* cpuid */
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xa3: /* bt */
        case 0xab: /* bts */
        case 0xb3: /* btr */
        case 0xba:
        case 0xbb: /* btc */
          {
            guint8 bit;
            gint32 value;
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if (code == 0xba)
              {
                guint8 imm8;

                if (ia32_memory_read_byte (thread->process,
                                                eip + m.leng + 1, &imm8) < 0)
                  return IA32_ERR_GP;
                value = (gint8)imm8;
              }
            else if (pope)
              {
                value = reg_get16 (thread, m.regop);
              }
            else
              {
                value = reg_get32 (thread, m.regop);
              }
            if (m.mod == 3)
              {
                value &= 31;
                bit = (reg_get32 (thread, m.regmem) >> value) & 1;
              }
            else
              {
                guint8 temp;

                m.address += value / 8;
                if (ia32_memory_read_byte (thread->process,
                                                        m.address, &temp) < 0)
                  return IA32_ERR_GP;
                value %= 8;
                bit = (temp >> value) & 1;
              }
            flag_put (thread->reg.eflags, FLAG_CF, bit);
            if (code == 0xab || (code == 0xba && m.regop == 5))
              {
                if (m.mod == 3)
                  {
                    reg_set32 (thread, m.regmem,
                                reg_get32 (thread, m.regmem) | (1 << value));
                  }
                else
                  {
                    guint8 temp;

                    if (ia32_memory_read_byte (thread->process,
                                                        m.address, &temp) < 0
                        || ia32_memory_write_byte (thread->process,
                                    m.address, temp | (1 << value), level) < 0)
                      return IA32_ERR_GP;
                  }
              }
            else if (code == 0xb3 || (code == 0xba && m.regop == 6))
              {
                if (m.mod == 3)
                  {
                    reg_set32 (thread, m.regmem,
                                reg_get32 (thread, m.regmem) & ~(1 << value));
                  }
                else
                  {
                    guint8 temp;

                    if (ia32_memory_read_byte (thread->process,
                                                        m.address, &temp) < 0
                        || ia32_memory_write_byte (thread->process,
                                m.address, temp & ~(1 << value), level) < 0)
                      return IA32_ERR_GP;
                  }
              }
            else if (code == 0xbb || (code == 0xba && m.regop == 7))
              {
                if (m.mod == 3)
                  {
                    reg_set32 (thread, m.regmem,
                                reg_get32 (thread, m.regmem) ^ (1 << value));
                  }
                else
                  {
                    guint8 temp;

                    if (ia32_memory_read_byte (thread->process,
                                                        m.address, &temp) < 0
                        || ia32_memory_write_byte (thread->process,
                                    m.address, temp ^ (1 << value), level) < 0)
                      return IA32_ERR_GP;
                  }
              }
            thread->reg.eip = eip + m.leng + (code == 0xba ? 2 : 1);
          }
          goto loop;
        case 0xa4: /* shld */
        case 0xa5:
        case 0xac: /* shrd */
        case 0xad:
          {
            guint8 count;
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if ((code & 1) == 0)
              {
                if (ia32_memory_read_byte (thread->process,
                                                eip + m.leng + 1, &count) < 0)
                  return IA32_ERR_GP;
              }
            else
              {
                count = reg_get_l (thread->reg.ecx);
              }
            count &= 63;
            if (count > 0)
              {
                if (pope)
                  {
                    guint16 temp;
                    guint32 value;

                    if (!modrm_get16 (thread, &m, &temp))
                      return IA32_ERR_GP;
                    value = ((guint32)temp << 16)
                                                | reg_get16 (thread, m.regop);
                    if ((code & 8) == 0)
                      {
                        flag_put (thread->reg.eflags, FLAG_CF,
                                value & ((guint32)0x80000000 >> (count - 1)));
                        value <<= count;
                      }
                    else
                      {
                        flag_put (thread->reg.eflags, FLAG_CF,
                                value & ((guint32)0x00000001 << (count - 1)));
                        value >>= count;
                      }
                    temp = value >> 16;
                    flag_put (thread->reg.eflags, FLAG_PF, flag_parity (temp));
                    flag_put (thread->reg.eflags, FLAG_ZF, temp == 0);
                    flag_put (thread->reg.eflags, FLAG_SF, temp & 0x8000);
                    if (!modrm_set16 (thread, &m, temp, level))
                      return IA32_ERR_GP;
                  }
                else
                  {
                    guint32 temp;
                    guint64 value;

                    if (!modrm_get32 (thread, &m, &temp))
                      return IA32_ERR_GP;
                    value = ((guint64)temp << 32)
                                                | reg_get32 (thread, m.regop);
                    if ((code & 8) == 0)
                      {
                        flag_put (thread->reg.eflags, FLAG_CF,
                            value & (G_GUINT64_CONSTANT (0x8000000000000000)
                                                            >> (count - 1)));
                        value <<= count;
                      }
                    else
                      {
                        flag_put (thread->reg.eflags, FLAG_CF,
                            value & (G_GUINT64_CONSTANT (0x0000000000000001)
                                                            << (count - 1)));
                        value >>= count;
                      }
                    temp = value >> 32;
                    flag_put (thread->reg.eflags, FLAG_PF, flag_parity (temp));
                    flag_put (thread->reg.eflags, FLAG_ZF, temp == 0);
                    flag_put (thread->reg.eflags, FLAG_SF, temp & 0x80000000);
                    if (!modrm_set32 (thread, &m, temp, level))
                      return IA32_ERR_GP;
                  }
              }
            thread->reg.eip = eip + m.leng + 1;
            if ((code & 1) == 0)
              thread->reg.eip++;
          }
          goto loop;
        case 0xaf: /* imul */
          {
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if (pope)
              {
                guint16 value;
                gint32 result;

                if (!modrm_get16 (thread, &m, &value))
                  return IA32_ERR_GP;
                result = (gint32)(gint16)reg_get16 (thread, m.regop)
                                                            * (gint16)value;
                if (result < G_MININT16 || G_MAXINT16 < result)
                  {
                    flag_set (thread->reg.eflags, FLAG_CF);
                    flag_set (thread->reg.eflags, FLAG_OF);
                  }
                else
                  {
                    flag_clr (thread->reg.eflags, FLAG_CF);
                    flag_clr (thread->reg.eflags, FLAG_OF);
                  }
                reg_set16 (thread, m.regop, result);
              }
            else
              {
                guint32 value;
                gint64 result;

                if (!modrm_get32 (thread, &m, &value))
                  return IA32_ERR_GP;
                result = (gint64)(gint32)reg_get32 (thread, m.regop)
                                                            * (gint32)value;
                if (result < G_MININT32 || G_MAXINT32 < result)
                  {
                    flag_set (thread->reg.eflags, FLAG_CF);
                    flag_set (thread->reg.eflags, FLAG_OF);
                  }
                else
                  {
                    flag_clr (thread->reg.eflags, FLAG_CF);
                    flag_clr (thread->reg.eflags, FLAG_OF);
                  }
                reg_set32 (thread, m.regop, result);
              }
            thread->reg.eip = eip + m.leng + 1;
          }
          goto loop;
        case 0xb0: /* cmpxchg */
        case 0xb1:
          {
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if ((code & 1) == 0)
              {
                guint8 value;

                if (!modrm_get8 (thread, &m, &value))
                  return IA32_ERR_GP;
                if (calc8 (thread, reg_get_l (thread->reg.eax), value, 7) == 0)
                  {
                    if (!modrm_set8 (thread, &m,
                                            reg_get8 (thread, m.regop), level))
                      return IA32_ERR_GP;
                    flag_set (thread->reg.eflags, FLAG_ZF);
                  }
                else
                  {
                    reg_set_l (thread->reg.eax, value);
                    flag_clr (thread->reg.eflags, FLAG_ZF);
                  }
              }
            else if (pope)
              {
                guint16 value;

                if (!modrm_get16 (thread, &m, &value))
                  return IA32_ERR_GP;
                if (calc16 (thread,
                                reg_get_l (thread->reg.eax), value, 7) == 0)
                  {
                    if (!modrm_set16 (thread, &m,
                                        reg_get16 (thread, m.regop), level))
                      return IA32_ERR_GP;
                    flag_set (thread->reg.eflags, FLAG_ZF);
                  }
                else
                  {
                    reg_set_x (thread->reg.eax, value);
                    flag_clr (thread->reg.eflags, FLAG_ZF);
                  }
              }
            else
              {
                guint32 value;

                if (!modrm_get32 (thread, &m, &value))
                  return IA32_ERR_GP;
                if (calc32 (thread, thread->reg.eax, value, 7) == 0)
                  {
                    if (!modrm_set32 (thread, &m,
                                        reg_get32 (thread, m.regop), level))
                      return IA32_ERR_GP;
                    flag_set (thread->reg.eflags, FLAG_ZF);
                  }
                else
                  {
                    thread->reg.eax = value;
                    flag_clr (thread->reg.eflags, FLAG_ZF);
                  }
              }
            thread->reg.eip = eip + m.leng + 1;
          }
        case 0xb6: /* movzx */
        case 0xb7:
        case 0xbe: /* movsx */
        case 0xbf:
          {
            guint32 value;
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if ((code & 1) == 0)
              {
                guint8 temp;

                if (!modrm_get8 (thread, &m, &temp))
                  return IA32_ERR_GP;
                value = (code & 8) == 0 ? temp : (gint8)temp;
              }
            else
              {
                guint16 temp;

                if (!modrm_get16 (thread, &m, &temp))
                  return IA32_ERR_GP;
                value = (code & 8) == 0 ? temp : (gint16)temp;
              }
            if (pope)
              reg_set16 (thread, m.regop, value);
            else
              reg_set32 (thread, m.regop, value);
            thread->reg.eip = eip + m.leng + 1;
          }
          goto loop;
        case 0xbc: /* bsf */
        case 0xbd: /* bsr */
          {
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if (pope)
              {
                guint16 value;

                if (!modrm_get16 (thread, &m, &value))
                  return IA32_ERR_GP;
                if (code == 0xbc)
                  {
                    gint i;

                    for (i = 0; i < 16; i++)
                      if ((value & (1 << i)) != 0)
                        {
                          reg_set16 (thread, m.regop, i);
                          break;
                        }
                  }
                else
                  {
                    gint i;

                    for (i = 15; i >= 0; i--)
                      if ((value & (1 << i)) != 0)
                        {
                          reg_set16 (thread, m.regop, i);
                          break;
                        }
                  }
                flag_put (thread->reg.eflags, FLAG_ZF, value == 0);
              }
            else
              {
                guint32 value;

                if (!modrm_get32 (thread, &m, &value))
                  return IA32_ERR_GP;
                if (code == 0xbc)
                  {
                    gint i;

                    for (i = 0; i < 32; i++)
                      if ((value & (1 << i)) != 0)
                        {
                          reg_set32 (thread, m.regop, i);
                          break;
                        }
                  }
                else
                  {
                    gint i;

                    for (i = 31; i >= 0; i--)
                      if ((value & (1 << i)) != 0)
                        {
                          reg_set32 (thread, m.regop, i);
                          break;
                        }
                  }
                flag_put (thread->reg.eflags, FLAG_ZF, value == 0);
              }
            thread->reg.eip = eip + m.leng + 1;
          }
          goto loop;
        case 0xc0: /* xadd */
        case 0xc1:
          {
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if ((code & 1) == 0)
              {
                guint8 v0, v1, result;

                if (!modrm_get8 (thread, &m, &v0))
                  return IA32_ERR_GP;
                v1 = reg_get8 (thread, m.regop);
                result = calc8 (thread, v0, v1, 0);
                reg_set8 (thread, m.regop, v0);
                if (!modrm_set8 (thread, &m, result, level))
                  return IA32_ERR_GP;
              }
            else if (pope)
              {
                guint16 v0, v1, result;

                if (!modrm_get16 (thread, &m, &v0))
                  return IA32_ERR_GP;
                v1 = reg_get16 (thread, m.regop);
                result = calc16 (thread, v0, v1, 0);
                reg_set16 (thread, m.regop, v0);
                if (!modrm_set16 (thread, &m, result, level))
                  return IA32_ERR_GP;
              }
            else
              {
                guint32 v0, v1, result;

                if (!modrm_get32 (thread, &m, &v0))
                  return IA32_ERR_GP;
                v1 = reg_get32 (thread, m.regop);
                result = calc32 (thread, v0, v1, 0);
                reg_set32 (thread, m.regop, v0);
                if (!modrm_set32 (thread, &m, result, level))
                  return IA32_ERR_GP;
              }
            thread->reg.eip = eip + m.leng + 1;
          }
          goto loop;
        case 0xc8: case 0xc9: case 0xca: case 0xcb: /* bswap */
        case 0xcc: case 0xcd: case 0xce: case 0xcf:
          {
            guint32 value;

            value = reg_get32 (thread, code);
            reg_set32 (thread, code, GUINT32_SWAP_LE_BE (value));
          }
          thread->reg.eip = eip + 1;
          goto loop;
        default:
          return IA32_ERR_UD;
      }
  return IA32_ERR_GP;
  loop:
  return IA32_ERR_NULL;
}


/*  ja:ステップ実行
    thread,スレッド構造体
       RET,例外(IA32_ERR_NULL:正常終了)                                     */
gint
ia32_cpu_step (Ia32Thread *thread)
{
  gboolean pope = FALSE, padr = FALSE;
  gint8 level;
  guint8 code, pseg = IA32_CPU_PSEG_NONE, prep = 0;
  guint32 eip;

  if (!thread || thread->terminate)
    return IA32_ERR_NULL;
  if (thread->process->export)
    {
      Ia32Api *api;

      api = g_hash_table_lookup (thread->process->export,
                                        GUINT_TO_POINTER (thread->reg.eip));
      if (api && api->func && !api->func (thread))
        {
          gchar *str;

          str = g_strdup_printf ("\'%s\' is not implemented.", api->name);
          ia32_message_record_error (thread, str);
          g_free (str);
          thread->terminate = TRUE;
          return IA32_ERR_NULL;
        }
    }
  eip = thread->reg.eip;
  while ((level = ia32_memory_read_byte (thread->process, eip, &code)) >= 0)
    switch (code)
      {
        case 0x00: case 0x01: case 0x02: /* add */
        case 0x03: case 0x04: case 0x05:
        case 0x08: case 0x09: case 0x0a: /* or */
        case 0x0b: case 0x0c: case 0x0d:
        case 0x10: case 0x11: case 0x12: /* adc */
        case 0x13: case 0x14: case 0x15:
        case 0x18: case 0x19: case 0x1a: /* sbb */
        case 0x1b: case 0x1c: case 0x1d:
        case 0x20: case 0x21: case 0x22: /* and */
        case 0x23: case 0x24: case 0x25:
        case 0x28: case 0x29: case 0x2a: /* sub */
        case 0x2b: case 0x2c: case 0x2d:
        case 0x30: case 0x31: case 0x32: /* xor */
        case 0x33: case 0x34: case 0x35:
        case 0x38: case 0x39: case 0x3a: /* cmp */
        case 0x3b: case 0x3c: case 0x3d:
          {
            guint8 type;

            type = (code >> 3) & 7;
            if ((code & 7) == 4)
              {
                guint8 imm8, result;

                if (ia32_memory_read_byte (thread->process,
                                                        eip + 1, &imm8) < 0)
                  return IA32_ERR_GP;
                result = calc8 (thread,
                                    reg_get_l (thread->reg.eax), imm8, type);
                if (type != 7)
                  reg_set_l (thread->reg.eax, result);
                thread->reg.eip = eip + 2;
              }
            else if ((code & 7) == 5)
              {
                if (pope)
                  {
                    guint16 imm16, result;

                    if (ia32_memory_read_word (thread->process,
                                                        eip + 1, &imm16) < 0)
                      return IA32_ERR_GP;
                    result = calc16 (thread,
                                    reg_get_x (thread->reg.eax), imm16, type);
                    if (type != 7)
                      reg_set_x (thread->reg.eax, result);
                    thread->reg.eip = eip + 3;
                  }
                else
                  {
                    guint32 imm32, result;

                    if (ia32_memory_read_dword (thread->process,
                                                        eip + 1, &imm32) < 0)
                      return IA32_ERR_GP;
                    result = calc32 (thread, thread->reg.eax, imm32, type);
                    if (type != 7)
                      thread->reg.eax = result;
                    thread->reg.eip = eip + 5;
                  }
              }
            else
              {
                ModRM m;

                if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
                  return IA32_ERR_GP;
                if ((code & 1) == 0)
                  {
                    guint8 v0, v1, result;

                    if (!modrm_get8 (thread, &m, &v0))
                      return IA32_ERR_GP;
                    v1 = reg_get8 (thread, m.regop);
                    if ((code & 2) != 0)
                      {
                        guint8 temp;

                        temp = v0;
                        v0 = v1;
                        v1 = temp;
                      }
                    result = calc8 (thread, v0, v1, type);
                    if (type != 7)
                      {
                        if ((code & 2) != 0)
                          reg_set8 (thread, m.regop, result);
                        else if (!modrm_set8 (thread, &m, result, level))
                          return IA32_ERR_GP;
                      }
                  }
                else if (pope)
                  {
                    guint16 v0, v1, result;

                    if (!modrm_get16 (thread, &m, &v0))
                      return IA32_ERR_GP;
                    v1 = reg_get16 (thread, m.regop);
                    if ((code & 2) != 0)
                      {
                        guint16 temp;

                        temp = v0;
                        v0 = v1;
                        v1 = temp;
                      }
                    result = calc16 (thread, v0, v1, type);
                    if (type != 7)
                      {
                        if ((code & 2) != 0)
                          reg_set16 (thread, m.regop, result);
                        else if (!modrm_set16 (thread, &m, result, level))
                          return IA32_ERR_GP;
                      }
                  }
                else
                  {
                    guint32 v0, v1, result;

                    if (!modrm_get32 (thread, &m, &v0))
                      return IA32_ERR_GP;
                    v1 = reg_get32 (thread, m.regop);
                    if ((code & 2) != 0)
                      {
                        guint32 temp;

                        temp = v0;
                        v0 = v1;
                        v1 = temp;
                      }
                    result = calc32 (thread, v0, v1, type);
                    if (type != 7)
                      {
                        if ((code & 2) != 0)
                          reg_set32 (thread, m.regop, result);
                        else if (!modrm_set32 (thread, &m, result, level))
                          return IA32_ERR_GP;
                      }
                  }
                thread->reg.eip = eip + m.leng + 1;
              }
          }
          goto loop;
        case 0x0f:
          {
            gint vector;

            vector = ia32_cpu_extend (thread, eip + 1, pope, padr, pseg, prep);
            if (vector != IA32_ERR_NULL)
              return vector;
          }
          goto loop;
        case 0x26: /* es */
        case 0x2e: /* cs */
        case 0x36: /* ss */
        case 0x3e: /* ds */
        case 0x64: /* fs */
        case 0x65: /* gs */
          if (pseg != IA32_CPU_PSEG_NONE)
            return IA32_ERR_UD;
          pseg = code;
          eip++;
          break;
        case 0x27: /* daa */
          {
            guint8 al;
            guint32 eflags;

            al = reg_get_l (thread->reg.eax);
            eflags = thread->reg.eflags;
            flag_clr (thread->reg.eflags, FLAG_CF);
            if (((reg_get_l (thread->reg.eax) & 0x0f) > 9)
                                || flag_get (thread->reg.eflags, FLAG_AF) != 0)
              {
                reg_set_l (thread->reg.eax, al + 6);
                if (flag_get (eflags, FLAG_CF) != 0 || al >= G_MAXUINT8 - 6)
                  flag_set (thread->reg.eflags, FLAG_CF);
                flag_set (thread->reg.eflags, FLAG_AF);
              }
            else
              {
                flag_clr (thread->reg.eflags, FLAG_AF);
              }
            if (al > 0x99 || flag_get (eflags, FLAG_CF) != 0)
              {
                reg_set_l (thread->reg.eax,
                                        reg_get_l (thread->reg.eax) + 0x66);
                flag_set (thread->reg.eflags, FLAG_CF);
              }
            else
              {
                flag_clr (thread->reg.eflags, FLAG_CF);
              }
            flag_put (thread->reg.eflags, FLAG_PF,
                                        flag_parity (thread->reg.eax));
            flag_put (thread->reg.eflags, FLAG_ZF,
                                        reg_get_l (thread->reg.eax) == 0);
            flag_put (thread->reg.eflags, FLAG_SF,
                                        reg_get_l (thread->reg.eax) & 0x80);
          }
          goto loop;
        case 0x2f: /* das */
          {
            guint8 al;
            guint32 eflags;

            al = reg_get_l (thread->reg.eax);
            eflags = thread->reg.eflags;
            flag_clr (thread->reg.eflags, FLAG_CF);
            if (((reg_get_l (thread->reg.eax) & 0x0f) > 9)
                                || flag_get (thread->reg.eflags, FLAG_AF) != 0)
              {
                reg_set_l (thread->reg.eax, al - 6);
                if (flag_get (eflags, FLAG_CF) != 0 || al < 6)
                  flag_set (thread->reg.eflags, FLAG_CF);
                flag_set (thread->reg.eflags, FLAG_AF);
              }
            else
              {
                flag_clr (thread->reg.eflags, FLAG_AF);
              }
            if (al > 0x99 || flag_get (eflags, FLAG_CF) != 0)
              {
                reg_set_l (thread->reg.eax,
                                        reg_get_l (thread->reg.eax) - 0x66);
                flag_set (thread->reg.eflags, FLAG_CF);
              }
            else
              {
                flag_clr (thread->reg.eflags, FLAG_CF);
              }
            flag_put (thread->reg.eflags, FLAG_PF,
                                        flag_parity (thread->reg.eax));
            flag_put (thread->reg.eflags, FLAG_ZF,
                                        reg_get_l (thread->reg.eax) == 0);
            flag_put (thread->reg.eflags, FLAG_SF,
                                        reg_get_l (thread->reg.eax) & 0x80);
          }
          goto loop;
        case 0x37: /* aaa */
          if (((reg_get_l (thread->reg.eax) & 0x0f) > 9)
                                || flag_get (thread->reg.eflags, FLAG_AF) != 0)
            {
              reg_set_l (thread->reg.eax, reg_get_l (thread->reg.eax) + 6);
              reg_set_h (thread->reg.eax, reg_get_h (thread->reg.eax) + 1);
              flag_set (thread->reg.eflags, FLAG_AF);
              flag_set (thread->reg.eflags, FLAG_CF);
            }
          else
            {
              flag_clr (thread->reg.eflags, FLAG_AF);
              flag_clr (thread->reg.eflags, FLAG_CF);
            }
          reg_set_l (thread->reg.eax, reg_get_l (thread->reg.eax) & 0x0f);
          goto loop;
        case 0x3f: /* aas */
          if (((reg_get_l (thread->reg.eax) & 0x0f) > 9)
                                || flag_get (thread->reg.eflags, FLAG_AF) != 0)
            {
              reg_set_l (thread->reg.eax, reg_get_l (thread->reg.eax) - 6);
              reg_set_h (thread->reg.eax, reg_get_h (thread->reg.eax) - 1);
              flag_set (thread->reg.eflags, FLAG_AF);
              flag_set (thread->reg.eflags, FLAG_CF);
            }
          else
            {
              flag_clr (thread->reg.eflags, FLAG_AF);
              flag_clr (thread->reg.eflags, FLAG_CF);
            }
          reg_set_l (thread->reg.eax, reg_get_l (thread->reg.eax) & 0x0f);
          goto loop;
        case 0x40: case 0x41: case 0x42: case 0x43: /* inc */
        case 0x44: case 0x45: case 0x46: case 0x47:
        case 0x48: case 0x49: case 0x4a: case 0x4b: /* dec */
        case 0x4c: case 0x4d: case 0x4e: case 0x4f:
          if (pope)
            reg_set16 (thread, code,
                incdec16 (thread, reg_get16 (thread, code), (code & 8) == 0));
          else
            reg_set32 (thread, code,
                incdec32 (thread, reg_get32 (thread, code), (code & 8) == 0));
          thread->reg.eip = eip + 1;
          goto loop;
        case 0x50: case 0x51: case 0x52: case 0x53: /* push */
        case 0x54: case 0x55: case 0x56: case 0x57:
          if (!(pope ? ia32_stack_push16 (thread, reg_get16 (thread, code))
                     : ia32_stack_push32 (thread, reg_get32 (thread, code))))
            return IA32_ERR_SS;
          thread->reg.eip = eip + 1;
          goto loop;
        case 0x58: case 0x59: case 0x5a: case 0x5b: /* pop */
        case 0x5c: case 0x5d: case 0x5e: case 0x5f:
          if (pope)
            {
              guint16 value;

              if (!ia32_stack_pop16 (thread, &value))
                return IA32_ERR_SS;
              reg_set16 (thread, code, value);
            }
          else
            {
              guint32 value;

              if (!ia32_stack_pop32 (thread, &value))
                return IA32_ERR_SS;
              reg_set32 (thread, code, value);
            }
          thread->reg.eip = eip + 1;
          goto loop;
        case 0x60: /* pusha */
          if (pope)
            {
              guint16 temp;

              temp = thread->reg.esp;
              if (!ia32_stack_push16 (thread, thread->reg.eax)
               || !ia32_stack_push16 (thread, thread->reg.ecx)
               || !ia32_stack_push16 (thread, thread->reg.edx)
               || !ia32_stack_push16 (thread, thread->reg.ebx)
               || !ia32_stack_push16 (thread, temp)
               || !ia32_stack_push16 (thread, thread->reg.ebp)
               || !ia32_stack_push16 (thread, thread->reg.esi)
               || !ia32_stack_push16 (thread, thread->reg.edi))
                return IA32_ERR_SS;
            }
          else
            {
              guint32 temp;

              temp = thread->reg.esp;
              if (!ia32_stack_push32 (thread, thread->reg.eax)
               || !ia32_stack_push32 (thread, thread->reg.ecx)
               || !ia32_stack_push32 (thread, thread->reg.edx)
               || !ia32_stack_push32 (thread, thread->reg.ebx)
               || !ia32_stack_push32 (thread, temp)
               || !ia32_stack_push32 (thread, thread->reg.ebp)
               || !ia32_stack_push32 (thread, thread->reg.esi)
               || !ia32_stack_push32 (thread, thread->reg.edi))
                return IA32_ERR_SS;
            }
          thread->reg.eip = eip + 1;
          goto loop;
        case 0x61: /* popa */
          if (pope)
            {
              guint16 value;

              if (!ia32_stack_pop16 (thread, &value))
                return IA32_ERR_SS;
              reg_set_x (thread->reg.edi, value);
              if (!ia32_stack_pop16 (thread, &value))
                return IA32_ERR_SS;
              reg_set_x (thread->reg.esi, value);
              if (!ia32_stack_pop16 (thread, &value))
                return IA32_ERR_SS;
              reg_set_x (thread->reg.ebp, value);
              thread->reg.esp += 2;
              if (!ia32_stack_pop16 (thread, &value))
                return IA32_ERR_SS;
              reg_set_x (thread->reg.ebx, value);
              if (!ia32_stack_pop16 (thread, &value))
                return IA32_ERR_SS;
              reg_set_x (thread->reg.edx, value);
              if (!ia32_stack_pop16 (thread, &value))
                return IA32_ERR_SS;
              reg_set_x (thread->reg.ecx, value);
              if (!ia32_stack_pop16 (thread, &value))
                return IA32_ERR_SS;
              reg_set_x (thread->reg.eax, value);
            }
          else
            {
              if (!ia32_stack_pop32 (thread, &thread->reg.edi)
               || !ia32_stack_pop32 (thread, &thread->reg.esi)
               || !ia32_stack_pop32 (thread, &thread->reg.ebp))
                return IA32_ERR_SS;
              thread->reg.esp += 4;
              if (!ia32_stack_pop32 (thread, &thread->reg.ebx)
               || !ia32_stack_pop32 (thread, &thread->reg.edx)
               || !ia32_stack_pop32 (thread, &thread->reg.ecx)
               || !ia32_stack_pop32 (thread, &thread->reg.eax))
                return IA32_ERR_SS;
            }
          thread->reg.eip = eip + 1;
          goto loop;
        case 0x66:
          pope = TRUE;
          eip++;
          break;
        case 0x67:
          padr = TRUE;
          eip++;
          break;
        case 0x68: /* push */
          if (pope)
            {
              guint16 imm16;

              if (ia32_memory_read_word (thread->process, eip + 1, &imm16) < 0)
                return IA32_ERR_GP;
              if (!ia32_stack_push16 (thread, imm16))
                return IA32_ERR_SS;
              thread->reg.eip = eip + 3;
            }
          else
            {
              guint32 imm32;

              if (ia32_memory_read_dword (thread->process,
                                                        eip + 1, &imm32) < 0)
                return IA32_ERR_GP;
              if (!ia32_stack_push32 (thread, imm32))
                return IA32_ERR_SS;
              thread->reg.eip = eip + 5;
            }
          goto loop;
        case 0x69: /* imul */
        case 0x6b:
          {
            gsize leng = 4;
            guint32 value0, value1;
            gint64 result;
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if (pope)
              {
                guint16 temp;

                if (!modrm_get16 (thread, &m, &temp))
                  return IA32_ERR_GP;
                value0 = (gint16)temp;
              }
            else if (!modrm_get32 (thread, &m, &value0))
              return IA32_ERR_GP;
            if (code == 0x6b)
              {
                guint8 imm8;

                if (ia32_memory_read_byte (thread->process,
                                                eip + m.leng + 1, &imm8) < 0)
                  return IA32_ERR_GP;
                value1 = (gint8)imm8;
                leng = 1;
              }
            else if (pope)
              {
                guint16 imm16;

                if (ia32_memory_read_word (thread->process,
                                                eip + m.leng + 1, &imm16) < 0)
                  return IA32_ERR_GP;
                value1 = (gint16)imm16;
                leng = 2;
              }
            else if (ia32_memory_read_dword (thread->process,
                                                eip + m.leng + 1, &value1) < 0)
              return IA32_ERR_GP;
            result = (gint64)(gint32)value0 * (gint32)value1;
            if (pope)
              {
                if (result < G_MININT16 || G_MAXINT16 < result)
                  {
                    flag_set (thread->reg.eflags, FLAG_CF);
                    flag_set (thread->reg.eflags, FLAG_OF);
                  }
                else
                  {
                    flag_clr (thread->reg.eflags, FLAG_CF);
                    flag_clr (thread->reg.eflags, FLAG_OF);
                  }
                reg_set16 (thread, m.regop, result);
              }
            else
              {
                if (result < G_MININT32 || G_MAXINT32 < result)
                  {
                    flag_set (thread->reg.eflags, FLAG_CF);
                    flag_set (thread->reg.eflags, FLAG_OF);
                  }
                else
                  {
                    flag_clr (thread->reg.eflags, FLAG_CF);
                    flag_clr (thread->reg.eflags, FLAG_OF);
                  }
                reg_set32 (thread, m.regop, result);
              }
            thread->reg.eip = eip + m.leng + leng + 1;
          }
          goto loop;
        case 0x6a: /* push */
          {
            guint8 imm8;

            if (ia32_memory_read_byte (thread->process, eip + 1, &imm8) < 0)
              return IA32_ERR_GP;
            if (!(pope ? ia32_stack_push16 (thread, (gint8)imm8)
                       : ia32_stack_push32 (thread, (gint8)imm8)))
              return IA32_ERR_SS;
          }
          thread->reg.eip = eip + 2;
          goto loop;
        case 0x6c: /* insb */
        case 0x6d: /* insw/insd */
          if (code == 0x6c)
            while (prep == 0 || thread->reg.ecx > 0)
              {
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  thread->reg.edi++;
                else
                  thread->reg.edi--;
                if (prep == 0)
                  break;
                thread->reg.ecx--;
              }
          else if (pope)
            while (prep == 0 || thread->reg.ecx > 0)
              {
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  thread->reg.edi += 2;
                else
                  thread->reg.edi -= 2;
                if (prep == 0)
                  break;
                thread->reg.ecx--;
              }
          else
            while (prep == 0 || thread->reg.ecx > 0)
              {
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  thread->reg.edi += 4;
                else
                  thread->reg.edi -= 4;
                if (prep == 0)
                  break;
                thread->reg.ecx--;
              }
          thread->reg.eip = eip + 1;
          goto loop;
        case 0x6e: /* outsb */
        case 0x6f: /* outsw/outsd */
          if (code == 0x6e)
            while (prep == 0 || thread->reg.ecx > 0)
              {
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  thread->reg.esi++;
                else
                  thread->reg.esi--;
                if (prep == 0)
                  break;
                thread->reg.ecx--;
              }
          else if (pope)
            while (prep == 0 || thread->reg.ecx > 0)
              {
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  thread->reg.esi += 2;
                else
                  thread->reg.esi -= 2;
                if (prep == 0)
                  break;
                thread->reg.ecx--;
              }
          else
            while (prep == 0 || thread->reg.ecx > 0)
              {
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  thread->reg.esi += 4;
                else
                  thread->reg.esi -= 4;
                if (prep == 0)
                  break;
                thread->reg.ecx--;
              }
          thread->reg.eip = eip + 1;
          goto loop;
        case 0x70: case 0x71: case 0x72: case 0x73: /* jcc */
        case 0x74: case 0x75: case 0x76: case 0x77:
        case 0x78: case 0x79: case 0x7a: case 0x7b:
        case 0x7c: case 0x7d: case 0x7e: case 0x7f:
          {
            guint8 imm8;

            if (ia32_memory_read_byte (thread->process, eip + 1, &imm8) < 0)
              return IA32_ERR_GP;
            thread->reg.eip = eip + 2;
            switch (code)
              {
                case 0x70: /* jo */
                  if (flag_get (thread->reg.eflags, FLAG_OF) == 1)
                    thread->reg.eip += (gint8)imm8;
                  break;
                case 0x71: /* jno */
                  if (flag_get (thread->reg.eflags, FLAG_OF) == 0)
                    thread->reg.eip += (gint8)imm8;
                  break;
                case 0x72: /* jb */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 1)
                    thread->reg.eip += (gint8)imm8;
                  break;
                case 0x73: /* jae */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 0)
                    thread->reg.eip += (gint8)imm8;
                  break;
                case 0x74: /* jz */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 1)
                    thread->reg.eip += (gint8)imm8;
                  break;
                case 0x75: /* jnz */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 0)
                    thread->reg.eip += (gint8)imm8;
                  break;
                case 0x76: /* jbe */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 1
                                || flag_get (thread->reg.eflags, FLAG_ZF) == 1)
                    thread->reg.eip += (gint8)imm8;
                  break;
                case 0x77: /* ja */
                  if (flag_get (thread->reg.eflags, FLAG_CF) == 0
                                && flag_get (thread->reg.eflags, FLAG_ZF) == 0)
                    thread->reg.eip += (gint8)imm8;
                  break;
                case 0x78: /* js */
                  if (flag_get (thread->reg.eflags, FLAG_SF) == 1)
                    thread->reg.eip += (gint8)imm8;
                  break;
                case 0x79: /* jns */
                  if (flag_get (thread->reg.eflags, FLAG_SF) == 0)
                    thread->reg.eip += (gint8)imm8;
                  break;
                case 0x7a: /* jp */
                  if (flag_get (thread->reg.eflags, FLAG_PF) == 1)
                    thread->reg.eip += (gint8)imm8;
                  break;
                case 0x7b: /* jnp */
                  if (flag_get (thread->reg.eflags, FLAG_PF) == 0)
                    thread->reg.eip += (gint8)imm8;
                  break;
                case 0x7c: /* jl */
                  if (flag_get (thread->reg.eflags, FLAG_SF)
                                    != flag_get (thread->reg.eflags, FLAG_OF))
                    thread->reg.eip += (gint8)imm8;
                  break;
                case 0x7d: /* jge */
                  if (flag_get (thread->reg.eflags, FLAG_SF)
                                    == flag_get (thread->reg.eflags, FLAG_OF))
                    thread->reg.eip += (gint8)imm8;
                  break;
                case 0x7e: /* jle */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 1
                                    || flag_get (thread->reg.eflags, FLAG_SF)
                                    != flag_get (thread->reg.eflags, FLAG_OF))
                    thread->reg.eip += (gint8)imm8;
                  break;
                case 0x7f: /* jg */
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 0
                                    && flag_get (thread->reg.eflags, FLAG_SF)
                                    == flag_get (thread->reg.eflags, FLAG_OF))
                    thread->reg.eip += (gint8)imm8;
              }
          }
          goto loop;
        case 0x80: /* add/or/adc/sbb/and/sub/xor/cmp */
        case 0x81:
        case 0x82:
        case 0x83:
          {
            gsize leng = 4;
            guint32 value;
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if (code != 0x81)
              {
                guint8 imm8;

                if (ia32_memory_read_byte (thread->process,
                                                eip + m.leng + 1, &imm8) < 0)
                  return IA32_ERR_GP;
                value = (gint8)imm8;
                leng = 1;
              }
            else if (pope)
              {
                guint16 imm16;

                if (ia32_memory_read_word (thread->process,
                                                eip + m.leng + 1, &imm16) < 0)
                  return IA32_ERR_GP;
                value = (gint16)imm16;
                leng = 2;
              }
            else if (ia32_memory_read_dword (thread->process,
                                                eip + m.leng + 1, &value) < 0)
              return IA32_ERR_GP;
            if (code == 0x80 || code == 0x82)
              {
                guint8 temp, result = 0;

                if (!modrm_get8 (thread, &m, &temp))
                  return IA32_ERR_GP;
                result = calc8 (thread, temp, value, m.regop);
                if (m.regop != 7 && !modrm_set8 (thread, &m, result, level))
                  return IA32_ERR_GP;
              }
            else if (pope)
              {
                guint16 temp, result = 0;

                if (!modrm_get16 (thread, &m, &temp))
                  return IA32_ERR_GP;
                result = calc16 (thread, temp, value, m.regop);
                if (m.regop != 7 && !modrm_set16 (thread, &m, result, level))
                  return IA32_ERR_GP;
              }
            else
              {
                guint32 temp, result = 0;

                if (!modrm_get32 (thread, &m, &temp))
                  return IA32_ERR_GP;
                result = calc32 (thread, temp, value, m.regop);
                if (m.regop != 7 && !modrm_set32 (thread, &m, result, level))
                  return IA32_ERR_GP;
              }
            thread->reg.eip = eip + m.leng + leng + 1;
          }
          goto loop;
        case 0x84: /* test */
        case 0x85:
          {
            guint32 result;
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if (code == 0x84)
              {
                guint8 value0, value1;

                if (!modrm_get8 (thread, &m, &value0))
                  return IA32_ERR_GP;
                value1 = reg_get8 (thread, m.regop);
                result = (gint8)(value0 & value1);
              }
            else if (pope)
              {
                guint16 value0, value1;

                if (!modrm_get16 (thread, &m, &value0))
                  return IA32_ERR_GP;
                value1 = reg_get16 (thread, m.regop);
                result = (gint16)(value0 & value1);
              }
            else
              {
                guint32 value0, value1;

                if (!modrm_get32 (thread, &m, &value0))
                  return IA32_ERR_GP;
                value1 = reg_get32 (thread, m.regop);
                result = value0 & value1;
              }
            flag_clr (thread->reg.eflags, FLAG_CF);
            flag_clr (thread->reg.eflags, FLAG_OF);
            flag_put (thread->reg.eflags, FLAG_PF, flag_parity (result));
            flag_put (thread->reg.eflags, FLAG_ZF, result == 0);
            flag_put (thread->reg.eflags, FLAG_SF, result & 0x80000000);
            thread->reg.eip = eip + m.leng + 1;
          }
          goto loop;
        case 0x86: /* xchg */
        case 0x87:
          {
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if (code == 0x86)
              {
                guint8 temp;

                if (!modrm_get8 (thread, &m, &temp)
                    || !modrm_set8 (thread,
                                    &m, reg_get8 (thread, m.regop), level))
                  return IA32_ERR_GP;
                reg_set8 (thread, m.regop, temp);
              }
            else if (pope)
              {
                guint16 temp;

                if (!modrm_get16 (thread, &m, &temp)
                    || !modrm_set16 (thread,
                                    &m, reg_get16 (thread, m.regop), level))
                  return IA32_ERR_GP;
                reg_set16 (thread, m.regop, temp);
              }
            else
              {
                guint32 temp;

                if (!modrm_get32 (thread, &m, &temp)
                    || !modrm_set32 (thread,
                                    &m, reg_get32 (thread, m.regop), level))
                  return IA32_ERR_GP;
                reg_set32 (thread, m.regop, temp);
              }
            thread->reg.eip = eip + m.leng + 1;
          }
          goto loop;
        case 0x88: /* mov */
        case 0x89:
          {
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if (code == 0x88)
              {
                if (!modrm_set8 (thread,
                                    &m, reg_get8 (thread, m.regop), level))
                  return IA32_ERR_GP;
              }
            else if (pope)
              {
                if (!modrm_set16 (thread,
                                    &m, reg_get16 (thread, m.regop), level))
                  return IA32_ERR_GP;
              }
            else
              {
                if (!modrm_set32 (thread,
                                    &m, reg_get32 (thread, m.regop), level))
                  return IA32_ERR_GP;
              }
            thread->reg.eip = eip + m.leng + 1;
          }
          goto loop;
        case 0x8a: /* mov */
        case 0x8b:
          {
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if (code == 0x8a)
              {
                guint8 temp;

                if (!modrm_get8 (thread, &m, &temp))
                  return IA32_ERR_GP;
                reg_set8 (thread, m.regop, temp);
              }
            else if (pope)
              {
                guint16 temp;

                if (!modrm_get16 (thread, &m, &temp))
                  return IA32_ERR_GP;
                reg_set16 (thread, m.regop, temp);
              }
            else
              {
                guint32 temp;

                if (!modrm_get32 (thread, &m, &temp))
                  return IA32_ERR_GP;
                reg_set32 (thread, m.regop, temp);
              }
            thread->reg.eip = eip + m.leng + 1;
          }
          goto loop;
        case 0x8d: /* lea */
          {
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if (m.mod == 3)
              return IA32_ERR_UD;
            if (pope)
              reg_set16 (thread, m.regop, m.address);
            else
              reg_set32 (thread, m.regop, m.address);
            thread->reg.eip = eip + m.leng + 1;
          }
          goto loop;
        case 0x8f: /* pop */
          {
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if (m.regop != 0)
              return IA32_ERR_UD;
            if (pope)
              {
                guint16 value;

                if (!ia32_stack_pop16 (thread, &value))
                  return IA32_ERR_SS;
                if (!modrm_set16 (thread, &m, value, level))
                  return IA32_ERR_GP;
              }
            else
              {
                guint32 value;

                if (!ia32_stack_pop32 (thread, &value))
                  return IA32_ERR_SS;
                if (!modrm_set32 (thread, &m, value, level))
                  return IA32_ERR_GP;
              }
            thread->reg.eip = eip + m.leng + 1;
          }
          goto loop;
        case 0x90: /* nop */
          thread->reg.eip = eip + 1;
          goto loop;
        case 0x91: case 0x92: case 0x93: /* xchg */
        case 0x94: case 0x95: case 0x96: case 0x97:
          if (pope)
            {
              guint16 temp;

              temp = reg_get16 (thread, code);
              reg_set16 (thread, code, reg_get_x (thread->reg.eax));
              reg_set_x (thread->reg.eax, temp);
            }
          else
            {
              guint32 temp;

              temp = reg_get32 (thread, code);
              reg_set32 (thread, code, thread->reg.eax);
              thread->reg.eax = temp;
            }
          thread->reg.eip = eip + 1;
          goto loop;
        case 0x98: /* cbw/cwde */
          if (pope)
            reg_set_x (thread->reg.eax, (gint8)reg_get_l (thread->reg.eax));
          else
            thread->reg.eax = (gint16)reg_get_x (thread->reg.eax);
          thread->reg.eip = eip + 1;
          goto loop;
        case 0x99: /* cwd/cdq */
          if (pope)
            {
              guint32 temp;

              temp = (gint16)reg_get_x (thread->reg.eax);
              reg_set_x (thread->reg.eax, temp);
              reg_set_x (thread->reg.edx, temp >> 16);
            }
          else
            {
              guint64 temp;

              temp = (gint32)thread->reg.eax;
              thread->reg.eax = temp;
              thread->reg.edx = temp >> 32;
            }
          thread->reg.eip = eip + 1;
          goto loop;
        case 0x9b: /* wait */
          eip++;
          break;
        case 0x9c: /* pushf */
          if (!(pope ? ia32_stack_push16 (thread, thread->reg.eflags)
                     : ia32_stack_push32 (thread, thread->reg.eflags)))
            return IA32_ERR_SS;
          thread->reg.eip = eip + 1;
          goto loop;
        case 0x9d: /* popf */
          if (pope)
            {
              guint16 value;

              if (!ia32_stack_pop16 (thread, &value))
                return IA32_ERR_SS;
              reg_set_x (thread->reg.eflags, value);
            }
          else
            {
              if (!ia32_stack_pop32 (thread, &thread->reg.eflags))
                return IA32_ERR_SS;
            }
          thread->reg.eip = eip + 1;
          goto loop;
        case 0x9e: /* sahf */
          reg_set_l (thread->reg.eflags,
                                (reg_get_h (thread->reg.eax) & 215) | 2);
          thread->reg.eip = eip + 1;
          goto loop;
        case 0x9f: /* lahf */
          reg_set_h (thread->reg.eax,
                                (reg_get_l (thread->reg.eflags) & 215) | 2);
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xa0: /* mov */
        case 0xa1:
        case 0xa2:
        case 0xa3:
          {
            guint32 address;

            if (padr)
              {
                guint16 temp;

                if (ia32_memory_read_word (thread->process,
                                                        eip + 1, &temp) < 0)
                  return IA32_ERR_GP;
                address = temp;
              }
            else
              {
                if (ia32_memory_read_dword (thread->process,
                                                        eip + 1, &address) < 0)
                  return IA32_ERR_GP;
              }
            address = ia32_cpu_seg (thread, address, pseg);
            switch (code)
              {
                case 0xa0:
                  {
                    guint8 temp;

                    if (ia32_memory_read_byte (thread->process,
                                                        address, &temp) < 0)
                      return IA32_ERR_GP;
                    reg_set_l (thread->reg.eax, temp);
                  }
                  break;
                case 0xa1:
                  if (pope)
                    {
                      guint16 temp;

                      if (ia32_memory_read_word (thread->process,
                                                        address, &temp) < 0)
                        return IA32_ERR_GP;
                      reg_set_x (thread->reg.eax, temp);
                    }
                  else
                    {
                      if (ia32_memory_read_dword (thread->process,
                                                address, &thread->reg.eax) < 0)
                        return IA32_ERR_GP;
                    }
                  break;
                case 0xa2:
                  if (ia32_memory_write_byte (thread->process,
                            address, reg_get_l (thread->reg.eax), level) < 0)
                    return IA32_ERR_GP;
                  break;
                case 0xa3:
                  if (pope)
                    {
                      if (ia32_memory_write_word (thread->process,
                            address, reg_get_x (thread->reg.eax), level) < 0)
                        return IA32_ERR_GP;
                    }
                  else
                    {
                      if (ia32_memory_write_dword (thread->process,
                                        address, thread->reg.eax, level) < 0)
                        return IA32_ERR_GP;
                    }
              }
          }
          thread->reg.eip = eip + (padr ? 3 : 5);
          goto loop;
        case 0xa4: /* movsb */
        case 0xa5: /* movsw/movsd */
          if (code == 0xa4)
            while (prep == 0 || thread->reg.ecx > 0)
              {
                guint8 temp;

                if (ia32_memory_read_byte (thread->process,
                    ia32_cpu_seg (thread, thread->reg.esi, pseg), &temp) < 0
                    || ia32_memory_write_byte (thread->process,
                                        ia32_cpu_seg (thread, thread->reg.edi,
                                        IA32_CPU_PSEG_ES), temp, level) < 0)
                  return IA32_ERR_GP;
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  {
                    thread->reg.esi++;
                    thread->reg.edi++;
                  }
                else
                  {
                    thread->reg.esi--;
                    thread->reg.edi--;
                  }
                if (prep == 0)
                  break;
                thread->reg.ecx--;
              }
          else if (pope)
            while (prep == 0 || thread->reg.ecx > 0)
              {
                guint16 temp;

                if (ia32_memory_read_word (thread->process,
                    ia32_cpu_seg (thread, thread->reg.esi, pseg), &temp) < 0
                    || ia32_memory_write_word (thread->process,
                                        ia32_cpu_seg (thread, thread->reg.edi,
                                        IA32_CPU_PSEG_ES), temp, level) < 0)
                  return IA32_ERR_GP;
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  {
                    thread->reg.esi += 2;
                    thread->reg.edi += 2;
                  }
                else
                  {
                    thread->reg.esi -= 2;
                    thread->reg.edi -= 2;
                  }
                if (prep == 0)
                  break;
                thread->reg.ecx--;
              }
          else
            while (prep == 0 || thread->reg.ecx > 0)
              {
                guint32 temp;

                if (ia32_memory_read_dword (thread->process,
                    ia32_cpu_seg (thread, thread->reg.esi, pseg), &temp) < 0
                    || ia32_memory_write_dword (thread->process,
                                        ia32_cpu_seg (thread, thread->reg.edi,
                                        IA32_CPU_PSEG_ES), temp, level) < 0)
                  return IA32_ERR_GP;
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  {
                    thread->reg.esi += 4;
                    thread->reg.edi += 4;
                  }
                else
                  {
                    thread->reg.esi -= 4;
                    thread->reg.edi -= 4;
                  }
                if (prep == 0)
                  break;
                thread->reg.ecx--;
              }
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xa6: /* cmpsb */
        case 0xa7: /* cmpsw/cmpsd */
          if (code == 0xa6)
            while (prep == 0 || thread->reg.ecx > 0)
              {
                guint8 v0, v1;

                if (ia32_memory_read_byte (thread->process,
                        ia32_cpu_seg (thread, thread->reg.esi, pseg), &v0) < 0
                    || ia32_memory_read_byte (thread->process,
                                        ia32_cpu_seg (thread, thread->reg.edi,
                                                IA32_CPU_PSEG_ES), &v1) < 0)
                  return IA32_ERR_GP;
                calc8 (thread, v0, v1, 7);
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  {
                    thread->reg.esi++;
                    thread->reg.edi++;
                  }
                else
                  {
                    thread->reg.esi--;
                    thread->reg.edi--;
                  }
                if (prep == 0)
                  break;
                thread->reg.ecx--;
                if ((prep == 0xf2
                            && flag_get (thread->reg.eflags, FLAG_ZF) != 0)
                    || (prep == 0xf3
                            && flag_get (thread->reg.eflags, FLAG_ZF) == 0))
                  break;
              }
          else if (pope)
            while (prep == 0 || thread->reg.ecx > 0)
              {
                guint16 v0, v1;

                if (ia32_memory_read_word (thread->process,
                        ia32_cpu_seg (thread, thread->reg.esi, pseg), &v0) < 0
                    || ia32_memory_read_word (thread->process,
                                        ia32_cpu_seg (thread, thread->reg.edi,
                                                IA32_CPU_PSEG_ES), &v1) < 0)
                  return IA32_ERR_GP;
                calc16 (thread, v0, v1, 7);
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  {
                    thread->reg.esi += 2;
                    thread->reg.edi += 2;
                  }
                else
                  {
                    thread->reg.esi -= 2;
                    thread->reg.edi -= 2;
                  }
                if (prep == 0)
                  break;
                thread->reg.ecx--;
                if ((prep == 0xf2
                            && flag_get (thread->reg.eflags, FLAG_ZF) != 0)
                    || (prep == 0xf3
                            && flag_get (thread->reg.eflags, FLAG_ZF) == 0))
                  break;
              }
          else
            while (prep == 0 || thread->reg.ecx > 0)
              {
                guint32 v0, v1;

                if (ia32_memory_read_dword (thread->process,
                        ia32_cpu_seg (thread, thread->reg.esi, pseg), &v0) < 0
                    || ia32_memory_read_dword (thread->process,
                                        ia32_cpu_seg (thread, thread->reg.edi,
                                                IA32_CPU_PSEG_ES), &v1) < 0)
                  return IA32_ERR_GP;
                calc32 (thread, v0, v1, 7);
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  {
                    thread->reg.esi += 4;
                    thread->reg.edi += 4;
                  }
                else
                  {
                    thread->reg.esi -= 4;
                    thread->reg.edi -= 4;
                  }
                if (prep == 0)
                  break;
                thread->reg.ecx--;
                if ((prep == 0xf2
                            && flag_get (thread->reg.eflags, FLAG_ZF) != 0)
                    || (prep == 0xf3
                            && flag_get (thread->reg.eflags, FLAG_ZF) == 0))
                  break;
              }
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xa8: /* test */
        case 0xa9:
          {
            guint32 result;

            if (code == 0xa8)
              {
                guint8 imm8;

                if (ia32_memory_read_byte (thread->process,
                                                        eip + 1, &imm8) < 0)
                  return IA32_ERR_GP;
                result = (gint8)(reg_get_l (thread->reg.eax) & imm8);
                thread->reg.eip = eip + 2;
              }
            else if (pope)
              {
                guint16 imm16;

                if (ia32_memory_read_word (thread->process,
                                                        eip + 1, &imm16) < 0)
                  return IA32_ERR_GP;
                result = (gint16)(reg_get_x (thread->reg.eax) & imm16);
                thread->reg.eip = eip + 3;
              }
            else
              {
                guint32 imm32;

                if (ia32_memory_read_dword (thread->process,
                                                        eip + 1, &imm32) < 0)
                  return IA32_ERR_GP;
                result = thread->reg.eax & imm32;
                thread->reg.eip = eip + 5;
              }
            flag_clr (thread->reg.eflags, FLAG_CF);
            flag_clr (thread->reg.eflags, FLAG_OF);
            flag_put (thread->reg.eflags, FLAG_PF, flag_parity (result));
            flag_put (thread->reg.eflags, FLAG_ZF, result == 0);
            flag_put (thread->reg.eflags, FLAG_SF, result & 0x80000000);
          }
          goto loop;
        case 0xaa: /* stosb */
        case 0xab: /* stosw/stosd */
          if (code == 0xaa)
            while (prep == 0 || thread->reg.ecx > 0)
              {
                if (ia32_memory_write_byte (thread->process,
                    ia32_cpu_seg (thread, thread->reg.edi, IA32_CPU_PSEG_ES),
                                    reg_get_l (thread->reg.eax), level) < 0)
                  return IA32_ERR_GP;
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  thread->reg.edi++;
                else
                  thread->reg.edi--;
                if (prep == 0)
                  break;
                thread->reg.ecx--;
              }
          else if (pope)
            while (prep == 0 || thread->reg.ecx > 0)
              {
                if (ia32_memory_write_word (thread->process,
                    ia32_cpu_seg (thread, thread->reg.edi, IA32_CPU_PSEG_ES),
                                    reg_get_x (thread->reg.eax), level) < 0)
                  return IA32_ERR_GP;
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  thread->reg.edi += 2;
                else
                  thread->reg.edi -= 2;
                if (prep == 0)
                  break;
                thread->reg.ecx--;
              }
          else
            while (prep == 0 || thread->reg.ecx > 0)
              {
                if (ia32_memory_write_dword (thread->process,
                    ia32_cpu_seg (thread, thread->reg.edi, IA32_CPU_PSEG_ES),
                                                thread->reg.eax, level) < 0)
                  return IA32_ERR_GP;
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  thread->reg.edi += 4;
                else
                  thread->reg.edi -= 4;
                if (prep == 0)
                  break;
                thread->reg.ecx--;
              }
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xac: /* lodsb */
        case 0xad: /* lodsw/lodsd */
          if (code == 0xac)
            while (prep == 0 || thread->reg.ecx > 0)
              {
                guint8 temp;

                if (ia32_memory_read_byte (thread->process,
                    ia32_cpu_seg (thread, thread->reg.esi, pseg), &temp) < 0)
                  return IA32_ERR_GP;
                reg_set_l (thread->reg.eax, temp);
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  thread->reg.esi++;
                else
                  thread->reg.esi--;
                if (prep == 0)
                  break;
                thread->reg.ecx--;
              }
          else if (pope)
            while (prep == 0 || thread->reg.ecx > 0)
              {
                guint16 temp;

                if (ia32_memory_read_word (thread->process,
                    ia32_cpu_seg (thread, thread->reg.esi, pseg), &temp) < 0)
                  return IA32_ERR_GP;
                reg_set_x (thread->reg.eax, temp);
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  thread->reg.esi += 2;
                else
                  thread->reg.esi -= 2;
                if (prep == 0)
                  break;
                thread->reg.ecx--;
              }
          else
            while (prep == 0 || thread->reg.ecx > 0)
              {
                if (ia32_memory_read_dword (thread->process,
                                ia32_cpu_seg (thread, thread->reg.esi, pseg),
                                                        &thread->reg.eax) < 0)
                  return IA32_ERR_GP;
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                  thread->reg.esi += 4;
                else
                  thread->reg.esi -= 4;
                if (prep == 0)
                  break;
                thread->reg.ecx--;
              }
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xae: /* scasb */
        case 0xaf: /* scasw/scasd */
          if (code == 0xae)
            while (prep == 0 || thread->reg.ecx > 0)
              {
                guint8 value;

                if (ia32_memory_read_byte (thread->process,
                                        ia32_cpu_seg (thread, thread->reg.edi,
                                                IA32_CPU_PSEG_ES), &value) < 0)
                  return IA32_ERR_GP;
                calc8 (thread, reg_get_l (thread->reg.eax), value, 7);
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                    thread->reg.edi++;
                else
                    thread->reg.edi--;
                if (prep == 0)
                  break;
                thread->reg.ecx--;
                if ((prep == 0xf2
                            && flag_get (thread->reg.eflags, FLAG_ZF) != 0)
                    || (prep == 0xf3
                            && flag_get (thread->reg.eflags, FLAG_ZF) == 0))
                  break;
              }
          else if (pope)
            while (prep == 0 || thread->reg.ecx > 0)
              {
                guint16 value;

                if (ia32_memory_read_word (thread->process,
                                        ia32_cpu_seg (thread, thread->reg.edi,
                                                IA32_CPU_PSEG_ES), &value) < 0)
                  return IA32_ERR_GP;
                calc16 (thread, reg_get_x (thread->reg.eax), value, 7);
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                    thread->reg.edi += 2;
                else
                    thread->reg.edi -= 2;
                if (prep == 0)
                  break;
                thread->reg.ecx--;
                if ((prep == 0xf2
                            && flag_get (thread->reg.eflags, FLAG_ZF) != 0)
                    || (prep == 0xf3
                            && flag_get (thread->reg.eflags, FLAG_ZF) == 0))
                  break;
              }
          else
            while (prep == 0 || thread->reg.ecx > 0)
              {
                guint32 value;

                if (ia32_memory_read_dword (thread->process,
                                        ia32_cpu_seg (thread, thread->reg.edi,
                                                IA32_CPU_PSEG_ES), &value) < 0)
                  return IA32_ERR_GP;
                calc32 (thread, thread->reg.eax, value, 7);
                if (flag_get (thread->reg.eflags, FLAG_DF) == 0)
                    thread->reg.edi += 4;
                else
                    thread->reg.edi -= 4;
                if (prep == 0)
                  break;
                thread->reg.ecx--;
                if ((prep == 0xf2
                            && flag_get (thread->reg.eflags, FLAG_ZF) != 0)
                    || (prep == 0xf3
                            && flag_get (thread->reg.eflags, FLAG_ZF) == 0))
                  break;
              }
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xb0: case 0xb1: case 0xb2: case 0xb3: /* mov */
        case 0xb4: case 0xb5: case 0xb6: case 0xb7:
          {
            guint8 imm8;

            if (ia32_memory_read_byte (thread->process, eip + 1, &imm8) < 0)
              return IA32_ERR_GP;
            reg_set8 (thread, code, imm8);
          }
          thread->reg.eip = eip + 2;
          goto loop;
        case 0xb8: case 0xb9: case 0xba: case 0xbb: /* mov */
        case 0xbc: case 0xbd: case 0xbe: case 0xbf:
          if (pope)
            {
              guint16 imm16;

              if (ia32_memory_read_word (thread->process, eip + 1, &imm16) < 0)
                return IA32_ERR_GP;
              reg_set16 (thread, code, imm16);
              thread->reg.eip = eip + 3;
            }
          else
            {
              guint32 imm32;

              if (ia32_memory_read_dword (thread->process,
                                                        eip + 1, &imm32) < 0)
                return IA32_ERR_GP;
              reg_set32 (thread, code, imm32);
              thread->reg.eip = eip + 5;
            }
          goto loop;
        case 0xc0: case 0xc1: /* rol/ror/rcl/rcr/shl/shr/sal/sar */
        case 0xd0: case 0xd1: case 0xd2: case 0xd3:
          {
            guint8 count = 1;
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if (code == 0xc0 || code == 0xc1)
              {
                if (ia32_memory_read_byte (thread->process,
                                                eip + m.leng + 1, &count) < 0)
                  return IA32_ERR_GP;
              }
            else if (code == 0xd2 || code == 0xd3)
              {
                count = reg_get_l (thread->reg.ecx);
              }
            count &= 31;
            if ((code & 1) == 0)
              {
                gint i;
                guint8 value;

                if (!modrm_get8 (thread, &m, &value))
                  return IA32_ERR_GP;
                for (i = 0; i < count; i++)
                  switch (m.regop)
                    {
                      case 0: /* rol */
                        flag_put (thread->reg.eflags, FLAG_CF, value & 0x80);
                        value = (value << 1)
                                    | flag_get (thread->reg.eflags, FLAG_CF);
                        break;
                      case 1: /* ror */
                        flag_put (thread->reg.eflags, FLAG_CF, value & 0x01);
                        value = (value >> 1)
                                    | (flag_get (thread->reg.eflags, FLAG_CF)
                                                                ? 0x80 : 0x00);
                        break;
                      case 2: /* rcl */
                        {
                          guint8 cf;

                          cf = flag_get (thread->reg.eflags, FLAG_CF);
                          flag_put (thread->reg.eflags, FLAG_CF, value & 0x80);
                          value = (value << 1) | cf;
                        }
                        break;
                      case 3: /* rcr */
                        {
                          guint8 cf;

                          cf = flag_get (thread->reg.eflags, FLAG_CF);
                          flag_put (thread->reg.eflags, FLAG_CF, value & 0x01);
                          value = (value >> 1) | (cf ? 0x80 : 0x00);
                        }
                        break;
                      case 4: /* shl */
                      case 6: /* sal */
                        flag_put (thread->reg.eflags, FLAG_CF, value & 0x80);
                        value <<= 1;
                        break;
                      case 5: /* shr */
                        flag_put (thread->reg.eflags, FLAG_CF, value & 0x01);
                        value >>= 1;
                        break;
                      case 7: /* sar */
                        flag_put (thread->reg.eflags, FLAG_CF, value & 0x01);
                        value = (gint8)value >> 1;
                    }
                if (count == 1)
                  {
                    if ((m.regop & 1) == 0)
                      flag_put (thread->reg.eflags, FLAG_OF, (value >> 7)
                                    ^ flag_get (thread->reg.eflags, FLAG_CF));
                    else
                      flag_put (thread->reg.eflags, FLAG_OF,
                                            (value >> 7) ^ ((value >> 6) & 1));
                  }
                if ((m.regop & 4) != 0 && count > 0)
                  {
                    flag_put (thread->reg.eflags, FLAG_PF,
                                                        flag_parity (value));
                    flag_put (thread->reg.eflags, FLAG_ZF, value == 0);
                    flag_put (thread->reg.eflags, FLAG_SF, value & 0x80);
                  }
                if (!modrm_set8 (thread, &m, value, level))
                  return IA32_ERR_GP;
              }
            else if (pope)
              {
                gint i;
                guint16 value;

                if (!modrm_get16 (thread, &m, &value))
                  return IA32_ERR_GP;
                for (i = 0; i < count; i++)
                  switch (m.regop)
                    {
                      case 0: /* rol */
                        flag_put (thread->reg.eflags, FLAG_CF, value & 0x8000);
                        value = (value << 1)
                                    | flag_get (thread->reg.eflags, FLAG_CF);
                        break;
                      case 1: /* ror */
                        flag_put (thread->reg.eflags, FLAG_CF, value & 0x0001);
                        value = (value >> 1)
                                    | (flag_get (thread->reg.eflags, FLAG_CF)
                                                            ? 0x8000 : 0x0000);
                        break;
                      case 2: /* rcl */
                        {
                          guint16 cf;

                          cf = flag_get (thread->reg.eflags, FLAG_CF);
                          flag_put (thread->reg.eflags, FLAG_CF,
                                                            value & 0x8000);
                          value = (value << 1) | cf;
                        }
                        break;
                      case 3: /* rcr */
                        {
                          guint16 cf;

                          cf = flag_get (thread->reg.eflags, FLAG_CF);
                          flag_put (thread->reg.eflags, FLAG_CF,
                                                            value & 0x0001);
                          value = (value >> 1) | (cf ? 0x8000 : 0x0000);
                        }
                        break;
                      case 4: /* shl */
                      case 6: /* sal */
                        flag_put (thread->reg.eflags, FLAG_CF, value & 0x8000);
                        value <<= 1;
                        break;
                      case 5: /* shr */
                        flag_put (thread->reg.eflags, FLAG_CF, value & 0x0001);
                        value >>= 1;
                        break;
                      case 7: /* sar */
                        flag_put (thread->reg.eflags, FLAG_CF, value & 0x0001);
                        value = (gint16)value >> 1;
                    }
                if (count == 1)
                  {
                    if ((m.regop & 1) == 0)
                      flag_put (thread->reg.eflags, FLAG_OF, (value >> 15)
                                    ^ flag_get (thread->reg.eflags, FLAG_CF));
                    else
                      flag_put (thread->reg.eflags, FLAG_OF,
                                        (value >> 15) ^ ((value >> 14) & 1));
                  }
                if ((m.regop & 4) != 0 && count > 0)
                  {
                    flag_put (thread->reg.eflags, FLAG_PF,
                                                        flag_parity (value));
                    flag_put (thread->reg.eflags, FLAG_ZF, value == 0);
                    flag_put (thread->reg.eflags, FLAG_SF, value & 0x8000);
                  }
                if (!modrm_set16 (thread, &m, value, level))
                  return IA32_ERR_GP;
              }
            else
              {
                gint i;
                guint32 value;

                if (!modrm_get32 (thread, &m, &value))
                  return IA32_ERR_GP;
                for (i = 0; i < count; i++)
                  switch (m.regop)
                    {
                      case 0: /* rol */
                        flag_put (thread->reg.eflags, FLAG_CF,
                                                        value & 0x80000000);
                        value = (value << 1)
                                    | flag_get (thread->reg.eflags, FLAG_CF);
                        break;
                      case 1: /* ror */
                        flag_put (thread->reg.eflags, FLAG_CF,
                                                        value & 0x00000001);
                        value = (value >> 1)
                                    | (flag_get (thread->reg.eflags, FLAG_CF)
                                                    ? 0x80000000 : 0x00000000);
                        break;
                      case 2: /* rcl */
                        {
                          guint32 cf;

                          cf = flag_get (thread->reg.eflags, FLAG_CF);
                          flag_put (thread->reg.eflags, FLAG_CF,
                                                        value & 0x80000000);
                          value = (value << 1) | cf;
                        }
                        break;
                      case 3: /* rcr */
                        {
                          guint32 cf;

                          cf = flag_get (thread->reg.eflags, FLAG_CF);
                          flag_put (thread->reg.eflags, FLAG_CF,
                                                        value & 0x00000001);
                          value = (value >> 1)
                                            | (cf ? 0x80000000 : 0x00000000);
                        }
                        break;
                      case 4: /* shl */
                      case 6: /* sal */
                        flag_put (thread->reg.eflags, FLAG_CF,
                                                        value & 0x80000000);
                        value <<= 1;
                        break;
                      case 5: /* shr */
                        flag_put (thread->reg.eflags, FLAG_CF,
                                                        value & 0x00000001);
                        value >>= 1;
                        break;
                      case 7: /* sar */
                        flag_put (thread->reg.eflags, FLAG_CF,
                                                        value & 0x00000001);
                        value = (gint32)value >> 1;
                    }
                if (count == 1)
                  {
                    if ((m.regop & 1) == 0)
                      flag_put (thread->reg.eflags, FLAG_OF, (value >> 31)
                                    ^ flag_get (thread->reg.eflags, FLAG_CF));
                    else
                      flag_put (thread->reg.eflags, FLAG_OF,
                                        (value >> 31) ^ ((value >> 30) & 1));
                  }
                if ((m.regop & 4) != 0 && count > 0)
                  {
                    flag_put (thread->reg.eflags, FLAG_PF,
                                                        flag_parity (value));
                    flag_put (thread->reg.eflags, FLAG_ZF, value == 0);
                    flag_put (thread->reg.eflags, FLAG_SF, value & 0x80000000);
                  }
                if (!modrm_set32 (thread, &m, value, level))
                  return IA32_ERR_GP;
              }
            thread->reg.eip = eip + m.leng + 1;
            if (code == 0xc0 || code == 0xc1)
              thread->reg.eip++;
          }
          goto loop;
        case 0xc2: /* retn */
          {
            guint16 imm16;

            if (code == 0xc2 && ia32_memory_read_word (thread->process,
                                                        eip + 1, &imm16) < 0)
              return IA32_ERR_GP;
            if (pope)
              {
                guint16 value;

                if (!ia32_stack_pop16 (thread, &value))
                  return IA32_ERR_SS;
                reg_set_x (thread->reg.eip, value);
              }
            else
              {
                if (!ia32_stack_pop32 (thread, &thread->reg.eip))
                  return IA32_ERR_SS;
              }
            if (padr)
              reg_set_x (thread->reg.esp, reg_get_x (thread->reg.esp) + imm16);
            else
              thread->reg.esp += imm16;
          }
          goto loop;
        case 0xc3: /* ret */
          if (pope)
            {
              guint16 value;

              if (!ia32_stack_pop16 (thread, &value))
                return IA32_ERR_SS;
              reg_set_x (thread->reg.eip, value);
            }
          else
            {
              if (!ia32_stack_pop32 (thread, &thread->reg.eip))
                return IA32_ERR_SS;
            }
          goto loop;
        case 0xc6: /* mov */
        case 0xc7:
          {
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if (m.regop != 0)
              return IA32_ERR_UD;
            if (code == 0xc6)
              {
                guint8 imm8;

                if (ia32_memory_read_byte (thread->process,
                                                eip + m.leng + 1, &imm8) < 0
                                    || !modrm_set8 (thread, &m, imm8, level))
                  return IA32_ERR_GP;
                thread->reg.eip = eip + m.leng + 2;
              }
            else if (pope)
              {
                guint16 imm16;

                if (ia32_memory_read_word (thread->process,
                                                eip + m.leng + 1, &imm16) < 0
                                    || !modrm_set16 (thread, &m, imm16, level))
                  return IA32_ERR_GP;
                thread->reg.eip = eip + m.leng + 3;
              }
            else
              {
                guint32 imm32;

                if (ia32_memory_read_dword (thread->process,
                                                eip + m.leng + 1, &imm32) < 0
                                    || !modrm_set32 (thread, &m, imm32, level))
                  return IA32_ERR_GP;
                thread->reg.eip = eip + m.leng + 5;
              }
          }
          goto loop;
        case 0xc8: /* enter */
          {
            guint8 imm8;
            guint16 imm16;

            if (ia32_memory_read_word (thread->process, eip + 1, &imm16) < 0
                || ia32_memory_read_byte (thread->process, eip + 3, &imm8) < 0)
              return IA32_ERR_GP;
            imm8 &= 31;
            if (pope)
              {
                guint16 frame;

                if (!ia32_stack_push16 (thread, reg_get_x (thread->reg.ebp)))
                  return IA32_ERR_SS;
                frame = reg_get_x (thread->reg.esp);
                if (imm8 > 0)
                  {
                    gint i;

                    for (i = 1; i < imm8; i++)
                      {
                        guint16 temp;

                        thread->reg.ebp -= 2;
                        if (ia32_memory_read_word (thread->process,
                                        ia32_cpu_seg (thread, thread->reg.ebp,
                                                IA32_CPU_PSEG_SS), &temp) < 0)
                          return IA32_ERR_GP;
                        if (!ia32_stack_push16 (thread, temp))
                          return IA32_ERR_SS;
                      }
                    if (!ia32_stack_push16 (thread, frame))
                      return IA32_ERR_SS;
                  }
                reg_set_x (thread->reg.ebp, frame);
                reg_set_x (thread->reg.esp,
                                        reg_get_x (thread->reg.ebp) - imm16);
              }
            else
              {
                guint32 frame;

                if (!ia32_stack_push32 (thread, thread->reg.ebp))
                  return IA32_ERR_SS;
                frame = thread->reg.esp;
                if (imm8 > 0)
                  {
                    gint i;

                    for (i = 1; i < imm8; i++)
                      {
                        guint32 temp;

                        thread->reg.ebp -= 4;
                        if (ia32_memory_read_dword (thread->process,
                                        ia32_cpu_seg (thread, thread->reg.ebp,
                                                IA32_CPU_PSEG_SS), &temp) < 0)
                          return IA32_ERR_GP;
                        if (!ia32_stack_push32 (thread, temp))
                          return IA32_ERR_SS;
                      }
                    if (!ia32_stack_push32 (thread, frame))
                      return IA32_ERR_SS;
                  }
                thread->reg.ebp = frame;
                thread->reg.esp = thread->reg.ebp - imm16;
              }
          }
          thread->reg.eip = eip + 4;
          goto loop;
        case 0xc9: /* leave */
          if (pope)
            {
              guint16 value;

              reg_set_x (thread->reg.esp, reg_get_x (thread->reg.ebp));
              if (!ia32_stack_pop16 (thread, &value))
                return IA32_ERR_SS;
              reg_set_x (thread->reg.ebp, value);
            }
          else
            {
              thread->reg.esp = thread->reg.ebp;
              if (!ia32_stack_pop32 (thread, &thread->reg.ebp))
                return IA32_ERR_SS;
            }
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xd4: /* aam */
        case 0xd5: /* aad */
          {
            guint8 imm8;

            if (ia32_memory_read_byte (thread->process, eip + 1, &imm8) < 0
                                                                || imm8 == 0)
              return IA32_ERR_GP;
            if (code == 0xd4)
              {
                guint8 al;

                al = reg_get_l (thread->reg.eax);
                reg_set_h (thread->reg.eax, al / imm8);
                reg_set_l (thread->reg.eax, al % imm8);
              }
            else
              {
                reg_set_l (thread->reg.eax, reg_get_l (thread->reg.eax)
                                        + reg_get_h (thread->reg.eax) * imm8);
                reg_set_h (thread->reg.eax, 0);
              }
            flag_put (thread->reg.eflags, FLAG_PF,
                                                flag_parity (thread->reg.eax));
            flag_put (thread->reg.eflags, FLAG_ZF,
                                            reg_get_l (thread->reg.eax) == 0);
            flag_put (thread->reg.eflags, FLAG_SF, thread->reg.eax & 0x80);
          }
          thread->reg.eip = eip + 2;
          goto loop;
        case 0xcc: /* int 3 */
          return IA32_ERR_BP;
        case 0xd6: /* setalc */
          reg_set_l (thread->reg.eax,
                    flag_get (thread->reg.eflags, FLAG_CF) == 0 ? 0x00 : 0xff);
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xd7: /* xlatb */
          {
            guint8 temp;

            
            if (ia32_memory_read_byte (thread->process, ia32_cpu_seg (thread,
                        (padr ? reg_get_x (thread->reg.ebx) : thread->reg.ebx)
                            + reg_get_l (thread->reg.eax), pseg), &temp) < 0)
              return IA32_ERR_GP;
            reg_set_l (thread->reg.eax, temp);
          }
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xd8: case 0xd9: case 0xda: case 0xdb: /* x87 FPU */
        case 0xdc: case 0xdd: case 0xde: case 0xdf:
          {
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            thread->reg.eip = eip + m.leng + 1;
          }
          goto loop;
        case 0xe0: /* loopnz */
        case 0xe1: /* loopz */
        case 0xe2: /* loop */
          if (padr)
            reg_set_x (thread->reg.ecx, reg_get_x (thread->reg.ecx) - 1);
          else
            thread->reg.ecx--;
        case 0xe3: /* jecxz */
          {
            gboolean result;
            guint8 imm8;

            if (ia32_memory_read_byte (thread->process, eip + 1, &imm8) < 0)
              return IA32_ERR_GP;
            result = padr ? reg_get_x (thread->reg.ecx) != 0
                          : thread->reg.ecx != 0;
            switch (code)
              {
                case 0xe0:
                  if (flag_get (thread->reg.eflags, FLAG_ZF) != 0)
                    result = FALSE;
                  break;
                case 0xe1:
                  if (flag_get (thread->reg.eflags, FLAG_ZF) == 0)
                    result = FALSE;
                  break;
                case 0xe3:
                    result = !result;
              }
            thread->reg.eip = eip + 2;
            if (result)
              thread->reg.eip += (gint8)imm8;
          }
          goto loop;
        case 0xe4: /* in */
        case 0xe5:
        case 0xe6: /* out */
        case 0xe7:
          thread->reg.eip = eip + 2;
          goto loop;
        case 0xe8: /* call */
          if (!(pope ? ia32_stack_push16 (thread, eip + 3)
                     : ia32_stack_push32 (thread, eip + 5)))
            return IA32_ERR_SS;
        case 0xe9: /* jmp */
          if (pope)
            {
              guint16 imm16;

              if (ia32_memory_read_word (thread->process, eip + 1, &imm16) < 0)
                return IA32_ERR_GP;
              thread->reg.eip = eip + 3 + (gint16)imm16;
            }
          else
            {
              guint32 imm32;

              if (ia32_memory_read_dword (thread->process,
                                                        eip + 1, &imm32) < 0)
                return IA32_ERR_GP;
              thread->reg.eip = eip + 5 + imm32;
            }
          goto loop;
        case 0xeb: /* jmp */
          {
            guint8 imm8;

            if (ia32_memory_read_byte (thread->process, eip + 1, &imm8) < 0)
              return IA32_ERR_GP;
            thread->reg.eip = eip + 2 + (gint8)imm8;
          }
          goto loop;
        case 0xec: /* in */
        case 0xed:
        case 0xee: /* out */
        case 0xef:
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xf0: /* lock */
        case 0xf2: /* repnz */
        case 0xf3: /* rep/repz */
          prep = code;
          eip++;
          break;
        case 0xf1: /* icebp */
          return IA32_ERR_DB;
        case 0xf5: /* cmc */
          flag_put (thread->reg.eflags, FLAG_CF,
                                flag_get (thread->reg.eflags, FLAG_CF) == 0);
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xf6: /* test/not/neg/mul/imul/div/idiv */
        case 0xf7:
          {
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            switch (m.regop)
              {
                case 0: /* test */
                  {
                    guint32 result;

                    if (code == 0xf6)
                      {
                        guint8 value0, value1;

                        if (!modrm_get8 (thread, &m, &value0))
                          return IA32_ERR_GP;
                        if (ia32_memory_read_byte (thread->process,
                                                eip + m.leng + 1, &value1) < 0)
                          return IA32_ERR_GP;
                        result = (gint8)(value0 & value1);
                        thread->reg.eip = eip + m.leng + 2;
                      }
                    else if (pope)
                      {
                        guint16 value0, value1;

                        if (!modrm_get16 (thread, &m, &value0))
                          return IA32_ERR_GP;
                        if (ia32_memory_read_word (thread->process,
                                                eip + m.leng + 1, &value1) < 0)
                          return IA32_ERR_GP;
                        result = (gint16)(value0 & value1);
                        thread->reg.eip = eip + m.leng + 3;
                      }
                    else
                      {
                        guint32 value0, value1;

                        if (!modrm_get32 (thread, &m, &value0))
                          return IA32_ERR_GP;
                        if (ia32_memory_read_dword (thread->process,
                                                eip + m.leng + 1, &value1) < 0)
                          return IA32_ERR_GP;
                        result = value0 & value1;
                        thread->reg.eip = eip + m.leng + 5;
                      }
                    flag_clr (thread->reg.eflags, FLAG_CF);
                    flag_clr (thread->reg.eflags, FLAG_OF);
                    flag_put (thread->reg.eflags, FLAG_PF,
                                                        flag_parity (result));
                    flag_put (thread->reg.eflags, FLAG_ZF, result == 0);
                    flag_put (thread->reg.eflags, FLAG_SF,
                                                        result & 0x80000000);
                  }
                  break;
                case 2: /* not */
                  if (code == 0xf6)
                    {
                      guint8 value;

                      if (!modrm_get8 (thread, &m, &value))
                        return IA32_ERR_GP;
                      value = ~value;
                      if (!modrm_set8 (thread, &m, value, level))
                        return IA32_ERR_GP;
                    }
                  else if (pope)
                    {
                      guint16 value;

                      if (!modrm_get16 (thread, &m, &value))
                        return IA32_ERR_GP;
                      value = ~value;
                      if (!modrm_set16 (thread, &m, value, level))
                        return IA32_ERR_GP;
                    }
                  else
                    {
                      guint32 value;

                      if (!modrm_get32 (thread, &m, &value))
                        return IA32_ERR_GP;
                      value = ~value;
                      if (!modrm_set32 (thread, &m, value, level))
                        return IA32_ERR_GP;
                    }
                  thread->reg.eip = eip + m.leng + 1;
                  break;
                case 3: /* neg */
                  {
                    guint32 result;

                    if (code == 0xf6)
                      {
                        guint8 value;

                        if (!modrm_get8 (thread, &m, &value))
                          return IA32_ERR_GP;
                        flag_put (thread->reg.eflags, FLAG_OF,
                                                    (gint8)value == G_MININT8);
                        value = -(gint8)value;
                        if (!modrm_set8 (thread, &m, value, level))
                          return IA32_ERR_GP;
                        result = (gint8)value;
                      }
                    else if (pope)
                      {
                        guint16 value;

                        if (!modrm_get16 (thread, &m, &value))
                          return IA32_ERR_GP;
                        flag_put (thread->reg.eflags, FLAG_OF,
                                                (gint16)value == G_MININT16);
                        value = -(gint16)value;
                        if (!modrm_set16 (thread, &m, value, level))
                          return IA32_ERR_GP;
                        result = (gint16)value;
                      }
                    else
                      {
                        guint32 value;

                        if (!modrm_get32 (thread, &m, &value))
                          return IA32_ERR_GP;
                        flag_put (thread->reg.eflags, FLAG_OF,
                                                (gint32)value == G_MININT32);
                        value = -(gint32)value;
                        if (!modrm_set32 (thread, &m, value, level))
                          return IA32_ERR_GP;
                        result = (gint32)value;
                      }
                    flag_put (thread->reg.eflags, FLAG_CF, result != 0);
                    flag_put (thread->reg.eflags, FLAG_PF,
                                                        flag_parity (result));
                    flag_put (thread->reg.eflags, FLAG_ZF, result == 0);
                    flag_put (thread->reg.eflags, FLAG_SF,
                                                        result & 0x80000000);
                    thread->reg.eip = eip + m.leng + 1;
                  }
                  break;
                case 4: /* mul */
                  {
                    gboolean result;

                    if (code == 0xf6)
                      {
                        guint8 value;

                        if (!modrm_get8 (thread, &m, &value))
                          return IA32_ERR_GP;
                        reg_set_x (thread->reg.eax,
                                        reg_get_l (thread->reg.eax) * value);
                        result = reg_get_h (thread->reg.eax) != 0;
                      }
                    else if (pope)
                      {
                        guint16 value;
                        guint32 temp;

                        if (!modrm_get16 (thread, &m, &value))
                          return IA32_ERR_GP;
                        temp = reg_get_x (thread->reg.eax) * value;
                        reg_set_x (thread->reg.eax, temp);
                        reg_set_x (thread->reg.edx, temp >> 16);
                        result = reg_get_x (thread->reg.edx) != 0;
                      }
                    else
                      {
                        guint32 value;
                        guint64 temp;

                        if (!modrm_get32 (thread, &m, &value))
                          return IA32_ERR_GP;
                        temp = thread->reg.eax * value;
                        thread->reg.eax = temp;
                        thread->reg.edx = temp >> 32;
                        result = thread->reg.edx != 0;
                      }
                    if (result)
                      {
                        flag_set (thread->reg.eflags, FLAG_CF);
                        flag_set (thread->reg.eflags, FLAG_OF);
                      }
                    else
                      {
                        flag_clr (thread->reg.eflags, FLAG_CF);
                        flag_clr (thread->reg.eflags, FLAG_OF);
                      }
                    thread->reg.eip = eip + m.leng + 1;
                  }
                  break;
                case 5: /* imul */
                  {
                    gboolean result;

                    if (code == 0xf6)
                      {
                        guint8 value;

                        if (!modrm_get8 (thread, &m, &value))
                          return IA32_ERR_GP;
                        reg_set_x (thread->reg.eax,
                            (gint8)reg_get_l (thread->reg.eax) * (gint8)value);
                        result = reg_get_h (thread->reg.eax) != 0x00
                                        && reg_get_h (thread->reg.eax) != 0xff;
                      }
                    else if (pope)
                      {
                        guint16 value;
                        guint32 temp;

                        if (!modrm_get16 (thread, &m, &value))
                          return IA32_ERR_GP;
                        temp = (gint16)reg_get_x (thread->reg.eax)
                                                            * (gint16)value;
                        reg_set_x (thread->reg.eax, temp);
                        reg_set_x (thread->reg.edx, temp >> 16);
                        result = reg_get_x (thread->reg.eax) != 0x0000
                                    && reg_get_x (thread->reg.eax) != 0xffff;
                      }
                    else
                      {
                        guint32 value;
                        guint64 temp;

                        if (!modrm_get32 (thread, &m, &value))
                          return IA32_ERR_GP;
                        temp = (gint32)thread->reg.eax * (gint32)value;
                        thread->reg.eax = temp;
                        thread->reg.edx = temp >> 32;
                        result = thread->reg.edx != 0x00000000
                                            && thread->reg.edx != 0xffffffff;
                      }
                    if (result)
                      {
                        flag_set (thread->reg.eflags, FLAG_CF);
                        flag_set (thread->reg.eflags, FLAG_OF);
                      }
                    else
                      {
                        flag_clr (thread->reg.eflags, FLAG_CF);
                        flag_clr (thread->reg.eflags, FLAG_OF);
                      }
                    thread->reg.eip = eip + m.leng + 1;
                  }
                  break;
                case 6: /* div */
                  if (code == 0xf6)
                    {
                      guint8 value;

                      if (!modrm_get8 (thread, &m, &value) || value == 0)
                        return IA32_ERR_GP;
                      reg_set_l (thread->reg.eax,
                                        reg_get_x (thread->reg.eax) / value);
                      reg_set_h (thread->reg.eax,
                                        reg_get_x (thread->reg.eax) % value);
                    }
                  else if (pope)
                    {
                      guint16 value;
                      guint32 temp;

                      if (!modrm_get16 (thread, &m, &value) || value == 0)
                        return IA32_ERR_GP;
                      temp = reg_get_x (thread->reg.eax)
                                | ((guint32)reg_get_x (thread->reg.edx) << 16);
                      reg_set_x (thread->reg.eax, temp / value);
                      reg_set_x (thread->reg.edx, temp % value);
                    }
                  else
                    {
                      guint32 value;
                      guint64 temp;

                      if (!modrm_get32 (thread, &m, &value) || value == 0)
                        return IA32_ERR_GP;
                      temp = reg_get_x (thread->reg.eax)
                                | ((guint64)reg_get_x (thread->reg.edx) << 32);
                      thread->reg.eax = temp / value;
                      thread->reg.edx = temp % value;
                    }
                  thread->reg.eip = eip + m.leng + 1;
                  break;
                case 7: /* idiv */
                  if (code == 0xf6)
                    {
                      guint8 value;

                      if (!modrm_get8 (thread, &m, &value) || value == 0)
                        return IA32_ERR_GP;
                      reg_set_l (thread->reg.eax,
                        (gint16)reg_get_x (thread->reg.eax) / (gint8)value);
                      reg_set_h (thread->reg.eax,
                        (gint16)reg_get_x (thread->reg.eax) % (gint8)value);
                    }
                  else if (pope)
                    {
                      guint16 value;
                      gint32 temp;

                      if (!modrm_get16 (thread, &m, &value) || value == 0)
                        return IA32_ERR_GP;
                      temp = reg_get_x (thread->reg.eax)
                                | ((gint32)reg_get_x (thread->reg.edx) << 16);
                      reg_set_x (thread->reg.eax, temp / (gint16)value);
                      reg_set_x (thread->reg.edx, temp % (gint16)value);
                    }
                  else
                    {
                      guint32 value;
                      gint64 temp;

                      if (!modrm_get32 (thread, &m, &value) || value == 0)
                        return IA32_ERR_GP;
                      temp = reg_get_x (thread->reg.eax)
                                | ((gint64)reg_get_x (thread->reg.edx) << 32);
                      thread->reg.eax = temp / (gint32)value;
                      thread->reg.edx = temp % (gint32)value;
                    }
                  thread->reg.eip = eip + m.leng + 1;
                  break;
                default:
                  return IA32_ERR_UD;
              }
          }
          goto loop;
        case 0xf8: /* clc */
          flag_clr (thread->reg.eflags, FLAG_CF);
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xf9: /* stc */
          flag_set (thread->reg.eflags, FLAG_CF);
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xfa: /* cli */
          flag_clr (thread->reg.eflags, FLAG_IF);
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xfb: /* sti */
          flag_set (thread->reg.eflags, FLAG_IF);
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xfc: /* cld */
          flag_clr (thread->reg.eflags, FLAG_DF);
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xfd: /* std */
          flag_set (thread->reg.eflags, FLAG_DF);
          thread->reg.eip = eip + 1;
          goto loop;
        case 0xfe: /* inc/dec */
          {
            guint8 value;
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m)
                                        || !modrm_get8 (thread, &m, &value))
              return IA32_ERR_GP;
            switch (m.regop)
              {
                case 0: /* inc */
                  value = incdec8 (thread, value, TRUE);
                  break;
                case 1: /* dec */
                  value = incdec8 (thread, value, FALSE);
                  break;
                default:
                  return IA32_ERR_UD;
              }
            if (!modrm_set8 (thread, &m, value, level))
              return IA32_ERR_GP;
            thread->reg.eip = eip + m.leng + 1;
          }
          goto loop;
        case 0xff: /* inc/dec/call/jmp/push */
          {
            ModRM m;

            if (!cpu_modrm (thread, eip + 1, padr, pseg, &m))
              return IA32_ERR_GP;
            if (pope)
              {
                guint16 value;

                if (!modrm_get16 (thread, &m, &value))
                  return IA32_ERR_GP;
                switch (m.regop)
                  {
                    case 0: /* inc */
                      value = incdec16 (thread, value, TRUE);
                      break;
                    case 1: /* dec */
                      value = incdec16 (thread, value, FALSE);
                      break;
                    case 2: /* call */
                      if (!ia32_stack_push16 (thread, eip + m.leng + 1))
                        return IA32_ERR_SS;
                    case 4: /* jmp */
                      reg_set_x (thread->reg.eip, value);
                      break;
                    case 6: /* push */
                      if (!ia32_stack_push16 (thread, value))
                        return IA32_ERR_SS;
                      thread->reg.eip = eip + m.leng + 1;
                      break;
                    default:
                      return IA32_ERR_UD;
                  }
                if (m.regop == 0 || m.regop == 1)
                  {
                    if (!modrm_set16 (thread, &m, value, level))
                      return IA32_ERR_GP;
                    thread->reg.eip = eip + m.leng + 1;
                  }
              }
            else
              {
                guint32 value;

                if (!modrm_get32 (thread, &m, &value))
                  return IA32_ERR_GP;
                switch (m.regop)
                  {
                    case 0: /* inc */
                      value = incdec32 (thread, value, TRUE);
                      break;
                    case 1: /* dec */
                      value = incdec32 (thread, value, FALSE);
                      break;
                    case 2: /* call */
                      if (!ia32_stack_push32 (thread, eip + m.leng + 1))
                        return IA32_ERR_SS;
                    case 4: /* jmp */
                      thread->reg.eip = value;
                      break;
                    case 6: /* push */
                      if (!ia32_stack_push32 (thread, value))
                        return IA32_ERR_SS;
                      thread->reg.eip = eip + m.leng + 1;
                      break;
                    default:
                      return IA32_ERR_UD;
                  }
                if (m.regop == 0 || m.regop == 1)
                  {
                    if (!modrm_set32 (thread, &m, value, level))
                      return IA32_ERR_GP;
                    thread->reg.eip = eip + m.leng + 1;
                  }
              }
          }
          goto loop;
        default:
          return IA32_ERR_UD;
      }
  return IA32_ERR_GP;
  loop:
  level = ia32_memory_get_level (thread->process, thread->reg.eip);
  if (thread->process->level < level)
    {
      thread->process->winmain = thread->reg.eip;
      thread->process->level = level;
    }
  return IA32_ERR_NULL;
}


/*  ja:命令実行
       process,プロセス構造体
    breakpoint,ブレークポイント
          tail,記録するステップ数(0:随時出力)
         flags,フラグ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
ia32_cpu_run (Ia32Process *process,
              GList       *breakpoint,
              const gint   tail,
              const guint  flags)
{
  gboolean running = FALSE;
  GList *glist;

  for (glist = g_list_first (process->thread);
                                            glist; glist = g_list_next (glist))
    {
      Ia32Thread *thread;

      thread = glist->data;
      if (!thread->terminate)
        {
          gint vector;
          GList *bp;

          /* ja:トレース */
          if (flags & (IA32_MESSAGE_MNEMONIC | IA32_MESSAGE_REGISTER))
            ia32_message_record_trace (thread, tail, flags);
          /* ja:ブレークポイント */
          for (bp = g_list_first (breakpoint); bp; bp = g_list_next (bp))
            {
              Ia32BreakPoint *b;

              b = bp->data;
              if (b->start <= thread->reg.eip && thread->reg.eip <= b->end)
                {
                  ia32_message_record_exception (thread, IA32_ERR_BP);
                  thread->terminate = TRUE;
                  break;
                }
            }
          if (thread->terminate)
            continue;
          /* ja:実行 */
          process->current = thread;
          vector = ia32_cpu_step (thread);
          /* ja:例外処理 */
          if (vector != IA32_ERR_NULL)
            {
              gboolean result = FALSE;
              guint32 exreg_address, next, handler;

              if (ia32_memory_read_dword (process,
                                    ia32_cpu_seg (thread, 0, IA32_CPU_PSEG_FS),
                                                        &exreg_address) >= 0
                && ia32_memory_read_dword (process, exreg_address, &next) >= 0
                && ia32_memory_read_dword (process,
                                            exreg_address + 4, &handler) >= 0)
                {
                  guint32 esp, exrec_address;
                  ExceptionRecord *exrec;
                  const static guint32 exception_code[] = {
    EXCEPTION_INT_DIVIDE_BY_ZERO,       /*  0 Divide Error */
    EXCEPTION_SINGLE_STEP,              /*  1 Debug */
    (guint32)-1,                        /*  2 NMI Interrupt */
    EXCEPTION_BREAKPOINT,               /*  3 Breakpoint */
    EXCEPTION_INT_OVERFLOW,             /*  4 Overflow */
    EXCEPTION_ARRAY_BOUNDS_EXCEEDED,    /*  5 Bound Range Exceeded */
    EXCEPTION_ILLEGAL_INSTRUCTION,      /*  6 Invalid Opcode */
    EXCEPTION_INVALID_HANDLE,           /*  7 Device Not Available */
    EXCEPTION_NONCONTINUABLE_EXCEPTION, /*  8 Double Fault */
    EXCEPTION_FLT_OVERFLOW,             /*  9 CoProcessor Segment Overrun */
    EXCEPTION_PRIV_INSTRUCTION,         /* 10 Invalid TSS */
    EXCEPTION_IN_PAGE_ERROR,            /* 11 Segment Not Present */
    EXCEPTION_STACK_OVERFLOW,           /* 12 Stack Segment Fault */
    EXCEPTION_ACCESS_VIOLATION,         /* 13 General Protection */
    EXCEPTION_GUARD_PAGE,               /* 14 Page Fault */
    0,                                  /* 15 Reserved */
    EXCEPTION_FLT_INVALID_OPERATION,    /* 16 Floating-Point Error */
    EXCEPTION_DATATYPE_MISALIGNMENT,    /* 17 Alignment Check */
    EXCEPTION_PRIV_INSTRUCTION,         /* 18 Machine Check */
    EXCEPTION_FLT_INEXACT_RESULT};      /* 19 SIMD Floating-Point Exception */

                  esp = thread->reg.esp;
                  /* ExceptionRecord */
                  thread->reg.esp -= (sizeof (ExceptionRecord) + 3) / 4 * 4;
                  exrec_address = thread->reg.esp;
                  exrec = ia32_memory_real_address (process, exrec_address);
                  if (exrec && ia32_memory_real_size (process, exrec_address)
                                                >= sizeof (ExceptionRecord))
                    {
                      guint32 ctxt_address;
                      Context *ctxt;

                      exrec_set_exception_code (exrec,
                                        vector < G_N_ELEMENTS (exception_code)
                                                ? exception_code[vector] : 0);
                      exrec_set_exception_flag (exrec, 0);
                      exrec_set_exception_record (exrec, 0);
                      exrec_set_exception_address (exrec, thread->reg.eip);
                      exrec_set_number_parameters (exrec, 0);
                      /* Context */
                      thread->reg.esp -= (sizeof (Context) + 3) / 4 * 4;
                      ctxt_address = thread->reg.esp;
                      ctxt = ia32_memory_real_address (process, ctxt_address);
                      if (ctxt && ia32_memory_real_size (process, ctxt_address)
                                                        >= sizeof (Context))
                        {
                          g_memset (ctxt, 0, sizeof (Context));
                          ctxt_set_gs (ctxt, thread->reg.gs);
                          ctxt_set_fs (ctxt, thread->reg.fs);
                          ctxt_set_es (ctxt, thread->reg.es);
                          ctxt_set_ds (ctxt, thread->reg.ds);
                          ctxt_set_edi (ctxt, thread->reg.edi);
                          ctxt_set_esi (ctxt, thread->reg.esi);
                          ctxt_set_ebx (ctxt, thread->reg.ebx);
                          ctxt_set_edx (ctxt, thread->reg.edx);
                          ctxt_set_ecx (ctxt, thread->reg.ecx);
                          ctxt_set_eax (ctxt, thread->reg.eax);
                          ctxt_set_ebp (ctxt, thread->reg.ebp);
                          ctxt_set_eip (ctxt, thread->reg.eip);
                          ctxt_set_cs (ctxt, thread->reg.cs);
                          ctxt_set_eflags (ctxt, thread->reg.eflags);
                          ctxt_set_esp (ctxt, esp);
                          ctxt_set_ss (ctxt, thread->reg.ss);
                          /* ja:スタック */
                          if (ia32_stack_push32 (thread, esp)
                           && ia32_stack_push32 (thread, GPOINTER_TO_UINT
                                        (g_hash_table_lookup (process->system,
                                                                "Terminate")))
                           && ia32_stack_push32 (thread, next)
                           && ia32_memory_write_dword (process,
                                    ia32_cpu_seg (thread, 0, IA32_CPU_PSEG_FS),
                                                    thread->reg.esp, -1) >= 0
                           && ia32_stack_push32 (thread, 0)
                           && ia32_stack_push32 (thread, ctxt_address)
                           && ia32_stack_push32 (thread, exreg_address)
                           && ia32_stack_push32 (thread, exrec_address)
                           && ia32_stack_push32 (thread, GPOINTER_TO_UINT
                                        (g_hash_table_lookup (process->system,
                                                                "Exception"))))
                            {
                              thread->reg.eip = handler;
                              result = TRUE;
                            }
                        }
                    }
                }
              if (!result)
                {
                  ia32_message_record_exception (thread, vector);
                  thread->terminate = TRUE;
                }
            }
          if (!thread->terminate)
            running = TRUE;
        }
    }
  return running;
}


/*  ja:プロセス実行
       process,プロセス構造体
    breakpoint,ブレークポイント
         timer,制限時間(秒)
          step,最大実行ステップ数(0:無制限)
          tail,記録するステップ数(0:随時出力)
         flags,フラグ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
ia32_cpu_loop (Ia32Process *process,
               GList       *breakpoint,
               const gint   step,
               const gint   timer,
               const gint   tail,
               const guint  flags)
{
  gint s = 1;
  time_t t;

  t = time (NULL);
  while (ia32_cpu_run (process, breakpoint, tail, flags))
    {
      if (step > 0 && ++s > step)
        {
          GList *glist;

          for (glist = g_list_first (process->thread);
                                            glist; glist = g_list_next (glist))
            {
              Ia32Thread *thread;

              thread = glist->data;
              if (!thread->terminate)
                ia32_message_record_error (thread, "Step out.");
            }
          break;
        }
      if (timer > 0 && time (NULL) - t > timer)
        {
          GList *glist;

          for (glist = g_list_first (process->thread);
                                            glist; glist = g_list_next (glist))
            {
              Ia32Thread *thread;

              thread = glist->data;
              if (!thread->terminate)
                ia32_message_record_error (thread, "Time out.");
            }
          break;
        }
    }
  return TRUE;
}
