/*
    IA32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "system.h"
#include "cpu.h"
#include "memory.h"
#include "message.h"
#include "stack.h"


/******************************************************************************
*                                                                             *
* ja:システム関数群                                                           *
*                                                                             *
******************************************************************************/
/*  ja:例外処理
    thread,スレッド構造体
       RET,TRUE:正常終了,FALSE:エラー                                       */
static gboolean
ia32_system_exception (Ia32Thread *thread)
{
  guint32 address;

  if (thread->reg.eax == EXCEPTION_CONTINUE_EXECUTION)
    {
      Context *ctxt;

      if (ia32_memory_read_dword (thread->process,
                                            thread->reg.esp + 4, &address) >= 0
            && ia32_memory_write_dword (thread->process,
                ia32_cpu_seg (thread, 0, IA32_CPU_PSEG_FS), address, -1) >= 0
            && ia32_memory_read_dword (thread->process,
                                            thread->reg.esp + 8, &address) >= 0
            && (ctxt = ia32_memory_real_address (thread->process, address)))
        {
          thread->reg.gs = ctxt_get_gs (ctxt);
          thread->reg.fs = ctxt_get_fs (ctxt);
          thread->reg.es = ctxt_get_es (ctxt);
          thread->reg.ds = ctxt_get_ds (ctxt);
          thread->reg.edi = ctxt_get_edi (ctxt);
          thread->reg.esi = ctxt_get_esi (ctxt);
          thread->reg.ebx = ctxt_get_ebx (ctxt);
          thread->reg.edx = ctxt_get_edx (ctxt);
          thread->reg.ecx = ctxt_get_ecx (ctxt);
          thread->reg.eax = ctxt_get_eax (ctxt);
          thread->reg.ebp = ctxt_get_ebp (ctxt);
          thread->reg.eip = ctxt_get_eip (ctxt);
          thread->reg.cs = ctxt_get_cs (ctxt);
          thread->reg.eflags = ctxt_get_eflags (ctxt);
          thread->reg.esp = ctxt_get_esp (ctxt);
          thread->reg.ss = ctxt_get_ss (ctxt);
          return TRUE;
        }
    }
  else if (ia32_memory_read_dword (thread->process,
                                            thread->reg.esp + 4, &address) >= 0
            && ia32_memory_read_dword (thread->process, address, &address) >= 0
            && address != (guint32)-1)
    {
      ia32_stack_push32 (thread,
                GPOINTER_TO_UINT (g_hash_table_lookup (thread->process->system,
                                                                "Exception")));
      thread->reg.eip = address;
      return TRUE;
    }
  ia32_stack_push32 (thread, 0);
  thread->reg.eip
            = GPOINTER_TO_UINT (g_hash_table_lookup (thread->process->system,
                                                                "Terminate"));
  return TRUE;
}


/*  ja:強制終了
    thread,スレッド構造体
       RET,TRUE:正常終了,FALSE:エラー                                       */
static gboolean
ia32_system_terminate (Ia32Thread *thread)
{
  guint32 address, code;

  if (ia32_memory_read_dword (thread->process,
                                            thread->reg.esp + 4, &address) >= 0
            && ia32_memory_read_dword (thread->process, address, &code) >= 0)
    {
      gint i;
      const static guint32 exception_code[] = {
    EXCEPTION_INT_DIVIDE_BY_ZERO,       /*  0 Divide Error */
    EXCEPTION_SINGLE_STEP,              /*  1 Debug */
    (guint32)-1,                        /*  2 NMI Interrupt */
    EXCEPTION_BREAKPOINT,               /*  3 Breakpoint */
    EXCEPTION_INT_OVERFLOW,             /*  4 Overflow */
    EXCEPTION_ARRAY_BOUNDS_EXCEEDED,    /*  5 Bound Range Exceeded */
    EXCEPTION_ILLEGAL_INSTRUCTION,      /*  6 Invalid Opcode */
    EXCEPTION_INVALID_HANDLE,           /*  7 Device Not Available */
    EXCEPTION_NONCONTINUABLE_EXCEPTION, /*  8 Double Fault */
    EXCEPTION_FLT_OVERFLOW,             /*  9 CoProcessor Segment Overrun */
    EXCEPTION_PRIV_INSTRUCTION,         /* 10 Invalid TSS */
    EXCEPTION_IN_PAGE_ERROR,            /* 11 Segment Not Present */
    EXCEPTION_STACK_OVERFLOW,           /* 12 Stack Segment Fault */
    EXCEPTION_ACCESS_VIOLATION,         /* 13 General Protection */
    EXCEPTION_GUARD_PAGE,               /* 14 Page Fault */
    0,                                  /* 15 Reserved */
    EXCEPTION_FLT_INVALID_OPERATION,    /* 16 Floating-Point Error */
    EXCEPTION_DATATYPE_MISALIGNMENT,    /* 17 Alignment Check */
    EXCEPTION_PRIV_INSTRUCTION,         /* 18 Machine Check */
    EXCEPTION_FLT_INEXACT_RESULT};      /* 19 SIMD Floating-Point Exception */

      for (i = 0; i < G_N_ELEMENTS (exception_code); i++)
        if (exception_code[i] == code)
          {
            ia32_message_record_exception (thread, i);
            break;
          }
    }
  thread->terminate = TRUE;
  return TRUE;
}


Ia32Export ia32_system_exports[] = {
{ 0,    1, "Exception", ia32_system_exception},
{ 0,    1, "Terminate", ia32_system_terminate},
{ 0,    0, NULL,        NULL}};
