/*
copyright (c) 2007-2009 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/
#include <ctype.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


typedef struct msglist
{
  char *id;
  char *str;
  struct msglist *next;
} MSGLIST;


static char *
msg_strconcat (char *string, ...)
{
  char *concat = NULL;

  if (string)
    {
      char *str;
      size_t len;
      va_list args;

      len = strlen (string) + 1;
      va_start (args, string);
      while (str = va_arg (args, char *))
        len += strlen (str);
      va_end (args);
      concat = realloc (string, len * sizeof (char));
      va_start (args, string);
      while (str = va_arg (args, char *))
        strcat (concat, str);
      va_end (args);
    }
  return concat;
}


static MSGLIST *
msg_list_insert (MSGLIST *msglist,
                 MSGLIST *m)
{
  MSGLIST *n, *prev = NULL;

  if (!m)
    return msglist;
  for (n = msglist; n && strcmp (m->id, n->id) > 0; prev = n, n = n->next);
  m->next = n;
  if (prev)
    prev->next = m;
  return prev ? msglist : m;
}


int
main (int   argc,
      char *argv[])
{
  char *infile = NULL, *outfile = NULL, buf[BUFSIZ], *str = NULL;
  int i, comment = 0, quot = 0, msg = -1;
  unsigned offset = 28;
  const unsigned magic = 0x950412de, revision = 0, hash = 0;
  size_t s, len = 0;
  FILE *fp;
  MSGLIST *msglist = NULL, *m = NULL;

  /* コマンドライン解析 */
  for (i = 1; i < argc; i++)
    if (strnicmp (argv[i], "-o", 2) == 0)
      {
        outfile = (argv[i])[2] == '=' ? argv[i] + 3
                : (argv[i])[2] != '\0' ? argv[i] + 2 : argv[++i];
      }
    else if (strnicmp (argv[i], "--output-file", 13) == 0)
      {
        outfile = (argv[i])[13] == '=' ? argv[i] + 14
                : (argv[i])[13] != '\0' ? argv[i] + 13 : argv[++i];
      }
    else if (stricmp (argv[i], "--h") == 0 || stricmp (argv[i], "--help") == 0)
      {
        fputs (
"Usage: msgfmt [OPTION] filename.po ...\n"
"\n"
"  -o, --output-file=FILE      write output to specified file\n"
"  -h, --help                  display this help and exit\n", stderr);
        return 0;
      }
    else if (stricmp (argv[i], "--no-hash") != 0)
      {
        infile = argv[i];
      }
  /* ファイル読み込み */
  fp = infile ? fopen (infile, "rt") : stdin;
  if (!fp)
    return -1;
  while ((s = fread (buf,
                        sizeof (char), sizeof (buf) / sizeof (char), fp)) > 0)
    {
      str = realloc (str, len + s + 1);
      memcpy (str + len, buf, s);
      len += s;
    }
  if (infile)
    fclose (fp);
  if (!str)
    return -1;
  str[len] = '\0';
  /* 解析 */
  for (i = 0; i < len; i++)
    {
      if (comment)
        {
          if (str[i] == '\n')
            comment = 0;
        }
      else if (quot)
        {
          char c = '\0';

          if (str[i] == '\"')
            {
              quot = 0;
            }
          else if (str[i] == '\\')
            {
              switch (tolower (str[++i]))
                {
                  case 'a'  : c = '\a'; break;
                  case 'b'  : c = '\b'; break;
                  case 'f'  : c = '\f'; break;
                  case 'n'  : c = '\n'; break;
                  case 'r'  : c = '\r'; break;
                  case 't'  : c = '\t'; break;
                  case 'v'  : c = '\v'; break;
                  case '\\' : c = '\\'; break;
                  case '\"' : c = '\"'; break;
                  case '\'' : c = '\''; break;
                  case 'x'  :
                    if (isxdigit (str[i + 1]) && isxdigit (str[i + 2]))
                      {
                        int j;

                        for (j = i + 1; j <= i + 2; j++)
                          {
                            c *= 16;
                            if (isdigit (str[j]))
                              c += str[j] - '0';
                            else if (isupper (str[j]))
                              c += str[j] - 'A' + 10;
                            else
                              c += str[j] - 'a' + 10;
                          }
                      }
                    i += 2;
                    break;
                  case '0': case '1': case '2': case '3':
                  case '4': case '5': case '6': case '7':
                    {
                      int j;

                      for (j = i; j <= i + 2; j++)
                        {
                          if (str[j] < '0' || '7' < str[j])
                            break;
                          c *= 8;
                          c += str[j] - '0';
                        }
                      i = j - 1;
                    }
                }
              if (c == '\0')
                i--;
            }
          else
            {
              c = str[i];
            }
          if (c != '\0' && m)
            {
              char buf[2];

              buf[0] = c;
              buf[1] = '\0';
              switch (msg)
                {
                  case 0:
                  case 1: m->id  = msg_strconcat (m->id,  buf, NULL); break;
                  case 2: m->str = msg_strconcat (m->str, buf, NULL);
                }
            }
        }
      else
        {
          if (str[i] == '\"')
            {
              quot = 1;
            }
          else if (str[i] == '#')
            {
              comment = 1;
            }
          else
            {
              const char *tag[3] = {"msgctxt", "msgid", "msgstr"};
              int j;

              for (j = 0; j < 3; j++)
                if (strnicmp (str + i, tag[j], strlen (tag[j])) == 0)
                  {
                    switch (j)
                      {
                        case 1:
                          if (msg == 0)
                            {
                              m->id = msg_strconcat (m->id, "\004", NULL);
                              break;
                            }
                        case 0:
                          msglist = msg_list_insert (msglist, m);
                          m = malloc (sizeof (MSGLIST));
                          m->id = malloc (sizeof (char));
                          m->str = malloc (sizeof (char));
                          m->id[0] = m->str[0] = '\0';
                          m->next = NULL;
                      }
                    msg = j;
                    break;
                  }
            }
        }
    }
  msglist = msg_list_insert (msglist, m);
  free (str);
  len = 0;
  for (m = msglist; m; m = m->next)
    len++;
  /* ファイル書き込み */
  fp = outfile ? fopen (outfile, "wb") : stdout;
  if (!fp)
    return -1;
  fwrite (&magic, sizeof (unsigned), 1, fp);/* マジックナンバー */
  fwrite (&revision, sizeof (unsigned), 1, fp);/* リビジョン */
  fwrite (&len, sizeof (size_t), 1, fp);/* 文字列領域の長さ */
  /* オリジナル文字列テーブルへのオフセット */
  fwrite (&offset, sizeof (unsigned), 1, fp);
  offset += len * sizeof (unsigned) * 2;
  /* 翻訳文字列テーブルへのオフセット */
  fwrite (&offset, sizeof (unsigned), 1, fp);
  fwrite (&hash, sizeof (unsigned), 1, fp);/* ハッシュテーブルのサイズ */
  fwrite (&hash, sizeof (unsigned), 1, fp);/* ハッシュテーブルのオフセット */
  offset += len * sizeof (unsigned) * 2;
  for (m = msglist; m; m = m->next)
    {
      s = strlen (m->id);
      fwrite (&s, sizeof (size_t), 1, fp);/* 文字列の長さ */
      fwrite (&offset, sizeof (unsigned), 1, fp);/* 文字列のオフセット */
      offset += s + 1;
    }
  for (m = msglist; m; m = m->next)
    {
      s = strlen (m->str);
      fwrite (&s, sizeof (size_t), 1, fp);/* 文字列の長さ */
      fwrite (&offset, sizeof (unsigned), 1, fp);/* 文字列のオフセット */
      offset += s + 1;
    }
  for (m = msglist; m; m = m->next)
    fwrite (m->id, sizeof (char), strlen (m->id) + 1, fp);
  for (m = msglist; m; m = m->next)
    fwrite (m->str, sizeof (char), strlen (m->str) + 1, fp);
  if (outfile)
    fclose (fp);
  /* メモリ解放 */
  while (msglist)
    {
      m = msglist->next;
      free (msglist->id);
      free (msglist->str);
      free (msglist);
      msglist = m;
    }
  return 0;
}
