/*
    align32
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "misc/peimage.h"


#define ALIGNMENT 0x00001000
#define IMAGEBASE 0x00400000


/*  ja:ヘルプを表示する
      fp,ファイルポインタ
    file,ファイル名                                                         */
static void
help (FILE        *fp,
      const gchar *file)
{
  g_fprintf (fp,
"alignw32 "VERSION" ("BUILD_ENVIRONMENT")\n"
"copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org\n"
"\n"
"Usage: %s [option...] infile [outfile]\n"
"\n"
"  -a, --align=BYTE       section alignment\n"
"  -b, --base=ADDRESS     image base\n"
"  -e, --entry=ADDRESS    address of entry point\n"
"\n", file);
}


int
main (int   argc,
      char *argv[])
{
  gboolean result;
  gsize size;
  gchar *infile = NULL, *outfile = NULL;
  gint i;
  guint8 *image, *data;
  guint32 align = ALIGNMENT, base = (guint32)-1, entry = (guint32)-1;

  /* ja:初期化 */
#if ! GTK_CHECK_VERSION (2,24,0)
  gtk_set_locale ();
#endif /* not GTK_CHECK_VERSION (2,24,0) */
  gtk_init (&argc, &argv);

  /* ja:引数 */
  for (i = 1; i < argc; i++)
    if (g_ascii_strcasecmp (argv[i], "-a") == 0
                            || g_ascii_strcasecmp (argv[i], "--align") == 0)
      {
        align = g_strtol (argv[++i], NULL, 16);
      }
    else if (g_ascii_strncasecmp (argv[i], "--align=", 8) == 0)
      {
        align = g_strtol (argv[i] + 8, NULL, 16);
      }
    else if (g_ascii_strcasecmp (argv[i], "-b") == 0
                            || g_ascii_strcasecmp (argv[i], "--base") == 0)
      {
        base = g_strtol (argv[++i], NULL, 16);
      }
    else if (g_ascii_strncasecmp (argv[i], "--base=", 7) == 0)
      {
        base = g_strtol (argv[i] + 7, NULL, 16);
      }
    else if (g_ascii_strcasecmp (argv[i], "-e") == 0
                            || g_ascii_strcasecmp (argv[i], "--entry") == 0)
      {
        entry = g_strtol (argv[++i], NULL, 16);
      }
    else if (g_ascii_strncasecmp (argv[i], "--entry=", 8) == 0)
      {
        entry = g_strtol (argv[i] + 8, NULL, 16);
      }
    else if (g_ascii_strcasecmp (argv[i], "-h") == 0
                                || g_ascii_strcasecmp (argv[i], "-?") == 0
                                || g_ascii_strcasecmp (argv[i], "--help") == 0)
      {
        help (stdout, argv[0]);
        return 0;
      }
    else if (!infile)
      {
        infile = argv[i];
      }
    else if (!outfile)
      {
        outfile = argv[i];
      }
    else
      break;
  if (!infile || i < argc || align <= 0)
    {
      help (stderr, argv[0]);
      return -1;
    }

  if ((image = peimage_file_load (infile)))
    {
      gint sections;
      ImageSectionHeader *ish;

      if (base != (guint32)-1)
        pe_ioh_set_image_base (image, base);
      if (entry != (guint32)-1)
        pe_ioh_set_address_of_entry_point (image, entry);
      sections = pe_ifh_get_number_of_sections (image);
      ish = pe_image_section_header (image);
      for (i = 0; i < sections; i++)
        {
          ish_set_pointer_to_raw_data (ish, ish_get_virtual_address (ish));
          ish++;
        }
      size = pe_ioh_get_size_of_image (image);
    }
  else if (g_file_get_contents (infile, (gchar **)&data, &size, NULL))
    {
      image = peimage_file_image_from_binary (data, size, base, entry, align);
      g_free (data);
      size = pe_ioh_get_size_of_image (image);
    }
  if (!image)
    {
      g_fprintf (stderr, "\'%s\' load error.\n", infile);
      return -1;
    }
  if (outfile)
    {
      if (!g_file_set_contents (outfile, (const gchar *)image, size, NULL))
        {
          g_fprintf (stderr, "\'%s\' save error.\n", outfile);
          result = FALSE;
        }
    }
  else
    {
      if (fwrite (image, sizeof (guint8), size, stdout) != size)
        {
          g_fprintf (stderr, "Output error.\n");
          result = FALSE;
        }
    }
  g_free (image);
  return result ? 0 : -1;
}
