/*
    disw32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "data.h"


/******************************************************************************
*                                                                             *
* ja:データ関数                                                               *
*                                                                             *
******************************************************************************/
/*  ja:BYTE単位データを作る
      info,イメージ情報構造体
     index,インデックス
      size,サイズ(バイト単位)
    belong,所属
       RET,作られたバイト数                                                 */
gsize
dw32_data_make_byte (Dw32Info    *info,
                     const gint   index,
                     const gsize  size,
                     const gint   belong)
{
  gint i;
  gsize s = 0;

  if (index < 0)
    return 0;
  while (index + s < info->size
                            && info->status[index + s] == DW32_STAT_UNKNOWN)
    s++;
  if (size < s)
    s = size;
  i = index;
  while (s > 0)
    {
      gchar str[51];
      gint j;

      g_strcpy (str, "db ");
      for (j = 0; j < 16; j++)
        {
          gchar tmp[3];

          info->status[i + j] = DW32_STAT_DATA;
          info->belong[i + j] = belong;
          g_sprintf (tmp, "%02x", *(guint8 *)(info->image + i + j));
          if (j != 0)
            g_strcat (str, ",");
          g_strcat (str, tmp);
          s--;
          if (s < 1)
            break;
        }
      info->mnemonic[i] = g_strdup (str);
      i += j;
    }
  return i - index;
}


/*  ja:WORD単位データを作る
      info,イメージ情報構造体
     index,インデックス
      size,サイズ(バイト単位)
    belong,所属
       RET,作られたバイト数                                                 */
gsize
dw32_data_make_word (Dw32Info    *info,
                     const gint   index,
                     const gsize  size,
                     const gint   belong)
{
  gint i;
  gsize s = 0;

  if (index < 0)
    return 0;
  while (index + s < info->size
                            && info->status[index + s] == DW32_STAT_UNKNOWN)
    s++;
  if (size < s)
    s = size;
  i = index;
  while (s >= 2)
    {
      gchar str[43];
      gint j;

      g_strcpy (str, "dw ");
      for (j = 0; j < 16; j += 2)
        {
          gchar tmp[5];

          info->status[i + j] = DW32_STAT_DATA;
          info->status[i + j + 1] = DW32_STAT_INTER;
          info->belong[i + j] = belong;
          info->belong[i + j + 1] = belong;
          g_sprintf (tmp, "%04x",
                        GUINT16_FROM_LE (*(guint16 *)(info->image + i + j)));
          if (j != 0)
            g_strcat (str, ",");
          g_strcat (str, tmp);
          s -= 2;
          if (s < 2)
            break;
        }
      info->mnemonic[i] = g_strdup (str);
      i += j;
    }
  return i - index + dw32_data_make_byte (info, i, s, belong);
}


/*  ja:DWORD単位データを作る
      info,イメージ情報構造体
     index,インデックス
      size,サイズ(バイト単位)
    belong,所属
       RET,作られたバイト数                                                 */
gsize
dw32_data_make_dword (Dw32Info    *info,
                      const gint   index,
                      const gsize  size,
                      const gint   belong)
{
  gint i;
  gsize s = 0;

  if (index < 0)
    return 0;
  while (index + s < info->size
                            && info->status[index + s] == DW32_STAT_UNKNOWN)
    s++;
  if (size < s)
    s = size;
  i = index;
  while (s >= 4)
    {
      gchar str[39];
      gint j;

      g_strcpy (str, "dd ");
      for (j = 0; j < 16; j += 4)
        {
          gchar tmp[9];

          info->status[i + j] = DW32_STAT_DATA;
          info->status[i + j + 1] = DW32_STAT_INTER;
          info->status[i + j + 2] = DW32_STAT_INTER;
          info->status[i + j + 3] = DW32_STAT_INTER;
          info->belong[i + j] = belong;
          info->belong[i + j + 1] = belong;
          info->belong[i + j + 2] = belong;
          info->belong[i + j + 3] = belong;
          g_sprintf (tmp, "%08x",
                        GUINT32_FROM_LE (*(guint32 *)(info->image + i + j)));
          if (j != 0)
            g_strcat (str, ",");
          g_strcat (str, tmp);
          s -= 4;
          if (s < 4)
            break;
        }
      info->mnemonic[i] = g_strdup (str);
      i += j;
    }
  return i - index + dw32_data_make_word (info, i, s, belong);
}


/*  ja:構造体データを作る
      info,イメージ情報構造体
     index,インデックス
      data,構造体定義
    belong,所属
       RET,作られたバイト数                                                 */
gsize
dw32_data_make_struct (Dw32Info       *info,
                       const gint      index,
                       const Dw32Data *data,
                       const gint      belong)
{
  gint i, j;

  if (index < 0)
    return 0;
  i = index;
  for (j = 0; data[j].size > 0; j++)
    {
      gint k;

      if (i + data[j].size >= info->size)
        return 0;
      for (k = 0; k < data[j].size; k++)
        if (info->status[i + k] != DW32_STAT_UNKNOWN)
          return 0;
    }
  i = index;
  for (j = 0; data[j].size > 0; j++)
    if (i + data[j].size < info->size)
      {
        dw32_data_make_dword (info, i, data[j].size, belong);
        info->comment[i] = g_strdup (data[j].name);
        i += data[j].size;
      }
  return i - index;
}


/*  ja:文字列データを作る
      info,イメージ情報構造体
     index,インデックス
    belong,所属
       RET,作られたバイト数                                                 */
gsize
dw32_data_make_string (Dw32Info   *info,
                       const gint  index,
                       const gint  belong)
{
  gint i;
  gsize s = 0;

  if (index < 0)
    return 0;
  while (index + s < info->size && *(gchar *)(info->image + index + s) != 0
                            && info->status[index + s] == DW32_STAT_UNKNOWN)
    s++;
  if (index + s >= info->size || info->status[index + s] != DW32_STAT_UNKNOWN)
    return 0;
  s++;
  i = index;
  while (s > 0)
    {
      gchar str[134];
      gint j;

      if (s > 1)
        {
          g_strcpy (str, "db \'");
          for (j = 0; j < 32; j++)
            {
              gchar tmp[5];
              gint c;

              info->status[i + j] = DW32_STAT_INTER;
              info->belong[i + j] = belong;
              c = *(guint8 *)(info->image + i + j);
              if (c == '\"')
                g_strcpy (tmp, "\\\"");
              else if (0x20 <= c && c <= 0x7e)
                g_sprintf (tmp, "%c", c);
              else if (c == '\0')
                g_strcpy (tmp, "\',0");
              else if (c == '\n')
                g_strcpy (tmp, "\\n");
              else if (c == '\r')
                g_strcpy (tmp, "\\r");
              else if (c == '\t')
                g_strcpy (tmp, "\\t");
              else
                g_sprintf (tmp, "\\x%02x", c);
              g_strcat (str, tmp);
              s--;
              if (s < 1)
                break;
            }
          if (s > 0)
            g_strcat (str, "\'");
        }
      else
        {
          j = 1;
          g_strcpy (str, "db 0");
          s--;
        }
      info->mnemonic[i] = g_strdup (str);
      i += j;
    }
  info->status[index] = DW32_STAT_DATA;
  return i - index;
}


/*  ja:ワイド文字列データを作る
      info,イメージ情報構造体
     index,インデックス
    belong,所属
       RET,作られたバイト数                                                 */
gsize
dw32_data_make_stringw (Dw32Info   *info,
                        const gint  index,
                        const gint  belong)
{
  gint i;
  gsize s = 0;

  while (index + s < info->size - 1
                        && *(guint16 *)(info->image + index + s) != 0
                        && info->status[index + s] == DW32_STAT_UNKNOWN
                        && info->status[index + s + 1] == DW32_STAT_UNKNOWN)
    s += 2;
  if (index + s > info->size - 2
                        || info->status[index + s] != DW32_STAT_UNKNOWN
                        || info->status[index + s + 1] != DW32_STAT_UNKNOWN)
    return 0;
  s += 2;
  i = index;
  while (s > 0)
    {
      gchar str[230];
      gint j;

      if (s > 2)
        {
          g_strcpy (str, "dw \'");
          for (j = 0; j < 32; j++)
            {
              gchar tmp[7];
              gint c;

              info->status[i + j * 2] = DW32_STAT_INTER;
              info->status[i + j * 2 + 1] = DW32_STAT_INTER;
              info->belong[i + j * 2] = belong;
              info->belong[i + j * 2 + 1] = belong;
              c = *(guint16 *)(info->image + i + j * 2);
              if (c == '\"')
                g_strcpy (tmp, "\\\"");
              else if (0x20 <= c && c <= 0x7e)
                g_sprintf (tmp, "%c", c);
              else if (c == '\0')
                g_strcpy (tmp, "\',0");
              else if (c == '\n')
                g_strcpy (tmp, "\\n");
              else if (c == '\r')
                g_strcpy (tmp, "\\r");
              else if (c == '\t')
                g_strcpy (tmp, "\\t");
              else
                g_sprintf (tmp, "\\u%04x", c);
              g_strcat (str, tmp);
              s -= 2;
              if (s < 2)
                break;
            }
          if (s > 0)
            g_strcat (str, "\'");
        }
      else
        {
          j = 2;
          g_strcpy (str, "dw 0");
          s -= 2;
        }
      info->mnemonic[i] = g_strdup (str);
      i += j;
    }
  info->status[index] = DW32_STAT_DATA;
  return i - index;
}


/*  ja:文字列を判定する
     info,イメージ情報構造体
    index,インデックス
      RET,バイト数,-1:文字列ではない                                        */
gint
dw32_data_is_string (Dw32Info   *info,
                     const gint  index)
{
  gint c = -1;
  gsize s = 0;

  if (index < 0)
    return -1;
  while (index + s < info->size
                            && info->status[index + s] == DW32_STAT_UNKNOWN)
    {
      c = info->image[index + s];
      if (c != '\n' && c != '\r' && c != '\t' && !(0x20 <= c && c <= 0x7e))
        break;
      s++;
    }
  return c == '\0' ? s : -1;
}


/*  ja:ワイド文字列を判定する
     info,イメージ情報構造体
    index,インデックス
      RET,バイト数,-1:ワイド文字列ではない                                  */
gint
dw32_data_is_stringw (Dw32Info   *info,
                      const gint  index)
{
  gint c = -1;
  gsize s = 0;

  if (index < 0)
    return -1;
  while (index + s < info->size - 1
                        && info->status[index + s] == DW32_STAT_UNKNOWN
                        && info->status[index + s + 1] == DW32_STAT_UNKNOWN)
    {
      c = *(guint16 *)(info->image + index + s);
      if (c != '\n' && c != '\r' && c != '\t' && !(0x20 <= c && c <= 0x7e))
        break;
      s += 2;
    }
  return c == '\0' ? s : -1;
}
