/*
    gcommon
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "gcommon.h"


/******************************************************************************
* Hash Tables                                                                 *
******************************************************************************/
#ifdef USE_GTK_EMULATE
# define HASH_TABLE_SHIFT 13
# define HASH_TABLE_SIZE (1 << HASH_TABLE_SHIFT)
# define HASH_TABLE_MASK (HASH_TABLE_SIZE - 1)


struct _GHashTable
{
  GHashFunc hash_func;
  GEqualFunc key_equal_func;
  GDestroyNotify key_destroy_func;
  GDestroyNotify value_destroy_func;
  GList *glist[HASH_TABLE_SIZE];
};
typedef struct _GHashNode
{
  gpointer key;
  gpointer value;
} GHashNode;


GHashTable *
g_hash_table_new (GHashFunc  hash_func,
                  GEqualFunc key_equal_func)
{
  return g_hash_table_new_full (hash_func, key_equal_func, NULL, NULL);
}


GHashTable *
g_hash_table_new_full (GHashFunc      hash_func,
                       GEqualFunc     key_equal_func,
                       GDestroyNotify key_destroy_func,
                       GDestroyNotify value_destroy_func)
{
  GHashTable *hash_table;

  hash_table = g_malloc0 (sizeof (GHashTable));
  hash_table->hash_func = hash_func ? hash_func : g_direct_hash;
  hash_table->key_equal_func = key_equal_func ? key_equal_func
                                              : g_direct_equal;
  hash_table->key_destroy_func = key_destroy_func;
  hash_table->value_destroy_func = value_destroy_func;
  return hash_table;
}


static GHashNode *
g_hash_table_lookup_node (GHashTable    *hash_table,
                          gconstpointer  key)
{
  if (hash_table)
    {
      guint hash;
      GList *glist;

      hash = hash_table->hash_func (key) & HASH_TABLE_MASK;
      for (glist = hash_table->glist[hash]; glist; glist = g_list_next (glist))
        {
          GHashNode *node;

          node = glist->data;
          if (hash_table->key_equal_func (node->key, key))
            {
              hash_table->glist[hash]
                        = g_list_delete_link (hash_table->glist[hash], glist);
              hash_table->glist[hash]
                        = g_list_prepend (hash_table->glist[hash], node);
              return node;
            }
        }
    }
  return NULL;
}


void
g_hash_table_insert (GHashTable *hash_table,
                     gpointer    key,
                     gpointer    value)
{
  if (hash_table)
    {
      GHashNode *node;

      node = g_hash_table_lookup_node (hash_table, key);
      if (node)
        {
          if (hash_table->key_destroy_func)
            hash_table->key_destroy_func (key);
          if (hash_table->value_destroy_func)
            hash_table->value_destroy_func (node->value);
          node->value = value;
        }
      else
        {
          guint hash;

          hash = hash_table->hash_func (key) & HASH_TABLE_MASK;
          node = g_malloc (sizeof (GHashNode));
          node->key = key;
          node->value = value;
          hash_table->glist[hash] = g_list_append (hash_table->glist[hash],
                                                                        node);
        }
    }
}


guint
g_hash_table_size (GHashTable *hash_table)
{
  guint size = 0;

  if (hash_table)
    {
      gint i;

      for (i = 0; i < HASH_TABLE_SIZE; i++)
        size += g_list_length (hash_table->glist[i]);
    }
  return size;
}


gpointer
g_hash_table_lookup (GHashTable    *hash_table,
                     gconstpointer  key)
{
  GHashNode *node;

  node = g_hash_table_lookup_node (hash_table, key);
  return node ? node->value : NULL;
}


void
g_hash_table_foreach (GHashTable *hash_table,
                      GHFunc      func,
                      gpointer    user_data)
{
  if (hash_table && func)
    {
      gint i;

      for (i = 0; i < HASH_TABLE_SIZE; i++)
        {
          GList *glist;

          for (glist = hash_table->glist[i];
                                            glist; glist = g_list_next (glist))
            {
              GHashNode *node;

              node = glist->data;
              func (node->key, node->value, user_data);
            }
        }
    }
}


gboolean
g_hash_table_remove (GHashTable    *hash_table,
                     gconstpointer  key)
{
  if (hash_table)
    {
      guint hash;
      GList *glist;

      hash = hash_table->hash_func (key) & HASH_TABLE_MASK;
      for (glist = hash_table->glist[hash]; glist; glist = g_list_next (glist))
        {
          GHashNode *node;

          node = glist->data;
          if (hash_table->key_equal_func (node->key, key))
            {
              hash_table->glist[hash]
                        = g_list_delete_link (hash_table->glist[hash], glist);
              if (hash_table->key_destroy_func)
                hash_table->key_destroy_func (node->key);
              if (hash_table->value_destroy_func)
                hash_table->value_destroy_func (node->value);
              g_free (node);
              return TRUE;
            }
        }
    }
  return FALSE;
}


guint
g_hash_table_foreach_remove (GHashTable *hash_table,
                             GHRFunc     func,
                             gpointer    user_data)
{
  guint removed = 0;

  if (hash_table && func)
    {
      gint i;

      for (i = 0; i < HASH_TABLE_SIZE; i++)
        {
          GList *glist;

          glist = hash_table->glist[i];
          while (glist)
            {
              GList *gl;
              GHashNode *node;

              gl = g_list_next (glist);
              node = glist->data;
              if (func (node->key, node->value, user_data))
                {
                  hash_table->glist[i]
                            = g_list_delete_link (hash_table->glist[i], glist);
                  if (hash_table->key_destroy_func)
                    hash_table->key_destroy_func (node->key);
                  if (hash_table->value_destroy_func)
                    hash_table->value_destroy_func (node->value);
                  g_free (node);
                  removed++;
                }
              glist = gl;
            }
        }
    }
  return removed;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,14,0)
static void
g_hash_table_get_keys_func (gpointer key,
                            gpointer value,
                            gpointer user_data)
{
  GList **glist;

  glist = user_data;
  *glist = g_list_append (*glist, key);
}


GList *
g_hash_table_get_keys (GHashTable *hash_table)
{
  GList *glist = NULL;

  if (hash_table)
    g_hash_table_foreach (hash_table, g_hash_table_get_keys_func, &glist);
  return glist;
}


static void
g_hash_table_get_values_func (gpointer key,
                              gpointer value,
                              gpointer user_data)
{
  GList **glist;

  glist = user_data;
  *glist = g_list_append (*glist, value);
}


GList *
g_hash_table_get_values (GHashTable *hash_table)
{
  GList *glist = NULL;

  if (hash_table)
    g_hash_table_foreach (hash_table, g_hash_table_get_values_func, &glist);
  return glist;
}
#endif /* not GLIB_CHECK_VERSION(2,14,0) */


#ifdef USE_GTK_EMULATE
void
g_hash_table_destroy (GHashTable *hash_table)
{
  if (hash_table)
    {
      gint i;

      for (i = 0; i < HASH_TABLE_SIZE; i++)
        while (hash_table->glist[i])
          {
            GHashNode *node;

            node = hash_table->glist[i]->data;
            if (hash_table->key_destroy_func)
              hash_table->key_destroy_func (node->key);
            if (hash_table->value_destroy_func)
              hash_table->value_destroy_func (node->value);
            g_free (node);
            hash_table->glist[i] = g_list_delete_link (hash_table->glist[i],
                                                       hash_table->glist[i]);
          }
      g_free (hash_table);
    }
}


gboolean
g_direct_equal (gconstpointer v1,
                gconstpointer v2)
{
  return v1 == v2;
}


guint
g_direct_hash (gconstpointer v)
{
  return GPOINTER_TO_UINT (v);
}


gboolean
g_int_equal (gconstpointer v1,
             gconstpointer v2)
{
  return *((const gint *)v1) == *((const gint *)v2);
}


guint
g_int_hash (gconstpointer v)
{
  return *(const gint *)v;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,22,0)
gboolean
g_int64_equal (gconstpointer v1,
               gconstpointer v2)
{
  return *((const gint64 *)v1) == *((const gint64 *)v2);
}


guint
g_int64_hash (gconstpointer v)
{
  return (guint)*(const gint64 *)v;
}


gboolean
g_double_equal (gconstpointer v1,
                gconstpointer v2)
{
  return *((const gdouble *)v1) == *((const gdouble *)v2);
}


guint
g_double_hash (gconstpointer v)
{
  return (guint)*(const gdouble *)v;
}
#endif /* not GLIB_CHECK_VERSION(2,22,0) */


#ifdef USE_GTK_EMULATE
gboolean
g_str_equal (gconstpointer v1,
             gconstpointer v2)
{
  return g_strcmp ((const gchar *)v1, (const gchar *)v2) == 0;
}


guint
g_str_hash (gconstpointer v)
{
  const gchar *p;
  guint32 h = 5381;

  if (v)
    for (p = v; *p != '\0'; p++)
      h = (h << 5) + h + *p;
  return h;
}
#endif /* USE_GTK_EMULATE */
