/*
    gcommon
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "gcommon.h"
#ifdef G_OS_WIN32
# include <windows.h>
# include <tchar.h>
#endif /* G_OS_WIN32 */


/******************************************************************************
* Directory Utilities                                                         *
******************************************************************************/
#ifdef USE_GTK_EMULATE
# if defined (G_OS_WIN32) && ! defined (HAVE_DIRENT_H)
#  ifdef UNICODE
#   define tdirent wdirent
#   define TDIR WDIR
#   define topendir wopendir
#   define treaddir wreaddir
#   define tclosedir wclosedir
typedef struct wdirent
{
  long           d_ino;
  unsigned short d_reclen;
  unsigned short d_namlen;
  wchar_t        d_name[MAX_PATH];
} WDIRENT;
typedef struct wdir
{
  int            nStat;
  LPWSTR         lpszPath;
  HANDLE         hFind;
  struct wdirent dd_dir;
} WDIR;
#  else /* not UNICODE */
#   define tdirent dirent
#   define TDIR DIR
#   define topendir opendir
#   define treaddir readdir
#   define tclosedir closedir
typedef struct dirent
{
  long           d_ino;
  unsigned short d_reclen;
  unsigned short d_namlen;
  char           d_name[MAX_PATH];
} DIRENT;
typedef struct dir
{
  int           nStat;
  LPSTR         lpszPath;
  HANDLE        hFind;
  struct dirent dd_dir;
} DIR;
#  endif /* not UNICODE */


TDIR *
topendir (const TCHAR *name)
{
  int i;
  DWORD dwResult;
  HANDLE hFind;
  LPTSTR lpszFile, lpszName;
  WIN32_FIND_DATA fnData;
  TDIR *nDir;

  if (!name)
    {
      errno = EFAULT;
      return NULL;
    }
  if (name[0] == '\0')
    {
      errno = ENOTDIR;
      return NULL;
    }
  lpszName = HeapAlloc (GetProcessHeap (), HEAP_ZERO_MEMORY,
                                        (lstrlen (name) + 1) * sizeof (TCHAR));
  lstrcpy (lpszName, name);
#  ifdef UNICODE
  for (i = 0; lpszName[i] != '\0'; i++)
    if (lpszName[i] == '/')
      lpszName[i] = '\\';
#  else /* not UNICODE */
  i = 0;
  while (lpszName[i] != '\0')
    if (IsDBCSLeadByteEx (CP_ACP, lpszName[i]))
      {
        i += 2;
      }
    else
      {
        if (lpszName[i] == '/')
          lpszName[i] = '\\';
        i++;
      }
#  endif /* not UNICODE */
  dwResult = GetFullPathName (lpszName, 0, NULL, &lpszFile);
  if (dwResult <= 0)
    {
      HeapFree (GetProcessHeap (), 0, lpszName);
      errno = ENOTDIR;
      return NULL;
    }
  nDir = HeapAlloc (GetProcessHeap (), HEAP_ZERO_MEMORY, sizeof (TDIR));
  nDir->lpszPath = HeapAlloc (GetProcessHeap (), HEAP_ZERO_MEMORY,
                                            (dwResult + 2) * sizeof (TCHAR));
  dwResult = GetFullPathName (lpszName, dwResult, nDir->lpszPath, &lpszFile);
  HeapFree (GetProcessHeap (), 0, lpszName);
  if (dwResult <= 0)
    {
      HeapFree (GetProcessHeap (), 0, nDir->lpszPath);
      HeapFree (GetProcessHeap (), 0, nDir);
      errno = ENOTDIR;
      return NULL;
    }
  if (nDir->lpszPath[dwResult - 1] != '\\')
    lstrcat (nDir->lpszPath, _T("\\"));
  lstrcat (nDir->lpszPath, _T("*"));
  hFind = FindFirstFile (nDir->lpszPath, &fnData);
  if (hFind == INVALID_HANDLE_VALUE)
    {
      HeapFree (GetProcessHeap (), 0, nDir->lpszPath);
      HeapFree (GetProcessHeap (), 0, nDir);
      errno = ENOTDIR;
      return NULL;
    }
  if (!FindClose (hFind))
    {
      HeapFree (GetProcessHeap (), 0, nDir->lpszPath);
      HeapFree (GetProcessHeap (), 0, nDir);
      errno = EFAULT;
      return NULL;
    }
  nDir->hFind = INVALID_HANDLE_VALUE;
  errno = 0;
  return nDir;
}


struct tdirent *
treaddir (TDIR *nDir)
{
  WIN32_FIND_DATA fnData;

  if (!nDir)
    {
      errno = EFAULT;
      return NULL;
    }
  if (nDir->nStat == 0)
    {
      nDir->hFind = FindFirstFile (nDir->lpszPath, &fnData);
      nDir->nStat = nDir->hFind == INVALID_HANDLE_VALUE ? -1 : 1;
    }
  else if (nDir->nStat > 0)
    {
      if (FindNextFile (nDir->hFind, &fnData))
        {
          nDir->nStat++;
        }
      else
        {
          FindClose (nDir->hFind);
          nDir->hFind = INVALID_HANDLE_VALUE;
          nDir->nStat = -1;
        }
    }
  errno = 0;
  if (nDir->nStat <= 0)
    return NULL;
  nDir->dd_dir.d_namlen = lstrlen (fnData.cFileName);
  lstrcpy (nDir->dd_dir.d_name, fnData.cFileName);
  return &nDir->dd_dir;
}


int
tclosedir (TDIR *nDir)
{
  int ret;

  if (nDir)
    {
      ret = nDir->hFind == INVALID_HANDLE_VALUE || FindClose (nDir->hFind)
                                                                    ? 0 : -1;
      HeapFree (GetProcessHeap (), 0, nDir->lpszPath);
      HeapFree (GetProcessHeap (), 0, nDir);
      errno = 0;
    }
  else
    {
      ret = -1;
      errno = EFAULT;
    }
  return ret;
}
# endif /* defined (G_OS_WIN32) && ! defined (HAVE_DIRENT_H) */


struct _GDir
{
# if defined (G_OS_WIN32) && defined (UNICODE)
  WDIR *dirp;
  gchar *name;
# else /* not defined (G_OS_WIN32) && defined (UNICODE) */
  DIR *dirp;
# endif /* not defined (G_OS_WIN32) && defined (UNICODE) */
};


GDir *
g_dir_open (const gchar  *path,
            guint         flags,
            GError      **error)
{
  if (path)
    {
      GDir *dir;
# if defined (G_OS_WIN32) && defined (UNICODE)
      gchar *utf8str;
      wchar_t *wpath;

      dir = g_malloc (sizeof (GDir));
      utf8str = g_filename_to_utf8 (path, -1, NULL, NULL, NULL);
      wpath = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
      g_free (utf8str);
      dir->dirp = wopendir (wpath);
      g_free (wpath);
      dir->name = NULL;
# else /* not defined (G_OS_WIN32) && defined (UNICODE) */
      dir = g_malloc (sizeof (GDir));
      dir->dirp = opendir (path);
# endif /* not defined (G_OS_WIN32) && defined (UNICODE) */
      if (dir->dirp)
        return dir;
      g_free (dir);
    }
  return NULL;
}


const gchar *
g_dir_read_name (GDir *dir)
{
# if defined (G_OS_WIN32) && defined (UNICODE)
  struct wdirent *entry = NULL;

  g_free (dir->name);
  dir->name = NULL;
  if (dir)
    {
      entry = wreaddir (dir->dirp);
      while (entry && (wcscmp (entry->d_name, L".") == 0
                                        || wcscmp (entry->d_name, L"..") == 0))
        entry = wreaddir (dir->dirp);
    }
  if (entry)
    {
      gchar *utf8str;

      utf8str = g_utf16_to_utf8 (entry->d_name, -1, NULL, NULL, NULL);
      dir->name = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
      g_free (utf8str);
    }
  return dir->name;
# else /* not defined (G_OS_WIN32) && defined (UNICODE) */
  struct dirent *entry = NULL;

  if (dir)
    {
      entry = readdir (dir->dirp);
      while (entry && (g_strcmp (entry->d_name, ".") == 0
                                    || g_strcmp (entry->d_name, "..") == 0))
        entry = readdir (dir->dirp);
    }
  return entry ? entry->d_name : NULL;
# endif /* not defined (G_OS_WIN32) && defined (UNICODE) */
}


void
g_dir_close (GDir *dir)
{
  if (dir)
    {
# if defined (G_OS_WIN32) && defined (UNICODE)
      wclosedir (dir->dirp);
      g_free (dir->name);
# else /* not defined (G_OS_WIN32) && defined (UNICODE) */
      closedir (dir->dirp);
# endif /* not defined (G_OS_WIN32) && defined (UNICODE) */
      g_free (dir);
    }
}
#endif /* USE_GTK_EMULATE */
