/*
    IA32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "module.h"
#include "memory.h"
#include "message.h"
#include "w32osdll/w32osdll.h"
#include "misc/peimage.h"


/******************************************************************************
*                                                                             *
* ja:モジュール関数群                                                         *
*                                                                             *
******************************************************************************/
typedef struct _Ia32Module
{
  guint32 address;
  gchar *file, *name;
  gint counter;
  GList *library;
} Ia32Module;


static gchar *
ia32_get_basename (const gchar *file)
{
  gchar *name = NULL;

  if (file)
    {
      name = g_path_get_basename (file);
      if (!g_strrchr (name, '.'))
        {
          gchar *tmp;

          tmp = g_strconcat (name, ".dll", NULL);
          g_free (name);
          name = tmp;
        }
    }
  return name;
}


/*  ja:既にロードされているイメージを取得する
    process,プロセス
       file,イメージ名
        RET,イメージ,IA32_MEMORY_NULL:ロードされていない                    */
guint32
ia32_module_get_library (Ia32Process *process,
                         const gchar *file)
{
  guint32 address = IA32_MEMORY_NULL;

  if (file)
    {
      gchar *base, *name;
      GList *glist;

      base = g_path_get_basename (file);
      name = ia32_get_basename (file);
      for (glist = g_list_first (process->module);
                                            glist; glist = g_list_next (glist))
        {
          Ia32Module *module;

          module = glist->data;
          if (g_ascii_strcasecmp (module->name, base) == 0
                            || g_ascii_strcasecmp (module->name, name) == 0)
            {
              address = module->address;
              break;
            }
        }
      g_free (base);
      g_free (name);
    }
  return address;
}


/*  ja:モジュール構造体を取得する
    process,プロセス
    address,イメージ
        RET,モジュール構造体,NULL:エラー                                    */
static Ia32Module *
ia32_module_get_info (Ia32Process   *process,
                      const guint32  address)
{
  if (address != IA32_MEMORY_NULL)
    {
      GList *glist;

      for (glist = g_list_first (process->module);
                                            glist; glist = g_list_next (glist))
        if (((Ia32Module *)glist->data)->address == address)
          return glist->data;
    }
  return NULL;
}


/*  ja:システムイメージを作成する
    process,プロセス
      osdll,エクスポート
        RET,イメージ,IA32_MEMORY_NULL:エラー                                */
static guint32
ia32_module_create_library (Ia32Process     *process,
                            const Ia32Osdll *osdll)
{
  gchar *q;
  gint i, num = 0;
  gint ordinal_min = G_MAXINT, ordinal_max = G_MININT;
  gsize leng, leng_head, leng_code, leng_data;
  guint8 *p, *image;
  guint16 *ordinal;
  guint32 address, *func, *name;
  ImageFileHeader *ifh;
  ImageOptionalHeader *ioh;
  ImageSectionHeader *ish;
  ImageExportDirectory *ied;

  leng_head = sizeof (ImageDosHeader)
            + sizeof (guint32)
            + sizeof (ImageFileHeader)
            + sizeof (ImageOptionalHeader)
            + sizeof (ImageSectionHeader);
  leng_code = 0;
  leng_data = sizeof (ImageExportDirectory) + g_strlen (osdll->name) + 1;
  for (i = 0; osdll->exports[i].argument != G_MAXINT; i++)
    {
      if (osdll->exports[i].argument < 0)
        leng_code -= osdll->exports[i].argument;
      else if (osdll->exports[i].argument == 0)
        leng_code++;
      else
        leng_code += 3;
      if (osdll->exports[i].name)
        {
          num++;
          leng_data += g_strlen (osdll->exports[i].name) + 1
                                        + sizeof (guint32) + sizeof (guint16);
        }
      if (ordinal_min > osdll->exports[i].ordinal)
        ordinal_min = osdll->exports[i].ordinal;
      if (ordinal_max < osdll->exports[i].ordinal)
        ordinal_max = osdll->exports[i].ordinal;
    }
  leng_data += (ordinal_max - ordinal_min + 1) * sizeof (guint32);
  leng = leng_head + leng_code + leng_data;
  address = osdll->address != IA32_MEMORY_NULL
                            ? ia32_memory_alloc (process, osdll->address, leng,
                                IA32_MEMORY_ATTR_SYSTEM) : IA32_MEMORY_NULL;
  if (address == IA32_MEMORY_NULL)
    address = ia32_memory_alloc (process, IA32_MEMORY_NULL, leng,
                                                    IA32_MEMORY_ATTR_SYSTEM);
  image = ia32_memory_real_address (process, address);
  if (!image)
    return IA32_MEMORY_NULL;
  /* ja:設定 */
  pe_idh_set_magic (image, PEIMAGE_DOS_SIGNATURE);
  pe_idh_set_lfanew (image, sizeof (ImageDosHeader));
  pe_set_signature (image, PEIMAGE_NT_SIGNATURE);
  ifh = pe_image_file_header (image);
  ifh_set_machine (ifh, PEIMAGE_FILE_MACHINE_I386);
  ifh_set_number_of_sections (ifh, 1);
  ifh_set_size_of_optional_header (ifh, sizeof (ImageOptionalHeader));
  ifh_set_characteristics (ifh, PEIMAGE_FILE_EXECUTABLE_IMAGE
                                            | PEIMAGE_FILE_LINE_NUMS_STRIPPED
                                            | PEIMAGE_FILE_LOCAL_SYMS_STRIPPED
                                            | PEIMAGE_FILE_32BIT_MACHINE
                                            | PEIMAGE_FILE_DLL);
  ioh = pe_image_optional_header (image);
  ioh_set_magic (ioh, PEIMAGE_NT_OPTIONAL_HDR_MAGIC);
  ioh_set_size_of_code (ioh, leng_code);
  ioh_set_size_of_initialized_data (ioh, leng_data);
  ioh_set_base_of_code (ioh, leng_head);
  ioh_set_base_of_data (ioh, leng_head + leng_code);
  ioh_set_image_base (ioh,
                osdll->address != IA32_MEMORY_NULL ? osdll->address : address);
  ioh_set_section_alignment (ioh, 1);
  ioh_set_file_alignment (ioh, 1);
  ioh_set_major_operating_system_version (ioh, 4);
  ioh_set_minor_operating_system_version (ioh, 0);
  ioh_set_major_subsystem_version (ioh, 4);
  ioh_set_minor_subsystem_version (ioh, 0);
  ioh_set_size_of_image (ioh, leng);
  ioh_set_size_of_headers (ioh, leng_head);
  ioh_set_subsystem (ioh, PEIMAGE_SUBSYSTEM_WINDOWS_GUI);
  ioh_set_size_of_stack_reserve (ioh, 0x100000);
  ioh_set_size_of_stack_commit (ioh, 0x1000);
  ioh_set_size_of_heap_reserve (ioh, 0x100000);
  ioh_set_size_of_heap_commit (ioh, 0x1000);
  ioh_set_number_of_rva_and_sizes (ioh, PEIMAGE_NUMBEROF_DIRECTORY_ENTRIES);
  ioh_set_data_directory_virtual_address (ioh,
                        PEIMAGE_DIRECTORY_ENTRY_EXPORT, leng_head + leng_code);
  ioh_set_data_directory_size (ioh,
                        PEIMAGE_DIRECTORY_ENTRY_EXPORT, leng_data);
  ish = pe_image_section_header (image);
  ish_set_virtual_size (ish, leng_code + leng_data);
  ish_set_virtual_address (ish, leng_head);
  ish_set_size_of_raw_data (ish, leng_code + leng_data);
  ish_set_pointer_to_raw_data (ish, leng_head);
  ish_set_characteristics (ish, PEIMAGE_SCN_CNT_CODE
                                            | PEIMAGE_SCN_CNT_INITIALIZED_DATA
                                            | PEIMAGE_SCN_MEM_EXECUTE
                                            | PEIMAGE_SCN_MEM_READ);
  /* ja:ポインタ */
  p = (guint8 *)(ish + 1);
  ied = (ImageExportDirectory *)(p + leng_code);
  func = (guint32 *)(ied + 1);
  name = func + ordinal_max - ordinal_min + 1;
  ordinal = (guint16 *)(name + num);
  q = (gchar *)(ordinal + num);
  ied_set_name (ied, GPOINTER_TO_UINT (q) - GPOINTER_TO_UINT (image));
  ied_set_base (ied, ordinal_min);
  ied_set_number_of_functions (ied, ordinal_max - ordinal_min + 1);
  ied_set_number_of_names (ied, num);
  ied_set_address_of_functions (ied, GPOINTER_TO_UINT (func)
                                                - GPOINTER_TO_UINT (image));
  ied_set_address_of_names (ied, GPOINTER_TO_UINT (name)
                                                - GPOINTER_TO_UINT (image));
  ied_set_address_of_name_ordinals (ied, GPOINTER_TO_UINT (ordinal)
                                                - GPOINTER_TO_UINT (image));
  g_strcpy (q, osdll->name);
  q += g_strlen (osdll->name) + 1;
  for (i = 0; osdll->exports[i].argument != G_MAXINT; i++)
    {
      gint index;

      index = osdll->exports[i].ordinal - ordinal_min;
      func[index] = GUINT32_TO_LE (GPOINTER_TO_UINT (p)
                                                - GPOINTER_TO_UINT (image));
      if (osdll->exports[i].argument >= 0)
        {
          Ia32Api *api;

          api = g_malloc (sizeof (Ia32Api));
          api->file = osdll->name;
          api->name = osdll->exports[i].name;
          api->ordinal = osdll->exports[i].ordinal;
          api->func = osdll->exports[i].func;
          g_hash_table_insert (process->export, GUINT_TO_POINTER
                            (ia32_memory_virtual_address (process, p)), api);
          if (osdll->exports[i].argument == 0)
            {
              *p++ = 0xc3;
            }
          else
            {
              *p++ = 0xc2;
              *(guint16 *)p = GINT16_TO_LE (osdll->exports[i].argument);
              p += 2;
            }
        }
      else
        {
          g_memmove (p, osdll->exports[i].func, - osdll->exports[i].argument);
          p -= osdll->exports[i].argument;
        }
      if (osdll->exports[i].name)
        {
          *name = GUINT32_TO_LE (GPOINTER_TO_UINT (q)
                                                - GPOINTER_TO_UINT (image));
          g_strcpy (q, osdll->exports[i].name);
          q += g_strlen (osdll->exports[i].name) + 1;
          *ordinal = index;
          ordinal++;
          name++;
        }
    }
  return address;
}


/*  ja:システムイメージをロードする
    process,プロセス
       file,イメージ名
        RET,イメージ,IA32_MEMORY_NULL:エラー                                */
static guint32
ia32_module_load_system_library (Ia32Process *process,
                                 const gchar *file)
{
  gint i;

  for (i = 0; ia32_osdll[i].name; i++)
    if (g_ascii_strcasecmp (ia32_osdll[i].name, file) == 0)
      return ia32_module_create_library (process, ia32_osdll + i);
  return IA32_MEMORY_NULL;
}


/*  ja:イメージをロードする
    process,プロセス
       file,イメージ名
        RET,イメージ,IA32_MEMORY_NULL:エラー                                */
guint32
ia32_module_load_library (Ia32Process *process,
                          const gchar *file)
{
  gboolean result = TRUE;
  gint i;
  guint8 *image;
  guint32 address;
  Ia32Module *module;

  if (!process || !file)
    return IA32_MEMORY_NULL;
  address = ia32_module_get_library (process, file);
  if (address)
    {
      /* ja:既読 */
      module = ia32_module_get_info (process, address);
      if (module)
        {
          module->counter++;
          return module->address;
        }
    }
  module = g_malloc0 (sizeof (Ia32Module));
  module->counter = 1;
  module->name = ia32_get_basename (file);
  /* ja:システムファイルをロードする */
  module->address = ia32_module_load_system_library (process, module->name);
  if (module->address != IA32_MEMORY_NULL)
    {
      module->file = g_build_filename
                                (g_get_home_dir (), "bin", module->name, NULL);
      process->module = g_list_append (process->module, module);
      return module->address;
    }
  g_free (module->name);

 /* ja:ファイル読み込み */
  module->name = g_path_get_basename (file);
  module->file = g_path_get_absolute (file, NULL);
  image = peimage_file_load (module->file);
  if (!image)
    {
      g_free (image);
      g_free (module->name);
      g_free (module->file);
      g_free (module);
      return IA32_MEMORY_NULL;
    }
  /* ja:メモリ登録 */
  module->address = pe_ioh_get_image_base (image);
  if (!ia32_memory_register (process, module->address,
                                      pe_ioh_get_size_of_image (image),
                                      IA32_MEMORY_ATTR_USER,
                                      image))
    {
      if (pe_ioh_get_data_directory_size (image,
                                        PEIMAGE_DIRECTORY_ENTRY_BASERELOC) <= 0
        || !ia32_memory_register (process, IA32_MEMORY_NULL,
                                           pe_ioh_get_size_of_image (image),
                                           IA32_MEMORY_ATTR_USER,
                                           image))
        {
          ia32_message_record_error (process->current,
                                "The base address or the file offset specified"
                                " does not have the proper alignment.");
          g_free (image);
          g_free (module->name);
          g_free (module->file);
          g_free (module);
          return IA32_MEMORY_NULL;
        }
      module->address = ia32_memory_virtual_address (process, image);
    }

  /* ja:インポート */
  if (pe_ioh_get_data_directory_size (image, PEIMAGE_DIRECTORY_ENTRY_IMPORT))
    {
      guint16 sections;
      guint32 size;

      sections = pe_ifh_get_number_of_sections (image);
      size = pe_ioh_get_size_of_image (image);
      for (address = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_IMPORT);
                            address + sizeof (ImageImportDescriptor) <= size;
                                    address += sizeof (ImageImportDescriptor))
        {
          gchar *name;
          guint32 import;
          ImageImportDescriptor *iid;

          iid = (ImageImportDescriptor *)(image + address);
          for (i = iid_get_name (iid); 0 < i && i < size; i++)
            if (image[i] == '\0')
              break;
          if (i <= 0 || size <= i)
            break;
          name = (gchar *)(image + iid_get_name (iid));
          import = ia32_module_load_library (process, name);
          if (import == IA32_MEMORY_NULL)
            {
              gint n = 0;
              guint32 lookup, thunk;
              ImageSectionHeader *ish;
              Ia32Osdll osdll;

              osdll.address = IA32_MEMORY_NULL;
              osdll.name = name;
              osdll.exports = NULL;
              ish = pe_image_section_header (image);
              for (i = 0; i < sections; i++)
                {
                  if (ish_get_virtual_address (ish)
                                        <= iid_get_original_first_thunk (iid)
                    && iid_get_original_first_thunk (iid)
                                                < ish_get_virtual_address (ish)
                                                + ish_get_virtual_size (ish))
                    break;
                  ish++;
                }
              lookup = i < sections ? iid_get_original_first_thunk (iid)
                                    : iid_get_first_thunk (iid);
              thunk = iid_get_first_thunk (iid);
              while (lookup + sizeof (guint32) <= size
                                        && thunk + sizeof (guint32) <= size)
                {
                  ImageThunkData *ilt, *iat;

                  ilt = (ImageThunkData *)(image + lookup);
                  iat = (ImageThunkData *)(image + thunk);
                  if (!itd_get_address_of_data (ilt)
                                            || !itd_get_address_of_data (iat))
                    break;
                  if (itd_get_ordinal (ilt) & PEIMAGE_ORDINAL_FLAG)
                    {
                      osdll.exports = g_realloc (osdll.exports,
                                                (n + 1) * sizeof (Ia32Export));
                      osdll.exports[n].argument = 0;
                      osdll.exports[n].ordinal = itd_get_ordinal (ilt)
                                                    & ~PEIMAGE_ORDINAL_FLAG;
                      osdll.exports[n].name
                        = g_strdup_printf ("#%d", osdll.exports[n].ordinal);
                      osdll.exports[n].func = NULL;
                      n++;
                    }
                  else if (itd_get_address_of_data (ilt)
                                                    + sizeof (guint16) <= size)
                    {
                      for (i = itd_get_address_of_data (ilt)
                                + sizeof (guint16); 0 <= i && i < size; i++)
                        if (image[i] == '\0')
                          {
                            const ImageImportByName *iibn;

                            osdll.exports = g_realloc (osdll.exports,
                                                (n + 1) * sizeof (Ia32Export));
                            iibn = (const ImageImportByName *)(image
                                            + itd_get_address_of_data (ilt));
                            osdll.exports[n].argument = 0;
                            osdll.exports[n].ordinal = G_MAXUINT16;
                            osdll.exports[n].name
                                            = g_strdup (iibn_get_name (iibn));
                            osdll.exports[n].func = NULL;
                            n++;
                            break;
                          }
                    }
                  lookup += sizeof (guint32);
                  thunk += sizeof (guint32);
                }
              osdll.exports = g_realloc (osdll.exports,
                                                (n + 1) * sizeof (Ia32Export));
              osdll.exports[n].argument = G_MAXINT;
              osdll.exports[n].ordinal = 0;
              osdll.exports[n].name = NULL;
              osdll.exports[n].func = NULL;
              import = ia32_module_create_library (process, &osdll);
              for (i = 0; i < n; i++)
                g_free (osdll.exports[n].name);
              g_free (osdll.exports);
            }
          if (import != IA32_MEMORY_NULL)
            {
              guint8 *library;
              guint32 lookup, thunk;
              ImageSectionHeader *ish;

              library = ia32_memory_real_address (process, import);
              ish = pe_image_section_header (image);
              for (i = 0; i < sections; i++)
                {
                  if (ish_get_virtual_address (ish)
                                        <= iid_get_original_first_thunk (iid)
                    && iid_get_original_first_thunk (iid)
                                                < ish_get_virtual_address (ish)
                                                + ish_get_virtual_size (ish))
                    break;
                  ish++;
                }
              lookup = i < sections ? iid_get_original_first_thunk (iid)
                                    : iid_get_first_thunk (iid);
              thunk = iid_get_first_thunk (iid);
              while (lookup + sizeof (guint32) <= size
                                        && thunk + sizeof (guint32) <= size)
                {
                  gchar *funcname = NULL;
                  gconstpointer func = NULL;
                  ImageThunkData *ilt, *iat;

                  ilt = (ImageThunkData *)(image + lookup);
                  iat = (ImageThunkData *)(image + thunk);
                  if (!itd_get_address_of_data (ilt)
                                            || !itd_get_address_of_data (iat))
                    break;
                  if (itd_get_ordinal (ilt) & PEIMAGE_ORDINAL_FLAG)
                    {
                      guint16 ordinal;

                      ordinal = itd_get_ordinal (ilt) & ~PEIMAGE_ORDINAL_FLAG;
                      func = peimage_file_get_proc (library, ordinal);
                      if (!func)
                        funcname = g_strdup_printf ("#%d", ordinal);
                    }
                  else if (itd_get_address_of_data (ilt)
                                                    + sizeof (guint16) <= size)
                    {
                      for (i = itd_get_address_of_data (ilt)
                                + sizeof (guint16); 0 <= i && i < size; i++)
                        if (image[i] == '\0')
                          {
                            const ImageImportByName *iibn;

                            iibn = (const ImageImportByName *)(image
                                            + itd_get_address_of_data (ilt));
                            func = peimage_file_get_func (library,
                                                        iibn_get_name (iibn));
                            if (!func)
                              funcname = g_strdup_printf ("\'%s\'",
                                                        iibn_get_name (iibn));
                            break;
                          }
                    }
                  if (!func)
                    {
                      gchar *str;

                      if (funcname)
                        {
                          str = g_strdup_printf (
                                    "%s could not be located in \'%s\'.",
                                                            funcname, name);
                          g_free (funcname);
                        }
                      else
                        {
                          str = g_strdup_printf ("The procedure entry point"
                                    " could not be located in \'%s\'.", name);
                        }
                      ia32_message_record_error (process->current, str);
                      g_free (str);
                      result = FALSE;
                      goto loop;
                    }
                  itd_set_function (iat,
                                ia32_memory_virtual_address (process, func));
                  lookup += sizeof (guint32);
                  thunk += sizeof (guint32);
                }
              module->library = g_list_append (module->library,
                                                    GUINT_TO_POINTER (import));
            }
          else
            {
              gchar *str;

              str = g_strdup_printf ("This application failed to start"
                                     " because \'%s\' was not found.", name);
              ia32_message_record_error (process->current, str);
              g_free (str);
              result = FALSE;
              goto loop;
            }
        }
    }
  loop:
  if (!result)
    {
      for (module->library = g_list_first (module->library); module->library;
                                        module->library = g_list_delete_link
                                            (module->library, module->library))
        ia32_module_free_library (process,
                                    GPOINTER_TO_UINT (module->library->data));
      ia32_memory_free (process, module->address);
      g_free (module->name);
      g_free (module->file);
      g_free (module);
      return IA32_MEMORY_NULL;
    }

  /* ja:再配置 */
  peimage_file_relocate (image,
                            module->address - pe_ioh_get_image_base (image));

  /* ja:プロセス登録 */
  process->module = g_list_append (process->module, module);

  return module->address;
}


/*  ja:イメージを解放する
    process,プロセス
    address,イメージ
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
ia32_module_free_library (Ia32Process   *process,
                          const guint32  address)
{
  gboolean result = FALSE;
  Ia32Module *module;

  module = ia32_module_get_info (process, address);
  if (module)
    {
      result = TRUE;
      module->counter--;
      if (module->counter <= 0)
        {
          guint8 *image;
          guint32 size;

          image = ia32_memory_real_address (process, address);
          size = ia32_memory_real_size (process, address);
          if (image && peimage_file_is_valid (image, size)
                                    && pe_ioh_get_data_directory_size (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT))
            {
              guint32 export;

              export = pe_ioh_get_data_directory_virtual_address (image,
                                            PEIMAGE_DIRECTORY_ENTRY_EXPORT);
              size = pe_ioh_get_size_of_image (image);
              if (export + sizeof (ImageExportDirectory) <= size)
                {
                  const ImageExportDirectory *ied;

                  ied = (const ImageExportDirectory *)(image + export);
                  if (ied_get_address_of_functions (ied)
                                            + ied_get_number_of_functions (ied)
                                                    * sizeof (guint32) <= size)
                    {
                      gint i;
                      const guint32 *functions;

                      functions = (const guint32 *)(image
                                        + ied_get_address_of_functions (ied));
                      for (i = 0; i < ied_get_number_of_functions (ied); i++)
                        {
                          guint32 rva;

                          rva = GUINT32_FROM_LE (functions[i]);
                          if (rva)
                            g_hash_table_remove (process->export,
                                            GUINT_TO_POINTER (address + rva));
                        }
                    }
                }
            }
          process->module = g_list_remove (process->module, module);
          result = ia32_memory_free (process, address);
          for (module->library = g_list_first (module->library);
                                                            module->library;
                                        module->library = g_list_delete_link
                                            (module->library, module->library))
            if (!ia32_module_free_library (process,
                                    GPOINTER_TO_UINT (module->library->data)))
              result = FALSE;
          g_free (module->name);
          g_free (module->file);
          g_free (module);
        }
    }
  return result;
}


/*  ja:イメージの名前を取得する
    process,プロセス
    address,イメージ
        RET,名前,NULL:エラー                                                */
const gchar *
ia32_module_get_filename (Ia32Process   *process,
                          const guint32  address)
{
  Ia32Module *module;

  module = ia32_module_get_info (process, address);
  return module ? module->file : NULL;
}


/*  ja:プロセスのイメージ一覧を取得する
    process,プロセス
        RET,リスト,NULL:エラー                                              */
GList *
ia32_module_get_list (Ia32Process *process)
{
  GList *module = NULL, *glist;

  for (glist = g_list_first (process->module);
                                            glist; glist = g_list_next (glist))
    module = g_list_append (module,
                    GUINT_TO_POINTER (((Ia32Module *)glist->data)->address));
  return module;
}
