/*
    wcommon
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "wcommon.h"
#include <shlwapi.h>
#include <shlobj.h>


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ファイルをメモリに読み込む
    lpszFile,ファイル名
    lpdwSize,ファイルサイズ
         RET,メモリ                                                         */
LPVOID WINAPI
LoadFileW (LPCWSTR lpszFile,
           LPDWORD lpdwSize)
{
  LPVOID lpBuffer = NULL;

  if (lpszFile)
    {
      BOOL fResult = FALSE;
      DWORD dwSize;
      HANDLE hFile;

      hFile = CreateFileW (lpszFile, GENERIC_READ, FILE_SHARE_READ, NULL,
                                OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
      if (hFile != INVALID_HANDLE_VALUE)
        {
          dwSize = GetFileSize (hFile, NULL);
          if (dwSize != INVALID_FILE_SIZE)
            {
              DWORD dwRead;

              lpBuffer = MemoryAlloc (((dwSize + 1) / sizeof (WCHAR) + 1)
                                                            * sizeof (WCHAR));
              if (lpBuffer)
                fResult = ReadFile (hFile, lpBuffer, dwSize, &dwRead, NULL)
                                                        && dwSize == dwRead;
            }
          if (!CloseHandle (hFile))
            fResult = FALSE;
        }
      if (!fResult)
        {
          MemoryFree (lpBuffer);
          lpBuffer = NULL;
        }
      else if (lpdwSize)
        {
          *lpdwSize = dwSize;
        }
    }
  return lpBuffer;
}


/*  ファイルをメモリに読み込む
    lpszFile,ファイル名
    lpdwSize,ファイルサイズ
         RET,メモリ                                                         */
LPVOID WINAPI
LoadFileA (LPCSTR  lpszFile,
           LPDWORD lpdwSize)
{
  LPVOID lpBuffer = NULL;

  if (lpszFile)
    {
      BOOL fResult = FALSE;
      DWORD dwSize;
      HANDLE hFile;

      hFile = CreateFileA (lpszFile, GENERIC_READ, FILE_SHARE_READ, NULL,
                                OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
      if (hFile != INVALID_HANDLE_VALUE)
        {
          dwSize = GetFileSize (hFile, NULL);
          if (dwSize != INVALID_FILE_SIZE)
            {
              DWORD dwRead;

              lpBuffer = MemoryAlloc (((dwSize + 1) / sizeof (WCHAR) + 1)
                                                            * sizeof (WCHAR));
              if (lpBuffer)
                fResult = ReadFile (hFile, lpBuffer, dwSize, &dwRead, NULL)
                                                        && dwSize == dwRead;
            }
          if (!CloseHandle (hFile))
            fResult = FALSE;
        }
      if (!fResult)
        {
          MemoryFree (lpBuffer);
          lpBuffer = NULL;
        }
      else if (lpdwSize)
        {
          *lpdwSize = dwSize;
        }
    }
  return lpBuffer;
}


/*  メモリをファイルに書き込む
    lpszFile,ファイル名
    lpBuffer,メモリ
      nSize,バイト数(負:NULL終端文字列)
         RET,TRUE:正常終了,FALSE:エラー                                     */
BOOL WINAPI
SaveFileW (LPCWSTR lpszFile,
           LPCVOID lpBuffer,
           SSIZE_T nSize)
{
  BOOL fResult = FALSE;

  if (lpszFile)
    {
      HANDLE hFile;
      WCHAR szDir[MAX_PATH];

      lstrcpyW (szDir, lpszFile);
      PathRemoveFileSpecW (szDir);
      SHCreateDirectoryExW (NULL, szDir, NULL);
      hFile = CreateFileW (lpszFile, GENERIC_WRITE, FILE_SHARE_READ, NULL,
                                CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
      if (hFile != INVALID_HANDLE_VALUE)
        {
          DWORD dwSize, dwWrite;

          dwSize = nSize >= 0 ? nSize: lstrlenW (lpBuffer) * sizeof (WCHAR);
          fResult = WriteFile (hFile, lpBuffer, dwSize, &dwWrite, NULL)
                                                        && dwSize == dwWrite;
          if (!CloseHandle (hFile))
            fResult = FALSE;
        }
    }
  return fResult;
}


/*  メモリをファイルに書き込む
    lpszFile,ファイル名
    lpBuffer,メモリ
      nSize,バイト数(負:NULL終端文字列)
         RET,TRUE:正常終了,FALSE:エラー                                     */
BOOL WINAPI
SaveFileA (LPCSTR  lpszFile,
           LPCVOID lpBuffer,
           SSIZE_T nSize)
{
  BOOL fResult = FALSE;

  if (lpszFile)
    {
      HANDLE hFile;
      CHAR szDir[MAX_PATH];

      lstrcpyA (szDir, lpszFile);
      PathRemoveFileSpecA (szDir);
      SHCreateDirectoryExA (NULL, szDir, NULL);
      hFile = CreateFileA (lpszFile, GENERIC_WRITE, FILE_SHARE_READ, NULL,
                                CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
      if (hFile != INVALID_HANDLE_VALUE)
        {
          DWORD dwSize, dwWrite;

          dwSize = nSize >= 0 ? nSize: lstrlenA (lpBuffer) * sizeof (CHAR);
          fResult = WriteFile (hFile, lpBuffer, dwSize, &dwWrite, NULL)
                                                        && dwSize == dwWrite;
          if (!CloseHandle (hFile))
            fResult = FALSE;
        }
    }
  return fResult;
}
