/*
    wcommon
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "wcommon.h"


/******************************************************************************
*                                                                             *
******************************************************************************/
typedef struct _HASHNODE {
  struct _HASHNODE *lpNext;
  LPVOID lpKey, lpValue;
} HASHNODE, *PHASHNODE, *LPHASHNODE;
typedef struct _HASH {
  HASHNODE *lpNode;
  HashCompare_t lpCompare;
  HashFree_t lpFreeKey, lpFreeValue;
  LPVOID lpData;
} HASH, *LPHASH;


/*  鍵を直接比較する
    lpKey1,鍵1
    lpKey2,鍵2
    lpData,ユーザデータ
       RET,TRUE:等しい,FALSE:異なる                                         */
BOOL WINAPI
HashCompareDirect (LPVOID lpKey1,
                   LPVOID lpKey2,
                   LPVOID lpData)
{
  return lpKey1 == lpKey2;
}


/*  鍵を文字列として比較する
    lpKey1,鍵1
    lpKey2,鍵2
    lpData,ユーザデータ
       RET,TRUE:等しい,FALSE:異なる                                         */
BOOL WINAPI
HashCompareStringA (LPVOID lpKey1,
                    LPVOID lpKey2,
                    LPVOID lpData)
{
  return lstrcmpA (lpKey1, lpKey2) == 0;
}


/*  鍵を文字列として比較する
    lpKey1,鍵1
    lpKey2,鍵2
    lpData,ユーザデータ
       RET,TRUE:等しい,FALSE:異なる                                         */
BOOL WINAPI
HashCompareStringW (LPVOID lpKey1,
                    LPVOID lpKey2,
                    LPVOID lpData)
{
  return lstrcmpW (lpKey1, lpKey2) == 0;
}


/*  ハッシュの鍵または値を解放する
    lpObject,鍵または値
      lpData,ユーザデータ                                                   */
VOID WINAPI
HashFree (LPVOID lpObject,
          LPVOID lpData)
{
  MemoryFree (lpObject);
}


/*  ハッシュテーブルを作成する
      lpHashCompare,比較関数
      lpHashFreeKey,鍵解放関数
    lpHashFreeValue,値解放関数
             lpData,ユーザデータ
                RET,ハッシュテーブル                                        */
LPHASH WINAPI
HashCreate (HashCompare_t lpHashCompare,
            HashFree_t    lpHashFreeKey,
            HashFree_t    lpHashFreeValue,
            LPVOID        lpData)
{
  LPHASH lpHash;

  lpHash = MemoryAlloc (sizeof (HASH));
  lpHash->lpCompare = lpHashCompare ? lpHashCompare : HashCompareDirect;
  lpHash->lpFreeKey = lpHashFreeKey;
  lpHash->lpFreeValue = lpHashFreeValue;
  lpHash->lpData = lpData;
  return lpHash;
}


/*  ハッシュテーブルを解放する
    lpHash,ハッシュテーブル                                                 */
VOID WINAPI
HashRelease (LPHASH lpHash)
{
  if (lpHash)
    while (lpHash->lpNode)
      {
        LPHASHNODE lpNode;

        lpNode = lpHash->lpNode;
        lpHash->lpNode = lpHash->lpNode->lpNext;
        if (lpHash->lpFreeKey)
          lpHash->lpFreeKey (lpNode->lpKey, lpHash->lpData);
        if (lpHash->lpFreeValue)
          lpHash->lpFreeValue (lpNode->lpValue, lpHash->lpData);
        MemoryFree (lpNode);
      }
  MemoryFree (lpHash);
}


/*  ハッシュテーブルに要素を挿入する
     lpHash,ハッシュテーブル
      lpKey,鍵
    lpValue,値
        RET,ハッシュテーブル                                                */
LPHASH WINAPI
HashInsert (LPHASH lpHash,
            LPVOID lpKey,
            LPVOID lpValue)
{
  LPHASHNODE lpNode;

  if (!lpHash)
    lpHash = HashCreate (NULL, NULL, NULL, NULL);
  if (lpHash->lpNode)
    {
      LPHASHNODE lpPrev = NULL;

      for (lpNode = lpHash->lpNode;
        lpNode && !lpHash->lpCompare (lpNode->lpKey, lpKey, lpHash->lpData);
                                                    lpNode = lpNode->lpNext)
        lpPrev = lpNode;
      if (lpNode)
        {
          if (lpHash->lpFreeKey)
            lpHash->lpFreeKey (lpNode->lpKey, lpHash->lpData);
          if (lpHash->lpFreeValue)
            lpHash->lpFreeValue (lpNode->lpValue, lpHash->lpData);
        }
      else
        {
          lpNode = MemoryAlloc (sizeof (HASHNODE));
          lpPrev->lpNext = lpNode;
        }
    }
  else
    {
      lpNode = lpHash->lpNode = MemoryAlloc (sizeof (HASHNODE));
    }
  lpNode->lpKey = lpKey;
  lpNode->lpValue = lpValue;
  return lpHash;
}


/*  ハッシュテーブルの値を求める
    lpHash,ハッシュテーブル
     lpKey,鍵
       RET,値(NULL:値なし)                                                  */
LPVOID WINAPI
HashLookup (LPHASH lpHash,
            LPVOID lpKey)
{
  LPHASHNODE lpNode;

  if (lpHash)
    for (lpNode = lpHash->lpNode; lpNode; lpNode = lpNode->lpNext)
      if (lpHash->lpCompare (lpNode->lpKey, lpKey, lpHash->lpData))
        return lpNode->lpValue;
  return NULL;
}


/*  ハッシュテーブルの要素を削除する
    lpHash,ハッシュテーブル
     lpKey,鍵
       RET,TRUE:削除できた,FALSE:削除できない                               */
BOOL WINAPI
HashRemove (LPHASH lpHash,
            LPVOID lpKey)
{
  if (lpHash)
    {
      LPHASHNODE lpNode, lpPrev = NULL;

      for (lpNode = lpHash->lpNode; lpNode; lpNode = lpNode->lpNext)
        {
          if (lpHash->lpCompare (lpNode->lpKey, lpKey, lpHash->lpData))
            {
              if (lpPrev)
                lpPrev->lpNext = lpNode->lpNext;
              else
                lpHash->lpNode = lpNode->lpNext;
              if (lpHash->lpFreeKey)
                lpHash->lpFreeKey (lpNode->lpKey, lpHash->lpData);
              if (lpHash->lpFreeValue)
                lpHash->lpFreeValue (lpNode->lpValue, lpHash->lpData);
              MemoryFree (lpNode);
              return TRUE;
            }
          lpPrev = lpNode;
        }
    }
  return FALSE;
}


/*  ハッシュテーブルの要素を列挙する
    lpHash,ハッシュテーブル
    lpProc,列挙関数
    lpData,ユーザデータ
       RET,TRUE:すべて列挙,FALSE:中断                                       */
BOOL WINAPI
HashEnum (LPHASH         lpHash,
          HashEnumProc_t lpProc,
          LPVOID         lpData)
{
  if (lpHash)
    {
      LPHASHNODE lpNode;

      for (lpNode = lpHash->lpNode; lpNode; lpNode = lpNode->lpNext)
        if (!lpProc (lpNode->lpKey, lpNode->lpValue, lpData))
          return FALSE;
    }
  return TRUE;
}
