/*
    wcommon
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "wcommon.h"


/******************************************************************************
*                                                                             *
******************************************************************************/
typedef struct _LIST {
  struct _LIST *lpNext;
  LPVOID lpValue;
} LIST, *LPLIST;


/*  値を直接比較する
    lpValue1,値1
    lpValue2,値2
      lpData,ユーザデータ
         RET,負:値1が小さい,0:等しい,正:値1が大きい                         */
int WINAPI
ListCompareDirect (LPVOID lpValue1,
                   LPVOID lpValue2,
                   LPVOID lpData)
{
  return (UINT_PTR)lpValue1 - (UINT_PTR)lpValue2;
}


/*  値を文字列として比較する
    lpValue1,値1
    lpValue2,値2
      lpData,ユーザデータ
         RET,負:値1が小さい,0:等しい,正:値1が大きい                         */
int WINAPI
ListCompareStringW (LPVOID lpValue1,
                    LPVOID lpValue2,
                    LPVOID lpData)
{
  return lstrcmpW (lpValue1, lpValue2);
}


/*  値を文字列として比較する
    lpValue1,値1
    lpValue2,値2
      lpData,ユーザデータ
         RET,負:値1が小さい,0:等しい,正:値1が大きい                         */
int WINAPI
ListCompareStringA (LPVOID lpValue1,
                    LPVOID lpValue2,
                    LPVOID lpData)
{
  return lstrcmpA (lpValue1, lpValue2);
}


/*  リストの値を解放する
    lpValue,値
     lpData,ユーザデータ                                                    */
VOID WINAPI
ListFree (LPVOID lpValue,
          LPVOID lpData)
{
  MemoryFree (lpValue);
}


/*  線形リストの末尾に追加する
    lpStart,線形リスト(NULL:新規作成)
    lpValue,値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListAppend (LPLIST lpStart,
            LPVOID lpValue)
{
  LPLIST lpNew, lpList;

  lpNew = MemoryAlloc (sizeof (LIST));
  lpNew->lpValue = lpValue;
  if (!lpStart)
    return lpNew;
  for (lpList = lpStart; lpList->lpNext; lpList = lpList->lpNext);
  lpList->lpNext = lpNew;
  return lpStart;
}


/*  線形リストの先頭に追加する
    lpStart,線形リスト(NULL:新規作成)
    lpValue,値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListPrepend (LPLIST lpStart,
             LPVOID lpValue)
{
  LPLIST lpNew;

  lpNew = MemoryAlloc (sizeof (LIST));
  lpNew->lpValue = lpValue;
  lpNew->lpNext = lpStart;
  return lpNew;
}


/*  線形リストに追加する
      lpStart,線形リスト(NULL:新規作成)
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListInsert (LPLIST        lpStart,
            LPVOID        lpValue,
            ListCompare_t lpCompare,
            LPVOID        lpData)
{
  LPLIST lpNew, lpList, lpPrev = NULL;

  lpNew = MemoryAlloc (sizeof (LIST));
  lpNew->lpValue = lpValue;
  if (!lpStart)
    return lpNew;
  if (!lpCompare)
    lpCompare = ListCompareDirect;
  for (lpList = lpStart; lpList->lpNext; lpList = lpList->lpNext)
    {
      if (lpCompare (lpList->lpValue, lpValue, lpData) > 0)
        {
          lpNew->lpNext = lpList;
          if (lpPrev)
            {
              lpPrev->lpNext = lpNew;
              return lpStart;
            }
          return lpNew;
        }
      lpPrev = lpList;
    }
  lpList->lpNext = lpNew;
  return lpStart;
}


/*  線形リストをソートする
      lpStart,線形リスト
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListSort (LPLIST        lpStart,
          ListCompare_t lpCompare,
          LPVOID        lpData)
{
  LPLIST lpFirst = NULL, lpLast = NULL;

  if (!lpCompare)
    lpCompare = ListCompareDirect;
  while (lpStart)
    {
      LPLIST lpList, lpPrev = NULL, lpSmall, lpSmallPrev = NULL;

      lpSmall = lpStart;
      for (lpList = lpStart->lpNext; lpList; lpList = lpList->lpNext)
        {
          if (lpCompare (lpList->lpValue, lpSmall->lpValue, lpData) < 0)
            {
              lpSmall = lpList;
              lpSmallPrev = lpPrev;
            }
          lpPrev = lpList;
        }
      if (lpSmallPrev)
        lpSmallPrev->lpNext = lpSmall->lpNext;
      else
        lpStart = lpSmall->lpNext;
      if (lpLast)
        lpLast->lpNext = lpSmall;
      else
        lpFirst = lpSmall;
      lpLast = lpSmall;
    }
  if (lpLast)
    lpLast->lpNext = NULL;
  return lpFirst;
}


/*  線形リストの先頭を求める
    lpStart,線形リスト
        RET,値(NULL:要素なし)                                               */
LPVOID WINAPI
ListFirst (LPLIST lpStart)
{
  return lpStart ? lpStart->lpValue : NULL;
}


/*  線形リストの末尾を求める
    lpStart,線形リスト
        RET,値(NULL:要素なし)                                               */
LPVOID WINAPI
ListLast (LPLIST lpStart)
{
  if (lpStart)
    {
      while (lpStart->lpNext)
        lpStart = lpStart->lpNext;
      return lpStart->lpValue;
    }
  return NULL;
}


/*  線形リストのn番目を求める
    lpStart,線形リスト
     nIndex,n番(負:後方から,正or0:前方から)
        RET,値(NULL:要素なし)                                               */
LPVOID WINAPI
ListIndex (LPLIST lpStart,
           int    nIndex)
{
  int nCount = 0;

  if (nIndex < 0)
    nIndex += ListLength (lpStart);
  while (lpStart)
    {
      if (nCount == nIndex)
        return lpStart->lpValue;
      nCount++;
      lpStart = lpStart->lpNext;
    }
  return NULL;
}


/*  線形リストの要素数を求める
    lpStart,線形リスト
        RET,要素数                                                          */
int WINAPI
ListLength (LPLIST lpStart)
{
  int nCount = 0;

  while (lpStart)
    {
      nCount++;
      lpStart = lpStart->lpNext;
    }
  return nCount;
}


/*  線形リストの一致する要素の数を求める
      lpStart,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,一致数                                                        */
int WINAPI
ListHasValue (LPLIST        lpStart,
              LPVOID        lpValue,
              ListCompare_t lpCompare,
              LPVOID        lpData)
{
  int nCount = 0;

  while (lpStart)
    {
      if (lpCompare ? lpCompare (lpStart->lpValue, lpValue, lpData) == 0
                    : lpStart->lpValue == lpValue)
        nCount++;
      lpStart = lpStart->lpNext;
    }
  return nCount;
}


/*  線形リストの一致する要素を削除する
      lpStart,線形リスト
      lpValue,値
         fAll,TRUE:すべて削除,FALSE:最初の要素のみ削除
    lpCompare,比較関数
       lpFree,解放関数
       lpData,ユーザデータ
          RET,線形リスト(NULL:すべて削除)                                   */
LPLIST WINAPI
ListRemove (LPLIST        lpStart,
            LPVOID        lpValue,
            BOOL          fAll,
            ListCompare_t lpCompare,
            ListFree_t    lpFree,
            LPVOID        lpData)
{
  LPLIST lpList, lpPrev = NULL;

  lpList = lpStart;
  while (lpList)
    if (lpCompare ? lpCompare (lpList->lpValue, lpValue, lpData) == 0
                  : lpList->lpValue == lpValue)
      {
        LPLIST lpNext;

        lpNext = lpList->lpNext;
        if (lpPrev)
          lpPrev->lpNext = lpNext;
        else
          lpStart = lpNext;
        if (lpFree)
          lpFree (lpList->lpValue, lpData);
        MemoryFree (lpList);
        if (!fAll)
          break;
        lpList = lpNext;
      }
    else
      {
        lpPrev = lpList;
        lpList = lpList->lpNext;
      }
  return lpStart;
}


/*  線形リストを解放する
    lpStart,線形リスト
     lpFree,解放関数
     lpData,ユーザデータ                                                    */
VOID WINAPI
ListRelease (LPLIST     lpStart,
             ListFree_t lpFree,
             LPVOID     lpData)
{
  while (lpStart)
    {
      LPLIST lpNext;

      lpNext = lpStart->lpNext;
      if (lpFree)
        lpFree (lpStart->lpValue, lpData);
      MemoryFree (lpStart);
      lpStart = lpNext;
    }
}


/*  線形リストの要素を列挙する
    lpStart,線形リスト
    lpFree,列挙関数
    lpData,ユーザデータ
       RET,TRUE:すべて列挙,FALSE:中断                                       */
BOOL WINAPI
ListEnum (LPLIST         lpStart,
          ListEnumProc_t lpProc,
          LPVOID         lpData)
{
  if (lpProc)
    while (lpStart)
      {
        if (!lpProc (lpStart->lpValue, lpData))
          return FALSE;
        lpStart = lpStart->lpNext;
      }
  return lpProc != NULL;
}
