/*
    wcommon
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "wcommon.h"


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  メモリの内容を埋める
    lpvDest,メモリ
      bByte,内容
     uBytes,バイト数                                                        */
VOID WINAPI
MemorySet (LPVOID lpvDest,
           BYTE   bByte,
           SIZE_T uBytes)
{
  if (lpvDest)
    {
      LPBYTE p, q;

      p = lpvDest;
      q = p + uBytes;
      while (p < q)
        *p++ = bByte;
    }
}


/*  メモリの内容をコピーする
    lpvDest,コピー先
     lpvSrc,コピー元
     uBytes,バイト数                                                        */
VOID WINAPI
MemoryCopy (LPVOID  lpvDest,
            LPCVOID lpvSrc,
            SIZE_T  uBytes)
{
  if (lpvDest && lpvSrc && uBytes > 0)
    {
      LPBYTE p;
      LPCBYTE q;

      p = lpvDest;
      q = lpvSrc;
      if ((UINT_PTR)lpvDest < (UINT_PTR)lpvSrc)
        {
          LPBYTE r;

          r = p + uBytes;
          while (p < r)
            *p++ = *q++;
        }
      else if ((UINT_PTR)lpvDest > (UINT_PTR)lpvSrc)
        {
          LPBYTE r;

          r = p;
          uBytes--;
          p += uBytes;
          q += uBytes;
          while (p >= r)
            *p-- = *q--;
        }
    }
}


/*  メモリの内容を比較する
    lpvBuf1,バッファ1
    lpvBuf1,バッファ2
     uBytes,バイト数
        RET,0:等しい,負:バッファ1は小さい,正:バッファ1は大きい              */
int WINAPI
MemoryCompare (LPCVOID lpvBuf1,
               LPCVOID lpvBuf2,
               SIZE_T  uBytes)
{
  if (lpvBuf1 && lpvBuf2)
    {
      LPCBYTE p, q, r;

      p = lpvBuf1;
      q = lpvBuf2;
      r = p + uBytes;
      while (p < r)
        {
          int n;

          n = *p - *q;
          if (n != 0)
            return n;
          p++;
          q++;
        }
    }
  else if (lpvBuf1)
    return INT_MAX;
  else if (lpvBuf2)
    return INT_MIN;
  return 0;
}


/*  メモリを確保する
    uBytes,サイズ
       RET,確保したメモリ,NULL:エラー                                       */
LPVOID WINAPI
MemoryAlloc (SIZE_T uBytes)
{
  return uBytes > 0 ? HeapAlloc (GetProcessHeap (), HEAP_ZERO_MEMORY, uBytes)
                    : NULL;
}


/*  メモリを解放する
    lpMemory,確保したメモリ
         RET,TRUE:正常終了,FALSE:エラー                                     */
BOOL WINAPI
MemoryFree (LPVOID lpMemory)
{
  return !lpMemory || HeapFree (GetProcessHeap (), 0, lpMemory);
}


/*  メモリのサイズを変更する
    lpMemory,確保したメモリ
      uBytes,サイズ
         RET,確保したメモリ,NULL:エラー                                     */
LPVOID WINAPI
MemoryReAlloc (LPVOID lpMemory,
               SIZE_T uBytes)
{
  return uBytes > 0 ? lpMemory ? HeapReAlloc (GetProcessHeap (),
                                            HEAP_ZERO_MEMORY, lpMemory, uBytes)
                               : MemoryAlloc (uBytes)
                    : (MemoryFree (lpMemory), NULL);
}


/*  メモリを複製する
    lpMemory,複製するメモリ
      uBytes,サイズ
         RET,確保したメモリ,NULL:エラー                                     */
LPVOID WINAPI
MemoryDuplicate (LPCVOID lpMemory,
                 SIZE_T  uBytes)
{
  LPVOID lpResult;

  if (lpMemory)
    {
      lpResult = MemoryAlloc (uBytes);
      MemoryCopy (lpResult, lpMemory, uBytes);
    }
  else
    {
      lpResult = NULL;
    }
  return lpResult;
}
