/*
    wcommon
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "wcommon.h"
#include <shlwapi.h>


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  文字列を結合する
    lpszSeparator,区切り文字列
              RET,文字列                                                    */
LPWSTR WINAPI
StringJoinW (LPCWSTR lpszSeparator, ...)
{
  LPWSTR lpszResult = NULL;
  LPCWSTR lpszString;
  int i = 0, nLen = 1;
  va_list arglist;

  va_start (arglist, lpszSeparator);
  while (lpszString = va_arg (arglist, LPCWSTR))
    {
      nLen += lstrlenW (lpszString);
      i++;
    }
  va_end (arglist);
  if (lpszSeparator && i > 1)
    nLen += lstrlenW (lpszSeparator) * (i - 1);
  if (nLen > 1)
    {
      lpszResult = MemoryAlloc (nLen * sizeof (WCHAR));
      va_start (arglist, lpszSeparator);
      lpszString = va_arg (arglist, LPCWSTR);
      while (lpszString)
        {
          lstrcatW (lpszResult, lpszString);
          lpszString = va_arg (arglist, LPCWSTR);
          if (lpszSeparator && lpszString)
            lstrcatW (lpszResult, lpszSeparator);
        }
      va_end (arglist);
    }
  return lpszResult;
}


/*  文字列を結合する
    lpszSeparator,区切り文字列
              RET,文字列                                                    */
LPSTR WINAPI
StringJoinA (LPCSTR lpszSeparator, ...)
{
  LPSTR lpszResult = NULL;
  LPCSTR lpszString;
  int i = 0, nLen = 1;
  va_list arglist;

  va_start (arglist, lpszSeparator);
  while (lpszString = va_arg (arglist, LPCSTR))
    {
      nLen += lstrlenA (lpszString);
      i++;
    }
  va_end (arglist);
  if (lpszSeparator && i > 1)
    nLen += lstrlenA (lpszSeparator) * (i - 1);
  if (nLen > 1)
    {
      lpszResult = MemoryAlloc (nLen * sizeof (CHAR));
      va_start (arglist, lpszSeparator);
      lpszString = va_arg (arglist, LPCSTR);
      while (lpszString)
        {
          lstrcatA (lpszResult, lpszString);
          lpszString = va_arg (arglist, LPCSTR);
          if (lpszSeparator && lpszString)
            lstrcatA (lpszResult, lpszSeparator);
        }
      va_end (arglist);
    }
  return lpszResult;
}


/*  文字列を結合する
    lpszSeparator,区切り文字列
       plpszArray,分割された文字列
              RET,文字列                                                    */
LPWSTR WINAPI
StringJoinVW (LPCWSTR  lpszSeparator,
              LPCWSTR *plpszArray)
{
  LPWSTR lpszResult = NULL;

  if (plpszArray)
    {
      int i, nLen = 1;

      for (i = 0; plpszArray[i]; i++)
        nLen += lstrlenW (plpszArray[i]);
      if (lpszSeparator && i > 1)
        nLen += lstrlenW (lpszSeparator) * (i - 1);
      if (nLen > 1)
        {
          lpszResult = MemoryAlloc (nLen * sizeof (WCHAR));
          while (*plpszArray)
            {
              lstrcatW (lpszResult, *plpszArray);
              plpszArray++;
              if (lpszSeparator && *plpszArray)
                lstrcatW (lpszResult, lpszSeparator);
            }
        }
    }
  return lpszResult;
}


/*  文字列を結合する
    lpszSeparator,区切り文字列
       plpszArray,分割された文字列
              RET,文字列                                                    */
LPSTR WINAPI
StringJoinVA (LPCSTR  lpszSeparator,
              LPCSTR *plpszArray)
{
  LPSTR lpszResult = NULL;

  if (plpszArray)
    {
      int i, nLen = 1;

      for (i = 0; plpszArray[i]; i++)
        nLen += lstrlenA (plpszArray[i]);
      if (lpszSeparator && i > 1)
        nLen += lstrlenA (lpszSeparator) * (i - 1);
      if (nLen > 1)
        {
          lpszResult = MemoryAlloc (nLen * sizeof (CHAR));
          while (*plpszArray)
            {
              lstrcatA (lpszResult, *plpszArray);
              plpszArray++;
              if (lpszSeparator && *plpszArray)
                lstrcatA (lpszResult, lpszSeparator);
            }
        }
    }
  return lpszResult;
}


/*  分割された文字列の数を求める
    plpszArray,分割された文字列
           RET,分割された文字列の数                                         */
int WINAPI
StringLengthVW (LPCWSTR *plpszArray)
{
  int nResult = 0;

  if (plpszArray)
    while (plpszArray[nResult])
      nResult++;
  return nResult;
}


/*  分割された文字列の数を求める
    plpszArray,分割された文字列
           RET,分割された文字列の数                                         */
int WINAPI
StringLengthVA (LPCSTR *plpszArray)
{
  int nResult = 0;

  if (plpszArray)
    while (plpszArray[nResult])
      nResult++;
  return nResult;
}


/*  分割された文字列をコピーする
    plpszArray,分割された文字列
           RET,分割された文字列                                             */
LPWSTR * WINAPI
StringCopyVW (LPCWSTR *plpszArray)
{
  LPWSTR *plpszResult = NULL;

  if (plpszArray)
    {
      int nCount = 0, nLength = 0;
      LPCWSTR *pp;

      for (pp = plpszArray; *pp; pp++)
        {
          nLength += lstrlenW (*pp) + 1;
          nCount++;
        }
      if (nCount > 0)
        {
          LPWSTR q, *qq;

          qq = plpszResult = MemoryAlloc (++nCount * sizeof (LPWSTR)
                                                + nLength * sizeof (WCHAR));
          q = (LPWSTR)(plpszResult + nCount);
          for (pp = plpszArray; *pp; pp++)
            {
              *qq++ = q;
              nLength = lstrlenW (*pp) + 1;
              MemoryCopy (q, *pp, nLength * sizeof (WCHAR));
              q += nLength;
            }
        }
    }
  return plpszResult;
}


/*  分割された文字列をコピーする
    plpszArray,分割された文字列
           RET,分割された文字列                                             */
LPSTR * WINAPI
StringCopyVA (LPCSTR *plpszArray)
{
  LPSTR *plpszResult = NULL;

  if (plpszArray)
    {
      int nCount = 0, nLength = 0;
      LPCSTR *pp;

      for (pp = plpszArray; *pp; pp++)
        {
          nLength += lstrlenA (*pp) + 1;
          nCount++;
        }
      if (nCount > 0)
        {
          LPSTR q, *qq;

          qq = plpszResult = MemoryAlloc (++nCount * sizeof (LPSTR)
                                                + nLength * sizeof (CHAR));
          q = (LPSTR)(plpszResult + nCount);
          for (pp = plpszArray; *pp; pp++)
            {
              *qq++ = q;
              nLength = lstrlenA (*pp) + 1;
              MemoryCopy (q, *pp, nLength * sizeof (CHAR));
              q += nLength;
            }
        }
    }
  return plpszResult;
}


/*  分割された文字列のNULL文字列を削除する
    plpszArray,分割された文字列
           RET,分割された文字列                                             */
LPWSTR * WINAPI
StringShrinkVW (LPCWSTR *plpszArray)
{
  LPWSTR *plpszResult = NULL;

  if (plpszArray)
    {
      int nCount = 0, nLength = 0;
      LPCWSTR *pp;

      for (pp = plpszArray; *pp; pp++)
        if (**pp != '\0')
          {
            nLength += lstrlenW (*pp) + 1;
            nCount++;
          }
      if (nCount > 0)
        {
          LPWSTR q, *qq;

          qq = plpszResult = MemoryAlloc (++nCount * sizeof (LPWSTR)
                                                + nLength * sizeof (WCHAR));
          q = (LPWSTR)(plpszResult + nCount);
          for (pp = plpszArray; *pp; pp++)
            if (**pp != '\0')
              {
                *qq++ = q;
                nLength = lstrlenW (*pp) + 1;
                MemoryCopy (q, *pp, nLength * sizeof (WCHAR));
                q += nLength;
              }
        }
    }
  return plpszResult;
}


/*  分割された文字列のNULL文字列を削除する
    plpszArray,分割された文字列
           RET,分割された文字列                                             */
LPSTR * WINAPI
StringShrinkVA (LPCSTR *plpszArray)
{
  LPSTR *plpszResult = NULL;

  if (plpszArray)
    {
      int nCount = 0, nLength = 0;
      LPCSTR *pp;

      for (pp = plpszArray; *pp; pp++)
        if (**pp != '\0')
          {
            nLength += lstrlenA (*pp) + 1;
            nCount++;
          }
      if (nCount > 0)
        {
          LPSTR q, *qq;

          qq = plpszResult = MemoryAlloc (++nCount * sizeof (LPSTR)
                                                + nLength * sizeof (CHAR));
          q = (LPSTR)(plpszResult + nCount);
          for (pp = plpszArray; *pp; pp++)
            if (**pp != '\0')
              {
                *qq++ = q;
                nLength = lstrlenA (*pp) + 1;
                MemoryCopy (q, *pp, nLength * sizeof (CHAR));
                q += nLength;
              }
        }
    }
  return plpszResult;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
static int CALLBACK
StringSplitDelimiterCallbackW (LPCWSTR lpszString,
                               LPVOID  lpData)
{
  return lpszString && lpData && StrChrW (lpData, *lpszString) ? 1 : 0;
}


static int CALLBACK
StringSplitDelimiterCallbackA (LPCSTR lpszString,
                               LPVOID lpData)
{
  return lpszString && lpData && StrChrA (lpData, *lpszString) ? 1 : 0;
}


static int CALLBACK
StringSplitDelimiterCallbackIW (LPCWSTR lpszString,
                                LPVOID  lpData)
{
  return lpszString && lpData && StrChrIW (lpData, *lpszString) ? 1 : 0;
}


static int CALLBACK
StringSplitDelimiterCallbackIA (LPCSTR lpszString,
                                LPVOID lpData)
{
  return lpszString && lpData && StrChrIA (lpData, *lpszString) ? 1 : 0;
}


static int CALLBACK
StringSplitDelimitersCallbackW (LPCWSTR lpszString,
                                LPVOID  lpData)
{
  return lpszString && lpData ? StrSpnW (lpszString, lpData) : 0;
}


static int CALLBACK
StringSplitDelimitersCallbackA (LPCSTR lpszString,
                                LPVOID lpData)
{
  return lpszString && lpData ? StrSpnA (lpszString, lpData) : 0;
}


static int CALLBACK
StringSplitDelimitersCallbackIW (LPCWSTR lpszString,
                                 LPVOID  lpData)
{
  if (lpszString && lpData)
    {
      LPCWSTR p;

      for (p = lpszString; *p != '\0'; p++)
        if (!StrChrIW (lpData, *p))
          break;
      return p - lpszString;
    }
  return 0;
}


static int CALLBACK
StringSplitDelimitersCallbackIA (LPCSTR lpszString,
                                 LPVOID lpData)
{
  if (lpszString && lpData)
    {
      LPCSTR p;

      for (p = lpszString; *p != '\0'; p++)
        if (IsDBCSLeadByteEx (CP_ACP, *p) || !StrChrIA (lpData, *p))
          break;
      return p - lpszString;
    }
  return 0;
}


static int CALLBACK
StringSplitSeparatorCallbackW (LPCWSTR lpszString,
                               LPVOID  lpData)
{
  if (lpszString && lpData)
    {
      int n;

      n = lstrlenW (lpData);
      if (StrCmpNW (lpszString, lpData, n) == 0)
        return n;
    }
  return 0;
}


static int CALLBACK
StringSplitSeparatorCallbackA (LPCSTR lpszString,
                               LPVOID lpData)
{
  if (lpszString && lpData)
    {
      int n;

      n = lstrlenA (lpData);
      if (StrCmpNA (lpszString, lpData, n) == 0)
        return n;
    }
  return 0;
}


static int CALLBACK
StringSplitSeparatorCallbackIW (LPCWSTR lpszString,
                                LPVOID  lpData)
{
  if (lpszString && lpData)
    {
      int n;

      n = lstrlenW (lpData);
      if (StrCmpNIW (lpszString, lpData, n) == 0)
        return n;
    }
  return 0;
}


static int CALLBACK
StringSplitSeparatorCallbackIA (LPCSTR lpszString,
                                LPVOID lpData)
{
  if (lpszString && lpData)
    {
      int n;

      n = lstrlenA (lpData);
      if (StrCmpNIA (lpszString, lpData, n) == 0)
        return n;
    }
  return 0;
}


static int CALLBACK
StringSplitComplementCallbackW (LPCWSTR lpszString,
                                LPVOID  lpData)
{
  return lpszString && lpData ? StrCSpnW (lpszString, lpData) : 0;
}


static int CALLBACK
StringSplitComplementCallbackA (LPCSTR lpszString,
                                LPVOID lpData)
{
  return lpszString && lpData ? StrCSpnA (lpszString, lpData) : 0;
}


static int CALLBACK
StringSplitComplementCallbackIW (LPCWSTR lpszString,
                                 LPVOID  lpData)
{
  return lpszString && lpData ? StrCSpnIW (lpszString, lpData) : 0;
}


static int CALLBACK
StringSplitComplementCallbackIA (LPCSTR lpszString,
                                 LPVOID lpData)
{
  return lpszString && lpData ? StrCSpnIA (lpszString, lpData) : 0;
}


/*  文字列を分割する
         lpszString,文字列
    StringSplitProc,コールバック関数
             lpData,ユーザデータ
                RET,分割された文字列                                        */
LPWSTR * WINAPI
StringSplitW (LPCWSTR            lpszString,
              StringSplitProcW_t StringSplitProc,
              LPVOID             lpData)
{
  LPWSTR *plpszArray = NULL;

  if (lpszString && StringSplitProc)
    {
      int nCount = 0, nLength = 0, *lpnStatus, *q;
      LPCWSTR p;
      LPWSTR r, *pp;

      p = lpszString;
      q = lpnStatus = MemoryAlloc ((lstrlenW (lpszString) + 1) * sizeof (int));
      while (*p != '\0')
        {
          *q = StringSplitProc (p, lpData);
          if (*q > 0)
            {
              nCount++;
              p += *q;
              q += *q;
            }
          else if (*q < 0)
            {
              goto final;
            }
          else
            {
              nLength++;
              p++;
              q++;
            }
        }
      pp = plpszArray = MemoryAlloc ((nCount + 2) * sizeof (LPWSTR)
                                    + (nLength + nCount + 1) * sizeof (WCHAR));
      p = lpszString;
      q = lpnStatus;
      r = (LPWSTR)(plpszArray + nCount + 2);
      *pp++ = r;
      while (*p != '\0')
        {
          if (*q > 0)
            {
              p += *q;
              q += *q;
              *pp++ = ++r;
            }
          else
            {
              *r++ = *p++;
              q++;
            }
        }
      final:
      MemoryFree (lpnStatus);
    }
  return plpszArray;
}


/*  文字列を分割する
         lpszString,文字列
    StringSplitProc,コールバック関数
             lpData,ユーザデータ
                RET,分割された文字列                                        */
LPSTR * WINAPI
StringSplitA (LPCSTR             lpszString,
              StringSplitProcA_t StringSplitProc,
              LPVOID             lpData)
{
  LPSTR *plpszArray = NULL;

  if (lpszString && StringSplitProc)
    {
      int nCount = 0, nLength = 0, *lpnStatus, *q;
      LPCSTR p;
      LPSTR r, *pp;

      p = lpszString;
      q = lpnStatus = MemoryAlloc ((lstrlenA (lpszString) + 1) * sizeof (int));
      while (*p != '\0')
        {
          *q = StringSplitProc (p, lpData);
          if (*q > 0)
            {
              nCount++;
              p += *q;
              q += *q;
            }
          else if (*q < 0)
            {
              goto final;
            }
          else if (IsDBCSLeadByteEx (CP_ACP, *p))
            {
              nLength += 2;
              p += 2;
              q += 2;
            }
          else
            {
              nLength++;
              p++;
              q++;
            }
        }
      pp = plpszArray = MemoryAlloc ((nCount + 2) * sizeof (LPSTR)
                                    + (nLength + nCount + 1) * sizeof (CHAR));
      p = lpszString;
      q = lpnStatus;
      r = (LPSTR)(plpszArray + nCount + 2);
      *pp++ = r;
      while (*p != '\0')
        {
          if (*q > 0)
            {
              p += *q;
              q += *q;
              *pp++ = ++r;
            }
          else
            {
              *r++ = *p++;
              q++;
            }
        }
      final:
      MemoryFree (lpnStatus);
    }
  return plpszArray;
}


#define StringSplitProcedureW(prefix,suffix) \
LPWSTR * WINAPI \
StringSplit ## prefix ## suffix (LPCWSTR lpszString, \
                                 LPCWSTR lpszData) \
{ \
  return StringSplitW (lpszString, \
                       StringSplit ## prefix ## Callback ## suffix, \
                       (LPVOID)lpszData); \
}


#define StringSplitProcedureA(prefix,suffix) \
LPSTR * WINAPI \
StringSplit ## prefix ## suffix (LPCSTR lpszString, \
                                 LPCSTR lpszData) \
{ \
  return StringSplitA (lpszString, \
                       StringSplit ## prefix ## Callback ## suffix, \
                       (LPVOID)lpszData); \
}


StringSplitProcedureW(Delimiter,W)
StringSplitProcedureA(Delimiter,A)
StringSplitProcedureW(Delimiter,IW)
StringSplitProcedureA(Delimiter,IA)
StringSplitProcedureW(Delimiters,W)
StringSplitProcedureA(Delimiters,A)
StringSplitProcedureW(Delimiters,IW)
StringSplitProcedureA(Delimiters,IA)
StringSplitProcedureW(Separator,W)
StringSplitProcedureA(Separator,A)
StringSplitProcedureW(Separator,IW)
StringSplitProcedureA(Separator,IA)
StringSplitProcedureW(Complement,W)
StringSplitProcedureA(Complement,A)
StringSplitProcedureW(Complement,IW)
StringSplitProcedureA(Complement,IA)
