/*
    common
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef __COMMON_H__
#define __COMMON_H__


#include <windows.h>


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


#define n_elements(arr) (sizeof (arr) / sizeof ((arr)[0]))


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  文字が数字であるかどうかを判断する
     ch,文字
    RET,TRUE:数字,FALSE:その他                                              */
BOOL WINAPI
IsCharNumericW (WCHAR ch);


/*  文字が数字であるかどうかを判断する
     ch,文字
    RET,TRUE:数字,FALSE:その他                                              */
BOOL WINAPI
IsCharNumericA (CHAR ch);


/*  文字がコントロールコードであるかどうかを判断する
     ch,文字
    RET,TRUE:コントロールコード,FALSE:その他                                */
BOOL WINAPI
IsCharControlW (WCHAR ch);


/*  文字がコントロールコードであるかどうかを判断する
     ch,文字
    RET,TRUE:コントロールコード,FALSE:その他                                */
BOOL WINAPI
IsCharControlA (CHAR ch);


#ifdef UNICODE
# define IsCharNumeric IsCharNumericW
# define IsCharControl IsCharControlW
#else /* not UNICODE */
# define IsCharNumeric IsCharNumericA
# define IsCharControl IsCharControlA
#endif /* not UNICODE */


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  10進数文字列を数値に変換する
    lpszText,文字列
         RET,数値                                                           */
DWORD WINAPI
DecW (LPCWSTR lpszText);


/*  10進数文字列を数値に変換する
    lpszText,文字列
         RET,数値                                                           */
DWORD WINAPI
DecA (LPCSTR lpszText);


/*  16進数文字列を数値に変換する
    lpszText,文字列
         RET,数値                                                           */
DWORD WINAPI
HexW (LPCWSTR lpszText);


/*  16進数文字列を数値に変換する
    lpszText,文字列
         RET,数値                                                           */
DWORD WINAPI
HexA (LPCSTR lpszText);


/*  ja:数値→文字列
        nValue,数値
    lpszString,文字列
        nRadix,基数
         nWide,桁数
       fSigned,TRUE:符号あり,FALSE:符号なし                                 */
VOID WINAPI
ValStrW (int    nValue,
         LPWSTR lpszString,
         int    nRadix,
         int    nWide,
         BOOL   fSigned);


/*  ja:数値→文字列
        nValue,数値
    lpszString,文字列
        nRadix,基数
         nWide,桁数
       fSigned,TRUE:符号あり,FALSE:符号なし                                 */
VOID WINAPI
ValStrA (int   nValue,
         LPSTR lpszString,
         int   nRadix,
         int   nWide,
         BOOL  fSigned);


/*  ja:文字列→数値
        nValue,数値
    lpszString,文字列
        nRadix,基数
           RET,TRUE:正常終了,FALSE:エラー                                   */
BOOL WINAPI
StrValW (int     *nValue,
         LPCWSTR  lpszString,
         int      nRadix);


/*  ja:文字列→数値
        nValue,数値
    lpszString,文字列
        nRadix,基数
           RET,TRUE:正常終了,FALSE:エラー                                   */
BOOL WINAPI
StrValA (int    *nValue,
         LPCSTR  lpszString,
         int     nRadix);


#ifdef UNICODE
# define Dec DecW
# define Hex HexW
# define ValStr ValStrW
# define StrVal StrValW
#else /* not UNICODE */
# define Dec DecA
# define Hex HexA
# define ValStr ValStrA
# define StrVal StrValA
#endif /* not UNICODE */


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  メモリの内容を埋める
    lpvDest,メモリ
      bByte,内容
     uBytes,バイト数                                                        */
VOID WINAPI
MemorySet (LPVOID lpvDest,
           BYTE   bByte,
           SIZE_T uBytes);


/*  メモリの内容をコピーする
    lpvDest,コピー先
     lpvSrc,コピー元
     uBytes,バイト数                                                        */
VOID WINAPI
MemoryCopy (LPVOID  lpvDest,
            LPCVOID lpvSrc,
            SIZE_T  uBytes);


/*  メモリの内容を比較する
    lpvBuf1,バッファ1
    lpvBuf1,バッファ2
     uBytes,バイト数
        RET,0:等しい,負:バッファ1は小さい,正:バッファ1は大きい              */
int WINAPI
MemoryCompare (LPCVOID lpvBuf1,
               LPCVOID lpvBuf2,
               SIZE_T  uBytes);


/*  メモリを確保する
    uBytes,サイズ
       RET,確保したメモリ,NULL:エラー                                       */
LPVOID WINAPI
MemoryAlloc (SIZE_T uBytes);


/*  メモリを解放する
    lpMemory,確保したメモリ
         RET,TRUE:正常終了,FALSE:エラー                                     */
BOOL WINAPI
MemoryFree (LPVOID lpMemory);


/*  メモリのサイズを変更する
    lpMemory,確保したメモリ
      uBytes,サイズ
         RET,確保したメモリ,NULL:エラー                                     */
LPVOID WINAPI
MemoryReAlloc (LPVOID lpMemory,
               SIZE_T uBytes);


/*  メモリを複製する
    lpMemory,複製するメモリ
      uBytes,サイズ
         RET,確保したメモリ,NULL:エラー                                     */
LPVOID WINAPI
MemoryDuplicate (LPCVOID lpMemory,
                 SIZE_T  uBytes);


#ifdef COMMON_NO_DEPRECATE
# pragma deprecated(MemoryString, MemoryStringW, MemoryStringA)
# define MemoryStringW StringDuplicateW
# define MemoryStringA StringDuplicateA
# ifdef UNICODE
#  define MemoryString StringDuplicateW
# else /* not UNICODE */
#  define MemoryString StringDuplicateA
# endif /* not UNICODE */
#endif /* COMMON_NO_DEPRECATE */


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  文字列を書式化する
    ppszString,文字列
    lpszFormat,書式文字列
       arglist,引数
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wvasprintfW (LPWSTR  *ppszString,
             LPCWSTR  lpszFormat,
             va_list  arglist);


/*  文字列を書式化する
    ppszString,文字列
    lpszFormat,書式文字列
       arglist,引数
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wvasprintfA (LPSTR  *ppszString,
             LPCSTR  lpszFormat,
             va_list arglist);


/*  文字列を書式化する
    ppszString,文字列
      lpFormat,書式文字列
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wasprintfW (LPWSTR  *ppszString,
            LPCWSTR  lpszFormat, ...);


/*  文字列を書式化する
    ppszString,文字列
      lpFormat,書式文字列
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wasprintfA (LPSTR  *ppszString,
            LPCSTR  lpszFormat, ...);


/*  文字列を複製する
    lpszText,文字列
       nSize,文字数(負:NULL終端文字列)
         RET,複製したメモリ,NULL:エラー                                     */
LPWSTR WINAPI
StringDuplicateExW (LPCWSTR lpszText,
                    SSIZE_T nSize);


/*  文字列を複製する
    lpszText,文字列
       nSize,文字数(負:NULL終端文字列)
         RET,複製したメモリ,NULL:エラー                                     */
LPSTR WINAPI
StringDuplicateExA (LPCSTR  lpszText,
                    SSIZE_T nSize);


/*  文字列を複製する
    lpszText,文字列
         RET,複製したメモリ,NULL:エラー                                     */
LPWSTR WINAPI
StringDuplicateW (LPCWSTR lpszText);


/*  文字列を複製する
    lpszText,文字列
         RET,複製したメモリ,NULL:エラー                                     */
LPSTR WINAPI
StringDuplicateA (LPCSTR lpszText);


/*  マルチバイト文字列をワイド文字列に変換する
    lpszMultiByte,マルチバイト文字列
     cchMultiByte,文字数(負:NULL終端文字列)
              RET,ワイド文字列                                              */
LPWSTR WINAPI
StringMultiByteToWideCharEx (LPCSTR  lpszMultiByte,
                             SSIZE_T cchMultiByte);


/*  ワイド文字列をマルチバイト文字列に変換する
    lpszWideChar,ワイド文字列
     cchWideChar,文字数(負:NULL終端文字列)
             RET,マルチバイト文字列                                         */
LPSTR WINAPI
StringWideCharToMultiByteEx (LPCWSTR lpszWideChar,
                             SSIZE_T cchWideChar);


/*  マルチバイト文字列をワイド文字列に変換する
    lpszMultiByte,マルチバイト文字列
              RET,ワイド文字列                                              */
LPWSTR WINAPI
StringMultiByteToWideChar (LPCSTR lpszMultiByte);


/*  ワイド文字列をマルチバイト文字列に変換する
    lpszWideChar,ワイド文字列
             RET,マルチバイト文字列                                         */
LPSTR WINAPI
StringWideCharToMultiByte (LPCWSTR lpszWideChar);


/*  文字列を正規化する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
           RET,正規化された文字列                                           */
LPWSTR WINAPI
StringCanonicalizeExW (LPCWSTR lpszString,
                       SSIZE_T nSize);


/*  文字列を正規化する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
           RET,正規化された文字列                                           */
LPSTR WINAPI
StringCanonicalizeExA (LPCSTR  lpszString,
                       SSIZE_T nSize);


/*  文字列を正規化する
    lpszString,文字列
           RET,正規化された文字列                                           */
LPWSTR WINAPI
StringCanonicalizeW (LPCWSTR lpszString);


/*  文字列を正規化する
    lpszString,文字列
           RET,正規化された文字列                                           */
LPSTR WINAPI
StringCanonicalizeA (LPCSTR lpszString);


#ifdef UNICODE
# define wvasprintf wvasprintfW
# define wasprintf wasprintfW
# define StringDuplicateEx StringDuplicateExW
# define StringDuplicate StringDuplicateW
# define StringFromMultiByteEx StringMultiByteToWideCharEx
# define StringFromMultiByte StringMultiByteToWideChar
# define StringFromWideCharEx StringDuplicateExW
# define StringFromWideChar StringDuplicateW
# define StringToMultiByteEx StringWideCharToMultiByteEx
# define StringToMultiByte StringWideCharToMultiByte
# define StringToWideCharEx StringDuplicateExW
# define StringToWideChar StringDuplicateW
# define StringCanonicalizeEx StringCanonicalizeExW
# define StringCanonicalize StringCanonicalizeW
#else /* not UNICODE */
# define wvasprintf wvasprintfA
# define wasprintf wasprintfA
# define StringDuplicateEx StringDuplicateExA
# define StringDuplicate StringDuplicateA
# define StringFromMultiByteEx StringDuplicateExA
# define StringFromMultiByte StringDuplicateA
# define StringFromWideCharEx StringWideCharToMultiByteEx
# define StringFromWideChar StringWideCharToMultiByte
# define StringToMultiByteEx StringDuplicateExA
# define StringToMultiByte StringDuplicateA
# define StringToWideCharEx StringMultiByteToWideCharEx
# define StringToWideChar StringMultiByteToWideChar
# define StringCanonicalizeEx StringCanonicalizeExA
# define StringCanonicalize StringCanonicalizeA
#endif /* not UNICODE */


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  文字列を結合する
    lpszSeparator,区切り文字列
              RET,文字列                                                    */
LPWSTR WINAPI
StringJoinW (LPCWSTR lpszSeparator, ...);


/*  文字列を結合する
    lpszSeparator,区切り文字列
              RET,文字列                                                    */
LPSTR WINAPI
StringJoinA (LPCSTR lpszSeparator, ...);


/*  文字列を結合する
    lpszSeparator,区切り文字列
       plpszArray,分割された文字列
              RET,文字列                                                    */
LPWSTR WINAPI
StringJoinVW (LPCWSTR  lpszSeparator,
              LPCWSTR *plpszArray);


/*  文字列を結合する
    lpszSeparator,区切り文字列
       plpszArray,分割された文字列
              RET,文字列                                                    */
LPSTR WINAPI
StringJoinVA (LPCSTR  lpszSeparator,
              LPCSTR *plpszArray);


/*  分割された文字列の数を求める
    plpszArray,分割された文字列
           RET,分割された文字列の数                                         */
int WINAPI
StringLengthVW (LPCWSTR *plpszArray);


/*  分割された文字列の数を求める
    plpszArray,分割された文字列
           RET,分割された文字列の数                                         */
int WINAPI
StringLengthVA (LPCSTR *plpszArray);


/*  分割された文字列をコピーする
    plpszArray,分割された文字列
           RET,分割された文字列                                             */
LPWSTR * WINAPI
StringCopyVW (LPCWSTR *plpszArray);


/*  分割された文字列をコピーする
    plpszArray,分割された文字列
           RET,分割された文字列                                             */
LPSTR * WINAPI
StringCopyVA (LPCSTR *plpszArray);


/*  分割された文字列のNULL文字列を削除する
    plpszArray,分割された文字列
           RET,分割された文字列                                             */
LPWSTR * WINAPI
StringShrinkVW (LPCWSTR *plpszArray);


/*  分割された文字列のNULL文字列を削除する
    plpszArray,分割された文字列
           RET,分割された文字列                                             */
LPSTR * WINAPI
StringShrinkVA (LPCSTR *plpszArray);


#ifdef UNICODE
# define StringJoin StringJoinW
# define StringJoinV StringJoinVW
# define StringLengthV StringLengthVW
# define StringCopyV StringCopyVW
# define StringShrinkV StringShrinkVW
#else /* not UNICODE */
# define StringJoin StringJoinA
# define StringJoinV StringJoinVA
# define StringLengthV StringLengthVA
# define StringCopyV StringCopyVA
# define StringShrinkV StringShrinkVA
#endif /* not UNICODE */


/******************************************************************************
*                                                                             *
******************************************************************************/
typedef int (CALLBACK *StringSplitProcW_t) (LPCWSTR lpszString,
                                            LPVOID  lpData);
typedef int (CALLBACK *StringSplitProcA_t) (LPCSTR lpszString,
                                            LPVOID lpData);


/*  文字列を分割する
         lpszString,文字列
    StringSplitProc,コールバック関数
             lpData,ユーザデータ
                RET,分割された文字列                                        */
LPWSTR * WINAPI
StringSplitW (LPCWSTR            lpszString,
              StringSplitProcW_t StringSplitProc,
              LPVOID             lpData);


/*  文字列を分割する
         lpszString,文字列
    StringSplitProc,コールバック関数
             lpData,ユーザデータ
                RET,分割された文字列                                        */
LPSTR * WINAPI
StringSplitA (LPCSTR             lpszString,
              StringSplitProcA_t StringSplitProc,
              LPVOID             lpData);


/*  文字列を分割文字列の中の1文字で分割する
    lpszString,文字列
      lpszData,分割文字列
           RET,分割された文字列                                             */
LPWSTR * WINAPI StringSplitDelimiterW   (LPCWSTR lpszString, LPCWSTR lpszData);
LPSTR  * WINAPI StringSplitDelimiterA   (LPCSTR  lpszString, LPCSTR  lpszData);


/*  文字列を分割文字列の中の1文字で分割する
    lpszString,文字列
      lpszData,分割文字列(大文字小文字を区別しない)
           RET,分割された文字列                                             */
LPWSTR * WINAPI StringSplitDelimiterIW  (LPCWSTR lpszString, LPCWSTR lpszData);
LPSTR  * WINAPI StringSplitDelimiterIA  (LPCSTR  lpszString, LPCSTR  lpszData);


/*  文字列を分割文字列の中の1文字に一致する文字列で分割する
    lpszString,文字列
      lpszData,分割文字列
           RET,分割された文字列                                             */
LPWSTR * WINAPI StringSplitDelimitersW  (LPCWSTR lpszString, LPCWSTR lpszData);
LPSTR  * WINAPI StringSplitDelimitersA  (LPCSTR  lpszString, LPCSTR  lpszData);


/*  文字列を分割文字列の中の1文字に一致する文字列で分割する
    lpszString,文字列
      lpszData,分割文字列(大文字小文字を区別しない)
           RET,分割された文字列                                             */
LPWSTR * WINAPI StringSplitDelimitersIW (LPCWSTR lpszString, LPCWSTR lpszData);
LPSTR  * WINAPI StringSplitDelimitersIA (LPCSTR  lpszString, LPCSTR  lpszData);


/*  文字列を分割文字列で分割する
    lpszString,文字列
      lpszData,分割文字列
           RET,分割された文字列                                             */
LPWSTR * WINAPI StringSplitSeparatorW   (LPCWSTR lpszString, LPCWSTR lpszData);
LPSTR  * WINAPI StringSplitSeparatorA   (LPCSTR  lpszString, LPCSTR  lpszData);


/*  文字列を分割文字列で分割する
    lpszString,文字列
      lpszData,分割文字列(大文字小文字を区別しない)
           RET,分割された文字列                                             */
LPWSTR * WINAPI StringSplitSeparatorIW  (LPCWSTR lpszString, LPCWSTR lpszData);
LPSTR  * WINAPI StringSplitSeparatorIA  (LPCSTR  lpszString, LPCSTR  lpszData);


/*  文字列を分割文字列の中のどの文字にも一致しない文字列で分割する
    lpszString,文字列
      lpszData,分割文字列
           RET,分割された文字列                                             */
LPWSTR * WINAPI StringSplitComplementW  (LPCWSTR lpszString, LPCWSTR lpszData);
LPSTR  * WINAPI StringSplitComplementA  (LPCSTR  lpszString, LPCSTR  lpszData);


/*  文字列を分割文字列の中のどの文字にも一致しない文字列で分割する
    lpszString,文字列
      lpszData,分割文字列(大文字小文字を区別しない)
           RET,分割された文字列                                             */
LPWSTR * WINAPI StringSplitComplementIW (LPCWSTR lpszString, LPCWSTR lpszData);
LPSTR  * WINAPI StringSplitComplementIA (LPCSTR  lpszString, LPCSTR  lpszData);


#ifdef UNICODE
# define StringSplit            StringSplitW
# define StringSplitDelimiter   StringSplitDelimiterW
# define StringSplitDelimiterI  StringSplitDelimiterIW
# define StringSplitDelimiters  StringSplitDelimitersW
# define StringSplitDelimitersI StringSplitDelimitersIW
# define StringSplitSeparator   StringSplitSeparatorW
# define StringSplitSeparatorI  StringSplitSeparatorIW
# define StringSplitComplement  StringSplitComplementW
# define StringSplitComplementI StringSplitComplementIW
#else /* not UNICODE */
# define StringSplit            StringSplitA
# define StringSplitDelimiter   StringSplitDelimiterA
# define StringSplitDelimiterI  StringSplitDelimiterIA
# define StringSplitDelimiters  StringSplitDelimitersA
# define StringSplitDelimitersI StringSplitDelimitersIA
# define StringSplitSeparator   StringSplitSeparatorA
# define StringSplitSeparatorI  StringSplitSeparatorIA
# define StringSplitComplement  StringSplitComplementA
# define StringSplitComplementI StringSplitComplementIA
#endif /* not UNICODE */


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ファイルをメモリに読み込む
    lpszFile,ファイル名
    lpdwSize,ファイルサイズ
         RET,メモリ                                                         */
LPVOID WINAPI
LoadFileW (LPCWSTR lpszFile,
           LPDWORD lpdwSize);


/*  ファイルをメモリに読み込む
    lpszFile,ファイル名
    lpdwSize,ファイルサイズ
         RET,メモリ                                                         */
LPVOID WINAPI
LoadFileA (LPCSTR  lpszFile,
           LPDWORD lpdwSize);


/*  メモリをファイルに書き込む
    lpszFile,ファイル名
    lpBuffer,メモリ
      nSize,バイト数(負:NULL終端文字列)
         RET,TRUE:正常終了,FALSE:エラー                                     */
BOOL WINAPI
SaveFileW (LPCWSTR lpszFile,
           LPCVOID lpBuffer,
           SSIZE_T nSize);


/*  メモリをファイルに書き込む
    lpszFile,ファイル名
    lpBuffer,メモリ
      nSize,バイト数(負:NULL終端文字列)
         RET,TRUE:正常終了,FALSE:エラー                                     */
BOOL WINAPI
SaveFileA (LPCSTR  lpszFile,
           LPCVOID lpBuffer,
           SSIZE_T nSize);


#ifdef UNICODE
# define LoadFile LoadFileW
# define SaveFile SaveFileW
#else /* not UNICODE */
# define LoadFile LoadFileA
# define SaveFile SaveFileA
#endif /* not UNICODE */


/******************************************************************************
*                                                                             *
******************************************************************************/
typedef struct _LIST *LPLIST;
typedef int (WINAPI *ListCompare_t)(LPVOID lpValue1, LPVOID lpValue2,
                                                                LPVOID lpData);
typedef VOID (WINAPI *ListFree_t)(LPVOID lpValue, LPVOID lpData);
typedef BOOL (CALLBACK *ListEnumProc_t)(LPVOID lpValue, LPVOID lpData);


/*  値を直接比較する
    lpValue1,値1
    lpValue2,値2
      lpData,ユーザデータ
         RET,負:値1が小さい,0:等しい,正:値1が大きい                         */
int WINAPI
ListCompareDirect (LPVOID lpValue1,
                   LPVOID lpValue2,
                   LPVOID lpData);


/*  値を文字列として比較する
    lpValue1,値1
    lpValue2,値2
      lpData,ユーザデータ
         RET,負:値1が小さい,0:等しい,正:値1が大きい                         */
int WINAPI
ListCompareStringW (LPVOID lpValue1,
                    LPVOID lpValue2,
                    LPVOID lpData);


/*  値を文字列として比較する
    lpValue1,値1
    lpValue2,値2
      lpData,ユーザデータ
         RET,負:値1が小さい,0:等しい,正:値1が大きい                         */
int WINAPI
ListCompareStringA (LPVOID lpValue1,
                    LPVOID lpValue2,
                    LPVOID lpData);


/*  リストの値を解放する
    lpValue,値
     lpData,ユーザデータ                                                    */
VOID WINAPI
ListFree (LPVOID lpValue,
          LPVOID lpData);


/*  線形リストの末尾に追加する
    lpStart,線形リスト(NULL:新規作成)
    lpValue,値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListAppend (LPLIST lpStart,
            LPVOID lpValue);


/*  線形リストの先頭に追加する
    lpStart,線形リスト(NULL:新規作成)
    lpValue,値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListPrepend (LPLIST lpStart,
             LPVOID lpValue);


/*  線形リストに追加する
      lpStart,線形リスト(NULL:新規作成)
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListInsert (LPLIST        lpStart,
            LPVOID        lpValue,
            ListCompare_t lpCompare,
            LPVOID        lpData);


/*  線形リストをソートする
      lpStart,線形リスト
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListSort (LPLIST        lpStart,
          ListCompare_t lpCompare,
          LPVOID        lpData);


/*  線形リストの先頭を求める
    lpStart,線形リスト
        RET,値(NULL:要素なし)                                               */
LPVOID WINAPI
ListFirst (LPLIST lpStart);


/*  線形リストの末尾を求める
    lpStart,線形リスト
        RET,値(NULL:要素なし)                                               */
LPVOID WINAPI
ListLast (LPLIST lpStart);


/*  線形リストのn番目を求める
    lpStart,線形リスト
     nIndex,n番(負:後方から,正or0:前方から)
        RET,値(NULL:要素なし)                                               */
LPVOID WINAPI
ListIndex (LPLIST lpStart,
           int    nIndex);


/*  線形リストの要素数を求める
    lpStart,線形リスト
        RET,要素数                                                          */
int WINAPI
ListLength (LPLIST lpStart);


/*  線形リストの一致する要素の数を求める
      lpStart,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,一致数                                                        */
int WINAPI
ListHasValue (LPLIST        lpStart,
              LPVOID        lpValue,
              ListCompare_t lpCompare,
              LPVOID        lpData);


/*  線形リストの一致する要素を削除する
      lpStart,線形リスト
      lpValue,値
         fAll,TRUE:すべて削除,FALSE:最初の要素のみ削除
    lpCompare,比較関数
       lpFree,解放関数
       lpData,ユーザデータ
          RET,線形リスト(NULL:すべて削除)                                   */
LPLIST WINAPI
ListRemove (LPLIST        lpStart,
            LPVOID        lpValue,
            BOOL          fAll,
            ListCompare_t lpCompare,
            ListFree_t    lpFree,
            LPVOID        lpData);


/*  線形リストを解放する
    lpStart,線形リスト
     lpFree,解放関数
     lpData,ユーザデータ                                                    */
VOID WINAPI
ListRelease (LPLIST     lpStart,
             ListFree_t lpFree,
             LPVOID     lpData);


/*  線形リストの要素を列挙する
    lpStart,線形リスト
    lpFree,列挙関数
    lpData,ユーザデータ
       RET,TRUE:すべて列挙,FALSE:中断                                       */
BOOL WINAPI
ListEnum (LPLIST         lpStart,
          ListEnumProc_t lpProc,
          LPVOID         lpData);


#ifdef UNICODE
# define ListCompareString ListCompareStringW
#else /* not UNICODE */
# define ListCompareString ListCompareStringA
#endif /* not UNICODE */


/******************************************************************************
*                                                                             *
******************************************************************************/
typedef struct _HASH *LPHASH;
typedef BOOL (WINAPI *HashCompare_t )(LPVOID lpKey1,
                                      LPVOID lpKey2,
                                      LPVOID lpData);
typedef VOID (WINAPI *HashFree_t) (LPVOID lpObject,
                                   LPVOID lpData);
typedef BOOL (CALLBACK *HashEnumProc_t) (LPVOID lpKey,
                                         LPVOID lpValue,
                                         LPVOID lpData);


/*  鍵を直接比較する
    lpKey1,鍵1
    lpKey2,鍵2
    lpData,ユーザデータ
       RET,TRUE:等しい,FALSE:異なる                                         */
BOOL WINAPI
HashCompareDirect (LPVOID lpKey1,
                   LPVOID lpKey2,
                   LPVOID lpData);


/*  鍵を文字列として比較する
    lpKey1,鍵1
    lpKey2,鍵2
    lpData,ユーザデータ
       RET,TRUE:等しい,FALSE:異なる                                         */
BOOL WINAPI
HashCompareStringA (LPVOID lpKey1,
                    LPVOID lpKey2,
                    LPVOID lpData);


/*  鍵を文字列として比較する
    lpKey1,鍵1
    lpKey2,鍵2
    lpData,ユーザデータ
       RET,TRUE:等しい,FALSE:異なる                                         */
BOOL WINAPI
HashCompareStringW (LPVOID lpKey1,
                    LPVOID lpKey2,
                    LPVOID lpData);


/*  ハッシュの鍵または値を解放する
    lpObject,鍵または値
      lpData,ユーザデータ                                                   */
VOID WINAPI
HashFree (LPVOID lpObject,
          LPVOID lpData);


/*  ハッシュテーブルを作成する
      lpHashCompare,比較関数
      lpHashFreeKey,鍵解放関数
    lpHashFreeValue,値解放関数
             lpData,ユーザデータ
                RET,ハッシュテーブル                                        */
LPHASH WINAPI
HashCreate (HashCompare_t lpHashCompare,
            HashFree_t    lpHashFreeKey,
            HashFree_t    lpHashFreeValue,
            LPVOID        lpData);


/*  ハッシュテーブルを解放する
    lpHash,ハッシュテーブル                                                 */
VOID WINAPI
HashRelease (LPHASH lpHash);


/*  ハッシュテーブルに要素を挿入する
     lpHash,ハッシュテーブル
      lpKey,鍵
    lpValue,値
        RET,ハッシュテーブル                                                */
LPHASH WINAPI
HashInsert (LPHASH lpHash,
            LPVOID lpKey,
            LPVOID lpValue);


/*  ハッシュテーブルの値を求める
    lpHash,ハッシュテーブル
     lpKey,鍵
       RET,値(NULL:値なし)                                                  */
LPVOID WINAPI
HashLookup (LPHASH lpHash,
            LPVOID lpKey);


/*  ハッシュテーブルの要素を削除する
    lpHash,ハッシュテーブル
     lpKey,鍵
       RET,TRUE:削除できた,FALSE:削除できない                               */
BOOL WINAPI
HashRemove (LPHASH lpHash,
            LPVOID lpKey);


/*  ハッシュテーブルの要素を列挙する
    lpHash,ハッシュテーブル
    lpProc,列挙関数
    lpData,ユーザデータ
       RET,TRUE:すべて列挙,FALSE:中断                                       */
BOOL WINAPI
HashEnum (LPHASH         lpHash,
          HashEnumProc_t lpProc,
          LPVOID         lpData);


#ifdef UNICODE
# define HashCompareString HashCompareStringW
#else /* not UNICODE */
# define HashCompareString HashCompareStringA
#endif /* not UNICODE */


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:コマンドラインの解析
    lpszCmdLine,コマンドライン文字列
            RET,引数へのポインタ,NULL:エラー                                */
LPWSTR * WINAPI
MakeArgumentW (LPCWSTR lpszCmdLine);


/*  ja:コマンドラインの解析
    lpszCmdLine,コマンドライン文字列
            RET,引数へのポインタ,NULL:エラー                                */
LPSTR * WINAPI
MakeArgumentA (LPCSTR lpszCmdLine);


/*  文字列をデバイスに出力する
    nStdHandle,デバイス
    lpszFormat,書式文字列                                                   */
VOID WINAPI
PrintConsoleW (DWORD   nStdHandle,
               LPCWSTR lpszFormat, ...);


/*  文字列をデバイスに出力する
    nStdHandle,デバイス
    lpszFormat,書式文字列                                                   */
VOID WINAPI
PrintConsoleA (DWORD  nStdHandle,
               LPCSTR lpszFormat, ...);


#ifdef UNICODE
# define MakeArgument MakeArgumentW
# define PrintConsole PrintConsoleW
#else /* not UNICODE */
# define MakeArgument MakeArgumentA
# define PrintConsole PrintConsoleA
#endif /* not UNICODE */


#ifdef COMMON_NO_DEPRECATE
# pragma deprecated(CanonicalStringEx, CanonicalStringExW, CanonicalStringExA)
# pragma deprecated(CanonicalString, CanonicalStringW, CanonicalStringA)
# pragma deprecated(SplitString, SplitStringW, SplitStringA)
# define CanonicalStringExW StringCanonicalExW
# define CanonicalStringW StringCanonicalW
# define SplitStringW StringSplitW
# define CanonicalStringExA StringCanonicalExA
# define CanonicalStringA StringCanonicalA
# define SplitStringA StringSplitA
# ifdef UNICODE
#  define CanonicalStringEx StringCanonicalExW
#  define CanonicalString StringCanonicalW
#  define SplitString StringSplitW
# else /* not UNICODE */
#  define CanonicalStringEx StringCanonicalExA
#  define CanonicalString StringCanonicalA
#  define SplitString StringSplitA
# endif /* not UNICODE */
#endif /* COMMON_NO_DEPRECATE */


#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /* __COMMON_H__ */
