/**
 * Copyright  2007 matsu@zuena.org.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.zuena.guiceex.jpa.utils.statements;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.persistence.Column;
import javax.persistence.EntityManager;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.zuena.guiceex.util.StringUtils;

import static org.zuena.guiceex.util.ClassUtils.*;


/**
 * @author matsu@zuena.org.
 * @param <T>
 */
@SuppressWarnings("unchecked")
public class Select<T> extends Statement<T>{

	private static final Log log = LogFactory.getLog(Select.class);
	private Columns<T>[] columnses;
	private Table[] tables;
	
	/**
	 * 
	 */
	public Select(Class<T> resultType) {
		this(new Columns<T>(resultType));
	}
	
	/**
	 * 
	 */
	public Select(Columns<T>... columnses) {
		this.columnses = columnses;
	}
	
	/**
	 * 
	 * @param entityClass
	 * @return
	 */
	public Select<T> from(Class entityClass){
		return from(new Table<T>(entityClass));
	}
	
	/**
	 * 
	 * @param tables
	 * @return
	 */
	public Select<T> from(Table... tables){
		this.tables = tables;
		return this;
	}

	/**
	 * 
	 * @param em
	 * @return
	 */
	public List<T> getResultList(EntityManager em) {
		List l = em.createNativeQuery(getStatementString())
					.getResultList();
		List<T> results = new ArrayList<T>();
		
		for (Object o : l) {
			for(Columns<T> columns : columnses){
				Object[] values = (Object[])o;
				T t = newInstance(columns.getResultClass());
				for (int i=0 ; i < values.length ; i++) {
					try {
						Field field = columns.getField(i);
						field.setAccessible(true);
						field.set(t, values[i]);
					} catch (Exception e) {
					}
				}
				results.add(t);
			}
		}
		return results;
	}

	/**
	 * 
	 * @return
	 */
	public String getStatementString() {
		
		Map<Table,Set<Columns>> entityGetterNames = new LinkedHashMap<Table,Set<Columns>>();

		StringBuffer sb = new StringBuffer("SELECT\n");
		boolean first = true;
		for (Columns<T> columns : columnses) {
			
			Table table = getTableByAlias(columns);

			for(int i=0 ; i < columns.columnCount() ; i++){
				
				Field field = table.getField(columns.getFieldName(i));
				if (field == null){
					continue;
				}
				
				if (first){
					first = false;
					sb.append('\t');
				}else{
					sb.append("\t,");
				}
				
				if (table.getAlias() != null){
					sb.append(table.getAlias() + '.');
				}
				
				String name = null;
				Column c = field.getAnnotation(Column.class);
				if (c != null){
					name = c.name();
				}		
				if (name == null){
					name = columns.getColumnName(i);
				}
				sb.append(name +'\n');
			}
		}
		
		sb.append("FROM\n");
		first = true;
		for (Table table : tables) {
			if (first){
				first = false;
				sb.append("\t");
			}else{
				sb.append("\t,");
			}
			sb.append(table.getTableNameWithAlias() + '\n');
		}		
		
		if (log.isDebugEnabled()){
			log.debug("\n"+sb);
		}
		return sb.toString();
	}
	
	
	private Table getTableByAlias(Columns columns) {
		for (Table table : tables) {
			String alias = columns.getAlias();
			if (!StringUtils.isEmpty(alias) && alias.equals(table.getAlias())
				|| columns.getResultClass().equals(table.getEntityClass())){
				return table;
			}
		}
		return tables[0];
	}
}
